package com.openexchange.office.tools.doc;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;

import com.openexchange.file.storage.File;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.office.tools.SessionUtils;
import com.openexchange.office.tools.StorageHelper;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.FolderHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.iterator.SearchIterator;


public class BackupFileHelper {

    private final static org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(BackupFileHelper.class);

    static public final String BACKUP_FILE_POSTFIX = "-bak";

    /**
     *
     * @param services
     * @param session
     * @param storageHelper
     * @param folderId
     * @param metaData
     * @return
     */
    public static boolean canCreateOrWriteBackupFile(final ServiceLookup services, final Session session, final StorageHelper storageHelper, final String folderId, final DocumentMetaData metaData) {
       Validate.notNull(services, "service lookup must be set");
       Validate.notNull(session, "session must be set");
       Validate.notNull(storageHelper, "storageHelper must be set");
       Validate.notNull(folderId, "folderId must be set");
       Validate.notNull(metaData, "metaData must be set");

       boolean canCreateWriteBackupFile = false;

       if (!storageHelper.supportsFileVersions()) {
           // we must check further requirements to answer the question
           try {
               final IDBasedFolderAccess folderAccess = FolderHelper.getFolderAccess(services, session);
               final IDBasedFileAccess fileAccess = FileHelper.getFileAccess(services, session);
               int[] permissions = FolderHelper.getFolderPermissions(folderAccess, folderId);
               if (null != permissions) {
                   int write  = permissions[FolderHelper.FOLDER_PERMISSIONS_WRITE];
                   int folder = permissions[FolderHelper.FOLDER_PERMISSIONS_FOLDER];
                   if ((folder >= com.openexchange.file.storage.FileStoragePermission.CREATE_OBJECTS_IN_FOLDER) &&
                       (write >= com.openexchange.file.storage.FileStoragePermission.WRITE_ALL_OBJECTS)) {
                       // permissions are sufficient to create/write a bak file - no further tests necessary
                       canCreateWriteBackupFile = true;
                   } else {
                       // We have again dig deeper and now check a possible bak-file existence
                       // First check file existence using the current file name
                       final String fileName = metaData.getFileName();
                       final String backupFileName = FileHelper.createFilenameWithPostfix(fileName, BackupFileHelper.BACKUP_FILE_POSTFIX);
                       SearchIterator<File> iter = FileHelper.getMetaDataFromFileName(fileAccess, folderId, backupFileName);

                       File bakMetaData = null;
                       if (!StringUtils.isEmpty(backupFileName) && (iter != null) && (iter.hasNext())) {
                           // BACKUP file exists, retrieve meta data
                           bakMetaData = iter.next();
                       }

                       if (null != bakMetaData) {
                           // BACKUP file exists - check write permissions
                           canCreateWriteBackupFile = ((write >= com.openexchange.file.storage.FileStoragePermission.WRITE_OWN_OBJECTS) &&
                                                        bakMetaData.getCreatedBy() == SessionUtils.getUserId(session));
                       } else {
                           // BACKUP file doesn't exists - check folder permissions
                           canCreateWriteBackupFile = (folder >= com.openexchange.file.storage.FileStoragePermission.CREATE_OBJECTS_IN_FOLDER);
                       }
                   }
               }
           } catch (Exception e) {
               LOG.error("RT connection: Exception caught while trying to determine backup file permissions", e);
           }
       } else {
           canCreateWriteBackupFile = true;
       }

       return canCreateWriteBackupFile;
   }

}
