/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import org.docx4j.dml.BaseStyles;
import org.docx4j.dml.CTColorScheme;
import org.docx4j.dml.Theme;
import org.docx4j.openpackaging.parts.ThemePart;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTCfIcon;
import org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTCfvo;
import org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTColorScale;
import org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.STCfvoType;
import org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.STDataBarAxisPosition;
import org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.STDataBarDirection;
import org.xlsx4j.sml.CTBorder;
import org.xlsx4j.sml.CTBorderPr;
import org.xlsx4j.sml.CTCellAlignment;
import org.xlsx4j.sml.CTCellProtection;
import org.xlsx4j.sml.CTCfRule;
import org.xlsx4j.sml.CTColor;
import org.xlsx4j.sml.CTColors;
import org.xlsx4j.sml.CTDataBar;
import org.xlsx4j.sml.CTFill;
import org.xlsx4j.sml.CTGradientStop;
import org.xlsx4j.sml.CTIconSet;
import org.xlsx4j.sml.CTIndexedColors;
import org.xlsx4j.sml.CTNumFmt;
import org.xlsx4j.sml.CTPatternFill;
import org.xlsx4j.sml.CTRgbColor;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.ICfRule;
import org.xlsx4j.sml.ICfvo;
import org.xlsx4j.sml.IColorScale;
import org.xlsx4j.sml.IDataBar;
import org.xlsx4j.sml.IIconSet;
import org.xlsx4j.sml.STBorderStyle;
import org.xlsx4j.sml.STHorizontalAlignment;
import org.xlsx4j.sml.STPatternType;
import org.xlsx4j.sml.STVerticalAlignment;
import org.xlsx4j.sml.SmlUtils;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.XlsxOperationDocument;


public class Utils {

    public static void createFill(JSONObject dest, CTStylesheet stylesheet, CTFill fill)
        throws JSONException {

        if(fill!=null) {
            if (null == fill.getPatternFill() && null != fill.getGradientFill()) {
                //fallback for gradients! taking its first color
                List<CTGradientStop> stop = fill.getGradientFill().getStop();
                if (null != stop && stop.size() > 0) {
                    CTColor color = stop.get(0).getColor();
                    final JSONObject jsonColor = createColor(stylesheet, color);
                    if(jsonColor!=null) {
                    	dest.put("fillColor", jsonColor);
                    }
                }
            }else {
                createPatternFill(dest, stylesheet, fill.getPatternFill());
            }
//          createGradientFill(dest, fill.getGradientFill());
        }
    }

    public static void createPatternFill(JSONObject dest, CTStylesheet stylesheet, CTPatternFill patternFill)
        throws JSONException {

        if(patternFill!=null) {
        	STPatternType patternType = patternFill.getPatternType();
        	if(patternType==null) {
        		patternType = STPatternType.SOLID;
        	}
            switch(patternType) {
                case NONE: {
                    JSONObject jsonColor = new JSONObject(1);
                    jsonColor.put("type", "auto");
                	dest.put("fillColor", jsonColor);
                	break;
                }
                case SOLID :
                default: {
                    JSONObject jsonColor = null;
                    if(patternFill.getFgColor()!=null) {
                    	jsonColor = createColor(stylesheet, patternFill.getFgColor());
                    	if(patternFill.getBgColor()!=null&&(jsonColor==null||jsonColor.optString("type", "auto").equals("auto"))) {
                    		jsonColor = createColor(stylesheet, patternFill.getBgColor());
                    	}
                    }
                    else if(patternFill.getBgColor()!=null) {
                    	jsonColor = createColor(stylesheet, patternFill.getBgColor());
                    }
                    if(jsonColor!=null) {
                        dest.put("fillColor", jsonColor);
                    }
                }
            }
        }
    }

    final static String themeColors[] = {

/* OOXML specification at a glance... if accessing theme colors
 * as described in the specification, then at least black and
 * white are swapped. Some says that only the first two pairs
 * within the index are swapped, other think (like in poi) that each
 * non alpha rbg white and black color is switched
 *
 * TODO: I have to test this. For now I only switch the index into
 * the color scheme.
 *

        "dark1",
        "light1",
        "dark2",
        "light2",
*/
        "light1",
        "dark1",
        "light2",
        "dark2",

        "accent1",
        "accent2",
        "accent3",
        "accent4",
        "accent5",
        "accent6",
        "hyperlink",
        "followedHyperlink"
    };

    final static String[] indexedColors = {
        "000000","FFFFFF","FF0000","00FF00","0000FF","FFFF00","FF00FF","00FFFF",
        "000000","FFFFFF","FF0000","00FF00","0000FF","FFFF00","FF00FF","00FFFF",
        "800000","008000","000080","808000","800080","008080","C0C0C0","808080",
        "9999FF","993366","FFFFCC","CCFFFF","660066","FF8080","0066CC","CCCCFF",
        "000080","FF00FF","FFFF00","00FFFF","800080","800000","008080","0000FF",
        "00CCFF","CCFFFF","CCFFCC","FFFF99","99CCFF","FF99CC","CC99FF","FFCC99",
        "3366FF","33CCCC","99CC00","FFCC00","FF9900","FF6600","666699","969696",
        "003366","339966","003300","333300","993300","993366","333399","333333"
    };
    
	public static JSONArray createColorScaleSteps(IColorScale iColorScale, int minColorScaleSize, int maxColorScaleSize)
			throws JSONException {
		JSONArray jsonColorScaleSteps = new JSONArray();
		if (iColorScale != null && iColorScale.getCfvo().size() == iColorScale.getColor().size()
				&& iColorScale.getCfvo().size() >= minColorScaleSize
				&& iColorScale.getCfvo().size() <= maxColorScaleSize) {

			for (int i = 0; i < iColorScale.getCfvo().size(); i++) {
				final JSONObject colorScaleStep = createRuleOperation(iColorScale.getCfvo().get(i));
				colorScaleStep.put("c", createColor(null, iColorScale.getColor().get(i)));
				jsonColorScaleSteps.put(colorScaleStep);
			}
		}

		return jsonColorScaleSteps;
	}
	
	/**
	 * Add the CTColorScale Object to the given CfRule.
	 * @throws JSONException
	 */
	public static void addColorScaleObjectToCfRule(XlsxOperationDocument operationDocument, ICfRule iCfRule,  JSONArray jsonColorScales) throws JSONException {
		if (iCfRule instanceof org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTCfRule) {
			CTColorScale colorScale = new CTColorScale();
			for (int i = 0; i < jsonColorScales.length(); i++) {
				JSONObject jsonRule = jsonColorScales.getJSONObject(i);
				colorScale.getCfvo().add((CTCfvo) createRuleObject(new CTCfvo(), jsonRule));
				colorScale.getColor().add(getColorByJson(operationDocument, jsonRule.getJSONObject("c")));
			}
			((org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTCfRule) iCfRule).setColorScale(colorScale);
		} else {
			org.xlsx4j.sml.CTColorScale colorScale = new org.xlsx4j.sml.CTColorScale();
			for (int i = 0; i < jsonColorScales.length(); i++) {
				JSONObject jsonRule = jsonColorScales.getJSONObject(i);
				colorScale.getCfvo().add((org.xlsx4j.sml.CTCfvo) createRuleObject(new org.xlsx4j.sml.CTCfvo(), jsonRule));
				colorScale.getColor().add(getColorByJson(operationDocument, jsonRule.getJSONObject("c")));
			}
			((CTCfRule) iCfRule).setColorScale(colorScale);
		}
	}

	public static JSONObject createDataBarOperation(IDataBar dataBar, IDataBar dataBar1) throws JSONException {
		JSONObject jsonDataBar = new JSONObject();
		if (dataBar != null) {
			jsonDataBar.put("r1", createRuleOperation(dataBar.getCfvo().get(0)));
			if (dataBar.getCfvo().size() > 1) {
				jsonDataBar.put("r2", createRuleOperation(dataBar.getCfvo().get(1)));				
			}
			
			JSONObject color;
			if (dataBar.getColor() == null && dataBar1 != null) {
				color = createColor(null, dataBar1.getColor());
			} else {
				color = createColor(null, dataBar.getColor());
			}
			jsonDataBar.put("c", color);
			jsonDataBar.put("s", dataBar.isShowValue());
			jsonDataBar.put("min", dataBar.getMinLength());
			jsonDataBar.put("max", dataBar.getMaxLength());
			
			if (dataBar instanceof org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTDataBar) {
				org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTDataBar dataBar2009 = (org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTDataBar) dataBar;
				
				jsonDataBar.put("b", dataBar2009.isBorder());
				jsonDataBar.put("bc", createColor(null, dataBar2009.getBorderColor()));
				
				jsonDataBar.put("ncs", dataBar2009.isNegativeBarColorSameAsPositive());
				jsonDataBar.put("nc", createColor(null, dataBar2009.getNegativeFillColor()));
				
				jsonDataBar.put("nbs", dataBar2009.isNegativeBarBorderColorSameAsPositive());
			    jsonDataBar.put("nbc", createColor(null, dataBar2009.getNegativeBorderColor()));
				
				jsonDataBar.put("ap", dataBar2009.getAxisPosition().value());				
				jsonDataBar.put("ac", createColor(null, dataBar2009.getAxisColor()));
				
				jsonDataBar.put("g", dataBar2009.isGradient());
				jsonDataBar.put("d", dataBar2009.getDirection().value());
			}
		}

		return jsonDataBar;
	}
	
	private static CTColor getColorByJson(XlsxOperationDocument operationDocument, JSONObject jsonColor) throws FilterException, JSONException {
		CTColor color = Context.getsmlObjectFactory().createCTColor();
		if (jsonColor != null) {
			applyColor(operationDocument, color, jsonColor);			
		}
		return color;
	}
	
	/**
	 * Add the CTDataBar to the CfRule Object.
	 * @throws JSONException
	 */
	public static void addDataBarObjectToCfRule(XlsxOperationDocument operationDocument, ICfRule iCfRule,  JSONObject jsonDataBar, boolean hasBothRules, Integer priority) throws JSONException {
		
		if (jsonDataBar != null) {
			if (iCfRule instanceof org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTCfRule) {
				/**
				 * Documentation for DataBar.
				 * 	https://msdn.microsoft.com/en-us/library/dd948147%28v=office.12%29.aspx
				 */
				org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTDataBar dataBar = new org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTDataBar();
				addDefaultDataBarAttributes(dataBar, jsonDataBar);
				dataBar.getCfvo().add((CTCfvo) createRuleObject(new CTCfvo(), jsonDataBar.getJSONObject("r1")));
				dataBar.getCfvo().add((CTCfvo) createRuleObject(new CTCfvo(), jsonDataBar.getJSONObject("r2")));

				// Fill Color and Priority only if org.xlsx4j.sml.CTCfRule not exists.
				if (!hasBothRules) {
					dataBar.setFillColor(getColorByJson(operationDocument, jsonDataBar.getJSONObject("c")));					
			  		if(priority!=null) {
			  			iCfRule.setPriority(priority.intValue());
			  		}
				}
				
				if (!jsonDataBar.isNull("b")) {
					dataBar.setBorder(jsonDataBar.getBoolean("b"));
				}
				
				if (dataBar.isBorder()) {
				    if (!jsonDataBar.isNull("bc")) {
						dataBar.setBorderColor(getColorByJson(operationDocument, jsonDataBar.getJSONObject("bc")));
				    }						
					if (!jsonDataBar.isNull("nbs")) {
						dataBar.setNegativeBarBorderColorSameAsPositive(jsonDataBar.getBoolean("nbs"));
					}
				}
				
				if (!dataBar.isNegativeBarBorderColorSameAsPositive()) {
					if (!jsonDataBar.isNull("nbc")) {
						dataBar.setNegativeBorderColor(getColorByJson(operationDocument, jsonDataBar.getJSONObject("nbc")));
					}					
				}
				
				if (!jsonDataBar.isNull("ncs")) {
					dataBar.setNegativeBarBorderColorSameAsPositive(jsonDataBar.getBoolean("ncs"));
				}
				
				if (!dataBar.isNegativeBarColorSameAsPositive()) {
					if (!jsonDataBar.isNull("nc")) {
						dataBar.setNegativeFillColor(getColorByJson(operationDocument, jsonDataBar.getJSONObject("nc")));
					}
				}
				
				if (!jsonDataBar.isNull("ap")) {
					dataBar.setAxisPosition(STDataBarAxisPosition.fromValue(jsonDataBar.getString("ap")));
				}
				
				if (!jsonDataBar.isNull("ac")) {
					dataBar.setAxisColor(getColorByJson(operationDocument, jsonDataBar.getJSONObject("ac")));
				}					


				if (!jsonDataBar.isNull("g")) {
					dataBar.setGradient(jsonDataBar.getBoolean("g"));
				}

				if (!jsonDataBar.isNull("d")) {
					dataBar.setDirection(STDataBarDirection.fromValue(jsonDataBar.getString("d")));
				}
				
				((org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTCfRule) iCfRule).setDataBar(dataBar);
			} else {
				CTDataBar dataBar = new CTDataBar();
		  		if(priority!=null) {
		  			iCfRule.setPriority(priority.intValue());
		  		}
				dataBar.getCfvo().add((org.xlsx4j.sml.CTCfvo) createRuleObject(new org.xlsx4j.sml.CTCfvo(), jsonDataBar.getJSONObject("r1")));
		  		dataBar.getCfvo().add((org.xlsx4j.sml.CTCfvo) createRuleObject(new org.xlsx4j.sml.CTCfvo(), jsonDataBar.getJSONObject("r2")));

				dataBar.setColor(getColorByJson(operationDocument, jsonDataBar.getJSONObject("c")));

				addDefaultDataBarAttributes(dataBar, jsonDataBar);

				((org.xlsx4j.sml.CTCfRule)iCfRule).setDataBar(dataBar);
			}
		}

	}
	
	private static void addDefaultDataBarAttributes(IDataBar dataBar, JSONObject jsonDataBar) throws JSONException {
		if (!jsonDataBar.isNull("s")) {
			dataBar.setShowValue(jsonDataBar.getBoolean("s"));
		}
		if (!jsonDataBar.isNull("min")) {
			dataBar.setMinLength(jsonDataBar.getLong("min"));
		}
		if (!jsonDataBar.isNull("max")) {
			dataBar.setMaxLength(jsonDataBar.getLong("max"));
		}
	}
			
	
	public static JSONObject createIconSet(IIconSet iconSet)  throws JSONException {
		org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTIconSet iconSet2009;
		JSONObject jsonIconSet = new JSONObject();
		jsonIconSet.put("is", iconSet.getIconSet());
		jsonIconSet.put("s", iconSet.isShowValue());
		jsonIconSet.put("r", iconSet.isReverse());
		jsonIconSet.put("p", iconSet.isPercent());
		if (iconSet instanceof org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTIconSet) {
			iconSet2009 = (org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTIconSet)iconSet;
			jsonIconSet.put("c", iconSet2009.isCustom());
		} else {
			iconSet2009 = null;
		}
		
		JSONArray jsonRules = new JSONArray();
		List<CTCfIcon> cfIcons = null;
		if (iconSet2009 != null) {
			cfIcons = iconSet2009.getCfIcon();
		}
		
		for (int i = 0; i < iconSet.getCfvo().size(); i++) {
			ICfvo cfvo = iconSet.getCfvo().get(i);
			final JSONObject rule = createRuleOperation(cfvo);
			rule.put("gte", cfvo.isGte());
			if (cfIcons != null && i < cfIcons.size()) {
				CTCfIcon icon = iconSet2009.getCfIcon().get(i);
				rule.put("is", icon.getIconSet());
				rule.put("id", icon.getIconId());
			}
			jsonRules.put(rule);
		}
		
		jsonIconSet.put("ir", jsonRules);
		
		return jsonIconSet;
	}
	
	/**
	 * Add CTIconSet to the CfRule Object.
	 * @throws JSONException
	 */
	public static void addIconSetObjectCfRule(XlsxOperationDocument operationDocument, ICfRule iCfRule,  JSONObject jsonIconSet) throws JSONException {
		if (jsonIconSet != null) {
			if (iCfRule instanceof org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTCfRule) {
				org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTIconSet iconSet = new org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTIconSet();
				if (!jsonIconSet.isNull("c")) {
					iconSet.setCustom(jsonIconSet.getBoolean("c"));
				}
				addDefaultIconSetAttributes(iconSet, jsonIconSet);
				
				JSONArray jsonRules = jsonIconSet.getJSONArray("ir");
				for (int i = 0; i < jsonRules.length(); i++) {
					JSONObject jsonRule = jsonRules.getJSONObject(i);
					CTCfvo rule = (CTCfvo) createRuleObject(new CTCfvo(), jsonRule);
					iconSet.getCfvo().add(rule);
					if (jsonRule.has("is") && jsonRule.has("id")) {
						CTCfIcon icon = new CTCfIcon();
						icon.setIconSet(jsonRule.getString("is"));
						icon.setIconId(jsonRule.getInt("id"));				
						iconSet.getCfIcon().add(icon);
					}
				}
				
				((org.xlsx4j.schemas.microsoft.com.office.spreadsheetml_2009_9.main.CTCfRule) iCfRule).setIconSet(iconSet);
			} else {
				CTIconSet iconSet = new CTIconSet();
				
				((org.xlsx4j.sml.CTCfRule)iCfRule).setIconSet(iconSet);
				addDefaultIconSetAttributes(iconSet, jsonIconSet);
				
				
				JSONArray jsonRules = jsonIconSet.getJSONArray("ir");
				for (int i = 0; i < jsonRules.length(); i++) {
					JSONObject jsonRule = jsonRules.getJSONObject(i);
					org.xlsx4j.sml.CTCfvo rule = (org.xlsx4j.sml.CTCfvo) createRuleObject(new org.xlsx4j.sml.CTCfvo(), jsonRule);
					iconSet.getCfvo().add(rule);
				}
			}
		}
	}
	
	private static void addDefaultIconSetAttributes(IIconSet iconSet, JSONObject jsonIconSet) throws JSONException {
		iconSet.setIconSet(jsonIconSet.getString("is"));
		if (jsonIconSet.isNull("s")) {
			iconSet.setShowValue(jsonIconSet.getBoolean("s"));
		}
		if (jsonIconSet.isNull("r")) {
			iconSet.setReverse(jsonIconSet.getBoolean("r"));
		}
		if (jsonIconSet.isNull("p")) {
			iconSet.setPercent(jsonIconSet.getBoolean("p"));
		}
	}

	private static JSONObject createRuleOperation(ICfvo cfvo) throws JSONException {
		JSONObject jsonRule = new JSONObject();

		if (cfvo.getType() != null) {
			String type;
			switch (cfvo.getType().ordinal()) {
			case 0 :
				type = "formula";
				break;
			case 1 :
				type = "percent";
				break;
			case 2:
				type = "max";
				break;
			case 3:
				type = "min";
				break;
			case 4:
				type = "formula";
				break;
			case 5:
				type = "percentile";
				break;
			case 6:
				type = "automin";
				break;
			case 7:
				type = "automax";
				break;
			default:
				type = null;
			}
			if (type != null) {
				jsonRule.put("t", type);
				jsonRule.put("v", cfvo.getVal());
			}
		}

		return jsonRule;
	}
	
	private static ICfvo createRuleObject(ICfvo iCfvo, JSONObject jsonRule) throws JSONException {
		if (jsonRule != null) {
			String typeName = jsonRule.getString("t");
			if (iCfvo  instanceof CTCfvo) {
				((CTCfvo)iCfvo).setType(STCfvoType.fromValue(typeName));
			} else {
				org.xlsx4j.sml.STCfvoType type = org.xlsx4j.sml.STCfvoType.fromValue(typeName);
				if (type == null) {
					if (typeName.equalsIgnoreCase("automin")) {
						typeName = "min";
					} else if (typeName.equalsIgnoreCase("automax")) {
						typeName = "max";
					}
				}
				((org.xlsx4j.sml.CTCfvo)iCfvo).setType(org.xlsx4j.sml.STCfvoType.fromValue(typeName));
			}
			if (jsonRule.has("v")) {
				iCfvo.setVal(jsonRule.getString("v"));				
			}
			if (!jsonRule.isNull("gte")) {
				iCfvo.setGte(jsonRule.getBoolean("gte"));				
			}

		}
		return iCfvo;
	}

    public static JSONObject createColor(CTStylesheet stylesheet, CTColor color)
        throws JSONException {

        if (color!=null) {

            JSONObject jsonColor = new JSONObject(3);

            Boolean auto    = color.isAuto();
            Long    theme   = color.getTheme();
            Long    indexed = color.getIndexed();
            byte[]  rgb     = color.getRgb();
            byte    alpha   = -1;

            if(theme!=null&&theme.longValue()<themeColors.length) {
                jsonColor.put("type",  "scheme");
                jsonColor.put("value", themeColors[theme.intValue()]);
            }
            else if(auto!=null&&auto.booleanValue()) {
                jsonColor.put("type", "auto");
            }
            else if(rgb!=null&&rgb.length==3) {
                jsonColor.put("type", "rgb");
                jsonColor.put("value", Commons.bytesToHexString(rgb, 0, 3));
            }
            else if(rgb!=null&&rgb.length==4) {
//                alpha = rgb[0];
//            	Alpha is ignored now because of excel and calc, they ignore it too
                jsonColor.put("type", "rgb");
                jsonColor.put("value", Commons.bytesToHexString(rgb, 1, 3));
            }
            else if(indexed!=null) {
                if(indexed.longValue()>=indexedColors.length) {
                    jsonColor.put("type", "auto");
                }
                else {
                    jsonColor.put("type", "rgb");
                    String colorValue = indexedColors[indexed.intValue()];
                    if(stylesheet!=null) {
                        // check if indexedColor is overwritten
	                    final CTColors ctColors = stylesheet.getColors();
	                    if(ctColors!=null) {
	                        CTIndexedColors _indexedColors = ctColors.getIndexedColors();
	                        if(_indexedColors!=null) {
	                            List<CTRgbColor> _indexedColorList = _indexedColors.getRgbColor();
	                            if(_indexedColorList.size()>indexed.intValue()) {
	                                CTRgbColor _indexedRgbColor = _indexedColorList.get(indexed.intValue());
	                                if(_indexedRgbColor!=null) {
	                                    byte[] _indexedRgbColorArray = _indexedRgbColor.getRgb();
	                                    if(_indexedRgbColorArray!=null&&_indexedRgbColorArray.length==4) {
	                                        // alpha = indRgbArray[0]; don't know if alpha is used here, value seems to be zero always
	                                        colorValue = Commons.bytesToHexString(_indexedRgbColorArray, 1, 3);
	                                    }
	                                }
	                            }
	                        }
	                    }
                    }
                    jsonColor.put("value", colorValue);
                }
            }
            final JSONArray transformations = createTransformations(color, alpha);
            if (transformations!=null) {
                jsonColor.put("transformations", transformations);
            }
            return jsonColor.isEmpty() ? null : jsonColor;
        }
        return null;
    }

    public static JSONObject createDefaultColor()
    	throws JSONException {

    	JSONObject jsonColor = new JSONObject(2);
        jsonColor.put("type",  "scheme");
        jsonColor.put("value", themeColors[1]);
        return jsonColor;
    }

    private static JSONArray createTransformations(CTColor color, byte alpha)
        throws JSONException {

        JSONArray jsonColorTransformations = new JSONArray();
        if(alpha!=-1) {
            jsonColorTransformations.put(createAlphaTransformation(alpha));
        }
        if (color!=null) {
            final Double tint = color.getTint();
            if (tint!=null) {
            	final JSONObject tintTransformation = createTintTransformation(tint.doubleValue());
            	if(tintTransformation!=null) {
            		jsonColorTransformations.put(tintTransformation);
            	}
            }
        }
        return jsonColorTransformations.length()>0?jsonColorTransformations:null;
    }

    private static JSONObject createAlphaTransformation(byte alpha)
        throws JSONException {

        JSONObject alphaTransform = new JSONObject();
        alphaTransform.put("type", "alpha");
        alphaTransform.put("value", (alpha&0xff)/255.0*100000.0);
        return alphaTransform;
    }

    private static JSONObject createTintTransformation(double _tint)
        throws JSONException {

        if(_tint >= 0) {
        	final int tint = 100000 - (int)(_tint * 100000.0);
        	if(tint!=100000) {
	            final JSONObject tintTransform = new JSONObject();
	            tintTransform.put("type", "tint");
	            tintTransform.put("value", tint);
	            return tintTransform;
        	}
        }
        else {
            final int shade = 100000 - (int)(_tint *-100000.0);
            if(shade!=100000) {
                final JSONObject tintTransform = new JSONObject();
	            tintTransform.put("type", "shade");
	            tintTransform.put("value", shade);
	            return tintTransform;
            }
        }
        return null;
    }

    // CTBorder

    public static void createBorders(JSONObject jsonDest, CTStylesheet stylesheet, CTBorder borders)
        throws JSONException {

        if(borders!=null) {
            Commons.jsonPut(jsonDest, "borderLeft", createJSONfromCTBorder(stylesheet, borders.getLeft()));
            Commons.jsonPut(jsonDest, "borderTop", createJSONfromCTBorder(stylesheet, borders.getTop()));
            Commons.jsonPut(jsonDest, "borderRight", createJSONfromCTBorder(stylesheet, borders.getRight()));
            Commons.jsonPut(jsonDest, "borderBottom", createJSONfromCTBorder(stylesheet, borders.getBottom()));
            Commons.jsonPut(jsonDest, "borderInsideHor", createJSONfromCTBorder(stylesheet, borders.getHorizontal()));
            Commons.jsonPut(jsonDest, "borderInsideVert", createJSONfromCTBorder(stylesheet, borders.getVertical()));
        }
    }

    public static int pixelToHmm(double pixel) {
    	return (int) Math.round(pixel / 96.0 * 2540.0);
    }

    public static double hmmToPixel(int hmm) {
    	return hmm * 96.0 / 2540.0;
    }

    public static int HAIR_WIDTH_HMM= pixelToHmm(0.5);
    public static int THIN_WIDTH_HMM = pixelToHmm(1);
    public static int MEDIUM_WIDTH_HMM = pixelToHmm(2);
    public static int THICK_WIDTH_HMM = pixelToHmm(3);

    public static JSONObject createJSONfromCTBorder(CTStylesheet stylesheet, CTBorderPr borderPr)
        throws JSONException {

        if (borderPr==null)
            return null;

        JSONObject jsonBorder = new JSONObject();
        String style = "single";
        double width = THIN_WIDTH_HMM;
        STBorderStyle borderStyle = borderPr.getStyle();
        if(borderStyle!=null) {
            if(borderStyle==STBorderStyle.NONE) {
                style = "none";
            }
            else if(borderStyle==STBorderStyle.HAIR) {
                style = "single";
                width = HAIR_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.THIN) {
                style = "single";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DOTTED) {
                style = "dotted";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DASH_DOT_DOT) {
                style = "dashDotDot";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DASH_DOT) {
                style = "dashDot";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DASHED) {
                style = "dashed";
                width = THIN_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.MEDIUM) {
                style = "single";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.MEDIUM_DASH_DOT_DOT) {
                style = "dashDotDot";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.MEDIUM_DASH_DOT) {
                style = "dashDot";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.MEDIUM_DASHED) {
                style = "dashed";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.SLANT_DASH_DOT) {
                style = "dashDot";
                width = MEDIUM_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.THICK) {
                style = "single";
                width = THICK_WIDTH_HMM;
            }
            else if(borderStyle==STBorderStyle.DOUBLE) {
                style = "double";
                width = THICK_WIDTH_HMM;
            }
        }
        if(borderStyle!=STBorderStyle.NONE) {
            jsonBorder.put("width", width);
        }
        jsonBorder.put("style", style);

        // creating border color
        Commons.jsonPut(jsonBorder, "color", createColor(stylesheet, borderPr.getColor()));

        return jsonBorder;
    }

    public static void createAlignment(JSONObject jsonDest, CTCellAlignment cellAlignment)
        throws JSONException {

        if(cellAlignment!=null) {
            STHorizontalAlignment horzAlignment = cellAlignment.getHorizontal();
            if(horzAlignment!=null) {
                String hAlign;
                switch(horzAlignment) {
                    case LEFT: hAlign = "left"; break;
                    case CENTER_CONTINUOUS:             // !!PASSTHROUGH INTENDED
                    case CENTER : hAlign = "center"; break;
                    case RIGHT : hAlign = "right"; break;
                    case JUSTIFY : hAlign = "justify"; break;
                    default: hAlign = "auto"; break;
                }
                jsonDest.put("alignHor", hAlign);
            }

            STVerticalAlignment vertAlignment = cellAlignment.getVertical();
            if(vertAlignment!=null) {
                String vAlign;
                switch(vertAlignment) {
                    case TOP: vAlign = "top"; break;
                    case CENTER : vAlign = "middle"; break;
                    case JUSTIFY : vAlign = "justify"; break;
                    default: vAlign = "bottom"; break;
                }
                jsonDest.put("alignVert", vAlign);
            }
            Boolean wrapText = cellAlignment.isWrapText();
            if(wrapText!=null) {
                jsonDest.put("wrapText", wrapText.booleanValue());
            }
        }
    }

    public static void createProtection(JSONObject jsonDest, CTCellProtection cellProtection)
        throws JSONException {

        if(cellProtection!=null) {
            if(cellProtection.isHidden()!=null && cellProtection.isHidden().booleanValue()) {
                jsonDest.put("hidden", true);
            }
            if(cellProtection.isLocked()!=null&&!cellProtection.isLocked().booleanValue()) {
                jsonDest.put("unlocked", true);
            }
        }
        else
            jsonDest.put("unlocked", false);
    }

    public static byte[] themeColorToBytes(ThemePart themePart, String themeColor) {
        byte[] rgbValue = null;
        if(themePart!=null) {
        	final Theme theme = themePart.getJaxbElement();
        	if(theme!=null) {
	            BaseStyles themeElements = theme.getThemeElements();
	            if(themeElements!=null) {
	                CTColorScheme clrScheme = themeElements.getClrScheme();
	                if(clrScheme!=null) {
	                    org.docx4j.dml.CTColor ctColor = null;
	                    if(themeColor.equals("light1")) {
	                        ctColor = clrScheme.getLt1();
	                    }
	                    else if(themeColor.equals("dark1")) {
	                        ctColor = clrScheme.getDk1();
	                    }
	                    else if(themeColor.equals("light2")) {
	                        ctColor = clrScheme.getLt1();
	                    }
	                    else if(themeColor.equals("dark2")) {
	                        ctColor = clrScheme.getDk2();
	                    }
	                    else if(themeColor.equals("accent1")) {
	                        ctColor = clrScheme.getAccent1();
	                    }
	                    else if(themeColor.equals("accent2")) {
	                        ctColor = clrScheme.getAccent2();
	                    }
	                    else if(themeColor.equals("accent3")) {
	                        ctColor = clrScheme.getAccent3();
	                    }
	                    else if(themeColor.equals("accent4")) {
	                        ctColor = clrScheme.getAccent4();
	                    }
	                    else if(themeColor.equals("accent5")) {
	                        ctColor = clrScheme.getAccent5();
	                    }
	                    else if(themeColor.equals("accent6")) {
	                        ctColor = clrScheme.getAccent6();
	                    }
	                    else if(themeColor.equals("hyperlink")) {
	                        ctColor = clrScheme.getHlink();
	                    }
	                    else if(themeColor.equals("followedHyperlink")) {
	                        ctColor = clrScheme.getFolHlink();
	                    }
	                    else if(themeColor.equals("text1")) {
	                        ctColor = clrScheme.getDk1();
	                    }
	                    else if(themeColor.equals("text2")) {
	                        ctColor = clrScheme.getDk2();
	                    }
	                    else if(themeColor.equals("background1")) {
	                        ctColor = clrScheme.getLt1();
	                    }
	                    else if(themeColor.equals("background2")) {
	                        ctColor = clrScheme.getLt2();
	                    }
	                    if(ctColor!=null) {
	                        rgbValue = Commons.ctColorToBytes(ctColor);
	                    }
	                }
	            }
            }
        }
        return rgbValue;
    }

    public static void applyColor(XlsxOperationDocument operationDocument, CTColor color, JSONObject jsonColor)
        throws FilterException, JSONException {

        if(color!=null&&jsonColor.has("type")) {
            String type = jsonColor.getString("type");
            color.setTint(null);
            if(type.equals("rgb")&&jsonColor.has("value")) {
                color.setRgb(Commons.hexStringToBytes(jsonColor.getString("value")));
                color.setTheme(null);
            } else if (type.equals("scheme")&&jsonColor.has("value")) {
                String themeColor = jsonColor.getString("value");

                long themeColorIndex = 0;
                for(String c:themeColors) {
                    if(c.equals(themeColor)) {
                        break;
                    }
                    themeColorIndex++;
                }
                if(themeColorIndex >= themeColors.length) {
                    if(themeColor.equals("text1")) {
                        themeColorIndex = 1;
                    }
                    else if(themeColor.equals("text2")) {
                        themeColorIndex = 3;
                    }
                    else if(themeColor.equals("background1")) {
                        themeColorIndex = 0;
                    }
                    else if(themeColor.equals("background2")) {
                        themeColorIndex = 2;
                    }
                    else {
                        themeColorIndex = 1;
                    }
                }
                color.setTheme(themeColorIndex);
                if (jsonColor.has("transformations")) {
                    JSONArray transformations = jsonColor.getJSONArray("transformations");
                    addTransformationsToColor(color, transformations);
                }
                color.setRgb(themeColorToBytes(operationDocument.getThemePart(true), themeColor));
            } else if (type.equals("auto")) {
                color.setAuto(true);
                color.setTheme(null);
            }
        }
    }

    private static void addTransformationsToColor(CTColor color, JSONArray transformations) {
        if ((transformations != null) && (transformations.length() > 0)) {
            Object temp = transformations.opt(0);
            if (temp instanceof JSONObject) {
                JSONObject transformation = (JSONObject)temp;
                String type = transformation.optString("type");
                Double value = transformation.optDouble("value");

                if ((type != null) && (value != null)) {
                    if (type.equals("tint")) {
                        color.setTint((100000.0-value) / 100000.0);
                    }
                    else if (type.equals("shade")) {
                        color.setTint((-(100000.0-value)) / 100000.0);
                    }
                }
            }
        }
    }

    public static void createColumnAttributeRanges(Cols cols, long start, long size) {
        List<Col> colList = cols.getCol();

        int i=0;
        for(;i<colList.size();i++) {
            Col col = colList.get(i);

            if(col.getMax()<start) {
                continue;
            }
            if(col.getMin()>((start+size)-1))
                break;
            long colLength = col.getMax() - col.getMin() + 1;
            if(col.getMin()>start) {
                Col newCol = Context.getsmlObjectFactory().createCol();
                newCol.setMin(start);
                long newEnd = start + size - 1;
                if(newEnd>=col.getMin()) {
                    newEnd = col.getMin() - 1;
                }
                newCol.setMax(newEnd);
                start = newEnd + 1;
                size -= (newCol.getMax() - newCol.getMin()) + 1;
                colList.add(i, newCol);
            }
            else if(col.getMin()==start) {
                if(colLength<size) {
                    start += colLength;
                    size -= colLength;
                }
                else if(colLength>size) {
                    final Col newCol = col.clone();
                    col.setMax(col.getMin()+size-1);
                    newCol.setMin(col.getMax()+1);
                    colList.add(i+1, newCol);
                    size = 0;
                    break;
                }
                else {
                    size = 0;
                    break;
                }
            }
            // colMin() < start
            else if(col.getMax()>=start) {
                final Col newCol = col.clone();
                newCol.setMin(start);
                newCol.setMax(col.getMax());
                col.setMax(start-1);
                colList.add(i+1, newCol);
            }
        }
        if(size>0) {
            Col newCol = Context.getsmlObjectFactory().createCol();
            newCol.setMin(start);
            newCol.setMax((start+size)-1);
            colList.add(i, newCol);
        }
    }

    public static void applySqrefRangeFromJsonArray(List<String> sqrefList, Object value)
    	throws JSONException {

    	if(value instanceof JSONArray) {
            final JSONArray ranges = (JSONArray)value;
            sqrefList.clear();
            for(int i=0; i<ranges.length();i++) {
                final JSONObject range = ranges.getJSONObject(i);
                final JSONArray start = range.getJSONArray("start");
                final JSONArray end = range.optJSONArray("end");
                final SmlUtils.CellRefRange cellRefRange = new SmlUtils.CellRefRange(
                		new SmlUtils.CellRef(start.getInt(0), start.getInt(1)),
                		end!=null ? new SmlUtils.CellRef(end.getInt(0), end.getInt(1)) : new SmlUtils.CellRef(start.getInt(0), start.getInt(1)));
                sqrefList.add(SmlUtils.getCellRefRange(cellRefRange));
            }
    	}
    	else {
    		sqrefList.clear();
    	}
    }

    public static JSONArray createJsonArrayFromCellRefRange(List<SmlUtils.CellRefRange> rangeList)
    	throws JSONException {

    	final JSONArray ranges = new JSONArray();
        for(SmlUtils.CellRefRange cellRefRange:rangeList) {
            final JSONObject range = new JSONObject(2);
            range.put("start", cellRefRange.getStart().getJSONArray());
            if((cellRefRange.getEnd().getColumn()!=cellRefRange.getStart().getColumn())||(cellRefRange.getEnd().getRow()!=cellRefRange.getStart().getRow())) {
                range.put("end", cellRefRange.getEnd().getJSONArray());
            }
            ranges.put(range);
        }
        return ranges;
    }

    public static List<SmlUtils.CellRefRange> createCellRefRangeListFromSqrefList(List<String> sqrefList) {
		final List<SmlUtils.CellRefRange> cellRefRangeList = new ArrayList<SmlUtils.CellRefRange>();
	    for(String sqRef:sqrefList) {
	        final SmlUtils.CellRefRange cellRefRange = SmlUtils.createCellRefRange(sqRef);
	        if(cellRefRange!=null) {
	            cellRefRangeList.add(cellRefRange);
	        }
	    }
	    return cellRefRangeList;
    }

	private static Pattern autoStyleIdPattern = Pattern.compile("a\\d+");

    /**
     * Parses the passed identifier of an auto-style, and returns the array index of the
     * corresponding cell XF element.
     * 
     * @param styleId
     * 	The identifier of an auto-style, as used in document operations. MUST NOT be null!
     *  Can be the empty string to refer to the default auto-style "a0".
     * 
     * @return
     * 	The zero-based index of the specified auto-style (e.g. the number 3 for the style
     * 	identifier "a3"); or null, if the passed string is not a valid auto-style identifier.
     */
    public static Long parseAutoStyleId(String styleId) {
    	if (styleId.length() == 0) { return 0L; }
    	Matcher matcher = autoStyleIdPattern.matcher(styleId);
    	return matcher.matches() ? Long.parseLong(styleId.substring(1)) : null;
    }
}
