/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odt.dom.components;

import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.text.TextHElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.DLList;
import com.openexchange.office.odf.DLNode;
import com.openexchange.office.odf.DrawingType;
import com.openexchange.office.odf.INodeAccessor;
import com.openexchange.office.odf.IParagraph;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.Styles;
import com.openexchange.office.odt.dom.Annotation;
import com.openexchange.office.odt.dom.AnnotationEnd;
import com.openexchange.office.odt.dom.DrawFrame;
import com.openexchange.office.odt.dom.DrawImage;
import com.openexchange.office.odt.dom.Paragraph;
import com.openexchange.office.odt.dom.Shape;
import com.openexchange.office.odt.dom.Text;
import com.openexchange.office.odt.dom.TextField;
import com.openexchange.office.odt.dom.TextLineBreak;
import com.openexchange.office.odt.dom.TextList;
import com.openexchange.office.odt.dom.TextListItem;
import com.openexchange.office.odt.dom.TextSpan;
import com.openexchange.office.odt.dom.TextTab;

public class ParagraphComponent extends Component implements IParagraph {

	public ParagraphComponent(ComponentContext parentContext, DLNode<Object> paragraphNode, int componentNumber) {
		super(parentContext, paragraphNode, componentNumber);
	}

	@Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {

        DLNode<Object> nextNode = previousChildContext != null ? previousChildContext.getNode().next : ((Paragraph)getObject()).getContent().getFirstNode();
        while(nextNode!=null) {
			final Object child = nextNode.getObject();
			if(child instanceof TextSpan) {
				final TextSpanContext textSpanContext =  new TextSpanContext(this, nextNode);
				return textSpanContext.getNextChildComponent(null, previousChildComponent);
			}
			nextNode = nextNode.next;
		}
		return null;
	}

	@Override
	public void applyAttrsFromJSON(OdfOperationDocument operationDocument, JSONObject attrs)
		throws JSONException, SAXException {

		final Paragraph paragraph = (Paragraph)getObject();
		if(attrs!=null) {

			final JSONObject paragraphAttrs = attrs.optJSONObject("paragraph");
			if(paragraphAttrs!=null) {
				final Object listLevel = paragraphAttrs.opt("listLevel");
				if(listLevel!=null) {
					if(listLevel instanceof Integer) {
						paragraph.setListLevel(((Integer)listLevel).intValue());
					}
					else if(listLevel==JSONObject.NULL) {
						paragraph.setListLevel(-1);
					}
				}
				final Object listStyleId = paragraphAttrs.opt("listStyleId");
				if(listStyleId!=null) {
					if(listStyleId instanceof String) {
						final TextListItem textListItem = paragraph.getTextListItem();
						if(textListItem!=null) {
							textListItem.getParentTextList().setStyleName((String)listStyleId);
						}
					}
					else if(listStyleId==JSONObject.NULL) {
						paragraph.setListLevel(-1);
					}
				}
			}

			// TODO: big fake ... to be able to use the old "addStyle" functionality
			// we temporarily create a PElement to provide the actual style
			final OdfFileDom ownerDocument = operationDocument.getOwnerDocument();
			final OdfStylableElement e = paragraph.isHeader() ? new TextHElement(ownerDocument) : new TextPElement(ownerDocument);
			e.setAttributeNS(Namespaces.TEXT, "text:style-name", paragraph.getStyleName());
			Styles.addStyle(attrs, e, ownerDocument);
			paragraph.setStyleName(e.getAttributeNS(Namespaces.TEXT, "style-name"));
		}
	}

	@Override
	public JSONObject createJSONAttrs(OdfOperationDocument operationDocument, JSONObject attrs)
		throws JSONException, SAXException {

		if(attrs==null) {
			attrs = new JSONObject();
		}
		final Paragraph paragraph = (Paragraph)getObject();
		if(paragraph.getStyleName()!=null&&!paragraph.getStyleName().isEmpty()) {
			attrs.put("styleId", paragraph.getStyleName());
		}

		final JSONObject paragraphAttributes = new JSONObject();
		TextListItem textListItem = paragraph.getTextListItem();
		if(textListItem!=null) {
			// TODO: listLabelHidden ?
			final String listId =  textListItem.getParentTextList().getId();
			if(listId!=null&&!listId.isEmpty()) {
				paragraphAttributes.put("listId", listId);
			}
			paragraphAttributes.put("listLevel", textListItem.getListLevel());
			if(paragraph.getStartValue()!=null) {
				paragraphAttributes.put("listStartValue", paragraph.getStartValue());
			}
			if(paragraph.getOutlineLevel()!=null) {
				paragraphAttributes.put("outlineLevel", paragraph.getOutlineLevel());
			}

			String listStyleId = null;
			do {
				final TextList textList = textListItem.getParentTextList();
				if(textList.getStyleName()!=null&&!textList.getStyleName().isEmpty()) {
					listStyleId = textList.getStyleName();
					break;
				}
				textListItem = textList.getParentTextListItem();
			}
			while(textListItem!=null);

			if(listStyleId==null&&paragraph.getStyleName()!=null) {
				final Styles styles = (Styles)operationDocument.getDocument().getStylesDom();
				OdfStyle odfStyle = styles.getAutomaticStyles().getStyle(paragraph.getStyleName(), OdfStyleFamily.Paragraph);
				if(odfStyle==null) {
					odfStyle = styles.getOfficeStyles().getStyle(paragraph.getStyleName(), OdfStyleFamily.Paragraph);
				}
				if(odfStyle!=null) {
					listStyleId = odfStyle.getStyleListStyleNameAttribute();
				}
			}
			if(listStyleId!=null) {
				paragraphAttributes.put("listStyleId", listStyleId);
			}
		}
		if(!paragraphAttributes.isEmpty()) {
			attrs.put("paragraph", paragraphAttributes);
		}
		return attrs;
	}

	@Override
	public void insertText(OdfOperationDocument operationDocument, int textPosition, String text, JSONObject attrs)
		throws JSONException, SAXException {

    	if(text.length()>0) {
        	final Paragraph paragraph = (Paragraph)getObject();

            Text t = null;
            Component childComponent = getNextChildComponent(null, null);
            Component cRet = null;

            if(childComponent!=null) {
            	if(textPosition>0) {
            		childComponent = childComponent.getComponent(textPosition-1);
            	}
                // check if the character could be inserted into an existing text:
                if(childComponent instanceof TextComponent) {
                    t = (Text)childComponent.getObject();
                    final StringBuffer s = new StringBuffer(t.getText());
                    s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                    t.setText(s.toString());
                    cRet = childComponent;
                }
                else {
                	t = new Text(text);
                	final TextSpanContext spanContext = (TextSpanContext)childComponent.getParentContext();
                    ((TextSpan)spanContext.getObject()).getContent().addNode(childComponent.getNode(), new DLNode<Object>(t), textPosition==0);
                    cRet = childComponent;

                    if(textPosition>0) {
                        cRet = childComponent.getNextComponent();
                    }
                    else {
                        cRet = getNextChildComponent(null, null);
                    }
                }
            }
            else {

            	// the paragraph is empty, we have to create R and its text
                final TextSpan newRun = new TextSpan();
                paragraph.getContent().add(newRun);
/*
                if(paragraph.getPPr()!=null) {
                    final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                    if(rPr!=null) {
                        rPr.setParent(newRun);
                        newRun.setRPr(rPr);
                    }
                }
*/
                t = new Text(text);
                newRun.getContent().add(t);
                cRet = getNextChildComponent(null, null);
            }
            if(attrs!=null) {
            	cRet.splitStart(textPosition);
            	cRet.splitEnd(textPosition+text.length()-1);
            	cRet.applyAttrsFromJSON(operationDocument, attrs);
            }
        }
	}

    @Override
    public Component insertChildComponent(OdfOperationDocument operationDocument, int textPosition, JSONObject attrs, Type childType)
        throws JSONException, SAXException {

        Object newChild = null;
        switch(childType) {
            case TAB : {
                newChild = new TextTab(null);
                break;
            }
            case HARDBREAK : {
            	newChild = new TextLineBreak();
            	break;
            }
            case FIELD : {
            	newChild = new TextField();
            	break;
            }
            case COMMENT_REFERENCE : {
            	newChild = new Annotation(null);
            	break;
            }
            case COMMENT_RANGE_END : {
            	newChild = new AnnotationEnd(null);
            	break;
            }
            case AC_IMAGE : {
            	newChild = new DrawFrame(new DrawImage());
            	break;
            }
            case AC_SHAPE : {
            	// TODO: we should take care of text-boxes when inserting, a new DrawingType is required .... new DrawFrame(new DrawTextBox());
            	newChild = new Shape(DrawingType.SHAPE, Namespaces.DRAW, "rect", "draw:rect");
            	break;
            }
            case AC_GROUP : {
            	newChild = new Shape(DrawingType.GROUP, Namespaces.DRAW, "g", "draw:g");
            	break;
            }
            default : {
                throw new UnsupportedOperationException();
            }
        }
        final Paragraph paragraph = (Paragraph)getObject();
        final DLList<Object> paragraphContent = paragraph.getContent();

        Component childComponent = getNextChildComponent(null, null);
        if(childComponent!=null) {
            if(textPosition>0) {
                childComponent = childComponent.getComponent(textPosition-1);
                childComponent.splitEnd(textPosition-1);
            }
        	if(childComponent instanceof TextSpan_Base) {

        		// the new child can be added into an existing textRun
                final TextSpan textSpan = ((TextSpan_Base)childComponent).getTextSpan();
                textSpan.getContent().addNode(((TextSpan_Base)childComponent).getNode(), new DLNode<Object>(newChild), textPosition==0);
            }
            else {

            	// we need to create a new textRun, check if we can get referenceAttributes that we will clone for the new run. this is
            	final TextSpan newTextSpan = new TextSpan();
            	newTextSpan.getContent().add(newChild);

            	final INodeAccessor parentContextObject = (INodeAccessor)childComponent.getParentContext().getObject();
            	((DLList<Object>)parentContextObject.getContent()).addNode(childComponent.getNode(), new DLNode<Object>(newTextSpan), textPosition==0);
            }
        }
        else {
            final TextSpan newTextSpan = new TextSpan();
            paragraphContent.add(newTextSpan);
            newTextSpan.getContent().add(newChild);
        }
        if(textPosition>0) {
            childComponent = childComponent.getNextComponent();
        }
        else {
            childComponent = getNextChildComponent(null, null);
        }
        if(attrs!=null) {
            childComponent.splitStart(textPosition);
            childComponent.splitEnd(textPosition);
            childComponent.applyAttrsFromJSON(operationDocument, attrs);
        }
        return childComponent;
    }

	@Override
    public void splitParagraph(int textPosition) {
    	final Paragraph paragraph = (Paragraph)getObject();
        final Paragraph destParagraph = new Paragraph(
        	paragraph.getTextListItem()!=null
        	? new TextListItem(paragraph.getTextListItem().getParentTextList())
        	: null);
        final DLNode<Object> destParagraphNode = new DLNode<Object>(destParagraph);
        ((INodeAccessor)getParentContext().getObject()).getContent().addNode(getNode(), destParagraphNode, false);
        destParagraph.setStyleName(paragraph.getStyleName());

        Component component = getNextChildComponent(null, null);
    	while(component!=null&&component.getNextComponentNumber()<=textPosition) {
    		component = component.getNextComponent();
    	}
        if(component!=null) {
        	component.splitStart(textPosition);

        	// moving text spans into the new paragraph
        	paragraph.getContent().moveNodes(component.getParentContext().getNode(), -1, destParagraph.getContent(), null, true);
        }
    }

	@Override
	public void mergeParagraph() {
    	final Component nextParagraphComponent = getNextComponent();
    	if(nextParagraphComponent instanceof ParagraphComponent) {
        	final Paragraph paragraph = (Paragraph)getObject();
        	final DLList<Object> sourceContent = ((Paragraph)nextParagraphComponent.getObject()).getContent();
            paragraph.getContent().mergeNodes(sourceContent);
            ((INodeAccessor)getParentContext().getObject()).getContent().removeNode(nextParagraphComponent.getNode());
    	}
	}
}
