/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.odf;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.Names;
import org.odftoolkit.odfdom.doc.OdfDocument;
import org.odftoolkit.odfdom.dom.OdfContentDom;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.OdfSchemaConstraint;
import org.odftoolkit.odfdom.dom.OdfSchemaDocument;
import org.odftoolkit.odfdom.dom.OdfStylesDom;
import org.odftoolkit.odfdom.dom.attribute.draw.DrawStyleNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.style.StyleCellProtectAttribute;
import org.odftoolkit.odfdom.dom.attribute.style.StyleRunThroughAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableDefaultCellStyleNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.table.TableStyleNameAttribute;
import org.odftoolkit.odfdom.dom.attribute.text.TextStyleNameAttribute;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.OdfStyleBase;
import org.odftoolkit.odfdom.dom.element.OdfStylePropertiesBase;
import org.odftoolkit.odfdom.dom.element.draw.DrawFrameElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeFontFaceDeclsElement;
import org.odftoolkit.odfdom.dom.element.style.StyleChartPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleFontFaceElement;
import org.odftoolkit.odfdom.dom.element.style.StyleGraphicPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleHeaderFooterPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleListLevelLabelAlignmentElement;
import org.odftoolkit.odfdom.dom.element.style.StyleListLevelPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StylePageLayoutPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleParagraphPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleRubyPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleSectionPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleStyleElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTabStopElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTabStopsElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableCellPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableColumnPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTablePropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTableRowPropertiesElement;
import org.odftoolkit.odfdom.dom.element.style.StyleTextPropertiesElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableColumnElement;
import org.odftoolkit.odfdom.dom.element.table.TableTableElement;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleBulletElement;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleElementBase;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleImageElement;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleNumberElement;
import org.odftoolkit.odfdom.dom.element.text.TextListStyleElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.dom.style.props.OdfStylePropertiesSet;
import org.odftoolkit.odfdom.incubator.doc.office.OdfOfficeStyles;
import org.odftoolkit.odfdom.incubator.doc.office.OdfStylesBase;
import org.odftoolkit.odfdom.incubator.doc.style.OdfDefaultStyle;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.odftoolkit.odfdom.pkg.OdfValidationException;
import org.odftoolkit.odfdom.type.Length;
import org.odftoolkit.odfdom.type.StyleName;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;

import com.openexchange.office.ods.dom.MapHelper;

@SuppressWarnings("serial")
public abstract class Styles extends OdfStylesDom {

	private OfficeStyles officeStyles;

    private Map<String, Boolean> knownStyles = new HashMap<String, Boolean>();
    private Set<String> knownListStyles = new HashSet<String>();

	public Styles(OdfSchemaDocument odfDocument, String packagePath) throws SAXException {
		super(odfDocument, packagePath);
		// TODO Auto-generated constructor stub
	}

	public OfficeStyles getStyles() {
		if(officeStyles==null) {
			officeStyles = new OfficeStyles();
		}
		return officeStyles;
	}

	/**
	 * Retrieves the Odf Document
	 * 
	 * @return The <code>OdfDocument</code>
	 */
	@Override
	public OdfDocument getDocument() {
		return (OdfDocument)mPackageDocument;
	}

	public static Map<String, Object> getAutomaticStyleHierarchyProps(String styleName, String styleFamily, OfficeStyles officeStyles, Map<String, Object> allHardFormatting)
    	throws JSONException {

    	if(styleName==null) {
    		return null;
    	}
    	// Hard formatted properties (automatic styles)
        Map<String, Map<String, String>> allOdfProps = null;
        // AUTOMATIC STYLE HANDLING
    	final OfficeStyleBase officeStyleBase = officeStyles.getAutomaticStyle(styleFamily, styleName);
    	if(officeStyleBase!=null) {
            OdfStyleBase style = officeStyleBase.getOdfStyleBase();

            // all ODF properties
            allOdfProps = new HashMap<String, Map<String, String>>();
            List<OdfStyleBase> parents = new LinkedList<OdfStyleBase>();
            parents.add(style);
            OdfStyleBase parent = style.getParentStyle();
            // if automatic style inheritance is possible
            while (parent != null) {
                Node n = parent.getParentNode();
                // if it is no longer an automatic style (template or default style)
                if (n instanceof OdfOfficeStyles) {
                    break;
                }
                parents.add(parent);
                parent = parent.getParentStyle();
            }
            // due to inheritance the top ancestor style have to be propagated first
            for (int i = parents.size() - 1; i >= 0; i--) {
                OdfStyleBase styleBase = parents.get(i);
                Map<String, OdfStylePropertiesSet> familyPropertyGroups = getAllOxStyleGroupingIdProperties(styleFamily);
                MapHelper.getStyleProperties(styleBase, familyPropertyGroups, allOdfProps);
            }
            Map<String, OdfStylePropertiesSet> familyPropertyGroups = getAllOxStyleGroupingIdProperties(styleFamily);
            allHardFormatting = MapHelper.mapStyleProperties(familyPropertyGroups, allHardFormatting, allOdfProps);
        }
        return allHardFormatting;
    }
	
    /**
     * At the the given newElement from the given ownerDocument the given style
     * changes from attrs are being applied. In addition automatic styles might
     * being created, as ODF does not apply hard styles directly to the element.
     *
     * @param attrs Map with style changes
     * @param newElement the element the style changes will be applied to
     * @param ownerDocument the XML file the element belongs to
     * @return the prior given newElement after all style changes had been
     * applied.
     */
    // ToDo-Clean-Up: Move this to styleable Element
    public static StyleStyleElement addStyle(JSONObject attrs, OdfStylableElement newElement, OdfFileDom ownerDocument)
    	throws JSONException, SAXException {

    	StyleStyleElement autoStyle = null;
        // temporary font properties taken into adapter
        OdfDocument doc4Fonts = ((OdfDocument) ownerDocument.getDocument());

        // if there are style changes
        if (attrs != null) {
            OdfStyleFamily styleFamily = newElement.getStyleFamily();
            boolean hasHardFormatting = hasHardProperties(attrs, styleFamily);
            // IF THERE IS A NEW TEMPLATE STYLE
            if (attrs.has("styleId") && !attrs.isNull("styleId")) {
                // IF ONLY TEMPLATE STYLE
                String styleName = attrs.optString("styleId");
                if (!hasHardFormatting) {
                    // Add template style, if there was no hard formatting it will be set directly, otherwise the template style has to reference
                    addStyleNameAttribute(newElement, styleFamily, ownerDocument, styleName);
                    // IF THERE ARE HARD FORMATTING STYLES AND TEMPLATE STYLES
                } else {
                    autoStyle = newElement.getOrCreateUnqiueAutomaticStyle();

                    // adding/removing list style reference from style
                    modifyListStyleName(attrs, autoStyle);

                    // adding the style name
                    addStyleNameAttribute(newElement, styleFamily, ownerDocument, autoStyle.getStyleNameAttribute());
                    autoStyle.setStyleParentStyleNameAttribute(styleName);
                    // APPLY HARD FORMATTING PROPERTIES OF ODF FAMILY
                    mapProperties(styleFamily, attrs, autoStyle, doc4Fonts);
                }
            } else { // IF NO NEW TEMPLATE STYLE
                // if Element has no Automatic Styles && styles will not add any --> skip
                String styleName = newElement.getStyleName();
                if (styleName != null) {
                    if (styleName.isEmpty() && !hasHardFormatting) {
                        // remove the invalid attribute
                        newElement.removeAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-name");
                    } else {
                        // if there are automatic styles adjust them
                        boolean removeTemplateStyle = attrs.has("styleId") && attrs.isNull("styleId");

                        if (newElement instanceof TableTableColumnElement) {
                            // if the template style is being removed
                            if (removeTemplateStyle) {
                                autoStyle = newElement.getOrCreateUnqiueAutomaticStyle();
                                autoStyle.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "parent-style-name");
                            }
                            if (attrs.has("column")) {
                                if (autoStyle == null) {
                                    autoStyle = newElement.getOrCreateUnqiueAutomaticStyle();
                                }
                                // APPLY HARD FORMATTING PROPERTIES OF ODF FAMILY
                                mapProperties(styleFamily, attrs, autoStyle, doc4Fonts);

                                // if the template style is being removed
                                if (removeTemplateStyle) {
                                    addStyleNameAttribute(newElement, styleFamily, ownerDocument, autoStyle.getStyleNameAttribute());
                                }
                            }
                            if (attrs.has("cell") || attrs.has("paragraph") || attrs.has("text")) {
                                autoStyle = newElement.getOrCreateUnqiueAutomaticStyle(Boolean.FALSE, OdfStyleFamily.TableCell);
                                // APPLY HARD FORMATTING PROPERTIES OF ODF FAMILY
                                mapProperties(OdfStyleFamily.TableCell, attrs, autoStyle, doc4Fonts);
                                //default-cell-style-name
                                TableDefaultCellStyleNameAttribute attr = new TableDefaultCellStyleNameAttribute(ownerDocument);
                                newElement.setOdfAttribute(attr);
                                attr.setValue(autoStyle.getStyleNameAttribute());
                            }
                        } else {
                            autoStyle = newElement.getOrCreateUnqiueAutomaticStyle();
                            // adding/removing list style reference from style
                            modifyListStyleName(attrs, autoStyle);

                            // APPLY HARD FORMATTING PROPERTIES OF ODF FAMILY
                            try {
                                JSONObject paragraphAttr = null;
                                if( newElement instanceof TextPElement && styleFamily.equals(OdfStyleFamily.Paragraph) && attrs.has("paragraph") && ( (paragraphAttr = attrs.getJSONObject("paragraph")).has("pageBreakBefore")  || paragraphAttr.has("pageBreakAfter"))){
                                    //search for the top table element, stop at top level and at frames
                                    TableTableElement tableElement = null;
                                    Node parent = newElement.getParentNode();
                                    while(parent != null && !(parent instanceof DrawFrameElement)){
                                        if(parent instanceof TableTableElement) {
                                            tableElement = (TableTableElement)parent;
                                        }
                                        parent = parent.getParentNode();
                                    }
                                    if(tableElement != null) {
                                        OdfStyle tableStyle = tableElement.getAutomaticStyle();
                                        OdfStylePropertiesBase propsElement = tableStyle.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableProperties);
                                        JSONObject paraAttrs = attrs.getJSONObject("paragraph");
                                        boolean isBefore = paragraphAttr.has("pageBreakBefore");
                                        String attrName = isBefore ? "pageBreakBefore" : "pageBreakAfter";
                                        boolean breakValue = !paraAttrs.isNull(attrName) && paraAttrs.getBoolean(attrName);
                                        paraAttrs.remove(attrName);
                                        if(breakValue) {
                                            propsElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), isBefore ? "fo:break-before" : "fo:break-after", "page");
                                            propsElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), isBefore ? "break-after" : "break-before");
                                        } else {
                                            propsElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), isBefore ? "break-before" : "break-after");
                                        }
                                    }
                                }
                            } catch (JSONException e) {
                                //no handling required
                            }
                            mapProperties(styleFamily, attrs, autoStyle, doc4Fonts);
                            // if the template style is being removed
                            if (removeTemplateStyle) {
                                autoStyle.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "parent-style-name");
                                addStyleNameAttribute(newElement, styleFamily, ownerDocument, autoStyle.getStyleNameAttribute());
                            }
                        }
                    }
                }
            }
            // apply numberFormat - uses exisiting 'Number' style or create a new one
            JSONObject cellAttrs = attrs.optJSONObject("cell");
            if (cellAttrs!=null) {
                String formatCode = cellAttrs.optString("formatCode");
                if(!formatCode.isEmpty()) {
                    String dataStyleName = MapHelper.findOrCreateDataStyle(formatCode, cellAttrs.optLong("formatId", -1), ownerDocument);
                    autoStyle.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:data-style-name", dataStyleName);
                } else {
                    autoStyle.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "data-style-name");
                }
            }
        }
        return autoStyle;
    }

	public void addListStyle(JSONArray operationQueue, String styleId)
		throws JSONException {

		if (styleId != null & !styleId.isEmpty()) {
            if (!knownListStyles.contains(styleId)) {
            	knownListStyles.add(styleId);

            	final OfficeStyles styles = getStyles();
            	final OfficeTextListStyle textListStyle = (OfficeTextListStyle)styles.getStyle("list-style", styleId);
            	if(textListStyle!=null) {
            		final OdfStyleBase odfStyleBase = textListStyle.getOdfStyleBase();
            		if(odfStyleBase instanceof TextListStyleElement) {
            			final TextListStyleElement textListStyleElement = (TextListStyleElement)odfStyleBase;
                        insertListStyle(operationQueue, textListStyleElement.getStyleNameAttribute(), textListStyleElement.getTextConsecutiveNumberingAttribute(), getListLevelDefinitions(textListStyleElement));
            		}
            	}
            }
        }
	}

    public void createInsertStyleOperations(JSONArray operationQueue, OdfStylesBase stylesBase, boolean autoStyle, boolean isTextDocument)
    	throws JSONException {

    	if (stylesBase!=null) {

    		if(isTextDocument) {
	        	for (OdfStyle style : stylesBase.getStylesForFamily(OdfStyleFamily.Paragraph)) {
	            	triggerStyleHierarchyOps(operationQueue, stylesBase, OdfStyleFamily.Paragraph, style, autoStyle, isTextDocument);
	            }
	            for (OdfStyle style : stylesBase.getStylesForFamily(OdfStyleFamily.Text)) {
	                triggerStyleHierarchyOps(operationQueue, stylesBase, OdfStyleFamily.Text, style, autoStyle, isTextDocument);
	            }
	            for (OdfStyle style : stylesBase.getStylesForFamily(OdfStyleFamily.Graphic)) {
	                triggerStyleHierarchyOps(operationQueue, stylesBase, OdfStyleFamily.Graphic, style, autoStyle, isTextDocument);
	            }
	            //always generate graphic default style
	            if(stylesBase instanceof OdfOfficeStyles) {
	            	triggerDefaultStyleOp(operationQueue, OdfStyleFamily.Graphic, ((OdfOfficeStyles)stylesBase).getDefaultStyle(OdfStyleFamily.Graphic));
	            }
    		}

//	        		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.Table)){
//	        			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.Table, style);
//	        		}
//	        		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.TableRow)){
//	        			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.TableRow, style);
//	        		}
//	        		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.TableColumn)){
//	        			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.TableColumn, style);
//	        		}

    		if(!isTextDocument) {
	            for (OdfStyle style : stylesBase.getStylesForFamily(OdfStyleFamily.TableCell)) {
	                triggerStyleHierarchyOps(operationQueue, stylesBase, OdfStyleFamily.TableCell, style, autoStyle, isTextDocument);
	            }
    		}

//	        		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.Section)){
//	        			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.Section, style);
//	        		}
//	        		for(OdfStyle style : officeStyles.getStylesForFamily(OdfStyleFamily.List)){
//	        			mJsonOperationProducer.triggerStyleHierarchyOps(officeStyles, OdfStyleFamily.List, style);
//	        		}
        }
    }
	
    /**
     * Modifying the @style:list-style-name to the style:style of (an automatic)
     * style, when the paragraph properties contain a "listStyleId
     */
    private static void modifyListStyleName(JSONObject attrs, OdfStyleBase autoStyle) {

        String listStyleName;
        if (attrs.has("paragraph")) {
            JSONObject paraProps = attrs.optJSONObject("paragraph");
            listStyleName = paraProps.optString("listStyleId");
            if (listStyleName == null) {
                autoStyle.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "list-style-name");
            } else if (!listStyleName.isEmpty()) {
                autoStyle.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:list-style-name", listStyleName);
            }
        }
    }

    private static void addStyleNameAttribute(OdfElement newElement, OdfStyleFamily styleFamily, OdfFileDom ownerDocument, String styleName) {
        // Add template style, if there was no hard formatting it will be set directly, otherwise the template style has to reference
        String attributeName = null;
        if (styleFamily.getName().equals("paragraph") || styleFamily.getName().equals("text")) {
            TextStyleNameAttribute attr = new TextStyleNameAttribute(ownerDocument);
            newElement.setOdfAttribute(attr);
            attr.setValue(styleName);
            attributeName = "text:style-name";
        } else if (styleFamily.getName().equals("table") || styleFamily.getName().equals("table-column") || styleFamily.getName().equals("table-row") || styleFamily.getName().equals("table-cell")) {
            TableStyleNameAttribute attr = new TableStyleNameAttribute(ownerDocument);
            newElement.setOdfAttribute(attr);
            attr.setValue(styleName);
            attributeName = "table:style-name";
        } else if (styleFamily.getName().equals("graphic")) {
            DrawStyleNameAttribute attr = new DrawStyleNameAttribute(ownerDocument);
            newElement.setOdfAttribute(attr);
            attr.setValue(styleName);
            attributeName = "draw:style-name";
        }
        ErrorHandler errorHandler = ownerDocument.getDocument().getPackage().getErrorHandler();
        if (errorHandler != null && attributeName != null) {
            // Is String from type NCName? == http://www.w3.org/TR/xmlschema-2/#NCName
            if (!StyleName.isValid(styleName)) {
                try {
                    errorHandler.error(new OdfValidationException(OdfSchemaConstraint.DOCUMENT_XML_INVALID_ATTRIBUTE_VALUE, styleName, attributeName));
                } catch (SAXException ex) {
                    Logger.getLogger(StyleStyleElement.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
        }
    }

    /**
     * Analyze the attrs if there is any new style properties given or only
     * existing should be removed
     */
    private static boolean hasHardProperties(JSONObject attrs, OdfStyleFamily styleFamily) {
        int attrsLength = attrs.length();
        if (attrs.has("styleId") && !attrs.isNull("styleId")) {
            attrsLength--;
        }
        if (attrs.has("changes")) {
            attrsLength--;
        }
        Map<String, OdfStylePropertiesSet> elementProps = getAllOxStyleGroupingIdProperties(styleFamily);
        for (String propertyId : elementProps.keySet()) {
            if (attrs.has(propertyId)) {
                JSONObject newProps = attrs.optJSONObject(propertyId);
                if (newProps != null) {
                    int propsLength = newProps.length();
                    Iterator<String> keys = newProps.keys();
                    String key;
                    while (keys.hasNext()) {
                        key = keys.next();
                        // if there is a deletion of a hard formatting or an URL, which is not being used for <text:span> but an own element (<text:a>)
                        if (newProps.has(key) && newProps.isNull(key) || key.equals("url")) {
                            propsLength--;
                        } else {
                            break;
                        }
                    }
                    if (propsLength == 0) {
                        attrsLength--;
                    } else {
                        break;
                    }
                } else {
                    // remove key, when value is null
                    attrs.remove(propertyId);
                }
            }
        }
        return attrsLength != 0;
    }

    private static JSONObject getListLevelDefinitions(TextListStyleElement listStyle)
    	throws JSONException {

    	JSONObject listDefinition = new JSONObject(9);
        NodeList listStyleChildren = listStyle.getChildNodes();
        int size = listStyleChildren.getLength();
        for (int i = 0; i < size; i++) {
            Node child = listStyleChildren.item(i);
            if (!(child instanceof Element)) {
                // avoid line breaks, when XML is indented
                continue;
            } else {
                TextListLevelStyleElementBase listLevelStyle = (TextListLevelStyleElementBase) child;
                // Transform mandatory attribute to integer

                String textLevel = listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "level");
                int listLevel = Integer.parseInt(textLevel) - 1;
                listDefinition.put("listLevel" + listLevel, createListLevelDefinition(listLevelStyle, listLevel));
            }
        }
        return listDefinition;
    }

    private static JSONObject createListLevelDefinition(TextListLevelStyleElementBase listLevelStyle, int listLevel) throws JSONException {
    	JSONObject listLevelDefinition = new JSONObject();

    	// NUMBERED LISTS
    	if (listLevelStyle instanceof TextListLevelStyleNumberElement) {
    		TextListLevelStyleNumberElement listLevelNumberStyle = (TextListLevelStyleNumberElement) listLevelStyle;
    		listLevelDefinition.put("levelText", getLabel(listLevelNumberStyle, listLevel));
    		if (listLevelStyle.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "start-value")) {
    			String listStartValue = getListStartValue(listLevelNumberStyle);
    			listLevelDefinition.put("listStartValue", Integer.parseInt(listStartValue));
    		}
    		// There is always the number format set
    		listLevelDefinition.put("numberFormat", getNumberFormat(listLevelNumberStyle));

    		// BULLET LISTS
    	} else if (listLevelStyle instanceof TextListLevelStyleBulletElement) {
    		TextListLevelStyleBulletElement listLevelBulletStyle = (TextListLevelStyleBulletElement) listLevelStyle;
    		listLevelDefinition.put("levelText", getLabel(listLevelBulletStyle, listLevel));
    		if (listLevelStyle.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "bullet-relative-size")) {
    			listLevelDefinition.put("bulletRelativeSize", listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "bullet-relative-size"));
    		}
    		listLevelDefinition.put("numberFormat", "bullet");

    		// IMAGE LISTS
    	} else if (listLevelStyle instanceof TextListLevelStyleImageElement) {
    		listLevelDefinition.put("levelPicBulletUri", listLevelStyle.getAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "href"));
    		listLevelDefinition.put("numberFormat", "bullet");
    	}

    	// ALL THREE TYPES: number, bullet and image list
    	if (listLevelStyle.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-name")) {
    		listLevelDefinition.put("styleId", listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "style-name"));
    	}

    	NodeList listLevelProps = listLevelStyle.getElementsByTagNameNS(OdfDocumentNamespace.STYLE.getUri(), "list-level-properties");
    	if (listLevelProps != null) {
    		StyleListLevelPropertiesElement styleListLevelProperties = (StyleListLevelPropertiesElement) listLevelProps.item(0);
    		if (styleListLevelProperties != null) {
    			//  fo:height
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.FO.getUri(), "height")) {
    				String heightValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "height");
    				if (heightValue != null) {
    					int height = MapHelper.normalizeLength(heightValue);
    					listLevelDefinition.put("height", height);
    				}
    			}

    			// fo:text-align
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-align")) {
    				listLevelDefinition.put("textAlign", MapHelper.mapFoTextAlign(styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-align")));
    			}

    			//  fo:width
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.FO.getUri(), "width")) {
    				String widthValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "width");
    				if (widthValue != null) {
    					int width = MapHelper.normalizeLength(widthValue);
    					listLevelDefinition.put("width", width);
    				}
    			}

    			// style:font-name
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name")) {
    				listLevelDefinition.put("fontName", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name"));
    			}

    			// style:vertical-pos
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-pos")) {
    				listLevelDefinition.put("verticalPos", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-pos"));
    			}

    			// style:vertical-rel
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-rel")) {
    				listLevelDefinition.put("verticalRel", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-rel"));
    			}

    			// svg:y
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y")) {
    				listLevelDefinition.put("y", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y"));
    			}

    			// text:list-level-position-and-space-mode
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-level-position-and-space-mode")) {
    				listLevelDefinition.put("listLevelPositionAndSpaceMode", styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-level-position-and-space-mode"));
    			}

    			// text:min-label-distance
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "min-label-distance")) {
    				String minLabelDistanceValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "min-label-distance");
    				if (minLabelDistanceValue != null && !minLabelDistanceValue.isEmpty()) {
    					int minLabelDistance = MapHelper.normalizeLength(minLabelDistanceValue);
    					listLevelDefinition.put("minLabelDistance", minLabelDistance);
    				}
    			}

    			// text:min-label-width
    			String minLabelWidthValue = null;
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "min-label-width")) {
    				minLabelWidthValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "min-label-width");
    				if (minLabelWidthValue != null && !minLabelWidthValue.isEmpty()) {
    					int width = MapHelper.normalizeLength(minLabelWidthValue);
    					listLevelDefinition.put("minLabelWidth", width);
    				}
    			}

    			// text:space-before
    			String spaceBeforeValue = null;
    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "space-before")) {
    				spaceBeforeValue = styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "space-before");
    				if (spaceBeforeValue != null && !spaceBeforeValue.isEmpty()) {
    					int spaceBefore = MapHelper.normalizeLength(spaceBeforeValue);
    					listLevelDefinition.put("spaceBefore", spaceBefore);
    				}
    			}

    			// Mapping list XML ODF 1.1 to ODF 1.2: Adding @text:min-label-width & @text:space-before to margin-left
    			listLevelDefinition = mapIndent(minLabelWidthValue, spaceBeforeValue, listLevelDefinition);

    			if (styleListLevelProperties.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-level-position-and-space-mode")) {
    				if ("label-alignment".equals(styleListLevelProperties.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-level-position-and-space-mode"))) {
    					NodeList nl = styleListLevelProperties.getElementsByTagNameNS(OdfDocumentNamespace.STYLE.getUri(), "list-level-label-alignment");
    					if (nl != null && nl.getLength() == 1) {
    						StyleListLevelLabelAlignmentElement labelAlignmentElement = (StyleListLevelLabelAlignmentElement) nl.item(0);
    						String marginLeft = labelAlignmentElement.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "margin-left");
    						int margin = 0;
    						if (marginLeft != null && !marginLeft.isEmpty()) {
    							margin = MapHelper.normalizeLength(marginLeft);
    							listLevelDefinition.put("indentLeft", margin);
    						} else {
    							listLevelDefinition = mapIndent(minLabelWidthValue, spaceBeforeValue, listLevelDefinition);
    						}
    						String textIndent = labelAlignmentElement.getAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-indent");
    						if (textIndent != null && !textIndent.isEmpty()) {
    							int indent = MapHelper.normalizeLength(textIndent);
    							listLevelDefinition.put("indentFirstLine", indent);
    						}
    						if (labelAlignmentElement.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-tab-stop-position")) {
    							String tabPosition = labelAlignmentElement.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "list-tab-stop-position");
    							if (tabPosition != null && !tabPosition.isEmpty()) {
                                   listLevelDefinition.put("tabStopPosition", MapHelper.normalizeLength(tabPosition));
    							}
    						}
    						if (labelAlignmentElement.hasAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "label-followed-by")) {
    							listLevelDefinition.put("labelFollowedBy", labelAlignmentElement.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "label-followed-by"));
    						}
    					}
    				}
    			}
    		}
    	}
    	return listLevelDefinition;
    }

    private static String getNumberFormat(TextListLevelStyleElementBase listLevelStyle) {
    	String numberFormat = listLevelStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "num-format");
    	String numFormat;
    	if (numberFormat == null || numberFormat.isEmpty()) {
    		numFormat = "none";
    	} else if (numberFormat.equals("1")) {
    		numFormat = "decimal";
    	} else if (numberFormat.equals("i")) {
    		numFormat = "lowerRoman";
    	} else if (numberFormat.equals("I")) {
    		numFormat = "upperRoman";
    	} else if (numberFormat.equals("a")) {
    		numFormat = "lowerLetter";
    	} else if (numberFormat.equals("A")) {
    		numFormat = "upperLetter";
    	} else {
    		// a value of type string 18.2. (COMPLEX NUMBERING ie. ASIAN oder ERSTENS..)
    		numFormat = numberFormat;
    	}
    	return numFormat;
    }

    private static String getListStartValue(TextListLevelStyleElementBase listLevelStyle) {
   		return listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "start-value");
   	}

    private static String getLabel(TextListLevelStyleElementBase listLevelStyle, int listLevel) {
    	StringBuilder levelText = new StringBuilder();

    	// creating label prefix
    	String labelPrefix = listLevelStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "num-prefix");
    	if (labelPrefix != null && !labelPrefix.isEmpty()) {
    		levelText.append(labelPrefix);
    	}

    	// creating label number
    	if (listLevelStyle instanceof TextListLevelStyleNumberElement) {
    		String displayLevels = listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "display-levels");
    		if (displayLevels != null && !displayLevels.isEmpty()) {
    			int showLevels = Integer.parseInt(displayLevels);
    			// Creating the label, in ODF always adding the low levelText first, adding each follow up level for display level
    			// Custom string with one of the placeholders from '%1' to '%9') for numbered lists.
    			for (int i = showLevels; i > 0; i--) {
    				levelText.append("%").append(listLevel + 2 - i);
    				// Although not commented in the specification a "." is being added to the text level
    				if (i != 1) {
    					levelText.append('.');
    				}
    			}
    		} else {
    			levelText.append("%").append(listLevel + 1);
    		}
    		// creating label suffix
    		String labelSuffix = listLevelStyle.getAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "num-suffix");
    		if (labelSuffix != null && !labelSuffix.isEmpty()) {
    			levelText.append(labelSuffix);
    		}
    	} else {
    		String bulletChar = listLevelStyle.getAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "bullet-char");
    		if (bulletChar != null && !bulletChar.isEmpty()) {
    			levelText.append(bulletChar);
    		}
    	}
    	return levelText.toString();
    }

    private static JSONObject mapIndent(String minLabelWidthValue, String spaceBeforeValue, JSONObject listLevelDefinition) throws JSONException {
    	int minLabelWidth = 0;
    	boolean isValidMinLabelWidth = Length.isValid(minLabelWidthValue);
    	if (isValidMinLabelWidth) {
    		minLabelWidth = MapHelper.normalizeLength(minLabelWidthValue);
    	}
    	int spaceBefore = 0;
    	boolean isValidSpaceBefore = Length.isValid(spaceBeforeValue);
    	if (isValidSpaceBefore) {
    		spaceBefore = MapHelper.normalizeLength(spaceBeforeValue);
    	}
    	if (isValidMinLabelWidth || isValidSpaceBefore) {
    		listLevelDefinition.put("indentLeft", minLabelWidth + spaceBefore);
    	}
    	return listLevelDefinition;
    }
	
    private void triggerStyleHierarchyOps(JSONArray operationQueue, OdfStylesBase stylesBase, OdfStyleFamily styleFamily, OdfStyleBase style, boolean isAutoStyle, boolean isTextDocument)
    	throws JSONException {

        if (style != null) {

            if (!(style instanceof OdfDefaultStyle)) {
                if (!knownStyles.containsKey(((OdfStyle) style).getStyleNameAttribute())) {
                    List<OdfStyleBase> parents = new LinkedList<OdfStyleBase>();
                    OdfStyleBase parent = style;

                    // Collecting hierachy, to go back through the style hierarchy from the end, to be able to neglect empty styles and adjust parent style attribute
                    while (parent != null
                        && (parent instanceof OdfDefaultStyle || !knownStyles.containsKey(((OdfStyle) parent).getStyleNameAttribute()))) {
                        if (parent instanceof OdfDefaultStyle && stylesBase instanceof OdfOfficeStyles) {
                            triggerDefaultStyleOp(operationQueue, styleFamily, ((OdfOfficeStyles)stylesBase).getDefaultStyle(styleFamily));
                            // NEXT: there is no style above a default in the style hierarchy
                            break;
                        } else if (parent != null) {
                            parents.add(parent);

                            // NEXT: get the next parent style and if the style parent name is the OX DEFAULT NAME remove it
                            Attr parentStyleName = parent.getAttributeNodeNS(OdfDocumentNamespace.STYLE.getUri(), "parent-style-name");
                            if (parentStyleName != null && parentStyleName.getValue().equals(Names.OX_DEFAULT_STYLE_PREFIX + getFamilyID(styleFamily) + Names.OX_DEFAULT_STYLE_SUFFIX)) {
                                parent.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "parent-style-name");
                                triggerDefaultStyleOp(operationQueue, styleFamily, style.getParentStyle());
                                break;
                            } else {
                                parent = parent.getParentStyle();
                            }
                        }

                        // trigger operation only for those style not already existing
                        // check if the named style already exists
                    }

                    String lastWrittenStyleName = null; // Only write out parents with mapped styles
                    boolean skippedEmptyParent = false;
                    // Intermediate ODF properties
                    Map<String, Map<String, String>> allOdfProps = new HashMap<String, Map<String, String>>();
                    // The property groups for this component, e.g. cell, paragraph, text for a cell with properties
                    Map<String, OdfStylePropertiesSet> familyPropertyGroups = getAllOxStyleGroupingIdProperties(styleFamily);
                    // Mapped properties
                    Map<String, Object> mappedFormatting = null;

                    // addChild named style to operation
                    String styleName;

                    // due to inheritance the top ancestor style have to be propagated first
                    for (int i = parents.size() - 1; i >= 0; i--) {
                        style = parents.get(i);
                        if(style instanceof OdfDefaultStyle) {
                        	continue;
                        }
                        styleName = ((OdfStyle) style).getStyleNameAttribute();
                        // get all ODF properties from this style
                        MapHelper.getStyleProperties(style, familyPropertyGroups, allOdfProps);
                        // mapping the ODF attribute style props to our component properties
                        mappedFormatting = MapHelper.mapStyleProperties(familyPropertyGroups, new HashMap<String, Object>(), allOdfProps);
                        if(isAutoStyle) {
                        	MapHelper.putNumberFormat(mappedFormatting, (OdfStyle)style, stylesBase, getOfficeStyles());
                        }
                        else {
                        	MapHelper.putNumberFormat(mappedFormatting, (OdfStyle)style, null, stylesBase);
                        }

                        // No OdfStyle, as the parent still might be a default style without name
                        OdfStyleBase parentStyle = style.getParentStyle();
                        String parentStyleName = null;

                        // Default styles do not have a name
                        if (parentStyle != null && !(parentStyle instanceof OdfDefaultStyle)) {
                            parentStyleName = ((OdfStyle) parentStyle).getStyleNameAttribute();
                        }
                        String nextStyle = ((OdfStyle) style).getStyleNextStyleNameAttribute();
                        Integer outlineLevel = ((OdfStyle) style).getStyleDefaultOutlineLevelAttribute();
                        // Do not trigger operations to create empty styles
                        if (skippedEmptyParent) {
                            parentStyleName = lastWrittenStyleName;
                        }
                        String familyId = getFamilyID(styleFamily);
                        if(isTextDocument) {
	                        if (parentStyleName != null && !parentStyleName.isEmpty()) {
	                            insertStyleSheet(operationQueue, styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, parentStyleName, nextStyle, outlineLevel, false, false, isAutoStyle);
	                        } else if(isAutoStyle) {	// the auto-style without parent
	                        	insertStyleSheet(operationQueue, styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, null, nextStyle, outlineLevel, false, true, isAutoStyle);
	                    	}
	                    	else {
	                    		insertStyleSheet(operationQueue, styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, Names.OX_DEFAULT_STYLE_PREFIX + familyId + Names.OX_DEFAULT_STYLE_SUFFIX, nextStyle, outlineLevel, false, false, isAutoStyle);
	                        }
                        }
                        else {
                        	if(isAutoStyle) {
                        		insertAutoStyle(operationQueue, familyId, styleName, mappedFormatting, parentStyleName, false);
                        	}
                        	else {
	                            insertStyleSheet(operationQueue, styleName, familyId, ((OdfStyle) style).getStyleDisplayNameAttribute(), mappedFormatting, 
	                               		parentStyleName != null && !parentStyleName.isEmpty() ? parentStyleName : Names.OX_DEFAULT_STYLE_PREFIX + familyId + Names.OX_DEFAULT_STYLE_SUFFIX,
	                               			nextStyle, null, false, false, false);
                        	}
                        }
                        lastWrittenStyleName = styleName;
                        mappedFormatting.clear();
                        allOdfProps.clear();
                        // addChild named style to known styles, so it will be only executed once
                        knownStyles.put(styleName, Boolean.TRUE);
                    }
                }
            } else {
                // DEFAULT STYLE PARENT
                // Default styles will receive a name and will be referenced by the root style (the default style for the web editor)
                if (styleFamily.equals(OdfStyleFamily.Paragraph)) {
                    triggerDefaultStyleOp(operationQueue, styleFamily, style);
                } else {
                    triggerDefaultStyleOp(operationQueue, styleFamily, style);
                }
            }
        }
    }

    private static Map<String, OdfStylePropertiesSet> getAllOxStyleGroupingIdProperties(OdfStyleFamily styleFamily) {
    	return getAllOxStyleGroupingIdProperties(styleFamily.getName());
    }
    private static Map<String, OdfStylePropertiesSet> getAllOxStyleGroupingIdProperties(String styleFamily) {
        Map<String, OdfStylePropertiesSet> familyProperties = new HashMap<String, OdfStylePropertiesSet>();
        switch(styleFamily) {
        	case "paragraph" : {
        		familyProperties.put("paragraph", OdfStylePropertiesSet.ParagraphProperties);
        		familyProperties.put("character", OdfStylePropertiesSet.TextProperties);
        		break;
        	}
        	case "text" : {
        		familyProperties.put("character", OdfStylePropertiesSet.TextProperties);
        		break;
        	}
        	case "table" : {
        		familyProperties.put("table", OdfStylePropertiesSet.TableProperties);
        		break;
        	}
        	case "table-row" : {
        		familyProperties.put("row", OdfStylePropertiesSet.TableRowProperties);
        		break;
        	}
        	case "table-cell" : {
        		familyProperties.put("cell", OdfStylePropertiesSet.TableCellProperties);
        		familyProperties.put("paragraph", OdfStylePropertiesSet.ParagraphProperties);
        		familyProperties.put("character", OdfStylePropertiesSet.TextProperties); //changed due to OX naming from text to character
        		break;
        	}
        	case "table-column" : {
        		familyProperties.put("column", OdfStylePropertiesSet.TableColumnProperties);
        		break;
        	}
        	case "section" : {
        		familyProperties.put("section", OdfStylePropertiesSet.SectionProperties);
        		break;
        	}
        	case "list" : {
        		familyProperties.put("list", OdfStylePropertiesSet.ListLevelProperties);
        		break;
        	}
        	case "chart" : {
        		familyProperties.put("chart", OdfStylePropertiesSet.ChartProperties);
        		familyProperties.put("drawing", OdfStylePropertiesSet.GraphicProperties); //changed due to OX naming from graphic to drawing
        		familyProperties.put("paragraph", OdfStylePropertiesSet.ParagraphProperties);
        		familyProperties.put("character", OdfStylePropertiesSet.TextProperties); //changed due to OX naming from text to character
        		break;
        	}
        	case "presentation" :
        	case "graphic" : {
        		familyProperties.put("drawing", OdfStylePropertiesSet.GraphicProperties); //changed due to OX naming from graphic to drawing
        		familyProperties.put("paragraph", OdfStylePropertiesSet.ParagraphProperties);
        		familyProperties.put("character", OdfStylePropertiesSet.TextProperties); //changed due to OX naming from text to character
        		break;
        	}
        	case "drawing-page" : {
        		familyProperties.put("drawing", OdfStylePropertiesSet.DrawingPageProperties);
        		break;
        	}
        	case "ruby" : {
        		familyProperties.put("ruby", OdfStylePropertiesSet.RubyProperties);
        		break;
        	}
        }
        return familyProperties;
    }

    private static String getFamilyID(OdfStyleFamily styleFamily) {
        String familyID = null;
        if (styleFamily.equals(OdfStyleFamily.Paragraph)) {
            familyID = "paragraph";
        } else if (styleFamily.equals(OdfStyleFamily.Text)) {
            familyID = "character";
        } else if (styleFamily.equals(OdfStyleFamily.Table)) {
            familyID = "table";
        } else if (styleFamily.equals(OdfStyleFamily.TableRow)) {
            familyID = "row";
        } else if (styleFamily.equals(OdfStyleFamily.TableCell)) {
            familyID = "cell";
        } else if (styleFamily.equals(OdfStyleFamily.TableColumn)) {
            familyID = "column";
        } else if (styleFamily.equals(OdfStyleFamily.Section)) {
            familyID = "section";
        } else if (styleFamily.equals(OdfStyleFamily.List)) {
            familyID = "list";
        } else if (styleFamily.equals(OdfStyleFamily.Chart)) {
            familyID = "chart";
        } else if (styleFamily.equals(OdfStyleFamily.Graphic) || styleFamily.equals(OdfStyleFamily.Presentation)) {
            familyID = "drawing";
        } else if (styleFamily.equals(OdfStyleFamily.DrawingPage)) {
            familyID = "drawing";
        } else if (styleFamily.equals(OdfStyleFamily.Ruby)) {
            familyID = "ruby";
        }
        return familyID;
    }

    private static String getFamilyDisplayName(OdfStyleFamily styleFamily) {
        String familyID = null;
        if (styleFamily.equals(OdfStyleFamily.Paragraph)) {
            familyID = "Paragraph";
        } else if (styleFamily.equals(OdfStyleFamily.Text)) {
            familyID = "Character";
        } else if (styleFamily.equals(OdfStyleFamily.Table)) {
            familyID = "Table";
        } else if (styleFamily.equals(OdfStyleFamily.TableRow)) {
            familyID = "Row";
        } else if (styleFamily.equals(OdfStyleFamily.TableCell)) {
            familyID = "Cell";
        } else if (styleFamily.equals(OdfStyleFamily.TableColumn)) {
            familyID = "Column";
        } else if (styleFamily.equals(OdfStyleFamily.Section)) {
            familyID = "Section";
        } else if (styleFamily.equals(OdfStyleFamily.List)) {
            familyID = "List";
        } else if (styleFamily.equals(OdfStyleFamily.Presentation)) {
            familyID = "Presentation";
        } else if (styleFamily.equals(OdfStyleFamily.Chart)) {
            familyID = "Chart";
        } else if (styleFamily.equals(OdfStyleFamily.Graphic)) {
            familyID = "Graphic";
        } else if (styleFamily.equals(OdfStyleFamily.DrawingPage)) {
            familyID = "Drawing";
        } else if (styleFamily.equals(OdfStyleFamily.Ruby)) {
            familyID = "Ruby";
        }
        return familyID;
    }

    public static OdfStyleFamily getFamily(String styleId) {
        OdfStyleFamily family = null;
        if (styleId.equals("paragraph")) {
            family = OdfStyleFamily.Paragraph;
        } else if (styleId.equals("character")) {
            family = OdfStyleFamily.Text;
        } else if (styleId.equals("table")) {
            family = OdfStyleFamily.Table;
        } else if (styleId.equals("row")) {
            family = OdfStyleFamily.TableRow;
        } else if (styleId.equals("cell")) {
            family = OdfStyleFamily.TableCell;
        } else if (styleId.equals("column")) {
            family = OdfStyleFamily.TableColumn;
        } else if (styleId.equals("drawing")) {
            family = OdfStyleFamily.Graphic;
        }
        return family;
    }

    /**
     * Tests first if the default style was already added to the document, than
     * triggers a insertStylesheet operation
     */
    private Integer triggerDefaultStyleOp(JSONArray operationQueue, OdfStyleFamily styleFamily, OdfStyleBase style)
    	throws JSONException {

    	Integer defaultTabStopWidth = null;
        // Intermediate ODF properties
        Map<String, Map<String, String>> allOdfProps = new HashMap<String, Map<String, String>>();
        // The property groups for this component, e.g. cell, paragraph, text for a cell with properties
        Map<String, OdfStylePropertiesSet> familyPropertyGroups = getAllOxStyleGroupingIdProperties(styleFamily);
        // Mapped properties
        Map<String, Object> mappedFormatting = null;

        // addChild named style to operation
        String styleName;

        if (style instanceof OdfDefaultStyle && !knownStyles.containsKey(Names.OX_DEFAULT_STYLE_PREFIX + getFamilyID(styleFamily) + Names.OX_DEFAULT_STYLE_SUFFIX)) {
            // get all ODF properties from this style
            MapHelper.getStyleProperties(style, familyPropertyGroups, allOdfProps);
            // mapping the ODF attribute style props to our component properties
            mappedFormatting = MapHelper.mapStyleProperties(familyPropertyGroups, new HashMap<String, Object>(), allOdfProps);
            // Tabulator default size is an attribute in the default style, will be received from static mapping functions
            if (mappedFormatting.containsKey("paragraph")) {
                JSONObject paraProps = (JSONObject) mappedFormatting.get("paragraph");
                if (paraProps.has("document")) {
                    JSONObject documentProps = paraProps.optJSONObject("document");
                    defaultTabStopWidth = documentProps.optInt("defaultTabStop");
                }
            }
            String familyId = getFamilyID(styleFamily);
            // Do not trigger operations to create empty styles
            if (!mappedFormatting.isEmpty()) {
                String displayName = "Default " + getFamilyDisplayName(styleFamily) + " Style";
                insertStyleSheet(operationQueue, Names.OX_DEFAULT_STYLE_PREFIX + familyId + Names.OX_DEFAULT_STYLE_SUFFIX, familyId, displayName, mappedFormatting, null, null, null, true, true, false);
            }
            // addChild named style to known styles, so it will be only executed once
            styleName = Names.OX_DEFAULT_STYLE_PREFIX + getFamilyID(styleFamily) + Names.OX_DEFAULT_STYLE_SUFFIX;
            knownStyles.put(styleName, Boolean.TRUE);
        }
        return defaultTabStopWidth;
    }

    private static void insertStyleSheet(JSONArray operationQueue, String styleId, String familyID, String displayName, Map<String, Object> componentProps, String parentStyle, String nextStyleId, Integer outlineLevel, boolean isDefaultStyle, boolean isHidden, boolean isAutoStyle)
    	throws JSONException {

        final JSONObject insertComponentObject = new JSONObject();
        insertComponentObject.put("name", "insertStyleSheet");
        if (styleId != null && !styleId.isEmpty()) {
            insertComponentObject.put("styleId", styleId);
        }
        insertComponentObject.put("type", familyID);
        if (displayName != null && !displayName.isEmpty()) {
            insertComponentObject.put("styleName", displayName);
        }
        if (familyID.equals("table")) {
            final JSONObject tableStyleAttrs = new JSONObject();
            tableStyleAttrs.put("wholeTable", componentProps);
            insertComponentObject.put("attrs", tableStyleAttrs);

        } else {
            insertComponentObject.put("attrs", componentProps);
        }
        if (parentStyle != null && !parentStyle.isEmpty()) {
            insertComponentObject.put("parent", parentStyle);
        }
        if (isDefaultStyle) {
            insertComponentObject.put("default", isDefaultStyle);
        }
        if (isAutoStyle) {
        	insertComponentObject.put("auto", true);
        }
        else if (isHidden) {
            insertComponentObject.put("hidden", isHidden);
        }
        if (outlineLevel != null || nextStyleId != null) {
            JSONObject paraProps;
            if (componentProps.containsKey("paragraph")) {
                paraProps = (JSONObject) componentProps.get("paragraph");
            } else {
                paraProps = new JSONObject();
                componentProps.put("paragraph", paraProps);
            }
            if (outlineLevel != null) {
                paraProps.put("outlineLevel", outlineLevel - 1);
            }
            if (nextStyleId != null && !nextStyleId.isEmpty()) {
                paraProps.put("nextStyleId", nextStyleId);
            }
            componentProps.put("paragraph", paraProps);
            insertComponentObject.put("attrs", componentProps);
        }
        operationQueue.put(insertComponentObject);
    }

    public void insertAutoStyle(JSONArray operationQueue, String family, String styleId, Map<String, Object> componentProps, String parentStyle, boolean isDefaultStyle)
    	throws JSONException {

    	final JSONObject insertAutoStyleOperation = new JSONObject(5);
    	insertAutoStyleOperation.put("name", "insertAutoStyle");
    	insertAutoStyleOperation.put("type", family);
    	insertAutoStyleOperation.put("styleId", styleId);
    	if(parentStyle!=null) {
    		componentProps.put("styleId", parentStyle);
    	}
    	insertAutoStyleOperation.put("attrs", componentProps);
    	if(isDefaultStyle) {
    		insertAutoStyleOperation.put("default", true);
    	}
    	operationQueue.put(insertAutoStyleOperation);
    }

    private void insertListStyle(JSONArray operationQueue, String styleName, boolean hasConsecutiveNumbering, JSONObject listDefinition)
    	throws JSONException {

    	final JSONObject insertComponentObject = new JSONObject();
        insertComponentObject.put("name", "insertListStyle");
        insertComponentObject.put("listStyleId", styleName);
//				insertComponentObject.put("displayName", displayName);
        if (hasConsecutiveNumbering) {
            insertComponentObject.put("listUnifiedNumbering", hasConsecutiveNumbering);
        }
        insertComponentObject.put("listDefinition", listDefinition);

        operationQueue.put(insertComponentObject);
    }

    public static void mapProperties(OdfStyleFamily styleFamily, JSONObject attrs, OdfStyleBase style, OdfDocument doc4Fonts)
    	throws JSONException, SAXException {

        if (attrs != null && styleFamily != null && style != null) {
            Map<String, OdfStylePropertiesSet> familyProperties = getAllOxStyleGroupingIdProperties(styleFamily);
            Set<String> propTypes = familyProperties.keySet();
            for (String type : propTypes) {
                if (type.equals("character") && attrs.hasAndNotNull("character")) {
                    JSONObject textProps = (JSONObject) attrs.opt("character");
                    if (textProps != null && textProps.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TextProperties);
                        mapCharacterProperties(textProps, (StyleTextPropertiesElement) propsElement, doc4Fonts);
                    }
                } else if (type.equals("paragraph") && attrs.hasAndNotNull("paragraph")) {
                    JSONObject paraProps = (JSONObject) attrs.opt("paragraph");
                    if (paraProps != null && paraProps.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.ParagraphProperties);
                        mapParagraphProperties(paraProps, (StyleParagraphPropertiesElement) propsElement);
                    }
                } else if (type.equals("table")) {
                    if (attrs.hasAndNotNull("table")) {
                        JSONObject tableProps = (JSONObject) attrs.opt("table");
                        if (tableProps != null && tableProps.length() > 0) {
                            OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableProperties);
                            mapTableProperties(tableProps, (StyleTablePropertiesElement) propsElement);
                        }
                    } else if (attrs.hasAndNotNull("sheet")) {
                        // currently the sheet are handled different than the tableElement
                        JSONObject sheetProps = (JSONObject) attrs.opt("sheet");
                        if (sheetProps != null && sheetProps.length() > 0) {
                            OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableProperties);
                            mapTableProperties(sheetProps, (StyleTablePropertiesElement) propsElement);
                        }
                    } else {
                        // some default values have to be set (width 100% for MSO15)
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableProperties);
                        mapTableProperties(null, (StyleTablePropertiesElement) propsElement);
                    }
                } else if (type.equals("row") && attrs.hasAndNotNull("row")) {
                    JSONObject props = (JSONObject) attrs.opt("row");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableRowProperties);
                        mapRowProperties(props, (StyleTableRowPropertiesElement) propsElement);
                    }
                } else if (type.equals("cell") && attrs.hasAndNotNull("cell")) {
                    JSONObject props = (JSONObject) attrs.opt("cell");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableCellProperties);
                        mapCellProperties(props, (StyleTableCellPropertiesElement) propsElement, style);
                    }
                } else if (type.equals("column") && attrs.hasAndNotNull("column")) {
                    JSONObject props = (JSONObject) attrs.opt("column");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.TableColumnProperties);
                        mapColumnProperties(props, (StyleTableColumnPropertiesElement) propsElement);
                    }
                } else if (type.equals("list") && attrs.hasAndNotNull("list")) {
                    JSONObject props = (JSONObject) attrs.opt("list");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.ListLevelProperties);
                        mapListProperties(props, (StyleListLevelPropertiesElement) propsElement);
                    }
                } else if (type.equals("section") && attrs.hasAndNotNull("section")) {
                    JSONObject props = (JSONObject) attrs.opt("section");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.SectionProperties);
                        mapSectionProperties(props, (StyleSectionPropertiesElement) propsElement);
                    }

                } else if (type.equals("drawing") || type.equals("presentation")) {
                    if (attrs.has("drawing") || attrs.has("shape")|| attrs.has("line")|| attrs.has("fill") ) {
                        JSONObject allDrawingProperties = new JSONObject();
                        String subs[] = {"shape", "drawing"};
                        for( String sub : subs ) {
                            if(attrs.has(sub)) {
                                JSONObject subAttrs = attrs.getJSONObject(sub);
                                Iterator<String> keyIt = subAttrs.keys();
                                while( keyIt.hasNext()) {
                                    String key = keyIt.next();
                                    allDrawingProperties.put(key, subAttrs.get(key));
                                }
                            }
                        }
                        if( attrs.has("fill") ) {
                            allDrawingProperties.put("fill", attrs.getJSONObject("fill"));
                        }
                        if( attrs.has("line") ) {
                            allDrawingProperties.put("line", attrs.getJSONObject("line"));
                        }
                        if (allDrawingProperties.length() > 0) {
                            OdfStyleBase parentStyle = style.getParentStyle();
                            OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.GraphicProperties);
                            mapGraphicProperties(allDrawingProperties, (StyleGraphicPropertiesElement) propsElement, parentStyle);
                        }
                    }
                } else if (type.equals("chart") || type.equals("chart")) {
                    JSONObject props = (JSONObject) attrs.opt("chart");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.ChartProperties);
                        mapChartProperties(props, (StyleChartPropertiesElement) propsElement);
                    }
                } else if (type.equals("page") || type.equals("page")) {
                    JSONObject props = (JSONObject) attrs.opt("page");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.PageLayoutProperties);
                        mapPageProperties(props, (StylePageLayoutPropertiesElement) propsElement);
                    }
                } else if (type.equals("ruby") || type.equals("ruby")) {
                    JSONObject props = (JSONObject) attrs.opt("ruby");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.RubyProperties);
                        mapRubyProperties(props, (StyleRubyPropertiesElement) propsElement);
                    }
                } else if (type.equals("headerFooter") || type.equals("headerFooter")) {
                    JSONObject props = (JSONObject) attrs.opt("headerFooter");
                    if (props != null && props.length() > 0) {
                        OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.HeaderFooterProperties);
                        mapHeaderFooterProperties(props, (StyleHeaderFooterPropertiesElement) propsElement);
                    }
                }
            }
        }
    }

    // The latter fontNames Set is a hack for the OX release as the 16 fonts supported do not write out their descriptions
    static private void mapCharacterProperties(JSONObject attrs, StyleTextPropertiesElement propertiesElement, OdfDocument doc4Fonts)
    	throws JSONException, SAXException {

    	if (attrs != null) {

        	Object language = null;
        	Object noProof = null;

        	final Iterator<Entry<String, Object>> entrySetIter = attrs.entrySet().iterator();
            while(entrySetIter.hasNext()) {
            	final Entry<String, Object> entrySet = entrySetIter.next();
            	final String key = entrySet.getKey();
            	final Object value = entrySet.getValue();
                if (key.equals("bold")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "font-weight");
                    } else {
                        Boolean isBold = (Boolean) value;
                        if (isBold) {
                            propertiesElement.setFoFontWeightAttribute("bold");
                        } else {
                            propertiesElement.setFoFontWeightAttribute("normal");
                        }
                    }
                } else if (key.equals("boldAsian")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "font-weight-asian");
                    } else {
                        Boolean isBold = (Boolean) value;
                        if (isBold) {
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-weight-asian", "bold");
                        } else {
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-weight-asian", "normal");
                        }
                    }
                } else if (key.equals("boldComplex")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "font-weight-complex");
                    } else {
                        Boolean isBold = (Boolean) value;
                        if (isBold) {
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-weight-complex", "bold");
                        } else {
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-weight-complex", "normal");
                        }
                    }
                } else if (key.equals("underline")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-underline-style");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-underline-width");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-underline-color");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-underline-type");
                    } else {
                        Boolean isUnderline = (Boolean) value;
                        if (isUnderline) {
                            propertiesElement.setStyleTextUnderlineStyleAttribute("solid");
                        } else {
                            propertiesElement.setStyleTextUnderlineStyleAttribute("none");
                        }
                    }
                } else if (key.equals("italic")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "font-style");
                    } else {
                        Boolean isItalic = (Boolean) value;
                        if (isItalic) {
                            propertiesElement.setFoFontStyleAttribute("italic");
                        } else {
                            propertiesElement.setFoFontStyleAttribute("normal");
                        }
                    }
                } else if (key.equals("italicAsian")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "font-style-asian");
                    } else {
                        Boolean isItalic = (Boolean) value;
                        if (isItalic) {
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-style-asian", "italic");

                        } else {
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-style-asian", "normal");
                        }
                    }
                } else if (key.equals("italicComplex")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "font-style-complex");
                    } else {
                        Boolean isItalic = (Boolean) value;
                        if (isItalic) {
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-style-complex", "italic");
                        } else {
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-style-complex", "normal");
                        }
                    }
                } else if (key.equals("color")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        if (color.hasAndNotNull("type")) {
                            String type = color.optString("type", "");
                            if (!type.equals(MapHelper.AUTO)) {
                                propertiesElement.setFoColorAttribute(getColor(color, null));
                                propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "use-window-font-color");
                            } else {
                                propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:use-window-font-color", "true");
                            }
                        } else { // DEFAULT IS AUTO
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:use-window-font-color", "true");
                        }
                    }
                } else if (key.equals("fillColor")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        propertiesElement.setFoBackgroundColorAttribute(getColor(color, MapHelper.TRANSPARENT));
                    }
                } else if (key.equals("fontSize")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "font-size");
                    } else {
                        propertiesElement.setFoFontSizeAttribute(value.toString() + "pt");
                    }
                } else if (key.equals("fontSizeAsian")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-size-asian");
                    } else {
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-size-asian", value.toString() + "pt");
                    }
                } else if (key.equals("fontSizeComplex")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-size-complex");
                    } else {
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-size-complex", value.toString() + "pt");
                    }
                } else if (key.equals("fontName")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name");
                    } else {
                        String fontName = (String) value;
                        propertiesElement.setStyleFontNameAttribute(fontName);
                        addFontToDocument(fontName, doc4Fonts);
                    }
                } else if (key.equals("fontNameAsian")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name-asian");
                    } else {
                        String fontName = (String) value;
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-name-asian", fontName);
                    }
                } else if (key.equals("fontNameComplex")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name-complex");
                    } else {
                        String fontName = (String) value;
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-name-complex", fontName);
                    }
                } else if (key.equals("vertAlign")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-position");
                    } else {
                        String alignment = (String) value;
                        if (alignment.equals("sub")) {
                            propertiesElement.setStyleTextPositionAttribute("sub");
                        } else if (alignment.equals("super")) {
                            propertiesElement.setStyleTextPositionAttribute("super");
                        } else { //baseline
                            propertiesElement.setStyleTextPositionAttribute("0% 100%");
                        }
                    }
                } else if (key.equals("strike")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-position");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-line-through-color");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-line-through-mode");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-line-through-style");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-line-through-text");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-line-through-text-style");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-line-through-type");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-line-through-width");
                    } else {
                        String strikeType = (String) value;
                        if (strikeType.equals("single")) {
                            propertiesElement.setStyleTextLineThroughTypeAttribute("single");
                            propertiesElement.setStyleTextLineThroughStyleAttribute("solid");
                            propertiesElement.setStyleTextLineThroughModeAttribute("continuous");
                            propertiesElement.setStyleTextUnderlineModeAttribute("continuous");
                            propertiesElement.setStyleTextOverlineModeAttribute("continuous");
                        } else if (!strikeType.equals("none")) { //double
                            propertiesElement.setStyleTextLineThroughTypeAttribute("double");
                            propertiesElement.setStyleTextLineThroughStyleAttribute("solid");
                            propertiesElement.setStyleTextLineThroughModeAttribute("continuous");
                            propertiesElement.setStyleTextUnderlineModeAttribute("continuous");
                            propertiesElement.setStyleTextOverlineModeAttribute("continuous");
                        }
                    }
                } else if (key.equals("language")) {
                	language = value;
                } else if (key.equals("noProof")) {
                	noProof = value;
                } else if (key.equals("letterSpacing")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "letter-spacing");
                    } else {
                        if (value.equals("normal")) {
                            propertiesElement.setFoLetterSpacingAttribute("normal");
                        } else {
                            propertiesElement.setFoLetterSpacingAttribute((getSafelyInteger(value)) / 100.0 + "mm");
                        }
                    }
                } else if (key.equals("url")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "href");
                    } else {
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "xlink:href", (String) value);
                    }
                }
            }
            if(noProof!=null || language!=null) {
            	Object newLanguage = language;
            	if((noProof instanceof Boolean && ((Boolean)noProof).booleanValue()) || ((language instanceof String) && ((String)language).equals("none"))) {
        			propertiesElement.setFoLanguageAttribute("zxx");
        			propertiesElement.setStyleLanguageAsianAttribute("zxx");
        			propertiesElement.setStyleCountryComplexAttribute("zxx");
        			propertiesElement.setFoCountryAttribute("none");
        			propertiesElement.setStyleCountryAsianAttribute("none");
        			propertiesElement.setStyleCountryComplexAttribute("none");
            	}
            	else if (newLanguage == null || newLanguage.equals(JSONObject.NULL)) {
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "country");
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "language");
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "country-asian");
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "language-asian");
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "country-complex");
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "language-complex");
                } else {
                    String locale = (String) newLanguage;
                    if (!locale.isEmpty()) {
                        int delimiterPos = locale.indexOf('-');
                        if (delimiterPos > -1) {
                            propertiesElement.setFoLanguageAttribute(locale.substring(0, delimiterPos));
                            propertiesElement.setFoCountryAttribute(locale.substring(delimiterPos + 1, locale.length()));
                        } else {
                            propertiesElement.setFoLanguageAttribute(locale);
                        }
                    }
                }
            }
        }
    }

    private static void mapParagraphProperties(JSONObject attrs, StyleParagraphPropertiesElement propertiesElement)
    	throws JSONException {

        if (attrs != null) {
            Set<String> propKeys = attrs.keySet();
            for (String key : propKeys) {
                addBorderProperties(key, attrs, propertiesElement, null);
                addPaddingProperties(key, attrs, propertiesElement);
                addMarginProperties(key, attrs, propertiesElement);

                if (key.equals("fillColor")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        propertiesElement.setFoBackgroundColorAttribute(getColor(color, MapHelper.TRANSPARENT));
                    }
                } else if (key.equals("lineHeight")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "line-height");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-height-at-least");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-spacing");
                    } else {
                        JSONObject lineHeight = (JSONObject) value;
                        setLineHeight(lineHeight, propertiesElement);
                    }
                } else if (key.equals("alignment")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-align");
                    } else {
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:text-align", (String) value);
                    }
                } else if (key.equals("indentFirstLine")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-indent");
                    } else {
                        Integer indent = getSafelyInteger(value);
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:text-indent", ((indent / 100.0) + "mm"));
                    }
                } else if (key.equals("pageBreakBefore")) {
                    Object value = attrs.get(key);
                    if( value != JSONObject.NULL && value.equals(Boolean.TRUE) ) {
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:break-before", "page");
                    } else {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "break-before");
                    }
                    // there can not be before and after break at the same paragraph
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "break-after");
                } else if (key.equals("pageBreakAfter")) {
                    Object value = attrs.get(key);
                    if( value != JSONObject.NULL && value.equals(Boolean.TRUE) ) {
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:break-after", "page");
                    } else {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "break-after");
                    }
                    // there can not be before and after break at the same paragraph
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "break-before");
                } else if (key.equals("tabStops")){
                    Object value = attrs.get(key);
                    StyleTabStopsElement tabsElement = OdfElement.findFirstChildNode(StyleTabStopsElement.class, propertiesElement);
                    if(tabsElement != null){
                        propertiesElement.removeChild(tabsElement);
                    }
                    if (value == null || value.equals(JSONObject.NULL)) {
                        // node already removed
                    } else {
                        JSONArray tabsValue = (JSONArray)value;

                        OdfFileDom fileDom = (OdfFileDom)propertiesElement.getOwnerDocument();
                        tabsElement = new StyleTabStopsElement(fileDom);
                        for( int idx = 0; idx < tabsValue.length(); ++idx){
                            JSONObject tab = tabsValue.getJSONObject(idx);
                            StyleTabStopElement tabElement = new StyleTabStopElement(fileDom);
                            if( tab.has("pos") ){
                                int tabPos = tab.getInt("pos");
                                tabElement.setStylePositionAttribute((tabPos / 1000F) + "cm");
                            }
                            if( tab.has("value")) {
                                String tabValue = tab.getString("value");
                                if(tabValue.equals("decimal")){
                                    tabValue = "char";
                                } else if(tabValue.equals("bar")){
                                    tabValue = "left";
                                } else if(tabValue.equals("clear")){
                                    continue; // clear unsupported
                                }
                                tabElement.setStyleTypeAttribute(tabValue); //center, char, left, right
                            }
                            tabsElement.insertBefore(tabElement, null);
                        }

                        propertiesElement.insertBefore(tabsElement, null);
                    }
                }
            }
        }
    }

    private static void mapListProperties(JSONObject attrs, StyleListLevelPropertiesElement propertiesElement)
    	throws JSONException {

    	if (attrs != null) {
            Set<String> propKeys = attrs.keySet();
            for (String key : propKeys) {
                if (key.equals("fontName")) {
                    Object value = attrs.get(key);
                    // == null does not work here..
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name");
                    } else {
                        String fontName = (String) value;
                        propertiesElement.setStyleFontNameAttribute(fontName);
                    }
                } else if (key.equals("fontNameAsian")) {
                    Object value = attrs.get(key);
                    // == null does not work here..
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name-asian");
                    } else {
                        String fontName = (String) value;
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-name-asian", fontName);
                    }
                } else if (key.equals("fontNameComplex")) {
                    Object value = attrs.get(key);
                    // == null does not work here..
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "font-name-complex");
                    } else {
                        String fontName = (String) value;
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-name-complex", fontName);
                    }
                }
            }
        }
    }

    private static void mapSectionProperties(JSONObject attrs, StyleSectionPropertiesElement propertiesElement)
    	throws JSONException {

    	if (attrs != null) {
            Set<String> propKeys = attrs.keySet();
            for (String key : propKeys) {
                // No margin in ODF
                addBorderProperties(key, attrs, propertiesElement, null);
                addPaddingProperties(key, attrs, propertiesElement);
                if (key.equals("fillColor")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        propertiesElement.setFoBackgroundColorAttribute(getColor(color, MapHelper.TRANSPARENT));
                    }
                }
            }
        }
    }

    public static void mapGraphicProperties(JSONObject attrs, StyleGraphicPropertiesElement propertiesElement, OdfStyleBase frameParentStyle )
    	throws JSONException {

    	if (attrs != null) {
            boolean isMirroredHorizontalRemoved = false;
            boolean isMirroredHorizontal = false;
            boolean isMirroredVerticalRemoved = false;
            boolean isMirroredVertical = false;
            Set<String> propKeys = attrs.keySet();
            for (String key : propKeys) {
                addBorderProperties(key, attrs, propertiesElement, frameParentStyle);
                addPaddingProperties(key, attrs, propertiesElement);
                addMarginProperties(key, attrs, propertiesElement);
                if( key.equals("fill")) {
                    JSONObject value = (JSONObject)attrs.get(key);
                    boolean flyFrame = frameParentStyle != null && !(frameParentStyle instanceof OdfDefaultStyle);
                    if (value == null || value.equals(JSONObject.NULL) || ( value.has("type") && (value.isNull("type") || value.getString("type").equals("none")))) {
                        if( flyFrame )
                        	propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                        else
                        {
                        	propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "fill-color");
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.DRAW.getUri(), "draw:fill", "none");
                        }
                    } else {
                        JSONObject color = value.getJSONObject("color");
                        String colorAttr = getColor(color, MapHelper.TRANSPARENT);
                        if( flyFrame )
                        	propertiesElement.setFoBackgroundColorAttribute(colorAttr);
                        else
                        {
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.DRAW.getUri(), "draw:fill-color", colorAttr);
                            propertiesElement.setAttributeNS(OdfDocumentNamespace.DRAW.getUri(), "draw:fill", "solid");
                        }
                    }

                } else if (key.equals("flipH")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        isMirroredHorizontalRemoved = true;
                    } else if ((Boolean) value) {
                        isMirroredHorizontal = true;
                    }
                } else if (key.equals("flipV")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        isMirroredVerticalRemoved = true;
                    } else if ((Boolean) value) {
                        isMirroredVertical = true;
                    }
                } else if (key.equals("anchorHorBase")) {
                    Object value = attrs.get(key);
                    if (value != null && !value.equals(JSONObject.NULL)) {
                        String horBase = (String)value;
                        Map<String, String> relMap = new HashMap<String, String>();
                        relMap.put("margin", "page-start-margin"      );
                        relMap.put("page", "page"                     );
                        relMap.put("column", "paragraph"              );
                        relMap.put("character", "char"                );
                        relMap.put("leftMargin", "page-start-margin"  );
                        relMap.put("rightMargin", "page-end-margin"   );
                        relMap.put("insideMargin", "page-start-margin"   ); //TODO: set horizontal-postion:from-inside
                        relMap.put("outsideMargin", "page-start-margin"  ); //TODO: set horizontal-postion:outside
                        String odfValue = relMap.get(horBase);
                        if(odfValue != null)
                            propertiesElement.setStyleHorizontalRelAttribute(odfValue);
                    }
                } else if (key.equals("anchorVertBase")) {
                    Object value = attrs.get(key);
                    if (value != null && !value.equals(JSONObject.NULL)) {
                        String horBase = (String)value;
                        Map<String, String> relMap = new HashMap<String, String>();
                        relMap.put("margin", "page-content"       );
                        relMap.put("page", "page"                 );
                        relMap.put("paragraph", "paragraph"       );
                        relMap.put("line", "line"                 );
                        relMap.put("topMargin", "page-content"    );
                        relMap.put("bottomMargin", "page-content" );
                        String odfValue = relMap.get(horBase);
                        if(odfValue != null)
                            propertiesElement.setStyleVerticalRelAttribute(odfValue);
                    }
                } else if (key.equals("textWrapMode")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "wrap");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "wrap-contour");
                    }
                    else {
                        String textWrapMode = (String) value;
                        if (textWrapMode.equals("topAndBottom")) {
                            propertiesElement.setStyleWrapAttribute("none");
                            propertiesElement.setStyleWrapContourAttribute(Boolean.FALSE);
                        } else {
                            String textWrapSide = attrs.optString("textWrapSide");
                            if (textWrapSide != null) {
                                if (textWrapMode.equals("square")) {
                                    if (textWrapSide.equals("largest")) {
                                        propertiesElement.setStyleWrapAttribute("biggest");
                                    } else if (textWrapSide.equals("left")) {
                                        propertiesElement.setStyleWrapAttribute("left");
                                    } else if (textWrapSide.equals("both")) {
                                        propertiesElement.setStyleWrapAttribute("parallel");
                                    } else if (textWrapSide.equals("right")) {
                                        propertiesElement.setStyleWrapAttribute("right");
                                    }
                                    propertiesElement.setStyleWrapContourAttribute(Boolean.FALSE);
                                } else if (textWrapMode.equals("through")) {
                                    propertiesElement.setStyleWrapAttribute("run-through");
                                    propertiesElement.setStyleWrapContourAttribute(Boolean.FALSE);
                                }
                            }
                        }
                    }
                } else if (key.equals("anchorHorAlign")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "horizontal-pos");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "horizontal-rel");
                    } else {
                        String anchorHorAlign = (String) value;
                        if (anchorHorAlign.equals("center")) {
                            propertiesElement.setStyleHorizontalPosAttribute("center");
                        } else if (anchorHorAlign.equals("offset")) {
                            propertiesElement.setStyleHorizontalPosAttribute("from-left");
                        } else if (anchorHorAlign.equals("left")) {
                            propertiesElement.setStyleHorizontalPosAttribute("left");
                        } else if (anchorHorAlign.equals("right")) {
                            propertiesElement.setStyleHorizontalPosAttribute("right");
                        } else if (anchorHorAlign.equals("inside")) {
                            propertiesElement.setStyleHorizontalPosAttribute("inside");
                        } else if (anchorHorAlign.equals("outside")) {
                            propertiesElement.setStyleHorizontalPosAttribute("outside");
                        }
                    }
                } else if (key.equals("anchorHorOffset")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.SVG.getUri(), "x");
                    } else {
                        Integer x = getSafelyInteger(value);

                        // This is the default in our constellation
                        propertiesElement.setSvgXAttribute(x / 100.0 + "mm");
                    }
                } else if (key.equals("anchorVertAlign")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-pos");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-rel");
                    } else {
                        String anchorVertAlign = (String) value;
                        if (anchorVertAlign.equals("center")) {
                            propertiesElement.setStyleVerticalPosAttribute("center");
                        } else if (anchorVertAlign.equals("offset")) {
                            propertiesElement.setStyleVerticalPosAttribute("from-top");
                        } else if (anchorVertAlign.equals("bottom")) {
                            propertiesElement.setStyleVerticalPosAttribute("bottom");
                        } else if (anchorVertAlign.equals("top")) {
                            propertiesElement.setStyleVerticalPosAttribute("top");
                        } else if (anchorVertAlign.equals("inside")) {
                            propertiesElement.setStyleVerticalPosAttribute("inside");
                        } else if (anchorVertAlign.equals("outside")) {
                            propertiesElement.setStyleVerticalPosAttribute("outside");
                        }
                    }
                } else if (key.equals("anchorVertOffset")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.SVG.getUri(), "y");
                    } else {
                        Integer y = getSafelyInteger(value);
                        // This is the default in our constellation
                        propertiesElement.setSvgYAttribute(y / 100.0 + "mm");
                    }
                } else if (key.equals("anchorBehindDoc")) {
                    boolean anchorBehindDoc = attrs.optBoolean(key, false);
                    StyleRunThroughAttribute.Value attr = null;                                
                    if(anchorBehindDoc){
                        attr = StyleRunThroughAttribute.Value.BACKGROUND;
                    }else {
                        attr =  StyleRunThroughAttribute.Value.FOREGROUND;
                    }                
                    propertiesElement.setStyleRunThroughAttribute(attr.toString());
                }
            }

            //					} else if (propName.contains("svg:x")) {
            //						int x = normalizeLength(odfProps.get("svg:x"));
            //						if(x != 0){
            //							newProps.put("anchorHorOffset", x);
            //						}
            // Two attributes are evaluated via boolean flags:
            if (isMirroredHorizontalRemoved && isMirroredVerticalRemoved) {
                propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "mirror");
            } else if (isMirroredHorizontal && isMirroredVertical) {
                propertiesElement.setStyleMirrorAttribute("horizontal vertical");
            } else if (isMirroredVertical) {
                propertiesElement.setStyleMirrorAttribute("vertical");
            } else if (isMirroredHorizontal) {
                propertiesElement.setStyleMirrorAttribute("horizontal");
            }
        }
    }

    private static void mapTableProperties(JSONObject attrs, StyleTablePropertiesElement propertiesElement)
    	throws JSONException {

    	if (attrs != null) {
            Set<String> propKeys = attrs.keySet();
            boolean isTableWidthGiven = false;
            for (String key : propKeys) {
                // no padding, no border
                addMarginProperties(key, attrs, propertiesElement);
                if (key.equals("width")) {
                    Object value = null;
                    value = attrs.get(key);

                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "width");
                    } else {
                        Object width = value;
                        if (width != null && !width.equals(MapHelper.AUTO)) {
                            // MSO 15 WORKAROUND..
                            isTableWidthGiven = true;
                            if (width instanceof Integer) {
                                propertiesElement.setStyleWidthAttribute(((Integer) width) / 1000.0 + "cm");
                            } else if (width instanceof String) {
                                propertiesElement.setStyleWidthAttribute(Integer.parseInt((String) width) / 1000.0 + "cm");
                            }
                            // LO/AO0 WORKSROUND: if there is a rel width with 100% and an absolute style the rel-width wins :(
                            // WIDTH COMES - REL-WIDTH GOES...
                            propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "rel-width");
                            // LO/AOO WORKAROUND: width is only recognized by LO/AO, when an alignment is provided. If none now left (or margin equal alignment is set to left!
                            if (!propertiesElement.hasAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "align") || propertiesElement.getAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "align").equalsIgnoreCase("margins")) {
                                propertiesElement.setTableAlignAttribute("left");
                            }
                        }
                    }
                } else if (key.equals("fillColor")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        propertiesElement.setFoBackgroundColorAttribute(getColor(color, MapHelper.TRANSPARENT));
                    }
                } else if (key.equals("visible")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.TABLE.getUri(), "display");
                    } else {
                        propertiesElement.setTableDisplayAttribute((Boolean) value);
                    }
                }
            }
            // MSO 15 WORKAROUND:
            if (!isTableWidthGiven) {
                // by default at least a 100 percent relative width for the table have to be given for MSO15
                propertiesElement.setStyleRelWidthAttribute("100%");
            }
        } else {
            if (propertiesElement != null) {
                // by default at least a 100 percent relative width for the table have to be given for MSO15
                propertiesElement.setStyleRelWidthAttribute("100%");
            }
        }
    }

    private static void mapRowProperties(JSONObject attrs, StyleTableRowPropertiesElement propertiesElement)
    	throws JSONException {

    	if (attrs != null) {
            Set<String> propKeys = attrs.keySet();
            for (String key : propKeys) {
                if (key.equals("fillColor")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        propertiesElement.setFoBackgroundColorAttribute(getColor(color, MapHelper.TRANSPARENT));
                    }
                } else if (key.equals("height")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "row-height");
                        // currently we are not differentiating between height and minimum height
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "min-row-height");
                    } else {
                        Integer rowHeight = getSafelyInteger(value);
                        propertiesElement.setStyleRowHeightAttribute(((rowHeight / 100.0) + "mm"));
                    }
                } else if (key.equals("customHeight")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "use-optimal-row-height");
                    } else {
                        propertiesElement.setStyleUseOptimalRowHeightAttribute(!((Boolean) value));
                    }
                }
            }
        }
    }

    private static void mapColumnProperties(JSONObject attrs, StyleTableColumnPropertiesElement propertiesElement)
    	throws JSONException {

    	if (attrs != null) {
            Set<String> propKeys = attrs.keySet();
            for (String key : propKeys) {
                if (key.equals("width")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "column-width");
                    } else {
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:column-width", (getSafelyInteger(value) / 100.0) + "mm");
                    }
                } else if (key.equals("customWidth")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "use-optimal-column-width");
                    } else {
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:use-optimal-column-width", value.toString());
                    }
                }
            }
        }
    }

    private static void mapCellProperties(JSONObject attrs, StyleTableCellPropertiesElement propertiesElement, OdfStyleBase style)
    	throws JSONException {

    	if (attrs!=null) {
    		String newCellProtect = null;
    		final Set<Entry<String, Object>> entrySet = attrs.entrySet();
            for (Entry<String, Object> entry : entrySet) {

            	final String key = entry.getKey();
            	final Object value = entry.getValue();

            	// No margin in ODF
            	addBorderProperties(key, attrs, propertiesElement, null);
            	addPaddingProperties(key, attrs, propertiesElement);
                if (key.equals("fillColor")) {
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        propertiesElement.setFoBackgroundColorAttribute(getColor(color, MapHelper.TRANSPARENT));
                    }
                }
                else if( key.equals("alignVert")){
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "vertical-align");
                    } else {
                        String align = (String)value;
                        propertiesElement.setStyleVerticalAlignAttribute(align);
                    }
                }
                else if(key.equals("alignHor")){
                    OdfStylePropertiesBase propsElement = style.getOrCreatePropertiesElement(OdfStylePropertiesSet.ParagraphProperties);
                    if (value == null || value.equals(JSONObject.NULL)) {
                    	propsElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "text-align");
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "text-align-source");
                    } else {
                        String align = (String)value;
                        if(align.equals("right")) {
                        	align = "end";
                        }
                        else if(align.equals("left")) {
                        	align = "start";
                        }
                        propsElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:text-align", align);
                        propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "style:text-align-source", "fix");
                    }
                }
                else if(key.equals("wrapText")) {
					if(value==null||!(Boolean)value) {
						propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "wrap-option");
					}
					else {
						propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:wrap-option", (Boolean)value ? "wrap" : "no-wrap");
					}
                }
                else if(key.equals("unlocked")) {
                	final String oldCellProtect = propertiesElement.getAttributeNS(StyleCellProtectAttribute.ATTRIBUTE_NAME.getUri(), StyleCellProtectAttribute.ATTRIBUTE_NAME.getLocalName());
                	if(newCellProtect==null) {
                		newCellProtect = "none";
                	}
                	if(value instanceof Boolean) {
                		if((Boolean)value) {
                			if(oldCellProtect.equals("hidden-and-protected")) {
                				newCellProtect = "formula-hidden";
                			}
                		}
                		else {
                			if(oldCellProtect.equals("formula-hidden")) {
                				newCellProtect = "hidden-and-protected";
                			}
                			else if(oldCellProtect.equals("hidden-and-protected")) {
                				newCellProtect = "hidden-and-protected";
                			}
                			else {
                				newCellProtect = "protected";
                			}
                		}
                	}
                	
                }
                else if(key.equals("hidden")) {
                	final String oldCellProtect = propertiesElement.getAttributeNS(StyleCellProtectAttribute.ATTRIBUTE_NAME.getUri(), StyleCellProtectAttribute.ATTRIBUTE_NAME.getLocalName());
                	if(newCellProtect==null) {
                		newCellProtect = "none";
                	}
                	if(value instanceof Boolean) {
                		if((Boolean)value) {
                			if(oldCellProtect.equals("hidden-and-protected")) {
                				newCellProtect = oldCellProtect;
                			}
                			else if(oldCellProtect.equals("protected")) {
                				newCellProtect = "hidden-and-protected";
                			}
                			else {
                				newCellProtect = "formula-hidden";
                			}
                		}
                		else {
                			if(oldCellProtect.equals("hidden-and-protected")) {
                				newCellProtect = "protected";
                			}
                		}
                	}
                }
            }
            if(newCellProtect!=null) {
            	propertiesElement.setAttributeNS(StyleCellProtectAttribute.ATTRIBUTE_NAME.getUri(), StyleCellProtectAttribute.ATTRIBUTE_NAME.getQName(), newCellProtect);
            }
        }
    }

    private static void mapChartProperties(JSONObject attrs, StyleChartPropertiesElement propertiesElement) {
    }

    private static void mapRubyProperties(JSONObject attrs, StyleRubyPropertiesElement propertiesElement) {
    }

    private static void mapHeaderFooterProperties(JSONObject attrs, StyleHeaderFooterPropertiesElement propertiesElement)
    	throws JSONException {

    	if (attrs != null) {
            Set<String> propKeys = attrs.keySet();
            for (String key : propKeys) {
                // No border, padding in ODF
                addMarginProperties(key, attrs, propertiesElement);
                if (key.equals("fillColor")) {
                    Object value = attrs.get(key);
                    // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        propertiesElement.setFoBackgroundColorAttribute(getColor(color, MapHelper.TRANSPARENT));

                    }
                }
            }
        }
    }

    private static void mapPageProperties(JSONObject attrs, StylePageLayoutPropertiesElement propertiesElement)
    	throws JSONException {

    	if (attrs != null) {
            Set<String> propKeys = attrs.keySet();
            for (String key : propKeys) {
                addBorderProperties(key, attrs, propertiesElement, null);
                addPaddingProperties(key, attrs, propertiesElement);
                addMarginProperties(key, attrs, propertiesElement);
                if (key.equals("fillColor")) {
                    Object value = attrs.get(key);
                    if (value == null || value.equals(JSONObject.NULL)) {
                        propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "background-color");
                    } else {
                        JSONObject color = (JSONObject) value;
                        propertiesElement.setFoBackgroundColorAttribute(getColor(color, MapHelper.TRANSPARENT));
                    }
                }
            }
        }
    }

    private static void addFontToDocument(String fontName, OdfDocument doc4Fonts) 
    	throws SAXException {

    	if (doc4Fonts != null) {
            Set<String> fontNames = doc4Fonts.getFontNames();
            if (fontName != null && !fontName.isEmpty()) {
                if (!fontNames.contains(fontName)) {
                    fontNames.add(fontName);
                    if (fontName.equals("Andale Mono")) {
                        insertFontDescription(doc4Fonts, "Andale Mono", null, "Andale Mono", null, null, null);
                    } else if (fontName.equals("Arial")) {
                        insertFontDescription(doc4Fonts, "Arial", null, "Arial", "swiss", "variable", null);
                    } else if (fontName.equals("Book Antiqua")) {
                        insertFontDescription(doc4Fonts, "Book Antiqua", null, "Book Antiqua", "roman", "variable", null);
                    } else if (fontName.equals("Calibri")) {
                        insertFontDescription(doc4Fonts, "Calibri", null, "Calibri", "swiss", "variable", "2 15 5 2 2 2 4 3 2 4");
                    } else if (fontName.equals("Cambria")) {
                        insertFontDescription(doc4Fonts, "Cambria", null, "Cambria", "roman", "variable", "2 4 5 3 5 4 6 3 2 4");
                    } else if (fontName.equals("Consolas")) {
                        insertFontDescription(doc4Fonts, "Consolas", null, "Consolas", "modern", "fixed", null);
                    } else if (fontName.equals("Courier New")) {
                        insertFontDescription(doc4Fonts, "Courier New", null, "Courier New", "modern", "fixed", null);
                    } else if (fontName.equals("Courier")) {
                        insertFontDescription(doc4Fonts, "Courier", null, "Courier", "modern", "fixed", null);
                    } else if (fontName.equals("Georgia")) {
                        insertFontDescription(doc4Fonts, "Georgia", null, "Georgia", "roman", "variable", null);
                    } else if (fontName.equals("Helvetica")) {
                        insertFontDescription(doc4Fonts, "Helvetica", null, "Helvetica", "swiss", null, null);
                    } else if (fontName.equals("Impact")) {
                        insertFontDescription(doc4Fonts, "Impact", null, "Impact", "swiss", "variable", null);
                    } else if (fontName.equals("Mangal")) {
                        insertFontDescription(doc4Fonts, "Mangal", null, "Mangal", "system", "variable", null);
                    } else if (fontName.equals("Mangal1")) {
                        insertFontDescription(doc4Fonts, "Mangal1", null, "Mangal", null, null, null);
                    } else if (fontName.equals("Microsoft YaHei")) {
                        insertFontDescription(doc4Fonts, "Microsoft YaHei", null, "Microsoft YaHei", "system", "variable", null);
                    } else if (fontName.equals("MS Gothic")) {
                        insertFontDescription(doc4Fonts, "MS Gothic", null, "MS Gothic", "modern", "fixed", "2 11 6 9 7 2 5 8 2 4");
                    } else if (fontName.equals("MS Mincho")) {
                        insertFontDescription(doc4Fonts, "MS Mincho", null, "MS Mincho", "modern", "fixed", "2 2 6 9 4 2 5 8 3 4");
                    } else if (fontName.equals("Palatino")) {
                        insertFontDescription(doc4Fonts, "Palatino", null, "Palatino", "roman", null, null);
                    } else if (fontName.equals("SimSun")) {
                        insertFontDescription(doc4Fonts, "SimSun", null, "SimSun", "system", "variable", null);
                    } else if (fontName.equals("Tahoma")) {
                        insertFontDescription(doc4Fonts, "Tahoma", null, "Tahoma", "swiss", "variable", null);
                    } else if (fontName.equals("Times New Roman")) {
                        insertFontDescription(doc4Fonts, "Times New Roman", null, "Times New Roman", "roman", "variable", "2 2 6 3 5 4 5 2 3 4");
                    } else if (fontName.equals("Times")) {
                        insertFontDescription(doc4Fonts, "Times", null, "Times", "roman", null, null);
                    } else if (fontName.equals("Verdana")) {
                        insertFontDescription(doc4Fonts, "Verdana", null, "Verdana", "swiss", "variable", null);
                    }
                }
            }
        }
    }

    private static void insertFontDescription(OdfDocument doc, String fontName, String[] altNames, String family, String familyGeneric, String pitch, String panose1)
    	throws SAXException {

    	// CREATING CONTENT FONT DECLARATION
        OdfContentDom contentDom = doc.getContentDom();
        StyleFontFaceElement newElement = new StyleFontFaceElement(contentDom);
        newElement.setStyleNameAttribute(fontName);
        if (family != null && !family.isEmpty()) {
            newElement.setSvgFontFamilyAttribute(family);
        }
        if (familyGeneric != null && !familyGeneric.isEmpty()) {
            newElement.setStyleFontFamilyGenericAttribute(familyGeneric);
        }
        if (pitch != null && !pitch.isEmpty()) {
            newElement.setStyleFontPitchAttribute(pitch);
        }
        if (panose1 != null && !panose1.isEmpty()) {
            if (panose1.contains("[")) {
                panose1 = panose1.substring(1, panose1.length() - 1);
            }
            if (panose1.contains(",")) {
                panose1 = panose1.replace(',', ' ');
            }
            newElement.setSvgPanose1Attribute(panose1);
        }
        insertFontFace(contentDom, newElement);

        OdfStylesDom stylesDom = doc.getStylesDom();
        StyleFontFaceElement newElement2 = (StyleFontFaceElement) newElement.cloneNode(true);
        newElement2 = (StyleFontFaceElement) stylesDom.adoptNode(newElement2);
        insertFontFace(stylesDom, newElement2);
    }

    private static void insertFontFace(OdfFileDom fileDom, StyleFontFaceElement fontFaceElement) {
        NodeList fontDecls = fileDom.getElementsByTagName("office:font-face-decls");
        if (fontDecls.getLength() == 0) {
            Element rootElement = fileDom.getRootElement();
            NodeList rootChildren = rootElement.getChildNodes();
            boolean hasInserted = false;
            OfficeFontFaceDeclsElement fontFaceDeclsElement = new OfficeFontFaceDeclsElement(fileDom);
            fontFaceDeclsElement.appendChild(fontFaceElement);
            for (int i = 0; i < rootChildren.getLength(); i++) {
                Node currentNode = rootChildren.item(i);
                if (currentNode instanceof Element) {
                    String elementName = ((Element) currentNode).getNodeName();
                    if (elementName.equals("office:automatic-styles") || elementName.equals("office:styles") || elementName.equals("office:body") || elementName.equals("office:master-styles")) {
                        rootElement.insertBefore(fontFaceDeclsElement, currentNode);
                        hasInserted = true;
                        break;
                    }
                } else {
                    continue;
                }
            }
            if (!hasInserted) {
                rootElement.appendChild(fontFaceDeclsElement);
            }
        } else {
            OfficeFontFaceDeclsElement fontFaceDeclsElement = (OfficeFontFaceDeclsElement) fontDecls.item(0);
            fontFaceDeclsElement.appendChild(fontFaceElement);
        }
    }

    private static String getBorder(JSONObject border)
    	throws JSONException {

    	String style = null;
        String borderValue = "";
        if (border.hasAndNotNull("style")) {
            //  'none', 'single', 'double', 'dotted', 'dashed', 'outset', or 'inset'
            style = border.optString("style");
            if (style.equals("none")) {
                borderValue = "none";
            } else {
                if (border.hasAndNotNull("width")) {
                    double width = border.optInt("width") / 100.0;
                    borderValue = width + "mm ";
                } else {
                    // DEFAULT BORDER: 0.002cm solid #000000"
                    borderValue = "0.02mm ";
                }
                if (border.hasAndNotNull("style")) {
                    //  'none', 'single', 'double', 'dotted', 'dashed', 'outset', or 'inset'
                    style = border.optString("style");
                    if (style.equals("single")) {
                        borderValue = borderValue.concat("solid ");
                    } else {
                        borderValue = borderValue.concat(style + " ");
                    }
                } else {
                    // DEFAULT BORDER: 0.002cm solid #000000"
                    borderValue = borderValue.concat("solid ");
                }
                if (border.hasAndNotNull("color")) {
                    JSONObject color = border.optJSONObject("color");
                    borderValue = borderValue.concat(getColor(color, "#000000"));
                } else {
                    // DEFAULT BORDER: 0.002cm solid #000000"
                    borderValue = borderValue.concat("#000000");
                }
            }
        }
        return borderValue;
    }

    private static void addBorderProperties(String key, JSONObject attrs, OdfStylePropertiesBase propertiesElement, OdfStyleBase frameParentStyle )
    	throws JSONException {

    	boolean putBorders = false;
        if(attrs.hasAndNotNull("line")) {
            boolean isEmptyBorder = true;
            String allBorderString      = propertiesElement.getAttribute("fo:border");
            String leftBorderString     = propertiesElement.getAttribute("fo:border-left");
            String rightBorderString    = propertiesElement.getAttribute("fo:border-right");
            String topBorderString      = propertiesElement.getAttribute("fo:border-top");
            String bottomBorderString   = propertiesElement.getAttribute("fo:border-bottom");
            boolean useOneBorder = allBorderString.length() > 0;
            if(!useOneBorder&&frameParentStyle!=null) {
                OdfStylePropertiesBase propsElement = frameParentStyle.getOrCreatePropertiesElement(OdfStylePropertiesSet.GraphicProperties);
                String styleAllBorderString = propsElement.getAttribute("fo:border");
                if( leftBorderString  .length() == 0 ) {
                    leftBorderString     = propsElement.getAttribute("fo:border-left");
                }
                if( rightBorderString .length() == 0 ) {
                    rightBorderString    = propsElement.getAttribute("fo:border-right");
                }
                if( topBorderString.length() == 0) {
                    topBorderString      = propsElement.getAttribute("fo:border-top");
                }
                if( bottomBorderString.length() == 0 ) {
                    bottomBorderString   = propsElement.getAttribute("fo:border-bottom");
                }
                if( styleAllBorderString.length() > 0 && leftBorderString  .length() == 0 && rightBorderString  .length() == 0 &&
                    topBorderString.length() == 0 && bottomBorderString  .length() == 0 ) {
                    allBorderString      = styleAllBorderString;
                    useOneBorder = true;
                } else {
                    useOneBorder = false;
                }
            }

            JSONObject allBorder        = MapHelper.createBorderMap(allBorderString);
            JSONObject oldLeftBorder    = MapHelper.createBorderMap(leftBorderString);
            JSONObject oldRightBorder   = MapHelper.createBorderMap(rightBorderString);
            JSONObject oldTopBorder     = MapHelper.createBorderMap(topBorderString);
            JSONObject oldBottomBorder  = MapHelper.createBorderMap(bottomBorderString);
            isEmptyBorder = useOneBorder ? (!allBorder.has("width")|| !allBorder.has("style")) : (!oldTopBorder.has("width")|| !oldTopBorder.has("style"));

            JSONObject line = attrs.getJSONObject("line");
            JSONObject lineColor = line.optJSONObject("color");
            boolean lineColorIsNull = line.has("color") && line.isNull("color");
            boolean lineTypeIsNull = line.has("type") && line.isNull("type");
            boolean lineWidthIsNull = line.has("width") && line.isNull("width");
            boolean lineStyleIsNull = line.has("style") && line.isNull("style");
            if(lineStyleIsNull && lineTypeIsNull && lineWidthIsNull ) {
                putBorders = true;
                useOneBorder= true;
                allBorder.reset();
            } else {
                String lineType = line.optString("type");
                String lineStyle = line.optString("style");
                int width = line.optInt("width");
                if(lineColor != null) {
                    if(useOneBorder) {
                        allBorder.put("color", lineColor);
                    } else {
                        oldLeftBorder  .put("color", lineColor);
                        oldRightBorder .put("color", lineColor);
                        oldTopBorder   .put("color", lineColor);
                        oldBottomBorder.put("color", lineColor);
                    }
                    putBorders = true;
                    if( isEmptyBorder ) {
                        if(lineStyle.isEmpty()) {
                            lineStyle = "solid";
                        }
                        if( width == 0) {
                            width = 1;
                        }
                    }
                } else if (lineColorIsNull) {
                    if(useOneBorder) {
                        allBorder.remove("color");
                    } else {
                        oldLeftBorder  .remove("color");
                        oldRightBorder .remove("color");
                        oldTopBorder   .remove("color");
                        oldBottomBorder.remove("color");
                    }
                    putBorders = true;
                }
                if( lineType != null && lineStyle != null && (lineStyle.length() > 0 || (lineType.equals("none")))) {
                    // type: none, auto, solid
                    // style:               solid,          dotted, dashed, dashDot, dashDotDot
                    //border-style: none,   single, double, dotted, dashed, dashDot, dashDotDot
                    String newStyle = lineStyle;;
                    if(lineType.equals("none")){
                        newStyle = "none";
                    } else if(lineStyle.equals("solid")){
                        newStyle = "single";
                    }
                    if(useOneBorder) {
                        allBorder   .put("style", newStyle);
                    } else {
                        oldLeftBorder   .put("style", newStyle);
                        oldRightBorder  .put("style", newStyle);
                        oldTopBorder    .put("style", newStyle);
                        oldBottomBorder .put("style", newStyle);
                    }
                    if( isEmptyBorder && width == 0 && !newStyle.equals("none")) {
                        width = 1;
                    }
                    putBorders = true;

                }
                if( width > 0 ) {
                    if(useOneBorder) {
                        allBorder       .put("width", width );
                    } else {
                        oldLeftBorder   .put("width", width );
                        oldRightBorder  .put("width", width );
                        oldTopBorder    .put("width", width );
                        oldBottomBorder .put("width", width );
                    }
                    putBorders = true;
                }
            }
            if(putBorders) {
                JSONObject copyAttrs = new JSONObject(attrs);
                copyAttrs.put( "borderLeft", useOneBorder? allBorder : oldLeftBorder);
                copyAttrs.put( "borderRight", useOneBorder? allBorder : oldRightBorder);
                copyAttrs.put( "borderTop", useOneBorder? allBorder : oldTopBorder);
                copyAttrs.put( "borderBottom", useOneBorder? allBorder : oldBottomBorder);
                attrs = copyAttrs;
                propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "border");
            }
        }
        if (putBorders || key.equals("borderLeft")) {
            Object value = attrs.get("borderLeft");
            // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
            if (value == null || value.equals(JSONObject.NULL)) {
                propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "border-left");
            } else {
                // {\"width\":2,
                // \"style\":\"solid\"
                // \"color\":{\"value\":\"000000\",\"type\":\"rgb\"}}
                JSONObject border = (JSONObject) value;
                propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:border-left", getBorder(border));
                // oppose to the other border styles, the padding was added to border
                if (border.hasAndNotNull("space")) {
                    double padding = border.optInt("space") / 100.0;
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:padding-left", padding + "mm");
                }
            }
        }
        if (putBorders || key.equals("borderRight")) {
            Object value = attrs.get("borderRight");
            // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
            if (value == null || value.equals(JSONObject.NULL)) {
                propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "border-right");
            } else {
                // {\"width\":15,
                // \"style\":\"solid\"
                // \"color\":{\"value\":\"000000\",\"type\":\"rgb\"}} / // \"color\":{\"type\":\"auto\"}}
                // \"space\":\140
                JSONObject border = (JSONObject) value;
                propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:border-right", getBorder(border));
                // oppose to the other border styles, the padding was added to border
                if (border.hasAndNotNull("space")) {
                    double padding = border.optInt("space") / 100.0;
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:padding-right", padding + "mm");
                }
            }
        }
        if (putBorders || key.equals("borderTop")) {
            Object value = attrs.get("borderTop");
            // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
            if (value == null || value.equals(JSONObject.NULL)) {
                propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "border-top");
            } else {
                // {\"width\":2,
                // \"style\":\"solid\"
                // \"color\":{\"value\":\"000000\",\"type\":\"rgb\"}}
                JSONObject border = (JSONObject) value;
                propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:border-top", getBorder(border));
                // oppose to the other border styles, the padding was added to border
                if (border.hasAndNotNull("space")) {
                    double padding = border.optInt("space") / 100.0;
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:padding-top", padding + "mm");
                }
            }
        }
        if (putBorders || key.equals("borderBottom")) {
            Object value = attrs.get("borderBottom");
            // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
            if (value == null || value.equals(JSONObject.NULL)) {
                propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "border-bottom");
            } else {
                // {\"width\":2,
                // \"style\":\"solid\"
                // \"color\":{\"value\":\"000000\",\"type\":\"rgb\"}}
                JSONObject border = (JSONObject) value;
                propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:border-bottom", getBorder(border));
                // oppose to the other border styles, the padding was added to border
                if (border.hasAndNotNull("space")) {
                    double padding = border.optInt("space") / 100.0;
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:padding-bottom", padding + "mm");
                }
            }
        }
    }

    private static void addMarginProperties(String key, JSONObject attrs, OdfStylePropertiesBase propertiesElement)
    	throws JSONException {

    	if (key.contains("margin") || key.contains("indent")) {
            if (key.equals("marginBottom")) {
                Object value = attrs.get(key);
                // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
                if (value == null || value.equals(JSONObject.NULL)) {
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "margin-bottom");
                } else {
                    Integer width = getSafelyInteger(value);
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:margin-bottom", ((width / 100.0) + "mm"));

                }
            } else if (key.equals("marginLeft") || key.equals("indentLeft")) {
                Object value = attrs.get(key);
                // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
                if (value == null || value.equals(JSONObject.NULL)) {
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "margin-left");
                } else {
                    Integer width = getSafelyInteger(value);
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:margin-left", ((width / 100.0) + "mm"));

                }
            } else if (key.equals("marginRight") || key.equals("indentRight")) {
                Object value = attrs.get(key);
                // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
                if (value == null || value.equals(JSONObject.NULL)) {
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "margin-right");
                } else {
                    Integer width = getSafelyInteger(value);
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:margin-right", ((width / 100.0) + "mm"));

                }
            } else if (key.equals("marginTop")) {
                Object value = attrs.get(key);
                // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
                if (value == null || value.equals(JSONObject.NULL)) {
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "margin-top");
                } else {
                    Integer width = getSafelyInteger(value);
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:margin-top", ((width / 100.0) + "mm"));

                }
            }
        }
    }

    private static void addPaddingProperties(String key, JSONObject attrs, OdfStylePropertiesBase propertiesElement)
    	throws JSONException {

    	if (key.contains("padding")) {
            if (key.equals("paddingBottom")) {
                Object value = attrs.get(key);
                // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
                if (value == null || value.equals(JSONObject.NULL)) {
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "padding-bottom");
                } else {
                    Integer width = getSafelyInteger(value);
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:padding-bottom", ((width / 100.0) + "mm"));

                }
            } else if (key.equals("paddingLeft")) {
                Object value = attrs.get(key);
                // ToDo: Test with latest JSON Library and adapt accordingly for all occurences
                if (value == null || value.equals(JSONObject.NULL)) {
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "padding-left");
                } else {
                    Integer width = getSafelyInteger(value);
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:padding-left", ((width / 100.0) + "mm"));

                }
            } else if (key.equals("paddingRight")) {
                Object value = attrs.get(key);
                if (value == null || value.equals(JSONObject.NULL)) {
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "padding-right");
                } else {
                    Integer width = getSafelyInteger(value);
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:padding-right", ((width / 100.0) + "mm"));

                }
            } else if (key.equals("paddingTop")) {
                Object value = attrs.get(key);
                if (value == null || value.equals(JSONObject.NULL)) {
                    propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "padding-top");
                } else {
                    Integer width = getSafelyInteger(value);
                    propertiesElement.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:padding-top", ((width / 100.0) + "mm"));

                }
            }
        }
    }

    /**
     * Supporting "normal", "percentage" and a none-negative number
     */
    private static void setLineHeight(JSONObject lineHeight, StyleParagraphPropertiesElement propertiesElement)
    	throws JSONException {

    	String lineHeightValue = null;
        String type = lineHeight.getString("type");
        if (type.equals("percent")) {
            String value = lineHeight.getString("value");
            if (value != null && !value.isEmpty()) {
                lineHeightValue = value.concat("%");
                propertiesElement.setFoLineHeightAttribute(lineHeightValue);
                propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-height-at-least");
                propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-spacing");
            }
        } else if (type.equals("fixed")) {
            double value = lineHeight.optInt("value") / 100.0;
            lineHeightValue = value + "mm";
            propertiesElement.setFoLineHeightAttribute(lineHeightValue);
            propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-height-at-least");
            propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-spacing");
        } else if (type.equals("atLeast")) {
            double value = lineHeight.optInt("value") / 100.0;
            lineHeightValue = value + "mm";
            propertiesElement.setStyleLineHeightAtLeastAttribute(lineHeightValue);
            propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "line-height");
            propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-spacing");

        } else if (type.equals("leading")) {
            double value = lineHeight.optInt("value") / 100.0;
            lineHeightValue = value + "mm";
            propertiesElement.setStyleLineSpacingAttribute(lineHeightValue);
            propertiesElement.removeAttributeNS(OdfDocumentNamespace.FO.getUri(), "line-height");
            propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-height-at-least");
        } else if (type.equals("normal")) {
            lineHeightValue = "normal";
            propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-height-at-least");
            propertiesElement.removeAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "line-spacing");
        }
    }

    private static String getColor(JSONObject color, String autoColor)
    	throws JSONException {

    	String colorValue = "";
        if (color != null) {
            String type = color.getString("type");
            if (type.equals(MapHelper.AUTO)) {
                colorValue = autoColor;
            } else if (type.equals("rgb")) {
                colorValue = '#' + color.getString("value");
            } else if (color.has("fallbackValue")) {
                // {"color":{"type":"scheme","value":"accent1","transformations":[{"type":"shade","value":74902}],"fallbackValue":"376092"}
                colorValue = '#' + color.getString("fallbackValue");
            } else {

            }
        }
        return colorValue;
    }

    private static int getSafelyInteger(Object value) {
    	if(value instanceof Number) {
    		return ((Number)value).intValue();
    	}
    	return 0;
    }
}
