/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils/sortedarray'
], function (SortedArray) {

    'use strict';

    // class SortedArray ======================================================

    describe('Spreadsheet class SortedArray', function () {

        it('should exist', function () {
            expect(SortedArray).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var StringArray = SortedArray.extend({
            constructor: function () { SortedArray.call(this, function (value) { return parseInt(value, 10); }); }
        });

        var ObjectArray = SortedArray.extend({
            constructor: function () { SortedArray.call(this, function (object) { return object.prop; }); }
        });

        // public methods -----------------------------------------------------

        describe('method "length"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('length');
            });
            it('should return the length of the array', function () {
                var array = new StringArray();
                expect(array.length()).to.equal(0);
                array._array.push('1', '3', '5', '7');
                expect(array.length()).to.equal(4);
            });
        });

        describe('method "empty"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('empty');
            });
            it('should return whether the array is empty', function () {
                var array = new StringArray();
                expect(array.empty()).to.equal(true);
                array._array.push('1', '3', '5', '7');
                expect(array.empty()).to.equal(false);
            });
        });

        describe('method "clear"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('clear');
            });
            it('should clear the array', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                expect(array.empty()).to.equal(false);
                expect(array.clear()).to.equal(array);
                expect(array.empty()).to.equal(true);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('clone');
            });
            it('should create a shallow clone', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                var clone = array.clone();
                expect(clone).to.be.an.instanceof(SortedArray);
                expect(clone.length()).to.equal(4);
                expect(clone).to.not.equal(array);
                expect(clone._sorter).to.equal(array._sorter);
                expect(clone._array).to.not.equal(array._array);
                expect(clone._array).to.deep.equal(array._array);
            });
            it('should create a deep clone', function () {
                var obj1 = { prop: 1 }, obj2 = { prop: 1 };
                obj1.clone = _.constant(obj2);
                var array = new ObjectArray();
                array._array.push(obj1);
                var clone1 = array.clone();
                var clone2 = array.clone(true);
                expect(clone1._array[0]).to.equal(obj1);
                expect(clone2._array[0]).to.equal(obj2);
            });
        });

        describe('method "values"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('values');
            });
            it('should return the internal array', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                expect(array.values()).to.equal(array._array);
            });
        });

        describe('method "first"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('first');
            });
            it('should return the first element of the array', function () {
                var array = new StringArray();
                expect(array.first()).to.equal(null);
                array._array.push('1', '3', '5', '7');
                expect(array.first()).to.equal('1');
            });
        });

        describe('method "last"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('last');
            });
            it('should return the last element of the array', function () {
                var array = new StringArray();
                expect(array.last()).to.equal(null);
                array._array.push('1', '3', '5', '7');
                expect(array.last()).to.equal('7');
            });
        });

        describe('method "find"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('find');
            });
            it('should return the descriptors of array elements', function () {
                var array = new StringArray();
                expect(array.find(1)).to.equal(null);
                expect(array.find(1, 'next')).to.equal(null);
                expect(array.find(1, 'prev')).to.equal(null);
                expect(array.find(1, 'exact')).to.equal(null);
                array._array.push('1', '3', '5', '7');
                var desc = array.find(0);
                expect(desc).to.have.a.property('array', array);
                expect(desc).to.have.a.property('value', '1');
                expect(desc).to.have.a.property('index', 1);
                expect(desc).to.have.a.property('_ai', 0);
                expect(array.find(0, 'next')).to.contain({ value: '1', index: 1, _ai: 0 });
                expect(array.find(0, 'prev')).to.equal(null);
                expect(array.find(0, 'exact')).to.equal(null);
                expect(array.find(1)).to.contain({ value: '1', index: 1, _ai: 0 });
                expect(array.find(1, 'next')).to.contain({ value: '1', index: 1, _ai: 0 });
                expect(array.find(1, 'prev')).to.contain({ value: '1', index: 1, _ai: 0 });
                expect(array.find(1, 'exact')).to.contain({ value: '1', index: 1, _ai: 0 });
                expect(array.find(2)).to.contain({ value: '3', index: 3, _ai: 1 });
                expect(array.find(2, 'next')).to.contain({ value: '3', index: 3, _ai: 1 });
                expect(array.find(2, 'prev')).to.contain({ value: '1', index: 1, _ai: 0 });
                expect(array.find(2, 'exact')).to.equal(null);
                expect(array.find(3)).to.contain({ value: '3', index: 3, _ai: 1 });
                expect(array.find(3, 'next')).to.contain({ value: '3', index: 3, _ai: 1 });
                expect(array.find(3, 'prev')).to.contain({ value: '3', index: 3, _ai: 1 });
                expect(array.find(3, 'exact')).to.contain({ value: '3', index: 3, _ai: 1 });
                expect(array.find(4)).to.contain({ value: '5', index: 5, _ai: 2 });
                expect(array.find(4, 'next')).to.contain({ value: '5', index: 5, _ai: 2 });
                expect(array.find(4, 'prev')).to.contain({ value: '3', index: 3, _ai: 1 });
                expect(array.find(4, 'exact')).to.equal(null);
                expect(array.find(5)).to.contain({ value: '5', index: 5, _ai: 2 });
                expect(array.find(5, 'next')).to.contain({ value: '5', index: 5, _ai: 2 });
                expect(array.find(5, 'prev')).to.contain({ value: '5', index: 5, _ai: 2 });
                expect(array.find(5, 'exact')).to.contain({ value: '5', index: 5, _ai: 2 });
                expect(array.find(6)).to.contain({ value: '7', index: 7, _ai: 3 });
                expect(array.find(6, 'next')).to.contain({ value: '7', index: 7, _ai: 3 });
                expect(array.find(6, 'prev')).to.contain({ value: '5', index: 5, _ai: 2 });
                expect(array.find(6, 'exact')).to.equal(null);
                expect(array.find(7)).to.contain({ value: '7', index: 7, _ai: 3 });
                expect(array.find(7, 'next')).to.contain({ value: '7', index: 7, _ai: 3 });
                expect(array.find(7, 'prev')).to.contain({ value: '7', index: 7, _ai: 3 });
                expect(array.find(7, 'exact')).to.contain({ value: '7', index: 7, _ai: 3 });
                expect(array.find(8)).to.equal(null);
                expect(array.find(8, 'next')).to.equal(null);
                expect(array.find(8, 'prev')).to.contain({ value: '7', index: 7, _ai: 3 });
                expect(array.find(8, 'exact')).to.equal(null);
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('has');
            });
            it('should return whether the array element exists', function () {
                var array = new StringArray();
                expect(array.has(1)).to.equal(false);
                array._array.push('1', '3', '5', '7');
                expect(array.has(0)).to.equal(false);
                expect(array.has(1)).to.equal(true);
                expect(array.has(2)).to.equal(false);
                expect(array.has(3)).to.equal(true);
                expect(array.has(4)).to.equal(false);
                expect(array.has(5)).to.equal(true);
                expect(array.has(6)).to.equal(false);
                expect(array.has(7)).to.equal(true);
                expect(array.has(8)).to.equal(false);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('get');
            });
            it('should return the array elements by sort index', function () {
                var array = new StringArray();
                expect(array.get(1)).to.equal(null);
                array._array.push('1', '3', '5', '7');
                expect(array.get(0)).to.equal(null);
                expect(array.get(1)).to.equal('1');
                expect(array.get(2)).to.equal(null);
                expect(array.get(3)).to.equal('3');
                expect(array.get(4)).to.equal(null);
                expect(array.get(5)).to.equal('5');
                expect(array.get(6)).to.equal(null);
                expect(array.get(7)).to.equal('7');
                expect(array.get(8)).to.equal(null);
            });
        });

        describe('method "getOrCreate"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('getOrCreate');
            });
            it('should create missing array elements', function () {
                var array = new StringArray();
                expect(array.getOrCreate(3, String)).to.equal('3');
                expect(array._array).to.deep.equal(['3']);
                expect(array.getOrCreate(7, String)).to.equal('7');
                expect(array._array).to.deep.equal(['3', '7']);
                expect(array.getOrCreate(1, String)).to.equal('1');
                expect(array._array).to.deep.equal(['1', '3', '7']);
                expect(array.getOrCreate(5, String)).to.equal('5');
                expect(array._array).to.deep.equal(['1', '3', '5', '7']);
            });
            it('should not create existing array elements', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                expect(array.getOrCreate(1, String)).to.equal('1');
                expect(array.getOrCreate(3, String)).to.equal('3');
                expect(array.getOrCreate(5, String)).to.equal('5');
                expect(array.getOrCreate(7, String)).to.equal('7');
                expect(array._array).to.deep.equal(['1', '3', '5', '7']);
            });
            it('should use calling context', function () {
                var array = new StringArray(), spy = sinon.spy(String), context = {};
                array.getOrCreate(1, spy, context);
                sinon.assert.alwaysCalledOn(spy, context);
            });
        });

        describe('method "slice"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('slice');
            });
            it('should return the array elements', function () {
                var array = new StringArray();
                expect(array.get(1)).to.equal(null);
                array._array.push('1', '3', '5', '7');
                expect(array.slice(0, 0)).to.deep.equal([]);
                expect(array.slice(0, 1)).to.deep.equal(['1']);
                expect(array.slice(1, 1)).to.deep.equal(['1']);
                expect(array.slice(1, 2)).to.deep.equal(['1']);
                expect(array.slice(2, 2)).to.deep.equal([]);
                expect(array.slice(2, 3)).to.deep.equal(['3']);
                expect(array.slice(5, 6)).to.deep.equal(['5']);
                expect(array.slice(6, 6)).to.deep.equal([]);
                expect(array.slice(6, 7)).to.deep.equal(['7']);
                expect(array.slice(7, 7)).to.deep.equal(['7']);
                expect(array.slice(7, 8)).to.deep.equal(['7']);
                expect(array.slice(8, 8)).to.deep.equal([]);
                expect(array.slice(2, 6)).to.deep.equal(['3', '5']);
                expect(array.slice(1, 7)).to.deep.equal(['1', '3', '5', '7']);
                expect(array.slice(0, 8)).to.deep.equal(['1', '3', '5', '7']);
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('forEach');
            });
            it('should visit all array elements', function () {
                var array = new StringArray(), spy = sinon.spy(), context = {};
                array._array.push('1', '3', '5', '7');
                expect(array.forEach(spy, context)).to.equal(array);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.callCount(spy, 4);
                sinon.assert.calledWithExactly(spy.getCall(0), '1', 1);
                sinon.assert.calledWithExactly(spy.getCall(1), '3', 3);
                sinon.assert.calledWithExactly(spy.getCall(2), '5', 5);
                sinon.assert.calledWithExactly(spy.getCall(3), '7', 7);
            });
        });

        describe('method "iterator"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('iterator');
            });
            it('should visit all array elements', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                var it = array.iterator();
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: '1', index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: '3', index: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: '5', index: 5 });
                expect(it.next()).to.deep.equal({ done: false, value: '7', index: 7 });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all array elements in reversed order', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                var it = array.iterator(true);
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: '7', index: 7 });
                expect(it.next()).to.deep.equal({ done: false, value: '5', index: 5 });
                expect(it.next()).to.deep.equal({ done: false, value: '3', index: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: '1', index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "intervalIterator"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('intervalIterator');
            });
            it('should visit all array elements', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                var it = array.intervalIterator(0, 8);
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: '1', index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: '3', index: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: '5', index: 5 });
                expect(it.next()).to.deep.equal({ done: false, value: '7', index: 7 });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit specific array elements', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                var it = array.intervalIterator(3, 5);
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: '3', index: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: '5', index: 5 });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all array elements in reversed order', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                var it = array.intervalIterator(0, 8, true);
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: '7', index: 7 });
                expect(it.next()).to.deep.equal({ done: false, value: '5', index: 5 });
                expect(it.next()).to.deep.equal({ done: false, value: '3', index: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: '1', index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit specific array elements in reversed order', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                var it = array.intervalIterator(3, 5, true);
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: '5', index: 5 });
                expect(it.next()).to.deep.equal({ done: false, value: '3', index: 3 });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "index"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('index');
            });
            it('should return the sort index of a value', function () {
                var array = new StringArray();
                expect(array.index('0')).to.equal(0);
                expect(array.index('1')).to.equal(1);
                expect(array.index('2')).to.equal(2);
                expect(array.index('3')).to.equal(3);
            });
        });

        describe('method "insert"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('insert');
            });
            it('should insert new array elements', function () {
                var array = new StringArray();
                expect(array.insert('3')).to.equal('3');
                expect(array._array).to.deep.equal(['3']);
                array.insert('7');
                expect(array._array).to.deep.equal(['3', '7']);
                array.insert('1');
                expect(array._array).to.deep.equal(['1', '3', '7']);
                array.insert('5');
                expect(array._array).to.deep.equal(['1', '3', '5', '7']);
            });
            it('should support the replace flag', function () {
                var array = new ObjectArray();
                var obj1 = { prop: 3 };
                var obj2 = { prop: 3 };
                expect(array.insert(obj1)).to.equal(obj1);
                expect(array.insert(obj2)).to.equal(obj1);
                expect(array.length()).to.equal(1);
                expect(array.insert(obj2, true)).to.equal(obj2);
                expect(array.length()).to.equal(1);
            });
        });

        describe('method "remove"', function () {
            it('should exist', function () {
                expect(SortedArray).to.respondTo('remove');
            });
            it('should remove existing array elements', function () {
                var array = new StringArray();
                array._array.push('1', '3', '5', '7');
                expect(array.remove('4')).to.equal(null);
                expect(array._array).to.deep.equal(['1', '3', '5', '7']);
                expect(array.remove('3')).to.equal('3');
                expect(array._array).to.deep.equal(['1', '5', '7']);
                array.remove('7');
                expect(array._array).to.deep.equal(['1', '5']);
                array.remove('1');
                expect(array._array).to.deep.equal(['5']);
                array.remove('5');
                expect(array._array).to.deep.equal([]);
            });
            it('should return the removed object', function () {
                var array = new ObjectArray();
                var obj = { prop: 3 };
                array.insert(obj);
                expect(array.length()).to.equal(1);
                expect(array.remove({ prop: 2 })).to.equal(null);
                expect(array.remove({ prop: 3 })).to.equal(obj);
                expect(array.length()).to.equal(0);
            });
        });
    });

    // ========================================================================
});
