/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils'
], function (Utils, IteratorUtils) {

    'use strict';

    // static class IteratorUtils =============================================

    describe('Toolkit module IteratorUtils', function () {

        it('should exist', function () {
            expect(IteratorUtils).to.be.an('object');
        });

        // -----------------------------------------------------

        describe('constant "ALWAYS_DONE_ITERATOR"', function () {
            it('should exist', function () {
                expect(IteratorUtils).to.have.a.property('ALWAYS_DONE_ITERATOR').that.is.an('object');
                expect(IteratorUtils.ALWAYS_DONE_ITERATOR).to.respondTo('next');
                expect(IteratorUtils.ALWAYS_DONE_ITERATOR.next()).to.deep.equal({ done: true });
            });
        });

        // static methods -----------------------------------------------------

        describe('method "createSingleIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).to.respondTo('createSingleIterator');
            });
            it('should visit a single result', function () {
                var it = IteratorUtils.createSingleIterator({ value: 42, other: 'abc' });
                expect(it.next()).to.deep.equal({ done: false, value: 42, other: 'abc' });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "createIndexIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).to.respondTo('createIndexIterator');
            });
            it('should visit all indexes', function () {
                var it = IteratorUtils.createIndexIterator(3);
                expect(it.next()).to.deep.equal({ done: false, value: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all indexes in reversed order', function () {
                var it = IteratorUtils.createIndexIterator(3, { reverse: true });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 0 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all indexes with an offset', function () {
                var it = IteratorUtils.createIndexIterator(3, { offset: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: 4 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all indexes in reversed order with an offset', function () {
                var it = IteratorUtils.createIndexIterator(3, { reverse: true, offset: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 4 });
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "createIntervalIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).to.respondTo('createIntervalIterator');
            });
            it('should visit all indexes', function () {
                var it = IteratorUtils.createIntervalIterator(3, 5);
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: 4 });
                expect(it.next()).to.deep.equal({ done: false, value: 5 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all indexes in reversed order', function () {
                var it = IteratorUtils.createIntervalIterator(3, 5, { reverse: true });
                expect(it.next()).to.deep.equal({ done: false, value: 5 });
                expect(it.next()).to.deep.equal({ done: false, value: 4 });
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "createArrayIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).to.respondTo('createArrayIterator');
            });
            it('should visit all elements of an array', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3]);
                expect(it.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array from a specific start index', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3], { begin: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array to a specific end index', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3], { end: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in a specific interval', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3], { begin: 1, end: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in reversed order', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3], { reverse: true });
                expect(it.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in reversed order from a specific start index', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3], { reverse: true, begin: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in reversed order to a specific end index', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3], { reverse: true, end: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array in reversed order in a specific interval', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3], { reverse: true, begin: 1, end: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all elements of an array-like object', function () {
                var it = IteratorUtils.createArrayIterator('abc');
                expect(it.next()).to.deep.equal({ done: false, value: 'a', index: 0 });
                expect(it.next()).to.deep.equal({ done: false, value: 'b', index: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 'c', index: 2 });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "createObjectIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).to.respondTo('createObjectIterator');
            });
            it('should visit all properties of an object', function () {
                var it = IteratorUtils.createObjectIterator({ a: 42, b: true, c: 'abc' });
                var results = [it.next(), it.next(), it.next()];
                results = _.sortBy(results, function (result) { return result.key; });
                expect(results[0]).to.deep.equal({ done: false, value: 42, key: 'a' });
                expect(results[1]).to.deep.equal({ done: false, value: true, key: 'b' });
                expect(results[2]).to.deep.equal({ done: false, value: 'abc', key: 'c' });
                expect(it.next()).to.deep.equal({ done: true });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "createWhileIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('createWhileIterator');
            });
            it('should create an iterator', function () {
                function generate(v) { return (v % 2) ? null : (v === 6) ? { done: true } : { value: v }; }
                var spy = sinon.spy(generate), context = {};
                var whileIt = IteratorUtils.createWhileIterator(spy, context);
                expect(whileIt).to.respondTo('next');
                expect(whileIt.next()).to.deep.equal({ done: false, value: 0 });
                expect(whileIt.next()).to.deep.equal({ done: false, value: 2 });
                expect(whileIt.next()).to.deep.equal({ done: false, value: 4 });
                expect(whileIt.next()).to.deep.equal({ done: true });
                sinon.assert.callCount(spy, 7);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 0);
                sinon.assert.calledWithExactly(spy.getCall(1), 1);
                sinon.assert.calledWithExactly(spy.getCall(2), 2);
                sinon.assert.calledWithExactly(spy.getCall(6), 6);
            });
        });

        describe('method "createTransformIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('createTransformIterator');
            });
            it('should create a transformation iterator', function () {
                function sqrt(v) { return (v < 0) ? null : (v === 0) ? { done: true } : { value: Math.sqrt(v) }; }
                var spy = sinon.spy(sqrt), context = {};
                var arrayIt = IteratorUtils.createArrayIterator([36, -4, 4, -36, 0, 1]);
                var transformIt = IteratorUtils.createTransformIterator(arrayIt, spy, context);
                expect(transformIt).to.respondTo('next');
                expect(transformIt.next()).to.deep.equal({ done: false, value: 6 });
                expect(transformIt.next()).to.deep.equal({ done: false, value: 2 });
                expect(transformIt.next()).to.deep.equal({ done: true });
                sinon.assert.callCount(spy, 5);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 36, { done: false, value: 36, index: 0 });
                sinon.assert.calledWithExactly(spy.getCall(1), -4, { done: false, value: -4, index: 1 });
                sinon.assert.calledWithExactly(spy.getCall(2), 4, { done: false, value: 4, index: 2 });
            });
            it('should create a filter iterator returning an object property', function () {
                var arrayIt = IteratorUtils.createArrayIterator([{ a: 1 }, { a: 0 }, { a: 'abc' }]);
                var transformIt = IteratorUtils.createTransformIterator(arrayIt, 'a');
                expect(transformIt).to.respondTo('next');
                expect(transformIt.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(transformIt.next()).to.deep.equal({ done: false, value: 0, index: 1 });
                expect(transformIt.next()).to.deep.equal({ done: false, value: 'abc', index: 2 });
                expect(transformIt.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "createFilterIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('createFilterIterator');
            });
            it('should create a filter iterator', function () {
                function even(v) { return (v % 2) === 0; }
                var spy = sinon.spy(even), context = {};
                var arrayIt = IteratorUtils.createArrayIterator([6, 3, 2, 7, 0, 4, 1, 5]);
                var filterIt = IteratorUtils.createFilterIterator(arrayIt, spy, context);
                expect(filterIt).to.respondTo('next');
                expect(filterIt.next()).to.deep.equal({ done: false, value: 6, index: 0 });
                expect(filterIt.next()).to.deep.equal({ done: false, value: 2, index: 2 });
                expect(filterIt.next()).to.deep.equal({ done: false, value: 0, index: 4 });
                expect(filterIt.next()).to.deep.equal({ done: false, value: 4, index: 5 });
                expect(filterIt.next()).to.deep.equal({ done: true });
                sinon.assert.callCount(spy, 8);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 6, { done: false, value: 6, index: 0 });
                sinon.assert.calledWithExactly(spy.getCall(1), 3, { done: false, value: 3, index: 1 });
                sinon.assert.calledWithExactly(spy.getCall(2), 2, { done: false, value: 2, index: 2 });
            });
            it('should create a filter iterator checking an object property', function () {
                var arrayIt = IteratorUtils.createArrayIterator([{ a: 1 }, { a: 0 }, {}, { a: 2 }]);
                var filterIt = IteratorUtils.createFilterIterator(arrayIt, 'a');
                expect(filterIt).to.respondTo('next');
                expect(filterIt.next()).to.deep.equal({ done: false, value: { a: 1 }, index: 0 });
                expect(filterIt.next()).to.deep.equal({ done: false, value: { a: 2 }, index: 3 });
                expect(filterIt.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "createReduceIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('createReduceIterator');
            });
            it('should create a reducing iterator', function () {
                function reduce(r1, r2) { return (r1.value === r2.value) ? r1 : null; }
                var spy = sinon.spy(reduce), context = {};
                var arrayIt = IteratorUtils.createArrayIterator([1, 2, 3, 3, 3, 2, 1, 1]);
                var reduceIt = IteratorUtils.createReduceIterator(arrayIt, spy, context);
                expect(reduceIt).to.respondTo('next');
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 1, index: 0 });
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 2, index: 1 });
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 3, index: 2 });
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 2, index: 5 });
                expect(reduceIt.next()).to.deep.equal({ done: false, value: 1, index: 6 });
                expect(reduceIt.next()).to.deep.equal({ done: true });
                sinon.assert.alwaysCalledOn(spy, context);
            });
        });

        describe('method "createNestedIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('createNestedIterator');
            });
            it('should create a nested iterator', function () {
                function createIndexIterator(value) {
                    return (value > 0) ? IteratorUtils.createIndexIterator(value) : null;
                }
                function combineResults(result1, result2) {
                    var v1 = result1.value, v2 = result2.value;
                    return (v1 === 42) ? null : (v1 > 9) ? { done: true } : { value: v1 + ',' + v2 };
                }
                var spy1 = sinon.spy(createIndexIterator), spy2 = sinon.spy(combineResults), context = {};
                var arrayIt = IteratorUtils.createArrayIterator([3, 1, 42, 0, 2, 999, 4]);
                var nestedIt = IteratorUtils.createNestedIterator(arrayIt, spy1, spy2, context);
                expect(nestedIt).to.respondTo('next');
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '3,0' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '3,1' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '3,2' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '1,0' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '2,0' });
                expect(nestedIt.next()).to.deep.equal({ done: false, value: '2,1' });
                expect(nestedIt.next()).to.deep.equal({ done: true });
                sinon.assert.alwaysCalledOn(spy1, context);
                sinon.assert.calledWithExactly(spy1.getCall(0), 3, { done: false, value: 3, index: 0 });
                sinon.assert.calledWithExactly(spy1.getCall(1), 1, { done: false, value: 1, index: 1 });
                sinon.assert.alwaysCalledOn(spy2, context);
            });
        });

        describe('method "createCombinedIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('createCombinedIterator');
            });
            it('should create a combined iterator', function () {
                var it1 = IteratorUtils.createArrayIterator([2, 3]);
                var it2 = IteratorUtils.ALWAYS_DONE_ITERATOR;
                var it3 = IteratorUtils.createArrayIterator([4, 5]);
                var combIt = IteratorUtils.createCombinedIterator(it1, it2, it3);
                expect(combIt).to.respondTo('next');
                expect(combIt.next()).to.deep.equal({ done: false, value: 2, index: 0 });
                expect(combIt.next()).to.deep.equal({ done: false, value: 3, index: 1 });
                expect(combIt.next()).to.deep.equal({ done: false, value: 4, index: 0 });
                expect(combIt.next()).to.deep.equal({ done: false, value: 5, index: 1 });
                expect(combIt.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "createSynchronizedIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('createSynchronizedIterator');
            });
            it('should create a synchronized iterator', function () {
                var it1 = IteratorUtils.createArrayIterator([2, 3, 4, 5, 6, 7, 8]);
                var it2 = IteratorUtils.createArrayIterator([0, 3, 6]);
                var it3 = IteratorUtils.createArrayIterator([-1, -9]);
                var spy1 = sinon.spy(_.identity);
                var spy2 = sinon.spy(_.identity);
                var spy3 = sinon.spy(function (n) { return -n; });
                var context = {};
                var syncIt = IteratorUtils.createSynchronizedIterator([it1, it2, it3], [spy1, spy2, spy3], context);
                expect(syncIt).to.respondTo('next');
                expect(syncIt.next()).to.deep.equal({ done: false, value: 0, index: 0, iteratorIndex: 1 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: -1, index: 0, iteratorIndex: 2 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: 2, index: 0, iteratorIndex: 0 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: 3, index: 1, iteratorIndex: 0 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: 3, index: 1, iteratorIndex: 1 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: 4, index: 2, iteratorIndex: 0 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: 5, index: 3, iteratorIndex: 0 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: 6, index: 4, iteratorIndex: 0 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: 6, index: 2, iteratorIndex: 1 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: 7, index: 5, iteratorIndex: 0 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: 8, index: 6, iteratorIndex: 0 });
                expect(syncIt.next()).to.deep.equal({ done: false, value: -9, index: 1, iteratorIndex: 2 });
                expect(syncIt.next()).to.deep.equal({ done: true });
                sinon.assert.alwaysCalledOn(spy1, context);
                sinon.assert.alwaysCalledOn(spy2, context);
                sinon.assert.alwaysCalledOn(spy3, context);
                sinon.assert.calledWithExactly(spy1.getCall(0), 2, { done: false, value: 2, index: 0 }, 0);
                sinon.assert.calledWithExactly(spy2.getCall(0), 0, { done: false, value: 0, index: 0 }, 1);
                sinon.assert.calledWithExactly(spy3.getCall(0), -1, { done: false, value: -1, index: 0 }, 2);
            });
        });

        describe('method "createParallelIterator"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('createParallelIterator');
            });
            it('should create a parallel iterator', function () {
                var it1 = IteratorUtils.createArrayIterator([0, 2, 6, 8, 10,     12]);
                var it2 = IteratorUtils.createArrayIterator([0, 1, 3,     5,      6, 7]);
                var it3 = IteratorUtils.createArrayIterator([1, 3, 7, 9, 11, 12, 13]);
                var spy1 = sinon.spy(_.identity);
                var spy2 = sinon.spy(function (n) { return 2 * n; });
                var spy3 = sinon.spy(function (n) { return n - 1; });
                var context = {};
                var parallelIt = IteratorUtils.createParallelIterator([it1, it2, it3], [spy1, spy2, spy3], context);
                expect(parallelIt).to.respondTo('next');
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [0, 0, 1], results: [{ done: false, value: 0, index: 0 }, { done: false, value: 0, index: 0 }, { done: false, value: 1, index: 0 }], offset: 0, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [2, 1, 3], results: [{ done: false, value: 2, index: 1 }, { done: false, value: 1, index: 1 }, { done: false, value: 3, index: 1 }], offset: 2, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [6, 3, 7], results: [{ done: false, value: 6, index: 2 }, { done: false, value: 3, index: 2 }, { done: false, value: 7, index: 2 }], offset: 6, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [8, null, 9], results: [{ done: false, value: 8, index: 3 }, null, { done: false, value: 9, index: 3 }], offset: 8, complete: false });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [10, 5, 11], results: [{ done: false, value: 10, index: 4 }, { done: false, value: 5, index: 3 }, { done: false, value: 11, index: 4 }], offset: 10, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [null, null, 12], results: [null, null, { done: false, value: 12, index: 5 }], offset: 11, complete: false });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [12, 6, 13], results: [{ done: false, value: 12, index: 5 }, { done: false, value: 6, index: 4 }, { done: false, value: 13, index: 6 }], offset: 12, complete: true });
                expect(parallelIt.next()).to.deep.equal({ done: false, value: [null, 7, null], results: [null, { done: false, value: 7, index: 5 }, null], offset: 14, complete: false });
                expect(parallelIt.next()).to.deep.equal({ done: true });
                sinon.assert.alwaysCalledOn(spy1, context);
                sinon.assert.alwaysCalledOn(spy2, context);
                sinon.assert.alwaysCalledOn(spy3, context);
                sinon.assert.calledWithExactly(spy1.getCall(0), 0, { done: false, value: 0, index: 0 }, 0);
                sinon.assert.calledWithExactly(spy2.getCall(0), 0, { done: false, value: 0, index: 0 }, 1);
                sinon.assert.calledWithExactly(spy3.getCall(0), 1, { done: false, value: 1, index: 0 }, 2);
            });
        });

        describe('method "toArray"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('toArray');
            });
            it('should collect all values of an iterator', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3]);
                expect(IteratorUtils.toArray(it)).to.deep.equal([1, 2, 3]);
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('forEach');
            });
            it('should visit all steps of an iterator', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3]), spy = sinon.spy(), context = {};
                expect(IteratorUtils.forEach(it, spy, context)).to.equal(undefined);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 1, { value: 1, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), 2, { value: 2, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 3, { value: 3, index: 2, done: false });
            });
            it('should break if callback returns Utils.BREAK', function () {
                var it = IteratorUtils.createArrayIterator([1, 2, 3], { reverse: true });
                var stub = sinon.stub();
                stub.onSecondCall().returns(Utils.BREAK);
                expect(IteratorUtils.forEach(it, stub)).to.equal(Utils.BREAK);
                sinon.assert.callCount(stub, 2);
                sinon.assert.calledWithExactly(stub.getCall(0), 3, { value: 3, index: 2, done: false });
                sinon.assert.calledWithExactly(stub.getCall(1), 2, { value: 2, index: 1, done: false });
            });
        });

        describe('method "some"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('some');
            });
            it('should visit an iterator until it finds a truthy value', function () {
                var it = IteratorUtils.createArrayIterator([0, false, 2, '', true]), spy = sinon.spy(_.identity), context = {};
                expect(IteratorUtils.some(it, spy, context)).to.equal(true);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 0, { value: 0, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), false, { value: false, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 2, { value: 2, index: 2, done: false });
            });
            it('should visit all falsy values', function () {
                var it = IteratorUtils.createArrayIterator([0, false, '']), spy = sinon.spy(_.identity), context = {};
                expect(IteratorUtils.some(it, spy, context)).to.equal(false);
                sinon.assert.callCount(spy, 3);
            });
            it('should return false for an empty iterator', function () {
                expect(IteratorUtils.some(IteratorUtils.ALWAYS_DONE_ITERATOR, _.identity)).to.equal(false);
            });
        });

        describe('method "every"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('every');
            });
            it('should visit an iterator until it finds a falsy value', function () {
                var it = IteratorUtils.createArrayIterator([1, true, 0, 'abc', false]), spy = sinon.spy(_.identity), context = {};
                expect(IteratorUtils.every(it, spy, context)).to.equal(false);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 1, { value: 1, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), true, { value: true, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 0, { value: 0, index: 2, done: false });
            });
            it('should visit all truthy values', function () {
                var it = IteratorUtils.createArrayIterator([1, true, 'abc']), spy = sinon.spy(_.identity), context = {};
                expect(IteratorUtils.every(it, spy, context)).to.equal(true);
                sinon.assert.callCount(spy, 3);
            });
            it('should return true for an empty iterator', function () {
                expect(IteratorUtils.every(IteratorUtils.ALWAYS_DONE_ITERATOR, _.identity)).to.equal(true);
            });
        });

        describe('method "map"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('map');
            });
            it('should return all results of the callback function', function () {
                var it = IteratorUtils.createArrayIterator([2, 3, 4]), spy = sinon.spy(function (a) { return a * a; }), context = {};
                expect(IteratorUtils.map(it, spy, context)).to.deep.equal([4, 9, 16]);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 2, { value: 2, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), 3, { value: 3, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 4, { value: 4, index: 2, done: false });
            });
        });

        describe('method "reduce"', function () {
            it('should exist', function () {
                expect(IteratorUtils).itself.to.respondTo('reduce');
            });
            it('should visit all steps of an iterator', function () {
                var it = IteratorUtils.createArrayIterator([2, 3, 4]), spy = sinon.spy(function (a, b) { return a + b; }), context = {};
                expect(IteratorUtils.reduce(1, it, spy, context)).to.equal(10);
                sinon.assert.callCount(spy, 3);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 1, 2, { value: 2, index: 0, done: false });
                sinon.assert.calledWithExactly(spy.getCall(1), 3, 3, { value: 3, index: 1, done: false });
                sinon.assert.calledWithExactly(spy.getCall(2), 6, 4, { value: 4, index: 2, done: false });
            });
        });
    });

    // ========================================================================
});
