/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'json!io.ox/office/tk/locale/localedata'
], function (LOCALE_DATA) {

    'use strict';

    // LocaleData JSON data ===================================================

    describe('Toolkit JSON locale data', function () {

        it('should exist', function () {
            expect(LOCALE_DATA).to.be.an('object');
            expect(LOCALE_DATA).to.not.be.empty;
        });

        // contents -----------------------------------------------------------

        describe('entry', function () {
            it('en_US should exist', function () {
                expect(LOCALE_DATA).to.have.a.property('en_US');
            });
            it('de_DE should exist', function () {
                expect(LOCALE_DATA).to.have.a.property('de_DE');
            });
        });

        describe('entries', function () {
            it('should have a valid locale code', function () {
                _.each(LOCALE_DATA, function (data, locale) {
                    expect(locale).to.match(/^[a-z]{2,}_[A-Z]{2,}$/);
                });
            });
            it('should be objects', function () {
                _.keys(LOCALE_DATA).forEach(function (key) {
                    expect(LOCALE_DATA).to.have.a.property(key).that.is.an('object');
                });
            });
            it('should contain a valid MS locale identifier', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('lcid').that.is.a('number');
                    expect(data.lcid).to.be.above(0);
                });
            });
            it('should contain valid separators', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('dec').that.is.a('string');
                    expect(data.dec).to.have.length(1);
                    expect(data).to.have.a.property('group').that.is.a('string');
                    expect(data.group).to.have.length(1);
                });
            });
            it('should contain a valid writing direction', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('dir').that.is.a('string');
                    expect(data.dir).to.match(/^(ltr|rtl)$/);
                });
            });
            it('should contain a valid CJK flag', function () {
                _.each(LOCALE_DATA, function (data) {
                    if ('cjk' in data) { expect(data.cjk).to.equal(true); }
                });
            });
            it('should contain a valid measurement unit', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('unit').that.is.a('string');
                    expect(data.unit).to.match(/^(cm|mm|in)$/);
                });
            });
            it('should contain valid currency symbols', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('currency').that.is.a('string');
                    expect(data.currency).to.have.length.at.least(1);
                    expect(data).to.have.a.property('isoCurrency').that.is.a('string');
                    expect(data.isoCurrency).to.match(/^[A-Z]{3}$/);
                });
            });
            it('should contain valid date/time formats', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('shortDate').that.is.a('string');
                    expect(data.shortDate).to.have.length.at.least(1);
                    expect(data).to.have.a.property('longDate').that.is.a('string');
                    expect(data.longDate).to.have.length.at.least(1);
                    expect(data).to.have.a.property('shortTime').that.is.a('string');
                    expect(data.shortTime).to.have.length.at.least(1);
                    expect(data).to.have.a.property('longTime').that.is.a('string');
                    expect(data.longTime).to.have.length.at.least(1);
                });
            });
            it('should contain translated format code tokens', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('tokens').that.is.a('string');
                    expect(data.tokens).to.have.length(9);
                    expect(data.tokens).to.not.match(/(.).*\1/); // no duplicate characters
                });
            });
            it('should contain quarter strings', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('quarter').that.is.an('array');
                    expect(data.quarter).to.have.length(4);
                    data.quarter.forEach(function (quarterString) {
                        expect(quarterString).to.be.a('string');
                        expect(quarterString).to.have.length.at.least(1);
                    });
                    expect(data).to.have.a.property('quarterAbbr').that.is.an('array');
                    expect(data.quarterAbbr).to.have.length(4);
                    data.quarterAbbr.forEach(function (quarterString) {
                        expect(quarterString).to.be.a('string');
                        expect(quarterString).to.have.length.at.least(1);
                    });
                });
            });
        });
    });

    // ========================================================================
});
