/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/components/comment/commentlayer',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position'
], function (AppHelper, CommentLayer, DOM, Position) {

    'use strict';

    // class CommentLayer ================================================

    describe('Text class CommentLayer', function () {

        // private helpers ----------------------------------------------------

        var firstParagraph = null,
            commentNode = null,
            commentReplyNode = null,
            commentPlaceHolderNode = null,
            commentThreadNode = null,
            bubbleLayerNode = null,

            //  the operations to be applied by the document model
            OPERATIONS = [
                { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 }, character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' }, paragraph: { lineHeight: { type: 'percent', value: 115 }, marginBottom: 352 } } }
            ],

            model = null,
            commentLayer = null;

        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            commentLayer = model.getCommentLayer();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(CommentLayer).to.be.a('function');
        });

        // constructor --------------------------------------------------------
        describe('constructor', function () {
            it('should create a CommentLayer class instance', function () {
                expect(commentLayer).to.be.an['instanceof'](CommentLayer);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "DOM.isImplicitParagraphNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isImplicitParagraphNode');
            });
            it('should be true for the first paragraph in the document', function () {
                firstParagraph = Position.getParagraphElement(model.getNode(), [0]);
                expect(DOM.isImplicitParagraphNode(firstParagraph)).to.equal(true);
            });
        });

        describe('method "DOM.getCommentLayerNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('getCommentLayerNode');
            });
            it('should return no comment layer node after loading an empty document', function () {
                expect(DOM.getCommentLayerNode(model.getNode()).length).to.equal(0);
            });
        });

        describe('method "DOM.hasCommentPlaceHolderNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('hasCommentPlaceHolderNode');
            });
            it('should return no comment layer node after loading an empty document', function () {
                expect(DOM.hasCommentPlaceHolderNode(firstParagraph)).to.equal(false);
            });
        });

        describe('method "commentLayer.insertComment"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('insertComment');
            });
            it('should insert a new comment into the implicit paragraph', function () {
                commentLayer.insertComment();
                firstParagraph = Position.getParagraphElement(model.getNode(), [0]);
                expect(DOM.isImplicitParagraphNode(firstParagraph)).to.equal(false);
                expect(DOM.hasCommentPlaceHolderNode(firstParagraph)).to.equal(true);
                expect(DOM.getCommentLayerNode(model.getNode()).length).to.equal(1);
            });
        });

        describe('method "commentLayer.getCommentLayer"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('getCommentLayer');
            });
            it('should return the comment layer node below the page node', function () {
                expect(DOM.isCommentLayerNode(commentLayer.getCommentLayer())).to.equal(true);
            });
        });

        describe('method "commentLayer.getOrCreateCommentLayerNode"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('getOrCreateCommentLayerNode');
            });
            it('should return the comment layer node below the page node', function () {
                expect(DOM.isCommentLayerNode(commentLayer.getOrCreateCommentLayerNode())).to.equal(true);
            });
        });

        describe('method "DOM.isCommentThreadNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isCommentThreadNode');
            });
            it('should return true for the first and only child of the comment layer node', function () {
                commentThreadNode = DOM.getCommentLayerNode(model.getNode()).children();
                expect(commentThreadNode.length).to.equal(1);
                expect(DOM.isCommentThreadNode(commentThreadNode)).to.equal(true);
            });
        });

        describe('method "DOM.isCommentNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isCommentNode');
            });
            it('should return true for the first and only child of the comment thread node', function () {
                commentNode = commentThreadNode.children();
                expect(commentNode.length).to.equal(1);
                expect(DOM.isCommentNode(commentNode)).to.equal(true);
                expect(commentNode.children().length).to.equal(3); // three children in each comment node
            });
            it('should have one non implicit paragraph in the comment node', function () {
                var commentParagraph = commentNode.find(DOM.PARAGRAPH_NODE_SELECTOR);
                expect(commentParagraph.length).to.equal(1);
                expect(DOM.isImplicitParagraphNode(commentParagraph)).to.equal(false);
            });
        });

        describe('method "DOM.getCommentPlaceHolderNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('getCommentPlaceHolderNode');
            });
            it('should return the comment node from its place holder node and vice versa', function () {
                commentPlaceHolderNode = $(firstParagraph).children(DOM.COMMENTPLACEHOLDER_NODE_SELECTOR);
                expect(commentPlaceHolderNode.length).to.equal(1);
                expect(DOM.getCommentPlaceHolderNode(commentPlaceHolderNode)).to.equal(commentNode[0]);
                expect(DOM.getCommentPlaceHolderNode(commentNode)).to.equal(commentPlaceHolderNode[0]);
            });
        });

        describe('method "DOM.getTargetContainerId"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('getTargetContainerId');
            });
            it('should return the id of a specified comment', function () {
                expect(DOM.getTargetContainerId(commentNode)).to.equal('cmt0');
                expect(DOM.getTargetContainerId(commentPlaceHolderNode)).to.equal('cmt0');
                expect(model.getActiveTarget()).to.equal('cmt0');
            });
        });

        describe('method "commentLayer.replyToComment"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('replyToComment');
            });
            it('should create a new comment as reply to a specified comment', function () {
                commentLayer.replyToComment(commentNode);
                expect(commentThreadNode.children().length).to.equal(2);
                commentReplyNode = commentThreadNode.children().last();
                expect(DOM.isCommentNode(commentReplyNode)).to.equal(true);
                expect(DOM.getTargetContainerId(commentReplyNode)).to.equal('cmt1');
                expect(commentReplyNode.children().length).to.equal(3); // three children in each comment node
                expect(model.getActiveTarget()).to.equal('cmt1');
            });
        });

        describe('method "DOM.isParentCommentNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isParentCommentNode');
            });
            it('should identify the first comment node as parent comment', function () {
                expect(DOM.isParentCommentNode(commentNode)).to.equal(true);
                expect(DOM.isParentCommentNode(commentReplyNode)).to.equal(false);
            });
        });

        describe('method "DOM.isChildCommentNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isChildCommentNode');
            });
            it('should identify the second comment node as child comment', function () {
                expect(DOM.isChildCommentNode(commentNode)).to.equal(false);
                expect(DOM.isChildCommentNode(commentReplyNode)).to.equal(true);
            });
        });

        describe('method "DOM.getCommentMetaInfoNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('getCommentMetaInfoNode');
            });
            it('should return the one and only meta info node for each comment', function () {
                expect(DOM.getCommentMetaInfoNode(commentNode).length).to.equal(1);
                expect(DOM.getCommentMetaInfoNode(commentReplyNode).length).to.equal(1);
            });
        });

        describe('method "commentLayer.selectNextComment"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('selectNextComment');
            });
            it('should select the next comment as active target', function () {
                expect(model.getActiveTarget()).to.equal('cmt1');
                commentLayer.selectNextComment();
                expect(model.getActiveTarget()).to.equal('cmt0');
                commentLayer.selectNextComment();
                expect(model.getActiveTarget()).to.equal('cmt1');
            });
        });

        describe('method "commentLayer.isCommentTarget"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('isCommentTarget');
            });
            it('should detect a specified target as comment target', function () {
                expect(commentLayer.isCommentTarget('cmt0')).to.equal(true);
                expect(commentLayer.isCommentTarget('cmt1')).to.equal(true);
                expect(commentLayer.isCommentTarget('cmt2')).to.equal(false);
            });
        });

        describe('method "commentLayer.getComments"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('getComments');
            });
            it('should return the collector for all comments', function () {
                expect(commentLayer.getComments().length).to.equal(2);
            });
        });

        describe('method "commentLayer.getCommentRootNode"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('getCommentRootNode');
            });
            it('should return the comment node corresponding to a specified target string', function () {
                expect(commentLayer.getCommentRootNode('cmt0')[0]).to.equal(commentNode[0]);
                expect(commentLayer.getCommentRootNode('cmt1')[0]).to.equal(commentReplyNode[0]);
                expect(commentLayer.getCommentRootNode('cmt2')).to.equal(null);
            });
        });

        describe('method "DOM.getCommentBubbleLayerNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('getCommentBubbleLayerNode');
            });
            it('should return the one and only comment bubble layer node', function () {
                bubbleLayerNode = DOM.getCommentBubbleLayerNode(model.getNode());
                expect(bubbleLayerNode.length).to.equal(1);
            });
        });

        describe('method "commentLayer.getDisplayMode"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('getDisplayMode');
            });
        });

        describe('method "commentLayer.switchCommentDisplayMode"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('switchCommentDisplayMode');
            });
            it('should switch to bubble mode for the comment layer', function () {
                commentLayer.switchCommentDisplayMode(CommentLayer.DISPLAY_MODE.BUBBLES);
                expect(commentLayer.getDisplayMode()).to.equal(CommentLayer.DISPLAY_MODE.BUBBLES);
                expect(bubbleLayerNode.children().length).to.equal(1);
            });
            it('should switch to selected mode for the comment layer', function () {
                commentLayer.switchCommentDisplayMode(CommentLayer.DISPLAY_MODE.SELECTED);
                expect(commentLayer.getDisplayMode()).to.equal(CommentLayer.DISPLAY_MODE.SELECTED);
                expect(bubbleLayerNode.children().length).to.equal(0);
            });
        });

        describe('method "commentLayer.deleteComment"', function () {
            it('should exist', function () {
                expect(commentLayer).to.respondTo('deleteComment');
            });
            it('should delete the parent comment and its child', function () {
                commentLayer.deleteComment(commentNode);
                expect(commentLayer.getCommentLayer()).to.equal(null);
                expect(commentLayer.getOrCreateCommentLayerNode().children().length).to.equal(0);
            });
        });
    });

    // ========================================================================
});
