/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/sheetselection'
], function (SheetHelper, SheetSelection) {

    'use strict';

    // class SheetSelection ===================================================

    describe('Spreadsheet class SheetSelection', function () {

        it('should exist', function () {
            expect(SheetSelection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var a = SheetHelper.a;
        var r = SheetHelper.r;
        var ra = SheetHelper.ra;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            var ranges = ra('B3:D5 D5:F7'), range = r('D5:F7'), address = a('D5'), drawings = [[1, 2], [1, 3]];
            it('should create a selection', function () {
                var s = new SheetSelection(ranges, 1, address, drawings);
                expect(s).to.have.a.property('ranges', ranges);
                expect(s).to.have.a.property('active', 1);
                expect(s).to.have.a.property('address', address);
                expect(s).to.have.a.property('origin', null);
                expect(s).to.have.a.property('drawings', drawings);
            });
            it('should accept single range address', function () {
                var s = new SheetSelection(range, 0, address, drawings);
                expect(s).to.have.a.property('ranges');
                expect(s.ranges).to.have.length(1);
                expect(s.ranges[0]).to.equal(range);
                expect(s).to.have.a.property('active', 0);
                expect(s).to.have.a.property('address', address);
                expect(s).to.have.a.property('origin', null);
                expect(s).to.have.a.property('drawings', drawings);
            });
            it('should accept missing drawings', function () {
                var s = new SheetSelection(ranges, 1, address);
                expect(s).to.have.a.property('ranges', ranges);
                expect(s).to.have.a.property('active', 1);
                expect(s).to.have.a.property('address', address);
                expect(s).to.have.a.property('origin', null);
                expect(s).to.have.a.property('drawings');
                expect(s.drawings).to.be.an('array');
                expect(s.drawings).to.be.empty;
            });
            it('should accept missing active cell', function () {
                var s = new SheetSelection(ranges, 1);
                expect(s).to.have.a.property('ranges', ranges);
                expect(s).to.have.a.property('active', 1);
                expect(s).to.have.a.property('address');
                expect(s.address).to.not.equal(ranges[1].start);
                expect(s.address).to.deep.equal(ranges[1].start);
                expect(s).to.have.a.property('origin', null);
                expect(s).to.have.a.property('drawings');
                expect(s.drawings).to.be.an('array');
                expect(s.drawings).to.be.empty;
            });
            it('should accept missing active range index', function () {
                var s = new SheetSelection(ranges);
                expect(s).to.have.a.property('ranges', ranges);
                expect(s).to.have.a.property('active', 0);
                expect(s).to.have.a.property('address');
                expect(s.address).to.not.equal(ranges[0].start);
                expect(s.address).to.deep.equal(ranges[0].start);
                expect(s).to.have.a.property('origin', null);
                expect(s).to.have.a.property('drawings');
                expect(s.drawings).to.be.an('array');
                expect(s.drawings).to.be.empty;
            });
            it('should create a default selection', function () {
                var s = new SheetSelection();
                expect(s).to.have.a.property('ranges');
                expect(s.ranges).to.be.empty;
                expect(s).to.have.a.property('active', -1);
                expect(s).to.have.a.property('address');
                expect(s.address).to.deep.equal(a('A1'));
                expect(s).to.have.a.property('origin', null);
                expect(s).to.have.a.property('drawings');
                expect(s.drawings).to.be.an('array');
                expect(s.drawings).to.be.empty;
            });
        });

        // static methods -----------------------------------------------------

        describe('method "createFromAddress"', function () {
            it('should exist', function () {
                expect(SheetSelection).itself.to.respondTo('createFromAddress');
            });
            it('should create a selection', function () {
                var s = SheetSelection.createFromAddress(a('C4'));
                expect(s).to.deep.equal(new SheetSelection(r('C4:C4'), 0, a('C4')));
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(SheetSelection).to.respondTo('clone');
            });
            it('should return a deep clone', function () {
                var ranges = ra('B3:D5 D5:F7'), address = a('B3'), origin = a('C4'), drawings = [[1, 2], [1, 3]];
                var s1 = new SheetSelection(ranges, 1, address, drawings);
                s1.origin = origin;
                var s2 = s1.clone();
                expect(s2).to.be.an.instanceof(SheetSelection);
                expect(s2).to.not.equal(s1);
                expect(s2).to.deep.equal(s1);
                expect(s2.ranges).to.not.equal(s1.ranges);
                expect(s2.address).to.not.equal(s1.address);
                expect(s2.origin).to.not.equal(s1.origin);
                expect(s2.drawings).to.not.equal(s1.drawings);
            });
        });

        describe('method "activeRange"', function () {
            it('should exist', function () {
                expect(SheetSelection).to.respondTo('activeRange');
            });
            it('should return the active range', function () {
                var ranges = ra('B3:D5 D5:F7'), s = new SheetSelection(ranges, 0);
                expect(s.activeRange()).to.equal(ranges[0]);
                s.active = 1;
                expect(s.activeRange()).to.equal(ranges[1]);
            });
            it('should return null for empty selection', function () {
                var s = new SheetSelection();
                expect(s.activeRange()).to.equal(null);
            });
        });

        describe('method "containsAddress"', function () {
            it('should exist', function () {
                expect(SheetSelection).to.respondTo('containsAddress');
            });
            it('should return the correct result', function () {
                var ranges = ra('B3:D5 D5:F7'), s = new SheetSelection(ranges, 0);
                expect(s.containsAddress(a('B3'))).to.equal(true);
                expect(s.containsAddress(a('C4'))).to.equal(true);
                expect(s.containsAddress(a('D5'))).to.equal(true);
                expect(s.containsAddress(a('E6'))).to.equal(true);
                expect(s.containsAddress(a('F7'))).to.equal(true);
                expect(s.containsAddress(a('A2'))).to.equal(false);
                expect(s.containsAddress(a('G8'))).to.equal(false);
                expect(s.containsAddress(a('E4'))).to.equal(false);
                expect(s.containsAddress(a('C6'))).to.equal(false);
            });
        });

        describe('method "isActive"', function () {
            it('should exist', function () {
                expect(SheetSelection).to.respondTo('isActive');
            });
            it('should return the correct result', function () {
                var ranges = ra('B3:D5 D5:F7'), address = a('D5'), s = new SheetSelection(ranges, 0, address);
                expect(s.isActive(a('B3'))).to.equal(false);
                expect(s.isActive(a('C4'))).to.equal(false);
                expect(s.isActive(a('D5'))).to.equal(true);
                expect(s.isActive(a('E6'))).to.equal(false);
                expect(s.isActive(a('F7'))).to.equal(false);
                expect(s.isActive(a('A2'))).to.equal(false);
                expect(s.isActive(a('G8'))).to.equal(false);
                expect(s.isActive(a('E4'))).to.equal(false);
                expect(s.isActive(a('C6'))).to.equal(false);
            });
            it('should return false for empty selection', function () {
                expect(new SheetSelection().isActive(a('B3'))).to.equal(false);
            });
        });
    });

    // ========================================================================
});
