/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/rangeset'
], function (SheetHelper, RangeArray, RangeSet) {

    'use strict';

    // class RangeSet =========================================================

    describe('Spreadsheet class RangeSet', function () {

        it('should exist', function () {
            expect(RangeSet).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var a = SheetHelper.a;
        var r = SheetHelper.r;
        var ra = SheetHelper.ra;
        var unorderedMatcher = SheetHelper.unorderedRangesMatcher;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a range set', function () {
                var set = new RangeSet();
                expect(set).to.be.an.instanceof(RangeSet);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "empty"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('empty');
            });
            it('should return true for an empty set', function () {
                var set = new RangeSet();
                expect(set.empty()).to.equal(true);
            });
        });

        describe('method "clear"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('clear');
            });
            it('should clear the set', function () {
                var set = new RangeSet();
                set.insert(r('A2:B3'));
                expect(set.empty()).to.equal(false);
                expect(set.clear()).to.equal(set);
                expect(set.empty()).to.equal(true);
            });
        });

        describe('method "values"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('values');
            });
            it('should return all values', function () {
                var set = new RangeSet();
                var r1 = r('A2:B3'), r2 = r('D5:G8'), r3 = r('I10:J11');
                set.insert(r1);
                set.insert(r2);
                set.insert(r3);
                var values = set.values();
                expect(values).to.be.an.instanceof(RangeArray);
                expect(values).to.have.length(3);
                expect(values.indexOf(r1)).to.be.at.least(0);
                expect(values.indexOf(r2)).to.be.at.least(0);
                expect(values.indexOf(r3)).to.be.at.least(0);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('clone');
            });
            var set = new RangeSet();
            var r1 = r('A2:B3'), r2 = r('D5:G8'), r3 = r('I10:J11');
            set.insert(r1);
            set.insert(r2);
            set.insert(r3);
            it('should create a shallow clone', function () {
                var clone = set.clone();
                expect(clone).to.be.an.instanceof(RangeSet);
                var values1 = set.values().sort();
                var values2 = clone.values().sort();
                expect(values2).to.deep.equal(values1);
                expect(values2[0]).to.equal(values1[0]);
            });
            it('should create a deep clone', function () {
                var clone = set.clone(true);
                expect(clone).to.be.an.instanceof(RangeSet);
                var values1 = set.values().sort();
                var values2 = clone.values().sort();
                expect(values2).to.deep.equal(values1);
                expect(values2[0]).to.not.equal(values1[0]);
            });
            it('should create a deep clone with callback', function () {
                var context = {}, spy = sinon.spy(), clone = set.clone(spy, context);
                expect(clone).to.be.an.instanceof(RangeSet);
                var values1 = set.values().sort();
                var values2 = clone.values().sort();
                expect(values2).to.deep.equal(values1);
                expect(values2[0]).to.not.equal(values1[0]);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.callCount(spy, 3);
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('has');
            });
            it('should return false for an empty set', function () {
                var set = new RangeSet();
                expect(set.has(r('A1:J20'))).to.equal(false);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('get');
            });
            it('should return null for an empty set', function () {
                var set = new RangeSet();
                expect(set.get(r('A1:J20'))).to.equal(null);
            });
        });

        describe('method "containsAddress"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('containsAddress');
            });
            it('should return false for an empty set', function () {
                var set = new RangeSet();
                expect(set.containsAddress(a('A1'))).to.equal(false);
                expect(set.containsAddress(a('J20'))).to.equal(false);
            });
        });

        describe('method "findByAddress"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('findByAddress');
            });
            it('should return an empty array for an empty set', function () {
                var set = new RangeSet();
                expect(set.findByAddress(a('A1'))).to.deep.equal(ra());
                expect(set.findByAddress(a('J20'))).to.deep.equal(ra());
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('forEach');
            });
            it('should visit all values', function () {
                var set = new RangeSet();
                set.insert(r('A2:B3'));
                set.insert(r('D5:G8'));
                set.insert(r('I10:J11'));
                var spy = sinon.spy(), context = {};
                set.forEach(spy, context);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.callCount(spy, 3);
            });
        });

        describe('method "iterator"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('iterator');
            });
            it('should visit all ranges', function () {
                var set = new RangeSet();
                set.insert(r('A2:B3'));
                set.insert(r('D5:G8'));
                set.insert(r('I10:J11'));
                var it = set.iterator();
                expect(it).to.respondTo('next');
                expect(it.next().done).to.equal(false);
                expect(it.next().done).to.equal(false);
                expect(it.next().done).to.equal(false);
                expect(it.next().done).to.equal(true);
            });
        });

        describe('method "insert"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('insert');
            });
            var set = new RangeSet();
            it('should insert a new range', function () {
                var r1 = r('A2:C4'), r2 = r('A2:C4');
                expect(set.insert(r1)).to.equal(r1);
                expect(set.has(r1)).to.equal(true);
                expect(set.has(r2)).to.equal(true);
                expect(set.get(r1)).to.equal(r1);
                expect(set.get(r2)).to.equal(r1);
                expect(set._colSet.values().length).to.equal(1);
                expect(set._map.values().length).to.equal(1);
                expect(set.insert(r2)).to.equal(r1);
                expect(set.has(r1)).to.equal(true);
                expect(set.has(r2)).to.equal(true);
                expect(set.get(r1)).to.equal(r1);
                expect(set.get(r2)).to.equal(r1);
                expect(set._colSet.values().length).to.equal(1);
                expect(set._map.values().length).to.equal(1);
                expect(set.insert(r2, true)).to.equal(r2);
                expect(set.has(r1)).to.equal(true);
                expect(set.has(r2)).to.equal(true);
                expect(set.get(r1)).to.equal(r2);
                expect(set.get(r2)).to.equal(r2);
                expect(set._colSet.values().length).to.equal(1);
                expect(set._map.values().length).to.equal(1);
            });
            it('should detect the ranges by address', function () {
                expect(set.containsAddress(a('A1'))).to.equal(false);
                expect(set.containsAddress(a('A2'))).to.equal(true);
                expect(set.containsAddress(a('A3'))).to.equal(true);
                expect(set.containsAddress(a('A4'))).to.equal(true);
                expect(set.containsAddress(a('A5'))).to.equal(false);
                expect(set.containsAddress(a('B3'))).to.equal(true);
                expect(set.containsAddress(a('C1'))).to.equal(false);
                expect(set.containsAddress(a('C2'))).to.equal(true);
                expect(set.containsAddress(a('C3'))).to.equal(true);
                expect(set.containsAddress(a('C4'))).to.equal(true);
                expect(set.containsAddress(a('C5'))).to.equal(false);
                expect(set.containsAddress(a('E6'))).to.equal(false);
            });
            it('should insert and find another range', function () {
                set.insert(r('E6:I10'));
                expect(set.containsAddress(a('E6'))).to.equal(true);
                expect(set.containsAddress(a('I10'))).to.equal(true);
                expect(set.containsAddress(a('J10'))).to.equal(false);
                expect(set.containsAddress(a('I11'))).to.equal(false);
                expect(set._colSet.values().length).to.equal(2);
                expect(set._map.values().length).to.equal(2);
            });
            it('should insert and find multiple ranges', function () {
                set.insert(r('A2:J3'));
                set.insert(r('B18:I18'));
                set.insert(r('C8:H12'));
                set.insert(r('C1:D20'));
                set.insert(r('E2:G19'));
                set.insert(r('J3:J18'));
                expect(set.findByAddress(a('A1'))).to.satisfy(unorderedMatcher(''));
                expect(set.findByAddress(a('A2'))).to.satisfy(unorderedMatcher('A2:C4 A2:J3'));
                expect(set.findByAddress(a('C3'))).to.satisfy(unorderedMatcher('A2:C4 A2:J3 C1:D20'));
                expect(set._colSet.values().length).to.equal(8);
                expect(set._map.values().length).to.equal(8);
            });
        });

        describe('method "remove"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('remove');
            });
            it('should remove ranges', function () {
                var set = new RangeSet();
                var r1 = r('A2:C4');
                set.insert(r1);
                set.insert(r('B3:C4'));
                set.insert(r('C4:E6'));
                expect(set.remove(r('A2:B3'))).to.equal(null);
                expect(set.containsAddress(a('A2'))).to.equal(true);
                expect(set._colSet.values().length).to.equal(3);
                expect(set._map.values().length).to.equal(3);
                expect(set.remove(r('A2:C4'))).to.equal(r1);
                expect(set.containsAddress(a('A2'))).to.equal(false);
                expect(set._colSet.values().length).to.equal(2);
                expect(set._map.values().length).to.equal(2);
            });
        });

        describe('method "merge"', function () {
            it('should exist', function () {
                expect(RangeSet).to.respondTo('merge');
            });
            it('should merge two range sets', function () {
                var set1 = new RangeSet();
                set1.insert(r('A2:B3'));
                set1.insert(r('D5:G8'));
                set1.insert(r('I10:J11'));
                var set2 = new RangeSet();
                var r1 = r('A2:B3');
                set2.insert(r1);
                set2.insert(r('G8:I10'));
                set1.merge(set2);
                var values = set1.values();
                expect(values).to.satisfy(unorderedMatcher('A2:B3 D5:G8 I10:J11 G8:I10'));
                expect(values.indexOf(r1)).to.equal(-1);
                set1.merge(set2, true);
                values = set1.values();
                expect(values).to.satisfy(unorderedMatcher('A2:B3 D5:G8 I10:J11 G8:I10'));
                expect(values.indexOf(r1)).to.be.at.least(0);
            });
        });
    });

    // ========================================================================
});
