/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper'
], function (AppHelper, SheetHelper) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;
    var r = SheetHelper.r;

    // class MergeCollection ==================================================

    describe('Spreadsheet class MergeCollection', function () {

        // the operations to be applied by the document model
        var noBorder        = { style: 'none' },
            defaultBorder   = { style: 'single', width: 26, color: { type: 'auto' } },
            specialBorder   = { style: 'single', width: 56, color: { type: 'auto' } },

            OPERATIONS = [
                { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a1', attrs: { character: { fontName: 'Arial', fontSize: 13, bold: true, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: false, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' }, borderLeft: { style: 'none' }, borderRight: { style: 'none' }, borderTop: { style: 'none' }, borderBottom: { style: 'none' }, borderDown: { style: 'none' }, borderUp: { style: 'none' }, unlocked: false, hidden: false }, styleId: 'Standard' } },
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a2', attrs: { character: { fontName: 'Arial', fontSize: 13, bold: true, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' }, borderLeft: { style: 'none' }, borderRight: { style: 'none' }, borderTop: { style: 'none' }, borderBottom: { style: 'none' }, borderDown: { style: 'none' }, borderUp: { style: 'none' }, unlocked: false, hidden: false }, styleId: 'Standard' } },

                // Multiple Border Autostyles
                // A3 - autostyle: border top/left corner
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a3', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     defaultBorder,
                    borderRight:    noBorder,
                    borderTop:      defaultBorder,
                    borderBottom:   noBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },
                // A4 - autostyle: border top/right corner
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a4', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     noBorder,
                    borderRight:    defaultBorder,
                    borderTop:      defaultBorder,
                    borderBottom:   noBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },
                // A5 - autostyle: border bottom/left corner
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a5', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     defaultBorder,
                    borderRight:    noBorder,
                    borderTop:      noBorder,
                    borderBottom:   defaultBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },
                // A6 - autostyle: border bottom/right corner
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a6', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     noBorder,
                    borderRight:    defaultBorder,
                    borderTop:      noBorder,
                    borderBottom:   defaultBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },
                // A7 - autostyle: border top
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a7', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     noBorder,
                    borderRight:    noBorder,
                    borderTop:      defaultBorder,
                    borderBottom:   noBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },
                // A8 - autostyle: border right
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a8', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     noBorder,
                    borderRight:    defaultBorder,
                    borderTop:      noBorder,
                    borderBottom:   noBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },
                // A9 - autostyle: border bottom
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a9', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     noBorder,
                    borderRight:    noBorder,
                    borderTop:      noBorder,
                    borderBottom:   defaultBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },
                // A10 - autostyle: border left
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a10', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     defaultBorder,
                    borderRight:    noBorder,
                    borderTop:      noBorder,
                    borderBottom:   noBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },
                // A11 - no border
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a11', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     noBorder,
                    borderRight:    noBorder,
                    borderTop:      noBorder,
                    borderBottom:   noBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },
                // A12 - autostyle: special border top
                { name: 'insertAutoStyle', type: 'cell', styleId: 'a12', attrs: { character: { fontName: 'Arial', fontSize: 11, bold: false, italic: false, underline: false, strike: 'none', color: { type: 'scheme', value: 'dark1' } }, apply: { font: true, fill: true, border: true, align: false, number: false, protect: false }, cell: { alignHor: 'auto', alignVert: 'bottom', wrapText: false, formatId: 0, formatCode: 'General', fillColor: { type: 'rgb', value: '00B050' },
                    borderLeft:     noBorder,
                    borderRight:    noBorder,
                    borderTop:      specialBorder,
                    borderBottom:   noBorder,
                    borderDown:     noBorder,
                    borderUp:       noBorder,
                    unlocked: false, hidden: false }, styleId: 'Standard' } },

                { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
                { name: 'changeCells', sheet: 0, start: [0, 4], contents: [
                    { c: [{ v: 1 }, { v: 4 }, { v: 'hier' }] },
                    { c: [{ v: 2 }, { v: 5 }, { v: 'ein' }] },
                    { c: [{ v: 3 }, { v: 6 }, { v: 'string' }] }
                ] },

                { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
                { name: 'changeCells', sheet: 1, start: [4, 4], contents: [
                    { c: [{ v: 1, s: 'a1' }, { v: 4 }, { v: 'hier' }] },
                    { c: [{ v: 2          }, { v: 5 }, { v: 'ein' }] },
                    { c: [{ v: 3          }, { v: 6 }, { v: 'string' }] }
                ] },

                { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
                { name: 'changeCells', sheet: 2, start: [4, 4], contents: [
                    { c: [{ v: 1          }, { v: 4 }, { v: 'hier' }] },
                    { c: [{ v: 2          }, { v: 5 }, { v: 'ein' }] },
                    { c: [{ v: 3, s: 'a1' }, { v: 6 }, { v: 'string' }] }
                ] },

                { name: 'insertSheet', sheet: 3, sheetName: 'Sheet4' },
                { name: 'changeCells', sheet: 3, start: [4, 4], contents: [
                    { c: [{ v: 1, s: 'a3'  }, { v: 4, s: 'a7' }, { v: 'hier',   s: 'a4' }] },
                    { c: [{ v: 2, s: 'a10' }, { v: 5          }, { v: 'ein',    s: 'a8' }] },
                    { c: [{ v: 3, s: 'a5'  }, { v: 6, s: 'a9' }, { v: 'string', s: 'a6' }] }
                ] },

                { name: 'insertSheet', sheet: 4, sheetName: 'Sheet5' },
                { name: 'changeCells', sheet: 4, start: [4, 4], contents: [
                    { c: [{ v: 1, s: 'a3'  }, { v: 4, s: 'a12' }, { v: 'hier',   s: 'a4' }] },
                    { c: [{ v: 2, s: 'a10' }, { v: 5           }, { v: 'ein',    s: 'a8' }] },
                    { c: [{ v: 3, s: 'a5'  }, { v: 6, s: 'a9'  }, { v: 'string', s: 'a6' }] }
                ] },

                { name: 'insertSheet', sheet: 5, sheetName: 'Sheet6' },
                { name: 'changeCells', sheet: 5, start: [4, 4], contents: [
                    { c: [{ v: 1, s: 'a3'  }, { v: 4, s: 'a7'  }, { v: 'hier',   s: 'a4' }] },
                    { c: [{ v: 2, s: 'a10' }, { v: 5           }, { v: 'ein',    s: 'a8' }] },
                    { c: [{ v: 2, s: 'a10' }, { v: 5, s: 'a10' }, { v: 'ein',    s: 'a8' }] },
                    { c: [{ v: 2, s: 'a10' }, { v: 5           }, { v: 'ein',    s: 'a8' }] },
                    { c: [{ v: 3, s: 'a5'  }, { v: 6, s: 'a9'  }, { v: 'string', s: 'a6' }] }
                ] },

                { name: 'insertSheet', sheet: 6, sheetName: 'Sheet7' },
                { name: 'changeCells', sheet: 6, start: [0, 4], contents: [
                    { c: [{ v: 1 }, { v: 4 }, { v: 'hier' }] },
                    { c: [{ v: 2 }, { v: 5 }, { v: 'ein' }] },
                    { c: [{ v: 3 }, { v: 6 }, { v: 'string' }] }
                ] }
            ];

        // initialize test document
        var docModel = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
        });

        // public methods -----------------------------------------------------

        it('should merge', function (done) {
            var sheet           = 0,
                sheetModel      = docModel.getSheetModel(sheet),
                cellCollection  = sheetModel.getCellCollection(),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('A5:C7'),
                mergedRanges    = null,

                cells = [
                    { adr: a('A5'), val_before: 1,          val_after: 1            },
                    { adr: a('A6'), val_before: 2,          val_after: 2            },
                    { adr: a('A7'), val_before: 3,          val_after: 3            },
                    { adr: a('B5'), val_before: 4,          val_after: 4            },
                    { adr: a('B6'), val_before: 5,          val_after: 5            },
                    { adr: a('B7'), val_before: 6,          val_after: 6            },
                    { adr: a('C5'), val_before: 'hier',     val_after: 'hier'       },
                    { adr: a('C6'), val_before: 'ein',      val_after: 'ein'        },
                    { adr: a('C7'), val_before: 'string',   val_after: 'string'     }
                ];

            cells.forEach(function (obj) {
                expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_before);
            });

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(0);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'merge');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_after);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(1);
                expect(mergedRanges[0]).to.deep.equal(doMerge);

                done();
            });
        });

        it('should merge #2', function (done) {
            var sheet           = 0,
                sheetModel      = docModel.getSheetModel(sheet),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('A15:C16'),
                mergedRanges    = null;

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(1);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'merge');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(2);
                expect(mergedRanges[0]).to.deep.equal(r('A5:C7'));
                expect(mergedRanges[1]).to.deep.equal(doMerge);

                done();
            });
        });

        it('should merge with overlapping two mergedRanges', function (done) {
            var sheet           = 0,
                sheetModel      = docModel.getSheetModel(sheet),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('A1:D20'),
                mergedRanges    = null;

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(2);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'merge');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(1);
                expect(mergedRanges[0]).to.deep.equal(doMerge);
                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(2);
                expect(mergedRanges[0]).to.deep.equal(r('A5:C7'));
                expect(mergedRanges[1]).to.deep.equal(r('A15:C16'));

                done();
            });
        });

        it('should unmerge', function (done) {
            var sheet           = 0,
                sheetModel      = docModel.getSheetModel(sheet),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doUnmerge       = r('A1:Z99'),
                mergedRanges    = null;

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(2);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doUnmerge, 'unmerge');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(0);
                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(2);
                expect(mergedRanges[0]).to.deep.equal(r('A5:C7'));
                expect(mergedRanges[1]).to.deep.equal(r('A15:C16'));

                done();
            });
        });

        it('should merge with eye on styles', function (done) {
            var sheet           = 1,
                sheetModel      = docModel.getSheetModel(sheet),
                cellCollection  = sheetModel.getCellCollection(),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('E5:G7'),
                mergedRanges    = null,

                cells = [
                    { adr: a('E5'), before: { v: 1,          s: 'a1'    }, after: { v: 1,          s: 'a2'  } },
                    { adr: a('E6'), before: { v: 2,          s: ''      }, after: { v: 2,          s: 'a2'  } },
                    { adr: a('E7'), before: { v: 3,          s: ''      }, after: { v: 3,          s: 'a2'  } },
                    { adr: a('F5'), before: { v: 4,          s: ''      }, after: { v: 4,          s: 'a2'  } },
                    { adr: a('F6'), before: { v: 5,          s: ''      }, after: { v: 5,          s: 'a2'  } },
                    { adr: a('F7'), before: { v: 6,          s: ''      }, after: { v: 6,          s: 'a2'  } },
                    { adr: a('G5'), before: { v: 'hier',     s: ''      }, after: { v: 'hier',     s: 'a2'  } },
                    { adr: a('G6'), before: { v: 'ein',      s: ''      }, after: { v: 'ein',      s: 'a2'  } },
                    { adr: a('G7'), before: { v: 'string',   s: ''      }, after: { v: 'string',   s: 'a2'  } }
                ];

            cells.forEach(function (obj) {
                expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
            });

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(0);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'merge');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.after.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.after.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(1);
                expect(mergedRanges[0]).to.deep.equal(doMerge);
                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                done();
            });
        });

        it('should merge with eye on styles #2', function (done) {
            var sheet           = 1,
                sheetModel      = docModel.getSheetModel(sheet),
                cellCollection  = sheetModel.getCellCollection(),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('D4:H8'),
                mergedRanges    = null,

                cells = [
                    { adr: a('D4'), before: { v: null,          s: ''      }, after: { v: 1,          s: 'a2'  } },
                    { adr: a('D5'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('D6'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('D7'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('D8'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },

                    { adr: a('E4'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('E5'), before: { v: 1,             s: 'a1'    }, after: { v: null,       s: 'a2'  } },
                    { adr: a('E6'), before: { v: 2,             s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('E7'), before: { v: 3,             s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('E8'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },

                    { adr: a('F4'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('F5'), before: { v: 4,             s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('F6'), before: { v: 5,             s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('F7'), before: { v: 6,             s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('F8'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },

                    { adr: a('G4'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('G5'), before: { v: 'hier',        s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('G6'), before: { v: 'ein',         s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('G7'), before: { v: 'string',      s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('G8'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },

                    { adr: a('H4'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('H5'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('H6'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('H7'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } },
                    { adr: a('H8'), before: { v: null,          s: ''      }, after: { v: null,       s: 'a2'  } }
                ];

            cells.forEach(function (obj) {
                expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
            });

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(0);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'merge');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.after.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.after.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(1);
                expect(mergedRanges[0]).to.deep.equal(doMerge);
                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(0);

                done();
            });
        });

        it('should merge with eye on styles #3', function (done) {
            var sheet           = 2,
                sheetModel      = docModel.getSheetModel(sheet),
                cellCollection  = sheetModel.getCellCollection(),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('D4:H8'),
                mergedRanges    = null,

                cells = [
                    { adr: a('D4'), before: { v: null,          s: ''      }, after: { v: 1,          s: ''  } },
                    { adr: a('D5'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('D6'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('D7'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('D8'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },

                    { adr: a('E4'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('E5'), before: { v: 1,             s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('E6'), before: { v: 2,             s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('E7'), before: { v: 3,             s: 'a1'    }, after: { v: null,       s: ''  } },
                    { adr: a('E8'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },

                    { adr: a('F4'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('F5'), before: { v: 4,             s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('F6'), before: { v: 5,             s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('F7'), before: { v: 6,             s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('F8'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },

                    { adr: a('G4'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('G5'), before: { v: 'hier',        s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('G6'), before: { v: 'ein',         s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('G7'), before: { v: 'string',      s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('G8'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },

                    { adr: a('H4'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('H5'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('H6'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('H7'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } },
                    { adr: a('H8'), before: { v: null,          s: ''      }, after: { v: null,       s: ''  } }
                ];

            cells.forEach(function (obj) {
                // console.log('before: ', obj.adr.toString());
                expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
            });

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(0);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'merge');

            promise = promise.then(function () {
                expect(promise.state()).to.equal('resolved');

                cells.forEach(function (obj) {
                    // console.log('after: ', obj.adr.toString());
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.after.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.after.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(1);
                expect(mergedRanges[0]).to.deep.equal(doMerge);
                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(0);

                done();
            });
        });

        it('should merge with framed border', function (done) {
            var sheet           = 3,
                sheetModel      = docModel.getSheetModel(sheet),
                cellCollection  = sheetModel.getCellCollection(),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('E5:G7'),
                mergedRanges    = null,

                cells = [
                    { adr: a('E5'), before: { v: 1,          s: 'a3'    }, after: { v: 1,          s: 'a3'  } },
                    { adr: a('E6'), before: { v: 2,          s: 'a10'   }, after: { v: 2,          s: 'a10' } },
                    { adr: a('E7'), before: { v: 3,          s: 'a5'    }, after: { v: 3,          s: 'a5'  } },
                    { adr: a('F5'), before: { v: 4,          s: 'a7'    }, after: { v: 4,          s: 'a7'  } },
                    { adr: a('F6'), before: { v: 5,          s: ''      }, after: { v: 5,          s: 'a11' } },
                    { adr: a('F7'), before: { v: 6,          s: 'a9'    }, after: { v: 6,          s: 'a9'  } },
                    { adr: a('G5'), before: { v: 'hier',     s: 'a4'    }, after: { v: 'hier',     s: 'a4'  } },
                    { adr: a('G6'), before: { v: 'ein',      s: 'a8'    }, after: { v: 'ein',      s: 'a8'  } },
                    { adr: a('G7'), before: { v: 'string',   s: 'a6'    }, after: { v: 'string',   s: 'a6'  } }
                ];

            cells.forEach(function (obj) {
                // console.log('before: ', obj.adr.toString());
                expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
            });

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(0);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'merge');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                cells.forEach(function (obj) {
                    // console.log('after: ', obj.adr.toString());
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.after.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.after.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(1);
                expect(mergedRanges[0]).to.deep.equal(doMerge);
                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(0);

                done();
            });
        });

        it('should merge with framed border except the top one', function (done) {
            var sheet           = 4,
                sheetModel      = docModel.getSheetModel(sheet),
                cellCollection  = sheetModel.getCellCollection(),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('E5:G7'),
                mergedRanges    = null,

                cells = [
                    { adr: a('E5'), before: { v: 1,          s: 'a3'    }, after: { v: 1,          s: 'a10' } },
                    { adr: a('E6'), before: { v: 2,          s: 'a10'   }, after: { v: 2,          s: 'a10' } },
                    { adr: a('E7'), before: { v: 3,          s: 'a5'    }, after: { v: 3,          s: 'a5'  } },
                    { adr: a('F5'), before: { v: 4,          s: 'a12'   }, after: { v: 4,          s: 'a11' } },
                    { adr: a('F6'), before: { v: 5,          s: ''      }, after: { v: 5,          s: 'a11' } },
                    { adr: a('F7'), before: { v: 6,          s: 'a9'    }, after: { v: 6,          s: 'a9'  } },
                    { adr: a('G5'), before: { v: 'hier',     s: 'a4'    }, after: { v: 'hier',     s: 'a8'  } },
                    { adr: a('G6'), before: { v: 'ein',      s: 'a8'    }, after: { v: 'ein',      s: 'a8'  } },
                    { adr: a('G7'), before: { v: 'string',   s: 'a6'    }, after: { v: 'string',   s: 'a6'  } }
                ];

            cells.forEach(function (obj) {
                // console.log('before: ', obj.adr.toString());
                expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
            });

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(0);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'merge');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                cells.forEach(function (obj) {
                    // console.log('after: ', obj.adr.toString());
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.after.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.after.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(1);
                expect(mergedRanges[0]).to.deep.equal(doMerge);
                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(0);

                done();
            });
        });

        it('should merge with framed border and reset the middle cells', function (done) {
            var sheet           = 5,
                sheetModel      = docModel.getSheetModel(sheet),
                cellCollection  = sheetModel.getCellCollection(),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('E5:G9'),
                mergedRanges    = null,

                cells = [
                    { adr: a('E5'), before: { v: 1,          s: 'a3'    }, after: { v: 1,          s: 'a3'  } },
                    { adr: a('E6'), before: { v: 2,          s: 'a10'   }, after: { v: 2,          s: 'a10' } },
                    { adr: a('E7'), before: { v: 2,          s: 'a10'   }, after: { v: 2,          s: 'a10' } },
                    { adr: a('E8'), before: { v: 2,          s: 'a10'   }, after: { v: 2,          s: 'a10' } },
                    { adr: a('E9'), before: { v: 3,          s: 'a5'    }, after: { v: 3,          s: 'a5'  } },

                    { adr: a('F5'), before: { v: 4,          s: 'a7'    }, after: { v: 4,          s: 'a7'  } },
                    { adr: a('F6'), before: { v: 5,          s: ''      }, after: { v: 5,          s: 'a11' } },
                    { adr: a('F7'), before: { v: 5,          s: 'a10'   }, after: { v: 5,          s: 'a11' } },
                    { adr: a('F8'), before: { v: 5,          s: ''      }, after: { v: 5,          s: 'a11' } },
                    { adr: a('F9'), before: { v: 6,          s: 'a9'    }, after: { v: 6,          s: 'a9'  } },

                    { adr: a('G5'), before: { v: 'hier',     s: 'a4'    }, after: { v: 'hier',     s: 'a4'  } },
                    { adr: a('G6'), before: { v: 'ein',      s: 'a8'    }, after: { v: 'ein',      s: 'a8'  } },
                    { adr: a('G7'), before: { v: 'ein',      s: 'a8'    }, after: { v: 'ein',      s: 'a8'  } },
                    { adr: a('G8'), before: { v: 'ein',      s: 'a8'    }, after: { v: 'ein',      s: 'a8'  } },
                    { adr: a('G9'), before: { v: 'string',   s: 'a6'    }, after: { v: 'string',   s: 'a6'  } }
                ];

            cells.forEach(function (obj) {
                // console.log('before: ', obj.adr.toString());
                expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
            });

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(0);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'merge');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                cells.forEach(function (obj) {
                    // console.log('after: ', obj.adr.toString());
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.after.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.after.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(1);
                expect(mergedRanges[0]).to.deep.equal(doMerge);
                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.before.v);
                    expect(cellCollection.getCellStyleId(obj.adr)).to.equal(obj.before.s);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);
                expect(mergedRanges.length).to.equal(0);

                done();
            });
        });

        it('should merge horizontal', function (done) {
            var sheet           = 6,
                sheetModel      = docModel.getSheetModel(sheet),
                cellCollection  = sheetModel.getCellCollection(),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('A5:C7'),
                mergedRanges    = null,

                cells = [
                    { adr: a('A5'), val_before: 1,          val_after: 1            },
                    { adr: a('A6'), val_before: 2,          val_after: 2            },
                    { adr: a('A7'), val_before: 3,          val_after: 3            },
                    { adr: a('B5'), val_before: 4,          val_after: 4            },
                    { adr: a('B6'), val_before: 5,          val_after: 5            },
                    { adr: a('B7'), val_before: 6,          val_after: 6            },
                    { adr: a('C5'), val_before: 'hier',     val_after: 'hier'       },
                    { adr: a('C6'), val_before: 'ein',      val_after: 'ein'        },
                    { adr: a('C7'), val_before: 'string',   val_after: 'string'     }
                ];

            cells.forEach(function (obj) {
                expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_before);
            });

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(0);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'horizontal');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                cells.forEach(function (obj) {
                    // console.log('after: ', obj.adr.toString());
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_after);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);

                expect(mergedRanges.length).to.equal(3);

                expect(mergedRanges[0]).to.deep.equal(r('A5:C5'));
                expect(mergedRanges[1]).to.deep.equal(r('A7:C7'));
                expect(mergedRanges[2]).to.deep.equal(r('A6:C6'));

                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                done();
            });
        });

        it('should merge vertical', function (done) {
            var sheet           = 6,
                sheetModel      = docModel.getSheetModel(sheet),
                cellCollection  = sheetModel.getCellCollection(),
                mergeCollection = sheetModel.getMergeCollection(),

                checkRange      = r('A1:Z99'),
                doMerge         = r('A5:C7'),
                mergedRanges    = null,

                cells = [
                    { adr: a('A5'), val_before: 1,          val_after: 1            },
                    { adr: a('A6'), val_before: 2,          val_after: 2            },
                    { adr: a('A7'), val_before: 3,          val_after: 3            },
                    { adr: a('B5'), val_before: 4,          val_after: 4            },
                    { adr: a('B6'), val_before: 5,          val_after: 5            },
                    { adr: a('B7'), val_before: 6,          val_after: 6            },
                    { adr: a('C5'), val_before: 'hier',     val_after: 'hier'       },
                    { adr: a('C6'), val_before: 'ein',      val_after: 'ein'        },
                    { adr: a('C7'), val_before: 'string',   val_after: 'string'     }
                ];

            cells.forEach(function (obj) {
                expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_before);
            });

            mergedRanges = mergeCollection.getMergedRanges(checkRange);
            expect(mergedRanges.length).to.equal(0);

            var generator   = sheetModel.createOperationsGenerator({ applyImmediately: true });
            var promise     = mergeCollection.generateMergeCellsOperations(generator, doMerge, 'vertical');

            promise.always(function () {
                expect(promise.state()).to.equal('resolved');

                cells.forEach(function (obj) {
                    expect(cellCollection.getCellValue(obj.adr)).to.equal(obj.val_after);
                });

                mergedRanges = mergeCollection.getMergedRanges(checkRange);

                expect(mergedRanges.length).to.equal(3);

                expect(mergedRanges[0]).to.deep.equal(r('B5:B7'));
                expect(mergedRanges[1]).to.deep.equal(r('A5:A7'));
                expect(mergedRanges[2]).to.deep.equal(r('C5:C7'));

                expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);

                done();
            });
        });
    });

    // ========================================================================
});
