/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/funcs/engineeringfuncs'
], function (AppHelper, SheetHelper, FormulaUtils, EngineeringFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;

    // module EngineeringFuncs ================================================

    describe('Spreadsheet module EngineeringFuncs', function () {

        // initialize the test
        var ooxAppPromise = AppHelper.createSpreadsheetApp('ooxml');
        var odfAppPromise = AppHelper.createSpreadsheetApp('odf');
        var moduleTester = SheetHelper.createFunctionModuleTester(EngineeringFuncs, ooxAppPromise, odfAppPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('BITAND', function (BITAND) {
            it('should return the binary AND of the numbers', function () {
                expect(BITAND(12, 10)).to.equal(8);
                expect(BITAND(0xFFFFFFFFFFFF, 0x123456789ABC)).to.equal(0x123456789ABC);
            });
            it('should throw #NUM! for invalid numbers', function () {
                expect(BITAND(-1, 1)).to.equal(ErrorCode.NUM);
                expect(BITAND(1, -1)).to.equal(ErrorCode.NUM);
                expect(BITAND(0x1000000000000, 1)).to.equal(ErrorCode.NUM);
                expect(BITAND(1, 0x1000000000000)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('BITLSHIFT', function (BITLSHIFT) {
            it('should return the number shifted to the left', function () {
                expect(BITLSHIFT(9, 0)).to.equal(9);
                expect(BITLSHIFT(9, 1)).to.equal(18);
                expect(BITLSHIFT(9, 2)).to.equal(36);
                expect(BITLSHIFT(9, 10)).to.equal(0x2400);
                expect(BITLSHIFT(9, 44)).to.equal(0x900000000000);
            });
            it('should return the number shifted to the right', function () {
                expect(BITLSHIFT(36, -1)).to.equal(18);
                expect(BITLSHIFT(36, -2)).to.equal(9);
                expect(BITLSHIFT(36, -3)).to.equal(4);
                expect(BITLSHIFT(36, -4)).to.equal(2);
                expect(BITLSHIFT(36, -5)).to.equal(1);
                expect(BITLSHIFT(36, -6)).to.equal(0);
                expect(BITLSHIFT(36, -48)).to.equal(0);
            });
            it('should throw #NUM! for invalid parameters', function () {
                expect(BITLSHIFT(9, 45)).to.equal(ErrorCode.NUM);
                expect(BITLSHIFT(9, 46)).to.equal(ErrorCode.NUM);
                expect(BITLSHIFT(9, 47)).to.equal(ErrorCode.NUM);
                expect(BITLSHIFT(9, 48)).to.equal(ErrorCode.NUM);
                expect(BITLSHIFT(9, 54)).to.equal(ErrorCode.NUM);
                expect(BITLSHIFT(9, -54)).to.equal(ErrorCode.NUM);
            });
            it('should throw zero for specific shift counts', function () {
                expect(BITLSHIFT(9, 49)).to.equal(0);
                expect(BITLSHIFT(9, 53)).to.equal(0);
                expect(BITLSHIFT(36, -49)).to.equal(0);
                expect(BITLSHIFT(36, -53)).to.equal(0);
            });
        });

        moduleTester.testFunction('BITOR', function (BITOR) {
            it('should return the binary OR of the numbers', function () {
                expect(BITOR(12, 10)).to.equal(14);
                expect(BITOR(0xFFFFFFFFFFFF, 0x123456789ABC)).to.equal(0xFFFFFFFFFFFF);
            });
            it('should throw #NUM! for invalid numbers', function () {
                expect(BITOR(-1, 1)).to.equal(ErrorCode.NUM);
                expect(BITOR(1, -1)).to.equal(ErrorCode.NUM);
                expect(BITOR(0x1000000000000, 1)).to.equal(ErrorCode.NUM);
                expect(BITOR(1, 0x1000000000000)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('BITRSHIFT', function (BITRSHIFT) {
            it('should return the number shifted to the right', function () {
                expect(BITRSHIFT(36, 0)).to.equal(36);
                expect(BITRSHIFT(36, 1)).to.equal(18);
                expect(BITRSHIFT(36, 2)).to.equal(9);
                expect(BITRSHIFT(36, 3)).to.equal(4);
                expect(BITRSHIFT(36, 4)).to.equal(2);
                expect(BITRSHIFT(36, 5)).to.equal(1);
                expect(BITRSHIFT(36, 6)).to.equal(0);
                expect(BITRSHIFT(36, 48)).to.equal(0);
            });
            it('should return the number shifted to the left', function () {
                expect(BITRSHIFT(9, -1)).to.equal(18);
                expect(BITRSHIFT(9, -2)).to.equal(36);
                expect(BITRSHIFT(9, -10)).to.equal(0x2400);
                expect(BITRSHIFT(9, -44)).to.equal(0x900000000000);
            });
            it('should throw #NUM! for invalid parameters', function () {
                expect(BITRSHIFT(9, 54)).to.equal(ErrorCode.NUM);
                expect(BITRSHIFT(9, -45)).to.equal(ErrorCode.NUM);
                expect(BITRSHIFT(9, -46)).to.equal(ErrorCode.NUM);
                expect(BITRSHIFT(9, -47)).to.equal(ErrorCode.NUM);
                expect(BITRSHIFT(9, -48)).to.equal(ErrorCode.NUM);
                expect(BITRSHIFT(9, -54)).to.equal(ErrorCode.NUM);
            });
            it('should throw zero for specific shift counts', function () {
                expect(BITRSHIFT(36, 49)).to.equal(0);
                expect(BITRSHIFT(36, 53)).to.equal(0);
                expect(BITRSHIFT(9, -49)).to.equal(0);
                expect(BITRSHIFT(9, -53)).to.equal(0);
            });
        });

        moduleTester.testFunction('BITXOR', function (BITXOR) {
            it('should return the binary XOR of the numbers', function () {
                expect(BITXOR(12, 10)).to.equal(6);
                expect(BITXOR(0xFFFFFFFFFFFF, 0x123456789ABC)).to.equal(0xEDCBA9876543);
            });
            it('should throw #NUM! for invalid numbers', function () {
                expect(BITXOR(-1, 1)).to.equal(ErrorCode.NUM);
                expect(BITXOR(1, -1)).to.equal(ErrorCode.NUM);
                expect(BITXOR(0x1000000000000, 1)).to.equal(ErrorCode.NUM);
                expect(BITXOR(1, 0x1000000000000)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DELTA', function (DELTA) {
            it('should return the correct result', function () {
                expect(DELTA(5, 4)).to.equal(0);
                expect(DELTA(5, 5)).to.equal(1);
                expect(DELTA(0.5, 0)).to.equal(0);
            });
        });

        moduleTester.testFunction('ERF', function (ERF_OOX, ERF_ODF) {
            it('should return the result of the error function', function () {
                expect(ERF_OOX(0)).to.equal(0);
                expect(ERF_OOX(1)).to.equal(FormulaUtils.erf(1));
                expect(ERF_OOX(2)).to.equal(FormulaUtils.erf(2));
                expect(ERF_ODF(0)).to.equal(0);
                expect(ERF_ODF(1)).to.equal(FormulaUtils.erf(1));
                expect(ERF_ODF(2)).to.equal(FormulaUtils.erf(2));
            });
            it('should return the difference of two function results', function () {
                expect(ERF_OOX(1, 1)).to.equal(0);
                expect(ERF_OOX(1, 2)).to.equal(FormulaUtils.erf(2) - FormulaUtils.erf(1));
                expect(ERF_OOX(2, 1)).to.equal(FormulaUtils.erf(1) - FormulaUtils.erf(2));
                expect(ERF_ODF(1, 1)).to.equal(0);
                expect(ERF_ODF(1, 2)).to.equal(FormulaUtils.erf(2) - FormulaUtils.erf(1));
                expect(ERF_ODF(2, 1)).to.equal(FormulaUtils.erf(1) - FormulaUtils.erf(2));
            });
            it('should throw #NUM! error code for negative numbers in OOXML mode', function () {
                expect(ERF_OOX(-1)).to.equal(ErrorCode.NUM);
                expect(ERF_OOX(-2)).to.equal(ErrorCode.NUM);
                expect(ERF_OOX(-1, 2)).to.equal(ErrorCode.NUM);
                expect(ERF_OOX(2, -2)).to.equal(ErrorCode.NUM);
            });
            it('should return a result for negative numbers in ODF mode', function () {
                expect(ERF_ODF(-1)).to.equal(FormulaUtils.erf(-1));
                expect(ERF_ODF(-2)).to.equal(FormulaUtils.erf(-2));
                expect(ERF_ODF(-1, 2)).to.equal(FormulaUtils.erf(2) + FormulaUtils.erf(1));
                expect(ERF_ODF(2, -2)).to.equal(-2 * FormulaUtils.erf(2));
            });
        });

        moduleTester.testFunction('ERF.PRECISE', function (ERF_PRECISE) {
            it('should return the result of the error function', function () {
                expect(ERF_PRECISE(0)).to.equal(0);
                expect(ERF_PRECISE(1)).to.equal(FormulaUtils.erf(1));
                expect(ERF_PRECISE(2)).to.equal(FormulaUtils.erf(2));
            });
            it('should return a result for negative numbers', function () {
                expect(ERF_PRECISE(-1)).to.equal(FormulaUtils.erf(-1));
                expect(ERF_PRECISE(-2)).to.equal(FormulaUtils.erf(-2));
            });
        });

        moduleTester.testFunction('ERFC', function (ERFC_OOX, ERFC_ODF) {
            it('should return the complementary error function', function () {
                expect(ERFC_OOX(0)).to.equal(1);
                expect(ERFC_OOX(1)).to.equal(FormulaUtils.erfc(1));
                expect(ERFC_OOX(2)).to.equal(FormulaUtils.erfc(2));
                expect(ERFC_ODF(0)).to.equal(1);
                expect(ERFC_ODF(1)).to.equal(FormulaUtils.erfc(1));
                expect(ERFC_ODF(2)).to.equal(FormulaUtils.erfc(2));
            });
            it('should throw #NUM! error for negative numbers in OOXML mode', function () {
                expect(ERFC_OOX(-1)).to.equal(ErrorCode.NUM);
                expect(ERFC_OOX(-2)).to.equal(ErrorCode.NUM);
            });
            it('should return a result for negative numbers in ODF mode', function () {
                expect(ERFC_ODF(-1)).to.equal(FormulaUtils.erfc(-1));
                expect(ERFC_ODF(-2)).to.equal(FormulaUtils.erfc(-2));
            });
        });

        moduleTester.testFunction('ERFC.PRECISE', function (ERFC_PRECISE) {
            it('should return the complementary error function', function () {
                expect(ERFC_PRECISE(0)).to.equal(1);
                expect(ERFC_PRECISE(1)).to.equal(FormulaUtils.erfc(1));
                expect(ERFC_PRECISE(2)).to.equal(FormulaUtils.erfc(2));
            });
            it('should return a result for negative numbers', function () {
                expect(ERFC_PRECISE(-1)).to.equal(FormulaUtils.erfc(-1));
                expect(ERFC_PRECISE(-2)).to.equal(FormulaUtils.erfc(-2));
            });
        });

        moduleTester.testFunction('GESTEP', function (GESTEP) {
            it('should return the correct result', function () {
                expect(GESTEP(5, 4)).to.equal(1);
                expect(GESTEP(5, 5)).to.equal(1);
                expect(GESTEP(-4, -5)).to.equal(1);
                expect(GESTEP(-1)).to.equal(0);
                expect(GESTEP(1)).to.equal(1);
            });
        });
    });

    // ========================================================================
});
