/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/presentation/model/objectoperationmixin',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/editframework/utils/attributeutils'
], function (AppHelper, ObjectOperationMixin, DOM, AttributeUtils) {

    'use strict';

    // mix-in class SlideOperationMixin =======================================

    describe('Presentation mix-in class ObjectOperationMixin', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            selection = null,
            layoutId_1 = 'layout1',
            layoutId_2 = 'layout2',
            masterId_1 = 'master1',
            activeSlide = null,
            allDrawings = null,
            drawing1 = null,
            drawing2 = null,
            drawing3 = null,
            drawing4 = null,
            drawing5 = null,
            operationCounter = 0,

            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: {
                        page: { width: 33866, height: 19050, orientation: 'landscape' },
                        defaultTextListStyles: {
                            l1: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 0 } },
                            l2: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 1270 } },
                            l3: { character: { fontSize: 18, fontName: '+mn-lt', color: { type: 'scheme', value: 'text1' } }, paragraph: { defaultTabSize: 2540, alignment: 'left', indentLeft: 2540 } }
                        }
                    }
                },
                { name: 'insertMasterSlide', id: masterId_1 },

                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1', left: 1905, top: 5918, width: 21590, height: 4083 } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_1, text: 'Mastertitelformat bearbeiten' },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Untertitel 2', left: 3810, top: 10795, width: 17780, height: 4868 } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_1, text: 'Master-Untertitelformat bearbeiten' },

                { name: 'insertLayoutSlide', id: layoutId_2, target: masterId_1 },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_2, type: 'shape', attrs: { presentation: { phType: 'title' }, drawing: { name: 'Titel 1' } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_2 },
                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_2, text: 'Mastertitelformat bearbeiten' },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_2, type: 'shape', attrs: { presentation: { phIndex: 1 }, drawing: { name: 'Drawing 2', left: 1270, top: 4445, width: 11218, height: 12572 } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_2, attrs: { paragraph: { level: 0 } } },
                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_2, text: 'Mastertextformat bearbeiten' },

                { name: 'insertSlide', start: [0], target: layoutId_1 },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow left', left: 5027, top: 6725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow up', left: 4027, top: 5725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '00FF00' } } } },
                { name: 'insertParagraph', start: [0, 1, 0] },
                { name: 'insertDrawing', start: [0, 2], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow right', left: 6027, top: 6725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '0000FF' } } } },
                { name: 'insertParagraph', start: [0, 2, 0] },
                { name: 'insertDrawing', start: [0, 3], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow down', left: 4027, top: 7725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF00FF' } } } },
                { name: 'insertParagraph', start: [0, 3, 0] },
                { name: 'insertDrawing', start: [0, 4], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Math Multiply', left: 5027, top: 3725, width: 4419, height: 2509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FFFF00' } } } },
                { name: 'insertParagraph', start: [0, 4, 0] },

                { name: 'insertSlide', start: [1], target: layoutId_1 },
                { name: 'insertDrawing', start: [1, 0], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow left', left: 5027, top: 6725, width: 9419, height: 6000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [1, 0, 0] },
                { name: 'insertDrawing', start: [1, 1], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow up', left: 4027, top: 5000, width: 9419, height: 6000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '00FF00' } } } },
                { name: 'insertParagraph', start: [1, 1, 0] },
                { name: 'insertDrawing', start: [1, 2], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow right', left: 6027, top: 10000, width: 9419, height: 5000 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: '0000FF' } } } },
                { name: 'insertParagraph', start: [1, 2, 0] },
                { name: 'insertDrawing', start: [1, 3], type: 'shape', attrs: { shape: { anchor: 'centered' }, drawing: { name: 'Arrow down', left: 4027, top: 7725, width: 9419, height: 6509 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF00FF' } } } },
                { name: 'insertParagraph', start: [1, 3, 0] }
            ];

        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(ObjectOperationMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "changeDrawingOrder"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('changeDrawingOrder');
            });

            // Check: Starting with cursor selection, single drawing selection and multi drawing selection
            // Check: One step forwards or backwards or completely to foreground or background
            // Check: Keeping the selection after applying the operation
            // Check: Generating only the required operations
            // Check: Undo and redo

            it('should bring the drawing that contains a text selection one step to the top', function () {

                selection.setTextSelection([0, 0, 0, 0]);  // setting the cursor into the first drawing on the first slide

                activeSlide = model.getSlideById(model.getActiveSlideId());

                allDrawings = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);  // getting all drawings on slide

                expect(allDrawings.length).to.equal(5);

                // assigning drawings
                drawing1 = $(allDrawings[0]);
                drawing2 = $(allDrawings[1]);
                drawing3 = $(allDrawings[2]);
                drawing4 = $(allDrawings[3]);
                drawing5 = $(allDrawings[4]);

                // checking order of drawings
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                operationCounter = model.changeDrawingOrder('forward');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // checking merge of text spans
                expect(drawing1.prevAll().length).to.equal(3); // drawing and two text spans

                // checking the new selection after moving the drawing
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 1])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 2])).to.equal(true);

            });

            it('should bring the selected drawing one step to the top', function () {

                // expecting selection [0,1] to [0,2]

                operationCounter = model.changeDrawingOrder('forward');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);

                // checking merge of text spans
                expect(drawing1.prevAll().length).to.equal(5); // two drawings and three text spans

                // checking the new selection after moving the drawing
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 2])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 3])).to.equal(true);

            });

            it('should bring the selected drawing to the top', function () {

                // expecting selection [0,2] to [0,3]

                operationCounter = model.changeDrawingOrder('front');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // checking merge of text spans
                expect(drawing1.prevAll().length).to.equal(9); // four drawings and five text spans
                expect(drawing1.nextAll().length).to.equal(2); // only one text span and the br element

                // checking the new selection after moving the drawing
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 4])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 5])).to.equal(true);

            });

            it('should not create any operation to bring the top drawing further to the top', function () {

                // expecting selection [0,4] to [0,5]

                operationCounter = model.changeDrawingOrder('front');

                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('forward');

                expect(operationCounter).to.equal(0);

            });

            it('should bring the selected drawing to the back', function () {

                operationCounter = model.changeDrawingOrder('back');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);

                // checking merge of text spans
                expect(drawing1.prevAll().length).to.equal(1); // only one text span
                expect(drawing1.nextAll().length).to.equal(10); // four drawings and five text spans and br element

                // checking the new selection after moving the drawing
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 0])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 1])).to.equal(true);

            });

            it('should not create any operation to bring the background drawing further to the background', function () {

                operationCounter = model.changeDrawingOrder('back');

                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('backward');

                expect(operationCounter).to.equal(0);

            });

            it('should undo the operation that brought the upmost drawing to the back', function (done) {

                var // the undo promise
                    promise = null;

                // check that drawing is still a position [0,0]
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 0])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 1])).to.equal(true);
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);

                // bringing drawing1 to the front again
                promise = model.getUndoManager().undo();

                promise.always(function () {

                    // checking move of drawing
                    expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                    // checking merge of text spans
                    expect(drawing1.prevAll().length).to.equal(9); // four drawings and five text spans
                    expect(drawing1.nextAll().length).to.equal(2); // only one text span and the br element

                    // checking the new selection after moving the drawing
                    expect(_.isEqual(model.getSelection().getStartPosition(), [0, 4])).to.equal(true);
                    expect(_.isEqual(model.getSelection().getEndPosition(), [0, 5])).to.equal(true);

                    done();
                });

            });

            it('should redo the operation that brought the background drawing to the top', function (done) {

                var // the redo promise
                    promise = null;

                // check that drawing is still a position [0,0]
                expect(_.isEqual(model.getSelection().getStartPosition(), [0, 4])).to.equal(true);
                expect(_.isEqual(model.getSelection().getEndPosition(), [0, 5])).to.equal(true);
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // bringing drawing1 to the front again
                promise = model.getUndoManager().redo();

                promise.always(function () {

                    // checking move of drawing
                    expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);

                    // checking merge of text spans
                    expect(drawing1.prevAll().length).to.equal(1); // only one text span

                    // checking the new selection after moving the drawing
                    expect(_.isEqual(model.getSelection().getStartPosition(), [0, 0])).to.equal(true);
                    expect(_.isEqual(model.getSelection().getEndPosition(), [0, 1])).to.equal(true);

                    done();
                });

            });

            // starting with multi selection, the drawings are in the correct order from drawing1 to drawing5

            it('should bring the selected drawings one step to the top with two operations', function () {

                // checking order of drawings
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                expect(selection.isMultiSelection()).to.equal(false);

                selection.setMultiDrawingSelectionByPosition([[0, 0], [0, 2]]);  // selecting drawing1 and drawing3

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(2);
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,2)');

                operationCounter = model.changeDrawingOrder('forward');

                expect(operationCounter).to.equal(2);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // checking the new selection after moving the drawing
                expect(selection.getListOfPositions()).to.equal('(0,1),(0,3)');

            });

            it('should undo the operations that brought the two drawing one step to the front', function (done) {

                var // the undo promise
                    promise = null;

                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);

                // bringing drawing1 and drawing 3 one step backwards
                promise = model.getUndoManager().undo();

                promise.always(function () {

                    // checking move of drawing
                    expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                    expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);

                    // checking the new selection after moving the drawing
                    expect(selection.getListOfPositions()).to.equal('(0,0),(0,2)');

                    done();
                });

            });

            it('should generate only one operation, if one of the selected drawings is already in the background', function () {

                expect(selection.getListOfPositions()).to.equal('(0,0),(0,2)');

                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);

                operationCounter = model.changeDrawingOrder('backward');

                expect(operationCounter).to.equal(1);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);

                // checking the new selection after moving the drawing
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,1)');
            });

            it('should undo the operations that brought the one drawing one step to the back', function (done) {

                var // the undo promise
                    promise = null;

                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);

                // bringing drawing1 and drawing 3 one step backwards
                promise = model.getUndoManager().undo();

                promise.always(function () {

                    // checking move of drawing
                    expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                    expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);

                    // checking the new selection after moving the drawing
                    // expect(selection.getListOfPositions()).to.equal('(0,0),(0,2)');

                    done();
                });

            });

            it('should bring both selected drawings to the top', function () {

                selection.setMultiDrawingSelectionByPosition([[0, 0], [0, 2], [0, 3]]);  // selecting drawing1, drawing3 and drawing4

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(3);
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,2),(0,3)');

                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);

                operationCounter = model.changeDrawingOrder('front');

                expect(operationCounter).to.equal(3);

                // checking move of drawing
                expect(drawing1.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(2);
                expect(drawing2.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(0);
                expect(drawing3.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(3);
                expect(drawing4.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(4);
                expect(drawing5.prevAll(DOM.ABSOLUTE_DRAWING_SELECTOR).length).to.equal(1);

                // checking the new selection after moving the drawing
                expect(selection.getListOfPositions()).to.equal('(0,2),(0,3),(0,4)');
            });

            it('should not create any operation, if all drawings are selected', function () {

                selection.setTextSelection([0, 0, 0, 0]); // setting the cursor into the first drawing on the first slide
                selection.selectAllDrawingsOnSlide();

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(5);
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,1),(0,2),(0,3),(0,4)');

                operationCounter = model.changeDrawingOrder('front');
                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('forward');
                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('back');
                expect(operationCounter).to.equal(0);

                operationCounter = model.changeDrawingOrder('backward');
                expect(operationCounter).to.equal(0);

                expect(selection.isMultiSelection()).to.equal(true);
                expect(selection.getMultiSelection().length).to.equal(5);
                expect(selection.getListOfPositions()).to.equal('(0,0),(0,1),(0,2),(0,3),(0,4)');
            });

            it('should not create move operations by changing the alignment', function () {

                function getDrawingFromSelection(drawingSelection) {
                    var drawingNode = selection.getDrawingNodeFromMultiSelection(drawingSelection);
                    return AttributeUtils.getExplicitAttributes(drawingNode).drawing;
                }

                function getAllSelectedDrawings() {
                    return selection.isMultiSelection() ? selection.getMultiSelection() : selection.generateMultiSelectionObjectFromDrawingSelection({ anyDrawing: true });
                }

                // Select the text of the first drawing
                selection.setTextSelection([1, 0, 0, 0]);

                // align the drawing to left
                operationCounter = model.changeAlignment('left');
                expect(operationCounter).to.equal(true);

                var allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(1);
                // drawing left position must be on the left side on the page with left 0
                expect(getDrawingFromSelection(allDrawingSelections[0]).left).to.equal(0);

                // Select the first and second drawing
                selection.setMultiDrawingSelectionByPosition([[1, 0], [1, 1]]);

                // algin the drawings to 'center'
                operationCounter = model.changeAlignment('center');

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(2);

                // check the new left position
                expect(getDrawingFromSelection(allDrawingSelections[0]).left).to.equal(2014);

                expect(getDrawingFromSelection(allDrawingSelections[1]).left).to.equal(2014);

                // algin the objects to top
                operationCounter = model.changeAlignment('top');

                expect(allDrawingSelections.length).to.equal(2);
                // both drawings must have the same top position, it is the top position of the second drawing
                expect(getDrawingFromSelection(allDrawingSelections[0]).top).to.equal(5000);
                expect(getDrawingFromSelection(allDrawingSelections[1]).top).to.equal(5000);

                // Select three drawing
                selection.setMultiDrawingSelectionByPosition([[1, 0], [1, 1], [1, 2]]);

                // algin the drawings to 'center'
                operationCounter = model.changeAlignment('middle');

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(3);
                //check the new top positions
                expect(getDrawingFromSelection(allDrawingSelections[0]).top).to.equal(7000);
                expect(getDrawingFromSelection(allDrawingSelections[1]).top).to.equal(7000);
                expect(getDrawingFromSelection(allDrawingSelections[2]).top).to.equal(7500);

                // Select the last drawing
                selection.setMultiDrawingSelectionByPosition([[1, 3]]);

                // algin the drawings to 'center'
                operationCounter = model.changeAlignment('right');

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(1);
                //check the new top positions
                expect(getDrawingFromSelection(allDrawingSelections[0]).left).to.equal(24447);

               // Select the first and last drawing
                selection.setMultiDrawingSelectionByPosition([[1, 0], [1, 3]]);

                // algin the drawings to 'center'
                operationCounter = model.changeAlignment('bottom');

                allDrawingSelections = getAllSelectedDrawings();

                expect(allDrawingSelections.length).to.equal(2);
                //check the new top positions
                expect(getDrawingFromSelection(allDrawingSelections[0]).top).to.equal(8234);
                expect(getDrawingFromSelection(allDrawingSelections[1]).top).to.equal(7725);
            });

        });

    });

});
