/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/portal/portalutils'
], function (PortalUtils) {

    'use strict';

    var TIME_FORMAT     = /^([01]?\d|2[0-3]):[0-5]\d [ap]m$/i;   // z.B. => 3:00 AM
    var DAY_FORMAT      = /^([0-2]?\d|3[01])\.[a-z]{3}$/i;       // z.B. => 28.Mar

    // static class PortalUtils ===============================================

    describe('Portal module PortalUtils', function () {

        it('should exist', function () {
            expect(PortalUtils).to.be.an('object');
        });

        it('should have a icon-create function', function () {
            expect(PortalUtils).to.respondTo('createDocumentIcon');
        });

        // it('should return the correct module-name', function () {
        //     expect(PortalUtils.getModuleNameForFile('bla.docx')).to.equal('io.ox/office/text');
        // });

        it('createDocumentIcon function should return the correct app-icon', function () {
            expect(PortalUtils.createDocumentIcon('nevermind')[0].outerHTML).to.equal('<i class="document-icon fa fa-file"></i>');
          //expect(PortalUtils.createDocumentIcon('bla.docx')[0].outerHTML).to.equal('<i class="document-icon fa fa-file-text text"></i>');
            expect(PortalUtils.createDocumentIcon('bla.xlsx')[0].outerHTML).to.equal('<i class="document-icon fa fa-table spreadsheet"></i>');
        });

        it('should return the correct application base name', function () {
            expect(PortalUtils.getAppBaseName('nevermind')).to.equal(null);
          //expect(PortalUtils.getAppBaseName('bla.docx')).to.equal('text');
            expect(PortalUtils.getAppBaseName('bla.bla.xlsx')).to.equal('spreadsheet');
        });

        it('should return the filename without extension', function () {
            expect(PortalUtils.removeFileExtension('nevermind')).to.equal('nevermind');
            expect(PortalUtils.removeFileExtension('bla.docx')).to.equal('bla');
            expect(PortalUtils.removeFileExtension('bla.bla.xlsx')).to.equal('bla.bla');
        });

        it('should return whether the filename end with the given suffix', function () {
            expect(PortalUtils.stringEndsWith('nevermind', 'docx')).to.equal(false);
            expect(PortalUtils.stringEndsWith('bla.docx', 'docx')).to.equal(true);
            expect(PortalUtils.stringEndsWith('bla.xlsx', 'docx')).to.equal(false);
        });

        it('should return the enabled applications', function () {
            expect(PortalUtils.getEnabledApps()).to.deep.equal(['text', 'spreadsheet', 'presentation']);
        });

        it('should return the title of the active app in lowercase', function () {
            expect(PortalUtils.getActiveApp({ attributes: { title: 'Text' } })).to.equal('text');
            expect(PortalUtils.getActiveApp({ attributes: { title: 'Text' } })).not.to.equal('Text');
        });

        it('moveFocus function should exist', function () {
            expect(PortalUtils).to.respondTo('moveFocus');
        });

        it('should set the focus and not return anything', function () {
            var firstNode = $('<div>'),
                secondNode = $('<div>'),
                thirdNode = $('<div>'),

                fourthNode = $('<div tabindex="1">'),
                fifthNode = $('<div tabindex="2">'),
                sixthNode = $('<div tabindex="3">');

            $('<div>').append(firstNode, secondNode, thirdNode);
            $('<div>').append(fourthNode, fifthNode, sixthNode);

            expect(PortalUtils.moveFocus()).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(secondNode)).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(secondNode, 'left')).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(secondNode, 'down')).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(secondNode, 'up')).to.be.equal(undefined);

            expect(PortalUtils.moveFocus(fifthNode)).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(fifthNode, 'left')).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(fifthNode, 'down')).to.be.equal(undefined);
            expect(PortalUtils.moveFocus(fifthNode, 'up')).to.be.equal(undefined);
        });

        it('keydownHandler should exist', function () {
            expect(PortalUtils).to.respondTo('keydownHandler');
        });

        it('should handle keydown events', function () {
            var spy = sinon.spy(),
                ele = $('<a>').on('click', spy),

                e1      = new $.Event('click', { target: ele }),
                e2      = new $.Event('click', { target: ele, data: { grid: true } }),
                e3a     = new $.Event('click', { target: ele, keyCode: 13 }),
                e3b     = new $.Event('click', { target: ele, keyCode: 37 }),
                e3c     = new $.Event('click', { target: ele, keyCode: 38 }),
                e3c2    = new $.Event('click', { target: ele, keyCode: 38, data: { grid: true } }),
                e3d     = new $.Event('click', { target: ele, keyCode: 39 }),
                e3e     = new $.Event('click', { target: ele, keyCode: 40 }),
                e3e2    = new $.Event('click', { target: ele, keyCode: 40, data: { grid: true } });

            expect(PortalUtils.keydownHandler()).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e1)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e2)).to.be.equal(undefined);

            expect(PortalUtils.keydownHandler(e3a)).to.be.equal(undefined);
            expect(spy.calledOnce).to.equal(true);

            expect(PortalUtils.keydownHandler(e3b)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3c)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3c2)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3d)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3e)).to.be.equal(undefined);
            expect(PortalUtils.keydownHandler(e3e2)).to.be.equal(undefined);
        });

    });

    describe('Portal module PortalUtils - Time tests (Now = 01.01.2016 14:00)', function () {
        var systemTime = new Date(2016, 0, 1, 14, 0, 0),
            clock = null;

        before(function () {
            clock = sinon.useFakeTimers(systemTime.getTime());
        });

        it('formatDate should return the correct formatted date', function () {
            var now             = new Date(),
                yesterday       = new Date(2015, 11, 31, 14, 0, 0),
                prevMonth       = new Date(2015, 11, 1, 14, 0, 0),
                prevYear        = new Date(2015, 0, 1, 14, 0, 0),
                TIME_FORMAT     = new RegExp('^([0-1]?[0-9]|2[0-3]):[0-5][0-9] [APap][mM]$');   // z.B. => 3:00 AM

            expect(PortalUtils.formatDate(now.getTime())).to.equal('2:00 PM');
            expect(PortalUtils.formatDate(now.getTime())).to.match(TIME_FORMAT);

            expect(PortalUtils.formatDate(yesterday.getTime())).to.equal('12/31/2015');

            expect(PortalUtils.formatDate(prevMonth.getTime())).to.equal('12/01/2015');

            expect(PortalUtils.formatDate(prevYear.getTime())).to.equal('01/01/2015');
        });

        after(function () {
            clock.restore();
        });
    });

    describe('Portal module PortalUtils - Time tests (Now = 01.05.2016 14:00)', function () {
        var systemTime = new Date(2016, 4, 1, 14, 0, 0),
            clock = null;

        before(function () {
            clock = sinon.useFakeTimers(systemTime.getTime());
        });

        it('formatDate should return the correct formatted date', function () {
            var now             = new Date(),
                yesterday       = new Date(2016, 3, 30, 14, 0, 0),
                prevMonth       = new Date(2016, 3, 1, 14, 0, 0),
                prevYear        = new Date(2015, 4, 1, 14, 0, 0);

            expect(PortalUtils.formatDate(now.getTime())).to.equal('2:00 PM');
            expect(PortalUtils.formatDate(now.getTime())).to.match(TIME_FORMAT);

            expect(PortalUtils.formatDate(yesterday.getTime())).to.equal('30.Apr');
            expect(PortalUtils.formatDate(yesterday.getTime())).to.match(DAY_FORMAT);

            expect(PortalUtils.formatDate(prevMonth.getTime())).to.equal('01.Apr');
            expect(PortalUtils.formatDate(prevMonth.getTime())).to.match(DAY_FORMAT);

            expect(PortalUtils.formatDate(prevYear.getTime())).to.equal('05/01/2015');
        });

        after(function () {
            clock.restore();
        });
    });

    describe('Portal module PortalUtils - Time tests (Now = 31.05.2016 14:00)', function () {
        var systemTime = new Date(2016, 4, 31, 14, 0, 0),
            clock = null;

        before(function () {
            clock = sinon.useFakeTimers(systemTime.getTime());
        });

        it('formatDate should return the correct formatted date', function () {
            var now             = new Date(),
                yesterday       = new Date(2016, 4, 30, 14, 0, 0),
                prevMonth       = new Date(2016, 3, 30, 14, 0, 0),
                prevYear        = new Date(2015, 4, 31, 14, 0, 0);

            expect(PortalUtils.formatDate(now.getTime())).to.equal('2:00 PM');
            expect(PortalUtils.formatDate(now.getTime())).to.match(TIME_FORMAT);

            expect(PortalUtils.formatDate(yesterday.getTime())).to.equal('30.May');
            expect(PortalUtils.formatDate(yesterday.getTime())).to.match(DAY_FORMAT);

            expect(PortalUtils.formatDate(prevMonth.getTime())).to.equal('30.Apr');
            expect(PortalUtils.formatDate(prevMonth.getTime())).to.match(DAY_FORMAT);

            expect(PortalUtils.formatDate(prevYear.getTime())).to.equal('05/31/2015');
        });

        after(function () {
            clock.restore();
        });
    });

    // ========================================================================
});
