/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('globals/apphelper', [
    'globals/sheethelper',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/model/editmodel',
    'io.ox/office/text/model/model',
    'io.ox/office/spreadsheet/model/model',
    'io.ox/office/presentation/model/model'
], function (SheetHelper, TimerMixin, EditModel, TextModel, SpreadsheetModel, PresentationModel) {

    'use strict';

    // helper function that does nothing but returns the calling context
    function noop() { return this; }

    // class BaseTestView =====================================================

    /**
     * A simple test view class for unit tests. Mimics the common public API of
     * a Documents view instance.
     *
     * @constructor
     */
    function BaseTestView(app, docModel) {

        var rootNode = $('<div>');

        this.on = this.one = this.off = this.trigger = noop;
        this.getDocModel = _.constant(docModel);
        this.getContentRootNode = _.constant(rootNode);
        this.getHiddenRootNode = _.constant(rootNode);
        this.isVisible = _.constant(true);
        this.getToolBarTabs = _.constant({ handleActiveToolbarTab: _.noop });
        this.createClipboardNode = function () { return $('<div class="clipboard">'); };
        this.grabFocus = noop;
        this.getZoomFactor = _.constant(100);
        this.enterBusy = noop;
        this.leaveBusy = noop;
        this.updateBusyProgress = noop;

    } // class BaseTestView

    // class TextFrameworkTestView ============================================

    function TextFrameworkTestView(app, docModel) {

        BaseTestView.call(this, app, docModel);

        this.getDisplayDateString = _.constant('A');
        this.recalculateDocumentMargin = noop;
        this.hideFieldFormatPopup = noop;
        this.isSearchActive = _.constant(false);
        this.clearVisibleDrawingAnchor = noop;
        this.setVisibleDrawingAnchor = noop;
        this.getDisplayDateString = _.constant('A');

    } // class TextTestView

    // class TextTestView =====================================================

    function TextTestView(app, docModel) {

        TextFrameworkTestView.call(this, app, docModel);

        this.recalculateDocumentMargin = noop;
        this.hideFieldFormatPopup = noop;

    } // class TextTestView

    // class PresentationTestView =============================================

    function PresentationTestView(app, docModel) {

        TextFrameworkTestView.call(this, app, docModel);

        this.getSlidePane = _.constant({
            setSidePaneWidthDuringLoad: noop,
            getSlidePaneContainer: $.noop // no return value like 'noop'
        });

        this.enableVerticalScrollBar = noop;
        this.setZoomType = noop;

    } // class TextTestView

    // class BaseTestController ===============================================

    /**
     * A simple test controller class for unit tests. Mimics the common public
     * API of a Documents controller instance.
     *
     * @constructor
     */
    function BaseTestController() {

        this.on = this.one = this.off = this.trigger = noop;

    } // class BaseTestController

    // static class AppHelper =================================================

    var AppHelper = {};

    /**
     * Creates a simple test application class for unit tests, containing a
     * real document model instance. Mimics the common public API of a
     * Documents application instance.
     *
     * @returns {CoreApplication}
     *  The application containing an instance of the passed model class.
     */
    AppHelper.createApp = function (appBaseName, fileFormat, ModelClass, ViewClass, operations, converters) {

        var docModel = null;
        var docView = null;
        var docController = null;
        var initDef = $.Deferred();
        var importDef = $.Deferred();

        var moduleName = 'io.ox/office/' + appBaseName;
        var app = ox.ui.createApp({ name: moduleName });
        var appWindow = ox.ui.createWindow({ name: moduleName, search: false, chromeless: true });
        appWindow.isBusy = _.constant(false);
        app.setWindow(appWindow);

        app.getModel = function () { return docModel; };
        app.getView = function () { return docView; };
        app.getController = function () { return docController; };

        app.registerDestructor = noop;
        app.onInit = function (callback, context) { initDef.done(callback.bind(context)); };
        app.isImportStarted = function () { return initDef.state() !== 'pending'; };
        app.getImportStartPromise = _.constant(initDef.promise());
        app.isImportFinished = function () { return importDef.state() !== 'pending'; };
        app.getImportFinishPromise = _.constant(importDef.promise());
        app.leaveBusyDuringImport = noop;
        app.getFileFormat = _.constant(fileFormat);
        app.isOOXML = _.constant(fileFormat === 'ooxml');
        app.isODF = _.constant(fileFormat === 'odf');
        app.registerVisibleStateHandler = noop;

        app.setState = noop;
        app.isInternalError = _.constant(false);
        app.setInternalError = function () { throw new Error('internal error'); };
        app.isLocallyModified = _.constant(true);
        app.setUserState = noop;
        app.sendLogMessage = noop;

        app.getUserSettingsValue = _.constant(null);
        app.destroyImageNodes = noop;
        app.addAuthors = noop;
        app.getAuthorColorIndex = _.constant(1);
        app.isMultiSelectionApplication = _.constant(appBaseName !== 'text');
        app.getActiveClients = _.constant([]);

        app.quit = function () {
            docModel.destroy();
            docModel = docView = docController = null;
        };

        // synchronous launcher
        app.setLauncher(function () {
            TimerMixin.call(app);
            docModel = new ModelClass(app);
            docView = new ViewClass(app, docModel);
            docController = new BaseTestController();
            initDef.resolve();
            if (_.isArray(operations)) {
                operations.forEach(function (operation, index) {
                    if (converters && (operation.name in converters)) {
                        operation = _.clone(operation);
                        converters[operation.name](operation);
                    }
                    if (!docModel.invokeOperationHandler(operation)) {
                        console.error(appBaseName + ' operation #' + index + ' (' + operation.name + ') failed', operation);
                        throw new Error('operation failed');
                    }
                });
            }
            importDef.resolve();
        });

        return app;
    };

    /**
     * Generic implementation to create a test application. The creation of the
     * test application will be placed in a before() block, and the application
     * will be destroyed in an after() block. This results in creation and
     * destruction of the test application inside the surrounding describe()
     * block.
     *
     * @param {Function} generator
     *  The generator callback function for the test application. MUST return a
     *  promise that resolves with the test application.
     *
     * @returns {jQuery.Promise}
     *  A promise that resolves with the test application.
     */
    function createTestApp(generator) {

        var testApp = null;
        var def = $.Deferred();

        before(function (done) {
            generator().done(function (app) {
                testApp = app;
                def.resolve(app);
                done();
            }).fail(function () {
                def.reject();
                done(new Error('cannot create application'));
            });
        });

        after(function () {
            if (testApp) { testApp.quit(); }
            testApp = null;
        });

        return def.promise();
    }

    /**
     * Creates a test application with a real instance of an EditModel, and
     * applies the passed document operations in order top initialize the
     * document model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will resolve with the application instance containing an
     *  edit model.
     */
    AppHelper.createEditApp = function (fileFormat, operations) {
        return createTestApp(function () {
            var app = AppHelper.createApp('edit', fileFormat, EditModel, BaseTestView, operations);
            app.launch();
            return $.when(app);
        });
    };

    /**
     * Creates a test application with a real text document model, and applies
     * the passed document operations in order top initialize the document
     * model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will resolve with the application instance containing a
     *  real text model.
     */
    AppHelper.createTextApp = function (fileFormat, operations) {
        return createTestApp(function () {

            var app = AppHelper.createApp('text', fileFormat, TextModel, TextTestView, operations);
            app.stopOperationDistribution = noop;
            app.isOperationsBlockActive = _.constant(false);
            app.getClientOperationName = _.constant('author');
            app.launch();

            var docModel = app.getModel();
            return docModel.updateDocumentFormatting().then(function () {
                docModel.setEditMode(true);
                docModel.getSelection().setTextSelection([0, 0]);
                return app;
            });
        });
    };

    var SPREADSHEET_DEFAULT_OPERATIONS = [
        { name: 'setDocumentAttributes', attrs: { document: { fileFormat: 'ooxml', cols: 16384, rows: 1048576, activeSheet: 0 } } },
        { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' }
    ];

    var SPREADSHEET_OPERATION_CONVERTERS = {
        changeCells: function (operation) {
            if (typeof operation.start === 'string') {
                operation.start = SheetHelper.a(operation.start).toJSON();
            }
            operation.contents = operation.contents.map(function (rowData) {
                if (_.isArray(rowData)) { rowData = { c: rowData }; }
                rowData.c = rowData.c.map(function (cellData) {
                    return (cellData === null) ? {} :
                        (cellData instanceof SheetHelper.ErrorCode) ? { e: cellData.toJSON() } :
                        _.isObject(cellData) ? cellData :
                        { v: cellData };
                });
                return rowData;
            });
        }
    };

    /**
     * Creates a test application with a real spreadsheet document model, and
     * applies the passed document operations in order top initialize the
     * document model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will resolve with the application instance containing a
     *  real spreadsheet model.
     */
    AppHelper.createSpreadsheetApp = function (fileFormat, operations) {
        return createTestApp(function () {

            operations = operations || SPREADSHEET_DEFAULT_OPERATIONS;
            var app = AppHelper.createApp('spreadsheet', fileFormat, SpreadsheetModel, BaseTestView, operations, SPREADSHEET_OPERATION_CONVERTERS);
            app.launch();

            var docModel = app.getModel();
            return docModel.postProcessImport().then(function () {
                docModel.setEditMode(true);
                docModel.getDependencyManager().stopListeningTo(docModel);
                return app;
            });
        });
    };

    /**
     * Creates a test application with a real presentation document model, and
     * applies the passed document operations in order top initialize the
     * document model.
     *
     * @returns {jQuery.Promise}
     *  A promise that will resolve with the application instance containing a
     *  real presentation model.
     */
    AppHelper.createPresentationApp = function (fileFormat, operations) {
        return createTestApp(function () {

            var app = AppHelper.createApp('presentation', fileFormat, PresentationModel, PresentationTestView, operations);
            app.isOperationsBlockActive = _.constant(false);
            app.stopOperationDistribution = noop;
            app.launch();

            var docModel = app.getModel();
            return docModel.updateDocumentFormatting().then(function () {
                docModel.setEditMode(true);
                return app;
            });
        });
    };

    // exports ================================================================

    return AppHelper;

});
