/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/logger', [
    'io.ox/office/tk/config'
], function (Config) {

    'use strict';

    var // global indent for all loggers
        indent = '',

        // prevent collapsing space characters in browser console
        INDENT_PARTICLE = '\xa0 ',

        // function returning the current system time stamp
        now = (window.performance && window.performance.now) ? function () { return window.performance.now(); } : _.now;

    // private global functions ===============================================

    /**
     * Increases the global indentation level for all logger messages.
     */
    function incIndent() {
        indent += INDENT_PARTICLE;
    }

    /**
     * Decreases the global indentation level for all logger messages.
     */
    function decIndent() {
        indent = indent.substring(INDENT_PARTICLE.length);
    }

    // class Logger ===========================================================

    /**
     * Provides methods to create logging messages in the browser console.
     * Logging will always be disabled without global debug mode specified in
     * the server-side configuration option 'debugavailable'.
     *
     * @constructor
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Boolean|String} [initOptions.enable=true]
     *      Specifies whether the logger is enabled. If set to a string, the
     *      application URL will be checked for an according anchor option set
     *      to the value 'true'.
     *  @param {String} [initOptions.prefix]
     *      If specified, all messages will be prefixed with the passed text
     *      (enclosed in bracket characters).
     */
    function Logger(initOptions) {

        var // the 'enable' option passed to the constructor (defaults to true)
            enableOpt = (_.isObject(initOptions) && ('enable' in initOptions)) ? initOptions.enable : true,

            // whether the logger is enabled
            enabled = _.isString(enableOpt) ? Config.getDebugUrlFlag(enableOpt) : (Config.DEBUG && _.isBoolean(enableOpt) && enableOpt),

            // the prefix printed before each message
            prefix = (_.isObject(initOptions) && _.isString(initOptions.prefix)) ? ('[' + initOptions.prefix  + ']') : null;

        // private methods ----------------------------------------------------

        /**
         * Writes a message to the browser output console.
         *
         * @param {String} level
         *  The log level of the message. The string 'log' will create a
         *  generic log message, the string 'info' will create an information,
         *  the string 'warn' will create a warning message, and the string
         *  'error' will create an error message.
         */
        function log(level, args) {

            // add white-space indent
            if (indent.length > 0) {
                args.unshift(indent);
            }

            // add logger prefix
            if (prefix) {
                args.unshift(prefix);
            }

            // check that the browser console supports the operation
            if (!_.isFunction(window.console[level])) {
                args.unshift(level.toUpperCase());
                level = 'log';
            }

            if (_.isFunction(window.console[level])) {
                window.console[level].apply(window.console, args);
            } else if (_.isNumber(_.browser.IE) && (_.browser.IE < 10)) {
                window.console.log(args.join(' '));
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether this logger is active.
         *
         * @returns {Boolean}
         *  Whether this logger is active.
         */
        this.isLoggingActive = function () {
            return enabled;
        };

        /**
         * Invokes the passed callback function if and only if this logger is
         * enabled. Can be used to prevent running rather complex calculations
         * that are needed for debug logging only.
         *
         * @param {Function} callback
         *  The callback function.
         *
         * @param {Object} [context]
         *  The context to be bound to the callback function.
         *
         * @returns {Logger}
         *  A reference to this instance.
         */
        this.withLogging = function (callback, context) {
            if (enabled) { callback.call(context); }
            return this;
        };

        /**
         * Writes a log message to the browser output console, if debug mode is
         * enabled in the global configuration.
         *
         * @param {Any} [...]
         *  The values to be written to the console.
         */
        this.log = enabled ? function () { log('log', _.toArray(arguments)); } : $.noop;

        /**
         * Writes an info message to the browser output console, if debug mode is
         * enabled in the global configuration.
         *
         * @param {Any} [...]
         *  The values to be written to the console.
         */
        this.info = enabled ? function () { log('info', _.toArray(arguments)); } : $.noop;

        /**
         * Writes a warning message to the browser output console, if debug mode is
         * enabled in the global configuration.
         *
         * @param {Any} [...]
         *  The values to be written to the console.
         */
        this.warn = enabled ? function () { log('warn', _.toArray(arguments)); } : $.noop;

        /**
         * Writes an error message to the browser output console, if debug mode is
         * enabled in the global configuration.
         *
         * @param {Any} [...]
         *  The values to be written to the console.
         */
        this.error = enabled ? function () { log('error', _.toArray(arguments)); } : $.noop;

        /**
         * Writes an error message to the browser output console describing the
         * passed exception object, if debug mode is enabled in the global
         * configuration.
         *
         * @param {Any} exception
         *  The exception as caught in a try/catch.
         *
         * @param {Any} [...]
         *  Additional values to be written to the console.
         */
        this.exception = enabled ? function (exception) {

            var // the stack contained in the exception object
                stack = _.isObject(exception) && (exception.stack || exception.stacktrace);

            log('error', ['Exception caught:', exception].concat(_.toArray(arguments).slice(1)));
            if (stack) { log('error', ['Stack trace:', stack]); }

        } : $.noop;

        /**
         * Registers an event handler for the specified event and writes a log
         * message whenever the event has been triggered.
         *
         * @param {Object} source
         *  The event source object.
         *
         * @param {String} type
         *  The name of the event. Can be a space-separated list.
         *
         * @param {String} [...]
         *  The names for additional parameters passed to the event handler.
         */
        this.logEvent = enabled ? function (source, type) {

            var // self reference
                logFunc = _.bind(function (args) { this.log.apply(this, args); }, this),
                // the event parameter names
                paramNames = _.toArray(arguments).slice(2),
                // if the source object provides a DOM node, write it to the log message too
                rootNode = _.isFunction(source.getNode) ? $(source.getNode())[0] : null;

            source.on(type, function (event) {

                var // all arguments passed to the log message
                    logArgs = ['type="' + event.type + '"', 'this=', this];

                // add the root node of the event source
                if (rootNode) { logArgs.push('node=', rootNode); }

                // add the additional event parameters
                _.chain(arguments).toArray().slice(1).each(function (arg, index) {
                    if (index < paramNames.length) { logArgs.push(paramNames[index] + '='); }
                    logArgs.push(arg);
                });

                logFunc(logArgs);
            });
        } : $.noop;

        /**
         * Executes the passed callback function and writes a message to the
         * browser console containing the execution time of the callback. The
         * callback function will be invoked ALWAYS, regardless whether logging
         * is enabled or not.
         *
         * @param {String} message
         *  The initial message printed to the browser console before invoking
         *  the callback function. The execution time will be shown after the
         *  callback function has returned.
         *
         * @param {Function} callback
         *  The callback function to be executed.
         *
         * @param {Object} [context]
         *  The context to be bound to the callback function.
         *
         * @returns {Any}
         *  The return value of the callback function.
         */
        this.takeTime = enabled ? function (message, callback, context) {
            var t0 = 0;
            try {
                this.log('=>', message);
                t0 = now();
                return Logger.indent(callback, context);
            } finally {
                this.log('<=', (Math.round((now() - t0) * 1000) / 1000) + 'ms', message);
            }
        } : function (message, callback, context) {
            return callback.call(context);
        };

        /**
         * Executes the passed asynchronous callback function and writes a
         * message to the browser console containing the time it needs to
         * resolve the promise returned by the callback. The callback function
         * will be invoked ALWAYS, regardless whether logging is enabled or
         * not.
         *
         * @param {String} message
         *  The initial message printed to the browser console before invoking
         *  the callback function. The execution time will be shown after the
         *  promise returned by the callback function has been resolved or
         *  rejected.
         *
         * @param {Function} callback
         *  The callback function to be executed. MUST return a Deferred object
         *  or a promise of a Deferred object.
         *
         * @param {Object} [context]
         *  The context to be bound to the callback function.
         *
         * @returns {jQuery.Promise}
         *  The promise returned by the callback function.
         */
        this.takeAsyncTime = enabled ? function (message, callback, context) {
            this.log('=> [async]', message);
            var logger = this, t0 = now();
            var promise = Logger.indent(callback, context);
            var state = promise.state();
            return promise.always(function () {
                logger.log('<=', (Math.round((now() - t0) * 1000) / 1000) + 'ms [' + promise.state() + ((state === 'pending') ? ']' : ' immediately]'), message);
            });
        } : function (message, callback, context) {
            return callback.call(context);
        };

        /**
         * Creates and returns a version of the passed callback function that
         * shows the execution time of each its invocation, if this logger is
         * enabled (wraps the callback function in a call to Logger.takeTime()
         * internally).
         *
         * @param {String} message
         *  The initial message printed to the browser console before invoking
         *  the callback function. The execution time will be shown after the
         *  callback function has returned.
         *
         * @param {Function} method
         *  The callback function to be profiled.
         *
         * @returns {Function}
         *  The passed callback function wrapped into a call to the method
         *  Logger.takeTime(). Forwards the current calling context to the
         *  callback function, and returns its return value.
         */
        this.profileMethod = enabled ? function (message, method) {
            var logger = this;
            return function () {
                var args = arguments;
                return logger.takeTime(message, function () {
                    return method.apply(this, args);
                }, this);
            };
        } : function (message, method) {
            return method;
        };

        /**
         * Creates and returns a version of the passed asynchronous callback
         * function that shows the time each of the promises it returns needs
         * to be resolved or rejected, if this logger is enabled (wraps the
         * callback function in a call to Logger.takeAsyncTime() internally).
         *
         * @param {String} message
         *  The initial message printed to the browser console before invoking
         *  the callback function. The execution time will be shown after the
         *  promise returned by the method has been resolved or rejected.
         *
         * @param {Function} method
         *  The callback function to be profiled. MUST return a Deferred object
         *  or a promise of a Deferred object.
         *
         * @returns {Function}
         *  The passed callback function wrapped into a call to the method
         *  Logger.takeAsyncTime(). Forwards the current calling context to the
         *  callback function, and returns its return value.
         */
        this.profileAsyncMethod = enabled ? function (message, method) {
            var logger = this;
            return function () {
                var args = arguments;
                return logger.takeAsyncTime(message, function () {
                    return method.apply(this, args);
                }, this);
            };
        } : function (message, method) {
            return method;
        };

    } // class Logger

    // static methods ---------------------------------------------------------

    /**
     * Extends the passed object or class with logging functionality.
     *
     * @param {Object|Function} obj
     *  An object that will be extended with the logging methods provided by
     *  the Logger class. Can be a class constructor function, in this case the
     *  class will contain static logging methods.
     *
     * @param {Object} [initOptions]
     *  Optional parameters passed to the constructor function of the class
     *  Logger.
     *
     * @returns {Object|Function}
     *  A reference to the passed and extended object or class.
     */
    Logger.extend = function (obj, initOptions) {
        Logger.call(obj, initOptions);
        return obj;
    };

    /**
     * Invokes the passed callback function with increased indentation for all
     * active loggers.
     *
     * @param {Function} callback
     *  The callback function to be executed.
     *
     * @param {Object} [context]
     *  The context to be bound to the callback function.
     *
     * @returns {Any}
     *  The return value of the callback function.
     */
    Logger.indent = function (callback, context) {
        try {
            incIndent();
            var result = callback.call(context);
            $.when(result).always(decIndent);
            return result;
        } catch (ex) {
            decIndent();
            throw ex;
        }
    };

    // exports ================================================================

    return Logger;

});
