/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/render/path', function () {

    'use strict';

    // convenience constants
    var PI2 = 2 * Math.PI;

    // class PathOperation ====================================================

    /**
     * A single operation contained in a canvas path object.
     *
     * @constructor
     *
     * @property {String} name
     *  The name of the path operation (the name of the related native method
     *  of the rendering context).
     *
     * @property {Array<Any>} args
     *  All arguments to be passed to the native context method.
     */
    function PathOperation(name, args) {

        this.name = name;
        this.args = args || [];

    } // PathOperation

    // class Path =============================================================

    /**
     * Represents a rendering path for a canvas rendering context, consisting
     * of several path operations.
     *
     * @constructor
     *
     * @property {Array<PathOperation>} ops
     *  All recorded operations of this path.
     */
    function Path() {

        this.ops = [];

    } // class Path

    // public methods ---------------------------------------------------------

    /**
     * Pushes a single path operation to th einternal operations array.
     *
     * @param {String} name
     *  The name of the path operation (the name of the related native method
     *  of the rendering context).
     *
     * @param {Array<Any>} [args]
     *  All arguments to be passed to the native context method.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.pushOp = function (name, args) {
        this.ops.push(new PathOperation(name, args));
        return this;
    };

    /**
     * Opens a new empty subpath in this path, starting at the specified
     * position.
     *
     * @param {Number} x
     *  The X coordinate of the starting point of the new subpath.
     *
     * @param {Number} y
     *  The Y coordinate of the starting point of the new subpath.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.moveTo = function (x, y) {
        return this.pushOp('moveTo', [x, y]);
    };

    /**
     * Adds a new position to the current (last) subpath in this path that will
     * be connected to the previous position with a straight line.
     *
     * @param {Number} x
     *  The X coordinate of the new position in the current subpath.
     *
     * @param {Number} y
     *  The Y coordinate of the new position in the current subpath.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.lineTo = function (x, y) {
        return this.pushOp('lineTo', [x, y]);
    };

    /**
     * Adds a circular arc to the current (last) subpath in this path. The
     * start position of the arc will be connected to the end position of the
     * subpath with a straight line.
     *
     * @param {Number} x
     *  The X coordinate of the center point of the arc.
     *
     * @param {Number} y
     *  The Y coordinate of the center point of the arc.
     *
     * @param {Number} r
     *  The radius of the arc.
     *
     * @param {Number} a1
     *  The angle (radiant) of the arc's start position. The value 0 represents
     *  a position on the positive part of the X axis.
     *
     * @param {Number} a2
     *  The angle (radiant) of the arc's end position. The value 0 represents a
     *  position on the positive part of the X axis.
     *
     * @param {Boolean} [ccw=false]
     *  Whether to draw the arc counterclockwise from its start position to its
     *  end position. By default, the arc will be drawn clockwise.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.arc = function (x, y, r, a1, a2, ccw) {
        return this.pushOp('arc', [x, y, r, a1, a2, ccw]);
    };

    /**
     * Adds an elliptic arc to the current (last) subpath in this path. The
     * start position of the arc will be connected to the end position of the
     * subpath with a straight line.
     *
     * @param {Number} x
     *  The X coordinate of the center point of the arc.
     *
     * @param {Number} y
     *  The Y coordinate of the center point of the arc.
     *
     * @param {Number} rx
     *  The radius of the arc on the X axis.
     *
     * @param {Number} ry
     *  The radius of the arc on the Y axis.
     *
     * @param {Number} a0
     *  The rotation angle (radiant) of the entire ellipse, including the
     *  following start and end angles.
     *
     * @param {Number} a1
     *  The angle (radiant) of the arc's start position. The value 0 represents
     *  a position on the positive part of the X axis.
     *
     * @param {Number} a2
     *  The angle (radiant) of the arc's end position. The value 0 represents a
     *  position on the positive part of the X axis.
     *
     * @param {Boolean} [ccw=false]
     *  Whether to draw the arc counterclockwise from its start position to its
     *  end position. By default, the arc will be drawn clockwise.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.ellipse = function (x, y, rx, ry, a0, a1, a2, ccw) {
        return this.pushOp('ellipse', [x, y, rx, ry, a0, a1, a2, ccw]);
    };

    /**
     * Closes the current (last) subpath by adding a straight line to its start
     * position, and opens a new subpath at the start position of the current
     * subpath.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.close = function () {
        return this.pushOp('closePath');
    };

    /**
     * Adds a new quadratic bezier subpath starting from the current point.
     *
     * @param {Number} cp1x
     *  The X coordinate of the control point.
     *
     * @param {Number} cp1y
     *  The Y coordinate of the control point.
     *
     * @param {Number} x
     *  The X coordinate of the end point of the line.
     *
     * @param {Number} y
     *  The Y coordinate of the end point of the line.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.quadraticCurveTo = function (cp1x, cp1y, x, y) {
        return this.pushOp('quadraticCurveTo', [cp1x, cp1y, x, y]);
    };

    /**
     * Adds a new cubic bezier subpath starting from the current point.
     *
     * @param {Number} cp1x
     *  The X coordinate of the first control point.
     *
     * @param {Number} cp1y
     *  The Y coordinate of the first control point.
     *
     * @param {Number} cp2x
     *  The X coordinate of the second control point.
     *
     * @param {Number} cp2y
     *  The Y coordinate of the second control point.
     *
     * @param {Number} x
     *  The X coordinate of the end point of the line.
     *
     * @param {Number} y
     *  The Y coordinate of the end point of the line.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.bezierCurveTo = function (cp1x, cp1y, cp2x, cp2y, x, y) {
        return this.pushOp('bezierCurveTo', [cp1x, cp1y, cp2x, cp2y, x, y]);
    };

    /**
     * Adds a new cubic bezier subpath starting from the current point.
     *
     * @param {Number} cx
     *  The x start point of the arc.
     *
     * @param {Number} cy
     *  The Y start point of the arc.
     *
     * @param {Number} rx
     *  The radius of the arc on the X axis.
     *
     * @param {Number} ry
     *  The radius of the arc on the Y axis.
     *
     * @param {Number} a0
     *  The start angle (radiant) of the arc.
     *
     * @param {Number} a1
     *  The end angle (radiant) of the arc.
     *
     * @param {Boolean} [ccw=false]
     *  Whether to draw the arc counterclockwise from its start position to its
     *  end position. By default, the arc will be drawn clockwise.
     *
     */
    Path.prototype.ellipseTo = function (cx, cy, rx, ry, a0, a1, ccw) {
        var dx = rx * Math.cos(a0);
        var dy = ry * Math.sin(a0);
        return this.pushOp('save').pushOp('translate', [cx - dx, cy - dy]).ellipse(0, 0, rx, ry, 0, a0, a1, ccw).pushOp('restore');
    };

    /**
     * Adds a new subpath consisting of a straight line. After that, opens
     * another new subpath at the starting point of the line.
     *
     * @param {Number} x1
     *  The X coordinate of the starting point of the line.
     *
     * @param {Number} y1
     *  The Y coordinate of the starting point of the line.
     *
     * @param {Number} x2
     *  The X coordinate of the end point of the line.
     *
     * @param {Number} y2
     *  The Y coordinate of the end point of the line.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.pushLine = function (x1, y1, x2, y2) {
        return this.moveTo(x1, y1).lineTo(x2, y2).moveTo(x1, y1);
    };

    /**
     * Adds a new subpath consisting of a closed rectangle. After that, opens
     * another new subpath at the starting corner of the rectangle.
     *
     * @param {Number|Object} x
     *  The X coordinate of the starting point of the rectangle, or a complete
     *  rectangle with the numeric properties 'left', 'top', 'width', and
     *  'height'.
     *
     * @param {Number} y
     *  The Y coordinate of the starting point of the rectangle (ignored, if
     *  parameter 'x' is a rectangle object).
     *
     * @param {Number} w
     *  The width of the rectangle (ignored, if parameter 'x' is a rectangle
     *  object).
     *
     * @param {Number} h
     *  The height of the rectangle (ignored, if parameter 'x' is a rectangle
     *  object).
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.pushRect = function (x, y, w, h) {
        if (_.isObject(x)) { y = x.top; w = x.width; h = x.height; x = x.left; }
        return this.moveTo(x, y).lineTo(x + w, y).lineTo(x + w, y + h).lineTo(x, y + h).close();
    };

    /**
     * Adds a new subpath consisting of a complete circle. After that, opens
     * another new subpath at the center point of the circle.
     *
     * @param {Number} x
     *  The X coordinate of the center point of the circle.
     *
     * @param {Number} y
     *  The Y coordinate of the center point of the circle.
     *
     * @param {Number} r
     *  The radius of the circle.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.pushCircle = function (x, y, r) {
        return this.moveTo(x + r, y).arc(x, y, r, 0, PI2).moveTo(x, y);
    };

    /**
     * Adds a new subpath consisting of a complete ellipse. After that, opens
     * another new subpath at the center point of the ellipse.
     *
     * @param {Number} x
     *  The X coordinate of the center point of the ellipse.
     *
     * @param {Number} y
     *  The Y coordinate of the center point of the ellipse.
     *
     * @param {Number} rx
     *  The radius of the ellipse on the X axis.
     *
     * @param {Number} ry
     *  The radius of the ellipse on the Y axis.
     *
     * @param {Number} [a=0]
     *  The rotation angle (radiant) of the ellipse.
     *
     * @returns {Path}
     *  A reference to this instance.
     */
    Path.prototype.pushEllipse = function (x, y, rx, ry, a) {
        return this.moveTo(x + rx, y).ellipse(x, y, rx, ry, a || 0, 0, PI2).moveTo(x, y);
    };

    // exports ================================================================

    return Path;

});
