/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/listmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/popup/basemenu',
    'settings!io.ox/office'
], function (Utils, KeyCodes, Forms, BaseMenu, Settings) {

    'use strict';

    // the CSS class for menu section nodes
    var SECTION_CLASS = 'section-container';

    // The internal identifier for the title section
    var TITLE_SECTION_ID = '\x00TITLE';

    // The internal identifier for the most recently used section.
    var MRU_SECTION_ID = '\x00MRU';

    // class ListMenu =========================================================

    /**
     * Wraps a pop-up menu element, shown on top of the application window, and
     * relative to an arbitrary DOM node. An item list contains a set of button
     * control elements (the list items) that can be stacked vertically, or can
     * be arranged in a tabular layout. The list items can be highlighted. The
     * items will be grouped into vertically stacked section nodes.
     *
     * Instances of this class trigger the following events (additionally to
     * the events triggered by the base class BaseMenu):
     * - 'create:section'
     *      After a new section node has been created with the method
     *      ListMenu.createSectionNode(). Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {jQuery} sectionNode
     *          The root node of the new section, as jQuery object.
     *      (3) {String} sectionId
     *          The section identifier.
     *      (4) {Object} [options]
     *          The options passed to the method ListMenu.createSectionNode().
     * - 'create:item'
     *      After a button node for a new list item has been created with the
     *      method ListMenu.createItemNode(). Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {jQuery} buttonNode
     *          The button element representing the new list item, as jQuery
     *          object.
     *      (3) {Any} value
     *          The value of the new list item.
     *      (4) {Object} [options]
     *          The options passed to the method ListMenu.createItemNode().
     *
     * @constructor
     *
     * @extends BaseMenu
     *
     * @param {Object} [initOptions]
     *  Initial options for the item list menu element. Supports all options
     *  also supported by the base class BaseMenu. Additionally, the following
     *  options are supported:
     *  @param {Function} [initOptions.itemMatcher=_.isEqual]
     *      A comparison function that returns whether a list item should be
     *      selected, deselected, or kept unmodified according to a specific
     *      value. Receives an arbitrary value as first parameter, and the
     *      value of the current list item element as second parameter. The
     *      function must return the Boolean value true to select the
     *      respective list item element, the Boolean value false to deselect
     *      the list item element, or any other value to leave the list item
     *      element unmodified. If omitted, uses _.isEqual() which compares
     *      arrays and objects deeply, and does not skip any list item.
     *  @param {Boolean} [options.multiSelect=true]
     *      If set to false, only the first matching element will be selected.
     *  @param {String} [options.title]
     *      A title label to be shown on top of the entire list menu.
     *  @param {String} [initOptions.itemDesign='list']
     *      The design mode of the list item elements:
     *      - 'list': All item elements will be stacked vertically, and will
     *          occupy the complete width of the pop-up menu. Highlighted item
     *          elements will be drawn with a changed background color.
     *      - 'grid': All item elements will be arranged in a tabular layout
     *          per menu section. Highlighted items will be drawn with a thick
     *          border, leaving the background untouched.
     *  @param {Boolean} [initOptions.sortItems=false]
     *      If set to true, the list items will be sorted according to the sort
     *      order that has been specified via the 'sortIndex' options of each
     *      inserted list item (see method ListMenu.createItemNode() for
     *      details).
     *  @param {Number} [options.gridColumns=10]
     *      The default number of columns in each menu section in 'grid' design
     *      mode (see option 'itemDesign' above). This value can be overridden
     *      for single menu sections by passing this option to the method
     *      ListMenu.createSectionNode().
     *  @param {Number} [options.mruSize=0]
     *      The maximum number of entries shown in the most recently used list.
     *      If the list size is exceeded, the first added item will be removed.
     *  @param {String} [options.mruSettingsKey]
     *      If specified, the contents of the most recently used list will be
     *      stored into the user settings under that key.
     *  @param {Array<String>} [options.mruBlackList]
     *      The item values which should not be added to the most recently used
     *      list.
     */
    function ListMenu(initOptions) {

        // self reference
        var self = this;

        // comparator for list item values
        var itemMatcher = Utils.getFunctionOption(initOptions, 'itemMatcher', _.isEqual);

        // whether to allow multi-selection of list items
        var multiSelect = Utils.getBooleanOption(initOptions, 'multiSelect', true);

        // the design mode of the list item nodes
        var design = Utils.getStringOption(initOptions, 'itemDesign', 'list');

        // whether the list items will be sorted
        var sorted = Utils.getBooleanOption(initOptions, 'sortItems', false);

        // default number of columns in each section node ('grid' design mode)
        var defColumns = Utils.getIntegerOption(initOptions, 'gridColumns', 10, 1);

        // the maximum size of the MRU list
        var mruSize = Utils.getIntegerOption(initOptions, 'mruSize', 0);

        // the key to store the MRU list in the user settings
        var mruSettingsKey = Utils.getStringOption(initOptions, 'mruSettingsKey', null);

        // values of all items not to be added to the MRU list
        var mruBlackList = Utils.getArrayOption(initOptions, 'mruBlackList', []);

        // the actual list of most recently used items
        var mruList = null;

        var selectedItemsOnBeforeOpen;

        // base constructor ---------------------------------------------------

        BaseMenu.call(this, Utils.extendOptions({
            scrollPadding: 5,
            preferFocusFilter: Forms.SELECTED_SELECTOR
        }, initOptions, {
            prepareLayoutHandler: prepareLayoutHandler
        }));

        // private methods ----------------------------------------------------

        /**
         * Prepares layouting this list in grid design. Calculates the maximum
         * width of the pop-up menu according to the widths of all existing
         * button elements, and distributes the available space equally to the
         * button nodes.
         */
        function prepareLayoutHandler() {

            if (design !== 'grid') {
                return;
            }

            var // all section nodes
                sectionNodes = self.getSectionNodes(),
                // the total width needed to show all sections with evenly sized columns
                totalWidth = 0;

            // remove explicit widths
            self.getItemNodes().css('width', '');

            // calculate required width of all sections
            sectionNodes.each(function () {

                var // all button nodes in the current section
                    buttonNodes = $(this).find(Forms.BUTTON_SELECTOR),
                    // maximum width of the button elements, used as grid column width
                    //maxWidth = _.max(buttonNodes.get(), function (node) { return Utils.getCeilNodeSize(node).width; });
                    maxWidth = _.max(_.map(buttonNodes.get(), function (node) {
                        return Utils.getCeilNodeSize(node).width;
                    }));

                // update total width with the width requited by the current section
                totalWidth = Math.max(totalWidth, maxWidth * $(this).data('grid-columns'));
            });

            // set button width in all sections
            sectionNodes.each(function () {
                $(this).find(Forms.BUTTON_SELECTOR).css('width', Math.floor(totalWidth / $(this).data('grid-columns')));
            });
        }

        /**
         * Sorts all list items in the specified section node debounced.
         */
        var sortSectionNode = (function () {

            var // all section nodes to be sorted
                sectionNodes = $();

            // direct callback: register a section node
            function registerSectionNode(sectionNode) {
                // jQuery keeps the collection unique
                sectionNodes = sectionNodes.add(sectionNode);
            }

            // deferred callback: sorts all registered sections debounced
            function sortAllSectionNodes() {

                // protect the focused list item
                self.guardFocusedListItem(function () {

                    // process all registered section nodes
                    sectionNodes.each(function () {

                        var // the current section node
                            sectionNode = $(this),
                            // all existing item buttons in the section, as plain array (for sorting)
                            buttonNodes = sectionNode.find(Forms.BUTTON_SELECTOR).get(),
                            // cell nodes in grid design
                            cellNodes = null;

                        // sort by fixed index, default to label text of the button
                        buttonNodes = _.sortBy(buttonNodes, function (btnNode) {
                            var index = $(btnNode).data('index');
                            return _.isNull(index) ? $(btnNode).text().toLowerCase() : index;
                        });

                        // insert button nodes depending on design mode
                        switch (design) {
                            case 'list':
                                sectionNode.append.apply(sectionNode, buttonNodes);
                                break;
                            case 'grid':
                                cellNodes = sectionNode.find('>.grid-row>.grid-cell');
                                _.each(buttonNodes, function (btnNode, index) {
                                    cellNodes.eq(index).append(btnNode);
                                });
                                break;
                        }
                    });

                    // reset the cache of dirty section nodes
                    sectionNodes = $();
                });
            }

            return self.createDebouncedMethod(registerSectionNode, sortAllSectionNodes);
        }());

        /**
         * Handles 'keydown' events in the root node of the pop-up menu.
         */
        function keyDownHandler(event) {

            var // the list item to be focused
                targetNode = self.getItemNodeForKeyEvent(event, $(event.target));

            // stop event propagation for all supported keys
            if (targetNode.length > 0) {
                targetNode.focus();
                self.scrollToChildNode(targetNode);
                return false;
            }
        }

        /**
         * Initializes the menu title.
         */
        function initializeTitle() {

            // the title shown above the list items
            var title = Utils.getStringOption(initOptions, 'title', null);
            if (!title) { return; }

            // the root node of the menu section
            var sectionNode = self.createSectionNode(TITLE_SECTION_ID, { gridColumns: 1 }).addClass('title');
            var titleNode = $('<span class="title">').text(title);

            // make section node visible (initially hidden)
            Forms.showNodes(sectionNode, true);

            // create a new row element for the button if required
            var rowNode = sectionNode.find('>.grid-row').last();
            if ((rowNode.length === 0) || (rowNode.children().length === sectionNode.data('grid-columns'))) {
                rowNode = $('<div class="grid-row" role="row">').appendTo(sectionNode);
            }
            // append the button to the row (wrap into a helper node for ARIA)
            var cellNode = $('<div class="grid-cell" role="gridcell">').attr('aria-label', titleNode.attr('aria-label') || '').append(titleNode);
            rowNode.append(cellNode);
        }

        /**
         * Check if the item is stored in the most recently list.
         * @param {String} value the value for check
         * @returns {Boolean} true an item the given value is stored to the list, otherwise false
         */
        function containsMostRecentlyItem(value) {
            return _.findWhere(mruList, { value: value });
        }

        /**
         * Add a value to the most recently list and create a node for the most recently used section.
         * If the item size is reached, the first added item will be removed.
         * @param {String} value see@ createItemNode(value, options)
         * @param {Object} options see@ createItemNode(value, options)
         */
        function addValueToMRUList(value, options) {
            if (mruList && value && !_.contains(mruBlackList, value) && !_.isEmpty(options) && !containsMostRecentlyItem(value)) {
                options.section = MRU_SECTION_ID;

                if (mruList.length >= mruSize) {
                    self.deleteItemNodes(mruList.shift().value, MRU_SECTION_ID);
                }
                mruList.push({ value: value, options: options });
                self.createItemNode(value, options);
                return true;
            }
            return false;
        }

        /**
         * Fills the most recently used list with the words from the settings
         * and add the items to the most recently used section. If the section
         * for the most recently items is not created, the section will be
         * added.
         */
        function initializeMRUList() {

            mruList = [];

            // resolve predefined list entries from configuration
            var mruValues = mruSettingsKey ? Settings.get(mruSettingsKey, []) : null;
            if (!_.isArray(mruValues)) { return; }

            mruValues.forEach(function (item) {
                var itemNodes = self.findItemNodes(item.value);
                if (itemNodes.length > 0) {
                    var data = itemNodes.data();
                    addValueToMRUList(data.value, data.options);
                }
            });
        }

        function updateMRUList() {
            _.each(self.getSelectedItemNodes(), function (obj, i, item) {

                if (!_.find(selectedItemsOnBeforeOpen, function (itemBefore) {
                    return item.data().value === $(itemBefore).data().value;
                })) {

                    var added = addValueToMRUList(item.data().value, item.data().options);
                    if (added && mruSettingsKey) {
                        Settings.set(mruSettingsKey, mruList).save();
                    }
                }
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Invokes the passed callback function which wants to modify the
         * contents of this drop-down list. The previously focused list item
         * will be focused again afterwards.
         *
         * @param {Function} callback
         *  The callback function that will be invoked while remembering the
         *  list item currently focused.
         *
         * @returns {ListMenu}
         *  A reference to this instance.
         */
        this.guardFocusedListItem = function (callback, context) {

            var // the focused list item
                focusNode = this.getItemNodes().filter(':focus'),
                // value of the focused list item (bug 33737, 33747: preserve browser focus)
                value = Forms.getButtonValue(focusNode.first());

            // invoke passed callback function
            callback.call(context);

            // restore browser focus by looking up the original focus node, or
            // by selecting a new focus node via the value of the old node
            if (focusNode.length > 0) {
                // first try to find the original node, resolve by value if this fails
                if (this.getItemNodes().filter(focusNode).length === 0) {
                    focusNode = this.findItemNodes(value);
                }
                focusNode.focus();
            }

            return this;
        };

        /**
         * Adds a new section to this pop-up menu, if it does not exist yet.
         *
         * @param {String} sectionId
         *  The unique identifier of the menu section.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.label]
         *      If specified, a heading label will be created for the section.
         *  @param {Number} [options.gridColumns]
         *      The number of columns in the menu section, if this menu is in
         *      'grid' design mode (see option 'itemDesign' passed to the
         *      constructor). If omitted, the default value for this option as
         *      specified in the constructor options will be used.
         *
         * @returns {jQuery}
         *  The root node of the menu section, as jQuery object.
         */
        this.createSectionNode = function (sectionId, options) {

            var // try to find an existing section node
                sectionNode = this.getSectionNode(sectionId);

            if (sectionNode.length === 0) {

                // create the header label mark-up
                var headerMarkup = Utils.getStringOption(options, 'label', '');
                if (headerMarkup.length > 0) {
                    headerMarkup = '<div class="header-label" aria-hidden="true">' + Utils.escapeHTML(headerMarkup) + '</div>';
                }

                // create the section root node
                sectionNode = $('<div class="' + SECTION_CLASS + ' group ' + Forms.HIDDEN_CLASS + '" data-section="' + sectionId + '">' + headerMarkup + '</div>');

                // add the number of grid columns as jQuery data attribute
                sectionNode.data('grid-columns', Utils.getIntegerOption(options, 'gridColumns', defColumns, 1));

                // add ARIA role for grid design
                if (design === 'grid') {
                    sectionNode.attr({ role: 'grid', 'aria-label': Utils.getStringOption(initOptions, 'tooltip', null, true) });
                }

                // insert the new section into the drop-down menu, notify listeners
                this.appendContentNodes(sectionNode);
                this.trigger('create:section', sectionNode, sectionId, options);
            }
            return sectionNode;
        };

        /**
         * Returns the DOM nodes of all menu sections.
         *
         * @returns {jQuery}
         *  The DOM nodes representing all menu sections, as jQuery collection.
         */
        this.getSectionNodes = function () {
            return this.getNode().find('.' + SECTION_CLASS);
        };

        /**
         * Returns the DOM node of the specified menu section.
         *
         * @param {String} sectionId
         *  The unique identifier of the menu section.
         *
         * @returns {jQuery}
         *  The DOM node representing the specified menu section. If the menu
         *  section does not exist, returns an empty jQuery collection.
         */
        this.getSectionNode = function (sectionId) {
            return this.getSectionNodes().filter('[data-section="' + sectionId + '"]');
        };

        /**
         * Adds a new item element to this list. If the items will be sorted
         * (see the sort options passed to the constructor), the item element
         * will be inserted according to these settings.
         *
         * @param {Any} value
         *  The unique value associated to the list item. MUST NOT be null.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all formatting options for button
         *  elements supported by the method Forms.createButtonMarkup().
         *  Additionally, the following options are supported:
         *  @param {String} [options.section='']
         *      The unique identifier of the menu section the new list item
         *      will be inserted into. If a section with this identifier does
         *      not exist yet, it will be created without a header label. Can
         *      be omitted for example for simple list menus without sections.
         *  @param {String} [options.dataValue]
         *      A string that will be inserted into the 'data-value' attribute
         *      of the button node. If omitted, the JSON string representation
         *      of the 'value' parameter will be used instead (all double-quote
         *      characters will be removed from the string though), unless the
         *      value is a function.
         *
         * @returns {jQuery}
         *  The button element representing the new list item, as jQuery
         *  object.
         */
        this.createItemNode = function (value, options) {

            var // the root node of the menu section
                sectionNode = this.createSectionNode(Utils.getStringOption(options, 'section', '')),
                // create the button element representing the item
                buttonNode = $(Forms.createButtonMarkup(options));

            // make section node visible (initially hidden)
            Forms.showNodes(sectionNode, true);

            // initialize button node
            Forms.setButtonValue(buttonNode, value, options);
            Forms.setToolTip(buttonNode, options);
            buttonNode.data('index', Utils.getOption(options, 'sortIndex', null));
            buttonNode.attr('role', 'option');

            // insert the new button according to the design mode
            switch (design) {
                case 'list':
                    sectionNode.append(buttonNode);
                    break;
                case 'grid':
                    // create a new row element for the button if required
                    var rowNode = sectionNode.find('>.grid-row').last();
                    if ((rowNode.length === 0) || (rowNode.children().length === sectionNode.data('grid-columns'))) {
                        rowNode = $('<div class="grid-row" role="row">').appendTo(sectionNode);
                    }
                    // append the button to the row (wrap into a helper node for ARIA)
                    var cellNode = $('<div class="grid-cell" role="gridcell">').attr('aria-label', buttonNode.attr('aria-label') || '').append(buttonNode);
                    rowNode.append(cellNode);
                    break;
            }

            // sort the button nodes in the section
            if (sorted) {
                sortSectionNode(sectionNode);
            }

            // notify listeners
            this.trigger('create:item', buttonNode, value, options);
            return buttonNode;
        };

        /**
         * Removes an item element from this list.
         *
         * @param {Any} value
         *  The value associated to the list item. MUST NOT be null.
         *
         * @param {String} [sectionId]
         *  The unique identifier of the menu section, to delete only the buttons in
         *  the given section, if the value is not unique. Otherwise delete all buttons
         *  with the given value.
         *
         * @returns {ListMenu}
         *  A reference to this instance.
         */
        this.deleteItemNodes = function (value, sectionId) {

            var // all item buttons to be deleted
                buttonNodes = this.findItemNodes(value),
                // all modified section nodes
                sectionNodes = $();

            // collect modified section nodes, notify listeners
            buttonNodes = buttonNodes.filter(function () {
                var buttonNode = $(this);
                if (!_.isString(sectionId) || buttonNode.parent().filter('[data-section="' + sectionId + '"]').length > 0) {
                    sectionNodes = sectionNodes.add(buttonNode.closest('.' + SECTION_CLASS));
                    self.trigger('delete:item', buttonNode, value);
                    return true;
                }
                return false;
            });

            // remove button nodes from DOM
            buttonNodes.remove();

            // update nodes in grid design mode, hide empty sections
            sectionNodes.each(function () {

                var // current section node
                    sectionNode = $(this);

                // all existing buttons in the section
                buttonNodes = sectionNode.find(Forms.BUTTON_SELECTOR);

                // reinsert buttons in grid design mode
                if (design === 'grid') {
                    // redistribute all buttons without moving the cell nodes
                    var rowNodes = sectionNode.find('>.grid-row'),
                        cellNodes = rowNodes.find('>.grid-cell');
                    buttonNodes.each(function (index) {
                        cellNodes.eq(index).append(this);
                    });
                    // remove trailing empty cell nodes and empty trailing row nodes
                    cellNodes.filter(':empty').remove();
                    rowNodes.filter(':empty').remove();
                }

                // hide empty sections
                Forms.showNodes(sectionNode, buttonNodes.length > 0);
            });

            return this;
        };

        /**
         * Returns all button elements representing the menu items of this
         * list.
         *
         * @returns {jQuery}
         *  All button elements representing the menu items in this item list,
         *  as jQuery collection.
         */
        this.getItemNodes = function () {
            return this.getNode().find(Forms.BUTTON_SELECTOR);
        };

        /**
         * Returns the button elements representing the list item with the
         * specified value.
         *
         * @param {Any} value
         *  The value whose list item buttons will be returned.
         *
         * @returns {jQuery}
         *  The button elements representing the list items with the specified
         *  value, as jQuery collection.
         */
        this.findItemNodes = function (value) {
            return Forms.filterButtonNodes(this.getItemNodes(), value, { matcher: itemMatcher, multiSelect: multiSelect });
        };

        /**
         * Returns the button elements representing all selected list items in
         * this list menu.
         *
         * @returns {jQuery}
         *  The button elements representing all selected list items, as jQuery
         *  collection.
         */
        this.getSelectedItemNodes = function () {
            return Forms.filterCheckedButtonNodes(this.getItemNodes());
        };

        /**
         * Selects the specified list item in this menu, and deselects all
         * other list items.
         *
         * @param {Number} index
         *  The zero-based index of the list item to be selected.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.scroll=false]
         *      If set to true, the menu will scroll to the selected item node
         *      automatically.
         *
         * @returns {jQuery}
         *  The button element representing the selected list item, as jQuery
         *  object. If the passed index is invalid, an empty jQuery collection
         *  will be returned.
         */
        this.selectItem = function (index, options) {
            return this.selectItemNode(this.getItemNodes().eq(index), options);
        };

        /**
         * Selects the specified button elements representing one or more list
         * items in this menu, and deselects all other list items.
         *
         * @param {HTMLElement|jQuery} buttonNodes
         *  The list item buttons to be selected, as plain DOM node, or as
         *  jQuery collection that may contain multiple list items.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.scroll=false]
         *      If set to true, the menu will scroll to the selected item node
         *      automatically.
         *
         * @returns {jQuery}
         *  The button elements representing the selected list items, as jQuery
         *  collection.
         */
        this.selectItemNode = function (buttonNode, options) {
            buttonNode = $(buttonNode);
            Forms.checkButtonNodes(this.getItemNodes(), false);
            Forms.checkButtonNodes(buttonNode, true);
            if ((buttonNode.length > 0) && Utils.getBooleanOption(options, 'scroll', false)) {
                this.scrollToChildNode(buttonNode);
            }
            return buttonNode;
        };

        /**
         * Selects the button elements representing the list item with the
         * passed value, and deselects all other list items.
         *
         * @param {Any} value
         *  The value whose list item buttons will be selected. List items will
         *  be selected or deselected according to the item matcher callback
         *  function passed to the constructor (option 'itemMatcher').
         *
         * @returns {jQuery}
         *  The button elements representing the selected list items with the
         *  specified value, as jQuery collection.
         */
        this.selectMatchingItemNodes = function (value) {
            Forms.checkMatchingButtonNodes(this.getItemNodes(), value, { matcher: itemMatcher, multiSelect: multiSelect });
            return this.getSelectedItemNodes();
        };

        /**
         * Returns the target list item to be focused or selected after moving
         * the cursor according to the passed keyboard event.
         *
         * @param {jQuery.Event} keyEvent
         *  The event object caught for a 'keydown' event.
         *
         * @param {HTMLElement|jQuery|Null} buttonNode
         *  The button element representing the source list item the cursor
         *  movement originates from. If omitted, the first or last list item
         *  will be selected instead according to the key code. All keys in
         *  'down' orientation (CURSOR_DOWN, PAGE_DOWN) and the HOME key will
         *  select the first list item, all keys in 'up' orientation
         *  (CURSOR_UP, PAGE_UP) and the END key will select the last list
         *  item.
         *
         * @returns {jQuery}
         *  The target list item to be focused or selected, as jQuery object.
         *  If the passed key event does not describe any key supported for
         *  cursor movement, or the passed button node is invalid, an empty
         *  jQuery collection will be returned instead.
         */
        this.getItemNodeForKeyEvent = function (keyEvent, buttonNode) {

            var // all visible button nodes
                buttonNodes = Forms.filterVisibleNodes(self.getItemNodes());

            // returns the number of list items currently visible
            function getListPageSize() {
                var itemHeight = buttonNodes.first().outerHeight();
                return (itemHeight > 0) ? Math.max(1, Math.floor(self.getNode()[0].clientHeight / itemHeight) - 1) : 1;
            }

            // returns the previous or next item node according to the passed distance
            function getNextItemNode(diff, wrap) {

                var // index of the focused button
                    index = buttonNodes.index(buttonNode);

                if (index < 0) {
                    index = (diff < 0) ? (buttonNodes.length - 1) : 0;
                } else if (wrap) {
                    index += (diff % buttonNodes.length) + buttonNodes.length;
                    index %= buttonNodes.length;
                } else {
                    index = Utils.minMax(index + diff, 0, buttonNodes.length - 1);
                }
                return buttonNodes.eq(index);
            }

            // returns the target item node for 'list' design
            function getItemNodeForList() {

                switch (keyEvent.keyCode) {
                    case KeyCodes.UP_ARROW:
                        return getNextItemNode(-1, true);
                    case KeyCodes.DOWN_ARROW:
                        return getNextItemNode(1, true);
                    case KeyCodes.PAGE_UP:
                        return getNextItemNode(-getListPageSize());
                    case KeyCodes.PAGE_DOWN:
                        return getNextItemNode(getListPageSize());
                }

                return $();
            }

            // returns the item node in the previous or next row in 'grid' design
            function getItemNodeInNextRow(diff) {

                var // the cell node containing the focused button
                    cellNode = buttonNode.parent(),
                    // the row node containing the focused button
                    rowNode = cellNode.parent(),
                    // all button elements in the focused row
                    rowButtonNodes = rowNode.children().children(),
                    // all row nodes in all sections
                    rowNodes = self.getSectionNodes().find('>.grid-row'),
                    // the column index of the cell in the row
                    colIndex = cellNode.index(),
                    // the global row index of the cell
                    rowIndex = rowNodes.index(rowNode),
                    // the relative column index of the cell
                    colRatio = (colIndex >= 0) ? (colIndex / rowButtonNodes.length) : 0;

                if ((rowNodes.length === 0) || (rowIndex < 0)) {
                    return (diff < 0) ? buttonNodes.last() : buttonNodes.first();
                }

                // move focus to the specified row, try to keep column position constant inside row
                rowIndex += (diff % rowNodes.length) + rowNodes.length;
                rowNode = rowNodes.eq(rowIndex % rowNodes.length);
                rowButtonNodes = rowNode.children().children();
                colIndex = Math.min(Math.floor(colRatio * rowNode.children().length), rowButtonNodes.length - 1);
                return rowButtonNodes.eq(colIndex);
            }

            // returns the target item node for 'grid' design
            function getItemNodeForGrid() {

                switch (keyEvent.keyCode) {
                    case KeyCodes.LEFT_ARROW:
                        return getNextItemNode(-1, true);
                    case KeyCodes.RIGHT_ARROW:
                        return getNextItemNode(1, true);
                    case KeyCodes.UP_ARROW:
                        return getItemNodeInNextRow(-1);
                    case KeyCodes.DOWN_ARROW:
                        return getItemNodeInNextRow(1);
                }

                return $();
            }

            // convert to jQuery collection
            buttonNode = $(buttonNode);

            // generic key shortcuts
            switch (keyEvent.keyCode) {
                case KeyCodes.HOME:
                    return buttonNodes.first();
                case KeyCodes.END:
                    return buttonNodes.last();
            }

            // special shortcuts dependent on design mode
            switch (design) {
                case 'list':
                    return getItemNodeForList();
                case 'grid':
                    return getItemNodeForGrid();
            }

            return $();
        };

        /**
         * Returns the contents of the MRU list.
         *
         * @returns {Array<Any>}
         *  The contents of the MRU list.
         */
        this.getMRUListEntries = function () {
            return mruList ? _.pluck(mruList, 'value') : [];
        };

        // initialization -----------------------------------------------------

        // add CSS marker for the item design
        this.getNode().addClass('list-menu').attr({ 'data-design': design, role: 'listbox' });

        // register event handlers
        this.getNode().on('keydown', keyDownHandler);

        // convert ENTER, SPACE, and TAB keys in the menu to click events
        Forms.setButtonKeyHandler(this.getNode(), { tab: true });

        // initialize the title section
        initializeTitle();

        // initialize the MRU list
        if (mruSize > 0) {
            this.createSectionNode(MRU_SECTION_ID);
            multiSelect = false;
            this.one('popup:beforeshow', initializeMRUList);
            this.on('popup:beforehide', updateMRUList);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = self = itemMatcher = null;
        });

    } // class ListMenu

    // exports ================================================================

    // derive this class from class BaseMenu
    return BaseMenu.extend({ constructor: ListMenu });

});
