/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/selection/selection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/textframework/selection/multiselectionmixin',
    'io.ox/office/textframework/selection/androidselection'
], function (Utils, KeyCodes, TriggerObject, TimerMixin, DrawingFrame, DOM, Position, Image, MultiSelectionMixin, AndroidSelection) {

    'use strict';

    var // whether the browser selection object supports the collapse() and expand() methods needed to create a backwards selection
        SELECTION_COLLAPSE_EXPAND_SUPPORT = (function () {
            var selection = window.getSelection();
            return _.isFunction(selection.collapse) && _.isFunction(selection.extend);
        }()),

        // if set to true, DOM selection will be logged to console
        LOG_SELECTION = false;

    // private static functions ===============================================

    /**
     * A jQuery selector for non-empty text spans and in-line component
     * nodes. Empty text spans are allowed as first elements in the paragraph, so
     * that positions before range markers can be reached (41957).
     */
    function inlineNodeSelector() {
        return DOM.isEditableInlineComponentNode(this) || (DOM.isTextSpan(this) && (!DOM.isEmptySpan(this) || DOM.isFirstTextSpanInParagraph(this)));
    }

    /**
     * A jQuery selector for empty and non-empty text spans and in-line component
     * nodes.
     */
    function allowEmptyNodeSelector() {
        return DOM.isEditableInlineComponentNode(this) || DOM.isTextSpan(this);
    }

    /**
     * Returns the first non-empty inline node of the specified paragraph (text
     * component nodes or non-empty text spans) if existing. Otherwise, if the
     * paragraph is empty, returns the trailing text span (following the
     * leading floating drawing objects).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.allowEmpty=false]
     *      If set to true, empty text spans will be included into the
     *      search for the first text cursor node. Otherwise only non empty
     *      text spans are allowed. The latter is the default.
     */
    function getFirstTextCursorNode(paragraph, options) {
        return Utils.findDescendantNode(paragraph, Utils.getBooleanOption(options, 'allowEmpty', false) ? allowEmptyNodeSelector : inlineNodeSelector, { children: true }) || DOM.findLastPortionSpan(paragraph);
    }

    /**
     * Returns the last non-empty inline node of the specified paragraph (text
     * component nodes or non-empty text spans) if existing. Otherwise, if the
     * paragraph is empty, returns the trailing text span (following the
     * leading floating drawing objects).
     */
    function getLastTextCursorNode(paragraph) {
        return Utils.findDescendantNode(paragraph, inlineNodeSelector, { children: true, reverse: true }) || DOM.findLastPortionSpan(paragraph);
    }

    // class Selection ========================================================

    /**
     * An instance of this class represents a selection in the edited document,
     * consisting of a logical start and end position representing a half-open
     * text range, or a rectangular table cell range.
     *
     * Triggers the following events:
     * - 'position:calculated': When the new logical positions startPosition and
     *      endPosition are calculated.
     * - 'change': After the selection has changed.
     * - 'update': After the selection has updated (triggers also when selection hasn't changed);
     *      used for example to open popups depending on cursor position (like field format popup)
     * - 'changecursorinfo': After the selection has changes (used only for Android)
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     * @extends MultiSelectionMixin
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The root node of the document. If this object is a jQuery collection,
     *  uses the first node it contains.
     */
    function Selection(app, rootNode) {

        var // self reference
            self = this,

            // logical start position
            startPosition = [],

            // logical end position (half-open range)
            endPosition = [],

            // whether the current text range has been selected backwards
            backwards = false,

            // whether the last cursor movement was to the left or upwards
            isBackwardCursor = false,

            // after switching the target, the change event must be triggered (even if start and end position stay the same)
            forceSelectionChangeEvent = false,

            // drawing node currently selected, as jQuery collection
            selectedDrawing = $(),

            // selecting additionally the text frame, if text is selected
            selectedTextFrameDrawing = $(),

            // table currently selected, as jQuery object
            selectedTable = $(),

            // whether this selection represents a rectangular table cell range
            cellRangeSelected = false,

            // whether the cell range selection covers the entire table
            tableSelected = false,

            // the anchor of a cell range of a cell selection in a table
            anchorCellRange = null,

            // copy of drawing contents with browser selection, for copy&paste
            clipboardNode = null,

            // the original focus() method of the root node DOM element
            originalFocusMethod = rootNode[0].focus,

            // Performance: Saving info from insertText operation to simplify scroll calculation
            insertTextPoint = null, isInsertTextOperation = false,

            // Performance: Saving paragraph node, logical paragraph position and text offset in specified operations
            paragraphCache = null,

            // whether the calculated logical (end) position was modified by triggering 'position:calculated'
            selectionExternallyModified = false, endSelectionExternallyModified = false,

            // FF: Saving the start position of the mouse down event
            mouseDownStartPosition = null,

            // the handler function for moving and resizing drawings
            drawingResizeHandler = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        MultiSelectionMixin.call(this, app);
        if (_.browser.Android) { AndroidSelection.call(this, app, rootNode); }

        // private methods ----------------------------------------------------

        /**
         * Returns the current logical anchor position.
         */
        function getAnchorPosition() {
            return backwards ? endPosition : startPosition;
        }

        /**
         * Returns the current logical focus position.
         */
        function getFocusPosition() {
            return backwards ? startPosition : endPosition;
        }

        /**
         * Returns the first logical position in the document.
         * If the optional parameter node is specified, it returns the
         * first logical position inside this node. This optional
         * parameter can be used for text frames.
         *
         * Info: This function supports an empty selection, that can
         * occur in slide mode after deleting the last slide (45742).
         *
         * @param {HTMLElement|jQuery} [node]
         *  An optional DOM element object whose first logical position
         *  will be searched. If this object is a jQuery collection,
         *  uses the first node it contains.
         *  If it is not specified, the page content node will be
         *  used as default.
         *
         * @param {Object} [options]
         *  Supports all options that are supported by 'getFirstTextCursorNode'.
         *
         * @returns {Number[]|Null}
         *  The first logical position in the document or the specified node.
         *  Or null, if this is the empty selection
         */
        function getFirstPosition(node, options) {

            var // the node, in which the first position is searched
                searchNode = null, //  node || (DOM.getPageContentNode(rootNode).length > 0 ? DOM.getPageContentNode(rootNode) : rootNode),
                // the page content node
                pageContentNode = null,
                // the first paragraph inside the specified node
                firstParagraph = null,
                // the first in-line node inside the paragraph (to avoid selection of absolute drawings at para start)
                firstInlineNode = null,
                // whether this is an empty selection in slide mode
                isEmptyPageContent = false;

            if (node) {
                searchNode = node;
            } else {
                pageContentNode = DOM.getPageContentNode(rootNode);

                if (pageContentNode.length > 0) {
                    searchNode = pageContentNode;
                    if (app.getModel().useSlideMode() && pageContentNode.children().length === 0) { isEmptyPageContent = true; }
                } else {
                    searchNode = rootNode;
                }
            }

            if (isEmptyPageContent) { return null; }

            firstParagraph = Utils.findDescendantNode(searchNode, DOM.PARAGRAPH_NODE_SELECTOR);
            firstInlineNode = getFirstTextCursorNode(firstParagraph, options);

            return Position.getOxoPosition(rootNode, firstInlineNode, 0);
        }

        /**
         * Returns the last logical position in the document.
         * If the optional parameter node is specified, it returns the
         * last logical position inside this node. This optional parameter
         * can be used for text frames.
         *
         * @param {HTMLElement|jQuery} [node]
         *  An optional DOM element object whose last logical position
         *  will be searched. If this object is a jQuery collection,
         *  uses the first node it contains.
         *  If it is not specified, the page content node will be
         *  used as default.
         */
        function getLastPosition(node) {

            var // the node, in which the first position is searched
                searchNode = node || (DOM.getPageContentNode(rootNode).length > 0 ? DOM.getPageContentNode(rootNode) : rootNode),
                // the last span inside the specified node
                lastSpan = Utils.findDescendantNode(searchNode, function () { return DOM.isPortionSpan(this); }, { reverse: true });

            return Position.getOxoPosition(rootNode, lastSpan, lastSpan.firstChild.nodeValue.length);
        }

        /**
         * Returns the first logical text cursor position in the document
         * (skips leading floating drawing objects in the first paragraph).
         *
         * @returns {Number[]|Null}
         *  The first logical text position in the document.
         *  Or null, if this is the empty selection (presentation only).
         */
        function getFirstTextPosition() {

            var // the page content node
                pageContentNode = DOM.getPageContentNode(rootNode),
                // the first in-line node inside the paragraph
                firstInlineNode = null;

            // special case for empty presentation documents
            if (app.getModel().useSlideMode() && pageContentNode.children().length === 0) { return null; }

            firstInlineNode = getFirstTextCursorNode(Utils.findDescendantNode(pageContentNode.length > 0 ? pageContentNode : rootNode, DOM.PARAGRAPH_NODE_SELECTOR));

            return Position.getOxoPosition(rootNode, firstInlineNode, 0);
        }

        /**
         * Clears the internal clipboard node.
         */
        function clearClipboardNode() {
            app.destroyImageNodes(clipboardNode);
            clipboardNode.empty().data('source-nodes', null);
        }

        /**
         * Checking whether a position is a possible text position.
         * This is a weak, but very fast check.
         * Another check is available in 'this.isValidTextSelection()',
         * that checks, whether the startPosition and the endPosition
         * contain valid positions.
         *
         * @param {Number[]} position
         *  The logical position to be checked.
         *
         * @returns {Boolean}
         *  Whether the given selection is a possible text selection.
         */
        function isPossibleTextSelection(position) {
            // only checking length of position array
            return position.length > 1;  // -> fast solution
            // the precise, but slow solution:
            // var localPos = _.clone(position);
            // localPos.pop();
            // return (Position.getParagraphElement(rootNode, localPos) !== null);
        }

        /**
         * Handling an optional selection of the text frame, if this is a (text)
         * selection inside a text frame. In this case this will be a 'double'
         * selection of text and surrounding text frame. This will not lead to
         * a drawing selection. The text selection stays the more important
         * selection.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM node containing the (text) selection.
         */
        function handleAdditionalTextFrameSelection(node) {

            var // a text frame node, that contains the specified node
                textFrameNode = $(node).closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR),
                // the content node inside the drawing
                contentNode = null;

            selectedTextFrameDrawing = $();

            if (textFrameNode.length > 0) {
                // it is necessary to make a drawing selection
                selectedTextFrameDrawing = $(node).closest(DrawingFrame.NODE_SELECTOR);

                // no additional selection, if the node is a comment
                if (DOM.isCommentNode(selectedTextFrameDrawing)) {
                    selectedTextFrameDrawing = $();
                    return;
                }

                // only groups can be selected, not the drawings inside the group
                if (selectedTextFrameDrawing.hasClass('grouped')) {
                    selectedTextFrameDrawing = DrawingFrame.getGroupNode(selectedTextFrameDrawing, { farthest: true, rootNode: rootNode }) || $();
                }

                // special handling for Mozilla browsers, to avoid, that the text frame
                // content is always in the foreground (36099)
                if ((_.browser.Firefox)) {
                    contentNode = DrawingFrame.getContentNode(selectedTextFrameDrawing);
                    if (_.isString(contentNode.attr('_moz_abspos'))) {
                        contentNode.removeAttr('_moz_abspos');
                    }
                }

                // drawing the selection around the drawing
                if (selectedTextFrameDrawing.length > 0) {
                    self.drawDrawingSelection(app, selectedTextFrameDrawing);
                    // -> this is not a real drawing selection, only the border is painted around the drawing
                    // -> the start and end position are not modified.
                }
            }

        }

        /**
         * Returns an array of DOM ranges representing the current browser
         * selection inside the passed container node.
         *
         * @param {HTMLElement|jQuery} containerNode
         *  The DOM node containing the returned selection ranges. Ranges not
         *  contained in this node will not be included into the result.
         *
         * @returns {Object}
         *  An object that contains a property 'active' with a DOM.Range object
         *  containing the current anchor point in its 'start' property, and
         *  the focus point in its 'end' property. The focus point may precede
         *  the anchor point if selecting backwards with mouse or keyboard. The
         *  returned object contains another property 'ranges' which is an
         *  array of DOM.Range objects representing all ranges currently
         *  selected. Start and end points of these ranges are already adjusted
         *  so that each start point precedes the end point.
         */
        function getBrowserSelection(containerNode) {

            var // the browser selection
                selection = window.getSelection(),
                // the result object
                result = { active: null, ranges: [] },
                // a single range object
                range = null,
                // the limiting points for valid ranges
                containerRange = DOM.Range.createRangeForNode(containerNode);

            // creates a DOM.Range object
            function createRange(startNode, startOffset, endNode, endOffset) {

                var // the range to be pushed into the array
                    range = DOM.Range.createRange(startNode, startOffset, endNode, endOffset),
                    // check that the nodes are inside the root node (with adjusted clone of the range)
                    adjustedRange = range.clone().adjust();

                return ((DOM.Point.comparePoints(containerRange.start, adjustedRange.start) <= 0) && (DOM.Point.comparePoints(adjustedRange.end, containerRange.end) <= 0)) ? range : null;
            }

            if (LOG_SELECTION) { Utils.info('Selection.getBrowserSelection(): reading browser selection...'); }

            // get anchor range which preserves direction of selection (focus node
            // may be located before anchor node)
            if (selection.rangeCount >= 1) {
                result.active = createRange(selection.anchorNode, selection.anchorOffset, selection.focusNode, selection.focusOffset);
                if (LOG_SELECTION) { Utils.log('  anchor=' + result.active); }
            }

            // read all selection ranges
            for (var index = 0; index < selection.rangeCount; index += 1) {
                // get the native selection Range object
                range = selection.getRangeAt(index);
                // translate to the internal text range representation
                range = createRange(range.startContainer, range.startOffset, range.endContainer, range.endOffset);
                // push, if range is valid
                if (range) {
                    if (LOG_SELECTION) { Utils.log('  ' + result.ranges.length + '=' + range); }
                    result.ranges.push(range.adjust());
                }
            }

            return result;
        }

        /**
         * Info: This must be a private function!
         * This function allows direct manipulation of the logical start and end
         * position. This is only allowed in private functions.
         * This function can be used in combination with the event 'position:calculated'.
         * Listener to this event must use this function to manipulate the logical
         * position(s).
         *
         * @param {Number[]} pos
         *  The new logical position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.start=true]
         *      If set to true, the logical start position will be overwritten
         *      with the specified logical position. Otherwise the logical end
         *      position will be overwritten.
         *  @param {Boolean} [options.selectionExternallyModified=false]
         *      If set to true, the call of the function restoreBrowserSelection
         *      will be forced, after the 'position:calculated' event was triggered.
         *      This forcing is necessary, when the selection was modified by the
         *      triggered function and if the selection is a backward selection in IE.
         *      Otherwise the new expanded selection will not be visible.
         *      Important:
         *      Unfortunately using this flag has a negative side effect.
         *      It destroys the backward selection with shift-button in IE, because
         *      the 'backwards' information is lost after calling the function
         *      'retoreBrowserSelection'.
         *      Additionally a Chrome browser selection problem can be fixed by reducing
         *      the logical end position temporary by 1. In a multi paragraph selection
         *      the final paragraph selection is not shown correctly.
         */
        function updateLogicalPosition(pos, options) {

            var // whether the start or end position will be updated
                isStart = Utils.getBooleanOption(options, 'start', true);

            if (isStart) {
                startPosition = _.clone(pos);
            } else {
                endPosition = _.clone(pos);
            }

            selectionExternallyModified = Utils.getBooleanOption(options, 'selectionExternallyModified', false);
            endSelectionExternallyModified = selectionExternallyModified && !isStart;
        }

        /**
         * Sets the browser selection to the passed DOM ranges.
         *
         * @param {HTMLElement|jQuery} containerNode
         *  The DOM node that must contain the passed selection ranges. This
         *  node must be focusable, and will be focused after the browser
         *  selection has been set (except if specified otherwise, see options
         *  below).
         *
         * @param {DOM.Range[]|DOM.Range} ranges
         *  The DOM ranges representing the new browser selection. May be an
         *  array of DOM range objects, or a single DOM range object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         */
        function setBrowserSelection(containerNode, ranges, options) {

            var // the browser selection
                selection = window.getSelection(),
                // whether to restore old focus element (always restore, if application is not active)
                restoreFocus = !app.getView().isVisible() || Utils.getBooleanOption(options, 'preserveFocus', false),
                // the current focus element
                focusNode = $(Utils.getActiveElement()),
                // whether this call was triggered by an operation
                operationTriggered = Utils.getBooleanOption(options, 'operationTriggered', false);

            if (LOG_SELECTION) { Utils.info('Selection.setBrowserSelection(): writing browser selection...'); }

            // Bug 26283: Browsers are picky how to correctly focus and select text
            // in a node that is content-editable: Chrome and IE will change the
            // scroll position of the editable node, if it has been focused
            // explicitly while not having an existing browser selection (or even
            // if the selection is not completely visible). Furthermore, the node
            // will be focused automatically when setting the browser selection.
            // Firefox, on the other hand, wants to have the focus already set
            // before the browser selection can be changed, otherwise it may throw
            // exceptions. Additionally, changing the browser selection does NOT
            // automatically set the focus into the editable node.
            // Performance: This is very expensive and should only be executed, if
            // it is really necessary. Setting text selection after insertOperations
            // should not execute this code.
            if (_.browser.Firefox && !Utils.getBooleanOption(options, 'simpleTextSelection', false)) {
                $(containerNode).focus();
            }

            // Clear the old browser selection.
            // Bug 28515, bug 28711: IE fails to clear the selection (and to modify
            // it afterwards), if it currently points to a DOM node that is not
            // visible anymore (e.g. the 'Show/hide side panel' button). Workaround
            // is to move focus to an editable DOM node which will cause IE to update
            // the browser selection object. The target container node cannot be used
            // for that, see comments above for bug 26283. Using another focusable
            // node (e.g. the body element) is not sufficient either. Interestingly,
            // even using the (editable) clipboard node does not work here. Setting
            // the new browser selection below will move the browser focus back to
            // the application pane.
            // This is typically not necessary, if 'setBrowserSelection' was called after
            // executing an operation (not only insertText operation)
            Utils.clearBrowserSelection({ rescueFocus: !operationTriggered });

            // convert to array
            ranges = _.getArray(ranges);

            // single range: use attributes of the Selection object (anchor/focus)
            // directly to preserve direction of selection when selecting backwards
            if (SELECTION_COLLAPSE_EXPAND_SUPPORT && (ranges.length === 1) && !$(ranges[0].start.node).is('tr')) {
                if (LOG_SELECTION) { Utils.log('  0=' + ranges[0]); }
                try {
                    selection.collapse(ranges[0].start.node, ranges[0].start.offset);
                    selection.extend(ranges[0].end.node, ranges[0].end.offset);
                    ranges = [];
                } catch (ex) {
                    if (!(_.browser.Firefox && $(ranges[0].start.node).is('div.clipboard'))) {  // Fix for 26645, no warning required in Firefox
                        Utils.warn('Selection.setBrowserSelection(): failed to collapse/expand to range: ' + ranges[0]);
                    }
                    // retry with regular code below
                    selection.removeAllRanges();
                }
            }

            // Performance for Internet Explorer! Alternative way to set selection
            // Can be removed , if IE supports selection API and SELECTION_COLLAPSE_EXPAND_SUPPORT becomes true.
            // Problem: textRange.select(); requires a long time in large documents, similar with
            // selection.addRange(docRange);
            /*
            if (!SELECTION_COLLAPSE_EXPAND_SUPPORT && document.body.createTextRange && (ranges.length === 1) && !$(ranges[0].start.node).is('tr')) {

                // the text range object
                var textRange = document.body.createTextRange();

                if (textRange) {
                    textRange.moveToElementText(ranges[0].start.node.parentNode);
                    textRange.collapse(true);
                    textRange.move('character', ranges[0].start.offset);
                    textRange.select();  // similar time consuming like selection.addRange(docRange);
                    ranges = [];
                }
            }
            */

            // create a multi-selection
            _(ranges).each(function (range, index) {

                var docRange = null;

                if (LOG_SELECTION) { Utils.log('  ' + index + '=' + range); }
                try {
                    range.adjust();  // 26574, backward selection
                    docRange = window.document.createRange();
                    docRange.setStart(range.start.node, range.start.offset);
                    docRange.setEnd(range.end.node, range.end.offset);
                    // 'selection.addRange' is time expensive in IE. It must be replaced by 'selection.modify'
                    // for selected operations like insertText, as soon as this is available in IE.
                    // Alternatively 'SELECTION_COLLAPSE_EXPAND_SUPPORT' should be supported, if available.
                    selection.addRange(docRange);
                } catch (ex) {
                    Utils.error('Selection.setBrowserSelection(): failed to add range to selection: ' + range);
                }
            });

            // No visible browser selection of text spans that are children of slides -> setting browser selection into clip board
            // -> this is the mode for the 'slide selection'
            if (app.getModel().useSlideMode() && (self.isTopLevelTextCursor() || self.isMultiSelection()) && Utils.getDomNode(containerNode) !== Utils.getDomNode(self.getClipboardNode())) {
                self.setFocusIntoClipboardNode({ specialTouchHandling: true });
                self.setBrowserSelectionToContents(self.getClipboardNode(), { preserveFocus: true });

                // setting new values for start and end position
                startPosition[startPosition.length - 1] = 0;
                endPosition[endPosition.length - 1] = 0;
            }

            // restore the old focus element if specified
            if (restoreFocus && !focusNode.is(containerNode)) {
                focusNode.focus();
            }
        }

        /**
         * Returns true if the clipboard contains exactly the given drawings.
         */
        function clipboardContainsDrawings(drawings) {
            var clipboardDrawings = clipboardNode.data('source-nodes') || [];

            if (!_.isArray(drawings) || _.isEmpty(drawings) || (drawings.length !== clipboardDrawings.length)) { return false; }
            return _.every(drawings, function (drawing) {
                return (_.indexOf(clipboardDrawings, drawing) !== -1);
            });
        }

        /**
         * Initializes this selection with the passed start and end points, and
         * validates the browser selection by moving the start and end points
         * to editable nodes.
         *
         * @param {Object} browserSelection
         *  The new browser selection descriptor. See the method
         *  Selection.getBrowserSelection() for details.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         *  @param {Object} [options.event]
         *      The original browser event that has caused the changed
         *      selection.
         *
         *  @param {Object} [newPositions]
         *   An object with a property 'start' and optionally a property 'end',
         *   that describes the logical position of the selection range. If this
         *   optional object is defined, it must be used for the simplified process
         *   of applying the browser selection. This simplified process avoids the
         *   recalculation of logical positions via Position.getTextLevelOxoPosition.
         */
        function applyBrowserSelection(browserSelection, options, newPositions) {

            var // active range from selection, preserving direction
                anchorPoint = null,
                focusPoint = null,
                // adjusted points (start before end)
                startPoint = null, endPoint = null,
                // a modified text node (for replacements of template texts)
                newTextNode = null,
                // whether position is a single cursor
                isCursor = false,
                // table containing the selection
                tableNode = null,
                // selected drawing node
                nodeInfo = null,
                // a temporary helper position
                tmpPosition = null,
                // the browser event passed to this method
                browserEvent = Utils.getOption(options, 'event'),
                // browser event from a touch device
                browserTouchEvent = Utils.TOUCHDEVICE && _.isObject(browserEvent),
                // old selection, used to detect changed selection
                oldStartPosition = _.clone(startPosition), oldEndPosition = _.clone(endPosition),
                // whether the browser selection shall be restored
                restoreSelection = Utils.getBooleanOption(options, 'restoreSelection', true),
                // whether this function was triggered by an insert operation, for example insertText
                insertOperation = Utils.getBooleanOption(options, 'insertOperation', false),
                // whether this function was triggered by an split paragraph operation
                splitOperation = Utils.getBooleanOption(options, 'splitOperation', false),
                // whether this function was triggered by an 'simplified text selection' operation, for example insertText, backspace, delete or return
                simpleTextSelection = Utils.getBooleanOption(options, 'simpleTextSelection', false),
                // whether the simplified process can be used to set the new text selection
                simplifiedProcess = !!newPositions,
                // whether the 'change' event must be triggered -> informing remote users about the current selection, even if it was not modified (31832)
                forceTrigger = Utils.getBooleanOption(options, 'forceTrigger', false),
                // whether an existing cell selection shall be restored
                keepCellSelection = Utils.getBooleanOption(options, 'keepCellSelection', false),
                // whether the change track pop up triggered this change of selection
                keepChangeTrackPopup = Utils.getBooleanOption(options, 'keepChangeTrackPopup', false),
                // whether a drawing inside a drawing group is selected
                drawingInGroupSelection = Utils.getBooleanOption(options, 'drawingInGroupSelection', false),
                // whether a drawing in the drawing layer is selected
                drawingLayerSelection = Utils.getBooleanOption(options, 'drawingLayerSelection', false),
                // whether this call was triggered by an operation (needs to be expanded)
                operationTriggered = insertOperation || splitOperation,
                // whether position is top level (not in tables, text frames, etc. ) and is simple selection
                isTopLevelPosition = null,
                // whether special handling for complex fields of the placeholder is required (39099)
                handlePlaceHolderComplexField = false;

            // checking, that a selection range is not inside and outside a text frame
            // -> modifying one position, so that both are inside the same text frame
            function handleTextFrameSelectionRange() {

                var // an optional text frame for the start position
                    startTextFrame = Position.getClosestDrawingTextframe(rootNode, startPosition),
                    // an optional text frame for the end position
                    endTextFrame = Position.getClosestDrawingTextframe(rootNode, endPosition);

                if (startTextFrame) {
                    if (endTextFrame) {
                        // are start and end position inside the same text frame?
                        if (startTextFrame[0] !== endTextFrame[0]) { endPosition = getLastPosition(startTextFrame); }
                    } else {
                        endPosition = getLastPosition(startTextFrame);
                    }
                } else {
                    if (endTextFrame) { startPosition = getFirstPosition(endTextFrame); }
                }
            }

            if (simplifiedProcess) {

                // check for cell range selection (must be in the same table)
                cellRangeSelected =  false;
                tableSelected = false;

                if (!_.isArray(newPositions.start)) {
                    Utils.error('Selection.applyBrowserSelection(): no start position set in newPositions: ' + JSON.stringify(newPositions));
                }

                // if we apply selection on top level elements, we can avoid checks for inside tables and text frames
                isTopLevelPosition = newPositions.start.length === 2 && !newPositions.end;

                startPosition = _.clone(newPositions.start);
                endPosition = newPositions.end ? _.clone(newPositions.end) : _.clone(newPositions.start);

            } else if (keepCellSelection) {

                // simplified behavior for modifying position on remote clients -> this must not destroy local cell selection
                cellRangeSelected = true;

            } else {

                anchorPoint = browserSelection.active.start;
                focusPoint = browserSelection.active.end;

                // check for cell range selection (must be in the same table)
                cellRangeSelected = DOM.isCellRangeSelected(focusPoint.node, anchorPoint.node);

                if (cellRangeSelected) {

                    // cell range selection is always ordered, no need to check for direction
                    backwards = false;

                    // convert multi-selection for cells to rectangular cell selection
                    startPoint = _(browserSelection.ranges).first().start;
                    endPoint = _(browserSelection.ranges).last().end;

                    // entire table selected, if number of cell range objects in selection is equal to number of table cells
                    tableNode = Utils.findClosest(rootNode, focusPoint.node, DOM.TABLE_NODE_SELECTOR);
                    tableSelected = browserSelection.ranges.length === DOM.getTableCells(tableNode).length;

                } else {

                    // get range direction (check for real range, DOM.Point.comparePoints() is expensive)
                    isCursor = browserSelection.active.isCollapsed();

                    // Fix for 28344 in combination with special Firefox behavior for ranges ending directly behind list labels (see Position.getTextNodeFromCurrentNode)
                    if ((anchorPoint.node === focusPoint.node) && (anchorPoint.offset === focusPoint.offset - 1) && (DOM.isParagraphNode(anchorPoint.node)) && (DOM.isListLabelNode(anchorPoint.node.childNodes[anchorPoint.offset]))) {
                        // removing listlabels from selection
                        anchorPoint.offset = 1;
                        isCursor = true;
                    }

                    backwards = !isCursor && (DOM.Point.comparePoints(anchorPoint, focusPoint) > 0);
                    tableSelected = false;

                    // adjust start and end position
                    startPoint = backwards ? focusPoint : anchorPoint;
                    endPoint = backwards ? anchorPoint : focusPoint;

                    // checking for text frame template text, that will be removed
                    if (DOM.isTextFrameTemplateTextNode(anchorPoint.node)) {

                        newTextNode = app.getModel().removeTemplateTextFromTextSpan(anchorPoint.node);

                        if (newTextNode) {
                            // Utils.getDomNode(mySpan).firstChild;
                            startPoint.node = newTextNode;
                            endPoint.node = startPoint.node;

                            startPoint.offset = 0;
                            endPoint.offset = 0;
                        }
                    }

                    // whether this is a selection inside a complex field of type PLACEHOLDER (to be handled in mouseup, not mousedown)
                    handlePlaceHolderComplexField = (!browserEvent || (browserEvent && browserEvent.type !== 'mousedown')) && DOM.isInsidePlaceHolderComplexField(startPoint.node.parentNode);
                }

                // calculate start and end position (always text positions, also in cell range mode)
                startPosition = Position.getTextLevelOxoPosition(startPoint, rootNode, false, !isCursor);
                endPosition = isCursor ? _.clone(startPosition) : Position.getTextLevelOxoPosition(endPoint, rootNode, true, !isCursor);

                // fast validity check, that start position is before end position (36249)
                if (!isCursor && !Position.isValidPositionOrder(startPosition, endPosition)) {
                    tmpPosition = startPosition;
                    startPosition = endPosition;
                    endPosition = tmpPosition;
                    Utils.warn('Selection.applyBrowserSelection(): switching of start and end position required');
                }

                // repairing the old value for endPosition, if this is a selected drawing in a drawing group
                if (drawingInGroupSelection || drawingLayerSelection) { endPosition = Position.increaseLastIndex(endPosition); }

                if (!_.isArray(startPosition)) {
                    Utils.error('Selection.applyBrowserSelection(): no position for node ' + startPoint);
                }
                if (!_.isArray(endPosition)) {
                    Utils.error('Selection.applyBrowserSelection(): no position for node ' + endPoint);
                }
                if (!_.isArray(startPosition) || !_.isArray(endPosition) || !isPossibleTextSelection(startPosition) || !isPossibleTextSelection(endPosition)) {
                    startPosition = getFirstTextPosition();
                    endPosition = _.clone(startPosition);
                    isCursor = true;
                }
            }

            // check for drawing selection
            if (selectedDrawing.length > 0) {
                nodeInfo = Position.getDOMPosition(rootNode, startPosition, true);
                // Not modifying selection, if this was a click on a drawing inside a drawing group
                if (nodeInfo && DrawingFrame.isDrawingFrame(nodeInfo.node) && $(nodeInfo.node).hasClass('grouped') && selectedDrawing[0] === DrawingFrame.getGroupNode(nodeInfo.node)[0]) {
                    return;
                }

                DrawingFrame.clearSelection(selectedDrawing);
                app.getView().clearVisibleDrawingAnchor();
                selectedDrawing = $();
            }

            // check for an additional text frame selection, that is no pure drawing selection
            if (selectedTextFrameDrawing.length > 0) {
                DrawingFrame.clearSelection(selectedTextFrameDrawing);
                app.getView().clearVisibleDrawingAnchor();
                if (app.getModel().useSlideMode() && !simplifiedProcess && (!browserEvent || browserEvent.type !== 'mouseup') && !Position.hasSameParentComponent(oldStartPosition, startPosition, 2)) {
                    app.getModel().getDrawingStyles().handleEmptyPlaceHolderDrawing(selectedTextFrameDrawing);
                }
                selectedTextFrameDrawing = $();
            }

            // removing a drawing multi selection
            if (self.isMultiSelectionSupported() && self.isMultiSelection()) { self.clearMultiSelection(); }

            // draw a new drawing selection
            if (!simplifiedProcess && !cellRangeSelected && (self.isSingleComponentSelection() || drawingInGroupSelection || drawingLayerSelection)) {
                nodeInfo = nodeInfo || Position.getDOMPosition(rootNode, startPosition, true);
                if (nodeInfo && DrawingFrame.isDrawingFrame(nodeInfo.node)) {
                    selectedDrawing = $(nodeInfo.node);
                    // only groups can be selected, not the drawings inside the group
                    if (selectedDrawing.hasClass('grouped')) {
                        selectedDrawing = DrawingFrame.getGroupNode(selectedDrawing, { farthest: true, rootNode: rootNode });
                        startPosition = Position.getOxoPosition(rootNode, selectedDrawing, 0);
                        endPosition = Position.getOxoPosition(rootNode, selectedDrawing, 1);
                    }
                    self.drawDrawingSelection(app, selectedDrawing);
                }
            }

            // update table selection
            // Performance: NOT required for insertText, insertTab, splitParagraph, ..., but might be necessary for backspace, delete and return
            if (!insertOperation && !splitOperation) {
                if (!isTopLevelPosition) {
                    selectedTable.removeClass('selected');
                    selectedTable = $(self.getEnclosingTable()).addClass('selected');
                }

                // update the clipboard (clone drawing contents, otherwise clear)
                // Performance: NOT required for insertText
                self.updateClipboardNode();
            }

            // checking, that a selection range is not inside and outside a textframe
            if (!isCursor && !isTopLevelPosition && selectedDrawing.length === 0) { handleTextFrameSelectionRange(); }

            // informing others about the new selection
            self.trigger('position:calculated', updateLogicalPosition, { handlePlaceHolderComplexField: handlePlaceHolderComplexField });

            // refreshing local variable isCursor might be necessary
            if (handlePlaceHolderComplexField) {
                isCursor = _.isEqual(self.getStartPosition(), self.getEndPosition());
                // updating 'backwards' after switching vom cursor to range; needs to be true for cursorLeft and cursorUp
                if (!isCursor) { backwards = isBackwardCursor; }
            }

            // draw correct browser selection (but not if browser does not
            // support backward selection mode, or on touch devices to keep the
            // text selection menu open after a double tap event)
            if (restoreSelection && (isCursor || !backwards || SELECTION_COLLAPSE_EXPAND_SUPPORT || selectionExternallyModified)) {
                if (!browserTouchEvent) {
                    options = Utils.extendOptions(options, { isCursor: isCursor, operationTriggered: operationTriggered });
                    self.restoreBrowserSelection(options);
                } else if (self.getSelectionType() === 'text') {
                    //handleAdditionalTextFrameSelection is a newer feature, which is called by restoreBrowserSelection
                    //but not on touch, so we call it now directly
                    startPoint = Position.getDOMPosition(rootNode, startPosition);
                    if (startPoint && startPoint.node) { handleAdditionalTextFrameSelection(startPoint.node); }
                }

                selectionExternallyModified = false; // call of restoreBrowserSelection was forced by 'updateLogicalPosition'
            }

            // notify listeners
            if ((!_.isEqual(startPosition, oldStartPosition) || !_.isEqual(endPosition, oldEndPosition)) || forceTrigger || forceSelectionChangeEvent) {
                self.trigger('change', { insertOperation: insertOperation, splitOperation: splitOperation, simpleTextSelection: simpleTextSelection, keepChangeTrackPopup: keepChangeTrackPopup, event: browserEvent });
                forceSelectionChangeEvent = false;  // needed after switching the target
            }
            self.trigger('update', { simpleTextSelection: simpleTextSelection, event: browserEvent });
            self.trigger('changecursorinfo', { start: startPosition, end: endPosition });
        }

        /**
         * Changes the current text position or selection by one character or
         * inline component.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.extend=false]
         *      If set to true, the current selection will be extended at the
         *      current focus point. Otherwise, the text cursor will be moved
         *      starting from the current focus point.
         *  @param {Boolean} [options.backwards=false]
         *      If set to true, the selection will be moved back by one
         *      character; otherwise the selection will be moved ahead.
         *  @param {Boolean} [options.verticalCellSelection=false]
         *      If set to true, the selection will be converted from a
         *      text selection to a vertical cell selection. This can be used
         *      in Firefox only, where multi-cell selection is possible.
         */
        function moveTextCursor(options) {

            var // whether to extend the selection
                extend = Utils.getBooleanOption(options, 'extend', false),
                // whether to move cursor back
                backwards = Utils.getBooleanOption(options, 'backwards', false),
                // whether the selection has to be converted into a vertical cell selection
                verticalCellSelection = Utils.getBooleanOption(options, 'verticalCellSelection', false),
                // text node at the current focus position
                focusNodeInfo = Position.getDOMPosition(rootNode, getFocusPosition()),
                // text node at the current anchor position (changes with focus node without SHIFT key)
                anchorNodeInfo = extend ? Position.getDOMPosition(rootNode, getAnchorPosition()) : focusNodeInfo,
                // the text node at the anchor position
                anchorTextNode = anchorNodeInfo && anchorNodeInfo.node && (anchorNodeInfo.node.nodeType === 3) ? anchorNodeInfo.node : null,
                // the text node at the focus position
                focusTextNode = focusNodeInfo && focusNodeInfo.node && (focusNodeInfo.node.nodeType === 3) ? focusNodeInfo.node : null,
                // space for other nodes
                node = null;

            // find the closest following inline node of the passed node
            function findNextNonEmptyInlineNode(node) {
                return Utils.findNextSiblingNode(node, inlineNodeSelector);
            }

            // find the closest preceding inline node of the passed node
            function findPreviousNonEmptyInlineNode(node) {
                return Utils.findPreviousSiblingNode(node, inlineNodeSelector);
            }

            // checks if node is before special complex field, and returns end range of that field
            function findNextValidNodeAfterSpecialField(node) {
                var nextSib = node.nextSibling,
                    nextNextSib = nextSib ? nextSib.nextSibling : null;

                if (DOM.isMarginalNode(node) && DOM.isRangeMarkerStartNode(nextSib) && (nextNextSib && DOM.isSpecialField(nextNextSib))) {
                    node = nextNextSib.nextSibling;
                    if (DOM.isRangeMarkerEndNode(node)) {
                        return node;
                    }
                }
                return null;
            }

            // check, if the specified point describes a position, that need to
            // be handled in a special way, caused by a preceding end range marker.
            // If the offset is '1', so that it becomes '0' after moving cursor
            // in backward direction, typically the node is changed to the previous
            // node. But not so, if there is a preceding range end marker node.
            // And also not, if the preceding node is a comment place holder node
            // that follows an end range marker node (typical schema in OOXML).
            // And if there is an empty text span between the end range marker node
            // and the following comment place holder node, this is also handled
            // within this function.
            //
            // If this function return 'true' this means, that the node defined by
            // point.node shall stay valid. And only the offset is reduced to 0.
            // Typically the previous node is searched, if offset is reduced to 0.
            function stayBehindRangeEndMarker(point) {

                var // the parent node of the specified point node
                    parent = null,
                    // the previous node
                    previous = null;

                if (point.offset === 1) {

                    parent = point.node.parentNode;
                    previous = parent.previousSibling;

                    if (!previous) { return false; }

                    // found range marker end node directly
                    if (DOM.isRangeMarkerEndNode(previous)) { return true; }

                    if (DOM.isCommentPlaceHolderNode(previous)) {

                        previous = previous.previousSibling;

                        if (!previous) { return false; }

                        // found range marker end node before a comment place holder node
                        if (DOM.isRangeMarkerEndNode(previous)) { return true; }

                        if (DOM.isEmptySpan(previous)) {

                            previous = previous.previousSibling;

                            if (!previous) { return false; }

                            // found range marker end node before an empty text span before a comment place holder node
                            if (DOM.isRangeMarkerEndNode(previous)) { return true; }
                        }
                    }
                }

                return false;
            }

            // find the first inline node or empty node of the following paragraph
            function findFirstInlineNodeInNextParagraph(node) {
                var containerNode = DOM.isPageNode(rootNode) ? DOM.getPageContentNode(rootNode) : rootNode,
                    skipSelector = DrawingFrame.NODE_SELECTOR + ', ' + DOM.PAGEBREAK_NODE_SELECTOR,
                    paragraph = Utils.findNextNode(containerNode, node, DOM.PARAGRAPH_NODE_SELECTOR, skipSelector);
                return paragraph && getFirstTextCursorNode(paragraph);
            }

            // find the last inline node or empty node of the preceding paragraph
            function findLastInlineNodeInPreviousParagraph(node) {
                var paragraph = Utils.findClosest(rootNode, node, DOM.PARAGRAPH_NODE_SELECTOR),
                    containerNode = DOM.isPageNode(rootNode) ? DOM.getPageContentNode(rootNode) : rootNode,
                    skipSelector = DrawingFrame.NODE_SELECTOR + ', ' + DOM.PAGEBREAK_NODE_SELECTOR;
                paragraph = paragraph && Utils.findPreviousNode(containerNode, paragraph, DOM.PARAGRAPH_NODE_SELECTOR, skipSelector);
                return paragraph && getLastTextCursorNode(paragraph);
            }

            // move to start of text span; or to end of text span preceding the inline component
            function jumpBeforeInlineNode(node) {
                if (DOM.isTextSpan(node)) {
                    // text span: jump to its beginning
                    focusNodeInfo.node = node.firstChild;
                    focusNodeInfo.offset = 0;
                } else if (DOM.isTextSpan(node.previousSibling)) {
                    // jump to end of the span preceding the inline component
                    focusNodeInfo.node = node.previousSibling.firstChild;
                    focusNodeInfo.offset = focusNodeInfo.node.nodeValue.length;
                } else {
                    Utils.warn('Selection.moveTextCursor.jumpBeforeInlineNode(): missing text span preceding a component node');
                }
            }

            // skip inline component; move to end or to specific offset of text span
            function jumpOverInlineNode(node, offset) {
                if (DOM.isTextSpan(node)) {
                    // text span: jump to passed offset, or to its end
                    focusNodeInfo.node = node.firstChild;
                    focusNodeInfo.offset = _.isNumber(offset) ? offset : focusNodeInfo.node.nodeValue.length;

                } else if (DOM.isTextSpan(DOM.getAllowedNeighboringNode(node))) {
                    // jump to beginning of the span following the inline component
                    // (may be an empty span before floating node)
                    focusNodeInfo.node = DOM.getAllowedNeighboringNode(node).firstChild;
                    focusNodeInfo.offset = 0;
                } else {
                    Utils.warn('Selection.moveTextCursor.jumpOverInlineNode(): missing text span following a component node');
                }
            }

            // setting a cell selection instead of a text selection
            function switchToCellSelectionHorz() {

                var // an array of ranges for cell selections
                    ranges = [],
                    // the previous or following row node required for cell selections
                    siblingRow = null,
                    // a temporarily required node
                    helperNode = null;

                // try to find the nearest table cell containing the text node
                focusNodeInfo = DOM.Point.createPointForNode($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR)[0]);
                anchorNodeInfo = DOM.Point.createPointForNode($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR)[0]);

                if (backwards) {
                    // selecting the cell and its previous sibling, if available (the previous sibling has to be the active cell)
                    focusNodeInfo.offset += 1;
                    if (anchorNodeInfo.offset > 0) {

                        helperNode = DOM.Point.createPointForNode($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR)[0]);
                        anchorNodeInfo.offset -= 1;

                        ranges.push({ start: anchorNodeInfo, end: helperNode });
                        ranges.push({ start: helperNode, end: focusNodeInfo });

                        self.setAnchorCellRange(ranges[1]);

                    } else {
                        // the complete row and the previous row need to be selected
                        if ($(focusNodeInfo.node).prev().length > 0) {
                            siblingRow = $(focusNodeInfo.node).prev();
                            ranges.push(new DOM.Range(new DOM.Point(siblingRow, 0), new DOM.Point(siblingRow, siblingRow.children().length)));
                            ranges.push(new DOM.Range(new DOM.Point(focusNodeInfo.node, 0), new DOM.Point(focusNodeInfo.node, $(focusNodeInfo.node).children().length)));
                            self.setAnchorCellRange(new DOM.Range(ranges[1].start, new DOM.Point(ranges[1].start.node, ranges[1].start.offset + 1)));
                        } else {
                            // Fix for 29402, not changing selection
                            if (self.isTextLevelSelection()) {
                                self.setTextSelection(self.getEndPosition(), Position.getFirstPositionInCurrentCell(rootNode, self.getEndPosition()));
                            }
                        }
                    }
                } else {
                    // selecting the cell and its following sibling, if available
                    if (focusNodeInfo.offset + 1 < $(focusNodeInfo.node).children().length) {
                        focusNodeInfo.offset += 2;  // selecting the current and the following cell
                        ranges.push({ start: anchorNodeInfo, end: focusNodeInfo });
                        self.setAnchorCellRange(new DOM.Range(anchorNodeInfo, new DOM.Point(anchorNodeInfo.node, anchorNodeInfo.offset + 1)));
                    } else {
                        // the complete row and the next row need to be selected
                        if ($(focusNodeInfo.node).next().length > 0) {
                            siblingRow = $(focusNodeInfo.node).next();
                            ranges.push(new DOM.Range(new DOM.Point(focusNodeInfo.node, 0), new DOM.Point(focusNodeInfo.node, $(focusNodeInfo.node).children().length)));
                            ranges.push(new DOM.Range(new DOM.Point(siblingRow, 0), new DOM.Point(siblingRow, siblingRow.children().length)));
                            self.setAnchorCellRange(new DOM.Range(new DOM.Point(ranges[0].end.node, ranges[0].end.offset - 1), ranges[0].end.node));
                        }
                    }
                }

                self.setCellSelection(ranges);
            }

            // setting a cell selection instead of a text selection
            function switchToCellSelectionVert() {

                var // an array of ranges for cell selections
                    ranges = [],
                    // the previous or following row node required for cell selections
                    siblingRow = null;

                // try to find the nearest table cell containing the text node
                if ($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR).length === 0) {
                    // Fix for 29402, not changing selection
                    if (self.isTextLevelSelection()) {
                        if (backwards) {
                            self.setTextSelection(Position.getFirstPositionInCurrentCell(rootNode, self.getEndPosition()), self.getEndPosition());
                        } else {
                            self.setTextSelection(self.getStartPosition(), Position.getLastPositionInCurrentCell(rootNode, self.getStartPosition()));
                        }
                    }
                } else {

                    focusNodeInfo = DOM.Point.createPointForNode($(focusTextNode).closest(DOM.TABLE_CELLNODE_SELECTOR)[0]);

                    if (backwards) {
                        // selecting the cell and its previous sibling in the row above, if available (the previous sibling has to be the active cell)
                        if ($(focusNodeInfo.node).prev().length > 0) {
                            siblingRow = $(focusNodeInfo.node).prev();
                            while (siblingRow.hasClass('pb-row') && siblingRow.prev().length > 0) { siblingRow = siblingRow.prev(); } // skip over page break row
                            ranges.push(new DOM.Range(new DOM.Point(siblingRow, focusNodeInfo.offset), new DOM.Point(siblingRow, focusNodeInfo.offset + 1)));
                            ranges.push(new DOM.Range(focusNodeInfo, new DOM.Point(focusNodeInfo.node, focusNodeInfo.offset + 1)));
                            self.setAnchorCellRange(ranges[ranges.length - 1]);
                        }
                    } else {
                        if ($(focusNodeInfo.node).next().length > 0) {
                            siblingRow = $(focusNodeInfo.node).next();
                            while (siblingRow.hasClass('pb-row') && siblingRow.next().length > 0) { siblingRow = siblingRow.next(); } // skip over page break row
                            ranges.push(new DOM.Range(focusNodeInfo, new DOM.Point(focusNodeInfo.node, focusNodeInfo.offset + 1)));
                            ranges.push(new DOM.Range(new DOM.Point(siblingRow, focusNodeInfo.offset), new DOM.Point(siblingRow, focusNodeInfo.offset + 1)));
                            self.setAnchorCellRange(ranges[0]);
                        }
                    }

                    self.setCellSelection(ranges);
                }
            }

            // check anchor and focus position
            if (!anchorTextNode || !focusTextNode) {

                if (!focusTextNode && focusNodeInfo && focusNodeInfo.node && DOM.isDrawingLayerNode(focusNodeInfo.node.parentNode)) {
                    focusNodeInfo.node = DOM.getDrawingPlaceHolderNode(focusNodeInfo.node).nextSibling.firstChild;
                    focusTextNode = focusNodeInfo && focusNodeInfo.node && (focusNodeInfo.node.nodeType === 3) ? focusNodeInfo.node : null;
                }

                if (!anchorTextNode) {
                    if (extend) {
                        if (anchorNodeInfo && anchorNodeInfo.node && DOM.isDrawingLayerNode(anchorNodeInfo.node.parentNode)) {
                            anchorNodeInfo.node = DOM.getDrawingPlaceHolderNode(anchorNodeInfo.node).nextSibling.firstChild;
                        }
                    } else {
                        anchorNodeInfo = focusNodeInfo;
                    }

                    anchorTextNode = anchorNodeInfo && anchorNodeInfo.node && (anchorNodeInfo.node.nodeType === 3) ? anchorNodeInfo.node : null;
                }

                if (!anchorTextNode || !focusTextNode) {
                    Utils.warn('Selection.moveTextCursor(): missing valid text position');
                    return false;
                }
            }

            // update focusNodeInfo according to the passed direction
            if (backwards) {

                if (verticalCellSelection && !cellRangeSelected && Position.isSameTableLevel(rootNode, self.getFocusPosition(), self.getAnchorPosition())) {
                    switchToCellSelectionVert();
                    return true;
                }

                // move back inside non-empty text span, but not always to the beginning
                if (focusNodeInfo.offset > 1) {
                    focusNodeInfo.offset -= 1;

                // stay behind range marker end nodes or comment place holder nodes
                } else if (stayBehindRangeEndMarker(focusNodeInfo)) {
                    focusNodeInfo.offset = 0;  // not changing the node

                // try to find the previous non-empty inline node in the own paragraph
                } else if ((node = findPreviousNonEmptyInlineNode(focusTextNode.parentNode))) {

                    // offset is 1, or preceding node is a text span: move cursor behind the previous inline node
                    if ((focusNodeInfo.offset === 1) || DOM.isTextSpan(node)) {
                        if (($(focusTextNode.parentNode).prev().hasClass('page-break') || DOM.isDrawingPlaceHolderNode(focusTextNode.parentNode.previousSibling) || DOM.isDrawingSpaceMakerNode(focusTextNode.parentNode.previousSibling) || DOM.isRangeMarkerNode(node.nextSibling)) && focusNodeInfo.offset === 0) {
                            // cursor is on first position before page break inside paragraph,
                            // and it has to jump to last position on previous page
                            // Check for image place holder node or space maker node is required
                            // for backwards travelling with cursor over node introduced with absolute
                            // positioned drawings.
                            // Also reducing the offset by 1, if there was a jump over a range marker end node.
                            jumpOverInlineNode(node, node.firstChild.nodeValue.length - 1); // TODO: special chars cannot be enumerated with text length of node

                            if ((focusNodeInfo.node.length === 1) && (focusNodeInfo.offset === 0) && (DOM.isFirstTextPositionBehindRangeStart(focusNodeInfo.node, focusNodeInfo.offset))) {
                                // special handling to jump completely over ranges, that have a length of 1
                                jumpOverRangeMarkerStartNode(focusNodeInfo); // jumping further to the left
                            }

                        } else {
                            jumpOverInlineNode(node);
                        }
                    // offset is 0, skip the previous inline component (jump to end of its preceding inline node)
                    } else {
                        // jump to end of preceding text span
                        jumpOverInlineNode(DOM.getAllowedNeighboringNode(node, { next: false }));

                        // try to find the correct trailing text span of the pre-preceding inline component
                        // (this may jump from the preceding empty span over floating nodes to the trailing
                        // text span of the previous inline component)
                        if (DOM.isEmptySpan(node.previousSibling) && (node = findPreviousNonEmptyInlineNode(node))) {
                            // skipping an inline node: jump to end of preceding text span (try to
                            // find text span following the next preceding component)
                            jumpOverInlineNode(node);
                        }
                    }

                // after first character in paragraph: go to beginning of the first text span
                } else if (focusNodeInfo.offset === 1) {
                    focusNodeInfo.offset = 0;

                // in Mozilla based browsers a cell selection can be created inside tables
                } else if (extend && _.browser.Firefox && !cellRangeSelected &&
                        Position.isPositionInTable(rootNode, getFocusPosition()) &&
                        Position.isFirstPositionInTableCell(rootNode, getFocusPosition()) &&
                        Position.isSameTableLevel(rootNode, getFocusPosition(), self.getAnchorPosition())) {
                    switchToCellSelectionHorz();
                    return true;

                // block cursor traversal, if this is the first position inside a text frame
                } else if (focusNodeInfo.offset === 0 && DOM.isFirstTextSpanInTextframe(focusTextNode.parentNode)) {
                    if (extend) {
                        self.restoreBrowserSelection(); // this call is asynchronous -> avoid uncontrolled horizontal move
                    } else {
                        self.setTextSelection(self.getStartPosition());
                    }
                    return true;

                // try to find the last text position in the preceding paragraph
                } else if ((node = findLastInlineNodeInPreviousParagraph(focusTextNode.parentNode))) {
                    jumpOverInlineNode(node);
                }

            } else {

                if (verticalCellSelection && !cellRangeSelected && Position.isSameTableLevel(rootNode, self.getAnchorPosition(), self.getFocusPosition())) {
                    switchToCellSelectionVert();
                    return true;
                }

                // move ahead inside non-empty text span (always up to the end of the span)
                if (focusNodeInfo.offset < focusTextNode.nodeValue.length) {
                    focusNodeInfo.offset += 1;

                    // check for following range marker end nodes
                    if ((focusNodeInfo.offset === focusTextNode.nodeValue.length) && DOM.isLastTextPositionBeforeRangeEnd(focusNodeInfo.node, focusNodeInfo.offset)) {
                        jumpOverRangeMarkerEndNode(focusNodeInfo);
                    }
                // try to jump over special page fields in header/footer
                } else if ((node = findNextValidNodeAfterSpecialField(focusTextNode.parentNode))) {
                    jumpOverInlineNode(node, 1);
                // try to find the next non-empty inline node in the own paragraph
                } else if ((node = findNextNonEmptyInlineNode(focusTextNode.parentNode))) {
                    // skip only the first character of following non-empty text span
                    jumpOverInlineNode(node, 1);

                    // this first character might be the only character inside a range, so that
                    // the position must be behind the range end marker
                    if (focusNodeInfo.node.length === 1 && DOM.isLastTextPositionBeforeRangeEnd(focusNodeInfo.node, focusNodeInfo.offset)) {
                        jumpOverRangeMarkerEndNode(focusNodeInfo);
                    }

                // in Mozilla based browsers a cell selection can be created inside tables
                } else if (extend && _.browser.Firefox && !cellRangeSelected &&
                        Position.isPositionInTable(rootNode, getFocusPosition()) &&
                        Position.isLastPositionInTableCell(rootNode, getFocusPosition()) &&
                        Position.isSameTableLevel(rootNode, self.getAnchorPosition(), getFocusPosition())) {
                    switchToCellSelectionHorz();
                    return true;

                // block cursor traversal, if this is the last position inside a text frame
                } else if (focusNodeInfo.offset === focusTextNode.nodeValue.length && DOM.isLastTextSpanInTextframe(focusTextNode.parentNode)) {
                    if (extend) {
                        self.restoreBrowserSelection(); // this call is asynchronous -> avoid uncontrolled horizontal move
                    } else {
                        self.setTextSelection(self.getEndPosition());
                    }
                    return true;
                // try to find the first text position in the next paragraph
                } else if ((node = findFirstInlineNodeInNextParagraph(focusTextNode.parentNode))) {
                    jumpBeforeInlineNode(node);
                // cursor right in Chrome at the last position inside the root node (simply stay at current position)
                } else if (_.browser.Chrome && (focusNodeInfo.offset === focusTextNode.nodeValue.length) && !Utils.isLastDescendant(rootNode, focusTextNode.parentNode)) {
                    self.restoreBrowserSelection();  // avoid uncontrolled vertical browser scrolling (40052)
                    app.getView().recalculateDocumentMargin();
                    return true;
                }
            }

            // update browser selection if focusNodeInfo still valid
            if (focusNodeInfo.node) {
                applyBrowserSelection({ active: new DOM.Range(anchorNodeInfo, focusNodeInfo), ranges: [] });
            }

            return true;
        }

        /**
         * Converts the passed logical text position to a DOM point pointing to
         * a DOM text node as used by the internal browser selection.
         *
         * @param {Number[]} position
         *  The logical position of the target node. Must be the position of a
         *  paragraph child node, either a text span, a text component (fields,
         *  tabs), or a drawing node.
         *
         * @param {Object} [point]
         *  Performance: The point containing node and offset as calculated
         *  from Position.getDOMPosition. Saving the object, that was calculated
         *  before, saves the call of Position.getDOMPosition.
         *
         * @returns {DOM.Point|Null}
         *  The DOM.Point object representing the passed logical position, or
         *  null, if the passed position is invalid.
         */
        function getPointForTextPosition(position, point) {

            var nodeInfo = point ? point : Position.getDOMPosition(rootNode, position, true),
                node = nodeInfo ? nodeInfo.node : null;

            if (point && node && node.nodeType === 3) {
                node = node.parentNode;  // saved node could be text node instead of text span
            }

            // check that the position selects a paragraph child node
            if (!node || !DOM.isParagraphNode(node.parentNode)) {
                // this might be a drawing inside a drawing group
                if (node && node.parentNode && DrawingFrame.isGroupContentNode(node.parentNode)) {
                    return new DOM.Point(node, 0);
                } else if (node && node.parentNode && DOM.isDrawingLayerNode(node.parentNode)) {
                    return new DOM.Point(node, 0);
                } else {
                    return null;
                }
            }

            if (DOM.isTextSpan(node)) {
                return new DOM.Point(node.firstChild, nodeInfo.offset);
            }

            node = Utils.findPreviousSiblingNode(node, function () { return DOM.isTextSpan(this); });
            if (DOM.isTextSpan(node)) {
                return new DOM.Point(node.firstChild, node.firstChild.nodeValue.length);
            }

            return null;
        }

        /**
         * Jumping with a specified point in backward direction over range marker start node(s).
         */
        function jumpOverRangeMarkerStartNode(point) {

            // helper function, that detects preceding range marker start nodes or complex field nodes
            function continueIteration(node) {
                return DOM.isRangeMarkerStartNode(node) || (node.previousSibling && DOM.isEmptySpan(node) && DOM.isRangeMarkerStartNode(node.previousSibling));
            }

            // skipping to the node before the range marker start node (this must be guaranteed by the caller)
            point.node = point.node.parentNode.previousSibling.previousSibling;

            // skipping also over following range marker end node or comment place holder nodes
            while (continueIteration(point.node)) {
                point.node = point.node.previousSibling;
            }

            // this must be a text span node
            point.node = point.node.firstChild;
            point.offset = point.node.length;

        }

        /**
         * Jumping with a specified point in forward direction over range marker end node(s).
         */
        function jumpOverRangeMarkerEndNode(point) {

            // helper function, that detects following range marker nodes or comment place holder nodes
            function continueIteration(node) {
                return DOM.isRangeMarkerEndNode(node) ||
                       DOM.isCommentPlaceHolderNode(node) ||
                       (node.nextSibling && DOM.isEmptySpan(node) && (DOM.isRangeMarkerEndNode(node.nextSibling) || DOM.isCommentPlaceHolderNode(node.nextSibling)));
            }

            // skipping to the node behind the range marker end node (this must be guaranteed by the caller)
            point.node = point.node.parentNode.nextSibling.nextSibling;

            // skipping also over following range marker end node or comment place holder nodes
            while (continueIteration(point.node)) {
                point.node = point.node.nextSibling;
            }

            // this must be a text span node
            point.node = point.node.firstChild;
            point.offset = 0;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether this selection contains a valid start and end
         * position.
         */
        this.isValid = function () {
            return (startPosition) && (endPosition) && (startPosition.length > 0) && (endPosition.length > 0);
        };

        /**
         * Returns the current logical start position. The start position is
         * always located before the end position (or, in case of a text
         * cursor, is equal to the end position), regardless of the direction
         * of the selection. To receive positions dependent on the direction,
         * see the methods Selection.getAnchorPosition() and
         * Selection.getFocusPosition().
         *
         * @returns {Number[]}
         *  The logical start position of this selection, as cloned array that
         *  can be changed by the caller.
         */
        this.getStartPosition = function () {
            return _.clone(startPosition);
        };

        /**
         * Returns the current logical end position. The end position is always
         * located behind the start position (or, in case of a text cursor,
         * is equal to the start position), regardless of the direction of the
         * selection. To receive positions dependent on the direction, see the
         * methods Selection.getAnchorPosition() and
         * Selection.getFocusPosition().
         *
         * @returns {Number[]}
         *  The logical end position of this selection, as cloned array that
         *  can be changed by the caller.
         */
        this.getEndPosition = function () {
            return _.clone(endPosition);
        };

        /**
         * Returns the current logical anchor position. The anchor position is
         * the position where the selection of a text range (by mouse or cursor
         * keys) has been started. The anchor position will be located after
         * the focus position if selecting backwards. To receive positions
         * independent from the direction, see the methods
         * Selection.getStartPosition() and Selection.getEndPosition().
         *
         * @returns {Number[]}
         *  The logical anchor position of this selection, as cloned array that
         *  can be changed by the caller.
         */
        this.getAnchorPosition = function () {
            return _.clone(getAnchorPosition());
        };

        /**
         * Returns the current logical focus position. The focus position is
         * the position that changes while modifying the selection of a range
         * (by mouse or cursor keys). The focus position will be located before
         * the anchor position if selecting backwards. To receive positions
         * independent from the direction, see the methods
         * Selection.getStartPosition() and Selection.getEndPosition().
         *
         * @returns {Number[]}
         *  The logical focus position of this selection, as cloned array that
         *  can be changed by the caller.
         */
        this.getFocusPosition = function () {
            return _.clone(getFocusPosition());
        };

        /**
         * Returns whether this selection represents a simple text cursor.
         *
         * @returns {Boolean}
         *  True, if this selection represents a simple text cursor.
         */
        this.isTextCursor = function () {
            return !cellRangeSelected && _.isEqual(startPosition, endPosition);
        };

        /**
         * Returns whether this selection represents a range that covers some
         * document contents. The result is the exact opposite of the method
         * Selection.isTextCursor().
         *
         * @returns {Boolean}
         *  True, if this selection represents a range in the document.
         */
        this.hasRange = function () {
            return !this.isTextCursor();
        };

        /**
         * Returns whether this selection has been created while selecting the
         * document contents backwards (by cursor keys or by mouse).
         *
         * @returns {Boolean}
         *  True, if the selection has been created backwards.
         */
        this.isBackwards = function () {
            return backwards;
        };

        /**
         * Returns the type of this selection as string.
         *
         * @returns {String}
         *  Returns 'text' for a text range or text cursor, or 'cell' for a
         *  rectangular cell range in a table, or 'drawing' for a drawing
         *  selection.
         */
        this.getSelectionType = function () {
            return cellRangeSelected ? 'cell' : (selectedDrawing.length > 0 || (self.isMultiSelectionSupported() && self.isMultiSelection())) ? 'drawing' : 'text';
        };

        /**
         * Returns whether the current selection is a selection that
         * additionally contains a selected text frame. This typically
         * happens for a text selection inside a text frame drawing.
         *
         * @returns {Boolean}
         *  Whether the current selection additionally contains a selection
         *  of a text frame.
         */
        this.isAdditionalTextframeSelection = function () {
            return selectedTextFrameDrawing.length > 0;
        };

        /**
         * Returns whether the current selection is a pure drawing selection
         * or is at least a selected text frame in a text selection.
         *
         * @returns {Boolean}
         *  Whether the current selection is a pure drawing selection
         *  or is at least a selected text frame in a text selection.
         */
        this.isAnyDrawingSelection = function () {
            return self.getSelectionType() === 'drawing' || self.isAdditionalTextframeSelection();
        };

        /**
         * Returns whether the current selection is a selection of a drawing group.
         *
         * @returns {Boolean}
         *  Whether the current selection is a selection of a drawing group.
         */
        this.isDrawingGroupSelection = function () {
            return self.getSelectionType() === 'drawing' && DrawingFrame.isGroupDrawingFrame(selectedDrawing);
        };

        /**
         * Returns whether the currently selected drawing is of type 'group' or if the drawing, that is
         * addtionally selected to the selected text, is a drawing of type 'group'.
         *
         * @returns {Boolean}
         *  Whether the selection drawing or the additionally selected drawing is of type 'group'.
         */
        this.isAnyDrawingGroupSelection = function () {
            return self.isDrawingGroupSelection() || (self.isAdditionalTextframeSelection() && DrawingFrame.isGroupDrawingFrame(selectedTextFrameDrawing));
        };

        /**
         * Returns whether the current selection is a selection of a drawing group and if this
         * group contains at least one text frame.
         *
         * @returns {Boolean}
         *  Whether the current selection is a selection of a drawing group and if this
         *  group contains at least one text frame.
         */
        this.isAtLeastOneTextFrameInGroupSelection = function () {
            return self.isDrawingGroupSelection() && selectedDrawing.find(DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR).length > 0;
        };

        /**
         * Returns a selected drawing or an additional text frame drawing.
         * Or null, if this is neither a drawing selection nor an additional
         * text frame selection.
         *
         * @returns {jQuery|Null}
         *  A jQuery object containing the currently selected drawing or
         *  additional text frame drawing. Or null, if none of them is
         *  selected.
         */
        this.getAnyDrawingSelection = function () {

            var // the currently selected drawing
                selDrawing = null;

            if (selectedDrawing.length > 0) {
                selDrawing = selectedDrawing;
            } else if (selectedTextFrameDrawing.length > 0) {
                selDrawing = selectedTextFrameDrawing;
            }

            return selDrawing;
        };

        /**
         * Whether the current selection is a selection of a drawing.
         *
         * @returns {Boolean}
         *  Whether the current selection is a selection of a drawing.
         */
        this.isDrawingSelection = function () {
            return self.getSelectionType() === 'drawing';
        };

        /**
         * Whether the current selection is a selection of a text frame.
         *
         * @returns {Boolean}
         *  Whether the current selection is a selection of a text frame. This does NOT include the additional text frame, that
         *  is selected, if the cursor is positioned inside a text frame.
         */
        this.isTextFrameSelection = function () {
            return self.isDrawingSelection() && DrawingFrame.isTextFrameShapeDrawingFrame(self.getSelectedDrawing());
        };

        /**
         * Returns whether the current selection is a text selection inside a text frame and the additionally
         * selected drawing is a text frame (and not a group), or if a text frame is directly selected as drawing.
         *
         * @param {Object} [options]
         *  An object containing some additional properties:
         *  @param {Boolean} [options.allDrawingTypesAllowed=false]
         *      Whether a selected drawing must be a text frame. Alternatively it is possible, that a 'group' drawing
         *      is selected additionally to a text selection inside a text frame. If 'allDrawingTypesAllowed' is set
         *      to true, this function returns also true, if a 'group' drawing node is selected additionally to a
         *      text selection inside a text frame.
         *
         * @returns {Boolean}
         *  Whether the current selection is a pure text frame drawing selection
         *  or is at least a selected text frame in a text selection.
         */
        this.isAnyTextFrameSelection = function (options) {

            var // whether the selected drawing must be a text frame
                allDrawingTypesAllowed = Utils.getBooleanOption(options, 'allDrawingTypesAllowed', false);

            return (self.isAdditionalTextframeSelection() && (allDrawingTypesAllowed || DrawingFrame.isTextFrameShapeDrawingFrame(self.getSelectedTextFrameDrawing()))) ||
                   (self.getSelectionType() === 'drawing' && DrawingFrame.isTextFrameShapeDrawingFrame(self.getSelectedDrawing()));
        };

        /**
         * Whether the current selection is a selection of a text frame (a drawing selection, not an additional text frame)
         * or if in a multi drawing seletion at least one drawing is a text frame. In any case a text frame drawing is
         * completely selected as drawing. Additionally this might be a group selection, in which the selected group
         * contains at least one text frame.
         *
         * @returns {Boolean}
         *  Whether the current selection is a selection of a text frame drawing or if in a multi drawing seletion at least
         *  one drawing is a text frame.
         */
        this.isAnyTextFrameDrawingSelection = function () {
            return self.isTextFrameSelection() || self.isAtLeastOneTextFrameInMultiSelection() || self.isAtLeastOneTextFrameInGroupSelection();
        };

        /**
         * Returns whether the selection represents a simple text cursor that is
         * located in a top level node (can be used for slide in presentation app).
         *
         * @param {Object} [selection]
         *  An optional object containing the properties 'start' and 'end'. If this
         *  is specified, it is evaluated. If this is not specified, the current
         *  selection is evaluated.
         *
         * @returns {Boolean}
         *  True, if this selection represents a simple text cursor inside a top
         *  level node.
         */
        this.isTopLevelTextCursor = function (selection) {
            return selection ? (_.isEqual(selection.start, selection.end) && selection.start.length === 2) : (startPosition.length === 2 && self.isTextCursor());
        };

        /**
         * Returns whether the current selected drawing is supported on not.
         * If the selected drawing is a group of drawings, iterate over all
         * children. If at least one of them is not supported, return false.
         *
         * @returns {Boolean}
         *  Whether the selection contains only supported drawings.
         */
        this.areOnlySupportedDrawingsSelected = function () {
            var selectedDrawing = null;

            if (self.getSelectionType() === 'drawing') {
                selectedDrawing = self.getSelectedDrawing();
            } else if (self.isAdditionalTextframeSelection()) {
                selectedDrawing = self.getSelectedTextFrameDrawing();
            }

            if (_.isNull(selectedDrawing)) { return false; }

            if (DrawingFrame.isGroupDrawingFrame(selectedDrawing)) {
                var reVal = true;
                _.each(DrawingFrame.getAllGroupDrawingChildren(selectedDrawing), function (drawing) {
                    if (DrawingFrame.isUnsupportedDrawing($(drawing))) {
                        reVal = false;
                    }
                });
                return reVal;
            } else {
                return !DrawingFrame.isUnsupportedDrawing(selectedDrawing);
            }
        };

        /**
         * Returns whether the current selection is a drawing selection and
         * the drawing is located inside a text frame.
         *
         * @returns {Boolean}
         *  Whether the current selection is a drawing selection and the drawing
         *  is located inside a text frame.
         */
        this.isDrawingInTextFrameSelection = function () {
            return self.getSelectionType() === 'drawing' && selectedDrawing.length > 0 && selectedDrawing.closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0;
        };

        /**
         * Returns whether the current selection is a drawing selection and
         * the drawing is located inside an ODF text frame.
         *
         * @returns {Boolean}
         *  Whether the current selection is a drawing selection and the drawing
         *  is located inside a text frame.
         */
        this.isDrawingInODFTextFrameSelection = function () {

            var // the text frame node inside the drawing node
                textFrameNode = null;

            if (self.getSelectionType() === 'drawing' && selectedDrawing.length > 0) {

                textFrameNode = selectedDrawing.closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR);

                if (textFrameNode && textFrameNode.length > 0 && DrawingFrame.isFullOdfTextframeNode(textFrameNode.closest(DrawingFrame.NODE_SELECTOR))) {
                    return true;
                }
            }

            return false;
        };

        /**
         * Returns whether the current selection is selection inside a paragraph.
         * So currently the selection type must be 'text' or 'drawing'.
         *
         * @returns {Boolean}
         *  Whether the current selection type is 'text' or 'drawing'.
         */
        this.isTextLevelSelection = function () {
            return !cellRangeSelected;
        };

        /**
         * Resetting an existing selection. This can be used for example during loading the
         * document, where an existing selection must be resetted after the operations were
         * applied. So it is possible to determine, if the user already modified the selection.
         */
        this.resetSelection = function () {
            startPosition = [];
            endPosition = [];
            cellRangeSelected = false;
            tableSelected = false;
        };

        /**
         * Checking, if the current values for startPosition and endPosition
         * represent a valid text position. It can happen, that these values
         * point to a non existing position (bug 28520).
         * Returns whether the current selection is a valid text selection.
         *
         * @returns {Boolean}
         *   Whether the current selection is a valid text selection.
         */
        this.isValidTextSelection = function () {

            var startPos = _.clone(startPosition),
                textPos = startPos.pop(),
                paragraph =  Position.getParagraphElement(rootNode, startPos),
                endPos = null;

            if ((paragraph === null) || (Position.getParagraphNodeLength(paragraph) < textPos)) {
                return false;
            }

            if (!_.isEqual(startPosition, endPosition)) {
                endPos = _.clone(endPosition);
                textPos = endPos.pop();
                paragraph =  Position.getParagraphElement(rootNode, endPos);

                if ((paragraph === null) || (Position.getParagraphNodeLength(paragraph) < textPos)) {
                    return false;
                }
            }

            return true;
        };

        /**
         * Checking, whether the current selection describes an inline component
         * node like a tab, field, drawing, ... and nothing else.
         *
         * @returns {Boolean}
         *   Whether the current selection describes an inline component.
         */
        this.isInlineComponentSelection = function () {

            var // the node element at the start position
                startNodeInfo = null;

            if (startPosition && endPosition && _.isEqual(_.initial(startPosition), _.initial(endPosition)) && (_.last(startPosition) + 1 === _.last(endPosition))) {
                startNodeInfo = Position.getDOMPosition(rootNode, startPosition, true);
                if (startNodeInfo && startNodeInfo.node && DOM.isInlineComponentNode(startNodeInfo.node)) {
                    return true;
                }
            }

            return false;
        };

        /**
         * Checking, whether the current selection describes a drawing component
         * and nothing else.
         *
         * @returns {Boolean}
         *   Whether the current selection describes a drawing component and
         *   nothing else.
         */
        this.isDrawingFrameSelection = function () {

            var // the node element at the start position
                startNodeInfo = null;

            if (startPosition && endPosition && _.isEqual(_.initial(startPosition), _.initial(endPosition)) && (_.last(startPosition) + 1 === _.last(endPosition))) {
                startNodeInfo = Position.getDOMPosition(rootNode, startPosition, true);
                if (startNodeInfo && startNodeInfo.node && DOM.isDrawingFrame(startNodeInfo.node)) {
                    return true;
                }
            }

            return false;
        };

        /**
         * Checks, whether the given node is (one of) the selected drawing(s)
         *
         * @param {HTMLElement|jQuery} drawingNode
         *
         * @returns {Boolean}
         *   Whether the node is selected or not
         */
        this.isDrawingSelected = function (drawingNode) {
            return _.contains(selectedDrawing, Utils.getDomNode(drawingNode));
        };

        /**
         * Returns whether the start and end position of this selection are
         * located in the same parent component (all array elements but the
         * last are equal).
         *
         * @param {Number} [parentLevel=1]
         *  The number of parent levels. If omitted, the direct parents of the
         *  start and end position will be checked (only the last element of
         *  the position array will be ignored). Otherwise, the specified
         *  number of trailing array elements will be ignored (for example, a
         *  value of 2 checks the grand parents).
         *
         * @returns {Boolean}
         *  Whether the start and end position are located in the same parent
         *  component.
         */
        this.hasSameParentComponent = function (parentLevel) {
            return Position.hasSameParentComponent(startPosition, endPosition, parentLevel);
        };

        /**
         * Returns whether this selection covers exactly one component.
         *
         * @returns {Boolean}
         *  Returns whether the selection is covering a single component. The
         *  start and end position must refer to the same parent component, and
         *  the last array element of the end position must be the last array
         *  element of the start position increased by the value 1.
         */
        this.isSingleComponentSelection = function () {
            return this.hasSameParentComponent() && (_.last(startPosition) === _.last(endPosition) - 1);
        };

        /**
         * Returns the logical position of the closest common component
         * containing all nodes covered by this selection (the leading array
         * elements that are equal in the start and end position arrays).
         *
         * @returns {Number[]}
         *  The logical position of the closest common component containing
         *  this selection. May be the empty array if the positions already
         *  differ in their first element.
         */
        this.getClosestCommonPosition = function () {

            var index = 0, length = Math.min(startPosition.length, endPosition.length);

            while ((index < length) && (startPosition[index] === endPosition[index])) {
                index += 1;
            }

            return startPosition.slice(0, index);
        };

        /**
         * Returns the closest paragraph that contains all nodes of this
         * selection completely.
         *
         * @returns {HTMLElement|Null}
         *  The closest paragraph containing this selection; or null, if the
         *  selection is not contained in a single paragraph.
         */
        this.getEnclosingParagraph = function () {

            var // position of closest common parent component containing the selection
                commonPosition = this.getClosestCommonPosition();

            // the closest paragraph containing the common parent component
            return (commonPosition.length > 0) ? Position.getCurrentParagraph(rootNode, commonPosition) : null;
        };

        /**
         * Returns the closest table that contains all nodes of this selection
         * completely.
         *
         * @returns {HTMLTableElement|Null}
         *  The closest table containing this selection; or null, if the
         *  selection is not contained in a single table.
         */
        this.getEnclosingTable = function () {

            var // position of closest common parent component containing the selection
                commonPosition = this.getClosestCommonPosition();

            // the closest table containing the common parent component
            return (commonPosition.length > 0) ? Position.getCurrentTable(rootNode, commonPosition) : null;
        };

        /**
         * Provides the text contents from the selection.
         *
         * @returns {String|Null}
         *  The text of the current selection or null if no text is available.
         */
        this.getSelectedText = function () {
            var text = null;

            this.iterateNodes(function (node, pos, start, length) {
                if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                    var nodeText = $(node).text();
                    if (nodeText) {
                        text = text || '';
                        text = text.concat(nodeText.slice(start, start + length));
                    }
                }
            });

            return text;
        };

        /**
         * Returns an object describing the table cell range that is currently
         * selected.
         *
         * @returns {Object|Null}
         *  If this selection is contained completely inside a table, returns
         *  an object containing the following attributes:
         *  - {HTMLTableElement} tableNode: the table element containing the
         *      selection,
         *  - {Number[]} tablePosition: the logical position of the table,
         *  - {Number[2]} firstCellPosition: the logical position of the first
         *      cell, relative to the table (contains exactly two elements:
         *      row, column),
         *  - {Number[2]} lastCellPosition: the logical position of the last
         *      cell, relative to the table (contains exactly two elements:
         *      row, column),
         *  - {Number} width: the number of columns covered by the cell range,
         *  - {Number} height: the number of rows covered by the cell range.
         *  Otherwise, this method returns null.
         */
        this.getSelectedCellRange = function () {

            var // the result object containing all info about the cell range
                result = { tableNode: this.getEnclosingTable() };

            if (!result.tableNode) {
                return null;
            }

            // logical position of the table
            result.tablePosition = Position.getOxoPosition(rootNode, result.tableNode, 0);

            // convert selection positions to cell positions relative to table
            if ((startPosition.length < result.tablePosition.length + 2) || (endPosition.length < result.tablePosition.length + 2)) {
                Utils.error('Selection.getSelectedCellRange(): invalid start or end position');
                return null;
            }
            result.firstCellPosition = startPosition.slice(result.tablePosition.length, result.tablePosition.length + 2);
            result.lastCellPosition = endPosition.slice(result.tablePosition.length, result.tablePosition.length + 2);

            // width and height of the range for convenience
            result.width = result.lastCellPosition[1] - result.firstCellPosition[1] + 1;
            result.height = result.lastCellPosition[0] - result.firstCellPosition[0] + 1;

            return result;
        };

        /**
         * Returns the anchor cell range of a cell selection.
         *
         * @returns {DOM.Range}
         *  A DOM range describing the anchor cell of a cell selection,
         *  or null, if not set.
         */
        this.getAnchorCellRange = function () {
            return anchorCellRange;
        };

        /**
         * Returns the anchor cell range of a cell selection.
         *
         * @param {DOM.Range} range
         *  The DOM range, that describes the position of the anchor
         *  cell in a cell selection.
         */
        this.setAnchorCellRange = function (range) {
            anchorCellRange = range;
        };

        /**
         * Returns the drawing node currently selected.
         *
         * @returns {jQuery}
         *  A jQuery collection containing the currently selected drawing, if
         *  existing; otherwise an empty jQuery collection.
         */
        this.getSelectedDrawing = function () {
            return selectedDrawing;
        };

        /**
         * Returns the type of the drawing node currently selected.
         *
         * @returns {String}
         *  The type of the drawing object currently selected ('image',
         *  'chart', 'shape', etc.); or the empty string, if no drawing object
         *  is selected.
         */
        this.getSelectedDrawingType = function () {
            return (selectedDrawing.length > 0) ? DrawingFrame.getDrawingType(selectedDrawing) : '';
        };

        /**
         * Returns the text frame drawing node, that is currently selected.
         *
         * @returns {jQuery}
         *  A jQuery collection containing the currently selected text frame
         *  drawing, if existing; otherwise an empty jQuery collection.
         */
        this.getSelectedTextFrameDrawing = function () {
            return selectedTextFrameDrawing;
        };

        /**
         * Returns the drawing node currently selected. In case of a text
         * selection inside a drawing object with a text frame, returns the
         * enclosing drawing node.
         *
         * @returns {jQuery}
         *  A jQuery collection containing the closest drawing node currently
         *  selected, if existing; otherwise an empty jQuery collection.
         */
        this.getClosestSelectedDrawing = function () {
            return (selectedTextFrameDrawing.length > 0) ? selectedTextFrameDrawing : selectedDrawing;
        };

        /**
         * Returns the text frame drawing node, that is currently selected. This can
         * be an additionally selected text frame node, if this is a text selection
         * inside a text frame, or a directly selected text frame node, if this is
         * a selection of type 'drawing'.
         *
         * @param {Object} [options]
         *  An object containing some additional properties:
         *  @param {Boolean} [options.forceTextFrame=false]
         *      Whether the returned  drawing node must be a text frame. Alternatively
         *      it is possible, that a 'group' drawing is selected additionally to a
         *      text selection inside a text frame. If 'forceTextFrame' is set to true,
         *      not this selected 'group' drawing node is returned, but the not selected
         *      text frame that contains the current text selection.
         *
         * @returns {jQuery}
         *  A jQuery collection containing the currently selected text frame
         *  drawing, if existing; otherwise an empty jQuery collection.
         */
        this.getAnyTextFrameDrawing = function (options) {

            var // whether the selected drawing must be a text frame
                forceTextFrame = Utils.getBooleanOption(options, 'forceTextFrame', false),
                // the text frame, that will be returned
                textFrame = $();

            if (self.isAdditionalTextframeSelection()) {
                if (DrawingFrame.isTextFrameShapeDrawingFrame(self.getSelectedTextFrameDrawing())) {
                    textFrame = self.getSelectedTextFrameDrawing();
                } else {
                    if (forceTextFrame) {
                        textFrame = Position.getClosestDrawingTextframe(rootNode, self.getStartPosition());  // finding the text frame, for example inside a group
                        textFrame = textFrame || $();
                    } else {
                        textFrame = self.getSelectedTextFrameDrawing();  // returning a group is also allowed
                    }
                }
            } else if ((self.getSelectionType() === 'drawing' && DrawingFrame.isTextFrameShapeDrawingFrame(self.getSelectedDrawing()))) {
                textFrame = self.getSelectedDrawing();
            }

            return textFrame;
        };

        /**
         * Collecting all drawing nodes in the current selection. If a filter is specified
         * the selection can be reduced to specific drawing nodes.
         *
         * @param {Function} filter
         *  A function that can be used to filter the drawings in the selection.
         *
         * @returns {jQuery[]|[]}
         *  An array that contains all drawing nodes in the selection. Or an empty array
         *  if there are no (filtered) drawings in the selection.
         */
        this.getAllDrawingsInSelection = function (filter) {

            var // the collector for all drawing nodes in the selection
                allDrawings = [];

            self.iterateDrawings(function (oneDrawing) {
                if (!filter || (filter && filter(oneDrawing))) {
                    allDrawings.push($(oneDrawing));
                }
            }, this);

            return allDrawings;
        };

        /**
         * Returns the type of the drawing node currently selected.
         *
         * @returns {String}
         *  The type of the drawing object currently selected ('image',
         *  'chart', 'shape', etc.); or the empty string, if no drawing object
         *  is selected.
         */
        this.getClosestSelectedDrawingType = function () {
            var drawingFrame = this.getClosestSelectedDrawing();
            return (drawingFrame.length > 0) ? DrawingFrame.getDrawingType(drawingFrame) : '';
        };

        /**
         * Returns the DOM clipboard node used to store a copy of the contents
         * of a selected drawing frame for copy&paste.
         *
         * @returns {jQuery}
         *  The clipboard node, as jQuery object.
         */
        this.getClipboardNode = function () {
            return clipboardNode;
        };

        /**
         * Setting the focus into the clipboard node. This is done, if one or more drawings
         * are selected or if this is a 'slide selection'.
         *
         * @param {Object} [options]
         *  An object containing some additional properties:
         *  @param {Boolean} [options.specialTouchHandling=false]
         *      Whether the slide pane node shall be used instead of the focus node. This
         *      is important to avoid, that the virtual keyboard appears, if the user makes
         *      a 'slide selection'. On touch devices the virtual keyboard shall not appear,
         *      if no object is selected on the slide.
         */
        this.setFocusIntoClipboardNode = function (options) {

            var // whether on touch devices an alternative shall be used to avoid keyboard flickering
                specialTouchHandling = Utils.getBooleanOption(options, 'specialTouchHandling', false);

            if (specialTouchHandling && Utils.TOUCHDEVICE && app.getView().getSlidePane() && app.getView().getSlidePane().getSlidePaneContainer()) {

                // on touch devices, the focus can alternatively be set into the slide pane. The keyboard
                // handling is not supported ('tab', 'ctrl-A', ...), so that the focus is not required in
                // the clipboard node. Copy/paste of drawings is also not supported on touch devices, yet.

                app.getView().getSlidePane().getSlidePaneContainer().focus();
            } else {

                if (clipboardNode.contents().length === 0) {
                    clipboardNode.text('\xa0');
                }
                clipboardNode.toggle(true).focus();
            }
        };

        /**
         * Updates the contents of the clipboard node, according to the current
         * selection.
         */
        this.updateClipboardNode = function () {
            // an array of the selected drawing nodes, or only one if multi selection is not supported
            var allSelectedDrawings = null;

            // inserts a clone of the passed image node into the clipboard
            function copyDrawingNode(drawingNode) {

                var // the source URL of the image
                    sourceUrl = null,
                    // the image node inside the drawing
                    imgNode = null;

                if (DOM.isImageNode(drawingNode)) {

                    imgNode = $(drawingNode).find('img');

                    if (imgNode.length > 0) {

                        // insert the image clone into the clipboard node
                        imgNode = imgNode.clone();
                        clipboardNode.append(imgNode);

                        // IE needs attributes for width/height instead of styles
                        imgNode.attr({ width: imgNode.width(), height: imgNode.height() });

                        if (DOM.isDocumentImageNode(imgNode)) {

                            // additional attributes to check for copy&paste inside the same editor instance
                            sourceUrl = DOM.getUrlParamFromImageNode(drawingNode, 'get_filename');
                            imgNode.attr('alt', JSON.stringify({
                                altsrc: sourceUrl,
                                sessionId: DOM.getUrlParamFromImageNode(drawingNode, 'session'),
                                fileId: DOM.getUrlParamFromImageNode(drawingNode, 'id')
                            }));

                            // replace source URL with Base64 data URL, doesn't work correctly on IE
                            if (!_.browser.IE) {
                                imgNode.attr('src', DOM.getBase64FromImageNode(imgNode, Image.getMimeTypeFromImageUri(sourceUrl)));
                            }
                        }
                    }
                }
            }

            // get the selected drawings for multi selection or the single selected drawing otherwise
            if (self.isMultiSelectionSupported() && self.isMultiSelection()) {
                allSelectedDrawings = self.getAllSelectedDrawingNodes() || [];
            } else {
                allSelectedDrawings = (selectedDrawing.length > 0) ? [selectedDrawing] : [];
            }

            // prevent multiple clones of the same drawing frame
            if (clipboardContainsDrawings(allSelectedDrawings)) {
                return;
            }

            // clear clip board node, register drawing nodes at clip board node
            clearClipboardNode();
            clipboardNode.toggle(allSelectedDrawings.length > 0).data('source-nodes', allSelectedDrawings);

            // clone contained image nodes
            _.each(allSelectedDrawings, copyDrawingNode);

            // make sure the clip board node has some content, so that it can get the browser selection
            if (clipboardNode.contents().length === 0) {
                clipboardNode.text('\xa0');
            }
        };

        /**
         * Returns the bounding rectangle of the focus position relative to the
         * entire document page, if this selection is of type 'text'.
         */
        this.getFocusPositionBoundRect = function () {

            var // the boundaries of the cursor
                boundRect = null,
                // the DOM point of the focus position
                focusPoint = null,
                // the DOM Range object
                docRange = null;

            if (this.getSelectionType() === 'text') {

                // Build DOM Range object from calculated focus point instead
                // of the settings in the core browser selection object (the
                // focusNode and focusOffset attributes). Needed because Chrome
                // reports wrong DOM nodes if cursor is located in an empty
                // paragraph (the browser selection object sometimes returns
                // the last text node of the preceding paragraph instead of the
                // empty text node of the current paragraph).
                // Performance: Special handling for insertText operation, where
                // the node was already saved before
                if (isInsertTextOperation && insertTextPoint) {
                    focusPoint = getPointForTextPosition(null, insertTextPoint);
                } else {
                    focusPoint = getPointForTextPosition(getFocusPosition());
                }

                if (focusPoint) {

                    // If the cursor points into an empty text span, calculate its
                    // position directly instead of using the getClientRects()
                    // method of the DOM Range object. This is needed because
                    // Chrome has problems calculating the bounding rectangle of an
                    // empty text node.
                    // Zoomed documents in Firefox needs this correction also, but only
                    // if the zoom factor is not 100.
                    // Allowing this for zoom factor 100 in Firefox leads to task 32397
                    if ((_.isString(focusPoint.node.nodeValue) && focusPoint.node.nodeValue.length === 0) || (_.browser.Firefox && app.getView().getZoomFactor() !== 100)) {
                        boundRect = Utils.getNodePositionInPage(focusPoint.node.parentNode);
                    } else {
                        try {
                            // Build a DOM range object needed to get the actual
                            // position of the focus position between two characters.
                            docRange = window.document.createRange();
                            docRange.setStart(focusPoint.node, focusPoint.offset);
                            docRange.setEnd(focusPoint.node, focusPoint.offset);
                            boundRect = docRange.getClientRects()[0];
                        } catch (e) {
                            Utils.error('Selection.getFocusPositionBoundRect(): failed to initialize focus range');
                        }
                    }
                }
            }

            return boundRect;
        };

        // low-level browser selection ----------------------------------------

        /**
         * Returns an array of DOM ranges representing the current browser
         * selection inside the editor root node.
         *
         * @returns {Object}
         *  An object that contains a property 'active' with a DOM.Range object
         *  containing the current anchor point in its 'start' property, and
         *  the focus point in its 'end' property. The focus point may precede
         *  the anchor point if selecting backwards with mouse or keyboard. The
         *  returned object contains another property 'ranges' which is an
         *  array of DOM.Range objects representing all ranges currently
         *  selected. Start and end points of these ranges are already adjusted
         *  so that each start point precedes the end point.
         */
        this.getBrowserSelection = function () {
            return getBrowserSelection(rootNode);
        };

        /**
         * Returns the text covered by the current browser selection as plain
         * text string.
         *
         * @returns {String}
         *  The text covered by the current browser selection.
         */
        this.getTextFromBrowserSelection = function () {
            return window.getSelection().toString();
        };

        /**
         * Returns the currently selected contents as HTML mark-up.
         *
         * @returns {String}
         *  The selected content, as HTML mark-up string.
         */
        this.getHtmlFromBrowserSelection = function () {

            var // the browser selection
                selection = window.getSelection(),
                // the result container
                resultNode = $('<div>');

            // read all selection ranges
            for (var index = 0; index < selection.rangeCount; index += 1) {
                resultNode.append(selection.getRangeAt(index).cloneContents());
            }
            resultNode.find('tr.pb-row').remove(); //filter rows that contains page breaks - they should not be copied

            return resultNode.html();
        };

        /**
         * Sets the browser selection to the passed DOM ranges in the editor
         * root node.
         *
         * @param {DOM.Range[]|DOM.Range} ranges
         *  The DOM ranges representing the new browser selection. May be an
         *  array of DOM range objects, or a single DOM range object. Must be
         *  contained in the editor root node.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.setBrowserSelection = function (ranges, options) {

            var // save custom implementation of the focus() method
                customFocusMethod = rootNode[0].focus;

            // Prevent popup closing if the document has readonly modus and the editor is typing. Bug 44088
            if (!app.getModel().getEditMode() && !$(document.activeElement).hasClass('page')) {
                return this;
            }
            rootNode[0].focus = originalFocusMethod;
            setBrowserSelection(rootNode, ranges, options);
            rootNode[0].focus = customFocusMethod;
            return this;
        };

        /**
         * Performance: Saving paragraph node, the logical position of the paragraph
         * and the text position during several operations. This values can be reused
         * during cursor setting after this operations or for directly following
         * operations. This reduces Position.getDOMPosition to the final value of
         * the logical position.
         *
         * @param {HTMLElement|jQuery|} paragraphNode
         *  The paragraph element as DOM node or jQuery object.
         *
         * @param {[Number]} pos
         *  The logical position of the paragraph node.
         *
         * @param {Number} offset
         *  The offset inside the paragraph node.
         */
        this.setParagraphCache = function (node, pos, offset) {
            paragraphCache = node ? { node: node, pos: pos, offset: offset } : null;
        };

        /**
         * Performance: Returning the paragraph node, the logical position of the paragraph
         * and the text position of the logical position of specific operations.
         * This values can be reused during cursor setting after the operation or for
         * following operations. This reduces Position.getDOMPosition to the
         * final value of the logical position.
         *
         * @returns {Object}
         *  An object containing the properties 'node', 'pos' and 'offset', that represent
         *  the paragraph node, the logical position of the paragraph and the offset of
         *  for the text level inside the paragraph. The paragraph node is the most inner
         *  paragraph, so that this paragraph together with the offset describes
         *  the complete cursor position and simplifies the usage of Postion.getDOMPosition
         *  to the last value. The logical paragraph position in the property 'pos' can be
         *  used to check, if in another following operation the paragraph node can be
         *  reused.
         */
        this.getParagraphCache = function () {
            return paragraphCache;
        };

        /**
         * Sets the browser selection to the contents of the passed DOM element
         * node.
         *
         * @param {HTMLElement|jQuery} containerNode
         *  The DOM node whose contents will be completely selected. This node
         *  must be focusable, and will be focused after the browser selection
         *  has been set (except if specified otherwise, see options below).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.setBrowserSelectionToContents = function (containerNode, options) {
            containerNode = Utils.getDomNode(containerNode);
            setBrowserSelection(containerNode, DOM.Range.createRange(containerNode, 0, containerNode, containerNode.childNodes.length), options);
            return this;
        };

        /**
         * Performance: Saving point calculated from Position.getDOMPosition for
         * insertText and splitParagraph operations.
         *
         * @param {Object} point
         *  The point calculated from Position.getDOMPosition, containing the
         *  html node element and the offset.
         */
        this.setInsertTextPoint = function (point) {
            insertTextPoint = point;
        };

        /**
         * Performance: Getting point calculated from Position.getDOMPosition for
         * insertText and splitParagraph operations.
         *
         * @returns {Object}
         *  The point calculated from Position.getDOMPosition, containing the
         *  html node element and the offset.
         */
        this.getInsertTextPoint = function () {
            return insertTextPoint;
        };

        /**
         * Performance: Register, whether the current operation is an insertText
         * operation. In this case the function 'getFocusPositionBoundRect' can use
         * the saved DOM in 'insertTextPoint'.
         */
        this.registerInsertText = function (options) {
            isInsertTextOperation = Utils.getBooleanOption(options, 'insertOperation', false);
        };

        // selection manipulation ---------------------------------------------

        /**
         * Restores the browser selection according to the current logical
         * selection represented by this instance.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preserveFocus=false]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.restoreBrowserSelection = function (options) {

            var // the DOM ranges representing the logical selection
                ranges = [],
                // start and end DOM point for text selection
                startPoint = null, endPoint = null,
                // the DOM element currently focused
                focusNode = $(Utils.getActiveElement()),
                // Performance: whether this is an insertText operation
                insertOperation = Utils.getBooleanOption(options, 'insertOperation', false),
                // Performance: whether this is an insertText operation
                splitOperation = Utils.getBooleanOption(options, 'splitOperation', false),
                // Performance: whether this is a 'simpleTextSelection' operation (backspace, delete, return, insertText)
                simpleTextSelection = Utils.getBooleanOption(options, 'simpleTextSelection', false),
                // Performance: whether this is a cursor selection
                isCursor = Utils.getBooleanOption(options, 'isCursor', false),
                // a local helper position
                localEndPosition = null,
                // logical text position for temporary usage
                localStartPos;

            if (app.isInQuit()) {
                return;
            }

            // Do not preserve focus, if it is inside root node or clipboard
            // (focus may switch nodes depending on new selection type).
            // Bug 28195: IE contains table cells that are content-editable on
            // their own and therefore are focused instead of the root node.
            // Another workaround: IE sometimes focuses the hidden root node if
            // drawing objects are deselected although this node is not focusable.
            if ((focusNode.length === 0) || focusNode.is(rootNode) || Utils.containsNode(rootNode, focusNode) || focusNode.is(clipboardNode) || focusNode.is(app.getView().getHiddenRootNode())) {
                options = Utils.extendOptions(options, { preserveFocus: false });
            }

            switch (this.getSelectionType()) {

                // text selection: select text range
                case 'text':
                    // Performance: Simple process for cursor selections, calling Position.getDOMPosition only once
                    if ((insertOperation || splitOperation) && paragraphCache && paragraphCache.node && _.isNumber(paragraphCache.offset)) {
                        startPoint = Position.getDOMPosition(paragraphCache.node, [paragraphCache.offset]);  // Performance: Using saved paragraph node with last oxo position value -> very fast getDOMPosition
                        self.setInsertTextPoint(startPoint); // Performance: Save for later usage
                    } else {
                        startPoint = Position.getDOMPosition(rootNode, startPosition);
                        // Fix for 32002, skipping floated drawings at beginning of document, webkit only, only complete selections (Ctrl-A)
                        if (_.browser.WebKit && !isCursor && _.isEqual(startPosition, getFirstPosition()) && _.isEqual(endPosition, getLastPosition()) &&
                            DOM.isEmptySpan(startPoint.node.parentNode) &&
                            startPoint.node.parentNode.nextSibling &&
                            (DOM.isFloatingDrawingNode(startPoint.node.parentNode.nextSibling) || DOM.isOffsetNode(startPoint.node.parentNode.nextSibling))) {
                            localStartPos = Position.skipFloatedDrawings(rootNode, startPosition); // Skipping floated drawings without modifying the original start position (32002)
                            startPoint = Position.getDOMPosition(rootNode, localStartPos);
                        }

                        // Problem with cursor traversal over drawing place holder
                        if (isCursor &&  startPoint && startPoint.node && DOM.isDrawingLayerNode(startPoint.node.parentNode)) {
                            startPoint.node = DOM.getDrawingPlaceHolderNode(startPoint.node).nextSibling;
                            startPoint.offset = 0;
                        }

                    }
                    if (startPoint && (simpleTextSelection || isCursor)) {
                        ranges.push(new DOM.Range(startPoint, startPoint));
                    } else {
                        if (_.browser.WebKit && selectionExternallyModified && endSelectionExternallyModified) {
                            // special Chrome trick. Otherwise the selection of the final paragraph
                            // in a multi-paragraph selection will not be displayed correctly.
                            localEndPosition = Position.decreaseLastIndex(endPosition);
                        }
                        endPoint = Position.getDOMPosition(rootNode, localEndPosition ? localEndPosition : endPosition);
                        if (startPoint && endPoint) {
                            ranges.push(new DOM.Range(backwards ? endPoint : startPoint, backwards ? startPoint : endPoint));
                        } else {
                            Utils.error('Selection.restoreBrowserSelection(): missing text selection range');
                        }
                    }

                    // enabling the possibility to select a range, that starts or ends with an absolutely positioned drawing (see 40739)
                    // -> switching from absolute drawing to its placeholder. Otherwise the browser cannot draw a visible selection.
                    if (!isCursor) {
                        if (startPoint && startPoint.node && DOM.isDrawingLayerNode(startPoint.node.parentNode) && endPoint && endPoint.node && endPoint.node.nodeType === 3) {
                            startPoint.node = DOM.getDrawingPlaceHolderNode(startPoint.node);
                        } else if (endPoint && endPoint.node && DOM.isDrawingLayerNode(endPoint.node.parentNode) && startPoint && startPoint.node && startPoint.node.nodeType === 3) {
                            endPoint.node = DOM.getDrawingPlaceHolderNode(endPoint.node);
                        }
                    }

                    // Now the startPoint is available and can be used to check, if next to the text selection, an
                    // additional selection of the text frame is required. Checking this inside applyBrowserSelection()
                    // would be less performant, because it would be necessary to use the startPosition array, to check,
                    // if the logical position is located inside a text frame.
                    if (startPoint && startPoint.node) { handleAdditionalTextFrameSelection(startPoint.node); }

                    break;

                // cell selection: iterate all cells
                case 'cell':
                    this.iterateTableCells(function (cell) {
                        ranges.push(DOM.Range.createRangeForNode(cell));
                    });
                    break;

                // drawing selection: select the clipboard node
                case 'drawing':
                    break;

                default:
                    Utils.error('Selection.restoreBrowserSelection(): unknown selection type');
            }

            // Set the new selection ranges. Bug 26283: when a drawing frame is
            // selected, do not focus root node to prevent scroll flickering.
            if (ranges.length > 0) {
                self.setBrowserSelection(ranges, options);
            } else {
                self.setBrowserSelectionToContents(clipboardNode, options);
            }
            return this;
        };

        /**
         * Handling cursor up and down event inside a text frame. The selection must
         * not leave the text frame.
         *
         * @param {jQuery.Event} event
         *  The jQuery browser event object.
         *
         * @returns {Boolean}
         *  Whether the event was handled within this function.
         */
        this.handleVerticalCursorKeyInTextFrame = function (event) {

            var // whether the event was handled inside this function
                handledEvent = false,
                // the text frame node containing the current selection
                textFrameNode = null,
                // the old logical position that needs to be checked and the new logical position
                oldPos = null, newPos = null,
                // the paragraph at the specified position
                paraNode = null;

            // avoiding that a vertical cursor in a text frame leaves the text frame
            if ((event.keyCode === KeyCodes.DOWN_ARROW || event.keyCode === KeyCodes.UP_ARROW) && self.isAdditionalTextframeSelection()) {

                textFrameNode = self.getSelectedTextFrameDrawing();
                oldPos = backwards ? self.getStartPosition() : self.getEndPosition();
                paraNode = Position.getParagraphElement(rootNode, _.initial(oldPos));

                if (event.keyCode === KeyCodes.DOWN_ARROW) {
                    if (paraNode && paraNode === Utils.getDomNode(textFrameNode.find(DOM.CONTENT_NODE_SELECTOR).last())) {
                        newPos = Position.getLastTextPositionInTextFrame(rootNode, textFrameNode);
                    }
                } else {
                    if (paraNode && paraNode === Utils.getDomNode(textFrameNode.find(DOM.CONTENT_NODE_SELECTOR).first())) {
                        newPos = Position.getFirstTextPositionInTextFrame(rootNode, textFrameNode);
                    }
                }

                // setting the new text selection
                if (newPos) {
                    if (event.shiftKey) {
                        if (backwards) {
                            self.setTextSelection(self.getEndPosition(), newPos);
                        } else {
                            self.setTextSelection(self.getStartPosition(), newPos);
                        }
                    } else {
                        self.setTextSelection(newPos);
                    }

                    handledEvent = true;
                }
            }

            return handledEvent;
        };

        /**
         * Processes a browser event that will change the current selection.
         * Supported are 'mousedown' events and 'keydown' events originating
         * from cursor navigation keys and 'selectionchange' events.
         *
         * @param {jQuery.Event} event
         *  The jQuery browser event object.
         *
         * @returns {jQuery.Promise}
         *  The promise of a deferred object that will be resolved after the
         *  browser has processed the passed event, and this selection instance
         *  has updated itself according to the new browser selection.
         */
        this.processBrowserEvent = function (event, options) {

            var // deferred return value
                def = $.Deferred(),
                // whether event is a keydown event
                keyDownEvent = event.type === 'keydown',
                // whether event is a keydown event without modifier keys except SHIFT
                simpleKeyDownEvent = keyDownEvent && !event.ctrlKey && !event.altKey && !event.metaKey,
                // whether event is a Left/Right cursor key
                leftRightCursor = (event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.RIGHT_ARROW),
                // whether event is a Left/Right cursor key
                upDownCursor = (event.keyCode === KeyCodes.UP_ARROW) || (event.keyCode === KeyCodes.DOWN_ARROW),
                // whethter this is a vertical cursor move
                isVerticalCursor = upDownCursor || (event.keyCode === KeyCodes.PAGE_UP) || (event.keyCode === KeyCodes.PAGE_DOWN),
                // original selection, for post-processing
                originalSelection = { start: this.getStartPosition(), end: this.getEndPosition() },
                // whether the document is in read-only mode
                readOnly = Utils.getBooleanOption(options, 'readonly', false);

            // skipping shrinked implicit paragraphs in read-only mode (28563)
            function skipImplicitShrinkedParagraph() {

                // Checking, if the new position is not inside an implicit paragraph with height 0
                var para = Position.getParagraphElement(rootNode, _.initial(startPosition)),
                    eventTriggered = false;

                if (para && DOM.isImplicitParagraphNode(para) && $(para).height() === 0) {
                    // simply skip this paragraph -> triggering this event again
                    $(rootNode).trigger(event);
                    eventTriggered = true;
                }

                return eventTriggered;
            }

            // saving backward cursor movement for later setting of 'backwards' for modified selections
            isBackwardCursor = (event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.UP_ARROW);

            // handle simple left/right cursor keys (with and without SHIFT) manually
            if (simpleKeyDownEvent && leftRightCursor) {
                // do not move selection manually with SHIFT, if there is no support for backward selection
                if (!event.shiftKey || SELECTION_COLLAPSE_EXPAND_SUPPORT) {
                    if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.LEFT_ARROW })) {
                        def.resolve();
                    } else {
                        def.reject();
                    }

                    if (readOnly && skipImplicitShrinkedParagraph()) {
                        def.resolve();
                        return;
                    }

                    event.preventDefault();
                    return def.promise();
                }

            }

            // Setting cursor to end of document for webkit browsers (32007)
            if (_.browser.WebKit && event.ctrlKey && !event.shiftKey && (event.keyCode === 35)) {
                self.setTextSelection(getLastPosition());
                def.resolve();
                return def.promise();
            }

            // browser needs to process pending events before its selection can be queried
            this.executeDelayed(function () {

                var // the current browser selection
                    browserSelection = self.getBrowserSelection(),
                    // a temporarily required start position
                    start,
                    // a temporarily required information about a position containing node and offset
                    nodeInfo,
                    // whether the browser selection shall be restored
                    restoreSelection = true,
                    // a current cell node in which the event occurred
                    currentCell,
                    // a current cell node in which the event occurred
                    currentCellContentNode,
                    // IE: A helper logical position
                    tempPosition = null,
                    // selector string for fetching page breaks and its children on click, keypress
                    helperSelectorInTable = 'td .page-break, .pb-row td, .pb-row, td .header.inactive-selection, td .footer.inactive-selection, td .inner-pb-line',
                    // helper string selector for fetching non-allowing nodes inside paragraph
                    helperSelectorInParagraph = '.page-break, .inner-pb-line, .header, .footer',
                    outOfBoundariesSelector = '.header.inactive-selection, .footer.inactive-selection, .page, .header-wrapper, .footer-wrapper',
                    // an optionally modified browser selection object
                    modifiedBrowserSel = null,
                    // a modified end position of the selection
                    newLastPosition = null;

                // helper function, that checks if selection received from the browser needs to be modified
                function modifyBrowserSelection() {

                    var $modifiedBrowserSelActStartNode = null,
                        $modifiedBrowserSelActEndNode = null,
                        modifiedBrowserSelActStartNode = null,
                        modifiedBrowserSelActEndNode = null,
                        pos = null,
                        index = null,  // holds position of row with page breaks inside ranges array
                        manualPbNode = $(),
                        $node = null,
                        $msHardbreak = null,
                        firstParagraph = null,
                        lastSpan = null;

                    // click on nodes like page break, header/footer, etc. and check if state is active or inactive
                    function selectionHasForbidenNodes(node) {
                        var parents = node.parentsUntil('.page', '.header, .footer');
                        if (parents.length) {
                            return parents.last().hasClass('inactive-selection');
                        } else {
                            return DOM.isPageBreakNode(node) || node.parentsUntil('.page', DOM.PAGE_BREAK_SELECTOR).length > 0;
                        }
                    }

                    // helper functions for cursor positioning in Firefox in tables with page break
                    function findNextPosInTd(node, pos) {
                        if (!node.hasClass('pb-row')) {
                            node = node.parentsUntil('.pagecontent', '.pb-row');
                        }
                        if (node.next().find('td').eq(pos).length > 0) {
                            return node.next().find('td').eq(pos).find('.p').first();
                        } else {
                            return node.next().find('td').last().find('.p').first();
                        }
                    }

                    function findPrevPosInTd(node, pos) {
                        if (!node.hasClass('pb-row')) {
                            node = node.parentsUntil('.pagecontent', '.pb-row');
                        }
                        if (node.prev().find('td').eq(pos).length > 0) {
                            return node.prev().find('td').eq(pos).find('.p').last();
                        } else {
                            return node.prev().find('td').last().find('.p').last();
                        }
                    }

                    // If selection start or selection end contains invalid node (such ase page break, header, footer...), determine next valid node
                    function modifyStartEndSelection($node) {
                        if (DOM.isPageBreakNode($node)) {
                            if (event.keyCode === KeyCodes.UP_ARROW) {
                                if ($node.prev().length > 0) {
                                    return $node.prev();
                                } else if ($node.parent().prev().length > 0) {
                                    return $node.parent().prev();
                                }
                            } else {
                                if ($node.next().length > 0) {
                                    return $node.next();
                                } else if ($node.parent().next().length > 0) {
                                    return $node.parent().next();
                                }
                            }
                        } else if ($node.hasClass('footer') || $node.hasClass('inner-pb-line')) {
                            if (DOM.isPageBreakNode($node.parent())) {
                                if (event.keyCode === KeyCodes.DOWN_ARROW) {
                                    if ($node.parent().next().length > 0) {
                                        return $node.parent().next();
                                    }
                                } else {
                                    if ($node.parent().prev().length > 0) {
                                        return $node.parent().prev();
                                    }
                                }
                            }
                        } else if ($node.hasClass('header')) {
                            if (DOM.isPageBreakNode($node.parent())) {
                                if (event.keyCode === KeyCodes.UP_ARROW) {
                                    if ($node.parent().prev().length > 0) {
                                        return $node.parent().prev();
                                    }
                                } else {
                                    if ($node.parent().next().length > 0) {
                                        return $node.parent().next();
                                    }
                                }
                            }
                        } else if ($node.is('td') && $node.parentsUntil('.pagecontent', DOM.PAGE_BREAK_SELECTOR).length > 0) {
                            if (event.keyCode === KeyCodes.DOWN_ARROW) {
                                if ($node.parentsUntil('.pagecontent', DOM.PAGE_BREAK_SELECTOR).next().length > 0) {
                                    return $node.parentsUntil('.pagecontent', DOM.PAGE_BREAK_SELECTOR).next();
                                }
                            } else {
                                if ($node.parentsUntil('.pagecontent', DOM.PAGE_BREAK_SELECTOR).prev().length > 0) {
                                    return $node.parentsUntil('.pagecontent', DOM.PAGE_BREAK_SELECTOR).prev();
                                }
                            }
                        } else if ($node.parentsUntil('.page', '.header-wrapper').length > 0) {
                            return DOM.getPageContentNode(rootNode).children('.p').first();
                        } else if ($node.parentsUntil('.page', '.footer-wrapper').length > 0) {
                            return DOM.getPageContentNode(rootNode).children('.p').last();
                        }
                        return false;
                    }

                    // Apply modifications on selection, after modifyStartEndSelection function is called
                    function modifyStartEndNodeForSelection($modBrowserSelNode, pos) {

                        var result = modifyStartEndSelection($modBrowserSelNode);

                        if (!result) {
                            // second try with parent node (IE sometimes returns children of header/footer par elements)
                            result = modifyStartEndSelection($modBrowserSelNode.parentsUntil('.page', helperSelectorInParagraph).first());
                        }
                        if (!result) {
                            Utils.warn('Selection: No valid node found to skip over page break!');
                            return false;
                        }
                        if (result.is('table')) { // fallback for IE
                            result = result.find('div.p span').first();
                        }
                        if (result.hasClass('p')) { // fix for IE
                            result = result.children('span').first();
                        }

                        if (pos === 'start') {
                            modifiedBrowserSel.active.start.node = (result)[0];
                            modifiedBrowserSel.active.start.offset = 0;
                        } else {
                            modifiedBrowserSel.active.end.node = (result)[0];
                            modifiedBrowserSel.active.end.offset = 0;
                        }
                    }

                    // start of modifyBrowserSelection

                    // Firefox - table on two pages cursor traversal
                    if (($(browserSelection.active.start.node).is(helperSelectorInTable) || $(browserSelection.active.end.node).is(helperSelectorInTable))) {

                        modifiedBrowserSel = _.clone(browserSelection);
                        modifiedBrowserSelActStartNode = modifiedBrowserSel.active.start.node;
                        modifiedBrowserSelActEndNode = modifiedBrowserSel.active.end.node;
                        pos = originalSelection.start[2];

                        if (event.keyCode === KeyCodes.DOWN_ARROW) {
                            //TODO: down-arrow in a mousedown event???
                            if ($(browserSelection.active.start.node).is(helperSelectorInTable)) {
                                modifiedBrowserSelActStartNode = findNextPosInTd($(browserSelection.active.start.node), pos);
                                while (modifiedBrowserSelActStartNode.is(DOM.PAGE_BREAK_SELECTOR)) {
                                    modifiedBrowserSelActStartNode = modifiedBrowserSelActStartNode.next();
                                }
                                modifiedBrowserSel.active.start.node = (modifiedBrowserSelActStartNode)[0];
                            }
                            if ($(browserSelection.active.end.node).is(helperSelectorInTable)) {
                                modifiedBrowserSelActEndNode = findNextPosInTd($(browserSelection.active.end.node), pos);
                                while (modifiedBrowserSelActEndNode.is(DOM.PAGE_BREAK_SELECTOR)) {
                                    modifiedBrowserSelActEndNode = modifiedBrowserSelActEndNode.next();
                                }
                                modifiedBrowserSel.active.end.node = (modifiedBrowserSelActEndNode)[0];
                            }
                        } else {
                            if ($(browserSelection.active.start.node).is(helperSelectorInTable)) {
                                modifiedBrowserSelActStartNode = findPrevPosInTd($(browserSelection.active.start.node), pos);
                                while (modifiedBrowserSelActStartNode.is(DOM.PAGE_BREAK_SELECTOR)) {
                                    modifiedBrowserSelActStartNode = modifiedBrowserSelActStartNode.prev();
                                }
                                modifiedBrowserSel.active.start.node = (modifiedBrowserSelActStartNode)[0];
                            }
                            if ($(browserSelection.active.end.node).is(helperSelectorInTable)) {
                                modifiedBrowserSelActEndNode = findPrevPosInTd($(browserSelection.active.end.node), pos);
                                while (modifiedBrowserSelActEndNode.is(DOM.PAGE_BREAK_SELECTOR)) {
                                    modifiedBrowserSelActEndNode = modifiedBrowserSelActEndNode.prev();
                                }
                                modifiedBrowserSel.active.end.node = (modifiedBrowserSelActEndNode)[0];
                            }

                            if ($(browserSelection.active.start.node).hasClass('pb-row')) { // we dont know direction (to move to next or previous node), so we will get it from ranges
                                _.each(browserSelection.ranges, function (range, i) {
                                    if ($(range.start.node).is('tr.pb-row')) {
                                        index = i;
                                    }
                                });
                                if (browserSelection.ranges[index + 1]) {
                                    modifiedBrowserSel.active.start.node = browserSelection.ranges[index + 1].start.node;
                                    modifiedBrowserSel.active.start.offset = browserSelection.ranges[index + 1].start.offset;
                                } else {
                                    modifiedBrowserSel.active.start.node = browserSelection.ranges[index - 1].start.node;
                                    modifiedBrowserSel.active.start.offset = browserSelection.ranges[index - 1].start.offset;
                                }
                            }
                            if ($(browserSelection.active.end.node).hasClass('pb-row')) { // we dont know direction (to move to next or previous node), so we will get it from ranges
                                if (_.isUndefined(index)) {
                                    _.each(browserSelection.ranges, function (range, i) {
                                        if ($(range.end.node).is('tr.pb-row')) {
                                            index = i;
                                        }
                                    });
                                }
                                if (browserSelection.ranges[index + 1]) {
                                    modifiedBrowserSel.active.end.node = browserSelection.ranges[index + 1].end.node;
                                    modifiedBrowserSel.active.end.offset = browserSelection.ranges[index + 1].end.offset;
                                } else {
                                    modifiedBrowserSel.active.end.node = browserSelection.ranges[index - 1].end.node;
                                    modifiedBrowserSel.active.end.offset = browserSelection.ranges[index - 1].end.offset;
                                }
                            }
                            if (!_.isUndefined(index)) { // remove range that contains page break row
                                modifiedBrowserSel.ranges.splice(index, 1);
                            }
                        }

                    } else if (selectionHasForbidenNodes($(browserSelection.active.start.node)) || selectionHasForbidenNodes($(browserSelection.active.end.node))) {

                        modifiedBrowserSel = _.clone(browserSelection);

                        $modifiedBrowserSelActStartNode = $(modifiedBrowserSel.active.start.node);
                        $modifiedBrowserSelActEndNode = $(modifiedBrowserSel.active.end.node);

                        if (selectionHasForbidenNodes($modifiedBrowserSelActStartNode)) {
                            modifyStartEndNodeForSelection($modifiedBrowserSelActStartNode, 'start');
                        }

                        if (selectionHasForbidenNodes($modifiedBrowserSelActEndNode)) {
                            modifyStartEndNodeForSelection($modifiedBrowserSelActEndNode, 'end');
                        }

                    } else if ($(event.target).hasClass('pagecontent') && (event.type === 'mousedown' || event.type === 'mouseup')) { // clicked on manual page break page
                        if (event.type === 'mouseup') {
                            if (!_.browser.Chrome) {
                                modifiedBrowserSel = null;
                            }
                        } else {
                            modifiedBrowserSel = _.clone(browserSelection);

                            if ($(browserSelection.active.start.node).hasClass('manual-page-break')) {
                                manualPbNode = $(browserSelection.active.start.node);
                            } else if ($(browserSelection.active.start.node).parentsUntil('.pagecontent', '.manual-page-break').length) {
                                manualPbNode = $(browserSelection.active.start.node).parentsUntil('.pagecontent', '.manual-page-break');
                            } else if (_.browser.Firefox && $(browserSelection.active.start.node).hasClass('page-break')) {
                                manualPbNode = $(browserSelection.active.start.node).prev();
                            }
                            if (manualPbNode.length) {
                                if (event.clientY < app.getModel().getNode().children('.pagecontent').children().last()[0].getBoundingClientRect().top && manualPbNode.length) {
                                    if (_.browser.Firefox) {
                                        $node = manualPbNode; // firefox will get the right node, not the one after
                                    } else {
                                        $node = manualPbNode.prev().prev(); // skip over pagebreak
                                    }
                                } else {
                                    $node = app.getModel().getNode().children('.pagecontent').children().last();
                                }

                                if (!$node.length) {
                                    // setting cursor to first document position, if no node is found
                                    $node = Utils.findDescendantNode(DOM.getPageContentNode(rootNode), DOM.PARAGRAPH_NODE_SELECTOR);
                                }

                                modifiedBrowserSel.active.start.node = Utils.findDescendantNode($node, function () { return DOM.isPortionSpan(this); }, { reverse: true });
                                modifiedBrowserSel.active.end.node = modifiedBrowserSel.active.start.node;
                                modifiedBrowserSel.active.start.offset = modifiedBrowserSel.active.start.node.firstChild.nodeValue.length;
                                modifiedBrowserSel.active.end.offset = modifiedBrowserSel.active.start.offset;

                            } else {
                                modifiedBrowserSel = null;
                            }
                        }
                    } else if ($(event.target).hasClass('manual-page-break') && event.type === 'mouseup') { // clicked on ms word page break

                        $msHardbreak = $(event.target).find('.ms-hardbreak-page');
                        modifiedBrowserSel = _.clone(browserSelection);

                        if ($msHardbreak.length && $msHardbreak.parent().next().next()[0].getBoundingClientRect().top > event.clientY && event.clientY > $msHardbreak[0].getBoundingClientRect().top + $msHardbreak.height()) {
                            // ms word manual pb handling
                            modifiedBrowserSel.active.start.node = $msHardbreak.parent().prev()[0];
                            modifiedBrowserSel.active.end.node = modifiedBrowserSel.active.start.node;
                            modifiedBrowserSel.active.start.offset = _.last(Position.getOxoPosition(rootNode, modifiedBrowserSel.active.start.node, modifiedBrowserSel.active.start.node.firstChild.nodeValue.length));
                            modifiedBrowserSel.active.end.offset = modifiedBrowserSel.active.start.offset;
                        }
                    } else if ($(browserSelection.active.start.node).is(outOfBoundariesSelector) || $(browserSelection.active.end.node).is(outOfBoundariesSelector)) {

                        modifiedBrowserSel = _.clone(browserSelection);

                        if (event.keyCode === KeyCodes.UP_ARROW || event.keyCode === KeyCodes.HOME || event.keyCode === KeyCodes.PAGE_UP) {
                            firstParagraph = Utils.findDescendantNode(DOM.getPageContentNode(rootNode), DOM.PARAGRAPH_NODE_SELECTOR);
                            if ($(browserSelection.active.start.node).is(outOfBoundariesSelector)) {
                                modifiedBrowserSel.active.start.node = firstParagraph;
                                modifiedBrowserSel.active.start.offset = 0;
                            }
                            if ($(browserSelection.active.end.node).is(outOfBoundariesSelector)) {
                                modifiedBrowserSel.active.end.node = firstParagraph;
                                modifiedBrowserSel.active.end.offset = 0;
                            }
                        } else if (event.keyCode === KeyCodes.DOWN_ARROW || event.keyCode === KeyCodes.END || event.keyCode === KeyCodes.PAGE_DOWN) {
                            lastSpan = Utils.findDescendantNode(DOM.getPageContentNode(rootNode), function () { return DOM.isPortionSpan(this); }, { reverse: true });
                            if ($(browserSelection.active.start.node).is(outOfBoundariesSelector)) {
                                modifiedBrowserSel.active.start.node = lastSpan;
                                modifiedBrowserSel.active.start.offset = lastSpan.firstChild.nodeValue.length;
                            }
                            if ($(browserSelection.active.end.node).is(outOfBoundariesSelector)) {
                                modifiedBrowserSel.active.end.node = lastSpan;
                                modifiedBrowserSel.active.end.offset = lastSpan.firstChild.nodeValue.length;
                            }
                            if (_.browser.Chrome && event.keyCode === KeyCodes.DOWN_ARROW) {
                                // in Chrome on last document position and arrow down, prevent document scrolls to begining
                                app.getView().scrollToChildNode(lastSpan);
                            }
                        } else if (event.type === 'mousedown' || event.type === 'mouseup') {
                            if (modifiedBrowserSel.active.start.node === modifiedBrowserSel.active.end.node) {
                                if ($(event.target).parents('.header, header-wrapper').length > 0) {
                                    firstParagraph = Utils.findDescendantNode(DOM.getPageContentNode(rootNode), DOM.PARAGRAPH_NODE_SELECTOR);
                                    modifiedBrowserSel.active.start.node = modifiedBrowserSel.active.end.node = firstParagraph;
                                    modifiedBrowserSel.active.start.offset = modifiedBrowserSel.active.end.offset = 0;
                                } else if ($(event.target).parents('.footer, footer-wrapper').length > 0) {
                                    lastSpan = Utils.findDescendantNode(DOM.getPageContentNode(rootNode), function () { return DOM.isPortionSpan(this); }, { reverse: true });
                                    modifiedBrowserSel.active.start.node = modifiedBrowserSel.active.end.node = lastSpan;
                                    modifiedBrowserSel.active.start.offset = modifiedBrowserSel.active.end.offset = lastSpan.firstChild.nodeValue.length;
                                }
                            } else if ($(browserSelection.active.end.node).is(outOfBoundariesSelector)) {
                                if ($(browserSelection.active.end.node).parents('.footer, footer-wrapper').length > 0) {
                                    lastSpan = Utils.findDescendantNode(DOM.getPageContentNode(rootNode), function () { return DOM.isPortionSpan(this); }, { reverse: true });
                                    modifiedBrowserSel.active.end.node = lastSpan;
                                    modifiedBrowserSel.active.end.offset = lastSpan.firstChild.nodeValue.length;
                                }
                                if ($(event.target).hasClass('cover-overlay')) {
                                    if ($(event.target).parents('.header, header-wrapper').length > 0) {
                                        firstParagraph = Utils.findDescendantNode(DOM.getPageContentNode(rootNode), DOM.PARAGRAPH_NODE_SELECTOR);
                                        modifiedBrowserSel.active.end.node = firstParagraph;
                                        modifiedBrowserSel.active.end.offset  = 0;
                                    } else if ($(event.target).parents('.footer, footer-wrapper').length > 0) {
                                        lastSpan = Utils.findDescendantNode(DOM.getPageContentNode(rootNode), function () { return DOM.isPortionSpan(this); }, { reverse: true });
                                        modifiedBrowserSel.active.end.node = lastSpan;
                                        modifiedBrowserSel.active.end.offset = lastSpan.firstChild.nodeValue.length;
                                    }
                                }
                            } else if ($(browserSelection.active.start.node).is(outOfBoundariesSelector)) {
                                if ($(browserSelection.active.start.node).parents('.header, header-wrapper').length > 0) {
                                    firstParagraph = Utils.findDescendantNode(DOM.getPageContentNode(rootNode), DOM.PARAGRAPH_NODE_SELECTOR);
                                    modifiedBrowserSel.active.start.node = firstParagraph;
                                    modifiedBrowserSel.active.start.offset  = 0;
                                }
                            }
                        }
                        if (_.browser.Firefox) {
                            app.getView().recalculateDocumentMargin();
                        }
                    }
                }

                // helper function that checks, if the vertical scroll position needs to be updated in special scenarios.
                // Returns true, if the process of setting the selection can be terminated, otherwise false.
                function checkVerticalPositions() {

                    if (_.browser.Firefox && self.isTextCursor() && DOM.isTextNodeInPortionSpan(browserSelection.active.start.node) && browserSelection.active.start.node.nodeValue.length === browserSelection.active.start.offset) {
                        // scrolling into view required for Firefox browser (38573)
                        app.getView().recalculateDocumentMargin();
                    } else if (_.browser.WebKit && DOM.isPageNode(rootNode) && !Utils.containsNode(DOM.getPageContentNode(rootNode), browserSelection.active.start.node)) {
                        // never leave the current root node with cursor down in Chrome (40052)
                        self.restoreBrowserSelection();
                        app.getView().recalculateDocumentMargin();
                        return true; // leaving selection setting
                    }

                    return false; // not leaving selection setting
                }

                // helper function that checks, if the browser selection needs to be shifted from the final text
                // position before a range marker end node to the first position behind the range marker node
                // (or even behind the following comment node).
                // also checking if this is the first position inside a text range -> select position before the range
                function checkRangeMarkerEndNodes() {

                    // checking start position of browser selection
                    if (browserSelection.active.start.offset === 0) {
                        if (DOM.isFirstTextPositionBehindRangeStart(browserSelection.active.start.node, browserSelection.active.start.offset)) {
                            jumpOverRangeMarkerStartNode(browserSelection.active.start); // jumping to the left
                        }
                    } else {
                        if (DOM.isLastTextPositionBeforeRangeEnd(browserSelection.active.start.node, browserSelection.active.start.offset)) {
                            jumpOverRangeMarkerEndNode(browserSelection.active.start); // jumping to the right
                        }
                    }

                    // checking end position of browser selection
                    if (browserSelection.active.end.offset === 0) {
                        if (DOM.isFirstTextPositionBehindRangeStart(browserSelection.active.end.node, browserSelection.active.end.offset)) {
                            jumpOverRangeMarkerStartNode(browserSelection.active.end); // jumping to the left
                        }
                    } else {
                        if (DOM.isLastTextPositionBeforeRangeEnd(browserSelection.active.end.node, browserSelection.active.end.offset)) {
                            jumpOverRangeMarkerEndNode(browserSelection.active.end); // jumping to the right
                        }
                    }

                }

                // helper function that checks, in case browser selection fails on page up/down traversals,
                // how to correct position, if possible
                function checkPageUpPageDownTraversal() {
                    var startPos = self.getStartPosition(),
                        originalStartPos = originalSelection.start,
                        compareArrays = Utils.compareNumberArrays(startPos, originalStartPos),
                        firefoxContition = _.browser.Firefox ? (event.keyCode === KeyCodes.PAGE_DOWN && (compareArrays < 0 || _.isEqual(startPos, self.getFirstDocumentPosition())) || (event.keyCode === KeyCodes.PAGE_UP && (compareArrays > 0 || _.isEqual(startPos, self.getFirstDocumentPosition())))) : false,
                        pos,
                        pageLayout,
                        pageNum,
                        numPages,
                        beginingElement,
                        newPos;

                    if (event.keyCode === KeyCodes.PAGE_UP && _.isEqual(originalStartPos, self.getFirstDocumentPosition())) {
                        // do nothing
                    } else if (event.keyCode === KeyCodes.PAGE_DOWN && _.isEqual(originalStartPos, self.getLastDocumentPosition())) {
                        // do nothing
                    } else if (firefoxContition || _.isEqual(startPos, originalStartPos)) {
                        pos = originalSelection.start;
                        pageLayout = app.getModel().getPageLayout();
                        pageNum = pageLayout.getPageNumber(pos);
                        numPages = pageLayout.getNumberOfDocumentPages();
                        if (event.keyCode === KeyCodes.PAGE_DOWN) {
                            if (pageNum === numPages) {
                                self.setTextSelection(self.getLastDocumentPosition());
                            } else {
                                beginingElement = pageLayout.getBeginingElementOnSpecificPage(pageNum + 1);
                                if (beginingElement) {
                                    beginingElement = beginingElement.parent(); // page brea
                                    if (beginingElement.next().length) {
                                        beginingElement = beginingElement.next(); // first next element
                                    } else if (beginingElement.parent().is('td')) {
                                        beginingElement = beginingElement.closest('tr').next().find('span').first();
                                    }
                                    if (beginingElement.hasClass('p') || beginingElement.is('table')) {
                                        beginingElement = beginingElement.find('span').first();
                                    }
                                    newPos = Position.getOxoPosition(rootNode, beginingElement, 0);
                                    self.setTextSelection(newPos);
                                }
                            }
                        } else if (event.keyCode === KeyCodes.PAGE_UP) {
                            if (!pageNum || pageNum < 3) {
                                self.setTextSelection(self.getFirstDocumentPosition());
                            } else {
                                beginingElement = pageLayout.getBeginingElementOnSpecificPage(pageNum - 1);
                                if (beginingElement) {
                                    beginingElement = beginingElement.parent(); // page break
                                    if (beginingElement.next().length) {
                                        beginingElement = beginingElement.next(); // first next element
                                    } else if (beginingElement.parent().is('td')) {
                                        beginingElement = beginingElement.closest('tr').next().find('span').first();
                                    }
                                    if (beginingElement.hasClass('p') || beginingElement.is('table')) {
                                        beginingElement = beginingElement.find('span').first();
                                    }
                                    newPos = Position.getOxoPosition(rootNode, beginingElement, 0);
                                    self.setTextSelection(newPos);
                                }
                            }
                        }
                    }
                }

                // start of delayed anonymous function

                if (browserSelection && browserSelection.active) {

                    // do not modify selection in IE (< 11) when starting to select with mouse
                    if (!((Utils.TOUCHDEVICE || _.browser.IE) && ((event.type === 'mousedown') || (event.type === 'touchstart')))) {

                        // Fix for 28222: Avoid calling restoreBrowserSelection for Ctrl-Shift-CursorLeft in Chrome and Firefox
                        if (event.shiftKey && event.ctrlKey && event.keyCode === KeyCodes.LEFT_ARROW && SELECTION_COLLAPSE_EXPAND_SUPPORT) {
                            restoreSelection = false;
                        }

                        // checking for range end marker nodes -> setting position behind those range end markers
                        if (leftRightCursor || event.type === 'mousedown' || event.type === 'mouseup') { //#38753, #41957
                            checkRangeMarkerEndNodes();
                        }

                        // modifying the browser selection if necessary
                        modifyBrowserSelection();

                        // specific checks for vertical cursor movements
                        if (isVerticalCursor && checkVerticalPositions()) { return; }

                        // calculates the new logical selection
                        applyBrowserSelection(modifiedBrowserSel ? modifiedBrowserSel : browserSelection, { event: event, restoreSelection: restoreSelection });

                        // fixing the position after Ctrl-CursorRight, so that cursor stays at end of document (37298)
                        if (event.keyCode === KeyCodes.RIGHT_ARROW && event.ctrlKey && self.isTextCursor() && _.isEqual(self.getStartPosition(), self.getFirstDocumentPosition())) {
                            self.setTextSelection(self.getLastDocumentPosition());
                        }

                        if ((event.keyCode === KeyCodes.PAGE_DOWN || event.keyCode === KeyCodes.PAGE_UP)) {
                            checkPageUpPageDownTraversal(); // fix for #39003
                        }

                        // Modifying selection, if Ctrl-Shift-Right includes start of following paragraph (40151)
                        if (event.ctrlKey && event.shiftKey && event.keyCode === KeyCodes.RIGHT_ARROW && _.last(endPosition) === 0) {
                            newLastPosition = Position.getLastPositionOfPreviousParagraph(rootNode, _.initial(endPosition));
                            if (newLastPosition) { self.setTextSelection(startPosition, newLastPosition); }
                        }

                        // Only for IE: Expanding selection after a double click or triple click (Fix for 29751)
                        if ((self.getSelectionType() === 'text') && (self.isTextCursor())) {
                            if (event.type === 'dblclick') {
                                self.setTextSelection.apply(self, Position.getWordBoundaries(rootNode, self.getStartPosition(), { addFinalSpaces: true }));
                            } else if (event.type === 'tripleclick') {
                                self.setTextSelection(Position.appendNewIndex(_.initial(self.getStartPosition()), 0), Position.appendNewIndex(_.initial(self.getStartPosition()), Position.getParagraphLength(rootNode, self.getStartPosition())));
                            }
                        }

                        // Fixing the position after pressing the 'End' key for MS IE and Webkit (26863) and Firefox (28309).
                        // The selection has to be reduced by 1 (if it is not the end of the paragraph) -> task 26863 and 28309
                        if ((event.keyCode === KeyCodes.END) && !event.shiftKey) {
                            start = self.getStartPosition();
                            nodeInfo = Position.getDOMPosition(rootNode, start);

                            if ($(nodeInfo.node).text().charCodeAt(nodeInfo.offset - 1) === 32) {
                                if (start[start.length - 1] < Position.getParagraphLength(rootNode, start)) {
                                    start[start.length - 1] -= 1;
                                    self.setTextSelection(start);
                                }
                            } else if ((_.browser.Firefox) && (nodeInfo.offset === 0) && (nodeInfo.node.parentNode && nodeInfo.node.parentNode.previousSibling && $(nodeInfo.node.parentNode.previousSibling).is(DOM.HARDBREAK_NODE_SELECTOR))) {
                                // Fix for 28544
                                start[start.length - 1] -= 1;
                                self.setTextSelection(start);
                            }
                        }

                        // Fixing in MS IE the handling of shift + 'end', that selects until the end of the document, if pressed
                        // in the last line of a paragraph (Fix for 28218).
                        if ((event.keyCode === KeyCodes.END) && event.shiftKey && _.browser.IE) {
                            if (!Position.hasSameParentComponent(startPosition, endPosition)) {
                                self.setTextSelection(startPosition, Position.getLastPositionInParagraph(rootNode, _.clone(startPosition)));
                            }
                        }

                        // adjust selection, if key event did not have any effect
                        if (keyDownEvent && _.isEqual(originalSelection, { start: startPosition, end: endPosition })) {

                            // we need different keys on OSX - CTRL+Left/Right cursor or on OSX ALT+Left/Right cursor: execute simple left/right step
                            if ((_.browser.MacOS) ? (leftRightCursor && !event.ctrlKey && event.altKey && !event.metaKey) : (leftRightCursor && event.ctrlKey && !event.altKey && !event.metaKey)) {
                                // move cursor to left and right by one step
                                if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.LEFT_ARROW })) {

                                    // Fix for Bug 41333: Extending the selection by a whole word on chrome/safari doesn't work in some situations. So after the cursor was moved with alt/ctrl+shift+left/right, extend the selection to the next word.
                                    if (_.browser.Chrome || _.browser.Safari) {
                                        // when the pressed key was arrow left, extend the selection to the left, to the start from the next word
                                        if (event.keyCode === KeyCodes.LEFT_ARROW) {
                                            // set the selection from the position where the selection of the text range has been started, to the first character from the word at the focus position
                                            self.setTextSelection(self.getAnchorPosition(), Position.getWordBoundaries(rootNode, self.getFocusPosition())[0]);

                                        // otherwise extend the selection to the right, to the end from next word
                                        } else {
                                            // set the selection from the start position of the selection to the last character from the word were the selection ends
                                            self.setTextSelection(self.getStartPosition(), Position.getWordBoundaries(rootNode, self.getEndPosition())[1]);
                                        }
                                    }

                                    def.resolve();
                                } else {
                                    def.reject();
                                }
                                return;
                            }

                            // Setting cell selections in Mozilla, if the selection did not change the content is separated into different cells
                            if (_.browser.Firefox && upDownCursor && event.shiftKey && Position.isPositionInTable(rootNode, startPosition) && !cellRangeSelected) {
                                // simple Up/Down cursor: execute simple left/right step
                                if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.UP_ARROW, verticalCellSelection: true })) {
                                    def.resolve();
                                } else {
                                    def.reject();
                                }
                                return;
                            } else if (upDownCursor && !event.ctrlKey && !event.altKey && !event.metaKey) {
                                // simple Up/Down cursor: execute simple left/right step
                                if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.UP_ARROW })) {
                                    def.resolve();
                                } else {
                                    def.reject();
                                }
                                return;
                            }
                            // see #42989
                            if (_.browser.Chrome) {
                                if (event.keyCode === KeyCodes.END) {
                                    self.setTextSelection(Position.getLastPositionInParagraph(rootNode, _.clone(startPosition)));
                                }
                                if (event.keyCode === KeyCodes.HOME) {
                                    self.setTextSelection(Position.getFirstPositionInParagraph(rootNode, _.clone(startPosition)));
                                }
                            }
                        } else {
                            // setting cell selections in Mozilla, if the content is separated into different cells
                            // Typically this is most often handled with 'verticalCellSelection' in the 'if', because
                            // the selection did not change. But sometimes it changes, and this can be recognized by
                            // the following call of moveTextCursor.
                            if (_.browser.Firefox && upDownCursor && event.shiftKey && Position.isPositionInTable(rootNode, startPosition) && !cellRangeSelected) {
                                // are start and end position in the same table cell?
                                if (!_.isEqual(Position.getLastPositionFromPositionByNodeName(rootNode, startPosition, DOM.TABLE_CELLNODE_SELECTOR), Position.getLastPositionFromPositionByNodeName(rootNode, endPosition, DOM.TABLE_CELLNODE_SELECTOR))) {
                                    if (moveTextCursor({ extend: event.shiftKey, backwards: event.keyCode === KeyCodes.UP_ARROW, verticalCellSelection: true })) {
                                        def.resolve();
                                    } else {
                                        def.reject();
                                    }
                                    return;
                                }
                            }

                            // setting selection in IE using shift and cursor-Right and jumping from one paragraph to the following paragraph
                            if (_.browser.IE && event.shiftKey) {
                                if (event.keyCode === KeyCodes.RIGHT_ARROW && Position.isLastPositionInParagraph(rootNode, endPosition)) {
                                    if (moveTextCursor({ extend: true, backwards: false, verticalCellSelection: false })) {
                                        def.resolve();
                                    } else {
                                        def.reject();
                                    }
                                    return;
                                } else if (event.keyCode === KeyCodes.LEFT_ARROW && startPosition[startPosition.length - 1] === 0) {
                                    if (moveTextCursor({ extend: true, backwards: true, verticalCellSelection: false })) {
                                        def.resolve();
                                    } else {
                                        def.reject();
                                    }
                                    return;
                                }
                            }
                        }

                        // Fix for 33246, setting cursor behind inline hardbreak in Firefox
                        if (_.browser.Firefox && ((event.type === 'mouseup') || (event.type === 'mousedown')) && event.target && DOM.isParagraphNode(event.target) && !self.hasRange()) {
                            // the current node must follow a div.inline.hardbreak and the selection must have offset of 0
                            start = self.getStartPosition();
                            if (Position.isPositionNeighboringSpecifiedNode(event.target, [_.last(start)], DOM.HARDBREAK_NODE_SELECTOR, { follow: true })) {
                                // -> reduce the current position by '1', so that the cursor is positioned before the hard break
                                start[start.length - 1] -= 1;
                                self.setTextSelection(start);
                            }
                        }

                        // Fixing broken selection in FF, when selection goes over a table
                        if (_.browser.Firefox) {
                            if (event.type === 'mousedown') {
                                mouseDownStartPosition = _.clone(startPosition);
                            } else if (event.type === 'mouseup') {
                                if (self.hasRange() && self.getSelectionType() === 'text') {
                                    if (backwards) {
                                        if (!_.isEqual(endPosition, mouseDownStartPosition)) {
                                            endPosition = mouseDownStartPosition;
                                            self.setTextSelection(startPosition, endPosition);
                                        }
                                    } else {
                                        if (!_.isEqual(startPosition, mouseDownStartPosition)) {
                                            startPosition = mouseDownStartPosition;
                                            self.setTextSelection(startPosition, endPosition);
                                        }
                                    }
                                }
                                mouseDownStartPosition = null;
                            }
                        }

                        // Skipping not expanded implicit paragraphs in read-only mode during up or down cursor (left or right cursor was already handled before)
                        if (simpleKeyDownEvent && upDownCursor && readOnly && skipImplicitShrinkedParagraph()) {
                            def.resolve();
                            return;
                        }

                    }

                    // special IE behavior
                    if (_.browser.IE) {
                        if ((event.type === 'mousedown') && (selectedDrawing.length > 0)) {
                            // deselect drawing node in IE (mousedown event has been ignored, see above)
                            DrawingFrame.clearSelection(selectedDrawing);
                            selectedDrawing = $();
                            self.updateClipboardNode();
                        } else if ((event.type === 'mousedown') || (event.type === 'mouseup')) {
                            // handling clicks behind last paragraph in table cell node in IE 11 (27287, 29409)
                            currentCell = $(event.target).closest(DOM.TABLE_CELLNODE_SELECTOR);
                            // if parent is row containing page break, don't do anything
                            if (!DOM.isTablePageBreakRowNode(currentCell.parent())) {
                                currentCellContentNode = $(event.target).closest(DOM.CELLCONTENT_NODE_SELECTOR);
                                if ((currentCell.length > 0) && (currentCellContentNode.length === 0)  && !DOM.isExceededSizeTableNode(currentCell.closest(DOM.TABLE_NODE_SELECTOR))) {
                                    // the mouse event happened behind the last paragraph in the cell. Therefore the new position has to be the last position inside the cell
                                    self.setTextSelection(Position.getLastPositionInCurrentCell(rootNode, Position.getOxoPosition(rootNode, currentCell[0], 0)));
                                }
                            }
                            // Handling of selections over several cells in tables in IE (evaluation in mouseup event)
                            if ((event.type === 'mouseup') && (Position.isPositionInTable(rootNode, self.getStartPosition())) && DOM.isCellContentNode(event.target)) {

                                // using event.offsetX and event.offsetY to calculate, if the mouseup event happened in the same table cell
                                // -> if not, a simplified process for selections outside the cell is required
                                if ((event.offsetX < 0) || (event.offsetY < 0) || (event.offsetX > $(event.target.parentNode).width()) || (event.offsetY > $(event.target.parentNode).height())) {

                                    tempPosition = Position.getOxoPositionFromPixelPosition(rootNode, event.pageX, event.pageY);

                                    if (tempPosition && tempPosition.start && (tempPosition.start.length > 0)) {
                                        self.setTextSelection(backwards ? tempPosition.start : self.getStartPosition(), backwards ? self.getEndPosition() : tempPosition.start);
                                    }
                                }
                            }
                        }
                    }

                    def.resolve();

                } else if (selectedDrawing.length > 0) {

                    // IE destroys text selection in clipboard node immediately after
                    // the mouseup event (this cannot be prevented by returning false)
                    if (_.browser.IE && (event.type === 'mouseup') && (selectedDrawing.length > 0)) {
                        self.restoreBrowserSelection(event);
                    } else if (Utils.CHROME_ON_ANDROID) {
                        //Fix for Bug 34409
                        DrawingFrame.clearSelection(selectedDrawing);
                        selectedDrawing = $();
                        self.updateClipboardNode();
                    }

                    def.resolve();
                } else {
                    if (keyDownEvent && _.browser.Chrome && _.isEqual(originalSelection, { start: startPosition, end: endPosition })) {
                        // see #42989
                        if (event.keyCode === KeyCodes.END) {
                            self.setTextSelection(Position.getLastPositionInParagraph(rootNode, _.clone(startPosition)));
                            def.resolve();
                            return;
                        }
                        if (event.keyCode === KeyCodes.HOME) {
                            self.setTextSelection(Position.getFirstPositionInParagraph(rootNode, _.clone(startPosition)));
                            def.resolve();
                            return;
                        }
                    }
                    if (isVerticalCursor) {
                        self.restoreBrowserSelection();  // avoid uncontrolled vertical browser scrolling (40052)
                        if (_.browser.Firefox || _.browser.Chrome) { app.getView().recalculateDocumentMargin(); }
                    }

                    // Handling of 'slide' selection
                    if (app.getModel().useSlideMode() && self.isTopLevelTextCursor()) {
                        def.resolve();
                        return;
                    }

                    Utils.warn('Selection.processBrowserEvent(): missing valid browser selection');
                    def.reject();
                }

            }, undefined, 'Text: Selection: Evaluating browser selection');

            return def.promise();
        };

        /**
         * Applies the passed cell selection.
         *
         * @param {DOM.Range[]} ranges
         *  The DOM ranges of the cell selection.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.setCellSelection = function (ranges) {
            if (ranges.length > 0) {
                // option 'active' required: point to any cell to be able (used to detect cell selection)
                applyBrowserSelection({ active: ranges[0], ranges: ranges });
            }
            return this;
        };

        /**
         * Drawing the border around a selected drawing and registering the handler.
         * This is the only function, that is allowed to draw the selection around
         * the drawing.
         *
         * @param {TextApplication} app
         *  The application instance containing the drawing.
         *
         * @param {HTMLElement|jQuery} drawingNode
         *  The drawing node to be selected, as DOM node or jQuery object.
         */
        this.drawDrawingSelection = function (app, drawing) {
            if (drawingResizeHandler) { drawingResizeHandler(app, drawing); }
        };

        /**
         * Selects the passed logical text range in the document.
         *
         * @param {Number[]} newStartPosition
         *  The logical position of the first text component in the selection.
         *  Must be the position of a paragraph child node, either a text span,
         *  a text component (fields, tabs), or a drawing node.
         *
         * @param {Number[]} [newEndPosition]
         *  The logical position behind the last text component in the
         *  selection (half-open range). Must be the position of a paragraph
         *  child node, either a text span, a text component (fields, tabs), or
         *  a drawing node. If omitted, sets a text cursor according to the
         *  passed start position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.backwards=false]
         *      If set to true, the selection will be created backwards. Note
         *      that not all browsers allow to create a reverse DOM browser
         *      selection. Nonetheless, the internal selection direction will
         *      always be set to backwards.
         *  @param {Boolean} [options.simpleTextSelection=false]
         *      If set to true, the selection will be set using newStartPosition
         *      and newEndPosition directly. The logical positions are not converted
         *      to a browserSelection and then calculated back in applyBrowserSelection
         *      using Position.getTextLevelOxoPosition. This is especially important
         *      for the very fast operations like 'insertText' (also used for 'Enter',
         *      'Delete' and 'Backspace').
         *  @param {Boolean} [options.insertOperation=false]
         *      If set to true, this setTextSelection is called from an insert operation,
         *      for example an insertText operation. This is important for performance
         *      reasons, because the cursor setting can be simplified.
         *  @param {Boolean} [options.splitOperation=false]
         *      If set to true, this setTextSelection is called from an 'Enter' operation.
         *      This is important for performance reasons, because the cursor setting can
         *      be simplified.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.setTextSelection = function (newStartPosition, newEndPosition, options) {

            var // DOM points for start and end position
                startPoint = null, endPoint = null,
                // whether to create the selection backwards
                selectBackwards = false,
                // the range containing start and end point
                range = null,
                // Performance: Whether the simplified text selection can be used
                simpleTextSelection = Utils.getBooleanOption(options, 'simpleTextSelection', false),
                // whether the selection shall be updated after an external operation (then a 'change' must be triggered, task 31832)
                remoteSelectionUpdate = Utils.getBooleanOption(options, 'remoteSelectionUpdate', false),
                // whether this call was triggered by the userDataUpdateHandler
                userDataUpdateHandler = Utils.getBooleanOption(options, 'userDataUpdateHandler', false),
                // whether the change track pop up triggered this change of selection
                keepChangeTrackPopup = Utils.getBooleanOption(options, 'keepChangeTrackPopup', false),
                // the browser event passed to this method
                browserEvent = Utils.getOption(options, 'event'),
                // whether a drawing inside a drawing group is selected
                drawingInGroupSelection = false,
                // whether a drawing inside the drawing layer is selected
                drawingLayerSelection = false,
                // whether an existing cell selection must not be destroyed
                keepCellSelection = false;

            if (simpleTextSelection) {
                applyBrowserSelection(null, { preserveFocus: true, insertOperation: Utils.getBooleanOption(options, 'insertOperation', false), splitOperation: Utils.getBooleanOption(options, 'splitOperation', false), simpleTextSelection: simpleTextSelection, keepChangeTrackPopup: keepChangeTrackPopup }, { start: newStartPosition, end: newEndPosition });
            } else {
                startPoint = _.isArray(newStartPosition) ? getPointForTextPosition(newStartPosition) : null;
                endPoint = _.isArray(newEndPosition) ? getPointForTextPosition(newEndPosition) : startPoint;

                if (startPoint && !endPoint && DrawingFrame.isDrawingFrame(startPoint.node) && DrawingFrame.isGroupContentNode(startPoint.node.parentNode)) {
                    endPoint = new DOM.Point(startPoint.node, 1);
                    drawingInGroupSelection = true;
                } else if (startPoint && startPoint.node && startPoint.node.parentNode && DOM.isDrawingLayerNode(startPoint.node.parentNode) && DOM.isFirstTextPositionBehindNode(endPoint, DOM.getDrawingPlaceHolderNode(startPoint.node.parentNode))) {
                    endPoint = new DOM.Point(startPoint.node, 1);
                    drawingLayerSelection = true;
                }

                if (startPoint && endPoint) {
                    keepCellSelection = userDataUpdateHandler && self.getSelectionType() === 'cell' && _.isEqual(newStartPosition, this.getStartPosition()) && _.isEqual(newEndPosition, this.getEndPosition());
                    selectBackwards = Utils.getBooleanOption(options, 'backwards', false);
                    range = new DOM.Range(selectBackwards ? endPoint : startPoint, selectBackwards ? startPoint : endPoint);
                    applyBrowserSelection({ active: range, ranges: [range] }, { preserveFocus: true, forceTrigger: remoteSelectionUpdate, keepCellSelection: keepCellSelection, keepChangeTrackPopup: keepChangeTrackPopup, drawingInGroupSelection: drawingInGroupSelection, drawingLayerSelection: drawingLayerSelection, event: browserEvent });
                } else {
                    var browserStartPoint = Utils.getArrayOption(options, 'browserStartPoint');
                    if (remoteSelectionUpdate && browserStartPoint) {
                        // using the selection given by the browser, if restoring the old text selection failed
                        this.setTextSelection(browserStartPoint);
                    } else {
                        Utils.warn('Selection.setTextSelection(): expecting text positions, start=' + JSON.stringify(newStartPosition) + ', end=' + JSON.stringify(newEndPosition));
                    }
                }
            }

            return this;
        };

        /**
         * Updating the selection of a remote read-only editor after executing an external operation. The new
         * selection created by the browser is used to calculate and set a new logical start and end position.
         * This cannot be used, if the selection before executing the external operation was a drawing selection
         * or a cell selection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preserveFocus=true]
         *      If set to true, the DOM element currently focused will be
         *      focused again after the browser selection has been set. Note
         *      that doing so will immediately lose the new browser selection,
         *      if focus is currently inside a text input element.
         *  @param {Boolean} [options.forceTrigger=true]
         *
         */
        this.updateSelectionAfterBrowserSelection = function (options) {

            var // the current browser selection
                browserSelection = self.getBrowserSelection(),
                preserveFocus = Utils.getOption(options, 'preserveFocus', true),
                forceTrigger = Utils.getOption(options, 'forceTrigger', true);

            // checking the current browser selection before using it (fix for 32182)
            if (browserSelection && browserSelection.active && browserSelection.active.start && browserSelection.active.end) {
                applyBrowserSelection(browserSelection, { preserveFocus: preserveFocus, forceTrigger: forceTrigger });
            } else {
                Utils.warn('Selection.updateSelectionAfterBrowserSelection(): incomplete browser selection, ignoring remote selection.');
            }
        };

        /**
         * Returns the first logical position in the document.
         *
         * @returns {Number[]}
         *  The first logical document position.
         */
        this.getFirstDocumentPosition = function () {
            return getFirstPosition();
        };

        /**
         * Returns the last logical position in the document.
         *
         * @returns {Number[]}
         *  The last logical document position.
         */
        this.getLastDocumentPosition = function () {
            return getLastPosition();
        };

        /**
         * Returns the logical start and end position in the document as an
         * object with the properties 'start' and 'end'.
         *
         * @returns {Object}
         *  The first and last logical document positions in an object with
         *  the properties 'start' and 'end'.
         */
        this.getDocumentRange = function () {
            return { start: getFirstPosition(), end: getLastPosition() };
        };

        /**
         * Setting the selection after the document was loaded successfully.
         * It is possible, that the selection was already modified during
         * loading the document. Additionally it is possible that the
         * document was scrolled, but the cursor was not set. In the latter
         * case, the scroll position needs to be restored.
         */
        this.selectDocumentLoadPosition = function () {
            // setting selection to top position, but keeping a selection,
            // that might have been set before during fastloading
            if (self.isUndefinedSelection()) {
                return self.setTextSelection(getFirstTextPosition());
            }
        };

        /**
         * Sets the text cursor to the first available cursor position in the
         * document. Skips leading floating drawings in the first paragraph. If
         * the first content node is a table, selects its first available
         * cell paragraph (may be located in a sub table in the first outer
         * cell).
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.selectTopPosition = function () {
            return this.setTextSelection(getFirstTextPosition());
        };

        /**
         * Check, whether the start position of the current selection is
         * the top position in the document. This is also the
         *
         * @returns {Boolean}
         *  Whether the start position of the current selection is the top
         *  position in the document.
         */
        this.isTopPositionSelected = function () {
            return _.isEqual(self.getStartPosition(), getFirstTextPosition());
        };

        /**
         * Check, whether the hole document is selected
         *
         * @returns {Boolean}
         *  Whether the hole document is selected
         */
        this.isAllSelected = function () {
            var selectionStartPos   = this.getStartPosition(),
                selectionEndPos     = this.getEndPosition(),
                startPos            = getFirstPosition(),
                endPos              = getLastPosition();

            return (_.isEqual(selectionStartPos, startPos) && _.isEqual(selectionEndPos, endPos));
        };

        /**
         * Check, whether the selection is undefined. This is the case,
         * if the start position is an empty array. This is only valid
         * during loading the document (fast load).
         *
         * @returns {Boolean}
         *  Whether the current selection is undefined.
         */
        this.isUndefinedSelection = function () {
            return (_.isArray(self.getStartPosition()) && _.isEmpty(self.getStartPosition()));
        };

        /**
         * Setting valid values for startPosition and endPosition, if a multiple drawing
         * selection is active. This can be the logical position of the first drawing
         * inside the multiple drawing selection. Using this function it is guaranteed, that
         * tests like 'isCursor' deliver the correct result for multiple drawing selections.
         *
         * @param {Object[]} multiSelection
         *  The container for the multiple drawing selection.
         */
        this.setValidMultiSelectionValues = function (multiSelection) {
            if (_.isObject(multiSelection[0])) {
                startPosition = multiSelection[0].startPosition;
                endPosition = multiSelection[0].endPosition;
            }
        };

        /**
         * Selects the entire document or the entire content of a text frame. This is dependent from
         * the current start position.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.selectAll = function () {

            var // whether all drawings can be selected
                // -> is requires support of multiple selection and that the current selection is a slide selection or
                //    a drawing selection or a multiple drawing selection
                selectAllDrawings = app.getModel().useSlideMode() && self.isMultiSelectionSupported() &&
                                    (self.isTopLevelTextCursor() || self.isDrawingSelection() || self.isMultiSelection()),
                // whether the selection is inside a text frame
                isInsideTextFrame = false,
                // whether the selection did not change (for example ctrl-A inside an empty text frame)
                doNothing = false,
                // the text frame
                textFrame = null,
                // the start and end position of the selection (allowing positions in empty text spans before drawings, comments, ...(38378))
                start = null, end = null;

            if (selectAllDrawings) {

                // removing existing drawing selections (single or multiple)
                if (self.isMultiSelection()) {
                    self.clearMultiSelection();
                } else if (self.isDrawingSelection()) {
                    DrawingFrame.clearSelection(selectedDrawing);
                    app.getView().clearVisibleDrawingAnchor();
                    selectedDrawing = $();
                }

                // selecting all drawings
                self.selectAllDrawingsOnSlide();

                // changing selection positions
                startPosition[startPosition.length - 1] = 0; // should already be the case
                endPosition[endPosition.length - 1] = 1;  // setting the first drawing as selected

                self.trigger('change'); // informing the listeners, not using default process
            } else {
                isInsideTextFrame = Position.isPositionInsideTextframe(rootNode, self.getStartPosition());
                textFrame = isInsideTextFrame ? Position.getClosestDrawingTextframe(rootNode, self.getStartPosition()) : null;
                start = getFirstPosition(isInsideTextFrame ? textFrame : null, { allowEmpty: true });
                end = getLastPosition(isInsideTextFrame ? textFrame : null);
                doNothing = _.isEqual(start, end) && _.isEqual(start, self.getStartPosition());
            }

            return selectAllDrawings || doNothing ? self : this.setTextSelection(start, end);
        };

        /**
         * If this selection selects a drawing node, changes the browser
         * selection to a range that starts directly before that drawing node,
         * and ends directly after that drawing.
         *
         * @returns {Selection}
         *  A reference to this instance.
         */
        this.selectDrawingAsText = function () {

            var // the selected drawing, as plain DOM node
                drawingNode = selectedDrawing[0],
                // whether the drawing is in inline mode
                inline = DOM.isInlineDrawingNode(drawingNode),
                // previous text span of the drawing node
                prevTextSpan = inline ? drawingNode.previousSibling : null,
                // next text span of the drawing node (skip following floating drawings)
                nextTextSpan = drawingNode ? Utils.findNextSiblingNode(drawingNode, function () { return DOM.isPortionSpan(this); }) : null,
                // DOM points representing the text selection over the drawing
                startPoint = null, endPoint = null;

            if (drawingNode) {

                // remove drawing selection boxes
                DrawingFrame.clearSelection(selectedDrawing);

                // start point after the last character preceding the drawing
                if (DOM.isPortionSpan(prevTextSpan)) {
                    startPoint = new DOM.Point(prevTextSpan.firstChild, prevTextSpan.firstChild.nodeValue.length);
                }
                // end point before the first character following the drawing
                if (DOM.isPortionSpan(nextTextSpan)) {
                    endPoint = new DOM.Point(nextTextSpan.firstChild, 0);
                }

                // set browser selection (do nothing if no start and no end point
                // have been found - but that should never happen)
                if (startPoint || endPoint) {
                    self.setBrowserSelection(backwards ?
                        new DOM.Range(endPoint || startPoint, startPoint || endPoint) :
                        new DOM.Range(startPoint || endPoint, endPoint || startPoint));
                }
            }

            return this;
        };

        /**
         * Modifies IE selection, after default browser selection is prevented,
         * avoiding resize rectangles (due to IE bug with contenteditable property, see #35678).
         * If header is clicked, returns first next valid node and sets text selection.
         * If footer is clicked, returns first previous valid node and sets selection inside.
         *
         * @param {jQuery} activeRootNode
         *  Root node for the document
         *
         * @param {jQuery} $node
         *  Event's target node, that is clicked on.
         */
        this.updateIESelectionAfterMarginalSelection = function (activeRootNode, $node) {
            var startPos;

            // helper function for IE, to modify browser selection on header&footers
            function modifySelection($node) {
                var $nodeParent = $node.parent();

                function getSiblingFromPageBreakTd($node, direction) {
                    var $nodeParentTr = $node.parentsUntil('.pagecontent', 'tr');

                    if ($nodeParentTr.length > 0) {
                        if (direction === 'prev' && $nodeParentTr.prev().length) {
                            return $nodeParentTr.prev();
                        } else if (direction === 'next' && $nodeParentTr.next().length) {
                            return $nodeParentTr.next();
                        }
                        return false;
                    }
                }

                if (DOM.isPageBreakNode($node)) {
                    if ($node.next().length > 0) {
                        return $node.next();
                    } else if ($nodeParent.next().length > 0) {
                        return $nodeParent.next();
                    }
                } else if (DOM.isFooterNode($node) || $node.hasClass('inner-pb-line')) {
                    if (DOM.isPageBreakNode($nodeParent)) {
                        if ($nodeParent.prev().length > 0) {
                            return $nodeParent.prev();
                        } else {
                            if ($nodeParent.parent().is('td')) {
                                return getSiblingFromPageBreakTd($nodeParent, 'prev');
                            }
                            return $nodeParent.parent().prev();
                        }
                    } else if ($node.parentsUntil('.page', '.footer-wrapper').length > 0) {
                        return DOM.getPageContentNode(activeRootNode).children('.p').last();
                    }
                } else if (DOM.isHeaderNode($node)) {
                    if (DOM.isPageBreakNode($nodeParent)) {
                        if ($nodeParent.next().length > 0) {
                            return $nodeParent.next();
                        } else {
                            if ($nodeParent.parent().is('td')) {
                                return getSiblingFromPageBreakTd($nodeParent, 'next');
                            }
                            return $nodeParent.parent().next();
                        }
                    } else if ($node.parentsUntil('.page', '.header-wrapper').length > 0) {
                        return DOM.getPageContentNode(activeRootNode).children('.p').first();
                    }
                }

                return false;
            }

            if ($node.hasClass('cover-overlay')) {
                $node = $node.parent();
            }

            if ($node.is('.header, header-wrapper')) {
                $node = modifySelection($node);
                if ($node.is('table, tr')) {
                    $node = $node.find('.p').first();
                }
                if ($node.children().length) {
                    $node = $node.children().first();
                }
                if ($node.length > 0) {
                    startPos = Position.getOxoPosition(activeRootNode, $node, 0);
                    if (startPos) {
                        self.setTextSelection(startPos);
                    } else {
                        Utils.warn('Selection.updateIESelectionAfterMarginalSelection: no valid position found!');
                    }
                }
            } else if ($node.is('.footer, footer-wrapper')) {
                $node = modifySelection($node);
                if ($node.is('table, tr')) {
                    $node = $node.find('.p').last();
                }
                if ($node.children().length) {
                    $node = $node.children().last();
                }
                startPos = Position.getOxoPosition(activeRootNode, $node, 0);
                if (startPos) {
                    self.setTextSelection(startPos);
                } else {
                    Utils.warn('Selection.updateIESelectionAfterMarginalSelection: no valid position found!');
                }
            }
        };

        /**
         * Updating the selection of a remote client without write privileges.
         */
        this.updateRemoteSelection = function () {

            var // whether the selection before applying the external operation was a range selection
                wasRange = false,
                // whether a drawing is selected
                drawingSelected = false,
                // whether the browser selection can be evaluated
                useBrowserSelection = true,
                // whether a cell range is selected
                cellSelected = false,
                // old logical start and end positions
                oldStartPosition = null, oldEndPosition = null,
                // the drawing node of a selected drawing
                drawingNode,
                // logical position of the selected drawing
                drawingStartPosition, drawingEndPosition,
                // a new calculated valid logical position
                validPos,
                // the page node
                editdiv = app.getModel().getNode(),
                // the node containing the focus
                focusNode = null;

            // checking if the selection before applying the external operation was a range selection
            if (self.hasRange()) {
                wasRange = true;
                oldStartPosition = _.clone(self.getStartPosition());
                oldEndPosition = _.clone(self.getEndPosition());
                drawingSelected = (self.getSelectionType() === 'drawing');
                cellSelected = (self.getSelectionType() === 'cell');
            }

            if (drawingSelected || cellSelected) { useBrowserSelection = false; }

            // using the browser selection as new selection
            if (useBrowserSelection) {
                self.updateSelectionAfterBrowserSelection();
                // check if it is necessary to restore a selection range (maybe the browser removed a selection range)
                if (wasRange && !self.hasRange()) {
                    validPos = Position.findValidSelection(editdiv, oldStartPosition, oldEndPosition, self.getLastDocumentPosition());
                    self.setTextSelection(validPos.start, validPos.end, { remoteSelectionUpdate: true });
                }
            } else if (drawingSelected) {

                drawingNode = self.getSelectedDrawing();

                if ($(drawingNode).closest(DOM.PAGECONTENT_NODE_SELECTOR).length > 0) {
                    // is the drawing still in the dom?
                    drawingStartPosition = Position.getOxoPosition(editdiv, drawingNode, 0);
                    drawingEndPosition = Position.increaseLastIndex(drawingStartPosition);

                    if (!_.isEqual(drawingStartPosition, self.getStartPosition())) {
                        self.setTextSelection(drawingStartPosition, drawingEndPosition, { remoteSelectionUpdate: true });
                    }
                } else {
                    validPos = Position.findValidSelection(editdiv, self.getStartPosition(), self.getEndPosition(), self.getLastDocumentPosition());
                    self.setTextSelection(validPos.start, validPos.end, { preserveFocus: false, remoteSelectionUpdate: true });

                    // The div.page must get the focus. Unfortunately this fails in Chrome,
                    // if Chrome does not have the focus.
                    if (!$(Utils.getActiveElement()).is(DOM.PAGE_NODE_SELECTOR)) {
                        focusNode = Utils.getActiveElement();
                        Utils.info('Missing browser focus -> changing focus failed. Focus at ' + focusNode.nodeName + '.' + focusNode.className);
                    }
                }

            } else if (cellSelected) {

                // Updating position for cell selection
                self.updateSelectionAfterBrowserSelection();

                // maybe the cell selection was destroyed by the operation -> self.getSelectedCellRange() returns null
                if (!self.getSelectedCellRange()) {
                    validPos = Position.findValidSelection(editdiv, oldStartPosition, oldEndPosition, self.getLastDocumentPosition());
                    self.setTextSelection(validPos.start, validPos.end, { remoteSelectionUpdate: true });
                }

            }
        };

        // iterators ----------------------------------------------------------

        /**
         * Calls the passed iterator function for each table cell, if this
         * selection is located inside a table. Processes a rectangular cell
         * selection (if supported by the browser), otherwise a row-oriented
         * text selection inside a table.
         * This function might also be called, when the selection is not inside
         * a table. In this case it leaves immediately with Utils.BREAK. This
         * happens for example during loading a document (Fix for 29021).
         *
         * @param {Function} iterator
         *  The iterator function that will be called for every table cell node
         *  covered by this selection. Receives the following parameters:
         *      (1) {HTMLTableCellElement} the visited DOM cell element,
         *      (2) {Number[]} its logical position (the last two elements in
         *          this array represent the row and column index of the cell),
         *      (3) {Number} the row offset, relative to the first row
         *          in the rectangular cell range,
         *      (4) {Number} the column offset, relative to the first column
         *          in the rectangular cell range.
         *  The last two parameters will be undefined, if the current selection
         *  is a text range in a table. If the iterator returns the Utils.BREAK
         *  object, the iteration process will be stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateTableCells = function (iterator, context) {

            var // information about the cell range and containing table
                cellRangeInfo = this.getSelectedCellRange(),

                // the DOM cells
                firstCellInfo = null, lastCellInfo = null,
                // current cell, and its logical position
                cellInfo = null, cellNode = null, cellPosition = null,

                // row/column indexes for loops
                row = 0, col = 0;

            // check enclosing table, get its position
            if (!cellRangeInfo) {
                return Utils.BREAK;
            }

            // resolve position to closest table cell
            firstCellInfo = Position.getDOMPosition(cellRangeInfo.tableNode, cellRangeInfo.firstCellPosition, true);
            lastCellInfo = Position.getDOMPosition(cellRangeInfo.tableNode, cellRangeInfo.lastCellPosition, true);
            if (!firstCellInfo || !$(firstCellInfo.node).is('td') || !lastCellInfo || !$(lastCellInfo.node).is('td')) {
                Utils.error('Selection.iterateTableCells(): no table cells found for cell positions');
                return Utils.BREAK;
            }

            // visit all cells for rectangular cell selection mode
            if (cellRangeSelected) {

                // loop over all cells in the range
                for (row = 0; row < cellRangeInfo.height; row += 1) {
                    for (col = 0; col < cellRangeInfo.width; col += 1) {

                        // cell position relative to table
                        cellPosition = [cellRangeInfo.firstCellPosition[0] + row, cellRangeInfo.firstCellPosition[1] + col];
                        cellInfo = Position.getDOMPosition(cellRangeInfo.tableNode, cellPosition);

                        // cellInfo will be undefined, if current position is covered by a merged cell
                        if (cellInfo && $(cellInfo.node).is('td')) {
                            cellPosition = cellRangeInfo.tablePosition.concat(cellPosition);
                            if (iterator.call(context, cellInfo.node, cellPosition, row, col) === Utils.BREAK) {
                                return Utils.BREAK;
                            }
                        }
                    }
                }

            // otherwise: visit all cells row-by-row (text selection mode)
            } else {

                cellNode = firstCellInfo.node;
                while (cellNode) {

                    // visit current cell
                    cellPosition = Position.getOxoPosition(cellRangeInfo.tableNode, cellNode, 0);
                    row = cellPosition[0] - cellRangeInfo.firstCellPosition[0];
                    col = cellPosition[1] - cellRangeInfo.firstCellPosition[1];
                    cellPosition = cellRangeInfo.tablePosition.concat(cellPosition);
                    if (iterator.call(context, cellNode, cellPosition, row, col) === Utils.BREAK) { return Utils.BREAK; }

                    // last cell reached
                    if (cellNode === lastCellInfo.node) { return; }

                    // find next cell node (either next sibling, or first child
                    // of next row, but skip embedded tables)
                    cellNode = Utils.findNextNode(cellRangeInfo.tableNode, cellNode, 'td', DOM.TABLE_NODE_SELECTOR);
                }

                // in a valid DOM tree, there must always be valid cell nodes until
                // the last cell has been reached, so this point should never be reached
                Utils.error('Selection.iterateTableCells(): iteration exceeded end of selection');
                return Utils.BREAK;
            }
        };

        /**
         * Calls the passed iterator function for each drawing, if this is a drawing
         * selection or a multiple drawing selection.
         * This function might also be called, when no drawing node is selected. In
         * this case it leaves immediately with Utils.BREAK.
         *
         * @param {Function} iterator
         *  The iterator function that will be called for every drawing node
         *  covered by this selection. Receives the following parameter:
         *      (1) {jQuery} the visited jQueryfied DOM drawing element
         *  If the iterator returns the Utils.BREAK
         *  object, the iteration process will be stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateDrawings = function (iterator, context) {

            var // the container of all drawings
                allDrawings = (self.isMultiSelectionSupported() && self.isMultiSelection()) ? self.getAllSelectedDrawingNodes() : (self.isAdditionalTextframeSelection() ? self.getSelectedTextFrameDrawing() : selectedDrawing);

            // exit if no drawing is selected
            if (allDrawings.length === 0) { return Utils.BREAK; }

            _.each(allDrawings, function (oneDrawing) {
                if (DrawingFrame.isGroupDrawingFrame(oneDrawing)) {
                    _.each($(oneDrawing).find(DrawingFrame.NODE_SELECTOR), function (drawingInGroup) {
                        if (iterator.call(context, drawingInGroup) === Utils.BREAK) {
                            return Utils.BREAK;
                        }
                    });
                } else {
                    if (iterator.call(context, oneDrawing) === Utils.BREAK) {
                        return Utils.BREAK;
                    }
                }
            });

        };

        /**
         * Calls the passed iterator function for specific content nodes
         * (tables and paragraphs) selected by this selection instance. It is
         * possible to visit all paragraphs embedded in all covered tables and
         * nested tables, or to iterate on the 'shortest path' by visiting
         * tables exactly once if they are covered completely by the selection
         * range and skipping the embedded paragraphs and sub tables. If the
         * selection range end at the very beginning of a paragraph (before the
         * first character), this paragraph is not considered to be included in
         * the selected range.
         *
         * @param {Function} iterator
         *  The iterator function that will be called for every content node
         *  (paragraphs and tables) covered by this selection. Receives the
         *  following parameters:
         *      (1) {HTMLElement} the visited content node,
         *      (2) {Number[]} its logical position,
         *      (3) {Number|Undefined} the logical index of the first text
         *          component covered by the FIRST paragraph; undefined for
         *          all other paragraphs and tables (may point after the last
         *          existing child text component, if the selection starts at
         *          the very end of a paragraph),
         *      (4) {Number|Undefined} the logical index of the last child text
         *          component covered by the LAST paragraph (closed range);
         *          undefined for all other paragraphs and tables,
         *      (5) {Boolean} whether the selection includes the beginning of
         *          the parent container of the current content node,
         *      (6) {Boolean} whether the selection includes the end of the
         *          parent container of the current content node.
         *  If the selection represents a text cursor, the start position will
         *  exceed the end position by 1. Thus, a text cursor in an empty
         *  paragraph will be represented by the text range [0, -1]. If the
         *  iterator returns the Utils.BREAK object, the iteration process will
         *  be stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.shortestPath=false]
         *      If set to true, tables that are covered completely by this
         *      selection will be visited, but their descendant components
         *      (paragraphs and embedded tables) will be skipped in the
         *      iteration process. By default, this method visits all
         *      paragraphs embedded in all tables and their sub tables, but
         *      does not visit the table objects. Has no effect for tables that
         *      contain the end paragraph, because these tables are not fully
         *      covered by the selection. Tables that contain the start
         *      paragraph will never be visited, because they start before the
         *      selection.
         *  @param {Number[]} [options.startPos]
         *      An optional logical start position that can be used to reduce the
         *      iteration range. If not specified, the current start position
         *      of the selection is used.
         *  @param {Number[]} [options.endPos]
         *      An optional logical end position that can be used to reduce the
         *      iteration range. If not specified, the current start position
         *      of the selection is used.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateContentNodes = function (iterator, context, options) {

            // checking, if a selection already exists. This is not the case for example in fast loading process.
            if (self.isUndefinedSelection()) { return Utils.BREAK; }

            var // the start position, that is used inside this function
                localStartPos = Utils.getArrayOption(options, 'startPos', startPosition),
                // the end position, that is used inside this function
                localEndPos = Utils.getArrayOption(options, 'endPos', endPosition),

                // start node and offset (pass true to NOT resolve text spans to text nodes)
                startInfo = Position.getDOMPosition(rootNode, localStartPos, true),
                // end node and offset (pass true to NOT resolve text spans to text nodes)
                endInfo = Position.getDOMPosition(rootNode, localEndPos, true),

                // whether to iterate on shortest path (do not traverse into completely covered tables)
                shortestPath = Utils.getBooleanOption(options, 'shortestPath', false),

                // max iterations
                maxIterations = Utils.getOption(options, 'maxIterations', -1),
                // iterator-counter
                iterateCount = 0,

                // paragraph nodes containing the passed start and end positions
                firstParagraph = null, lastParagraph = null,
                // current content node while iterating
                contentNode = null, parentCovered = false;

            // visit the passed content node (paragraph or table); or table child nodes, if not in shortest-path mode
            function visitContentNode(contentNode, parentCovered) {

                var // each call of the iterator get its own position array (iterator is allowed to modify it)
                    position = Position.getOxoPosition(rootNode, contentNode),
                    // start text offset in first paragraph
                    startOffset = (contentNode === firstParagraph) ? _.last(localStartPos) : undefined,
                    // end text offset in last paragraph (convert half-open range to closed range)
                    endOffset = (contentNode === lastParagraph) ? (_.last(localEndPos) - 1) : undefined;

                // If the localEndPos is an empty paragraph in a cell range selection, the attribute
                // must also be assigned to this end position. But in this case is the endOffset '-1' in the
                // last cell. Concerning to task 27361 the attribute is not assigned to the empty paragraph
                // in the bottom right cell of a table. To fix this, it is in this scenario necessary to
                // increase the endOffset to the value '0'! (Example selection: [0,0,0,0,0] to [0,1,1,0,0])
                // On the other hand, it is important that a selection that ends at the beginning of a
                // paragraph, ignores the attribute assignment for the following paragraph. Example:
                // Selection [0,0] to [1,0]: Paragraph alignment 'center' must not be assigned to
                // paragraph [1].

                if ((cellRangeSelected) && (endOffset < 0)) { endOffset = 0; }  // Task 27361

                // visit the content node, but not the last paragraph, if selection
                // does not start in that paragraph and ends before its beginning
                // (otherwise, it's a cursor in an empty paragraph)
                if ((contentNode === firstParagraph) || (contentNode !== lastParagraph) || (endOffset >= Position.getFirstTextNodePositionInParagraph(contentNode))) {
                    return iterator.call(context, contentNode, position, startOffset, endOffset, parentCovered);
                }
            }

            // find the first content node in passed root node (either table or embedded paragraph depending on shortest-path option)
            function findFirstContentNode(rootNode) {

                // in shortest-path mode, use first table or paragraph in cell,
                // otherwise find first paragraph which may be embedded in a sub table)
                return Utils.findDescendantNode(rootNode, shortestPath ? DOM.CONTENT_NODE_SELECTOR : DOM.PARAGRAPH_NODE_SELECTOR);
            }

            // find the next content node in DOM tree (either table or embedded paragraph depending on shortest-path option)
            function findNextContentNode(rootNode, contentNode, lastParagraph) {

                // find next content node in DOM tree (searches in own siblings, AND in other nodes
                // following the parent node, e.g. the next table cell, or paragraphs following the
                // containing table, etc.; but skips drawing nodes that may contain their own paragraphs)
                contentNode = Utils.findNextNode(rootNode, contentNode, DOM.CONTENT_NODE_SELECTOR, DOM.SKIP_DRAWINGS_AND_P_BREAKS_SELECTOR);

                // iterate into a table, if shortest-path option is off, or the end paragraph is inside the table
                while (DOM.isTableNode(contentNode) && (!shortestPath || (lastParagraph && Utils.containsNode(contentNode, lastParagraph)))) {
                    contentNode = Utils.findDescendantNode(contentNode, DOM.CONTENT_NODE_SELECTOR);
                }

                return contentNode;
            }

            // check validity of passed positions
            if (!startInfo || !startInfo.node || !endInfo || !endInfo.node) {
                Utils.error('Selection.iterateContentNodes(): invalid selection, cannot find first or last DOM node');
                return Utils.BREAK;
            }

            // find first and last paragraph node (also in table cell selection mode)
            firstParagraph = Utils.findClosest(rootNode, startInfo.node, DOM.PARAGRAPH_NODE_SELECTOR);
            lastParagraph = Utils.findClosest(rootNode, endInfo.node, DOM.PARAGRAPH_NODE_SELECTOR);
            if (!firstParagraph || !lastParagraph) {
                // this can happen for absolutely positioned drawings in drawing layer
                // -> switching from drawing layer node to page content
                if (!firstParagraph && startInfo.node.parentNode && DOM.isDrawingLayerNode(startInfo.node.parentNode)) {
                    firstParagraph = Utils.findClosest(rootNode, DOM.getDrawingPlaceHolderNode(startInfo.node), DOM.PARAGRAPH_NODE_SELECTOR);
                }

                if (!lastParagraph && endInfo.node.parentNode && DOM.isDrawingLayerNode(endInfo.node.parentNode)) {
                    lastParagraph = Utils.findClosest(rootNode, DOM.getDrawingPlaceHolderNode(endInfo.node), DOM.PARAGRAPH_NODE_SELECTOR);
                }

                if (!firstParagraph || !lastParagraph) {
                    Utils.error('Selection.iterateContentNodes(): invalid selection, cannot find containing paragraph nodes');
                    return Utils.BREAK;
                }
            }

            // rectangular cell range selection
            if (cellRangeSelected) {

                // entire table selected
                if (shortestPath && tableSelected) {
                    return visitContentNode(this.getEnclosingTable(), false);
                }

                // visit all table cells, iterate all content nodes according to 'shortest-path' option
                return this.iterateTableCells(function (cell) {
                    for (contentNode = findFirstContentNode(cell); contentNode; contentNode = findNextContentNode(cell, contentNode)) {
                        if (visitContentNode(contentNode, true) === Utils.BREAK) { return Utils.BREAK; }
                    }
                }, this);

            } else if (self.isAnyTextFrameDrawingSelection() && !shortestPath) {
                // iterating over all text frame drawings (one text frame is selected or this
                // is a multiple drawing selection with at least one text frame drawing
                return this.iterateDrawings(function (oneDrawing) {

                    if (DrawingFrame.isTextFrameShapeDrawingFrame(oneDrawing)) {
                        for (contentNode = findFirstContentNode(oneDrawing); contentNode; contentNode = findNextContentNode(oneDrawing, contentNode)) {
                            if (visitContentNode(contentNode, true) === Utils.BREAK) { return Utils.BREAK; }
                        }
                    }

                }, this);

            }

            // check, if the first paragraph can be expanded to a table, if this table is completely covered
            // This is necessary for deleting complete tables at the beginning of the document, after pressing Ctrl-A.
            // Tables at the end of the document are already fine, because of the following implicit paragraph.
            // 'shortestPath' must be checked, so that setting of attributes still works after Ctrl-A.
            if (shortestPath && _(localStartPos).all(function (value) { return (value === 0); }) && localEndPos[0] > 0 && Position.isPositionInTable(rootNode, localStartPos)) {
                firstParagraph = Utils.findFarthest(rootNode, firstParagraph, DOM.TABLE_NODE_SELECTOR);
            }

            // iterate through all paragraphs and tables until the end paragraph has been reached
            contentNode = firstParagraph;
            while (contentNode) {

                // check whether the parent container is completely covered
                parentCovered = !Utils.containsNode(contentNode.parentNode, firstParagraph) && !Utils.containsNode(contentNode.parentNode, lastParagraph);

                // visit current content node
                if (visitContentNode(contentNode, parentCovered) === Utils.BREAK) { return Utils.BREAK; }

                // maxIterations is set and reached
                if ((maxIterations !== -1) && (iterateCount + 1 === maxIterations)) { return Utils.BREAK; }

                // end of selection, or max iterations reached
                if (contentNode === lastParagraph) { return; }

                // find next content node in DOM tree (next sibling paragraph or
                // table, or first node in next cell, or out of last table cell...)
                contentNode = findNextContentNode(rootNode, contentNode, lastParagraph);
                iterateCount++;
            }

            // in a valid DOM tree, there must always be valid content nodes until end
            // paragraph has been reached, so this point should never be reached
            Utils.error('Selection.iterateContentNodes(): iteration exceeded end of selection');
            return Utils.BREAK;
        };

        /**
         * Calls the passed iterator function for specific nodes selected by
         * this selection instance. It is possible to visit all child nodes
         * embedded in all covered paragraphs (also inside tables and nested
         * tables), or to iterate on the 'shortest path' by visiting content
         * nodes (paragraphs or tables) exactly once if they are covered
         * completely by this selection and skipping the embedded paragraphs,
         * sub tables, and text contents.
         *
         * @param {Function} iterator
         *  The iterator function that will be called for every node covered by
         *  this selection. Receives the following parameters:
         *      (1) {HTMLElement} the visited DOM node object,
         *      (2) {Number[]} the logical start position of the visited node
         *          (if the node is a partially covered text span, this is the
         *          logical position of the first character covered by the
         *          selection, NOT the start position of the span itself),
         *      (3) {Number} the relative start offset inside the visited node
         *          (usually 0; if the visited node is a partially covered text
         *          span, this offset is relative to the first character in the
         *          span covered by the selection),
         *      (4) {Number} the logical length of the visited node (1 for
         *          content nodes, character count of the covered part of text
         *          spans).
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.shortestPath=false]
         *      If set to true, tables and paragraphs that are covered
         *      completely by this selection will be visited directly and once,
         *      and their descendant components will be skipped in the
         *      iteration process. By default, this method visits the child
         *      nodes of all paragraphs and tables embedded in tables. Has no
         *      effect for tables that contain the end paragraph, because these
         *      tables are not fully covered by the selection. Tables that
         *      contain the start paragraph will never be visited, because they
         *      start before the selection.
         *  @param {Boolean} [options.split=false]
         *      If set to true, the first and last text span not covered
         *      completely by this selection will be split before the iterator
         *      function will be called. The iterator function will always
         *      receive a text span that covers the contained text completely.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateNodes = function (iterator, context, options) {

            var // whether to iterate on shortest path (do not traverse into completely covered content nodes)
                shortestPath = Utils.getBooleanOption(options, 'shortestPath', false),
                // split partly covered text spans before visiting them
                split = Utils.getBooleanOption(options, 'split', false),

                // start node and offset
                startInfo = null;

            // special case 'simple cursor': visit the text span
            if (this.isTextCursor()) {

                // start node and offset (pass true to NOT resolve text spans to text nodes)
                startInfo = Position.getDOMPosition(rootNode, startPosition, true);
                if (!startInfo || !startInfo.node) {
                    Utils.error('Selection.iterateNodes(): invalid selection, cannot find DOM node at start position ' + JSON.stringify(startPosition));
                    return Utils.BREAK;
                }

                // if located at the beginning of a component: use end of preceding text span if available
                if ((startInfo.offset === 0) && DOM.isPortionSpan(startInfo.node.previousSibling)) {
                    startInfo.node = startInfo.node.previousSibling;
                    startInfo.offset = startInfo.node.firstChild.nodeValue.length;
                }

                // visit the text component node (clone, because iterator is allowed to change passed position)
                return iterator.call(context, startInfo.node, _.clone(startPosition), startInfo.offset, 0);
            }

            // iterate the content nodes (paragraphs and tables) covered by the selection
            return this.iterateContentNodes(function (contentNode, position, startOffset, endOffset) {

                var // single-component selection
                    singleComponent = false,
                    // text span at the very beginning or end of a paragraph
                    textSpan = null;

                // visit fully covered content node in 'shortest-path' mode
                if (shortestPath && !_.isNumber(startOffset) && !_.isNumber(endOffset)) {
                    return iterator.call(context, contentNode, position, 0, 1);
                }

                // if selection starts after the last character in a paragraph, visit the last text span
                if (_.isNumber(startOffset) && (startOffset >= Position.getLastTextNodePositionInParagraph(contentNode))) {
                    if ((textSpan = DOM.findLastPortionSpan(contentNode))) {
                        return iterator.call(context, textSpan, position.concat([startOffset]), textSpan.firstChild.nodeValue.length, 0);
                    }
                    Utils.error('Selection.iterateNodes(): cannot find last text span in paragraph at position ' + JSON.stringify(position));
                    return Utils.BREAK;
                }

                // visit covered text components in the paragraph
                singleComponent = _.isNumber(startOffset) && _.isNumber(endOffset) && (startOffset === endOffset);
                return Position.iterateParagraphChildNodes(contentNode, function (node, nodeStart, nodeLength, nodeOffset, offsetLength) {

                    // skip floating drawings (unless they are selected directly) and helper nodes
                    if (DOM.isTextSpan(node) || DOM.isInlineComponentNode(node) || (singleComponent && DOM.isFloatingDrawingNode(node))) {
                        // create local copy of position, iterator is allowed to change the array
                        return iterator.call(context, node, position.concat([nodeStart + nodeOffset]), nodeOffset, offsetLength);
                    }

                    // options for Position.iterateParagraphChildNodes(): visit empty text spans
                }, this, { allNodes: true, start: startOffset, end: endOffset, split: split });

                // options for Selection.iterateContentNodes()
            }, this, { shortestPath: shortestPath, maxIterations: Utils.getOption(options, 'maxIterations') });

        };

        /**
         * Setter for new rootNode inside selection. Happens when, for ex., when edit mode or header/footer is edited,
         * or return from this edit mode, to normal document edit mode.
         * In future may be expanded to other rootNodes
         *
         * @param {Node|jQuery} newNode
         */
        this.setNewRootNode = function (newNode) {
            rootNode = $(newNode);
            forceSelectionChangeEvent = true;
        };

        /**
         *  Getter for the current root node of the selection.
         *
         * @returns {HTMLElement|jQuery}
         *  The current active root node for the selection.
         */
        this.getRootNode = function () {
            return rootNode;
        };

        /**
         * Getter for the current target ID of the root node of the selection,
         * if there is one.
         *
         * @returns {String}
         *  The target of the current active root node for the selection.
         */
        this.getRootNodeTarget = function () {
            return DOM.getTargetContainerId(rootNode);
        };

        /**
         * Sets the current selection to the first, respectively to the next, drawing.
         * Except drawings in headers/footers.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.backwards=false]
         *      decides, whether the next selected drawing should be before or after
         *      the current
         */
        this.selectNextDrawing = function (options) {

            var // the page node and the slide node
                editdiv = null, slide = null,
                // the container for all drawings
                arrDrawings = null,
                // the index specifying the selected drawing
                selectedIndex = 0,
                // the logical start and end positions
                start = null, end = null,
                // whether the next or the previous drawing shall be selected
                backwards = Utils.getBooleanOption(options, 'backwards', false),
                // the model object
                model = app.getModel();

            // helper function to determine the next or previous drawing index
            function getNewDrawingIndex(oldIndex, maxLength, localBackwards) {

                var newIndex = 0;

                if (localBackwards) {
                    // count one down to select the prev drawing, or restart at zero
                    newIndex = (oldIndex > 0) ? (oldIndex - 1) : (maxLength - 1);
                } else {
                    // count one up to select the next drawing, or restart at zero
                    newIndex = (oldIndex + 1 < maxLength) ? (oldIndex + 1) : 0;
                }

                return newIndex;
            }

            if (model.useSlideMode()) {

                // using only the drawings on the current slide
                slide = model.getSlideById(model.getActiveSlideId());
                arrDrawings = slide.children(DrawingFrame.NODE_SELECTOR);

                if (arrDrawings.length > 0) {
                    selectedIndex = _.indexOf(arrDrawings, selectedDrawing[0]);
                    selectedIndex = getNewDrawingIndex(selectedIndex, arrDrawings.length, backwards);
                }

                start = Position.getOxoPosition(slide, arrDrawings[selectedIndex], 0);
                start = model.getActiveSlidePosition().concat(start);

            } else {

                editdiv = model.getNode();
                arrDrawings = editdiv.find('.pagecontent > :not(.header, .footer) > .drawing, .pagecontent > :not(.header, .footer) > .drawingplaceholder');

                if (arrDrawings.length > 0) {

                    // if already a drawing is selected, select next one
                    if (selectedDrawing.length > 0) {

                        selectedIndex = _.indexOf(arrDrawings, selectedDrawing[0]);
                        // if the selected drawing is not in the arr, it should be found via the placeholder
                        if (selectedIndex === -1) {
                            selectedIndex = _.indexOf(arrDrawings, $('.drawingplaceholder[data-drawingid="' + selectedDrawing.data('drawingid') + '"]').get(0));
                        }

                        selectedIndex = getNewDrawingIndex(selectedIndex, arrDrawings.length, backwards);
                    }

                    // get start/end positon of drawing
                    start = Position.getOxoPosition(editdiv, arrDrawings[selectedIndex], 0);
                }
            }

            if (start) {
                // the end position is always increased by '1'
                end = Position.increaseLastIndex(start);

                // select drawing
                self.setTextSelection(start, end);

                // scroll to position of drawing
                app.getView().scrollToChildNode(arrDrawings[selectedIndex]);
            }
        };

        /**
         * Setting the cursor into the text frame at the last text position inside the text frame.
         * Does nothing, if this is not a drawing or if the selected drawing is not a text frame.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.complete=false]
         *      Whether the complete content inside the text frame shall be selected. This is the
         *      default in the presentation app. In the text app, the cursor is set to the last
         *      text position inside the text frame.
         */
        this.setSelectionIntoTextframe = function (options) {

            var // the new text position
                startPosition = null, endPosition = null,
                // whether the complete content shall be selected
                complete = Utils.getBooleanOption(options, 'complete', false);

            if (selectedDrawing.length > 0 && DrawingFrame.isTextFrameShapeDrawingFrame(selectedDrawing)) {
                // getting the first text position inside the text frame
                endPosition = Position.getLastTextPositionInTextFrame(app.getModel().getCurrentRootNode(), selectedDrawing);

                // selecting the complete content inside the text frame
                if (complete) { startPosition = Position.getFirstTextPositionInTextFrame(app.getModel().getCurrentRootNode(), selectedDrawing); }
            }

            // setting the text selection
            if (!complete && endPosition) {
                self.setTextSelection(endPosition);
            } else if (complete && startPosition && endPosition) {
                self.setTextSelection(startPosition, endPosition);
            }
        };

        /**
         * Toggling the selection between a text frame and its content. In the presentation app
         * this is triggered by pressing 'F2'. If a text is selected, the new selection will be
         * the surrounding text frame. And if a text frame is selected, its content will be
         * selected (dependent from the options).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.complete=false]
         *      Whether the complete content inside the text frame shall be selected. This is the
         *      default in the presentation app. In the text app, the cursor is set to the last
         *      text position inside the text frame.
         */
        this.toggleTextFrameAndTextSelection = function (options) {

            var // the logical start position of the text frame
                start = null;

            if (self.isAdditionalTextframeSelection()) {
                // select the text frame itself
                start = Position.getOxoPosition(app.getModel().getCurrentRootNode(), self.getSelectedTextFrameDrawing(), 0);
                if (start) { self.setTextSelection(start, Position.increaseLastIndex(start)); }
            } else {
                // jumping into the text frame (if one is selected)
                self.setSelectionIntoTextframe(options);
            }
        };

        /**
         * Getting a valid text position, if the current selection is a drawing selection.
         * If this drawing is a text frame, the cursor will be set into the drawing. If this
         * drawing is no text frame, the cursor is set before the drawing.
         *
         * @returns {Array<Number>|null}
         *  The new logical text position. Or null, if could not be determined.
         */
        this.getValidTextPositionAccordingToDrawingFrame = function () {

            var // the new text position
                textPosition = null;

            if (selectedDrawing.length > 0) {

                if (DrawingFrame.isTextFrameShapeDrawingFrame(selectedDrawing)) {
                    // setting the selection into the text frame
                    textPosition = Position.getLastTextPositionInTextFrame(app.getModel().getCurrentRootNode(), selectedDrawing);
                } else {
                    // setting the selection before the drawing (application specific!)
                    textPosition = DOM.isSlideNode(selectedDrawing.parent()) ? null : self.getStartPosition();
                }
            }

            return textPosition;
        };

        /**
         * Setting the handler for resizing and moving drawing. This handler is
         * different in the several applications.
         *
         * @params {Function} handler
         *  The handler function for selection drawings and moving and resizing
         *  of drawings.
         */
        this.setDrawingResizeHandler = function (handler) {
            drawingResizeHandler = handler;
        };

        /**
         * Setting the logical position for the active slide. For the slide at position [3], this
         * selection is done on position [3,0]. All other selections are removed, also the multi
         * drawing selections.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.triggerChange=true]
         *      If set to false, the change of the selection will not cause the 'change' event
         *      to be triggered. This can be used for performance reason in special circumstances.
         */
        this.setSlideSelection = function (options) {

            var // whether the change event shall be triggered
                doTrigger = Utils.getBooleanOption(options, 'triggerChange', true);

            self.setTextSelection(Position.appendNewIndex(app.getModel().getActiveSlidePosition()));
            self.getClipboardNode().text('\xa0').toggle(true); //.focus(); TODO is this necessary here?
            if (doTrigger) { self.trigger('change'); }
        };

        /**
         * Clearing all kinds of drawing selections.
         */
        this.clearAllDrawingSelections = function () {

            // clearing the selection of the selected drawing
            if (self.getSelectedDrawing().length > 0) {
                DrawingFrame.clearSelection(selectedDrawing);
                selectedDrawing = $();
            }

            // clearing the selection of an additional text frame selection
            if (self.isAdditionalTextframeSelection()) {
                app.getModel().addTemplateTextIntoTextSpan(selectedTextFrameDrawing);
                DrawingFrame.clearSelection(selectedTextFrameDrawing);
                selectedTextFrameDrawing = $();
            }

            // clearing a multi selection
            if (self.isMultiSelectionSupported() && self.isMultiSelection()) {
                self.clearMultiSelection();
            }

            app.getView().clearVisibleDrawingAnchor();
        };

        /**
         * Clearing the complete current selection without setting a new selection.
         *
         * Info: This is only valid in OX Presentation. It happens, if the last slide
         * in the active view was deleted.
         */
        this.setEmptySelection = function () {
            startPosition = [];
            endPosition = [];
            backwards = false;
            isBackwardCursor = false;
            selectedDrawing = $();
            selectedTextFrameDrawing = $();
            selectedTable = $();
            cellRangeSelected = false;

            self.getClipboardNode().text('\xa0').toggle(true).focus(); // TODO is this necessary here?
            self.trigger('change');
        };

        /**
         * Return the selection object to send it to the server
         * @returns {Array<Object>}  { type, start, end }
         */
        this.getRemoteSelectionsObject = function () {
            var selections = [];
            if (self.isMultiSelectionSupported() && self.isMultiSelection()) {
                self.getMultiSelection().forEach(function (drawing) {
                    selections.push({
                        type: 'drawing',
                        start: drawing.startPosition,
                        end: drawing.endPosition
                    });
                });
            } else {
                selections.push({
                    type: self.getSelectionType(),
                    start: self.getStartPosition(),
                    end: self.getEndPosition()
                });
            }
            return selections;
        };

        /**
         * Checks which is the correct root node of given node, and sets it.
         * Three types of root node: comment root node, header/footer, and page root node.
         * Also check bugs #42671 and #45752.
         *
         * @param {Node|jQuery} node
         */
        this.swichToValidRootNode = function (node) {
            var rootNode;

            if (DOM.isFieldInsideComment(node)) {
                if (app.getModel().isHeaderFooterEditState()) {
                    app.getModel().getPageLayout().leaveHeaderFooterEditMode();
                }
                rootNode = DOM.getSurroundingCommentNode(node);
                app.getModel().getCommentLayer().activateCommentNode(rootNode, self, { deactivate: true });
            } else if (DOM.isMarginalNode(node)) {
                rootNode = DOM.getClosestMarginalTargetNode(node);
                if (app.getModel().isHeaderFooterEditState() && app.getModel().getActiveTarget() !== DOM.getTargetContainerId(rootNode)) {
                    app.getModel().getPageLayout().leaveHeaderFooterEditMode();
                }
                if (!app.getModel().isHeaderFooterEditState()) {
                    app.getModel().getPageLayout().enterHeaderFooterEditMode(rootNode);
                }
                self.setNewRootNode(rootNode);
            } else {
                if (app.getModel().isHeaderFooterEditState()) {
                    app.getModel().getPageLayout().leaveHeaderFooterEditMode();
                }
                rootNode = app.getModel().getNode();
                app.getModel().setActiveTarget();
                self.setNewRootNode(rootNode);
            }
        };

        // initialization -----------------------------------------------------

        // lazy initialization with access to the application model and view
        app.onInit(function () {
            clipboardNode = app.getView().createClipboardNode();

            self.listenTo(app.getView(), 'change:zoom', function () {
                var drawingNodes = null;

                if (self.isMultiSelectionSupported() && self.isMultiSelection()) {
                    drawingNodes = self.getAllSelectedDrawingNodes();
                } else {
                    drawingNodes = self.getAnyDrawingSelection();
                }

                if (drawingNodes && drawingNodes.length) {
                    DrawingFrame.updateScaleHandles(drawingNodes, { scaleHandles: (100 / app.getView().getZoomFactor()) });
                }
            });
        });

        // Bug 26283: Overwrite the focus() method of the passed editor root
        // node. When focusing the editor, the text selection needs to be
        // restored first, or the internal clipboard node will be focused,
        // depending on the current selection type. By overwriting the DOM
        // focus() method, all focus attempts will be covered, also global
        // focus traveling with the F6 key triggered by the UI core.
        rootNode[0].focus = function () {
            self.restoreBrowserSelection();
        };

        // destroy all class members on destruction
        this.registerDestructor(function () {
            // clear clipboard node and remove it from DOM
            if (clipboardNode) {
                clearClipboardNode();
                clipboardNode.remove();
            }
            // restore original focus() method of the root node DOM element (prevent leaks)
            rootNode[0].focus = originalFocusMethod;
        });

    } // class Selection

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: Selection });

});
