/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/hyperlinkmixin', [
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/components/hyperlink/hyperlink',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/view/textdialogs'
], function (AttributeUtils, Hyperlink, DOM, Operations, Position, Dialogs) {

    'use strict';

    // mix-in class HyperlinkMixin ======================================

    /**
     * A mix-in class for the hyper link handling in the document model.
     *
     * @constructor
     *
     * @param {EditApplication} app
     *  The application instance.
     */
    function HyperlinkMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        // public methods -----------------------------------------------------

        /**
         * Inserting a hyper link during pasting.
         *
         * @param {String} url
         *  The URL of the hyper link.
         *
         * @param {String} text
         *  The text for the hyper link.
         *
         * @param {Number[]} [dropPosition]
         *  An optional logical position at which the text will be inserted. If not specified
         *  the current selection is used.
         */
        this.insertHyperlinkDirect = function (url, text, dropPosition) {

            var generator = self.createOperationsGenerator(),
                hyperlinkStyleId = self.useSlideMode() ? null : self.getDefaultUIHyperlinkStylesheet();

            if (url && url.length > 0) {

                self.getUndoManager().enterUndoGroup(function () {

                    // helper function to insert hyper link after selected content is removed
                    function doInsertHyperlink() {

                        var newText = text || url,
                            // created operation
                            newOperation = null,
                            // target string property of operation
                            target = self.getActiveTarget(),
                            // the logical start and end positions
                            start = null, end = null,
                            // the selection object
                            selection = self.getSelection();

                        // reading start and end position (after calling deleteSelected())
                        start = dropPosition ? dropPosition.start : selection.getStartPosition();

                        // insert new text
                        self.doCheckImplicitParagraph(start);
                        newOperation = { text: newText, start: _.clone(start) };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.TEXT_INSERT, newOperation);

                        // Calculate end position of new text
                        // will be used for setAttributes operation
                        end = Position.increaseLastIndex(start, newText.length);

                        if (self.getCharacterStyles().isDirty(hyperlinkStyleId)) {
                            // insert hyper link style to document
                            self.generateInsertStyleOp(generator, 'character', hyperlinkStyleId);
                        }

                        newOperation = {
                            attrs: { styleId: hyperlinkStyleId, character: { url: url } },
                            start: _.clone(start),
                            end: _.clone(end)
                        };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                        // apply all collected operations
                        self.applyOperations(generator);
                    }

                    // delete selected range (this creates new operations)
                    if (self.getSelection().hasRange()) {
                        return self.deleteSelected()
                        .done(function () {
                            doInsertHyperlink();
                        });
                    }

                    doInsertHyperlink();
                    return $.when();
                });
            }
        };

        /**
         * Shows the hyper link dialog for the current selection, and returns a
         * promise that allows to wait for the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         */
        this.insertHyperlinkDialog = function () {

            var // the selection object
                selection = self.getSelection(),
                // the generator object for the operations
                generator = self.createOperationsGenerator(),
                text = '', url = '',
                startPos = null,
                start = selection.getStartPosition(),
                end = selection.getEndPosition(),
                // target string for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // currently active root node
                activeRootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null,
                // the dialog promise
                promise = null,
                // a modified selection object
                newSelection = null,
                // whether a specified selection was already expanded
                alreadyExpanded = false;

            // helper function to expand the given selection
            function expandSelection() {
                newSelection = Hyperlink.findSelectionRange(self, selection);
                if (newSelection.start !== null && newSelection.end !== null) {
                    startPos = selection.getStartPosition();
                    start[start.length - 1] = newSelection.start;
                    end[end.length - 1] = newSelection.end;
                    selection.setTextSelection(start, end); // this also handles range markers!
                    start = selection.getStartPosition(); // reload start position, to guarantee range marker handling (42095)
                    end = selection.getEndPosition();     // reload end position, to guarantee range marker handling (42095)
                }
            }

            // helper function to expand selection range inside complex fields, if they have already an URL specified.
            function expandRangeInsideComplexField() {

                var // the start point at the selections start position
                    startPoint = Position.getDOMPosition(activeRootNode, selection.getStartPosition()),
                    // the explicit attributes for the text span at the start position
                    charAttrs = null;

                if (startPoint && startPoint.node && startPoint.node.parentNode && DOM.isTextSpan(startPoint.node.parentNode) && DOM.isInsideComplexFieldRange(startPoint.node.parentNode)) {

                    // further check, if the text span also already has an url defined
                    charAttrs = AttributeUtils.getExplicitAttributes(startPoint.node.parentNode);

                    if (charAttrs && charAttrs.character && charAttrs.character.url) {
                        selection.setTextSelection(selection.getStartPosition());  // removing selection range
                        expandSelection();
                    }
                }
            }

            if (!selection.hasRange()) {
                expandSelection();
                alreadyExpanded = true;
            }

            // avoiding that hyper links that expand over complex fields are splitted inside the complex
            // field. This can happen, if the new selection is only a part inside a complex field. In this
            // case the new hyper link has to expand to the range of the old hyper link.
            // The filter cannot handle the case, in which a hyper link starts outside of a complex field
            // and ends inside a complex field.
            if (selection.hasRange() && !alreadyExpanded) { expandRangeInsideComplexField(); }

            if (!self.hasEnclosingParagraph()) {
                return $.Deferred().reject();
            }

            // use range to retrieve text and possible url
            if (selection.hasRange()) {

                // Find out the text/url of the selected text to provide them to the
                // hyper link dialog
                selection.iterateNodes(function (node, pos, start, length) {
                    if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                        var nodeText = $(node).text();
                        if (nodeText) {
                            text = text.concat(nodeText.slice(start, start + length));
                        }
                        if (url.length === 0) {
                            var charAttributes = self.getCharacterStyles().getElementAttributes(node).character;
                            if (charAttributes.url && charAttributes.url.length > 0) {
                                url = charAttributes.url;
                            }
                        }
                    }
                });
            }

            // show hyperlink dialog
            promise = new Dialogs.HyperlinkDialog(app.getView(), url, text).show();

            return promise.done(function (data) {
                // set url to selected text
                var hyperlinkStyleId = self.useSlideMode() ? null : self.getDefaultUIHyperlinkStylesheet(),
                    url = data.url,
                    oldAttrs = null,
                    attrs = null,
                    hyperlinkNode = null,
                    insertedLink = false,
                    // the change track object
                    changeTrack = self.getChangeTrack();

                self.getUndoManager().enterUndoGroup(function () {

                    if (data.url === null && data.text === null) {
                        // remove hyper link
                        // setAttribute uses a closed range therefore -1
                        attrs = Hyperlink.CLEAR_ATTRIBUTES;

                        if (changeTrack.isActiveChangeTracking()) {
                            hyperlinkNode = Position.getDOMPosition(activeRootNode, selection.getEndPosition()).node;
                            if (hyperlinkNode && (hyperlinkNode.nodeType === 3)) { hyperlinkNode = hyperlinkNode.parentNode; }
                            // Expanding operation for change tracking with old explicit attributes
                            oldAttrs = changeTrack.getOldNodeAttributes(hyperlinkNode);
                            // adding the old attributes, author and date for change tracking
                            if (oldAttrs) {
                                oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                                attrs.changes = { modified: oldAttrs };
                            }
                        }
                    } else {

                        // if the data.text is a emtpy string, a default value is set (fix for Bug 40313)
                        if (data.text === '') {
                            data.text = data.url;
                        }

                        // insert/change hyperlink
                        if (data.text !== text) {

                            // text has been changed
                            if (selection.hasRange()) {
                                self.deleteSelected();
                                start = selection.getStartPosition();
                            }

                            if (changeTrack.isActiveChangeTracking()) {
                                attrs = attrs || {};
                                attrs.changes = { inserted: changeTrack.getChangeTrackInfo() };
                                insertedLink = true;
                            }

                            // insert new text
                            self.doCheckImplicitParagraph(start);
                            newOperation = { text: data.text, start: _.clone(start), attrs: attrs };
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.TEXT_INSERT, newOperation);

                            // Calculate end position of new text, will be used for setAttributes operation
                            end = Position.increaseLastIndex(start, data.text.length);
                        }

                        if (self.getCharacterStyles().isDirty(hyperlinkStyleId)) {
                            // insert hyperlink style to document
                            self.generateInsertStyleOp(generator, 'character', hyperlinkStyleId);
                        }

                        attrs = { styleId: hyperlinkStyleId, character: { url: url } };

                        if (changeTrack.isActiveChangeTracking() && !insertedLink) {
                            hyperlinkNode = Position.getDOMPosition(activeRootNode, end).node;
                            if (hyperlinkNode && (hyperlinkNode.nodeType === 3)) { hyperlinkNode = hyperlinkNode.parentNode; }
                            // Expanding operation for change tracking with old explicit attributes
                            oldAttrs = changeTrack.getOldNodeAttributes(hyperlinkNode);
                            // adding the old attributes, author and date for change tracking
                            if (oldAttrs) {
                                oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                                attrs.changes = { modified: oldAttrs };
                            }
                        }
                    }

                    end[end.length - 1] -= 1;
                    newOperation = {
                        attrs: attrs,
                        start: _.clone(start),
                        end: _.clone(end)
                    };
                    self.extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                    // apply all collected operations
                    self.applyOperations(generator);

                }, self); // enterUndoGroup()

            }).always(function () {
                if (startPos) {
                    selection.setTextSelection(startPos);
                }
            });
        };

        /**
         * Removing a hyper link at the current selection.
         */
        this.removeHyperlink = function () {

            var // the selection object
                selection = self.getSelection(),
                // the generator object for the operations
                generator = self.createOperationsGenerator(),
                startPos = null,
                start = selection.getStartPosition(),
                end = selection.getEndPosition(),
                oldAttrs = null,
                attrs = null,
                hyperlinkNode = null,
                // target string for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // currently active root node
                activeRootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null,
                // a modified selection object
                newSelection = null,
                // the change track object
                changeTrack = self.getChangeTrack();

            if (!selection.hasRange()) {
                newSelection = Hyperlink.findSelectionRange(this, selection);
                if (newSelection.start !== null && newSelection.end !== null) {
                    startPos = selection.getStartPosition();
                    start[start.length - 1] = newSelection.start;
                    end[end.length - 1] = newSelection.end;
                    selection.setTextSelection(start, end);
                }
            }

            if (selection.hasRange() && self.hasEnclosingParagraph()) {

                attrs = Hyperlink.CLEAR_ATTRIBUTES;

                if (changeTrack.isActiveChangeTracking()) {
                    hyperlinkNode = Position.getDOMPosition(activeRootNode, selection.getEndPosition()).node;
                    if (hyperlinkNode && (hyperlinkNode.nodeType === 3)) { hyperlinkNode = hyperlinkNode.parentNode; }
                    // Expanding operation for change tracking with old explicit attributes
                    oldAttrs = changeTrack.getOldNodeAttributes(hyperlinkNode);
                    // adding the old attributes, author and date for change tracking
                    if (oldAttrs) {
                        oldAttrs = _.extend(oldAttrs, changeTrack.getChangeTrackInfo());
                        attrs.changes = { modified: oldAttrs };
                    }
                }

                // remove hyperlink
                // setAttribute uses a closed range therefore -1
                end[end.length - 1] -= 1;

                newOperation = {
                    attrs: attrs,
                    start: _.clone(start),
                    end: _.clone(end)
                };
                self.extendPropertiesWithTarget(newOperation, target);
                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                // apply the operations (undo group is created automatically)
                self.applyOperations(generator);

                self.executeDelayed(function () {
                    app.getView().grabFocus();
                    if (startPos) {
                        selection.setTextSelection(startPos);
                    }
                }, undefined, 'Text: removeHyperLink');
            }
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class HyperlinkMixin

    // exports ================================================================

    return HyperlinkMixin;

});
