/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/deleteoperationmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/components/table/table',
    'io.ox/office/textframework/format/tablestyles',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/snapshot',
    'io.ox/office/textframework/utils/textutils',
    'gettext!io.ox/office/textframework/main'
], function (DrawingFrame, AttributeUtils, Table, TableStyles, DOM, Position, Operations, Snapshot, Utils, gt) {

    'use strict';

    // mix-in class DeleteOperationMixin ======================================

    /**
     * A mix-in class for the document model class providing the operation
     * handling for deleting all kind of content used in a presentation
     * and text document.
     *
     * @constructor
     *
     * @param {EditApplication} app
     *  The application instance.
     */
    function DeleteOperationMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Checking whether a node contains an element, that cannot be restored
         * in undo action. This are tables with exceeded size or drawings that
         * are not of type 'image'.
         *
         * @param {Node|jQuery|Number[]} node
         *  The node, whose descendants are investigated. For convenience reasons
         *  this can also be a logical position that is transformed into a node
         *  automatically.
         *
         * @param {Number} [startOffset]
         *  An optional offset that can be used to reduce the search range inside
         *  the specified node.
         *
         * @param {Number} [endOffset]
         *  An optional offset that can be used to reduce the search range inside
         *  the specified node.
         *
         * @returns {Boolean}
         *  Returns true, if the node contains an unrestorable descendant,
         *  otherwise false.
         */
        function containsUnrestorableElements(node, startOffset, endOffset) {

            var // whether the node contains an unrestorable descendant
                contains = false,
                // a dom point calculated from logical position
                domPos = null,
                // a locally defined helper node
                searchNode = null,
                // the logical position of a non-image drawing
                drawingPos = null;

            // no unrestorable elements in slide mode
            if (self.useSlideMode()) { return contains; }

            // supporting for convenience also logical positions, that are automatically
            // converted into dom nodes
            if (_.isArray(node)) {
                domPos = Position.getDOMPosition(self.getCurrentRootNode(), node, true);
                if (domPos && domPos.node) { searchNode = domPos.node; }
            } else {
                searchNode = node;
            }

            if (searchNode) {
                if ($(searchNode).find(DOM.TABLE_SIZE_EXCEEDED_NODE_SELECTOR).length > 0) {
                    contains = true;
                } else {
                    // checking if the table or paragraph contains drawings, that are not of type 'image'
                    // or if the table contains another table with exceeded size
                    $(DrawingFrame.NODE_SELECTOR + ', ' + DOM.DRAWINGPLACEHOLDER_NODE_SELECTOR, searchNode).each(function (i, drawing) {

                        if (DOM.isDrawingPlaceHolderNode(drawing)) { drawing = DOM.getDrawingPlaceHolderNode(drawing); }

                        if (DOM.isUnrestorableDrawingNode(drawing)) {
                            // if only a part of a paragraph is deleted, it has to be checked,
                            // if the non-image and non-textframe drawing is inside this selected part.
                            if (_.isNumber(startOffset) && (_.isNumber(endOffset))) {
                                // the logical position of the non-image drawing
                                drawingPos = Position.getOxoPosition(self.getCurrentRootNode(), drawing, 0);
                                if (Position.isNodePositionInsideRange(drawingPos, startOffset, endOffset)) {
                                    contains = true;
                                }
                            } else {
                                contains = true;
                            }
                        }
                    });
                }
                // if node contains special page number fields in header and footer restore original state before delete
                self.getFieldManager().checkRestoringSpecialFields(searchNode, startOffset, endOffset);
            }

            return contains;
        }

        /**
         * Checking whether there can be an undo for the delete operation.
         * Undo is not possible, if a table with exceeded size is removed,
         * or if a drawing, that is not of type image is removed.
         *
         * @param {String} type
         *  The type of the node to be removed. This can be 'text', 'paragraph',
         *  'cell', 'row' and table.
         *
         * @param {Node|jQuery} node
         *  The node to be removed.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  deleted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  deleted. This is only relevant for type 'text'.
         *
         * @param {String[]} [target]
         *  If element is marginal, target is needed in group with
         *  start and end to determine root container.
         */
        function checkDisableUndoStack(type, node, start, end, target) {

            var // info about the parent paragraph or table node
                position = null, paragraph = null,
                // last index in start and end position
                startOffset = 0, endOffset = 0;

            // deleting the different types:
            switch (type) {

                case 'text':
                    // get paragraph node from start position
                    if (!_.isArray(start) || (start.length === 0)) {
                        Utils.warn('Editor.disableUndoStack(): missing start position');
                        return false;
                    }
                    position = start.slice(0, -1);
                    paragraph = Position.getParagraphElement(self.getCurrentRootNode(target), position);
                    if (!paragraph) {
                        Utils.warn('Editor.disableUndoStack(): no paragraph found at position ' + JSON.stringify(position));
                        return false;
                    }

                    // validate end position
                    if (_.isArray(end) && !Position.hasSameParentComponent(start, end)) {
                        Utils.warn('Editor.disableUndoStack(): range not in same paragraph');
                        return false;
                    }

                    // start and end offset in paragraph
                    startOffset = _.last(start);
                    endOffset = _.isArray(end) ? _.last(end) : startOffset;
                    if (endOffset === -1) { endOffset = undefined; }

                    // visit all covered child nodes (iterator allows to remove the visited node)
                    Position.iterateParagraphChildNodes(paragraph, function (localnode) {

                        if (DOM.isDrawingPlaceHolderNode(localnode)) { localnode = DOM.getDrawingPlaceHolderNode(localnode); }

                        if (DOM.isUnrestorableDrawingNode(localnode)) {
                            self.setDeleteUndoStack(true);
                        }
                        if (DOM.isSpecialField(localnode)) {
                            self.getFieldManager().restoreSpecialField(localnode);
                        }
                    }, undefined, { start: startOffset, end: endOffset });
                    break;

                case 'paragraph':
                    if (containsUnrestorableElements(node)) {
                        self.setDeleteUndoStack(true);
                    }
                    break;

                case 'cell':
                    if (containsUnrestorableElements(node)) {
                        self.setDeleteUndoStack(true);
                    }
                    break;

                case 'row':
                    if (containsUnrestorableElements(node)) {
                        self.setDeleteUndoStack(true);
                    }
                    break;

                case 'table':
                    if (DOM.isExceededSizeTableNode(node) || containsUnrestorableElements(node)) {
                        self.setDeleteUndoStack(true);
                    }
                    break;
            }
        }

        /**
         * Collecting all list style IDs of the paragraphs inside the specified elements.
         *
         * @param {Array|HTMLElement|jQuery} nodes
         *  The collection of nodes, that will be checked for paragraphs with list styles.
         *  This can be an array with html nodes or jQuery nodes. It can also be a html node
         *  or a jQuery object not included into an array. Above all nodes is iterated.
         *
         * @returns {[Array]}
         *  An array containing the list style IDs of the paragraphs inside the specified elements.
         */
        function getListStyleIDsOfParagraphs(nodes) {

            var listStyleIDs = null;

            if ((!nodes) || (_.isArray(nodes) && _.isEmpty(nodes))) { return listStyleIDs; }

            _.chain(nodes).getArray().any(function (oneNode) {
                $(oneNode).each(function (index, node) {
                    $(node).find(DOM.PARAGRAPH_NODE_SELECTOR).each(function (i, paragraph) {
                        // checking paragraph attributes for list styles
                        var paraAttrs = AttributeUtils.getExplicitAttributes(paragraph);
                        // updating lists, if required
                        if (self.isListStyleParagraph(null, paraAttrs) && paraAttrs && paraAttrs.paragraph) {
                            listStyleIDs = listStyleIDs || [];
                            if (!_.contains(listStyleIDs, paraAttrs.paragraph.listStyleId)) { listStyleIDs.push(paraAttrs.paragraph.listStyleId); }
                        }
                    });
                });
            });

            return listStyleIDs;
        }

        /**
         * Check, whether a given collection of nodes contain paragraphs with list
         * styles.
         *
         * @param {Array|HTMLElement|jQuery} nodes
         *  The collection of nodes, that will be checked for paragraphs with list styles.
         *  This can be an array with html nodes or jQuery nodes. It can also be a html node
         *  or a jQuery object not included into an array. Above all nodes is iterated.
         *
         * @returns {Boolean}
         *  Whether at least one of the specified nodes contains at least one list paragraph node.
         */
        function elementsContainListStyleParagraph(nodes) {

            var isListParagraph = false;

            if ((!nodes) || (_.isArray(nodes) && _.isEmpty(nodes))) { return isListParagraph; }

            _.chain(nodes).getArray().any(function (oneNode) {
                $(oneNode).each(function (index, node) {
                    $(node).find(DOM.PARAGRAPH_NODE_SELECTOR).each(function (i, paragraph) {
                        if (self.isListStyleParagraph(paragraph)) {
                            isListParagraph = true;
                            return !isListParagraph; // stopping 'each' iteration, if list paragraph was found
                        }
                    });
                    return !isListParagraph; // stopping 'each' iteration, if list paragraph was found
                });
                return isListParagraph; // stopping 'any' iteration, if list paragraph was found
            });

            return isListParagraph;
        }

        /**
         * Returns the type of the attributes supported by the passed DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose associated attribute type will be returned.
         *  If this object is a jQuery collection, returns its first node.
         *
         * @returns {String}
         *  A type for the specified element.
         */
        function resolveElementType(element) {

            var // the element, as jQuery object
                $element = $(element),
                // the resulting style type
                type = null;

            if (DOM.isPartOfParagraph($element)) {
                type = 'text';
            } else if (DOM.isParagraphNode($element)) {
                type = 'paragraph';
            } else if (DOM.isTableNode($element)) {
                type = 'table';
            } else if ($element.is('tr')) {
                type = 'row';
            } else if ($element.is('td')) {
                type = 'cell';
            } else {
                Utils.warn('Editor.resolveElementType(): unsupported element');
            }

            return type;
        }

        /**
         * Returns a valid text position at the passed component position
         * (paragraph or table). Used to calculate the new text cursor position
         * after deleting a component.
         *
         * @returns {Number[]}
         *  A valid logical text position.
         */
        function getValidTextPosition(position) {

            var // container root node of table
                rootNode = self.getCurrentRootNode(),
                // a logical text position
                textPosition = Position.getFirstPositionInParagraph(rootNode, position);

            // fall-back to last position in document (e.g.: last table deleted)
            if (!textPosition) {
                textPosition = Position.getLastPositionInParagraph(rootNode, [rootNode[0].childNodes.length - 1]);
            }

            return textPosition;
        }

        /**
         * Deletes a table at the specified position.
         *
         * @param {Number[]} position
         *  The logical position of the table to be deleted.
         *
         * @param {String} target
         *  Id of the container node, where operation is applied.
         *
         * @returns {Boolean}
         *  TRUE if the function has been processed successfully
         *  otherwise FALSE.
         */
        function implDeleteTable(position, target) {

            var // the target node
                rootNode = self.getRootNode(target),
                // the logical position of the table
                tablePosition = Position.getLastPositionFromPositionByNodeName(rootNode, position, DOM.TABLE_NODE_SELECTOR),
                // the table node
                tableNode = Position.getTableElement(rootNode, tablePosition),
                // whether there are list paragraphs inside the table
                containListStyleParagraph = false,
                // the list style IDs in the table
                allListStyleIDs = null;

            if (tableNode) {
                containListStyleParagraph = elementsContainListStyleParagraph(tableNode); // needs to be checked before removing the table
                if (containListStyleParagraph) { allListStyleIDs = getListStyleIDsOfParagraphs(tableNode); }
                $(tableNode).remove();
                self.setLastOperationEnd(getValidTextPosition(tablePosition));
            } else {
                Utils.warning('Editor.implDeleteTable(): not tableNode found ' + JSON.stringify(position));
                return false;
            }

            // the deleted paragraphs can be part of a list, update all lists
            if (containListStyleParagraph) {
                self.updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: allListStyleIDs ? _.uniq(allListStyleIDs) : null, listLevel: null });
            }

            return true;
        }

        /**
         * Deletes table row(s) at the specified position.
         *
         * @param {Number[]} position
         *  The logical position of the table.
         *
         * @param {Number} startRow
         *  The index of the first row to be deleted.
         *
         * @param {Number} endRow
         *  The index of the last row to be deleted.
         *
         * @param {String} target
         *  The id of the container node, where operation is applied.
         *
         * @returns {Boolean}
         *  Whether the rows could be deleted.
         */
        function implDeleteRows(pos, startRow, endRow, target) {

            var // the logical position
                localPosition = _.copy(pos, true),
                // the root node specified by the target string
                rootNode = self.getRootNode(target),
                // whether there are list paragraphs inside the deleted row(s)
                containListStyleParagraph = false,
                // the list style IDs in the deleted row(s)
                allListStyleIDs = null;

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                Utils.warning('Editor.implDeleteRows(): position not in table ' + JSON.stringify(pos));
                return false;
            }

            var table = Position.getDOMPosition(rootNode, localPosition).node,
                rowNodes = DOM.getTableRows(table).slice(startRow, endRow + 1);

            containListStyleParagraph = elementsContainListStyleParagraph(rowNodes); // needs to be checked before removing the row(s)
            if (containListStyleParagraph) { allListStyleIDs = getListStyleIDsOfParagraphs(rowNodes); }
            rowNodes.remove();

            if (self.isGUITriggeredOperation()) {  // called from 'deleteRows' -> temporary branding for table
                $(table).data('gui', 'remove');
                // Performance: Simple tables do not need new formatting.
                // Tables with styles need to be updated below the removed row. Also the
                // row above need to be updated, because it may have become the final row.

                // Bugfix 44230: introduced a check to prevent wrong values for '.data('reducedTableFormatting')' in updateRowFormatting(), when inserting/deleting
                // rows very quickly via button, so that more than one rows are inserted/deleted before the debounced updateRowFormatting() is executed
                if (_.isNumber($(table).data('reducedTableFormatting'))) {
                    $(table).data('reducedTableFormatting', Math.min((startRow > 0 ? startRow - 1 : 0), $(table).data('reducedTableFormatting')));
                } else {
                    $(table).data('reducedTableFormatting', startRow > 0 ? startRow - 1 : 0);
                }
            }

            // undo also needs table refresh for right border in Firefox (32374)
            if (self.isUndoRedoRunning()  && _.browser.Firefox) { $(table).data('undoRedoRunning', true); }

            // Setting cursor
            var lastRow = Table.getRowCount(table) - 1;

            if (lastRow >= 0) {
                if (endRow > lastRow) {
                    endRow = lastRow;
                }
                localPosition.push(endRow);
                localPosition.push(0);  // using first cell in row
                localPosition.push(0);  // using first paragraph or table
                localPosition = Position.getFirstPositionInParagraph(rootNode, localPosition);

                // recalculating the attributes of the table cells
                if (self.requiresElementFormattingUpdate()) {
                    self.implTableChanged(table);
                }

                self.setLastOperationEnd(localPosition);
            }

            // the deleted paragraphs can be part of a list, update all lists
            if (containListStyleParagraph) {
                self.updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: allListStyleIDs ? _.uniq(allListStyleIDs) : null, listLevel: null });
            }

            return true;
        }

        /**
         * Delete table cells at the specified position.
         *
         * @param {Array<Number>} pos
         *  The logical position of the table to be deleted.
         *
         * @param {Array<Number>} start
         *  The logical start position of the cell range.
         *
         * @param {Array<Number>} end
         *  The logical end position of the cell range.
         *
         * @param {String} target
         *  Id of the container node, where operation is applied.
         *
         * @returns {Boolean}
         *  TRUE if the function has been processed successfully,
         *  otherwise FALSE.
         */
        function implDeleteCells(pos, start, end, target) {

            var localPosition = _.copy(pos, true),
                tableRowDomPos = null,
                row = null,
                table = null,
                maxCell = 0,
                cellNodes = null,
                allCellNodes = [],
                rootNode = self.getRootNode(target),
                // whether there are list paragraphs inside the deleted cell(s)
                containListStyleParagraph = false,
                // the list style IDs in the deleted row(s)
                allListStyleIDs = null;

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                Utils.warn('Editor.implDeleteCells(): position not in table ' + JSON.stringify(pos));
                return false;
            }

            tableRowDomPos = Position.getDOMPosition(rootNode, localPosition);

            if (tableRowDomPos) {
                row = tableRowDomPos.node;
            }

            if (row) {

                maxCell = $(row).children().length - 1;

                if (start <= maxCell) {

                    if (end > maxCell) {
                        cellNodes = $(row).children().slice(start);

                    } else {
                        cellNodes = $(row).children().slice(start, end + 1);
                    }

                    if (elementsContainListStyleParagraph(cellNodes)) {
                        containListStyleParagraph = true;
                        allListStyleIDs = allListStyleIDs || [];
                        allListStyleIDs = allListStyleIDs.concat(getListStyleIDsOfParagraphs(cellNodes));
                    }

                    cellNodes.remove(); // removing all following cells
                    allCellNodes.push(cellNodes);
                }
            }

            // setting cursor position
            localPosition.push(0);
            localPosition.push(0);
            localPosition.push(0);

            if (row && self.requiresElementFormattingUpdate()) {
                table = $(row).closest('table');
                // undo of insertColumn also needs table refresh for right border in Firefox
                if (self.isUndoRedoRunning() && _.browser.Firefox) { table.data('undoRedoRunning', true); }
                self.implTableChanged(table);
            }

            self.setLastOperationEnd(localPosition);

            // the deleted paragraphs can be part of a list, update all lists
            if (containListStyleParagraph) {
                self.updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: allListStyleIDs ? _.uniq(allListStyleIDs) : null, listLevel: null });
            }

            return true;
        }

        /**
         * Deletes table column(s) at the specified position.
         *
         * @param {Number[]} start
         *  The logical position of the table.
         *
         * @param {Number} startGrid
         *  The index of the first column to be deleted.
         *
         * @param {Number} endGrid
         *  The index of the last column to be deleted.
         *
         * @param {String} target
         *  The id of the container node, where operation is applied.
         */
        function implDeleteColumns(start, startGrid, endGrid, target) {

            var localPosition = _.copy(start, true),
                cellNodes = null,
                allCellNodes = [],
                rootNode = self.getRootNode(target),
                // whether there are list paragraphs inside the deleted cell(s)
                containListStyleParagraph = false,
                // the list style IDs in the deleted row(s)
                allListStyleIDs = null,
                // whether the document is loaded completely
                documentLoaded = self.isImportFinished();

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                return;
            }

            var table = Position.getDOMPosition(rootNode, localPosition).node,
                allRows = DOM.getTableRows(table),
                endColInFirstRow = -1,
                lastColInFirstRow = 0;

            if (self.isGUITriggeredOperation()) {  // called from 'deleteColumns' -> temporary branding for table
                $(table).data('gui', 'remove');
            }

            allRows.each(
                function (i, row) {
                    var startCol = Table.getCellPositionFromGridPosition(row, startGrid, false, documentLoaded),
                        endCol = Table.getCellPositionFromGridPosition(row, endGrid, false, documentLoaded);

                    if ((i === 0) && (endCol !== -1)) {
                        endColInFirstRow = endCol;
                    }

                    if (startCol !== -1) {  // do nothing if startCol is out of range for this row

                        if (endCol === -1) {
                            // checking whether undo of operations is possible and remove cells
                            cellNodes = $(row).children().slice(startCol);
                        } else {
                            // checking whether undo of operations is possible and remove all cells in the range
                            cellNodes = $(row).children().slice(startCol, endCol + 1);
                        }

                        if (elementsContainListStyleParagraph(cellNodes)) {
                            containListStyleParagraph = true;
                            allListStyleIDs = allListStyleIDs || [];
                            allListStyleIDs = allListStyleIDs.concat(getListStyleIDsOfParagraphs(cellNodes));
                        }

                        cellNodes.each(function (i, cell) {
                            checkDisableUndoStack('cell', cell);
                            // updating all models, for drawings, comments, range markers, ...
                            self.updateCollectionModels(cell);
                        }).remove();  // removing cell nodes
                        allCellNodes.push(cellNodes);
                    }
                }
            );

            // Setting cursor
            lastColInFirstRow = DOM.getTableRows(table).first().children().length - 1;

            if ((endColInFirstRow > lastColInFirstRow) || (endColInFirstRow === -1)) {
                endColInFirstRow = lastColInFirstRow;
            }
            localPosition.push(0);
            localPosition.push(endColInFirstRow);
            localPosition.push(0);
            localPosition.push(0);

            // delete undo stack immediately if this is necessary and not a part of an undo group
            if (!self.isInUndoGroup() && self.deleteUndoStack()) {
                self.getUndoManager().clearUndoActions();
                self.setDeleteUndoStack(false);
            }

            // recalculating the attributes of the table cells
            if (self.requiresElementFormattingUpdate()) {
                self.implTableChanged(table);
            }

            // the deleted paragraphs can be part of a list, update all lists
            if (containListStyleParagraph) {
                self.updateListsDebounced({ useSelectedListStyleIDs: true, listStyleId: allListStyleIDs ? _.uniq(allListStyleIDs) : null, listLevel: null });
            }

            self.setLastOperationEnd(localPosition);
        }

        /**
         * Deletes text at the specified position.
         *
         * @param {Number[]} startPosition
         *  The logical start position.
         *
         * @param {Number[]} endPosition
         *  The logical end position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.allowEmptyResult=false]
         *      In internal calls (from other operations except delete), it can be allowed,
         *      that there is nothing to delete.
         *  @param {Boolean} [options.keepDrawingLayer=false]
         *      Whether the drawings in the drawing layer or comments in the comment layer
         *      shall not be removed -> this is necessary after splitting a paragraph.
         *
         * @param {String} target
         *  The id of the container node, where operation is applied.
         *
         * @returns {Boolean}
         *  Whether the text was deleted.
         */
        function implDeleteText(startPosition, endPosition, options, target) {

            var // info about the parent paragraph node
                position = null, paragraph = null,
                // last index in start and end position
                startOffset = 0, endOffset = 0,
                // next sibling text span of last visited child node
                nextTextSpan = null,
                // a counter for the removed elements
                removeCounter = 0,
                // the number of text positions to be removed
                removePositions = 0,
                // in internal calls (from other operations except delete), it can be allowed,
                // that there is nothing to delete
                allowEmptyResult = Utils.getBooleanOption(options, 'allowEmptyResult', false),
                // whether the drawings in the drawing layer or comments in the comment layer shall not be removed
                // -> this is necessary after splitting a paragraph
                keepDrawingLayer = Utils.getBooleanOption(options, 'keepDrawingLayer', false),
                // whether a character was deleted within this function
                noCharDeleted = false,
                // root node used for calculating position - if target is present - root node is header or footer
                rootNode = self.getRootNode(target),
                // a helper position
                lastOperationEndLocal = null;

            // get paragraph node from start position
            if (!_.isArray(startPosition) || (startPosition.length < 2)) {
                Utils.warn('Editor.implDeleteText(): missing start position');
                return false;
            }
            position = startPosition.slice(0, -1);
            paragraph = (self.useParagraphCache() && self.getParagraphCache()) || Position.getParagraphElement(rootNode, position);
            if (!paragraph) {
                Utils.warn('Editor.implDeleteText(): no paragraph found at position ' + JSON.stringify(position));
                return false;
            }

            // validate end position
            if (_.isArray(endPosition) && !Position.hasSameParentComponent(startPosition, endPosition)) {
                Utils.warn('Editor.implDeleteText(): range not in same paragraph');
                return false;
            }

            // start and end offset in paragraph
            startOffset = _.last(startPosition);
            endOffset = _.isArray(endPosition) ? _.last(endPosition) : startOffset;
            if (endOffset === -1) { endOffset = undefined; }
            if (endOffset !== undefined) {
                removePositions = endOffset - startOffset + 1;
            }

            // visit all covered child nodes (iterator allows to remove the visited node)
            Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength) {

                var // previous text span of current node
                    prevTextSpan = null,
                    // the field manager object
                    fieldManager = self.getFieldManager(),
                    // the drawing layer object
                    drawingLayer = self.getDrawingLayer();

                // remove preceding position offset node of floating drawing objects
                if (DOM.isFloatingDrawingNode(node)) {
                    $(node).prev(DOM.OFFSET_NODE_SELECTOR).remove();
                }

                // get sibling text spans
                prevTextSpan = DOM.isTextSpan(node.previousSibling) ? node.previousSibling : null;
                nextTextSpan = DOM.isTextSpan(node.nextSibling) ? node.nextSibling : null;

                // fix for msie, where empty text spans migth be removed
                // newly, also in other browsers (new jQuery version?)
                if (DOM.isTextSpanWithoutTextNode(node)) {
                    self.repairEmptyTextNodes(node);
                }
                if ((!prevTextSpan) && (DOM.isTextSpanWithoutTextNode(node.previousSibling))) {
                    prevTextSpan = node.previousSibling;
                    self.repairEmptyTextNodes(prevTextSpan);
                }
                if ((!nextTextSpan) && (DOM.isTextSpanWithoutTextNode(node.nextSibling))) {
                    nextTextSpan = node.previousSibling;
                    self.repairEmptyTextNodes(nextTextSpan);
                }

                // clear text in text spans
                if (DOM.isTextSpan(node)) {

                    // only remove the text span, if it has a sibling text span
                    // (otherwise, it separates other component nodes)
                    if (prevTextSpan || nextTextSpan) {
                        $(node).remove();
                    } else {
                        // remove text, but keep text span element
                        node.firstChild.nodeValue = '';
                        // but remove change track information immediately (38791)
                        if (DOM.isChangeTrackNode(node)) { self.getChangeTrack().updateChangeTrackAttributes($(node), {}); }
                    }
                    removeCounter += nodeLength;
                    return;
                }

                // other component nodes (drawings or text components)
                if (DOM.isTextComponentNode(node) || DrawingFrame.isDrawingFrame(node) || DOM.isCommentPlaceHolderNode(node) || DOM.isRangeMarkerNode(node)) {

                    // if we are removing manualy inserted page break by user, trigger the pagebreak recalculation on document
                    if (DOM.isTextComponentNode(node) && $(node.parentNode).hasClass('manual-page-break')) {
                        $(node.parentNode).removeClass('manual-page-break');
                        self.insertPageBreaks(node.parentNode, DrawingFrame.getClosestTextFrameDrawingNode(node.parentNode));
                    }

                    // node is the image place holder -> remove also the drawing from the drawing layer
                    if (DOM.isDrawingPlaceHolderNode(node)) {
                        drawingLayer.removeFromDrawingLayer(node, { keepDrawingLayer: keepDrawingLayer });
                    } else if (DOM.isAbsoluteParagraphDrawing(node)) {
                        drawingLayer.removeSpaceMakerNode(node);
                    } else if (DOM.isCommentPlaceHolderNode(node)) {
                        self.getCommentLayer().removeFromCommentLayer(node, { keepDrawingLayer: keepDrawingLayer });
                    } else if (DOM.isRangeMarkerNode(node) && !keepDrawingLayer) {
                        self.getRangeMarker().removeRangeMarker(node);
                    } else if (DOM.isComplexFieldNode(node) && !keepDrawingLayer) {
                        fieldManager.removeFromComplexFieldCollection(node);
                        // it is important to clean up class names when deleting cx field
                        fieldManager.cleanUpClassWhenDeleteCx(node);
                    } else if (DOM.isFieldNode(node)) {
                        fieldManager.removeSimpleFieldFromCollection(node);
                    }

                    // safely release image data (see comments in BaseApplication.destroyImageNodes())
                    app.destroyImageNodes(node);

                    // remove the visited node
                    $(node).remove();

                    // remove previous empty sibling text span (not next sibling which would break iteration)
                    if (DOM.isEmptySpan(prevTextSpan) && nextTextSpan) {
                        $(prevTextSpan).remove();
                    }
                    removeCounter += nodeLength;
                    return;
                }

                Utils.error('Editor.implDeleteText(): unknown paragraph child node');
                return Utils.BREAK;

            }, undefined, { start: startOffset, end: endOffset, split: true });

            // removed must be > 0 and it must be the difference from endOffset - startOffset + 1
            // check, if at least one element was removed
            if (!allowEmptyResult && (removeCounter === 0)) {
                Utils.warn('Editor.implDeleteText(): no component found from position: ' + JSON.stringify(startPosition) + ' to ' + JSON.stringify(endPosition));
                return false;
            }

            // check, if the removed part has the correct length
            if (!allowEmptyResult && (removePositions > 0) && (removePositions !== removeCounter)) {
                Utils.warn('Editor.implDeleteText(): incorrect number of removals: Required: ' + removeCounter + ' Done: ' + removeCounter);
                return false;
            }

            // remove next sibling text span after deleted range, if empty,
            // otherwise try to merge with equally formatted preceding text span
            if (nextTextSpan && DOM.isTextSpan(nextTextSpan.previousSibling)) {
                if (DOM.isEmptySpan(nextTextSpan)) {
                    $(nextTextSpan).remove();
                } else {
                    Utils.mergeSiblingTextSpans(nextTextSpan);
                }
            }

            // validate paragraph node, store operation position for cursor position
            if (self.isGUITriggeredOperation() || self.isUndoRedoRunning() || !self.isImportFinished()) {
                self.implParagraphChanged(paragraph);  // Performance and task 30597: Local client can defer attribute setting, Task 30603: deferring also during loading document
            } else {
                self.implParagraphChangedSync($(paragraph));  // Performance and task 30597: Remote client must set attributes synchronously
            }

            lastOperationEndLocal = _.clone(startPosition);
            if (noCharDeleted) {
                if (_.isArray(endPosition) && !_.isEqual(startPosition, endPosition)) { lastOperationEndLocal = _.clone(endPosition); }
                lastOperationEndLocal[lastOperationEndLocal.length - 1] += 1;
            }

            self.setLastOperationEnd(lastOperationEndLocal);

            return true;
        }

        /**
         * Removes a specified element or a text range. The type of the element
         * will be determined from the parameters start and end.
         * specified range. Currently, only single components can be deleted,
         * except for text ranges in a paragraph. A text range can include
         * characters, fields, and drawing objects, but must be contained in a
         * single paragraph.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  deleted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  deleted. Can be omitted, if the end position is equal to the start
         *  position (single component)
         *
         * @param {String} target - optional
         *  If target is existing, rootNode is not editdiv, but currently edited header/footer
         *
         *  @returns {Boolean}
         *   TRUE if the function has been processed successfully, otherwise
         *   FALSE.
         */
        function implDelete(start, end, target) {

            var // node info for start/end position
                startInfo = null, endInfo = null,
                // position description for cells
                rowPosition, startCell, endCell,
                // position description for rows
                tablePosition, startRow, endRow,
                // a new implicit paragraph node
                newParagraph = null,
                // a temporary helper position
                localPosition = null,
                // result of function (default true)
                result = true,
                // starting point for inserting page breaks downwards
                position = start.length > 1 ? start.slice(0, 1) : start.slice(0),
                // whehter the removed node is a slide node
                isSlideNode = false,
                // the parent node of the removed node
                parentNode = null,
                // used for pagebreaks
                currentElement = null,
                // the height of the paragraph before deleting content
                prevHeight = null,
                // the space maker node and the paragraph containing the space maker node
                spaceMakerNode = null, spaceMakerParagraph = null,
                // a top level drawing inside the drawing layer
                topLevelDrawing = null,
                // whether content in the drawing layer in a text frame was removed
                isDrawingLayerText = false,
                // a text frame node, that might contain the start position
                textFrameNode = null,
                // container element with target id
                rootNode = self.getRootNode(target);

            // resolve start and end position
            if (!_.isArray(start)) {
                Utils.warn('Editor.implDelete(): missing start position');
                return false;
            }
            startInfo = Position.getDOMPosition(rootNode, start, true);
            if (!startInfo || !startInfo.node) {
                Utils.warn('Editor.implDelete(): invalid start position: ' + JSON.stringify(start));
                return false;
            }
            endInfo = _.isArray(end) ? Position.getDOMPosition(rootNode, end, true) : startInfo;
            if (!endInfo || !endInfo.node) {
                Utils.warn('Editor.implDelete(): invalid end position: ' + JSON.stringify(end));
                return false;
            }

            end = end || start;

            // get attribute type of start and end node
            startInfo.type = resolveElementType(startInfo.node);
            endInfo.type = resolveElementType(endInfo.node);

            // check that start and end point to the same element type
            if ((!startInfo.type || !endInfo.type) || (startInfo.type !== endInfo.type)) {
                Utils.warn('Editor.implDelete(): problem with node types: ' + startInfo.type + ' and ' + endInfo.type);
                return false;
            }

            // check that start and end point to the same element for non text types (only one cell, row, paragraph, ...)
            if ((startInfo.type !== 'text') && (startInfo.node !== endInfo.node)) {
                Utils.warn('Editor.implDelete(): no ranges supported for node type "' + startInfo.type + '"');
                return false;
            }

            // check that for text nodes start and end point have the same parent
            if ((startInfo.type === 'text') && (startInfo.node.parentNode !== endInfo.node.parentNode)) {
                if (DOM.isDrawingLayerNode(endInfo.node.parentNode)) {
                    if (startInfo.node.parentNode !== DOM.getDrawingPlaceHolderNode(endInfo.node).parentNode) {
                        Utils.warn('Editor.implDelete(): deleting range only supported inside one paragraph.');
                        return false;
                    }
                } else if (DOM.isDrawingLayerNode(startInfo.node.parentNode)) {
                    if (endInfo.node.parentNode !== DOM.getDrawingPlaceHolderNode(startInfo.node).parentNode) {
                        Utils.warn('Editor.implDelete(): deleting range only supported inside one paragraph.');
                        return false;
                    }
                } else {
                    Utils.warn('Editor.implDelete(): deleting range only supported inside one paragraph.');
                    return false;
                }
            }

            // checking whether undo of operations is possible
            checkDisableUndoStack(startInfo.type, startInfo.node, _.clone(start), _.clone(end), target);

            // check for a text frame node
            textFrameNode = DrawingFrame.getClosestTextFrameDrawingNode(startInfo.node.parentNode);

            // deleting the different types:
            switch (startInfo.type) {

                case 'text':
                    var isLastCharInPar = !startInfo.node.nextSibling && (startInfo.offset === startInfo.node.textContent.length - 1);
                    // setting the paragraph node (required for page breaks)
                    currentElement = startInfo.node.parentNode;
                    if (DOM.isDrawingLayerNode(currentElement)) { currentElement = DOM.getDrawingPlaceHolderNode(startInfo.node).parentNode; }
                    if (!$(currentElement.parentNode).hasClass('pagecontent') && !DOM.isSlideNode(currentElement)) {
                        if (DrawingFrame.isTextFrameNode(currentElement.parentNode)) {
                            // is the text frame inside a drawing in the drawing layer?
                            if (DOM.isInsideDrawingLayerNode(currentElement)) {
                                isDrawingLayerText = true;
                                // measuring height changes at the current paragraph (that is already currentElement
                                // But for the top level measurement the paragraph containing the space maker node is required
                                topLevelDrawing = DOM.getTopLevelDrawingInDrawingLayerNode(currentElement);
                                spaceMakerNode = DOM.getDrawingSpaceMakerNode(topLevelDrawing);
                                if (DOM.getDrawingSpaceMakerNode(topLevelDrawing)) { spaceMakerParagraph = $(spaceMakerNode).parentsUntil(self.getNode(), DOM.PARAGRAPH_NODE_SELECTOR).last()[0]; }
                                // div.textframe height needs to be measured synchronously
                                currentElement = currentElement.parentNode;
                            } else if (DOM.isTopLevelNodeInComment(currentElement)) {
                                // comments are outside of the main document. Therefore it is only necessary to check the paragraph
                                // inside the comment, not the paragraph containing the placeholder for the comment.
                                currentElement = currentElement.parentNode;
                            } else {
                                currentElement = $(currentElement).parentsUntil(self.getNode(), DOM.PARAGRAPH_NODE_SELECTOR).last()[0];  // its a div.p inside paragraph
                            }
                        } else if (target) {
                            if (!DOM.isMarginalContentNode(currentElement.parentNode)) {
                                currentElement = $(currentElement).parents(DOM.TABLE_NODE_SELECTOR).last()[0]; //its a div.p inside table(s)
                            }
                            // TODO parent DrawingFrame
                        } else {
                            currentElement = $(currentElement).parentsUntil(self.getNode(), DOM.TABLE_NODE_SELECTOR).last()[0]; //its a div.p inside table(s)
                        }
                    }
                    // caching paragraph height before removing text
                    prevHeight = currentElement ? currentElement.offsetHeight : 0;
                    // removing text
                    result = implDeleteText(start, end, {}, target);

                    // block page breaks render if operation is targeted
                    if (target) {
                        self.setBlockOnInsertPageBreaks(true);
                    }

                    // using the same paragraph node again
                    if ((!isDrawingLayerText) && ($(currentElement).data('lineBreaksData'))) {
                        $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
                    }
                    self.setQuitFromPageBreaks(true); // quit from any currently running pagebreak calculus - performance optimization
                    // trigger repainting only if height changed or element is on two pages,
                    // or paragraph is on more pages, and last char in paragraph is not deleted (performance improvement)
                    if (prevHeight !== currentElement.offsetHeight || ($(currentElement).hasClass('contains-pagebreak')) && !isLastCharInPar) {
                        self.insertPageBreaks(isDrawingLayerText ? spaceMakerParagraph : currentElement, textFrameNode);
                        if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }
                    }

                    break;

                case 'paragraph':
                    if (DOM.isImplicitParagraphNode(startInfo.node)) {
                        Utils.warn('Editor.implDelete(): Error: Operation tries to delete an implicit paragraph!');
                        return false;
                    }
                    // used to check if next neighbour is table, and after deletion, selection has to be updated
                    var nextNodeNeighbour = $(startInfo.node).next();

                    isSlideNode = DOM.isSlideNode(startInfo.node);

                    // Setting lastOperationEnd for the special case, that insertParagraph was used instead of splitParagraph
                    // -> if there is a previous paragraph, use the last position of this previous paragraph (task 30742)
                    if (($(startInfo.node).prev().length > 0) && (DOM.isParagraphNode($(startInfo.node).prev()))) {
                        localPosition = _.clone(start);
                        localPosition[localPosition.length - 1]--;
                        localPosition = Position.getLastPositionInParagraph(rootNode, localPosition);
                        self.setLastOperationEnd(localPosition);
                    } else if ($(startInfo.node).prev().length === 0) {
                        // special case: Removing the first paragraph (undo-redo handling)
                        localPosition = _.clone(start);
                        localPosition = Position.getFirstPositionInParagraph(rootNode, localPosition);
                        self.setLastOperationEnd(localPosition);
                    }

                    if (!isSlideNode && ((DOM.isParagraphWithoutNeighbour(startInfo.node)) || (DOM.isFinalParagraphBehindTable(startInfo.node)))) {
                        newParagraph = DOM.createImplicitParagraphNode();
                        $(startInfo.node).parent().append(newParagraph);
                        self.validateParagraphNode(newParagraph);
                        // if original paragraph was marked as marginal, mark newly created implicit as marginal, too
                        if (DOM.isMarginalNode(startInfo.node)) {
                            $(newParagraph).addClass(DOM.MARGINAL_NODE_CLASSNAME);
                        }
                        self.implParagraphChanged(newParagraph);
                        // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                        // newly, also in other browsers (new jQuery version?)
                        self.repairEmptyTextNodes(newParagraph);
                    }

                    if (isSlideNode) { parentNode = startInfo.node.parentNode; }

                    // checking, if the paragraph contains drawing place holder nodes, comment place holder nodes or range
                    // marker nodes. In this case all models need to be updated. Furthermore the comments and the drawings
                    // in their layers need to be removed, too.
                    self.updateCollectionModels(startInfo.node);

                    $(startInfo.node).remove(); // remove the paragraph from the DOM
                    if (parentNode && DOM.isSlideContainerNode(parentNode)) { $(parentNode).remove(); }
                    // the deleted paragraphs can be part of a list, update all lists
                    if (!isSlideNode) { self.handleTriggeringListUpdate(startInfo.node); }

                    if (DOM.isTableNode(nextNodeNeighbour)) { // #34685 properly set cursor inside table after delete paragraph
                        self.getSelection().setTextSelection(Position.getFirstPositionInParagraph(rootNode, start), null, { simpleTextSelection: false, splitOperation: false });
                    }

                    // block page breaks render if operation is targeted
                    if (target) {
                        self.setBlockOnInsertPageBreaks(true);
                    }

                    // get currentElement again (maybe whole paragraph has deleted), and pass it to repaint pagebreaks from that page only
                    currentElement = Position.getContentNodeElement(rootNode, position);
                    self.insertPageBreaks(currentElement, textFrameNode);
                    if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

                    break;

                case 'cell':

                    // checking for absolute drawings, comments and range markers
                    self.updateCollectionModels(startInfo.node);

                    rowPosition = _.clone(start);
                    startCell = rowPosition.pop();
                    endCell = startCell;
                    result = implDeleteCells(rowPosition, startCell, endCell, target);
                    break;

                case 'row':

                    // checking for absolute drawings, comments and range markers
                    self.updateCollectionModels(startInfo.node);

                    tablePosition = _.clone(start);
                    startRow = tablePosition.pop();
                    endRow = startRow;
                    result = implDeleteRows(tablePosition, startRow, endRow, target);

                    // block page breaks render if operation is targeted
                    if (target) {
                        self.setBlockOnInsertPageBreaks(true);
                    }

                    // get currentElement again (maybe whole paragraph has deleted), and pass it to repaint pagebreaks from that page only
                    currentElement = Position.getContentNodeElement(rootNode, position);
                    self.insertPageBreaks(currentElement, textFrameNode);
                    if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

                    break;

                case 'table':

                    // checking for absolute drawings, comments and range markers
                    self.updateCollectionModels(startInfo.node);

                    result = implDeleteTable(start, target);

                    // block page breaks render if operation is targeted
                    if (target) {
                        // trigger header/footer content update on other elements of same type, if change was made inside header/footer
                        self.updateEditingHeaderFooterDebounced(self.getRootNode(target));
                        self.setBlockOnInsertPageBreaks(true);
                    }

                    // get currentElement again (maybe whole paragraph has deleted), and pass it to repaint pagebreaks from that page only
                    currentElement = Position.getContentNodeElement(rootNode, position);
                    self.insertPageBreaks(currentElement, textFrameNode);
                    if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

                    break;

                default:
                    Utils.error('Editor.implDelete(): unsupported node type: ' + startInfo.type);
            }

            // delete undo stack immediately if this is necessary and not a part of an undo group
            if (!self.isInUndoGroup() && self.deleteUndoStack()) {
                self.getUndoManager().clearUndoActions();
                self.setDeleteUndoStack(false);
            }

            return result;
        }

        // public methods -----------------------------------------------------

        /**
         * Generates the operations that will delete the current selection, and
         * executes the operations.
         * Info: With the introduction of 'rangeMarker.handlePartlyDeletedRanges' it is
         * now possible, that not only the end position of the current selection is
         * modified within 'deleteSelected', but also the start position. This happens
         * for example, if the selection includes an end range marker. In this case the
         * corresponding start range marker is also removed. If this is located in the
         * same paragraph as the current selection start position, this position will
         * also be updated. Therefore after executing 'deleteSelected', it is necessary
         * that the start position is read from the selection again without relying on
         * the value before executing 'deleteSelected'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.alreadyPasteInProgress=false]
         *      Whether deleteSelected was triggered from the pasting of clipboard. In
         *      this case the check function 'checkSetClipboardPasteInProgress()' does
         *      not need to be called, because blocking is already handled by the
         *      clipboard function.
         *  @param {Boolean} [options.deleteKey=false]
         *      Whether this function was triggered by the 'delete' or 'backspace' key.
         *      Or whether it was triggered via the controller function by tool bar or
         *      context menu.
         *  @param {Object} [options.snapshot]
         *      The snapshot that can be used, if the user cancels a long running
         *      delete action. Typically this is created inside this function, but if
         *      it was already created by the calling 'paste' function, it can be reused
         *      here.
         *  @param {Boolean} [options.saveStartAttrs=false]
         *      Whether the character attributes at the start position shall be saved.
         *      This attributes can be used later, if text is inserted after the content
         *      was deleted and this new text shall have the same character attributes.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.deleteSelected = function (options) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the logical position of the first and current partially covered paragraph
                firstParaPosition = null, currentParaPosition = null,
                // an optional new start position that can be caused by additionally removed start range nodes outside the selection range
                newStartPosition = null,
                // a helper paragraph for setting attributes
                paragraph = null,
                // the selection object
                selection = self.getSelection(),
                // whether the selection is a rectangular cell selection
                isCellSelection = selection.getSelectionType() === 'cell',
                // whether the selection is a multi drawing selection
                isMultiDrawingSelection = selection.isMultiSelectionSupported() && selection.isMultiSelection(),
                // whether it is necessary to ask the user before deleting content
                askUser = false,
                // whether an additional merge operation is required
                doMergeParagraph = false,
                // the paragraph at the end of the selection
                selectionEndParagraph = null,
                // the promise for generating the operations
                operationGeneratorPromise = null,
                // the promise for asking the user (if required)
                askUserPromise = null,
                // the promise for the asychronous execution of operations
                operationsPromise = null,
                // a node counter for the iterator function
                counter = 0,
                // the first and last node of iteration
                firstNode = null, lastNode = null, currentNode = null,
                // whehter a node is the first node  of a selection
                isFirstNode = false,
                // wheter the first or last paragraph of a selection are only partly removed
                firstParaRemovedPartly = false, lastParaRemovedPartly = false,
                // whether it is necessary to check, if the last paragraph of the selection need to be merged
                checkMergeOfLastParagraph = false,
                // changeTrack: Whether the content was really deleted (inserted by same user) or only attributes were set
                changeTrackDeletedContent = false,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // whether it is required to check for range markers inside the selection
                isRangeCheckRequired = !selection.isAllSelected() && !self.getRangeMarker().isEmpty(),
                // all classes used to identify the range markers
                allRangeMarkerSelector = isRangeCheckRequired ? DOM.RANGEMARKERNODE_SELECTOR : null,
                // the collector for all range marker nodes (including comments)
                allRangeMarkerNodes = $(),
                // created operation
                newOperation = null,
                // whether a progress bar shall be made visible (this might already be done by an outer function)
                alreadyPasteInProgress = Utils.getBooleanOption(options, 'alreadyPasteInProgress', false),
                // whether this function was triggered by pressing 'delete' or 'backspace'
                deleteKeyPressed = Utils.getBooleanOption(options, 'deleteKey', false),
                // whether the character attributes at the start position need to be saved for later usage
                saveStartAttrs = Utils.getBooleanOption(options, 'saveStartAttrs', false),
                // whether a valid snapshot was already created by the caller of this function (typically a 'paste' action over a selection range)
                snapshot = Utils.getObjectOption(options, 'snapshot', null),
                // whether the snapshot was created inside this function
                snapshotCreated = false,
                // whether the deleting will be done asynchronous (in large selections)
                asyncDelete = false,
                // the ratio of operation generation to applying of operation
                operationRatio = 0.3,
                // an optional array with selection ranges for large selections
                splittedSelectionRange = null,
                // whether the user aborted the delete process
                userAbort = false;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the content.
            function doDeleteSelected() {

                // clean up function after the operations were applied
                function postDeleteOperations() {

                    self.setGUITriggeredOperation(false);

                    if (firstParaPosition) {
                        // set paragraph attributes immediately, if paragraph is without content
                        paragraph = Position.getParagraphElement(rootNode, firstParaPosition);
                        if ((paragraph) && ($(paragraph).text().length === 0)) {
                            self.validateParagraphNode(paragraph);
                        }
                    }

                    if (!isCellSelection) {
                        if (currentParaPosition) {
                            paragraph = Position.getParagraphElement(rootNode, currentParaPosition);
                            if ((paragraph) && ($(paragraph).text().length === 0)) {
                                self.validateParagraphNode(paragraph);
                            }
                        } else {
                            Utils.warn('Editor.doDeleteSelected(): currentParaPosition is missing!');
                        }

                        // special handling for additionally removed range start handlers
                        if (newStartPosition) { self.setLastOperationEnd(newStartPosition); }

                        if (self.getChangeTrack().isActiveChangeTracking() && !changeTrackDeletedContent) {
                            // if change tracking is active, and the content was not deleted (inserted by the same user before?),
                            // then the old selection can be reused.
                            self.setLastOperationEnd(selection.getStartPosition());  // setting cursor before(!) removed selection
                        }
                    } else {
                        // cell selection
                        if (firstParaPosition) {   // not complete table selected
                            self.setLastOperationEnd(Position.getFirstPositionInCurrentCell(rootNode, firstParaPosition));
                        }
                    }

                    // collapse selection (but not after canceling delete process)
                    if (!userAbort) {
                        selection.setTextSelection(self.getLastOperationEnd());
                    }
                }

                // apply the operations (undo group is created automatically)
                self.setGUITriggeredOperation(true); // Fix for 30597

                if (asyncDelete) {

                    // applying all operations asynchronously
                    operationsPromise = self.applyTextOperationsAsync(generator, null, { showProgress: false, leaveOnSuccess: true, progressStart: operationRatio });

                    // handler for 'always', that is not triggered, if self is in destruction (document is closed, 42567)
                    self.waitForAny(operationsPromise, function () {
                        postDeleteOperations();
                    });

                } else {

                    // applying all operations synchronously
                    self.applyOperations(generator);

                    postDeleteOperations();

                    operationsPromise = $.when();
                }

                return operationsPromise;
            }

            // helper function to generate all delete operations in multi drawing selections
            function deleteDrawingsInMultiDrawingSelection() {

                _.each(selection.getMultiSelection(), function (drawingSelection) {

                    if (containsUnrestorableElements(selection.getDrawingNodeFromMultiSelection(drawingSelection))) { askUser = true; }

                    newOperation = { start: selection.getStartPositionFromMultiSelection(drawingSelection) };
                    self.extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.DELETE, newOperation);
                });
            }

            // helper function to generate all delete operations
            function generateAllDeleteOperations(selectionRange) {

                var // the logical start position, if specified
                    localStartPos = (selectionRange && selectionRange[0]) || null,
                    // the logical end position, if specified
                    localEndPos = (selectionRange && selectionRange[1]) || null,
                    // the change track object
                    changeTrack = self.getChangeTrack();

                // visit all content nodes (tables, paragraphs) in the selection
                selection.iterateContentNodes(function (node, position, startOffset, endOffset, parentCovered) {

                    var // whether the node is the last child of its parent
                        isLastChild = node === node.parentNode.lastChild,
                        // whether a paragraph is selected completely
                        paragraphSelected = false,
                        // a collector of all rows of a table
                        allRows = null;

                    counter++;  // counting the nodes
                    if (counter === 1) {
                        isFirstNode = true;
                        firstNode = node;
                    } else {
                        isFirstNode = false;
                        lastNode = node;
                        lastParaRemovedPartly = false;
                    }

                    // saving the current node
                    currentNode = node;

                    if (DOM.isParagraphNode(node)) {

                        // remember first and last paragraph
                        if (!firstParaPosition) { firstParaPosition = position; }
                        currentParaPosition = position;

                        if (_.isNumber(startOffset) && !_.isNumber(endOffset)) {
                            if (startOffset === 0) {
                                paragraphSelected = true;  // the full paragraph can be delete
                            } else {
                                checkMergeOfLastParagraph = true;  // for the final paragraph, a merge-check is necessary after iteration
                            }
                        }

                        // checking if a paragraph is selected completely and requires merge
                        paragraphSelected = _.isNumber(startOffset) && (startOffset === 0) && !_.isNumber(endOffset);

                        if (changeTrack.isActiveChangeTracking()) { // complete separation of code for change tracking

                            if (!DOM.isImplicitParagraphNode(node)) {

                                // do not delete the paragraph node, if it is only covered partially;
                                // or if it is the last paragraph when the parent container is cleared completely
                                if (parentCovered ? isLastChild : (_.isNumber(startOffset) || _.isNumber(endOffset))) {

                                    // 'deleteText' operation needs valid start and end position
                                    startOffset = _.isNumber(startOffset) ? startOffset : 0;
                                    endOffset = _.isNumber(endOffset) ? endOffset : (Position.getParagraphLength(rootNode, position) - 1);

                                    // delete the covered part of the paragraph
                                    if (isCellSelection) {
                                        // TODO
                                    } else if (paragraphSelected || ((startOffset === 0) && (endOffset === 0) && (Position.getParagraphLength(rootNode, position) === 0))) {
                                        if (changeTrack.isActiveChangeTracking()) {
                                            if (!DOM.isImplicitParagraphNode(node)) { // Checking if this is an implicit paragraph
                                                newOperation = { start: position, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                                self.extendPropertiesWithTarget(newOperation, target);
                                                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                                            }
                                        }
                                    } else if (startOffset <= endOffset) {
                                        // Will be handled during iteration over all paragraph children
                                    }
                                } else {
                                    newOperation = { start: position, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                    self.extendPropertiesWithTarget(newOperation, target);
                                    generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                                }

                                // if change tracking is active, three different cases need to be disinguished:
                                // - the node is already marked for removal -> no operation needs to be created
                                // - the node is marked for insertion of new text -> text can be removed with delete operation
                                // - if this is a text span without change track attribute -> a setAttributes operation is created
                                // - other content is simply removed, no change tracking implemented yet

                                // iterating over all nodes of the paragraph is required
                                Position.iterateParagraphChildNodes(node, function (subnode, nodestart, nodelength, offsetstart, offsetlength) {

                                    var // whether a delete operation needs to be created
                                        createDeleteOperation = true,
                                        // whether a delete operation needs to be created
                                        createSetAttributesOperation = false,
                                        // whether the node is a change track insert node
                                        isChangeTrackInsertNode = false;

                                    if (DOM.isTextSpan(subnode) || DOM.isTextComponentNode(subnode) || DrawingFrame.isDrawingFrame(subnode)) {
                                        if (changeTrack.isInsertNodeByCurrentAuthor(subnode)) {
                                            // the content was inserted during active change track
                                            createDeleteOperation = true;
                                            createSetAttributesOperation = false;
                                            isChangeTrackInsertNode = true;
                                        } else if (DOM.isChangeTrackRemoveNode(subnode)) {
                                            // the content was already removed during active change track
                                            createDeleteOperation = false;
                                            createSetAttributesOperation = false;
                                        } else {
                                            // a set attributes operation is required for text spans without change track attribute
                                            createDeleteOperation = false;
                                            createSetAttributesOperation = true;
                                        }
                                    }

                                    if (createSetAttributesOperation) {
                                        newOperation = { start: position.concat([nodestart + offsetstart]), end: position.concat([nodestart + offsetstart + offsetlength - 1]), attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                        self.extendPropertiesWithTarget(newOperation, target);
                                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                                    }

                                    if (createDeleteOperation) {
                                        newOperation = { start: position.concat([nodestart + offsetstart]), end: position.concat([nodestart + offsetstart + offsetlength - 1]) };
                                        self.extendPropertiesWithTarget(newOperation, target);
                                        generator.generateOperation(Operations.DELETE, newOperation);

                                        // the 'inserted' is no longer valid at a completely removed node
                                        // Info: The order is reversed -> 1. step: Setting attributes, 2. step: Removing all content
                                        if (isChangeTrackInsertNode && (offsetstart === 0) && (offsetlength === nodelength)) {
                                            newOperation = { start: position.concat([nodestart]), end: position.concat([nodestart + offsetlength - 1]), attrs: { changes: { mode: null } } };
                                            self.extendPropertiesWithTarget(newOperation, target);
                                            generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                                        }

                                    }

                                }, undefined, { start: startOffset, end: endOffset });
                            }

                        } else {  // no change tracking active

                            if (!DOM.isImplicitParagraphNode(node)) {

                                // do not delete the paragraph node, if it is only covered partially;
                                // or if it is the last paragraph when the parent container is cleared completely
                                if (parentCovered ? isLastChild : (_.isNumber(startOffset) || _.isNumber(endOffset))) {

                                    // 'deleteText' operation needs valid start and end position
                                    startOffset = _.isNumber(startOffset) ? startOffset : 0;
                                    endOffset = _.isNumber(endOffset) ? endOffset : (Position.getParagraphLength(rootNode, position) - 1);

                                    // delete the covered part of the paragraph
                                    if (isMultiDrawingSelection) {
                                        if (deleteKeyPressed) { deleteDrawingsInMultiDrawingSelection(); }
                                    } else if (isCellSelection) {
                                        if (containsUnrestorableElements(node)) { askUser = true; }
                                        if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector)); }
                                        newOperation = { start: position };
                                        self.extendPropertiesWithTarget(newOperation, target);
                                        generator.generateOperation(Operations.DELETE, newOperation);
                                    } else if (paragraphSelected || ((startOffset === 0) && (endOffset === 0) && (Position.getParagraphLength(rootNode, position) === 0))) {  // -> do not delete from [1,0] to [1,0]
                                        if (isRangeCheckRequired && paragraphSelected) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector)); }
                                        newOperation = { start: position };
                                        self.extendPropertiesWithTarget(newOperation, target);
                                        generator.generateOperation(Operations.DELETE, newOperation);
                                        self.handleTriggeringListUpdate(node);
                                    } else if (startOffset <= endOffset) {
                                        if (isFirstNode) {
                                            firstParaRemovedPartly = true;
                                        } else {
                                            lastParaRemovedPartly = true;
                                        }

                                        // saving the character attributes
                                        if (isFirstNode && saveStartAttrs) { self.addCharacterAttributesToPreselectedAttributes(node, startOffset, 'character'); }

                                        if (containsUnrestorableElements(node, startOffset, endOffset)) { askUser = true; }
                                        if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector, startOffset, endOffset)); }
                                        newOperation = { start: position.concat([startOffset]), end: position.concat([endOffset]) };
                                        self.extendPropertiesWithTarget(newOperation, target);
                                        generator.generateOperation(Operations.DELETE, newOperation);
                                    }
                                } else {
                                    if (containsUnrestorableElements(node)) { askUser = true; }
                                    if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector)); }
                                    newOperation = { start: position };
                                    self.extendPropertiesWithTarget(newOperation, target);
                                    generator.generateOperation(Operations.DELETE, newOperation);
                                }
                            }
                        }

                    } else if (DOM.isTableNode(node)) {

                        allRows = DOM.getTableRows(node);

                        if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(node) && !changeTrack.allAreInsertedNodesByCurrentAuthor(allRows)) {
                            // Table is not (necessarily) marked with 'inserted', but all rows
                            // -> setting changes attribute at all rows, not at the table
                            _.each(allRows, function (row, index) {
                                var rowPosition = _.clone(position);
                                rowPosition.push(index);
                                newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                self.extendPropertiesWithTarget(newOperation, target);
                                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                            });
                        } else {
                            // delete entire table
                            newOperation = { start: position };
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.DELETE, newOperation);
                            // checking, if this is a table with exceeded size
                            if (DOM.isExceededSizeTableNode(node) || containsUnrestorableElements(node)) { askUser = true; }
                            if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(node, allRangeMarkerSelector)); }
                            // important for the new position: Although change track might be activated, the content was really removed
                            changeTrackDeletedContent = true;
                        }
                    } else {
                        Utils.error('Editor.deleteSelected(): unsupported content node');
                        return Utils.BREAK;
                    }

                }, this, { shortestPath: true, startPos: localStartPos, endPos: localEndPos });

            }  // end of generateAllDeleteOperations

            // helper function to resort all delete operations and optionally add further delete operations
            // for example for range markers
            function resortDeleteOperations() {

                // operations MUST be executed in reverse order to preserve the positions
                generator.reverseOperations();

                // Merging paragraphs additionally is required, if:
                // - firstNode and lastNode of selection are (different) paragraphs
                // - firstNode and lastNode are not removed completely
                // - firstNode and lastNode have the same parent
                //
                // In Firefox tables there is a cell selection. In this case all paragraphs
                // are deleted completely, so that no merge is required.
                // Merging is also not allowed, if change tracking is active.

                doMergeParagraph = firstParaPosition && firstParaRemovedPartly && lastParaRemovedPartly && (firstNode.parentNode === lastNode.parentNode);

                // it is additionally necessary to merge paragraphs, if the selection ends at position 0 inside a paragraph.
                // In this case the 'end-paragraph' will not be iterated in 'selection.iterateContentNodes'. So this need
                // to be checked manually.
                if (!doMergeParagraph && checkMergeOfLastParagraph && (_.last(selection.getEndPosition()) === 0)) {
                    // there is another chance to merge paragraph, if mouse selection goes to start of following paragraph
                    // using currentParaPosition and end of selection and lastNode
                    // Additionally the first paragraph and the selectionEndParagraph need to have the same parent
                    selectionEndParagraph = Position.getDOMPosition(rootNode, _.initial(selection.getEndPosition())).node;
                    if (currentNode && currentNode.nextSibling && selectionEndParagraph && (currentNode.nextSibling === selectionEndParagraph) && (firstNode.parentNode === selectionEndParagraph.parentNode)) {
                        doMergeParagraph = true;
                    }
                }

                if (doMergeParagraph) {
                    if (self.getChangeTrack().isActiveChangeTracking()) {
                        newOperation = { start: firstParaPosition, attrs: { changes: { removed: self.getChangeTrack().getChangeTrackInfo() } } };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                    } else {
                        newOperation = { start: firstParaPosition };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.PARA_MERGE, newOperation);
                    }
                }

                // handling for all collected ranges (if content is removed before the current start of the selection
                // (for example range start nodes), if is necessary to know the new position of the current start
                // position.
                if (allRangeMarkerNodes.length > 0) {
                    newStartPosition = self.getRangeMarker().handlePartlyDeletedRanges(allRangeMarkerNodes, generator, selection.getStartPosition());
                }

            }  // end of resortDeleteOperations

            // helper function for asking the user, if unrestorable content shall be deleted
            function askUserHandler() {

                // Asking the user, if he really wants to remove the content, if it contains
                // a table with exceeded size or other unrestorable elements. This cannot be undone.
                if (askUser) {
                    askUserPromise = self.showDeleteWarningDialog(
                        gt('Delete Contents'),
                        gt('Deleting the selected elements cannot be undone. Do you want to continue?')
                    );
                    // return focus to editor after 'No' button
                    askUserPromise.then(function () {
                        // success handler
                    }, function () {
                        if (asyncDelete) {
                            // user abort during visible dialog -> no undo required -> leaving busy mode immediately
                            self.leaveAsyncBusy();
                        }
                        app.getView().grabFocus();
                    });
                } else {
                    askUserPromise = $.when();
                }

                return askUserPromise;
            }

            if (!selection.hasRange()) { return $.when(); }

            // Search not only for range markers, but also for comments, if required
            if (allRangeMarkerSelector && !self.getCommentLayer().isEmpty()) { allRangeMarkerSelector += ', ' + DOM.COMMENTPLACEHOLDER_NODE_SELECTOR; }

            // checking selection size -> make array with splitted selection [0,100], [101, 200], ... (39061)
            splittedSelectionRange = Position.splitLargeSelection(self.getCurrentRootNode(), selection.getStartPosition(), selection.getEndPosition(), self.getMaxTopLevelNodes());

            if (splittedSelectionRange && splittedSelectionRange.length > 1) {

                asyncDelete = true;

                // make sure that only one asynchronous call is processed at the same time
                if (!alreadyPasteInProgress && self.checkSetClipboardPasteInProgress()) { return $.when(); }

                // blocking keyboard input during generation and applying of operations
                self.setBlockKeyboardEvent(true);

                // creating a snapshot (but not, if it was already created by the caller of this function)
                if (!snapshot) {
                    snapshot = new Snapshot(app);
                    snapshotCreated = true;
                }

                // show a message with cancel button
                // -> immediately grabbing the focus, after calling enterBusy. This guarantees, that the
                // keyboard blocker works. Otherwise the keyboard events will not be catched by the page.
                app.getView().enterBusy({
                    cancelHandler: function () {
                        userAbort = true;  // user aborted the process
                        if (operationsPromise && operationsPromise.abort) { // order is important, the latter has to win
                            // restoring the old document state
                            snapshot.apply();
                            // calling abort function for operation promise
                            app.enterBlockOperationsMode(function () { operationsPromise.abort(); });
                        } else if (operationGeneratorPromise && operationGeneratorPromise.abort) {
                            // cancel during creation of operation -> no undo required
                            operationGeneratorPromise.abort();
                        }
                    },
                    immediate: true,
                    warningLabel: gt('Sorry, deleting content will take some time.')
                }).grabFocus();

                // generate operations asynchronously
                operationGeneratorPromise = self.iterateArraySliced(splittedSelectionRange, function (oneRange) {
                    // calling function to generate operations synchronously with reduced selection range
                    generateAllDeleteOperations(oneRange);
                }, { delay: 'immediate', infoString: 'Text: generateAllDeleteOperations' })
                    .then(function () {

                        // synchronous resorting of delete operations
                        resortDeleteOperations();

                        // Asking the user, if he really wants to remove the content, if it contains
                        // a table with exceeded size or other unrestorable elements. This cannot be undone.
                        // return askUserHandler();
                    }, function () {
                        // user abort during creation of operation -> no undo required
                        // -> leaving busy mode immediately
                        self.leaveAsyncBusy();
                    })
                    // add progress handling
                    .progress(function (progress) {
                        // update the progress bar according to progress of the operations promise
                        app.getView().updateBusyProgress(operationRatio * progress);
                    });

            } else {

                // synchronous handling for small selections
                asyncDelete = false;

                // calling function to generate operations synchronously
                generateAllDeleteOperations();

                // collecting further delete operations
                resortDeleteOperations();

                // in synchronous process the promise for operation generation is already resolved
                operationGeneratorPromise = $.when();
            }

            // delete contents on resolved promise
            return operationGeneratorPromise.then(askUserHandler).then(doDeleteSelected).always(function () {
                if (snapshotCreated) { snapshot.destroy(); }
            });
        };

        /**
         * Editor API function to generate 'delete' operations.
         * This is a generic function, that can be used to delete any component (text, paragraph,
         * cell, row, table, ...). Deleting columns is not supported, because columns cannot
         * be described with a logical position.
         * The parameter 'start' and 'end' are used to specify the position of the components that
         * shall be deleted. For all components except 'text' the 'end' position will be ignored.
         * For paragraphs, cells, ... only one specific component can be deleted within this
         * operation. Only on text level a complete range can be deleted.
         *
         * @param {Number[]} start
         *  The logical start position.
         *
         * @param {Number[]} [end]
         *  The logical end position (optional). This can be different from 'start' only for text ranges
         *  inside one paragraph. A text range can include characters, fields, and drawing objects,
         *  but must be contained in a single paragraph.
         *
         * @param {Object} [options]
         *  Additional optional options, that can be used for performance reasons.
         *  @param {Boolean} [options.setTextSelection=true]
         *      If set to false, the text selection will not be set within this
         *      function. This is useful, if the caller takes care of setting the
         *      cursor after the operation. This is the case for 'Backspace' and
         *      'Delete' operations. It is a performance issue, that the text
         *      selection is not set twice.
         *  @param {Boolean} [options.handleUnrestorableContent=false]
         *      If set to true, it will be checked if the range, that shall be
         *      deleted, contains unrestorable content. In this case a dialog appears,
         *      in which the user is asked, if he really wants to delete this content.
         *      This is the case for 'Backspace' and 'Delete' operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.deleteRange = function (start, end, options) {
            var // a helper dom position element
                domPos = null,
                // whether the cursor shall be set in this function call
                setTextSelection = Utils.getBooleanOption(options, 'setTextSelection', true),
                // whether start and end position are different
                handleUnrestorableContent = Utils.getBooleanOption(options, 'handleUnrestorableContent', false),
                // whether start and end logical position are different
                isSimplePosition = !_.isArray(end) || _.isEqual(start, end),
                // a paragraph node
                paragraph = null,
                // whether it is necessary to ask the user before deleting range
                askUser = false,
                // the resulting promise
                promise = null,
                // target ID of currently active root node, if existing
                target = self.getActiveTarget(),
                // currently active root node
                activeRootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the range.
            function doDeleteRange() {

                var // whether a delete operation needs to be created
                    createDeleteOperation = true,
                    // whether a delete operation needs to be created
                    createSetAttributesOperation = false,
                    // attribute object for changes attribute
                    attrs,
                    // the dom point at text level position
                    element,
                    // a logical helper position
                    lastOperationEndLocal = null,
                    // the change track object
                    changeTrack = self.getChangeTrack();

                // special handling for active change tracking
                if (changeTrack.isActiveChangeTracking()) {

                    // TODO: Also checking the end position?!
                    // -> not required, if this function is only called from deleting with 'backspace' or 'delete',
                    // that call first of all 'deleteSelected'.
                    // -> otherwise it might happen during active change tracking, that more than one operation
                    // needs to be generated.
                    element = Position.getDOMPosition(paragraph, [_.last(start)], true);

                    if (element && element.node && (DOM.isTextSpan(element.node) || DOM.isInlineComponentNode(element.node))) {
                        if (changeTrack.isInsertNodeByCurrentAuthor(element.node)) {
                            // the content was inserted during active change track (by the same user)
                            createDeleteOperation = true;
                            createSetAttributesOperation = false;
                        } else if (DOM.isChangeTrackRemoveNode(element.node)) {
                            // the content was already removed during active change track
                            createDeleteOperation = false;
                            createSetAttributesOperation = false;
                        } else {
                            // a set attributes operation is required for text spans without change track attribute
                            // or for spans inserted by another user
                            createDeleteOperation = false;
                            createSetAttributesOperation = true;
                        }
                    }

                    if (createSetAttributesOperation) {
                        createDeleteOperation = false;
                        // adding changes attributes
                        attrs = { changes: { removed: changeTrack.getChangeTrackInfo() } };
                        newOperation = { name: Operations.SET_ATTRIBUTES, start: _.clone(start), end: _.clone(end), attrs: attrs };
                        self.extendPropertiesWithTarget(newOperation, target);
                        self.applyOperations(newOperation);
                    }

                    // Setting the cursor to the correct position, if no character was deleted
                    if (!createDeleteOperation) {
                        lastOperationEndLocal = _.clone(end);
                        lastOperationEndLocal[lastOperationEndLocal.length - 1] += 1;
                        self.setLastOperationEnd(lastOperationEndLocal);
                    }

                }

                if (createDeleteOperation) {
                    // Using end as it is, not subtracting '1' like in 'deleteText'
                    self.setGUITriggeredOperation(true); // Fix for 30597
                    newOperation = { name: Operations.DELETE, start: _.clone(start), end: _.clone(end) };
                    self.extendPropertiesWithTarget(newOperation, target);
                    self.applyOperations(newOperation);
                    self.setGUITriggeredOperation(false);

                    // Setting paragraph attributes immediately, if paragraph is without content.
                    // This avoids jumping of following paragraphs.
                    if ((_.last(start) === 0) && (_.last(end) === 0)) {
                        domPos = Position.getDOMPosition(activeRootNode, _.chain(start).clone().initial().value(), true);
                        if ((domPos) && (domPos.node) && ($(domPos.node).is(DOM.PARAGRAPH_NODE_SELECTOR)) && ($(domPos.node).text().length === 0)) {
                            self.validateParagraphNode(domPos.node);
                        }
                    }

                    // setting the cursor position
                    if (setTextSelection) {
                        self.getSelection().setTextSelection(self.getLastOperationEnd());
                    }
                }
            }

            end = end || _.clone(start);

            // checking, if an unrestorable
            if (handleUnrestorableContent) {
                paragraph = Position.getParagraphElement(activeRootNode, _.initial(start));
                if (paragraph) {
                    if (isSimplePosition && containsUnrestorableElements(paragraph, _.last(start), _.last(start))) {
                        askUser = true;
                    } else {
                        // if node contains special page number fields in header and footer restore original state before delete
                        self.getFieldManager().checkRestoringSpecialFields(paragraph, _.last(start), _.last(end));
                    }
                }
            }

            // Asking the user, if he really wants to remove the content, if it contains
            // a table with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = self.showDeleteWarningDialog(
                    gt('Delete Contents'),
                    gt('Deleting the selected elements cannot be undone. Do you want to continue?')
                );
                // return focus to editor after 'No' button
                promise.fail(function () { app.getView().grabFocus(); });
            } else {
                promise = $.when();
            }

            // delete contents on resolved promise
            return promise.done(doDeleteRange);
        };

        /**
         * Editor API function to generate 'delete' operation for the complete table.
         * This method use the 'deleteRows' method with a predefined start and end index.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.deleteTable = function () {
            var // the start position of the selection
                position = self.getSelection().getStartPosition(),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // the index of the row inside the table, in which the selection ends
                endIndex = Position.getLastRowIndexInTable(rootNode, position);

            return self.deleteRows({ start: 0, end: endIndex });
        };

        /**
         * Editor API function to generate 'delete' operations for rows or  a complete table. This
         * function is triggered by an user event. It evaluates the current selection. If the
         * selection includes all rows of the table, the table is removed completely within one
         * operation. If the row or table contains unrestorable content, the user is asked, whether
         * he really wants to delete the rows.
         *
         * @param {Object} options
         *  Optional parameters:
         *  @param {Number} [options.start]
         *   The index of the start row
         *  @param {Number} [options.end]
         *   The index of the end row
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.deleteRows = function (options) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the selection object
                selection = self.getSelection(),
                // the start position of the selection
                position = selection.getStartPosition(),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // the index of the row inside the table, in which the selection starts
                start = Utils.getOption(options, 'start', Position.getRowIndexInTable(rootNode, position)),
                // the index of the row inside the table, in which the selection ends
                end = Utils.getOption(options, 'end', selection.hasRange() ? Position.getRowIndexInTable(rootNode, selection.getEndPosition()) : start),
                // the logical position of the table
                tablePos = Position.getLastPositionFromPositionByNodeName(rootNode, position, DOM.TABLE_NODE_SELECTOR),
                // the index of the last row in the table
                lastRow = Position.getLastRowIndexInTable(rootNode, position),
                // whether the complete table shall be deleted
                isCompleteTable = (start === 0) && (end === lastRow),
                // logical row position
                rowPosition = null,
                // the table node containing the row
                tableNode = null,
                // the row nodes to be deleted
                rowNode = null,
                // the collector of all table rows
                allRows = null,
                // loop counter
                i = 0,
                // whether it is necessary to ask the user before removing row or table
                askUser = false,
                // the resulting promise
                promise = null,
                // whether it is required to check for range markers inside the selection
                isRangeCheckRequired = !selection.isAllSelected() && !self.getRangeMarker().isEmpty(),
                // all classes used to identify the range markers
                allRangeMarkerSelector = isRangeCheckRequired ? DOM.RANGEMARKERNODE_SELECTOR : null,
                // the collector for all range marker nodes (including comments)
                allRangeMarkerNodes = $(),
                // the change track object
                changeTrack = self.getChangeTrack(),
                // created operation
                newOperation = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the selected rows.
            function doDeleteRows() {

                self.setGUITriggeredOperation(true);
                self.applyOperations(generator);
                self.setGUITriggeredOperation(false);

                // setting the cursor position after deleting content
                selection.setTextSelection(self.getLastOperationEnd());
            }

            // helper function to determine a valid new start position for range markers. This is necessary, if
            // from an existing range, only the start marker is removed, but not the end marker.
            function getValidInsertStartRangePosition() {

                var // the logical position for inserting range markers
                    insertStartRangePosition = null,
                    // whether the last row was deleted
                    lastRowDeleted = (end === lastRow);

                if (!lastRowDeleted) {
                    // a following remaining row comes to the current cursor position, so that it is still valid
                    insertStartRangePosition = selection.getStartPosition();
                    insertStartRangePosition[insertStartRangePosition.length - 1] = 0;
                } else {
                    // Find the first valid position behind the table
                    insertStartRangePosition = Position.getFirstPositionInParagraph(self.getCurrentRootNode(), Position.increaseLastIndex(tablePos));
                    if (isCompleteTable) {
                        // because the table is removed completely, the position needs to be reduced by 1
                        insertStartRangePosition[insertStartRangePosition.length - 2] -= 1;
                    }
                }

                return insertStartRangePosition;
            }

            // Search not only for range markers, but also for comments, if required
            if (allRangeMarkerSelector && !self.getCommentLayer().isEmpty()) { allRangeMarkerSelector += ', ' + DOM.COMMENTPLACEHOLDER_NODE_SELECTOR; }

            // Generating only one operation, if the complete table is removed.
            // Otherwise sending one operation for removing each row.
            if (isCompleteTable) {
                tableNode = Position.getTableElement(rootNode, tablePos);
                allRows = DOM.getTableRows(tableNode);
                // not setting attributes, if this table was inserted with change tracking by the current user
                if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(tableNode) && !changeTrack.allAreInsertedNodesByCurrentAuthor(allRows)) {
                    // Table is not (necessarily) marked with 'inserted', but all rows
                    // -> setting changes attribute at all rows, not at the table
                    _.each(allRows, function (row, index) {
                        rowPosition = _.clone(tablePos);
                        rowPosition.push(index);
                        newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                    });
                } else {
                    newOperation = { start: _.copy(tablePos, true) };
                    self.extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.DELETE, newOperation);
                    if (containsUnrestorableElements(tablePos)) { askUser = true; }
                    if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(tableNode, allRangeMarkerSelector)); }
                }
            } else {
                for (i = end; i >= start; i--) {
                    rowPosition = _.clone(tablePos);
                    rowPosition.push(i);
                    rowNode = Position.getTableRowElement(rootNode, rowPosition);
                    if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(rowNode)) {
                        newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                    } else {
                        newOperation = { start: rowPosition };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.DELETE, newOperation);
                        if (containsUnrestorableElements(rowPosition)) { askUser = true; }
                        if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(rowNode, allRangeMarkerSelector)); }
                    }
                }
            }

            // handling for all collected ranges
            if (allRangeMarkerNodes.length > 0) {
                self.getRangeMarker().handlePartlyDeletedRanges(allRangeMarkerNodes, generator, getValidInsertStartRangePosition());
            }

            // Asking the user, if he really wants to remove the content, if it contains
            // at least one row with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = self.showDeleteWarningDialog(
                    gt('Delete Rows'),
                    gt('Deleting the selected rows cannot be undone. Do you want to continue?')
                );
            } else {
                promise = $.when();
            }

            // delete rows on resolved promise
            return promise.done(doDeleteRows);
        };

        /**
         * Editor API function to generate 'delete' operations for columns or a complete table. This
         * function is triggered by an user event. It evaluates the current selection. If the
         * selection includes all columns of the table, the table is removed completely within one
         * operation. If the column or table contains unrestorable content, the user is asked, whether
         * he really wants to delete the columns.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.deleteColumns = function () {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the selection object
                selection = self.getSelection(),
                // the start position of the selection
                position = selection.getStartPosition(),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current root container node for element
                rootNode = self.getCurrentRootNode(),
                // the logical position of the table
                tablePos = Position.getLastPositionFromPositionByNodeName(rootNode, position, DOM.TABLE_NODE_SELECTOR),
                // the table node
                tableNode = Position.getDOMPosition(rootNode, tablePos).node,
                // the maximum grid number of the table
                maxGrid = Table.getColumnCount(tableNode) - 1,
                // the row node at the logical start position of the selection
                rowNode = Position.getLastNodeFromPositionByNodeName(rootNode, position, 'tr'),
                // the index of the column at the logical start position of the selection
                startColIndex = Position.getColumnIndexInRow(rootNode, position),
                // the index of the column at the logical end position of the selection
                endColIndex = selection.hasRange() ? Position.getColumnIndexInRow(rootNode, selection.getEndPosition()) : startColIndex,
                // a helper object for calculating grid positions in the table
                returnObj = Table.getGridPositionFromCellPosition(rowNode, startColIndex),
                // the start grid position in the table
                startGrid = returnObj.start,
                // the end grid position in the table
                endGrid = selection.hasRange() ? Table.getGridPositionFromCellPosition(rowNode, endColIndex).end : returnObj.end,
                // whether the complete table shall be deleted
                isCompleteTable = (startGrid === 0) && (endGrid === maxGrid),
                // logical row position
                rowPos = null,
                // logical cell position
                cellPos = null,
                // the index of the last row in the table
                maxRow = null,
                // whether all rows will be deleted
                deletedAllRows = false,
                // a jQuery collection containing all rows of the specified table
                allRows = null,
                // An array, that contains for each row an array with two integer values
                // for start and end position of the cells.
                allCellRemovePositions = null,
                // a row node
                currentRowNode = null,
                // a cell node
                cellNode = null,
                // Two integer values for start and end position of the cells in the row.
                oneRowCellArray = null,
                // helper numbers for cell descriptions inside a row
                start = 0, end = 0,
                // the table grid of a table
                tableGrid = null,
                // loop counter
                i = 0, j = 0,
                // whether it is necessary to ask the user before removing row or table
                askUser = false,
                // the resulting promise
                promise = null,
                // whether it is necessary to set the table grid again, after removing cells
                refreshTableGrid = false,
                // whether the delete column operation is required with activated change tracking
                // This delete is required, if the column was inserted with change tracking
                deleteForChangeTrackRequired = false,
                // whether it is required to check for range markers inside the selection
                isRangeCheckRequired = !selection.isAllSelected() && !self.getRangeMarker().isEmpty(),
                // all classes used to identify the range markers
                allRangeMarkerSelector = isRangeCheckRequired ? DOM.RANGEMARKERNODE_SELECTOR : null,
                // the collector for all range marker nodes (including comments)
                allRangeMarkerNodes = $(),
                // the change track object
                changeTrack = self.getChangeTrack(),
                // created operation
                newOperation = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the selected rows.
            function doDeleteColumns() {

                // apply the operations (undo group is created automatically)
                self.setGUITriggeredOperation(true);
                self.applyOperations(generator);
                self.setGUITriggeredOperation(false);
                self.setRequiresElementFormattingUpdate(true);

                // setting the cursor position after deleting the content
                selection.setTextSelection(self.getLastOperationEnd());
            }

            // helper function to determine a valid new start position for range markers. This is necessary, if
            // from an existing range, only the start marker is removed, but not the end marker.
            function getValidInsertStartRangePosition() {

                var // the logical position for inserting range markers
                    insertStartRangePosition = null;

                if (isCompleteTable || deletedAllRows) {
                    // Find the first valid position behind the table
                    insertStartRangePosition = Position.getFirstPositionInParagraph(self.getCurrentRootNode(), Position.increaseLastIndex(tablePos));
                    // because the table is removed completely, the position needs to be reduced by 1
                    insertStartRangePosition[insertStartRangePosition.length - 2] -= 1;
                }

                return insertStartRangePosition;
            }

            // Search not only for range markers, but also for comments, if required
            if (allRangeMarkerSelector && !self.getCommentLayer().isEmpty()) { allRangeMarkerSelector += ', ' + DOM.COMMENTPLACEHOLDER_NODE_SELECTOR; }

            // Generating only one operation, if the complete table is removed.
            // Otherwise sending operations for removing each columns, maybe for rows and
            // for setting new table attributes.
            if (isCompleteTable) {
                allRows = DOM.getTableRows(tableNode);
                // not setting attributes, if this table was inserted with change tracking by the current user
                if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(tableNode) && !changeTrack.allAreInsertedNodesByCurrentAuthor(allRows)) {
                    // Table is not (necessarily) marked with 'inserted', but all rows
                    // -> setting changes attribute at all rows, not at the table
                    _.each(allRows, function (row, index) {
                        var rowPosition = _.clone(tablePos);
                        rowPosition.push(index);
                        newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                    });
                } else {
                    newOperation = { start: _.copy(tablePos, true) };
                    self.extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.DELETE, newOperation);
                    if (containsUnrestorableElements(tablePos)) { askUser = true; }
                    if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(tableNode, allRangeMarkerSelector)); }
                }
            } else {
                // generating delete columns operation, but further operations might be necessary
                // -> if this is necessary for change tracking, this need to be evaluated using deleteForChangeTrackRequired
                if (!changeTrack.isActiveChangeTracking()) {
                    newOperation = { start: tablePos, startGrid: startGrid, endGrid: endGrid };
                    self.extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.COLUMNS_DELETE, newOperation);
                }

                // Checking, if there will be rows without cells after the columns are deleted
                maxRow = Table.getRowCount(tableNode) - 1;
                deletedAllRows = true;
                allRows = DOM.getTableRows(tableNode);
                allCellRemovePositions = Table.getAllRemovePositions(allRows, startGrid, endGrid);

                for (i = maxRow; i >= 0; i--) {
                    rowPos = _.clone(tablePos);
                    rowPos.push(i);
                    currentRowNode = Position.getDOMPosition(rootNode, rowPos).node;
                    oneRowCellArray =  allCellRemovePositions[i];
                    end = oneRowCellArray.pop();
                    start = oneRowCellArray.pop();

                    if ($(currentRowNode).children().length === (end - start + 1)) {
                        // checking row attributes
                        if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(currentRowNode)) { // not setting attributes, if this row was inserted with change tracking
                            newOperation = { start: _.copy(rowPos, true), attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                        } else {
                            newOperation = { start: rowPos };
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.DELETE, newOperation);
                            if (containsUnrestorableElements(rowPos)) { askUser = true; }
                            if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(currentRowNode, allRangeMarkerSelector)); }
                            refreshTableGrid = true;
                            deleteForChangeTrackRequired = true;
                        }
                    } else {
                        deletedAllRows = false;
                        // checking unrestorable content in all cells, that will be removed
                        for (j = start; j <= end; j++) {
                            cellPos = _.clone(rowPos);
                            cellPos.push(j);

                            // marking all cells with change track attribute (TODO: Also necessary to check the parent nodes?)
                            cellNode = Position.getSelectedTableCellElement(rootNode, cellPos);
                            if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(cellNode)) { // not setting attributes, if this cell was inserted with change tracking
                                newOperation = { start: _.copy(cellPos, true), attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                                self.extendPropertiesWithTarget(newOperation, target);
                                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                            } else {
                                if (containsUnrestorableElements(cellPos)) { askUser = true; }
                                if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(cellNode, allRangeMarkerSelector)); }
                                refreshTableGrid = true;
                                deleteForChangeTrackRequired = true;
                            }
                        }
                    }
                }

                if (changeTrack.isActiveChangeTracking() && deleteForChangeTrackRequired) {
                    newOperation = { start: tablePos, startGrid: startGrid, endGrid: endGrid };
                    self.extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.COLUMNS_DELETE, newOperation);
                }

                // Deleting the table explicitely, if all its content was removed
                if (deletedAllRows) {
                    // not setting attributes, if this table was inserted with change tracking by the current user
                    if (changeTrack.isActiveChangeTracking() && !changeTrack.isInsertNodeByCurrentAuthor(tableNode) && !changeTrack.allAreInsertedNodesByCurrentAuthor(allRows)) {
                        // Table is not (necessarily) marked with 'inserted', but all rows
                        // -> setting changes attribute at all rows, not at the table
                        _.each(allRows, function (row, index) {
                            var rowPosition = _.clone(tablePos);
                            rowPosition.push(index);
                            newOperation = { start: rowPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                        });
                    } else {
                        newOperation = { start: _.clone(tablePos) };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.DELETE, newOperation);
                        if (containsUnrestorableElements(tablePos)) { askUser = true; }
                        if (isRangeCheckRequired) { allRangeMarkerNodes = allRangeMarkerNodes.add(Position.collectElementsBySelector(tableNode, allRangeMarkerSelector)); }
                    }
                } else {
                    if (refreshTableGrid) {
                        // Setting new table grid attribute to table (but not, if change tracking is activated
                        tableGrid = _.clone(self.getTableStyles().getElementAttributes(tableNode).table.tableGrid);
                        tableGrid.splice(startGrid, endGrid - startGrid + 1);  // removing column(s) in tableGrid (automatically updated in table node)
                        newOperation = { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) };
                        self.extendPropertiesWithTarget(newOperation, target);
                        generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                        self.setRequiresElementFormattingUpdate(false);   // no call of implTableChanged -> attributes are already set in implSetAttributes
                    }
                }
            }

            // handling for all collected ranges
            if (allRangeMarkerNodes.length > 0) {
                self.getRangeMarker().handlePartlyDeletedRanges(allRangeMarkerNodes, generator, getValidInsertStartRangePosition());
            }

            // Asking the user, if he really wants to remove the columns, if it contains
            // at least one cell with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = self.showDeleteWarningDialog(
                    gt('Delete Columns'),
                    gt('Deleting the selected columns cannot be undone. Do you want to continue?')
                );
            } else {
                promise = $.when();
            }

            // delete columns on resolved promise
            return promise.done(doDeleteColumns);
        };

        /**
         * Editor API function to generate 'delete' operations for table cells.
         */
        this.deleteCells = function () {

            var // the selection object
                selection = self.getSelection(),
                isCellSelection = selection.getSelectionType() === 'cell',
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition(),
                localPos = null,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current element container node
                rootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            startPos.pop();  // removing character position and paragraph
            startPos.pop();
            endPos.pop();
            endPos.pop();

            var startCol = startPos.pop(),
                endCol = endPos.pop(),
                startRow = startPos.pop(),
                endRow = endPos.pop(),
                tablePos = _.clone(startPos),
                operations = [];

            for (var i = endRow; i >= startRow; i--) {

                var rowPosition = Position.appendNewIndex(tablePos, i),
                    localStartCol = startCol,
                    localEndCol = endCol;

                if (!isCellSelection && (i < endRow) && (i > startCol)) {
                    // removing complete rows
                    localStartCol = 0;
                    localEndCol = Position.getLastColumnIndexInRow(rootNode, rowPosition);
                }

                for (var j = localEndCol; j >= localStartCol; j--) {
                    localPos = Position.appendNewIndex(rowPosition, j);
                    newOperation = { name: Operations.DELETE, start: localPos };
                    self.extendPropertiesWithTarget(newOperation, target);
                    operations.push(newOperation);
                }

                // removing empty row
                var rowNode = Position.getDOMPosition(rootNode, rowPosition).node;
                if ($(rowNode).children().length === 0) {
                    localPos = Position.appendNewIndex(tablePos, i);
                    newOperation = { name: Operations.DELETE, start: localPos };
                    self.extendPropertiesWithTarget(newOperation, target);
                    operations.push(newOperation);
                }

                // checking if the table is empty
                var tableNode = Position.getDOMPosition(rootNode, tablePos).node;
                if (Table.getRowCount(tableNode) === 0) {
                    newOperation = { name: Operations.DELETE, start: _.clone(tablePos) };
                    self.extendPropertiesWithTarget(newOperation, target);
                    operations.push(newOperation);
                }
            }

            // apply the operations (undo group is created automatically)
            self.applyOperations(operations);

            // setting the cursor position
            selection.setTextSelection(self.getLastOperationEnd());
        };

        // public helper functions --------------------------------------------

        /**
         * Public helper function to make implDeleteText publicly available.
         * For parameters look at definition of 'implDeleteText'.
         */
        this.implDeleteText = function (startPosition, endPosition, options, target) {
            return implDeleteText(startPosition, endPosition, options, target);
        };

        // operation handler --------------------------------------------------

        /**
         * The handler for the delete operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the delete operation was successful.
         */
        this.deleteHandler = function (operation) {

            var // node info about the paragraph to be deleted
                nodeInfo = null,
                // attribute type of the start node
                type = null,
                // the undo manager
                undoManager = self.getUndoManager(),
                // generator for the undo/redo operations
                generator = undoManager.isUndoEnabled() ? self.createOperationsGenerator() : null,
                // if its header/footer editing, root node is header/footer node, otherwise editdiv
                // rootNode = self.getCurrentRootNode(operation.target),
                rootNode = self.getRootNode(operation.target),
                // undo operation for passed operation
                undoOperation = null;

            // undo/redo generation
            if (generator) {

                nodeInfo = Position.getDOMPosition(rootNode, operation.start, true);
                type = resolveElementType(nodeInfo.node);

                switch (type) {

                    case 'text':
                        var position = operation.start.slice(0, -1),
                            paragraph = Position.getCurrentParagraph(rootNode, position),
                            start = operation.start[operation.start.length - 1],
                            end = _.isArray(operation.end) ? operation.end[operation.end.length - 1] : start;

                        undoOperation = { start: start, end: end, target: operation.target, clear: true };
                        generator.generateParagraphChildOperations(paragraph, position, undoOperation);
                        undoManager.addUndo(generator.getOperations(), operation);
                        break;

                    case 'paragraph':
                        if (!DOM.isImplicitParagraphNode(nodeInfo.node)) {
                            generator.generateParagraphOperations(nodeInfo.node, operation.start, { target: operation.target });
                            undoManager.addUndo(generator.getOperations(), operation);
                        }
                        break;

                    case 'cell':
                        generator.generateTableCellOperations(nodeInfo.node, operation.start, { target: operation.target });
                        undoManager.addUndo(generator.getOperations(), operation);
                        break;

                    case 'row':
                        generator.generateTableRowOperations(nodeInfo.node, operation.start, { target: operation.target });
                        undoManager.addUndo(generator.getOperations(), operation);
                        break;

                    case 'table':
                        if (!DOM.isExceededSizeTableNode(nodeInfo.node)) {
                            generator.generateTableOperations(nodeInfo.node, operation.start, { target: operation.target }); // generate undo operations for the entire table
                            undoManager.addUndo(generator.getOperations(), operation);
                        }
                        if (DOM.isTableInTableNode(nodeInfo.node)) {
                            if (($(nodeInfo.node).next().length > 0) && (DOM.isImplicitParagraphNode($(nodeInfo.node).next()))) {
                                $(nodeInfo.node).next().css('height', '');  // making paragraph behind the table visible after removing the table
                            }
                        }
                        break;
                }
            }

            // finally calling the implementation function to delete the content
            if (operation.target) {
                return implDelete(operation.start, operation.end, operation.target);
            } else {
                return implDelete(operation.start, operation.end);
            }
        };

        /**
         * The handler for the delete column operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the delete operation was successful.
         */
        this.deleteColumnsHandler = function (operation) {

            var // the current root node
                rootNode = self.getRootNode(operation.target),
                // the table node
                table = Position.getTableElement(rootNode, operation.start),
                // the undo manager
                undoManager = self.getUndoManager();

            if (table) {

                if (undoManager.isUndoEnabled()) {

                    var allRows = DOM.getTableRows(table),
                        allCellRemovePositions = Table.getAllRemovePositions(allRows, operation.startGrid, operation.endGrid),
                        generator = self.createOperationsGenerator();

                    allRows.each(function (index) {

                        var rowPos = operation.start.concat([index]),
                            cells = $(this).children(),
                            oneRowCellArray =  allCellRemovePositions[index],
                            end = oneRowCellArray.pop(),
                            start = oneRowCellArray.pop();  // more than one cell might be deleted in a row

                        // start<0: no cell will be removed in this row
                        if (start >= 0) {

                            if (end < 0) {
                                // remove all cells until end of row
                                end = cells.length;
                            } else {
                                // closed range to half-open range
                                end = Math.min(end + 1, cells.length);
                            }

                            // generate operations for all covered cells
                            cells.slice(start, end).each(function (index) {
                                generator.generateTableCellOperations(this, rowPos.concat([start + index]), { target: operation.target });
                            });
                        }
                    });

                    undoManager.addUndo(generator.getOperations(), operation);
                }

                implDeleteColumns(operation.start, operation.startGrid, operation.endGrid, operation.target);
            }

            return true;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class DeleteOperationMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return DeleteOperationMixin;
});
