/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/textframework/components/searchhandler/searchhandler', [
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/view/labels',
    'io.ox/office/tk/object/triggerobject',
    'gettext!io.ox/office/textframework/main'
], function (AttributeUtils, Utils, Operations, DOM, Position, Labels, TriggerObject, gt) {

    'use strict';

    // class SearchHandler ====================================================

    /**
     * The search handler object.
     *
     * @param {TextApplication} app
     *  The application instance.
     *
     * @param {Object} model
     *  The text editor model.
     */
    function SearchHandler(app, model) {

        var // self reference for local functions
            self = this,
            // all text spans that are highlighted (for quick removal)
            highlightedSpans = [],
            // the results of a quick search as oxoPositions
            searchResults = [],
            // the current index of the searchResults array
            currentSearchResultId = 0,
            // indicates if the applied operations is due to a search&replace
            // TODO: piggyback the information onto the operation directly
            replaceOperation = false;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        function getCharacterStyles() {
            return model.getStyleCollection('character');
        }

        /**
         * Returns whether the document contains any highlighted ranges.
         */
        function hasHighlighting() {
            return highlightedSpans.length > 0;
        }

        /**
         * Helper function that generates a regular expression for a specified
         * search string.
         *
         * @param {String} str
         *  The query string.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.simpleString=false]
         *      If set to true, the regular expression will be generated directly
         *      with the specified string parameter. In this case the replacement
         *      value for the nodes inside the string ('\ufffc*') will not be used
         *      and those string elements will not be found. In this case the string,
         *      that include for example a floated drawing or a drawing place holder
         *      will not be found. This is used during replacement, so that those
         *      drawings are not automatically replaced. But it shall be possible
         *      to find those strings, that include such a drawing.
         *
         * @returns {RegExp}
         *  The generated regular expression object.
         */
        function generateRegExpForSearchString(str, options) {

            var // the string for creating the regular expression
                regExpString = '',
                // the loop iterator
                i = 0,
                // the length of the string
                len = 0,
                // whether only a simple string without special replacement characters shall be found
                simpleString = Utils.getBooleanOption(options, 'simpleString', false);

            //_.escapeRegExp for Bug 40404, search for \ or . or similar must work

            if (simpleString) {
                regExpString = _.escapeRegExp(str);
            } else {
                len = str.length;
                for (i = 0; i < len - 1; i++) { regExpString += _.escapeRegExp(str[i]) + '\\ufffc*'; }
                regExpString += _.escapeRegExp(str[len - 1]);
            }

            return new RegExp(regExpString, 'g');
        }

        /**
         * Helper function to check, whether there is a node inside the specified range, that hinders
         * this range from being replaced.
         *
         * @param {Number[]} startPosition
         *  The logical start position of the range.
         *
         * @param {Number[]} endPosition
         *  The logical end position of the range.
         *
         * @param {Number[]} exceptionList
         *  The list with the logical positions of those nodes, that hinder the range from being replaced.
         *  This list contains only the positions inside the paragraph, no array.
         *
         * @returns {Boolean}
         *  Whether the specified range contains an element that hinders it from being replaced.
         */
        function isReplaceableRange(startPosition, endPosition, exceptionList) {

            if (exceptionList.length === 0) { return true; }

            var // whether the specified range contains at least one element that cannot be replaced
                isReplaceable = true,
                // the start value inside the paragraph
                start = _.last(startPosition),
                // the end value inside the paragraph
                end = _.last(endPosition);

            // the exception list contains the logical positions of the not replaceable elements inside the paragraph,
            // only the numbers, not an array.
            Utils.iterateArray(exceptionList, function (number) {
                if (number > start && number < end) {
                    isReplaceable = false;
                    return Utils.BREAK;
                }
            });

            return isReplaceable;
        }

        /**
         * Comparing two search results, if they have the same target. This function also returns
         * 'true', if both results do not have a target.
         *
         * @param {Object} result1
         *  The first result object.
         *
         * @param {Object} result2
         *  The second result object.
         *
         * @returns {Boolean}
         *  Whether the two specified results have the same target or both have no target at all.
         */
        function hasSameTarget(result1, result2) {

            if (!result1.target && !result2.target) { return true; }

            if ((result1.target && !result2.target) || (!result1.target && result2.target)) { return false; }

            return result1.target === result2.target;
        }

        /**
         * Specifying those in-line nodes, that can be located inside a string and avoid, that the string
         * is found in a search. This is for example the case for in-line drawings. A floated drawing
         * or a placeholder for a drawing or a comment node do not disturb the search of a string.
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @returns {Boolean}
         *  Whether this is a node that hinders a string to be found in a search, if it is located inside
         *  the string.
         */
        function isAvoidFindingNode(node) {
            return DOM.isInlineDrawingNode(node);
        }

        /**
         * Sets the browser selection to the search result given by the id of the searchResults array
         *
         * @param {Number} id
         *  The id of search result to select.
         */
        function selectSearchResult(id) {

            var // the page layout object
                pageLayout = model.getPageLayout(),
                // the comment layer object
                commentLayer = model.getCommentLayer(),
                // the selection object
                selection = model.getSelection(),
                // the search result as logical positions
                result = searchResults[id],
                // if target available, root node for switching context between main document and marginal nodes
                $node = null;

            // Bug 28386: defer selection, otherwise Enter key event will delete selected search result
            model.executeDelayed(function () {

                // Important order:
                // 1. grabbing the focus
                // 2. evaluating and setting target, if necessary
                // 3. setting new text selection

                app.getView().grabFocus();

                if (result) {
                    currentSearchResultId = id;

                    if (model.getActiveTarget() && commentLayer.isCommentTarget(model.getActiveTarget())) {
                        commentLayer.deActivateCommentNode(model.getActiveTarget(), selection);
                    } else if (model.isHeaderFooterEditState()) {
                        pageLayout.leaveHeaderFooterEditMode();
                        selection.setNewRootNode(model.getNode()); // restore original rootNode
                        model.updateEditingHeaderFooterDebounced();
                    }

                    if (result.target) {
                        if (commentLayer.isCommentTarget(result.target)) {
                            commentLayer.activateCommentNode(commentLayer.getCommentRootNode(result.target), selection);
                        } else {
                            $node = pageLayout.getFirstMarginalByIdInDoc(result.target);
                            if (model.isHeaderFooterEditState()) {
                                if (result.target !== model.getActiveTarget()) {
                                    pageLayout.switchTargetNodeInHeaderMode($node);
                                }
                            } else {
                                pageLayout.enterHeaderFooterEditMode($node);
                            }
                        }
                    }

                    selection.setTextSelection(result.start, result.end);
                }
            }, undefined, 'Text: selectSearchResult');
        }

        function handleNoResult() {

            //reset text selection
            var selection = model.getSelection();
            selection.setTextSelection(selection.getStartPosition());

            //info
            app.getView().yell(Labels.NO_SEARCHRESULT_OPTIONS);

        }

        function handleAttrsAndCT(node) {
            var changeTrack = model.getChangeTrack();
            var attrs = AttributeUtils.getExplicitAttributes(node);
            if (changeTrack.isActiveChangeTracking()) {
                attrs.changes = { inserted: changeTrack.getChangeTrackInfo() };
            }
            return attrs;
        }

        // methods ------------------------------------------------------------

        /**
         * Removes all highlighting (e.g. from quick-search) from the document
         * and clears the search results.
         */
        this.removeHighlighting = function () {

            var // the documents page node
                rootNode = model.getNode();

            // remove highlighting and merge sibling text spans
            self.removeSpansHighlighting(highlightedSpans);
            highlightedSpans = [];

            // used for search and replace
            searchResults = [];
            currentSearchResultId = 0;

            // fade in entire document
            rootNode.removeClass('highlight');
        };

        /**
         * Removes highlighting (e.g. from quick-search) on assigned spans
         */
        this.removeSpansHighlighting = function (spans) {

            var // the documents character styles
                characterStyles = getCharacterStyles();

            // remove highlighting and merge sibling text spans
            _.each(spans, function (span) {
                characterStyles.setElementAttributes(span, { character: { highlight: null } }, { special: true });
                Utils.mergeSiblingTextSpans(span);
                Utils.mergeSiblingTextSpans(span, true);
            });
        };

        /**
         * Sets the browser selection to the next search result,
         * does nothing if no results are available.
         */
        this.selectNextSearchResult = function () {

            if (searchResults.length > 0) {
                var id = (currentSearchResultId < searchResults.length - 1) ? currentSearchResultId + 1 : 0;
                selectSearchResult(id);
            } else {
                handleNoResult();
            }
        };

        /**
         * Sets the browser selection to the previous search result,
         * does nothing if no results are available.
         */
        this.selectPreviousSearchResult = function () {

            if (searchResults.length > 0) {
                var id = (currentSearchResultId > 0) ? currentSearchResultId - 1 : searchResults.length - 1;
                selectSearchResult(id);
            } else {
                handleNoResult();
            }
        };

        /**
         * Replaces the search result defined by the given id with the replacement text.
         *
         * @param {Number} id
         *  The id of search result to replace.
         *
         * @param {String} replacement
         *  The replacement for the found text occurences.
         */
        this.replaceSearchResult = function (id, replacement) {

            var // the change track object
                changeTrack = model.getChangeTrack(),
                // the undo manager object
                undoManager = model.getUndoManager(),
                // the selection object
                selection = model.getSelection(),
                // the documents character styles
                characterStyles = getCharacterStyles(),
                // the search result as range of oxo positions
                currentResult = searchResults[id], result,
                // the delta of the searched and replaced text
                textDelta,
                // the highlighted text node
                node, position, startPosition, endPosition, fullLength,
                // whether the replacement string is empty
                isEmptyReplace = false,
                // the content length, that was inserted by the same author
                // with activated change tracking
                countInsertedBySameAuthor = 0,
                // the index inside the logical position that need to be updated
                changeIndex = 0,
                // currently active root node
                activeRootNode = model.getCurrentRootNode();

            // helper function to remove highlighting from a specified result
            function removeHighlighting(oneResult) {

                position = _.clone(oneResult.start);

                while (_.last(position) < _.last(oneResult.end)) {
                    // loop through the range and remove the highlight character style from the spans
                    node = Position.getSelectedElement(activeRootNode, position, 'span');
                    position[position.length - 1] += $(node).text().length || 1;
                    characterStyles.setElementAttributes(node, { character: { highlight: null } }, { special: true });
                }

            }

            // helper function to update the list of all search results
            function updateSearchResults(oneResult) {

                // remove replaced result from search results
                searchResults = _.without(searchResults, oneResult);

                if (currentSearchResultId > searchResults.length - 1) {
                    currentSearchResultId = 0;
                }

            }

            if (!currentResult) {
                return;
            }

            // handling search results that cannot be replaced, because the contain additional content,
            // like non in-line drawings or comments.
            if (!currentResult.isReplaceable) {
                // removing the highlighting without replacing the content
                removeHighlighting(currentResult);
                // removing the current result from the list of all results
                updateSearchResults(currentResult);
                // ... but do not replace this search result
                return;
            }

            startPosition = _.clone(currentResult.start);
            endPosition = _.clone(currentResult.end);
            fullLength = _.last(endPosition) - _.last(startPosition);
            replacement = _.isString(replacement) ? replacement : '';
            isEmptyReplace = replacement.length === 0;

            // calculating the difference of the length from the searched
            // and the replaced string. This is necessary to recalculate
            // the positions of the following results.
            textDelta = replacement.length - fullLength;

            // Fixing the text delta, if change tracking is active. In this case the delta is
            // typically the length of the replacement, because the old text is not removed, but
            // only marked for removal. But additionally it is possible, that the old text was
            // inserted before with activated change tracking, so that it will be really deleted.
            // Therefore textDelta need to be reduced by that content, that was inserted by the
            // same author.
            if (changeTrack.isActiveChangeTracking()) {
                // then remove the highlighting
                position = _.clone(currentResult.start);
                while (_.last(position) < _.last(currentResult.end)) {
                    // loop through the range and count the characters inserted by the same author
                    node = Position.getSelectedElement(activeRootNode, position, 'span');
                    countInsertedBySameAuthor = countInsertedBySameAuthor + changeTrack.isInsertNodeByCurrentAuthor(node) ? ($(node).text().length || 1) : 0;
                    position[position.length - 1] += $(node).text().length || 1;
                }
                textDelta = replacement.length - countInsertedBySameAuthor;
            }

            undoManager.enterUndoGroup(function () {
                // indicate that these operations should not trigger a remove of the highlighting
                // TODO: piggiback this onto the operations
                replaceOperation = true;

                // if we replace a text next to another higlighted text span,
                // we would inherit the highlighting if we do the replacement the common way.
                // In order to keep the span(s) which contains the text to replace,
                // we first add the replacement at the start position.
                // Taking care of empty replacement strings, task 30347
                if (!isEmptyReplace) {
                    var attrs = handleAttrsAndCT(Position.getSelectedElement(activeRootNode, startPosition, 'span'));
                    model.insertText(replacement, startPosition, attrs);
                    endPosition[endPosition.length - 1] += replacement.length;
                }

                // then remove the highlighting
                removeHighlighting(currentResult);

                // finally delete the previous text.
                startPosition[startPosition.length - 1] += replacement.length;
                selection.setTextSelection(startPosition, endPosition);
                model.deleteSelected();

                // reset indicator
                replaceOperation = false;
            });

            // update all positions that are in the same paragraph or table cell. Also handle
            // all following positions inside text frames, that are in the same paragraph (40911)
            changeIndex = currentResult.start.length - 1;

            for (var updateId = id + 1; updateId < searchResults.length; updateId++) {
                result = searchResults[updateId];
                if (result && Position.hasSameParentComponentAtSpecificIndex(currentResult.start, result.start, changeIndex) && (currentResult.start[changeIndex] < result.start[changeIndex]) && hasSameTarget(currentResult, result)) {
                    result.start[changeIndex] += textDelta;
                    result.end[changeIndex] += textDelta;
                }
            }

            // remove replaced result from search results
            updateSearchResults(currentResult);
        };

        /**
         * Replaces the current search result with the replacement text.
         *
         * @param {String} replacement
         *  The replacement for the found text occurences.
         */
        this.replaceSelectedSearchResult = function (replacement) {
            this.replaceSearchResult(currentSearchResultId, replacement);
            selectSearchResult(currentSearchResultId);
        };

        /**
         * Debounced quickSearch, will postpone its execution until 2 seconds
         * have elapsed since the last time it was invoked.
         *
         * @param {String} query
         *  The text that will be searched in the document.
         *
         *  @param {Number} [selectId]
         *  The id of search result to be selected after the search run.
         */
        this.debouncedQuickSearch = (function () {

            var lastQuery = null;
            var lastSelectId = null;

            // direct callback: called everytime the method Editor.debouncedQuickSearch() is called
            function storeParameters(query, selectId) {
                lastQuery = query;
                lastSelectId = selectId;
            }

            // deferred callback: called once after the timeout, uses the last parameters passed to direct callback
            function executeQuickSearch() {
                self.quickSearch(lastQuery, lastSelectId);
            }

            // create and return the debounced method Editor.debouncedQuickSearch()
            return model.createDebouncedMethod(storeParameters, executeQuickSearch, { delay: 2000 });

        }()); // Editor.debouncedQuickSearch()

        /**
         * Searches and highlights the passed text in the entire document.
         * If a selectId is given the corresponding search result is selected.
         *
         * @param {String} query
         *  The text that will be searched in the document.
         *
         *  @param {Number} [selectId]
         *  The id of search result to be selected after the search run.
         *
         * @returns {Boolean}
         *  Whether the passed text has been found in the document.
         */
        this.quickSearch = function (query, selectId) {

            var // the document root node
                rootNode = model.getNode(),
                // the documents character styles
                characterStyles = getCharacterStyles(),
                // helper array, used to store matches in target nodes
                targetSearchResults = [],
                // the regular expression object used to find the query string
                regexp = null;

            // remove old highlighting and clear previous search results
            this.removeHighlighting();

            // check input parameter
            if (!_.isString(query) || (query.length === 0)) {
                return false;
            }
            query = query.toLowerCase();

            regexp = generateRegExpForSearchString(query);

            // search in all paragraph nodes (also embedded in tables etc.)
            Utils.iterateSelectedDescendantNodes(rootNode, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {

                var // all text spans in the paragraph, as array
                    textSpans = [],
                    // the concatenated text from all text spans
                    elementText = '',
                    // all matching ranges of the query text in the complete paragraph text
                    matchingRanges = [], start = 0,
                    // information about a text span while iterating matching ranges
                    spanInfo = { index: 0, start: 0 },
                    // start, end position of the range
                    startPosition, endPosition,
                    // the match for the regular expression
                    arrMatch = null,
                    // the length of the found string
                    queryLength = 0,
                    // the target node of a found string
                    targetNode = null,
                    // the target string of a found string
                    target = '',
                    // a list with logical node positions for nodes, that do not avoid finding of
                    // queries, but avoid replacing of this strings, for example non-inline drawings
                    // (inline drawings even avoid finding of query)
                    // -> it is sufficient to save only the last position, because search happens
                    //    only inside one paragraph.
                    exceptionList = [];

                // adds information of the text span located at 'spanInfo.index' in the 'textSpans' array to 'spanInfo'
                function getTextSpanInfo() {
                    spanInfo.span = textSpans[spanInfo.index];
                    spanInfo.length = spanInfo.span ? spanInfo.span.firstChild.nodeValue.length : 0;
                }

                // goes to the next text span in the 'textSpans' array and updates all information in 'spanInfo'
                function getNextTextSpanInfo() {
                    spanInfo.index += 1;
                    spanInfo.start += spanInfo.length;
                    getTextSpanInfo();
                }

                // collect all non-empty text spans in the paragraph
                Position.iterateParagraphChildNodes(paragraph, function (node) {

                    var // the replacement string used for non text elements
                        replacementString = '';

                    // iterate child nodes...
                    if (DOM.isTextSpan(node)) {
                        // for spans add the text
                        textSpans.push(node);
                        elementText += $(node).text();
                    } else if (!DOM.isZeroLengthNode(node)) {
                        // for all but list labels, page breaks and drawing space makers (they don't have a length of 1) add a replacement (FFFC - Object replacement char)
                        // -> adding FFFD for in-line drawings, so that those drawings avoid finding text around it
                        replacementString = isAvoidFindingNode(node) ? '\ufffd' : '\ufffc';
                        textSpans.push(Utils.getDomNode(DOM.createTextSpan().text(replacementString)));
                        elementText += replacementString;
                        exceptionList.push(Position.getOxoPosition(paragraph, node, 0)[0]); // only the position inside the paragraph
                    }
                }, this, { allNodes: true });

                // replace all whitespace characters, and convert to lower case
                // for case-insensitive matching
                elementText = elementText.replace(/\s/g, ' ').toLowerCase();

                while ((arrMatch = regexp.exec(elementText))) {

                    start = arrMatch.index;
                    queryLength = arrMatch[0].length;

                    matchingRanges.push({ start: start, end: start + queryLength });

                    if (DOM.isMarginalNode(paragraph)) {

                        targetNode = DOM.getMarginalTargetNode(rootNode, paragraph);
                        target = DOM.getTargetContainerId(targetNode);

                        if (!DOM.isFirstMarginalInDocument(rootNode, targetNode, target)) {
                            return;
                        }
                        startPosition = Position.getOxoPosition(targetNode, paragraph, 0);
                        startPosition.push(start);
                        endPosition = Position.increaseLastIndex(startPosition, queryLength);
                        targetSearchResults.push({ start: startPosition, end: endPosition, target: target, isReplaceable: isReplaceableRange(startPosition, endPosition, exceptionList) });

                    } else if (!model.getCommentLayer().isEmpty() && DOM.isNodeInsideComment(paragraph)) {

                        targetNode = DOM.getSurroundingCommentNode(paragraph);
                        target = DOM.getTargetContainerId(targetNode);
                        startPosition = Position.getOxoPosition(targetNode, paragraph, 0);
                        startPosition.push(start);
                        endPosition = Position.increaseLastIndex(startPosition, queryLength);
                        targetSearchResults.push({ start: startPosition, end: endPosition, target: target, isReplaceable: isReplaceableRange(startPosition, endPosition, exceptionList) });

                    } else {
                        startPosition = Position.getOxoPosition(rootNode, paragraph, 0);
                        startPosition.push(start);
                        endPosition = Position.increaseLastIndex(startPosition, queryLength);
                        searchResults.push({ start: startPosition, end: endPosition, isReplaceable: isReplaceableRange(startPosition, endPosition, exceptionList) });
                    }

                    // continue after the currently handled text
                    start += queryLength;
                }

                // set highlighting to all occurrences
                getTextSpanInfo();
                _(matchingRanges).each(function (range) {

                    var newSpan = null;

                    // find first text span that contains text from current matching range
                    while (spanInfo.start + spanInfo.length <= range.start) {
                        getNextTextSpanInfo();
                    }

                    // process all text spans covered by the current matching range
                    while (spanInfo.start < range.end) {

                        // split beginning of text span not covered by the range
                        if (spanInfo.start < range.start) {
                            DOM.splitTextSpan(spanInfo.span, range.start - spanInfo.start);
                            // update spanInfo
                            spanInfo.length -= (range.start - spanInfo.start);
                            spanInfo.start = range.start;
                        }

                        // split end of text span NOT covered by the range
                        if (range.end < spanInfo.start + spanInfo.length) {
                            newSpan = DOM.splitTextSpan(spanInfo.span, range.end - spanInfo.start, { append: true });
                            // insert the new span into textSpans after the current span
                            textSpans.splice(spanInfo.index + 1, 0, newSpan[0]);
                            // update spanInfo
                            spanInfo.length = range.end - spanInfo.start;
                        }

                        // set highlighting to resulting text span and store it in the global list
                        characterStyles.setElementAttributes(spanInfo.span, { character: { highlight: true } }, { special: true });
                        highlightedSpans.push(spanInfo.span);

                        // go to next text span
                        getNextTextSpanInfo();
                    }

                }, this);

                // exit at a certain number of found ranges (for performance)
                if (highlightedSpans.length >= 100) {
                    return Utils.BREAK;
                }

            }, this);

            searchResults = _.union(searchResults, targetSearchResults);

            if (highlightedSpans.length > 0) {
                // fade out entire document
                rootNode.addClass('highlight');
                // make first highlighted text node visible
                //app.getView().scrollToChildNode(highlightedSpans[0]);

                // select the given search result
                // TODO: only if user does not change focus to other search elements, but there must be a better way!!!
                if (_.isNumber(selectId) && !$(Utils.getActiveElement()).parents('.group-container').children('[data-key="document/search/text"], [data-key="document/replace/text"]')) {
                    selectSearchResult(selectId);
                }
            } else {
                handleNoResult();
            }

            // return whether any text in the document matches the passed query text
            return hasHighlighting();
        };

        /**
         * Searches the passed text in the entire document and replaces it with
         * the replacement text
         *
         * @param {String} query
         *  The text that will be searched in the document.
         *
         * @param {String} replacement
         *  The replacement for the found text occurrences.
         *
         *  @param {Boolean} [matchCase=false]
         *  Determines if the search is case sensitive (optional, defaults to false).
         */
        this.searchAndReplaceAll = function (query, replacement, matchCase) {

            var // the selection object
                selection = model.getSelection(),
                // the undo manager object
                undoManager = model.getUndoManager(),
                // the change track object
                changeTrack = model.getChangeTrack(),
                // the documents root node
                rootNode = model.getNode(),
                // whether the replacement string is empty
                isEmptyReplace = false,
                // the length of the query and the replacement string
                queryLength = 0, replacementLength = 0,
                // the delta of the searched and replaced text
                textDelta = 0,
                // the initial text start position
                initialStart = selection.getStartPosition(),
                // the regular expression object used to find the query string
                regexp = null;

            // remove old quick search highlighting and clear previous search results
            this.removeHighlighting();

            // check input parameter
            if (!_.isString(query) || (query.length === 0)) { return; }

            query = matchCase ? query : query.toLowerCase();
            queryLength = query.length;
            replacement = _.isString(replacement) ? replacement : '';
            replacementLength = replacement.length;
            isEmptyReplace = replacementLength === 0;

            // creating the regular expression to find the query string
            // using 'simpleString', so that string with special character '\ufffc' is NOT found
            regexp = generateRegExpForSearchString(query, { simpleString: true });

            // search in all paragraph nodes (also embedded in tables etc.)
            undoManager.enterUndoGroup(function () {

                var // the deferred to keep the undo group open until it is resolved or rejected
                    undoDef = $.Deferred(),
                    // the apply actions deferred
                    applyDef = null,
                    // the operations generator
                    generator = model.createOperationsGenerator(),
                    // the collector for all paragraphs inside text frames
                    paragraphInTextFrameCollector = [],
                    // the collector for all paragraphs not inside text frames
                    paragraphCollector = [];

                // show a nice message with cancel button
                app.getView().enterBusy({
                    cancelHandler: function () { if (applyDef && applyDef.abort) { applyDef.abort(); } },
                    warningLabel: /*#. message shown while replacing text in entire document */ gt('Replacing text.')
                });

                // collecting all paragraphs. All paragraphs inside text frames need to be handled first (40911)
                Utils.iterateSelectedDescendantNodes(rootNode, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                    if (DOM.isNodeInsideTextFrame(paragraph)) {
                        paragraphInTextFrameCollector.push(paragraph); // paragraphs inside text frames need to be handled first
                    } else {
                        paragraphCollector.push(paragraph);
                    }
                });

                // merging both collectors for following iteration
                paragraphCollector = paragraphInTextFrameCollector.concat(paragraphCollector);

                // ... an iterating over all paragraphs
                _.each(paragraphCollector, function (paragraph) {

                    var // the concatenated text from all text spans
                        elementText = '', start = 0,
                        // start, end position of the matching range
                        startPosition, endPosition,
                        // the offset to apply to the start and end position of the range
                        offset = 0,
                        // attribute object for the generated operations
                        attrs = null,
                        // the node position before replacing content
                        origPosition = null,
                        // the node at the original position
                        node = null,
                        // whether the original content was inserted by the same author in change tracking mode
                        insertedBySameAuthor = false,
                        // object containing properties generated for operation
                        operationProperties = {},
                        // the match for the regular expression
                        arrMatch = null,
                        // the marginal node surrounding the found element
                        targetNode = null,
                        // the target node for the search
                        target = null;

                    // collect all non-empty text spans in the paragraph
                    Position.iterateParagraphChildNodes(paragraph, function (node) {

                        var // the replacement string used for non text elements
                            replacementString = '\ufffc';

                        // iterate child nodes...
                        if (DOM.isTextSpan(node)) {
                            // for spans add the text
                            elementText += $(node).text();
                        } else if (!DOM.isZeroLengthNode(node)) {
                            // for all but list labels, page breaks and drawing space makers (they don't have a length of 1) add a replacement (FFFC - Object replacement char)
                            elementText += replacementString;
                        }
                    }, this, { allNodes: true });

                    // replace all whitespace characters, and convert to lower case
                    // for case-insensitive matching
                    elementText = elementText.replace(/\s/g, ' ');

                    if (!matchCase) {
                        elementText = elementText.toLowerCase();
                    }

                    if (DOM.isMarginalNode(paragraph)) {
                        targetNode = DOM.getMarginalTargetNode(rootNode, paragraph);
                        target = DOM.getTargetContainerId(targetNode);

                        if (!DOM.isFirstMarginalInDocument(rootNode, targetNode, target)) {
                            return;
                        }

                    } else if (!model.getCommentLayer().isEmpty() && DOM.isNodeInsideComment(paragraph)) {

                        targetNode = DOM.getSurroundingCommentNode(paragraph);
                        target = DOM.getTargetContainerId(targetNode);

                    }

                    while ((arrMatch = regexp.exec(elementText))) {

                        start = arrMatch.index;
                        queryLength = arrMatch[0].length;
                        node = null;

                        startPosition = Position.getOxoPosition(targetNode || rootNode, paragraph, 0);

                        origPosition = _.clone(startPosition);
                        origPosition.push(start);
                        startPosition.push(start + offset);
                        endPosition = Position.increaseLastIndex(startPosition, queryLength - 1);

                        // if change tracking is active, the node need to be checked, if it
                        // was inserted by the same author before (sufficient to use the start position?)
                        insertedBySameAuthor = false;
                        if (changeTrack.isActiveChangeTracking()) {
                            node = Position.getSelectedElement(targetNode || rootNode, origPosition, 'span');
                            insertedBySameAuthor = changeTrack.isInsertNodeByCurrentAuthor(node);
                        }

                        if (changeTrack.isActiveChangeTracking() && !insertedBySameAuthor) {
                            operationProperties = { start: startPosition, end: endPosition, attrs: { changes: { removed: changeTrack.getChangeTrackInfo() } } };
                            model.extendPropertiesWithTarget(operationProperties, target);
                            generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);
                            // the delta of the searched and replaced text -> in this case no text was deleted
                            textDelta = replacementLength;
                        } else {
                            operationProperties = { start: startPosition, end: endPosition };
                            model.extendPropertiesWithTarget(operationProperties, target);
                            generator.generateOperation(Operations.DELETE, operationProperties);
                            // the delta of the searched and replaced text -> in this case, replaced text was deleted
                            textDelta = replacementLength - queryLength;
                        }

                        if (!isEmptyReplace) {
                            if (!node) { node = Position.getSelectedElement(targetNode || rootNode, startPosition, 'span'); }
                            //set the attributes of the first letter on the replacement
                            attrs = handleAttrsAndCT(node);

                            operationProperties = { text: replacement, start: startPosition, attrs: attrs };
                            model.extendPropertiesWithTarget(operationProperties, target);
                            generator.generateOperation(Operations.TEXT_INSERT, operationProperties);
                        }

                        // from elementText we get the occurences before any replacement takes place,
                        // all replacements but the first one need to handle the text offset due to
                        // the replacement offsets of the previous occurences.
                        offset += textDelta;

                        // continue after the currently handled text
                        start += queryLength;
                    }

                }, this);

                // apply generated operations
                applyDef = model.applyOperations(generator, { async: true })
                // set cursor behind last replacement text
                .done(function () {
                    selection.setTextSelection(initialStart);
                })
                // leave busy state and close undo group
                .always(function () {
                    app.getView().leaveBusy().grabFocus();
                    undoDef.resolve();
                })
                // add progress handling
                .progress(function (progress) {
                    app.getView().updateBusyProgress(progress);
                });

                return undoDef.promise();

            }, this); // enterUndoGroup
        };

        /**
         * removing search highlighting
         */
        this.clearHighlighting = function () {
            // removing search highlighting
            if (hasHighlighting()) {
                self.removeHighlighting();
            }
        };

        /**
         * clear search results and trigger the debounced quick search
         */
        this.reset = function () {
            if (!replaceOperation) {
                // clear search results and trigger the debounced quick search
                self.removeHighlighting();
                if (app.getView().isSearchActive()) {
                    self.debouncedQuickSearch(app.getView().getSearchSettings().query);
                }
            }
        };

        // initialization -----------------------------------------------------

        self.listenTo(model, 'operations:before', self.reset);

    } // class SearchHandler

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: SearchHandler });

});
