/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/formrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/popup/tablecolumnmenu',
    'io.ox/office/spreadsheet/view/popup/validationlistmenu'
], function (Utils, Forms, IteratorUtils, TriggerObject, TimerMixin, SheetUtils, RenderUtils, TableColumnMenu, ValidationListMenu) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;

    // class FormRenderer =====================================================

    /**
     * Renders the form controls (especially cell drop-down buttons) into the
     * DOM layers of a single grid pane.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this form layer renderer.
     */
    function FormRenderer(gridPane) {

        // self reference
        var self = this;

        // the spreadsheet model and view
        var docView = gridPane.getDocView();
        var docModel = docView.getDocModel();

        // the form control layer (container for drop-down buttons and other controls)
        var formLayerNode = gridPane.createLayerNode('form-layer');

        // the cell range covered by the layer nodes in the sheet area
        var layerRange = null;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Removes all cell drop-down button elements of the specified type.
         *
         * @param {String} menuType
         *  The type specifier of the button elements, as passed to the method
         *  FormRenderer.createCellButton().
         */
        function removeCellButtons(menuType) {
            formLayerNode.find('>.cell-dropdown-button[data-type="' + menuType + '"]').remove();
        }

        /**
         * Creates a drop-down button element intended to be attached to the
         * trailing border of the specified cell, and inserts it into the form
         * control layer.
         *
         * @param {String} menuType
         *  The type specifier of a pop-up menu that has been registered with
         *  the method GridPane.registerCellMenu(). If the created button will
         *  be clicked, the registered menu will be shown.
         *
         * @param {Address} address
         *  The address of the cell the button will be associated to.
         *
         * @param {Object} rectangle
         *  The position of the cell range the button is attached to (pixels).
         *  This rectangle may be larger than the area covered by the specified
         *  cell, e.g. if the cell is part of a merged range, and the drop-down
         *  button has to be displayed in the bottom-right corner of the range.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.icon]
         *      A specific icon to be shown in the button, instead of the
         *      default drop-down caret icon.
         *  @param {String} [options.tooltip]
         *      A tool tip text to be shown when the mouse hovers the drop-down
         *      button.
         *  @param {Boolean} [options.outside=false]
         *      If set to true, the button will be positioned outside the cell
         *      rectangle. By default, the button will be positioned inside.
         *  @param {Number} [options.distance=0]
         *      Distance of the button to the border of the cell rectangle.
         *  @param {Any} [options.userData]
         *      Any additional data that will be added to the jQuery data map
         *      of the button element, and that can be used during the
         *      initialization phase or event handling of pop-up menus
         *      associated to the drop-down button.
         *
         * @returns {jQuery}
         *  The created and inserted button element, as jQuery object. The
         *  jQuery data map of the button element will contain the following
         *  properties:
         *  - {Address} address
         *      The passed cell address.
         *  - {Number} anchorWidth
         *      The effective total width of the passed rectangle and the
         *      button rectangle.
         *  - {Object} [userData]
         *      The user data passed with the option 'userData'.
         */
        function createCellButton(menuType, address, rectangle, options) {

            // whether to place the button element outside the cell rectangle
            var outside = Utils.getBooleanOption(options, 'outside', false);
            // distance of the button element to the cell rectangle
            var distance = Utils.getIntegerOption(options, 'distance', 0, 0);
            // additional properties for the jQuery data map
            var userData = Utils.getOption(options, 'userData');
            // outer size of the button element (according to cell height)
            var size = Utils.minMax(rectangle.height - 1, 18, 30);
            // size of the sheet area hidden by frozen split
            var hiddenSize = gridPane.getHiddenSize();
            // position of the button in the sheet area
            var buttonRect = {
                left: Math.max(rectangle.left + rectangle.width + (outside ? distance : -(size + 1)), hiddenSize.width),
                top: Math.max(rectangle.top + rectangle.height - size - 1, hiddenSize.height),
                width: size,
                height: size
            };
            // the anchor rectangle to be used for attached drop-down menu
            var anchorRect = Utils.getBoundingRectangle(rectangle, buttonRect);

            // create button mark-up
            var markup = '<div class="cell-dropdown-button skip-tracking" tabindex="-1" data-type="' + menuType + '" data-address="' + address.key() + '"';
            markup += ' style="' + gridPane.getLayerRectangleStyleMarkup(buttonRect) + 'line-height:' + (buttonRect.height - 2) + 'px;">';
            markup += Forms.createIconMarkup(Utils.getStringOption(options, 'icon', 'fa-caret-down')) + '</div>';

            // create and return the button element
            var buttonNode = $(markup).data({ address: address, anchorWidth: anchorRect.width, userData: userData });
            Forms.setToolTip(buttonNode, options);
            formLayerNode.append(buttonNode);
            return buttonNode;
        }

        /**
         * Creates the form controls (cell drop-down buttons) currently visible
         * in this grid pane.
         */
        var renderFormControlsDebounced = this.createDebouncedMethod(_.noop, function () {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) {
                formLayerNode.empty();
                return;
            }

            // remove all filter buttons, do not render them in locked sheets
            removeCellButtons('table');
            if (docView.isSheetLocked()) { return; }

            // the collections of the active sheet
            var colCollection = docView.getColCollection();
            var rowCollection = docView.getRowCollection();

            // create drop-down buttons for all visible table ranges with activated filter/sorting
            docView.getTableCollection().findTables(layerRange, { active: true }).forEach(function (tableModel) {

                // the cell range covered by the table
                var tableRange = tableModel.getRange();
                // the settings of the header row
                var rowDesc = rowCollection.getEntry(tableRange.start[1]);

                // do nothing if the header row is outside the layer range, or hidden
                if ((tableRange.start[1] < layerRange.start[1]) || (rowDesc.size === 0)) { return; }

                // visit all visible columns of the table in the layer range
                var colInterval = tableRange.colInterval().intersect(layerRange.colInterval());
                var colIt = colCollection.createIterator(colInterval, { visible: true });
                IteratorUtils.forEach(colIt, function (colDesc) {

                    // relative table column index
                    var tableCol = colDesc.index - tableRange.start[0];

                    // bug 36152: do not render buttons covered by merged ranges
                    if (tableModel.getColumnAttributeSet(tableCol, true).filter.hideMergedButton) { return; }

                    // bug 36152: visible drop-down button in a merged range is placed in the last column of the
                    // merged range, but is associated to (shows cell data of) the first column in the merged range
                    while ((tableCol > 0) && tableModel.getColumnAttributeSet(tableCol - 1, true).filter.hideMergedButton) {
                        tableCol -= 1;
                    }

                    var address = new Address(colDesc.index, rowDesc.index);
                    var rectangle = { left: colDesc.offset, top: rowDesc.offset, width: colDesc.size, height: rowDesc.size };

                    createCellButton('table', address, rectangle, {
                        tooltip: TableColumnMenu.BUTTON_TOOLTIP,
                        icon: tableModel.isColumnFiltered(tableCol) ? 'fa-filter' : null,
                        userData: { tableName: tableModel.getName(), tableCol: tableCol }
                    });
                });
            });
        });

        /**
         * Handles a click on an embedded cell drop-down button, and triggers a
         * 'cellbutton:click' event.
         *
         * @param {jQuery.Event} event
         *  The jQuery click event.
         */
        function cellButtonClickHandler(event) {
            self.trigger('cellbutton:click', $(event.currentTarget));
        }

        /**
         * Creates or updates the drop-down button for list validation, after
         * the selection layer has been rendered.
         */
        function renderCellSelectionHandler() {

            // remove the old button element
            removeCellButtons('validation');

            // do not show in read-only mode, or while using auto-fill feature, or in locked cells
            if (!docModel.getEditMode() || docView.getSheetViewAttribute('autoFillData') || docView.isActiveCellLocked()) { return; }

            // the current selection in the document view
            var selection = docView.getSelection();
            // active cell in the current selection
            var activeCell = selection.address;
            // active cell, expanded to a merged range
            var activeRange = docView.getMergeCollection().expandRangeToMergedRanges(new Range(activeCell));

            // check that the active range is inside the row interval covered by this grid pane (needed to prevent
            // painting the upper part of the button, if cell is located in the top row of the lower part of a frozen split)
            var availableInterval = gridPane.getRowHeaderPane().getAvailableInterval();
            if (!availableInterval || (activeRange.end[1] < availableInterval.first)) { return; }

            // validation settings of the active cell in the selection
            var validationSettings = docView.getValidationCollection().getValidationSettings(activeCell);

            // check that the active cell provides a validation drop-down list
            if (!validationSettings || !/^(source|list)$/.test(validationSettings.attributes.type) || !validationSettings.attributes.showDropDown) { return; }

            // whether the right border of the active cell touches the right border of any selection range
            var rightBorder = selection.ranges.some(function (range) { return activeRange.end[0] === range.end[0]; });

            // create and insert the button node
            createCellButton('validation', activeCell, docView.getRangeRectangle(activeRange), {
                tooltip: ValidationListMenu.BUTTON_TOOLTIP,
                outside: true,
                distance: rightBorder ? 2 : 0
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Changes the layers according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('FormRenderer.setLayerRange()', function (newLayerRange) {
            layerRange = newLayerRange;
            renderFormControlsDebounced();
        });

        /**
         * Resets this renderer, clears the DOM layer nodes.
         */
        this.hideLayerRange = function () {
            layerRange = null;
            formLayerNode.empty();
        };

        // public methods -----------------------------------------------------

        /**
         * Returns all cell drop-down buttons attached to the specified cell.
         *
         * @param {Address} address
         *  The address of the cell whose drop-down menu buttons will be
         *  returned.
         *
         * @returns {jQuery}
         *  A jQuery collection with all cell drop-down menu buttons at the
         *  passed cell address.
         */
        this.getCellButtons = function (address) {
            return formLayerNode.find('>.cell-dropdown-button[data-address="' + address.key() + '"]');
        };

        // initialization -----------------------------------------------------

        // forward clicks on cell drop-down buttons, suppress double-clicks
        formLayerNode.on({ click: cellButtonClickHandler, dblclick: false }, '.cell-dropdown-button');

        // update layers according to changed contents (only, if the grid pane is visible)
        gridPane.listenToWhenVisible(this, docView, 'insert:table change:table delete:table', renderFormControlsDebounced);

        // render validation drop-down button after selection changes
        gridPane.listenToWhenVisible(this, gridPane, 'render:cellselection', renderCellSelectionHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = gridPane = docModel = docView = formLayerNode = null;
        });

    } // class FormRenderer

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: FormRenderer });

});
