/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/tablecolumnmenu', [
    'io.ox/office/baseframework/view/popup/compoundmenu',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/control/discretefiltergroup',
    'gettext!io.ox/office/spreadsheet/main'
], function (CompoundMenu, Labels, DiscreteFilterGroup, gt) {

    'use strict';

    // class TableColumnMenu ==================================================

    /**
     * A drop-down menu for a header cell in a table range, containing options
     * for sorting and filtering.
     *
     * @constructor
     *
     * @extends CompoundMenu
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this instance.
     */
    function TableColumnMenu(docView) {

        var // the check list control for discrete filters
            filterGroup = null,

            // the name of the table range represented by this menu
            tableName = null,

            // the column index in the table range
            tableCol = 0;

        // base constructor ---------------------------------------------------

        CompoundMenu.call(this, docView, {
            anchorAlign: 'trailing',
            actionKey: 'table/column/attributes',
            actionValue: getActionValue,
            enableActionHandler: isActionEnabled
        });

        // private methods ----------------------------------------------------

        /**
         * Returns a result object with the current filter settings as expected
         * by the controller item bound to the OK button of this menu.
         *
         * @returns {Object}
         *  The action value containing the current filter settings, with the
         *  following properties:
         *  - {String} tableName
         *      The name of the table range represented by this menu.
         *  - {Number} tableCol
         *      The column index in the table range.
         *  - {Object} attributes
         *      The attribute set with filter and sorting attributes to be set
         *      for the table column.
         */
        function getActionValue() {

            // the selected entries for a discrete filter (null, if all entries are selected)
            var entries = filterGroup.getValue();
            if (entries.length === filterGroup.getOptionButtons().length) { entries = null; }

            // create discrete filter, or clear filter (for undefined entries)
            var filterAttrs = entries ? { type: 'discrete', entries: entries } : { type: 'none' };

            // return the entire action value
            return { tableName: tableName, tableCol: tableCol, attributes: { filter: filterAttrs } };
        }

        /**
         * Returns whether the OK button is currently enabled. It is not
         * allowed to apply a discrete filter without any visible entries.
         */
        function isActionEnabled() {
            var entries = filterGroup.getValue();
            return _.isArray(entries) && (entries.length > 0);
        }

        // public methods -----------------------------------------------------

        /**
         * Collects and inserts the cell entries of a table column into this
         * menu, to be used to define discrete filters. Called by grid pane
         * instances from the click handler of cell drop-down buttons.
         *
         * @param {Address} address
         *  The address of the header cell of the table column.
         *
         * @param {Object} userData
         *  The user data stored during creation of the drop-down button
         *  associated to this menu. Contains the properties 'tableName' (name
         *  of the table model in the active sheet to be filtered), and
         *  'tableCol' (zero-based column index relative to the table range).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the menu has been initialized.
         */
        this.initialize = function (address, userData) {
            tableName = userData.tableName;
            tableCol = userData.tableCol;
            return filterGroup.initialize(tableName, tableCol);
        };

        // initialization -----------------------------------------------------

        this.addSectionLabel(Labels.FILTER_HEADER_LABEL)
            .addGroup(null, filterGroup = new DiscreteFilterGroup(docView));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView = filterGroup = null;
        });

    } // class TableColumnMenu

    // constants --------------------------------------------------------------

    /**
     * Tool tip for cell drop-down buttons attached to an instance of the class
     * TableColumnMenu.
     *
     * @constant
     */
    TableColumnMenu.BUTTON_TOOLTIP = gt.pgettext('filter', 'Select values for filtering');

    // exports ================================================================

    // derive this class from class CompoundMenu
    return CompoundMenu.extend({ constructor: TableColumnMenu });

});
