/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/namedrangeslayermenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/label',
    'io.ox/office/tk/control/button',
    'io.ox/office/baseframework/view/popup/layermenu',
    'io.ox/office/spreadsheet/view/control/namedrangeitem',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Label, Button, LayerMenu, NamedRangeItem, gt) {

    'use strict';

    // the names of all events for defined names triggered by the document model
    var DEFINED_NAME_EVENTS = 'insert:name change:name delete:name';

    // class NamedRangesLayerMenu =============================================

    /**
     * A layer menu with all named ranges.
     *
     * @constructor
     *
     * @extends LayerMenu
     */
    function NamedRangesLayerMenu(docView) {

        // self reference
        var self = this;

        // the document model
        var docModel = docView.getDocModel();

        // base constructor ---------------------------------------------------

        LayerMenu.call(this, docView, /*#. dialog title */ gt.pgettext('named ranges', 'Named Ranges'), {
            classes: 'defined-names',
            rootContainerNode: docView.getApp().getWindowNode()
        });

        // private methods ----------------------------------------------------

        function createAllGroups() {

            // the models of all visible defined names in the document (TODO: sheet-local names)
            var nameModels = docModel.getAllNames({ skipHidden: true, skipLocal: true }).sort(function (nameModel1, nameModel2) {
                return nameModel1.getLabel().localeCompare(nameModel2.getLabel());
            });

            // add a special entry, if no names are available at all
            if (nameModels.length === 0) {
                self.addGroup(null, new Label({ label: gt.pgettext('named ranges', 'No named ranges defined.') }));
                self.addSeparator();
            } else {
                nameModels.forEach(function addListEntry(nameModel) {
                    self.addGroup(null, new NamedRangeItem(docView, nameModel));
                    self.addSeparator();
                });
            }

            // add an extra button to define new named ranges
            self.addGroup('name/insert/dialog', new Button({ icon: 'fa-plus', label: gt.pgettext('named ranges', 'Add new named range') }));
        }

        var createAllGroupsDebounced = this.createDebouncedMethod(_.noop, createAllGroups, { delay: 10 });

        /**
         * Moves the menu to a default position when it has been shown for the
         * first time.
         */
        function setDefaultPosition() {
            var pos = Utils.getNodePositionInPage(docView.getAppPaneNode());
            self.setNodePosition(pos.left + (pos.width - self.getNode().outerWidth()) - 40, pos.top - 20);
        }

        // initialization -----------------------------------------------------

        // add event handlers for the menu
        this.on('popup:beforeshow', function () {
            createAllGroups();
            self.listenTo(docModel, DEFINED_NAME_EVENTS, function () {
                self.destroyAllGroups();
                createAllGroupsDebounced();
            });
        });

        // delete all menu entries when hiding the menu (entries are event listeners)
        this.on('popup:hide', function () {
            self.destroyAllGroups();
            self.stopListeningTo(docModel, DEFINED_NAME_EVENTS);
        });

        // set default position on first display of the menu
        this.one('popup:show', setDefaultPosition);

        // hide the menu automatically in read-only mode
        this.listenTo(docModel, 'change:editmode', function (event, editMode) {
            if (!editMode) { self.hide(); }
        });

        // destroy all class members
        this.registerDestructor(function () {
            self = docModel = docView = null;
        });

    } // class NamedRangesLayerMenu

    // exports ================================================================

    // derive this class from class LayerMenu
    return LayerMenu.extend({ constructor: NamedRangesLayerMenu });

});
