/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/celleditmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/tk/popup/tooltip',
    'io.ox/office/tk/popup/listmenu',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/sheetselection',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/view/popup/signaturetooltip',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, KeyCodes, Forms, DateUtils, IteratorUtils, Tracking, ToolTip, ListMenu, Color, AttributeUtils, SheetUtils, SheetSelection, TokenArray, SignatureToolTip, gt) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Range = SheetUtils.Range;

    // key codes of all keys used to navigate in text area, or in the sheet (quick edit mode)
    var NAVIGATION_KEYCODES = Utils.makeSet([
        KeyCodes.LEFT_ARROW,
        KeyCodes.RIGHT_ARROW,
        KeyCodes.UP_ARROW,
        KeyCodes.DOWN_ARROW,
        KeyCodes.PAGE_UP,
        KeyCodes.PAGE_DOWN,
        KeyCodes.HOME,
        KeyCodes.END
    ]);

    // mix-in class CellEditMixin =============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides extensions for
     * the cell in-place edit mode.
     *
     * @constructor
     */
    function CellEditMixin() {

        // self reference (spreadsheet view instance)
        var self = this;

        // the spreadsheet model, and other model objects
        var docModel = this.getDocModel();
        var undoManager = docModel.getUndoManager();
        var fontCollection = docModel.getFontCollection();
        var numberFormatter = docModel.getNumberFormatter();
        var formulaResource = docModel.getFormulaResource();
        var formulaGrammar = docModel.getFormulaGrammar('ui');
        var formulaParser = docModel.getFormulaParser();

        // the preferred date format for cell edit mode
        var DATE_FORMAT = numberFormatter.getPresetCode(14);
        // the preferred time format for cell edit mode
        var TIME_FORMAT = numberFormatter.getTimeCode({ seconds: true });

        // the underlay node containing highlighting for formulas
        var textUnderlayNode = $('<div class="underlay">');
        // the text area control
        var textAreaNode = $(Forms.createTextAreaMarkup());
        // the container element for the text underlay node and the text area control
        var textAreaContainer = $('<div class="textarea-container" data-focus-role="textarea">').append(textUnderlayNode, textAreaNode);

        // tooltip attached to the text area for function/parameter help
        var signatureToolTip = new SignatureToolTip(this, textAreaNode);
        // drop-down list attached to the text area (for function names)
        var textAreaListMenu = new ListMenu({ anchor: textAreaNode, autoFocus: false, focusableNodes: textAreaNode });
        // tooltip attached to the text area drop-down list (for the selected function)
        var textAreaListItemToolTip = new ToolTip({ anchor: textAreaListMenu.getNode(), anchorBorder: 'right left', anchorAlign: 'center', anchorAlignBox: getTextAreaListButtonNode });
        // unfiltered function entries for the auto-completion list menu
        var functionEntries = null;

        // whether the in-place cell edit mode is active
        var editActive = false;
        // the grid pane instance where in-place cell edit mode is active
        var editGridPane = null;
        // the model of the sheet containing the edited cell
        var sheetModel = null;
        // the cell collection of the sheet containing the edited cell
        var cellCollection = null;
        // the cell address of the edited cell
        var editAddress = null;
        // the cell attribute set of the edited cell
        var cellAttributeSet = null;
        // the effective font settings for the edited cell
        var fontDesc = null;
        // the effective line height of the edited cell, in pixels
        var lineHeight = null;
        // the token descriptors received form the formula parser for the current formula in the text area
        var tokenDescs = null;
        // the token array representing the current formula in the text area
        var tokenArray = null;

        // local undo/redo information while cell edit mode is active
        var undoStack = null;
        // pointer into the local undo/redo stack
        var undoIndex = 0;
        // additional attributes applied locally at the text area
        var editAttributeSet = null;

        // initial text at the beginning of the edit mode
        var originalText = null;
        // the text at the time the text area has been updated the last time
        var prevEditText = null;
        // collected entries for type-ahead suggestions
        var typeAheadEntries = null;

        // quick edit mode (true), or text cursor mode (false)
        var quickEditMode = null;
        // the formula expression, if the input string starts with specific characters
        var formulaString = null;
        // selection offset into the formula string (1, if formula starts with equality sign; or 0 for plus/minus)
        var formulaOffset = 0;
        // unique identifier of the current range highlighting mode for formulas
        var highlightUid = null;
        // whether a matrix formula is currently edited
        var matrixFormula = false;
        // whether to show auto-completion list for a function or defined name
        var funcAutoComplete = false;
        // token descriptor for function/name auto-completion
        var funcAutoTokenDesc = null;
        // whether to keep auto-completion list open although focus is not in text area
        var stickyAutoComplete = false;
        // how many closing parenthesis entered manually will be ignored
        var funcAutoCloseData = [];
        // the text position of the current range address while in range selection mode
        var rangeSelection = null;

        // private methods ----------------------------------------------------

        /**
         * Returns the initial text for the in-place cell edit mode. Returns
         * the formula expression for formula cells, otherwise the cell value
         * formatted with an appropriate default number format.
         *
         * @returns {String}
         *  The initial text for the in-place cell edit mode.
         */
        function getTextForCellEditMode() {

            // formulas: hide formula if the cell contains the hidden attribute
            var formula = cellCollection.getCellFormula(editAddress, 'ui');
            if (_.isString(formula)) {
                var cellAttrs = cellCollection.getCellAttributeSet(editAddress).cell;
                return (self.isSheetLocked() && cellAttrs.hidden) ? '' : ('=' + formula);
            }

            // the result value of the cell to be formatted
            var value = cellCollection.getCellValue(editAddress);

            // strings: use plain unformatted string for editing (bug 34421: add an apostrophe if necessary)
            if (_.isString(value) && (value.length > 0)) {
                return ((/^['=]/).test(value) || !_.isString(numberFormatter.parseValue(value))) ? ('\'' + value) : value;
            }

            // booleans: use plain boolean literal for editing
            if (_.isBoolean(value)) {
                return formulaGrammar.getBooleanName(value);
            }

            // error codes: use plain error code literal for editing
            if (value instanceof ErrorCode) {
                return formulaGrammar.getErrorName(value);
            }

            // numbers: use appropriate number representation according to number format category
            if (_.isNumber(value) && isFinite(value)) {

                // the resulting formatted value
                var formatted = null;

                // process different format categories
                var category = cellCollection.getCellParsedFormat(editAddress).category;
                switch (category) {

                    // percent: multiply by 100, add percent sign without whitespace
                    case 'percent':
                        value *= 100; // may become infinite
                        formatted = (isFinite(value) ? numberFormatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT) : '') + '%';
                        break;

                    // automatically show date and/or time, according to the number
                    case 'date':
                    case 'time':
                    case 'datetime':

                        // number of milliseconds
                        var milliSecs = Math.round(value * DateUtils.MSEC_PER_DAY);
                        // number of days
                        var date = Math.floor(milliSecs / DateUtils.MSEC_PER_DAY);
                        // number of milliseconds in the day
                        var time = Math.floor(milliSecs % DateUtils.MSEC_PER_DAY);
                        // whether to add the date and time part to the result
                        var showDate = (category !== 'time') || (date !== 0);
                        var showTime = (category !== 'date') || (time !== 0);
                        // the resulting format code
                        var formatCode = (showDate ? DATE_FORMAT : '') + ((showDate && showTime) ? ' ' : '') + (showTime ? TIME_FORMAT : '');
                        // the parsed number format
                        var parsedFormat = numberFormatter.getParsedFormat(formatCode);

                        // the resulting formatted value (may be null for invalid dates)
                        formatted = numberFormatter.formatValue(parsedFormat, value);
                        break;
                }

                // use standard number format for all other format codes
                return _.isString(formatted) ? formatted : numberFormatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT);
            }

            // empty cells
            return '';
        }

        /**
         * Applies a transformation to the passed color, and returns the CSS
         * color value.
         */
        function getModifiedCssColor(jsonColor, colorType) {

            // the passed color, as instance of the class Color
            var color = Color.parseJSON(jsonColor);
            // the resolved color, including the effective luma
            var colorDesc = docModel.resolveColor(color, colorType);

            if ((colorType === 'text') && (colorDesc.y > 0.25)) {
                color.transform('shade', 25000 / colorDesc.y);
            } else if ((colorType === 'fill') && (colorDesc.y < 0.75)) {
                color.transform('tint', 25000 / (1 - colorDesc.y));
            }
            return docModel.resolveColor(color, colorType).css;
        }

        /**
         * Updates the CSS formatting of the text area control used for cell
         * in-place edit mode according to the current formatting of the active
         * cell.
         */
        function updateTextAreaStyle() {

            // get cell attributes, and overwrite them with current edit attributes
            cellAttributeSet = docModel.extendAttributes(cellCollection.getCellAttributeSet(editAddress), editAttributeSet, { clone: true });

            // receive font settings, text line height, and text orientation
            var zoom = sheetModel.getEffectiveZoom();
            fontDesc = fontCollection.getFont(cellAttributeSet.character, zoom);
            fontDesc.fontSize = Utils.minMax(fontDesc.fontSize, 6, 48);
            lineHeight = docModel.getRowHeight(cellAttributeSet.character, zoom);
            var orientation = cellCollection.getCellTextOrientation(editAddress);

            // set CSS attributes according to formatting of active cell
            textAreaContainer.children().css({
                padding: '0 ' + sheetModel.getEffectiveTextPadding() + 'px',
                lineHeight: lineHeight + 'px',
                fontFamily: fontDesc.family,
                fontSize: fontDesc.size + 'pt',
                fontWeight: fontDesc.bold ? 'bold' : 'normal',
                fontStyle: fontDesc.italic ? 'italic' : 'normal',
                textDecoration: docModel.getCssTextDecoration(cellAttributeSet.character),
                textAlign: (cellCollection.isAnyFormulaCell(editAddress) && (cellAttributeSet.cell.alignHor === 'auto')) ? 'left' : orientation.cssTextAlign
            }).attr('dir', orientation.textDir);

            // set cell text color and fill color to the appropriate nodes
            textAreaNode.css('color', getModifiedCssColor(cellAttributeSet.character.color, 'text'));
            textUnderlayNode.css('background-color', getModifiedCssColor(cellAttributeSet.cell.fillColor, 'fill'));
        }

        /**
         * Updates the position and size of the text area control for the cell
         * in-place edit mode, according to the current length of the text in
         * the text area control, and the scroll position of the grid pane.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, the position of the text area will be updated
         *      immediately. By default, updating the text area will be done
         *      debounced.
         *  @param {Boolean} [options.size=true]
         *      If set to false, the size of the text area will not be changed,
         *      only its position. Used in situations where the content of the
         *      text area does not change (e.g. while scrolling).
         */
        var updateTextAreaPosition = (function () {

            // whether updating the position is pending
            var updatePosition = false;
            // whether updating the size is pending
            var updateSize = false;

            // direct callback, invoked immediately when calling updateTokenArray()
            function directCallback(options) {

                // set the pending update flags for the deferred callback
                updatePosition = true;
                updateSize = updateSize || Utils.getBooleanOption(options, 'size', true);

                // invoke deferred callback immediately if specified, or if the text has caused a new line break
                if (Utils.getBooleanOption(options, 'direct', false) || (textAreaNode.outerHeight() < textAreaNode[0].scrollHeight)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                // do nothing if in-place edit mode is not active
                if (!editActive || (!updatePosition && !updateSize)) { return; }

                // the position of the edited cell
                var position = null;
                // the visible area of this grid pane
                var visiblePosition = null;
                // whether the text in the cell wraps automatically
                var isWrapped = _.isString(cellCollection.getCellValue(editAddress)) && SheetUtils.hasWrappingAttributes(cellAttributeSet);
                // the effective horizontal CSS text alignment
                var textAlign = textAreaNode.css('text-align');
                // round up width to multiples of the line height
                var blockWidth = Math.max(20, lineHeight);
                // the new width and height of the text area
                var targetWidth = 0, targetHeight = 0;

                // returns required width needed to show all text lines in a single line
                function getMinimumWidth() {

                    // required width for all text lines
                    var requiredWidth = textAreaNode.val().split(/\n/).reduce(function (memo, textLine) {
                        return Math.max(memo, fontDesc.getTextWidth(textLine));
                    }, 0);

                    // round up to block width, enlarge to cell width
                    return Math.max(position.width, Utils.roundUp(requiredWidth, blockWidth) + blockWidth + Utils.SCROLLBAR_WIDTH);
                }

                // returns maximum width according to horizontal alignment and width of visible area
                function getMaximumWidth() {

                    // the distance from left visible border to right cell border (for right alignment)
                    var leftWidth = Utils.minMax(position.left + position.width - visiblePosition.left, 0, visiblePosition.width);
                    // the distance from left cell border to right visible border (for left alignment)
                    var rightWidth = Utils.minMax(visiblePosition.left + visiblePosition.width - position.left, 0, visiblePosition.width);
                    // effective available width
                    var availWidth = 0;

                    // get available width according to text alignment
                    switch (textAlign) {
                        case 'right':
                            availWidth = leftWidth;
                            break;
                        case 'center':
                            availWidth = Math.min(leftWidth, rightWidth) * 2 - position.width;
                            break;
                        default:
                            availWidth = rightWidth;
                    }

                    // enlarge to at least half of the width of the visible area
                    return Math.max(availWidth, Math.round(visiblePosition.width / 2));
                }

                // get horizontal and vertical cell position and size
                position = self.getCellRectangle(editAddress, { expandMerged: true });
                // reduce width by one pixel to exclude the grid line
                position.width = Math.max(blockWidth, position.width - 1);
                // reduce height by one pixel to exclude the grid line
                position.height -= 1;

                // get position and size of the visible area in the grid pane
                visiblePosition = editGridPane.getVisibleRectangle();

                // on small Android devices, soft keyboard can overlay complete grid pane,
                // then the automatic height calculation has zero height,
                // after closing the soft keyboard it still keeps this height and that looks really wrong
                if (Utils.CHROME_ON_ANDROID) {
                    visiblePosition.height = Math.max(visiblePosition.height, lineHeight);
                }

                // calculate and set the new size of the text area
                if (updateSize) {

                    // find best size, stick to cell width in auto-wrapping mode
                    var minWidth = isWrapped ? position.width : getMinimumWidth();
                    var maxWidth = isWrapped ? visiblePosition.width : getMaximumWidth();

                    // calculate width and height of the text area
                    targetWidth = Math.min(minWidth, maxWidth);
                    fontDesc.getCustomFontMetrics(function (node) {

                        // the text whose height will be determined
                        var text = textAreaNode.val();

                        // use NBSP instead of empty text; bug 35143: add NBSP to empty last line
                        if (/(^|\n)$/.test(text)) { text += '\xa0'; }

                        // prepare the helper node for multi-line text
                        node.style.width = targetWidth + 'px';
                        node.style.padding = '0 ' + sheetModel.getEffectiveTextPadding() + 'px';
                        node.style.lineHeight = lineHeight + 'px';
                        node.style.whiteSpace = 'pre-wrap';
                        node.style.wordWrap = 'break-word';

                        // add a text node
                        node.appendChild(document.createTextNode(text));

                        targetHeight = Math.min(Math.max(node.offsetHeight, position.height), visiblePosition.height);
                    });

                    textAreaNode.css({ width: targetWidth, height: targetHeight });

                } else {
                    targetWidth = textAreaNode.outerWidth();
                    targetHeight = textAreaNode.outerHeight();
                }

                // calculate and set the new position of the text area
                if (updatePosition) {

                    // left and top position of the text area
                    var leftOffset = position.left;
                    var topOffset = position.top;

                    // calculate horizontal position according to alignment
                    switch (textAlign) {
                        case 'right':
                            leftOffset += position.width - targetWidth;
                            break;
                        case 'center':
                            leftOffset += Math.round((position.width - targetWidth) / 2);
                            break;
                    }

                    // move text area into visible area, set effective position
                    leftOffset = Utils.minMax(leftOffset, visiblePosition.left, visiblePosition.left + visiblePosition.width - targetWidth);
                    topOffset = Utils.minMax(topOffset, visiblePosition.top, visiblePosition.top + visiblePosition.height - targetHeight);
                    textAreaContainer.css({ left: leftOffset - visiblePosition.left, top: topOffset - visiblePosition.top });
                }

                updatePosition = updateSize = false;
            }

            // return the debounced method updateTextAreaPosition()
            return self.createDebouncedMethod(directCallback, deferredCallback, { delay: 250, maxDelay: 500 });
        }());

        /**
         * Returns the selected list item node in the drop-down menu element
         * for the text area.
         *
         * @returns {jQuery}
         *  The selected menu item element in the text area drop-down menu.
         */
        function getTextAreaListButtonNode() {
            return textAreaListMenu.getSelectedItemNodes().first();
        }

        /**
         * Updates and shows or hides the tooltip attached to the text area,
         * containing the name, parameter list, and description of the current
         * function.
         */
        function updateFunctionDescriptionToolTip() {

            // the selected button element in the auto-completion menu
            var buttonNode = getTextAreaListButtonNode();
            // the name and type of the selected list item
            var itemValue = Forms.getButtonValue(buttonNode);
            // the resulting tooltip text
            var toolTipLabel = null;

            // create the tooltip text according to the type of the list entry
            switch (Utils.getStringOption(itemValue, 'type')) {

                case 'name':
                    // the model of the selected defined name
                    var nameModel = docModel.getNameCollection().getName(itemValue.key);
                    if (nameModel) {
                        toolTipLabel = nameModel.getFormula('ui', editAddress);
                        if (toolTipLabel) { toolTipLabel = '=' + toolTipLabel; }
                    }
                    break;

                case 'table':
                    // the model of the selected table range
                    var tableModel = docModel.getTable(itemValue.key);
                    if (tableModel) {
                        var sheetName = tableModel.getSheetModel().getName();
                        toolTipLabel = '=' + formulaParser.formatRangeList('ui', tableModel.getRange(), { sheetName: sheetName });
                    }
                    break;

                case 'func':
                    // the help descriptor for the selected function (nothing for defined names or tables)
                    var functionHelp = formulaResource.getFunctionHelp(itemValue.key);
                    if (functionHelp) { toolTipLabel = functionHelp.description; }
                    break;
            }

            // initialize the tooltip
            if (toolTipLabel) {
                textAreaListItemToolTip.setText(_.noI18n(toolTipLabel)).show();
            } else {
                textAreaListItemToolTip.hide();
            }
        }

        /**
         * Creates entries in the text area pop-up list menu.
         *
         * @param {Array<Object>} entries
         *  The unfiltered entries to be inserted into the list menu, as
         *  objects with the properties 'key' (inserted as property 'key' into
         *  the object values of the list entries) and 'name' (inserted as
         *  property 'name' into the object values of the list entries, used as
         *  visible labels of the list entries).
         *
         * @param {String} filter
         *  The filter string. Only entries with a 'name' property starting
         *  with this string (ignoring character case) will be inserted into
         *  the list menu.
         *
         * @param {String} type
         *  The type of the list entries. Will be used as name for a section in
         *  the list menu, and will be inserted as property 'type' into the
         *  object values of the list entries.
         *
         * @param {String} title
         *  Th UI label for the section title for the specified entry type.
         */
        function createListMenuEntries(entries, filter, type, title) {

            // create a section for the list entries
            textAreaListMenu.createSectionNode(type, { label: title });

            // filter for entries starting with the passed filter string
            filter = filter.toUpperCase();
            entries = entries.filter(function (entry) {
                return entry.name.substr(0, filter.length).toUpperCase() === filter;
            });

            // create the list entries
            entries.forEach(function (entry) {
                textAreaListMenu.createItemNode({ key: entry.key, name: entry.name, type: type }, { section: type, label: _.noI18n(entry.name) });
            });
        }

        /**
         * Returns information about the token containing or preceding the
         * passed text cursor position in the current formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the current formula string.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preferOperand=true]
         *      Specifies how to resolve the situation that the text cursor
         *      position is located between two tokens. If set to true, and one
         *      of the tokens is an operand token (constant value tokens, array
         *      literal tokens, reference tokens, name tokens, or function name
         *      tokens), this token will be returned. Otherwise, always returns
         *      the token preceding the text position.
         *
         * @returns {TokenDescriptor|Null}
         *  A descriptor for the token whose text representation contains the
         *  specified text cursor position. If the text cursor position is
         *  located outside of the formula text, or at its beginning (without
         *  the option 'preferOperand'), this method will return the value
         *  null.
         */
        function findTokenDescriptor(pos, options) {

            // returns whether the specified token is an operand token
            function isOperand(arrayIndex) {
                var tokenDesc = tokenDescs[arrayIndex];
                return tokenDesc && (/^(value|array|ref|name|func)$/).test(tokenDesc.token.getType());
            }

            // whether to prefer operand tokens
            var preferOperand = Utils.getBooleanOption(options, 'preferOperand', false);

            // cursor is located at the beginning of the formula, and operands are preferred:
            // return the first token (following the text cursor position)
            if (preferOperand && (pos === 0) && isOperand(0)) {
                return tokenDescs[0];
            }

            // invalid text position passed
            if (pos <= 0) { return null; }

            // find the token containing or preceding the passed text position
            var index = Utils.findFirstIndex(tokenDescs, function (tokenDesc) { return pos <= tokenDesc.end; }, { sorted: true });
            var currTokenDesc = tokenDescs[index];
            if (!currTokenDesc) { return null; }

            // check if the cursor is located between two tokens, and the trailing is the preferred operand
            if (preferOperand && (pos === currTokenDesc.end) && !isOperand(index) && isOperand(index + 1)) {
                return tokenDescs[index + 1];
            }

            // return the descriptor of the found token
            return currTokenDesc;
        }

        /**
         * Returns information about the innermost function containing the
         * passed text cursor position in the current formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the current formula string.
         *
         * @returns {Object|Null}
         *  A descriptor for the innermost function containing the specified
         *  text cursor position, in the following properties:
         *  - {String} funcKey
         *      The unique resource key of the function.
         *  - {Number} paramIndex
         *      The zero-based index of the function parameter covering the
         *      specified text cursor position.
         *  - {Number} paramCount
         *      The total number of parameters found in the function parameter
         *      list. If the formula is incomplete or contains errors, this
         *      number may be wrong, but it will always be greater than the
         *      value returned in the property 'paramIndex'.
         *  If the text cursor position is outside of any function, this method
         *  will return the value null.
         */
        function findFunctionDescriptor(pos) {

            // information about the token at the passed position
            var tokenDesc = findTokenDescriptor(pos, { preferOperand: true });
            // current token index, modified in seeking functions
            var index = 0;
            // the parameter index and count
            var param = 0, count = 0;

            // seek to the preceding opening parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToOpeningParenthesis() {
                var seps = 0;
                while (index >= 0) {
                    switch (tokenDescs[index].token.getType()) {
                        case 'open':
                            return seps;
                        case 'close':
                            index -= 1;
                            seekToOpeningParenthesis();
                            break;
                        case 'sep':
                            seps += 1;
                            break;
                    }
                    index -= 1;
                }
                return 0;
            }

            // seek to the following closing parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToClosingParenthesis() {
                var seps = 0;
                while (index < tokenDescs.length) {
                    switch (tokenDescs[index].token.getType()) {
                        case 'open':
                            index += 1;
                            seekToClosingParenthesis();
                            break;
                        case 'close':
                            return seps;
                        case 'sep':
                            seps += 1;
                            break;
                    }
                    index += 1;
                }
                return seps;
            }

            // invalid text cursor position passed
            if (!tokenDesc) { return null; }

            // try as long as a function has been found, or the beginning of the formula is reached
            index = tokenDesc.index;
            while (index >= 0) {

                // seek to the preceding opening parenthesis
                param = seekToOpeningParenthesis();

                // check if a function name precedes the parenthesis
                var funcTokenDesc = tokenDescs[index - 1];
                if (funcTokenDesc && funcTokenDesc.token.isType('func')) {
                    index += 1;
                    count = seekToClosingParenthesis() + 1;
                    return { funcKey: funcTokenDesc.token.getValue(), paramIndex: param, paramCount: count };
                }

                // continue seeking to preceding opening parenthesis
                index -= 1;
            }

            // cursor is not located inside a function, but it may point to a top-level
            // function name, in that case return this function with invalid parameter index
            if (tokenDesc.token.isType('func') && tokenDescs[tokenDesc.index + 1].token.isType('open')) {
                index = tokenDesc.index + 2;
                count = seekToClosingParenthesis() + 1;
                return { funcKey: tokenDesc.token.getValue(), paramIndex: -1, paramCount: count };
            }

            return null;
        }

        /**
         * Hides the function signature tooltip and the function
         * auto-completion pop-up menu.
         */
        function hideTextAreaPopups() {
            signatureToolTip.hide();
            textAreaListMenu.hide();
        }

        /**
         * Shows the function signature tooltip or the function auto-completion
         * pop-up menu if possible.
         *
         * @param {Boolean} [suppressAutoComplete=false]
         *  If set to true, the pop-up menu for auto-completion of function
         *  names will be suppressed regardless of the current formula text and
         *  cursor position.
         */
        function updateTextAreaPopups(suppressAutoComplete) {

            // show pop-ups only in formula mode, do not show them while dragging
            // a highlighted range, or when selecting new ranges for the formula
            if (!_.isString(formulaString) ||
                self.hasTrackedHighlightedRange() ||
                _.isObject(self.getSheetViewAttribute('activeSelection'))
            ) {
                hideTextAreaPopups();
                return;
            }

            // the cursor position in the text area (exclude the equality sign)
            var textPos = textAreaNode[0].selectionStart - formulaOffset;
            // the text to be used for function auto-completion
            var autoText = null;

            // prefer function auto-completion over function tooltip
            funcAutoTokenDesc = null;

            // auto completion of names: cursor must be located exactly at the end of a function or name token
            var tokenDesc = (funcAutoComplete && !suppressAutoComplete) ? findTokenDescriptor(textPos) : null;
            if (tokenDesc && (tokenDesc.end === textPos)) {

                // get the text entered before the text cursor (may be a name token, a table token, or a function token)
                // TODO: process sheet-local names too
                if ((tokenDesc.token.isType('name') && !tokenDesc.token.hasSheetRef()) || tokenDesc.token.isType('table') || tokenDesc.token.isType('func')) {
                    // Special case: do not show a pop-up menu when a literal number precedes the name.
                    // Otherwise, e.g. entering scientific numbers will show a pop-up for functions starting with 'E'.
                    if ((tokenDesc.index === 0) || !tokenDescs[tokenDesc.index - 1].token.isType('lit')) {
                        autoText = tokenDesc.text;
                    }
                }

                // store token info (needed to replace the entered text with the full function name)
                funcAutoTokenDesc = tokenDesc;
            }

            // fill pop-up menu for auto-completion
            if (_.isString(autoText)) {

                // insert all matching defined names, and all matching function names
                textAreaListMenu.clearContents();

                // insert defined names (the method createListMenuEntries() expects an array of objects with 'key' and 'name' properties)
                // TODO: sheet-local names
                var nameEntries = docModel.getAllNames({ skipHidden: true, skipLocal: true });
                nameEntries = nameEntries.map(function (nameModel) { return nameModel.getLabel(); });
                nameEntries.sort(function (label1, label2) { return label1.localeCompare(label2); });
                nameEntries = nameEntries.map(function (label) { return { key: label, name: label }; });
                createListMenuEntries(nameEntries, autoText, 'name',  gt.pgettext('named ranges', 'Named ranges'));

                // insert table ranges (the method createListMenuEntries() expects an array of objects with 'key' and 'name' properties)
                var tableEntries = docModel.getAllTables().map(function (tableModel) { return tableModel.getName(); });
                tableEntries.sort(function (name1, name2) { return name1.localeCompare(name2); });
                tableEntries = tableEntries.map(function (name) { return { key: name, name: name }; });
                createListMenuEntries(tableEntries, autoText, 'table', /*#. special cell ranges in a spreadsheet for filtering etc. */ gt('Table ranges'));

                // insert the matching function names
                createListMenuEntries(functionEntries, autoText, 'func', /*#. functions in spreadsheet formulas */ gt('Functions'));

                // show the menu, if it contains at least one entry
                if (textAreaListMenu.getItemNodes().length > 0) {
                    signatureToolTip.hide();
                    textAreaListMenu.show().selectItem(0, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return;
                }
            }

            // try to show a tooltip for the current function in a formula
            hideTextAreaPopups();
            var functionInfo = findFunctionDescriptor(textPos);
            if (functionInfo) {
                signatureToolTip.updateSignature(functionInfo.funcKey, functionInfo.paramIndex);
            }
        }

        /**
         * Updates the function tooltip, and other settings according to the
         * text selection in the text area control.
         */
        function updateTextAreaAfterSelection() {

            // do not open the auto-completion menu even if the new cursor position is valid
            funcAutoComplete = false;

            // do not ignore closing parentheses entered manually anymore (function auto-complete)
            funcAutoCloseData = [];

            // defer execution to let the browser process pending key events
            self.executeDelayed(function () {
                // fail-safe check that cell edit mode is still active
                if (editActive) { updateTextAreaPopups(); }
            });
        }

        /**
         * Updates the highlighting of cell references in the current formula
         * while cell in-place edit mode is active.
         */
        function updateTextAreaHighlighting() {

            // descriptor objects for all resolved cell ranges in the formula
            var rangeDescs = tokenArray.extractRanges({
                refAddress: editAddress,
                targetAddress: editAddress,
                resolveNames: true
            });

            // show the cell ranges located in the active sheet only
            var activeSheet = self.getActiveSheet();
            rangeDescs = rangeDescs.filter(function (rangeInfo) {
                return rangeInfo.range.containsSheet(activeSheet);
            });

            // nothing to do without any tokens to be highlighted
            if (rangeDescs.length === 0) {
                textUnderlayNode.empty();
                return;
            }

            // collect the scheme color index and the size (token count) of the range descriptors (map by token index)
            var rangeDataMap = {};
            var styleIndex = 0;
            rangeDescs.forEach(function (rangeDesc) {

                // skip multiple ranges based on the same name token
                if (rangeDesc.index in rangeDataMap) { return; }

                rangeDataMap[rangeDesc.index] = {
                    color: Utils.getSchemeColor(styleIndex),
                    size: rangeDesc.size
                };
                styleIndex += 1;
            });

            // start with the equality sign (unless the edit text does not contain it, e.g. in '+A1+A2')
            var markup = (formulaOffset === 0) ? '' : '<span>=</span>';

            // process the token descriptor array
            tokenDescs.forEach(function (tokenDesc, index) {

                // create a span element with highlight color, if a corresponding range exists
                markup += '<span ';
                var rangeData = rangeDataMap[index];
                if (rangeData) { markup += ' data-style="' + rangeData.color + '"';  }
                markup += '>';

                // add the token text representations of the token
                markup += Utils.escapeHTML(tokenDesc.text);
                markup += '</span>';
            });

            textUnderlayNode[0].innerHTML = markup;
        }

        /**
         * Updates the formula token array according to the current contents of
         * the text area control used for cell in-place edit mode. The token
         * array will invoke the changle callback handler that will cause to
         * redraw the highlighted ranges in all grid panes.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, the token array and all other dependent
         *      settings will be updated immediately. By default, updating the
         *      token array will be debounced.
         *  @param {Boolean} [options.suppressAutoComplete=false]
         *      If set to true, the pop-up menu for auto-completion of function
         *      names will be suppressed regardless of the current formula text
         *      and cursor position.
         */
        var updateTokenArray = (function () {

            // whether to suppress the auto-completion pop-up menu
            var suppressAutoComplete = false;

            // direct callback, invoked immediately when calling updateTokenArray()
            function directCallback(options) {

                // the current input string in the text area
                var value = textAreaNode.val();

                // update the formula expression immediately
                formulaString = /^[-+=]/.test(value) ? value.replace(/^=/, '') : null;
                formulaOffset = (value === formulaString) ? 0 : 1;

                // update the 'suppress auto-completion' flag
                suppressAutoComplete = suppressAutoComplete || Utils.getBooleanOption(options, 'suppressAutoComplete', false);

                // invoke deferred callback immediately if specified
                if (Utils.getBooleanOption(options, 'direct', false)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                // the cell edit mode may have been finished already
                if (!editActive) { return; }

                // parse the formula string
                if (_.isString(formulaString)) {
                    tokenDescs = tokenArray.parseFormula('ui', formulaString, { refAddress: editAddress });
                } else {
                    tokenArray.clearTokens();
                    tokenDescs = [];
                }

                // insert highlighting mark-up for the underlay node
                updateTextAreaHighlighting();

                // update pop-up nodes and other settings dependent on the text selection
                updateTextAreaPopups(suppressAutoComplete);
                suppressAutoComplete = false;
            }

            // return the debounced method updateTokenArray()
            return self.createDebouncedMethod(directCallback, deferredCallback, { delay: 100, maxDelay: 300 });
        }());

        /**
         * Callback handler for the token array used while cell edit mode is
         * active. Updates the edited formula expression while the token array
         * will be manipulated externally, e.g. when dragging reference ranges
         * in the sheet area.
         */
        function changeTokenHandler(token, index) {

            // return global token array changes (reparsed formula etc.)
            if (!token) { return; }

            // override the text representation of the changed token
            tokenDescs[index].text = token.getText(formulaGrammar);

            // update the text positions of the following tokens
            Utils.iterateArray(tokenDescs, function (tokenDesc, i) {
                tokenDesc.start = (i === 0) ? 0 : tokenDescs[i - 1].end;
                tokenDesc.end = tokenDesc.start + tokenDesc.text.length;
            }, { begin: index });

            // calculate the new formula string
            var formulaString = tokenDescs.reduce(function (formula, tokenDesc) {
                return formula + tokenDesc.text;
            }, '');

            // set the new formula, add an undo action
            setTextAreaContents(textAreaNode.val()[0] + formulaString, { start: tokenDescs[index].end + 1 });
            saveUndoState({ update: true });

            // immediately update the highlighted cell references
            updateTextAreaHighlighting();
            resetActiveCellSelection();
        }

        /**
         * Changes the contents and selection of the text area, and updates its
         * position and size.
         *
         * @param {String} text
         *  The new contents of the text area.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.start]
         *      The new start position of the selection. If omitted, the text
         *      cursor will be moved to the end of the new text.
         *  @param {Number} [options.end=options.start]
         *      The new end position of the selection. If omitted, and a start
         *      position has been specified with the option 'start', a simple
         *      text cursor will be shown at the start position.
         *  @param {Boolean} [options.suppressAutoComplete=false]
         *      If set to true, the pop-up menu for auto-completion of function
         *      names will be suppressed.
         */
        function setTextAreaContents(text, options) {

            var start = Utils.getIntegerOption(options, 'start', text.length, 0, text.length);
            var end = Utils.getIntegerOption(options, 'end', start, start, text.length);

            textAreaNode.val(text);
            prevEditText = text;
            Forms.setInputSelection(textAreaNode, start, end);
            updateTextAreaPosition({ direct: true });
            updateTokenArray(options);
        }

        /**
         * Replaces the token before the text cursor with the selected entry
         * from the auto-complete menu.
         */
        function applyAutoCompleteText(itemValue) {

            // always hide the pop-up menu
            textAreaListMenu.hide();

            // fail-safety check
            if (!funcAutoTokenDesc || !_.isObject(itemValue)) { return; }

            // get new text and new cursor position
            var newText = itemValue.name;
            var newStart = funcAutoTokenDesc.start + newText.length + 1;

            // insert parentheses after function name (unless the existing token is
            // a function token which means it already has an opening parenthesis)
            if (itemValue.type === 'func') {
                if (funcAutoTokenDesc.token.isType('name')) {
                    newText += '()';
                    // ignore the next closing parenthesis entered manually
                    funcAutoCloseData.push(true);
                }
                // move cursor behind the opening parenthesis
                newStart += 1;
            }

            // insert the new text into the formula string, prevent showing the
            // auo-completion pop-up list immediately again (important for defined names)
            newText = Utils.replaceSubString(textAreaNode.val(), funcAutoTokenDesc.start + 1, funcAutoTokenDesc.end + 1, newText);
            setTextAreaContents(newText, { start: newStart, suppressAutoComplete: true });
            saveUndoState({ update: true });
        }

        /**
         * Changes the quick edit mode while in-place cell edit mode is active,
         * and shows a status label at the bottom of the application pane.
         *
         * @param {Boolean} newQuickEdit
         *  The new value of the quick edit mode.
         */
        function setQuickEditMode(newQuickEditMode) {

            // change the quick edit state
            if (quickEditMode === newQuickEditMode) { return; }
            quickEditMode = newQuickEditMode;

            // update the status label
            self.setStatusLabel(quickEditMode ?
                //#. Special edit mode in spreadsheet document: quickly type text into cell, use cursor keys to move to next cell
                gt('Insert mode') :
                //#. Standard edit mode in spreadsheet document: edit existing cell text, cursor keys move text cursor inside cell
                gt('Edit mode')
            );
        }

        /**
         * Saves the current state of the cell edit mode on top of the undo
         * stack. If the current undo action is not on top of the stack, all
         * following actions will be deleted first.
         */
        function saveUndoState(options) {

            // try to update the current (top-most, not fixed) stack entry with current text
            var update = Utils.getBooleanOption(options, 'update', false);
            if (update && (undoIndex + 1 === undoStack.length) && !undoStack[undoIndex].fixed) {
                undoStack[undoIndex].text = textAreaNode.val();
                undoStack[undoIndex].sel = Forms.getInputSelection(textAreaNode);
            } else {
                undoStack.splice(undoIndex + 1);
                undoStack.push({
                    text: textAreaNode.val(),
                    sel: Forms.getInputSelection(textAreaNode),
                    attrs: _.copy(editAttributeSet, true),
                    fixed: !update
                });
            }

            // update the stack pointer, and notify listeners
            undoIndex = undoStack.length - 1;
            self.trigger('celledit:change');
        }

        /**
         * Restores the current state in the undo stack.
         */
        function restoreUndoState() {

            // the undo action to be restored
            var action = undoStack[undoIndex];

            editAttributeSet = _.copy(action.attrs, true);
            updateTextAreaStyle();
            setTextAreaContents(action.text, action.sel);
            self.trigger('celledit:change');
        }

        /**
         * Removes the activated ranges from the view that have been inserted
         * into the formula while cell edit mode is active.
         */
        function resetActiveCellSelection() {
            self.setSheetViewAttribute('activeSelection', null);
        }

        /**
         * Updates the position of the text area control after the grid pane
         * has been scrolled (or modified in any other way, e.g. zoom).
         */
        function gridPaneScrollHandler() {
            updateTextAreaPosition({ direct: true, size: false });
        }

        /**
         * Handles 'keydown' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyDownHandler(event) {

            // the list item currently selected in the auto-completion pop-up menu
            var itemNode = textAreaListMenu.getSelectedItemNodes().first();
            // the value of the item node
            var itemValue = Forms.getButtonValue(itemNode);

            // first, try to cancel active tracking mode on ESCAPE key
            if ((event.keyCode === KeyCodes.ESCAPE) && Tracking.cancelTracking()) {
                return false;
            }

            // short-cuts in visible pop-up menu for auto-completion
            if (textAreaListMenu.isVisible()) {

                // close pop-up menu on ESCAPE key
                if (event.keyCode === KeyCodes.ESCAPE) {
                    textAreaListMenu.hide();
                    return false;
                }

                // insert selected list item into the formula
                if (KeyCodes.matchKeyCode(event, 'TAB') || KeyCodes.matchKeyCode(event, 'ENTER')) {

                    // Bug 31999: For performance reasons, the token array and the auto-completion pop-up
                    // menu will be updated debounced. To prevent that ENTER/TAB keys insert a function
                    // name unintentionally, the pop-up menu must be updated before applying a list item.
                    updateTokenArray({ direct: true });

                    // list menu may have been hidden by the updateTextAreaPopups() call,
                    // do not apply any list item in this case, but leave cell edit mode
                    if (textAreaListMenu.isVisible()) {
                        applyAutoCompleteText(itemValue);
                        return false;
                    }
                }

                // otherwise, try to move the selected list item
                itemNode = textAreaListMenu.getItemNodeForKeyEvent(event, itemNode);
                if (itemNode.length > 0) {
                    textAreaListMenu.selectItemNode(itemNode, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return false;
                }
            }

            // quit in-place edit mode on ESCAPE key regardless of any modifier keys
            if (event.keyCode === KeyCodes.ESCAPE) {
                self.cancelCellEditMode();
                return false;
            }

            // toggle quick edit mode
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                setQuickEditMode(!quickEditMode);
                // forget last cell range selected in the sheet
                resetActiveCellSelection();
                return false;
            }

            // ENTER without modifier keys (but ignoring SHIFT): set value, move cell down/up
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                if (self.leaveCellEditMode('cell', { validate: true })) {
                    return; // let key event bubble up to move the cell cursor
                }
                return false;
            }

            // ENTER with CTRL or META key: fill value to all cells (no SHIFT), or insert matrix formula (SHIFT)
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null, ctrlOrMeta: true })) {
                self.leaveCellEditMode(event.shiftKey ? 'matrix' : 'fill', { validate: true });
                // do not let key event bubble up (no change of selection)
                return false;
            }

            // ENTER with ALT key without any other modifier keys: insert line break
            if (KeyCodes.matchKeyCode(event, 'ENTER', { alt: true })) {
                Forms.replaceTextInInputSelection(textAreaNode, '\n');
                textAreaInputHandler();
                return false;
            }

            // ENTER with any other modifier keys: discard the entire event
            if (event.keyCode === KeyCodes.ENTER) {
                return false;
            }

            // TAB key without modifier keys (but ignoring SHIFT): set value, move cell to next/previous cell
            if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                if (self.leaveCellEditMode('cell', { validate: true })) {
                    return; // let key event bubble up to move the cell cursor
                }
                return false;
            }

            // any cursor navigation key
            if (event.keyCode in NAVIGATION_KEYCODES) {
                if (quickEditMode) {
                    // quick edit mode: do not let the text area process the key events
                    event.preventDefault();
                } else {
                    // standard edit mode: do not process the key events (no change of cell selection)
                    event.stopPropagation();
                    // update pop-up nodes and other settings dependent on the new text selection
                    updateTextAreaAfterSelection();
                    // forget last cell range selected in the sheet
                    resetActiveCellSelection();
                }
                // let key event bubble up to move the cell cursor, or text cursor in the text area
                return;
            }

            // all other events will be processed by the text field
        }

        /**
         * Handles 'keypress' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyPressHandler(event) {

            // the entered Unicode character
            switch (String.fromCharCode(event.charCode)) {

                // Register an opening parenthesis entered manually. The related closing parenthesis
                // will be inserted even if it occurs among other closing parentheses inserted via
                // function auto-completion (see below).
                case '(':
                    funcAutoCloseData.push(false);
                    break;

                // When a function name has been inserted from the auto-completion pop-up menu, skip
                // a closing parenthesis because it has been inserted already by the auto-completion).
                case ')':
                    if (funcAutoCloseData.pop() === true) {
                        // the text cursor needs to be moved one character forward
                        Forms.setInputSelection(textAreaNode, Forms.getInputSelection(textAreaNode).start + 1);
                        updateTextAreaPopups();
                        return false;
                    }
                    break;
            }
        }

        /**
         * Handles any changes in the text area.
         */
        function textAreaInputHandler() {

            // the current value of the text area
            var currEditText = textAreaNode.val();
            // the text selection in the text area control
            var textSelection = Forms.getInputSelection(textAreaNode);
            // whether the selection in the text area is a cursor only
            var isCursor = textSelection.start === textSelection.end;

            // do nothing if the edit text did not change at all
            if (prevEditText === currEditText) { return; }

            // try to show drop-down list with function names and defined names
            funcAutoComplete = isCursor;

            // update text area settings (position, formula range highlighting)
            updateTextAreaPosition();
            updateTokenArray();
            saveUndoState({ update: true });

            // Mobile devices may add auto-correction text which modifies the selection after a text change event has
            // been processed. Recalculate the funcAutoComplete flag after the browser has set the final selection.
            self.executeDelayed(function () {
                textSelection = Forms.getInputSelection(textAreaNode);
                funcAutoComplete = textSelection.start === textSelection.end;
            });

            // forget last active cell range selected in the sheet (formulas with cell references)
            resetActiveCellSelection();

            // show type-ahead suggestion in the text area (not in formula edit mode)
            if (_.isString(prevEditText) && (prevEditText.length < currEditText.length) && isCursor && (textSelection.end === currEditText.length) && !_.isString(formulaString)) {

                // first cycle: collect the contents of all text cells and formula cells resulting in text above and below the edited cell
                if (!typeAheadEntries) {
                    typeAheadEntries = [];

                    // collect the values of all text cells, and the string results of formula cells
                    var boundRange = Range.create(editAddress[0], Math.max(0, editAddress[1] - 256), editAddress[0], Math.min(docModel.getMaxRow(), editAddress[1] + 256));
                    var cellIt = cellCollection.createLinearAddressIterator(editAddress, 'up down', { type: 'value', boundRange: boundRange, skipStart: true });
                    IteratorUtils.forEach(cellIt, function (address) {
                        var value = cellCollection.getCellValue(address);
                        if ((typeof value === 'string') && (value.length > 0)) {
                            typeAheadEntries.push({ text: value, dist: Math.abs(editAddress[1] - address[1]) });
                        }
                    });

                    // sort the entries in the array by distance to the edited cell
                    typeAheadEntries.sort(function (entry1, entry2) { return entry1.dist - entry2.dist; });

                    // reduce the entries to a simple string array, that is still sorted by distance
                    typeAheadEntries = _.pluck(typeAheadEntries, 'text');
                }

                // find the first matching type-ahead entry (the array is sorted by distance to edited cell, thus it will find the nearest matching entry)
                var suggestion = _.find(typeAheadEntries, function (entry) {
                    return entry.substr(0, currEditText.length).toUpperCase() === currEditText.toUpperCase();
                });

                // insert the remaining text of a matching entry, keep character case of the typed text
                if (suggestion) {
                    suggestion = currEditText + suggestion.substr(currEditText.length);
                    setTextAreaContents(suggestion, { start: currEditText.length, end: suggestion.length });
                    saveUndoState({ update: true });
                }
            }

            // store current text for next iteration
            prevEditText = currEditText;
        }

        /**
         * Handles mousedown or touchstart events in the text area.
         */
        function textAreaMouseTouchHandler() {
            // leave quick-edit mode (cursor keys will move the text cursor)
            setQuickEditMode(false);
            // update pop-up nodes and other settings dependent on the new text selection
            updateTextAreaAfterSelection();
            // forget last cell range selected in the sheet
            resetActiveCellSelection();
        }

        /**
         * Updates the scroll position of the underlay node, after the scroll
         * position of the text area node has been changed.
         */
        function textAreaScrollHandler() {
            textUnderlayNode.scrollTop(textAreaNode.scrollTop());
        }

        /**
         * Hides the pop-up nodes if the focus leaves the text area control,
         * unless the auto-completion menu is being clicked.
         */
        var textAreaFocusHandler = (function () {

            // whether to ignore focus event in the text area (no update of pop-up nodes)
            var ignoreFocusEvent = false;

            return function (event) {
                switch (event.type) {

                    // text area focused: rebuild all active pop-up nodes
                    case 'focus':
                        textAreaContainer.addClass(Forms.FOCUSED_CLASS);
                        if (!ignoreFocusEvent) {
                            updateTextAreaPopups();
                        }
                        break;

                    // text area lost focus: hide pop-up nodes (unless the auto-complete menu
                    // has been clicked, in this case, move focus back to text area immediately)
                    case 'blur':
                        self.executeDelayed(function () {
                            if (editActive && textAreaListMenu.hasFocus()) {
                                ignoreFocusEvent = true;
                                textAreaNode.focus();
                                // IE sends focus event deferred
                                self.executeDelayed(function () { ignoreFocusEvent = false; });
                            } else {
                                textAreaContainer.removeClass(Forms.FOCUSED_CLASS);
                                if (!stickyAutoComplete) { hideTextAreaPopups(); }
                            }
                        });
                        break;
                }
            };
        }());

        /**
         * Inserts the name clicked in the auto-completion pop-up menu into the
         * current formula text.
         */
        function textAreaListClickHandler(event) {
            var itemValue = Forms.getButtonValue(event.currentTarget);
            applyAutoCompleteText(itemValue);
        }

        /**
         * Returns whether the current text selection in the text area control
         * is valid to start selecting new ranges for the current formula. The
         * text area must contain a formula string (starting with an equality
         * sign), and the start position of the text selection must be located
         * behind the leading equality sign, an operator, a list separator, or
         * an opening parenthesis).
         */
        function isValidRangeSelectionPosition() {

            // the cursor position in the token array (exclude leading equality sign)
            var textPos = textAreaNode[0].selectionStart - formulaOffset;

            // invalid if not in cell in-place edit mode with a formula
            if (!editActive || !_.isString(formulaString)) { return false; }

            // existing view attribute, or cursor at the beginning of the formula is always valid
            if ((textPos === 0) || _.isObject(self.getSheetViewAttribute('activeSelection'))) { return true; }

            // find the formula token located before the text cursor
            var tokenDesc = findTokenDescriptor(textPos);

            // cursor must be located between tokens; leading token must be of correct
            // type (or cursor must be located at the beginning of the formula)
            return tokenDesc && (textPos === tokenDesc.end) && tokenDesc.token.matchesType(/^(op|sep|open)$/);
        }

        /**
         * When in cell edit mode with a formula, returns the active selection
         * which will be used for keyboard navigation and mouse/touch tracking.
         */
        function getCellSelectionHandler() {

            // the current value of the 'activeSelection' view attribute
            var activeSelection = self.getSheetViewAttribute('activeSelection');

            // always return existing view attribute
            if (_.isObject(activeSelection)) { return activeSelection; }

            // check the text selection in the text area control
            if (!isValidRangeSelectionPosition()) { return null; }

            // start with an empty active selection
            return new SheetSelection();
        }

        /**
         * When in cell edit mode with a formula, selects a single range and
         * inserts the range address into the formula string.
         */
        function setCellSelectionHandler(selection) {

            // do nothing if ranges cannot be selected with the current text selection
            if (!isValidRangeSelectionPosition()) { return false; }

            // store current text selection, if no range are activated yet
            // (selected text will be replaced with initial range addresses)
            if (!_.isObject(self.getSheetViewAttribute('activeSelection'))) {
                rangeSelection = Forms.getInputSelection(textAreaNode);
            }

            // build the new formula term to be inserted into the text
            var tmpTokenArray = new TokenArray(sheetModel, 'cell');
            tmpTokenArray.appendRangeList(selection.ranges);
            var references = tmpTokenArray.getFormula('ui');
            var formulaText = Utils.replaceSubString(textAreaNode.val(), rangeSelection.start, rangeSelection.end, references);

            // insert the new formula text, set cursor behind the range (no real selection)
            rangeSelection.end = rangeSelection.start + references.length;
            setTextAreaContents(formulaText, { start: rangeSelection.end });
            saveUndoState({ update: true });

            // update the view attribute
            self.setSheetViewAttribute('activeSelection', selection);

            // do not perform standard sheet selection
            return true;
        }

        /**
         * Cancels cell edit mode after the application loses edit rights.
         */
        function editModeHandler(event, editMode) {
            // leave cell in-place edit mode, if document changes to read-only mode
            if (!editMode && self.isImportFinished()) {
                self.cancelCellEditMode();
            }
        }

        /**
         * Updates the text area for a changed zoom factor.
         */
        function changeSheetViewAttributesHandler(event, attributes) {
            if (editActive && ('zoom' in attributes)) {
                updateTextAreaStyle();
                updateTextAreaPosition({ direct: true });
            }
        }

        function appendTextAreaContainer() {
            textAreaContainer.show();

            if (self.panesCombined()) {
                textAreaContainer.addClass('ontop');
                self.getApp().getWindowNode().append(textAreaContainer);
            } else {
                editGridPane.getNode().append(textAreaContainer);
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether the in-place cell edit mode is currently active.
         *
         * @returns {Boolean}
         *  Whether the in-place cell edit mode is currently active.
         */
        this.isCellEditMode = function () {
            return editActive;
        };

        /**
         * Starts in-place cell edit mode in the active grid pane.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {String} [options.text]
         *      The initial text to insert into the edit area. If omitted, the
         *      current value of the active cell will be inserted.
         *  @param {Number} [options.pos]
         *      The initial position of the text cursor. If omitted, the text
         *      cursor will be placed after the entire text. If negative, the
         *      text cursor will be placed from the end of the text.
         *  @param {Boolean} [options.quick=false]
         *      If set to true, the quick edit mode will be initiated. All
         *      cursor navigation keys will not move the text cursor in the
         *      text area control but will behave as in the normal sheet
         *      selection mode.
         *  @param {Boolean} [options.stickyList=false]
         *      If set to true, the auto-completion drop-down list will be kept
         *      open when the text edit control is not focused.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the cell edit mode has been
         *  started (or if the cell edit mode is already running), or that will
         *  be rejected, if the cell edit mode could not be started (e.g. if
         *  the active cell is locked).
         */
        this.enterCellEditMode = function (options) {

            // the initial text for the text area
            var initialText = Utils.getStringOption(options, 'text', null);
            // the initial cursor position for the text area
            var initialPos = Utils.getIntegerOption(options, 'pos');

            // first, check document edit mode
            if (!this.requireEditMode()) { return $.Deferred().reject(); }

            // check whether in-place edit mode is already active
            if (editActive) { return $.when(); }

            // cancel current tracking
            Tracking.cancelTracking();

            // store location of edited cell (sheet may change during range selection mode for formulas)
            editAddress = this.getActiveCell();
            sheetModel = this.getSheetModel();
            cellCollection = sheetModel.getCellCollection();
            editGridPane = this.getActiveGridPane();
            editAttributeSet = {};

            // initialize CSS formatting of the text area (this initializes most class members)
            updateTextAreaStyle();

            // scroll to the active cell (before checking edit mode); on touch devices,
            // scroll the cell to the top border of the grid pane (due to virtual keyboard)
            // don't scroll on small devices because there we put the textarea on top of the screen anyway
            editGridPane.scrollToCell(editAddress, { forceTop: Utils.TOUCHDEVICE && !self.panesCombined() });

            // Bug 34366: Touch devices scroll the window out of visible area, so that neither the bars
            // (tool and tab) are visible, nor the editable cell. Root cause is the soft-keyboard.
            // However, scroll back to 0,0 and everything is fine. (Delay of 1000 necessary for iOS)
            if (Utils.TOUCHDEVICE) {
                this.executeDelayed(function () { window.scrollTo(0, 0); }, 1000);
            }

            // the active cell must not be locked
            return this.ensureUnlockedActiveCell({ lockTables: 'header' }).then(function () {

                // initial state of quick edit mode and other settings
                editActive = true;
                setQuickEditMode(Utils.getBooleanOption(options, 'quick', false));

                // TODO: matrix formula support
                matrixFormula = false;

                // type-ahead suggestions will be collected on first access
                typeAheadEntries = null;

                // the token array stores the parsed formula for range highlighting
                tokenDescs = [];
                tokenArray = new TokenArray(sheetModel, 'cell');
                tokenArray.registerChangeHandler(changeTokenHandler);

                // highlight the ranges in the active sheet contained in the formula expression
                highlightUid = self.startRangeHighlighting(tokenArray, {
                    priority: true,
                    draggable: true,
                    refAddress: editAddress,
                    targetAddress: editAddress,
                    resolveNames: true
                });

                // insert the text area control into the DOM of the active grid pane, and activate it
                appendTextAreaContainer();
                editGridPane.registerFocusTargetNode(textAreaNode);
                editGridPane.on('change:scrollpos', gridPaneScrollHandler);

                // get the formatted edit string
                originalText = getTextForCellEditMode();
                textUnderlayNode.empty();

                // set initial contents, position, size, and selection of the text area
                var editText = _.isString(initialText) ? initialText : originalText;
                if (_.isNumber(initialPos) && (initialPos < 0)) { initialPos = Math.max(0, editText.length + initialPos); }
                setTextAreaContents(editText, { start: initialPos });

                // initialize local undo/redo stack
                undoStack = [{
                    text: originalText,
                    sel: _.isString(initialText) ? { start: 0, end: originalText.length } : Forms.getInputSelection(textAreaNode),
                    attrs: {},
                    fixed: true
                }];
                undoIndex = 0;

                // selection handler that changes cell ranges in formaulas in cell edit mode
                self.registerCellSelectionHandlers(getCellSelectionHandler, setCellSelectionHandler);

                // set last value and invoke input handler to handle custom initial text
                prevEditText = _.isString(initialText) ? '' : null;
                textAreaInputHandler();

                // special mode to keep auto-completion list menu open without focus in text area
                stickyAutoComplete = Utils.getBooleanOption(options, 'stickyList', false);
                textAreaListMenu.setAutoCloseMode(!stickyAutoComplete);

                // notify listeners
                self.trigger('celledit:enter');
            });
        };

        /**
         * Cancels in-place cell edit mode, without committing the current edit
         * text.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.cancelCellEditMode = function () {

            // nothing to do, if in-place edit mode is not active
            if (!editActive) { return this; }

            // deinitialize range highlighting
            editActive = false;
            this.endRangeHighlighting(highlightUid);
            this.unregisterCellSelectionHandlers();
            resetActiveCellSelection();
            sheetModel = cellCollection = null;
            tokenArray = tokenDescs = highlightUid = null;

            // back to cell selection mode
            quickEditMode = formulaString = typeAheadEntries = null;
            hideTextAreaPopups();

            // clean up grid pane
            editGridPane.off('change:scrollpos', gridPaneScrollHandler);
            editGridPane.unregisterFocusTargetNode();
            editGridPane = null;

            // hide the text area control (bug 40321: DO NOT detach from DOM, Egde may freeze)
            textAreaContainer.hide();

            // notify listeners
            this.setStatusLabel(null);
            this.trigger('celledit:leave');
            return this;
        };

        /**
         * Leaves in-place cell edit mode, and optionally commits the current
         * text. If the current text is a formula expression, it will be
         * validated before committing.
         *
         * @param {String} commitMode
         *  Specifies how to leave the cell edit mode. The following values are
         *  supported:
         *  - 'cell': The current value will be committed to the active cell of
         *      the selection.
         *  - 'fill': The current value will be filled to all cells of the
         *      selection.
         *  - 'matrix': The current value will be inserted as matrix formula
         *      into the selected cell range.
         *  - 'auto': Preserves the current formula type of the cell: either
         *      'cell' or 'matrix'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.validate=false]
         *      If set to true, a formula expression will be velidated. If it
         *      is invalid, a notification alert will be shown, and the cell
         *      edit mode will not be left.
         *
         * @returns {Boolean}
         *  Whether the cell edit mode has actually been left. Returns false,
         *  if the option 'validate' have been set to true, and the current
         *  formula expression is invalid.
         */
        this.leaveCellEditMode = function (commitMode, options) {

            // nothing to do, if in-place edit mode is not active
            if (!editActive) { return true; }

            // check validity of commit mode (e.g. disallow 'cell' in matrix formulas)
            switch (commitMode) {
                case 'cell':
                case 'fill':
                    if (matrixFormula) {
                        // TODO: define behavior (update entire matrix automatically?)
                        self.yellMessage('formula:matrix');
                        setQuickEditMode(false);
                        return false;
                    }
                    break;
                case 'matrix':
                    // TODO: check that selection matches range of matrix formula
                    break;
                case 'auto':
                    commitMode = matrixFormula ? 'matrix' : 'cell';
                    break;
                default:
                    Utils.error('CellEditMixin.leaveCellEditMode(): unknown commit mode: ' + commitMode);
                    commitMode = null;
            }

            // do nothing else, if the document does not have edit rights anymore
            if (!docModel.getEditMode() || !commitMode) {
                this.cancelCellEditMode();
                return true;
            }

            // the new cell text
            var editText = textAreaNode.val();
            // parse the entered text: convert to other data types, or interpret the formula
            // TODO: get value/matrix result depending on commit mode
            var contents = cellCollection.parseCellValue(editText, editAddress);
            // whether to validate the formula
            var validate = Utils.getBooleanOption(options, 'validate', false);

            // keep edit mode active, if an error has been found in the formula structure
            if (validate && contents.result && (contents.result.type === 'error')) {
                self.yellMessage('formula:invalid');
                setQuickEditMode(false);
                return false;
            }

            // deinitialize cell edit mode
            this.cancelCellEditMode();

            // add auto-wrapping attribute, if the text contains an explicit line break
            if (editText.indexOf('\n') >= 0) {
                AttributeUtils.insertAttribute(editAttributeSet, 'cell', 'wrapText', true);
            }

            // add hyperlink specific attributes
            if (contents.url) {
                AttributeUtils.insertAttribute(editAttributeSet, 'character', 'underline', true);
                AttributeUtils.insertAttribute(editAttributeSet, 'character', 'color', docModel.createJSONColor(Color.HYPERLINK, 'text'));
            }

            // performance: do not pass empty attributes object (prevent useless object processing)
            if (!_.isEmpty(editAttributeSet)) {
                contents.a = contents.a ? docModel.extendAttributes(contents.a, editAttributeSet) : editAttributeSet;
            }

            // send current value to document model after returning to selection mode
            switch (docModel.getEditMode() ? commitMode : 'discard') {
                case 'cell':
                    // single cell mode: do not send operations if text and formatting did not change
                    if ((originalText !== editText) || contents.a) {
                        this.setCellContents(contents);
                    }
                    break;
                case 'fill':
                    // fill mode: always fill all cells in the selection
                    this.fillRangeContents(contents);
                    break;
                case 'matrix':
                    // TODO: matrix formulas
                    Utils.warn('CellEditMixin.leaveCellEditMode(): matrix formulas not implemented');
                    break;
            }

            return true;
        };

        /**
         * Returns the additional formatting attributes used in in-place cell
         * edit mode.
         *
         * @returns {Object|Null}
         *  The new attributes applied while the in-place cell edit mode is
         *  active.
         */
        this.getEditAttributeSet = function () {
            return editActive ? _.copy(editAttributeSet, true) : null;
        };

        /**
         * Modifies formatting attributes while the in-place cell edit mode is
         * active.
         *
         * @param {Object} [attributes]
         *  The new attributes to be applied to the text area. If omitted, the
         *  call of this method will be ignored.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.setEditAttributeSet = function (attributes) {

            function extendAttributeFamily(family) {
                if (_.isObject(attributes[family])) {
                    _.extend(editAttributeSet[family] || (editAttributeSet[family] = {}), attributes[family]);
                }
            }

            if (editActive && _.isObject(attributes)) {

                // extend the edit attribute set
                extendAttributeFamily('cell');
                extendAttributeFamily('character');

                // update the formatting of the text area, and the undo/redo stack
                updateTextAreaStyle();
                updateTextAreaPosition({ direct: true });
                saveUndoState();
            }

            return this;
        };

        // extended undo/redo -------------------------------------------------

        /**
         * Returns whether at least one undo action is available on the undo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * are changed currently.
         *
         * @returns {Boolean}
         *  Whether at least one undo action is available on the stack.
         */
        this.isUndoAvailable = function () {
            return editActive ? (undoIndex > 0) : (undoManager.getUndoCount() > 0);
        };

        /**
         * Applies the topmost undo action on the undo stack. In cell in-place
         * edit mode, restores the original contents of the cell.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the undo action has been
         *  applied.
         */
        this.undo = function () {
            if (editActive) {
                if (undoIndex > 0) {
                    undoIndex -= 1;
                    restoreUndoState();
                }
                return $.when();
            }
            return undoManager.undo();
        };

        /**
         * Returns whether at least one redo action is available on the redo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * have been changed, but are not changed anymore (after undo).
         *
         * @returns {Boolean}
         *  Whether at least one redo action is available on the stack.
         */
        this.isRedoAvailable = function () {
            return editActive ? (undoIndex + 1 < undoStack.length) : (undoManager.getRedoCount() > 0);
        };

        /**
         * Applies the topmost redo action on the redo stack. In cell in-place
         * edit mode, restores the last changed contents of the cell, after
         * these changes have been undone.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the redo action has been
         *  applied.
         */
        this.redo = function () {
            if (editActive) {
                if (undoIndex + 1 < undoStack.length) {
                    undoIndex += 1;
                    restoreUndoState();
                }
                return $.when();
            }
            return undoManager.redo();
        };

        // initialization -----------------------------------------------------

        // get localized names of all visible functions, as sorted array of objects
        // with 'key' and 'name' properties (used for the auto-completion list menu)
        functionEntries = [];
        formulaResource.getFunctionCollection().forEach(function (descriptor, key) {
            if (!descriptor.hidden) {
                functionEntries.push({ key: key, name: descriptor.localName });
            }
        });
        functionEntries = _.sortBy(functionEntries, 'name');

        // cancel cell edit mode after the application loses edit rights
        this.listenTo(docModel, 'change:editmode', editModeHandler);

        // update zoom settings during edit mode
        this.on('change:sheet:viewattributes', changeSheetViewAttributesHandler);

        // events for in-place edit mode
        textAreaNode.on({
            keydown: textAreaKeyDownHandler,
            keypress: textAreaKeyPressHandler,
            input: textAreaInputHandler,
            'mousedown mouseup touchstart touchend touchcancel': textAreaMouseTouchHandler,
            scroll: textAreaScrollHandler,
            'focus blur': textAreaFocusHandler
        });

        // handle click events in the auto-completion pop-up menu
        textAreaListMenu.getNode().on('click', Forms.BUTTON_SELECTOR, textAreaListClickHandler);

        // hide function description tooltip when closing the auto-completion menu
        textAreaListMenu.on('popup:hide', function () { textAreaListItemToolTip.hide(); });

        // destroy all class members on destruction
        this.registerDestructor(function () {

            textAreaContainer.remove();
            signatureToolTip.destroy();
            textAreaListMenu.destroy();
            textAreaListItemToolTip.destroy();

            self = docModel = undoManager = fontCollection = null;
            numberFormatter = formulaResource = formulaGrammar = formulaParser = null;
            textAreaContainer = textAreaNode = textUnderlayNode = null;
            signatureToolTip = textAreaListMenu = textAreaListItemToolTip = null;
            undoStack = null;
        });

    } // class CellEditMixin

    // exports ================================================================

    return CellEditMixin;

});
