/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/dialog/unhidesheetsdialog', [
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/control/checkgroup',
    'io.ox/office/tk/utils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Dialogs, CheckGroup, Utils, gt) {

    'use strict';

    // class UnhideSheetsDialog ==================================================

    /**
     * Shows a dialog that allows to unhide sheets, it shows a checkbox and the name of the
     * hidden sheet and a checkbox for unhide all sheets.
     *
     * @constructor
     *
     * @extends ModalInputDialog
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view instance that has created this dialog.
     *
     * @param {Function} actionHandler
     *  The callback function invoked when the OK button of the dialog has
     *  been clicked (will be passed as 'validator' option to the dialog).
     *  Receives a Array<Number> with all sheets to unhide, and MUST return
     *  a boolean value or a promise that states whether the unhide sheet operation
     *  has been finished successfully. In case of an error, the
     *  dialog will be kept open.
     */
    var UnhideSheetsDialog = Dialogs.ModalDialog.extend({ constructor: function (docView, actionHandler) {

        var self = this,
            checkGroup = new CheckGroup({ sorted: sortComparator, boxed: true, checkAll: true });

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, { width: 400, title: gt('Unhide Sheets'), validator: function () { return actionHandler.call(self, checkGroup.getValue()); } });

        /**
         * Compares the passed sort order descriptors stored at the list
         * entries of this control.
         */
        function sortComparator(sortIndex1, sortIndex2) {
            return sortIndex1.value - sortIndex2.value;
        }

        // public methods -----------------------------------------------------

        /**
         * Shows the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved after the
         *  action handler callback function has been invoked successfully; or
         *  will be rejected, if the dialog has been canceled, or if the
         *  document has switched to read-only mode.
         */
        this.show = _.wrap(this.show, function (show) {
            return show.call(this).then(null, function (result) {
                // reopen the dialog, if the action handler has failed
                return (result === 'invalid') ? new UnhideSheetsDialog(docView, actionHandler).show() : result;
            });
        });

        // initialization -----------------------------------------------------

        // close dialog automatically after losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        if (Utils.SMALL_DEVICE) {
            this.on('show', function () {
                self.getOkButton().focus();
            });
        }

        checkGroup.getNode().addClass('dialog-group');
        this.append(checkGroup.getNode());

        docView.getHiddenSheets().forEach(function (sheet) {
            var options = { label: _.noI18n(sheet.name), sortIndex: { value: sheet.sheet, display: sheet.name } };
            checkGroup.createOptionButton(sheet.sheet, options);
        });

    } }); // class UnhideSheetsDialog

    // exports ================================================================

    return UnhideSheetsDialog;

});
