/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/namedrangeitem', [
    'io.ox/office/tk/control/group',
    'io.ox/office/tk/control/button'
], function (Group, Button) {

    'use strict';

    // class NamedRangeItem ===================================================

    /**
     * A item displaying properties of an named range entry
     *
     * @constructor
     *
     * @extends Group
     */
    function NamedRangeItem(docView, nameModel) {

        var // the node, which holds name + formula nodes
            nameNode = $('<div class="name">'),

            // the node, which holds the name
            textNode = $('<span class="text">'),

            // the status node at the right border
            formulaNode = $('<span class="formula">'),

            // the color box in the status node
            actionsNode = $('<div class="actions">'),

            // the edit button that will open the 'Edit Name' dialog
            editButton = new Button({ icon: 'fa-pencil-square-o', value: nameModel.getLabel() }),

            // the delete button to remove a defined name from the document
            deleteButton = new Button({ icon: 'fa-trash-o', value: nameModel.getLabel() }),

            // the token array of the defined name model
            tokenArray = nameModel.getTokenArray(),

            // unique identifier of the current range highlighting mode for mouse-hover
            highlightUid = null;

        // base constructor ---------------------------------------------------

        Group.call(this);

        // private methods ----------------------------------------------------

        function setFormula() {
            var oldFormula = formulaNode.attr('title');
            var newFormula = nameModel.getFormula('ui', docView.getActiveCell());
            if (oldFormula !== newFormula) {
                formulaNode.attr('title', newFormula).text(_.noI18n(newFormula));
            }
        }

        function getExtractRangeOptions() {
            return {
                refSheet: docView.getActiveSheet(),
                targetAddress: docView.getActiveCell(),
                resolveNames: true
            };
        }

        function startHoverHighlighting() {
            highlightUid = docView.startRangeHighlighting(tokenArray, getExtractRangeOptions());
        }

        function endHoverHighlighting() {
            docView.endRangeHighlighting(highlightUid);
        }

        function selectNamedRange() {

            var // all ranges contained in the defined name
                rangeInfos = tokenArray.extractRanges(getExtractRangeOptions()),
                // select a single range in the token array
                range = (rangeInfos.length === 1) ? rangeInfos[0].range : null;

            // nothing to do for names without range address
            if (!range) { return; }

            // switch to the first sheet referred by the range
            if (!range.containsSheet(docView.getActiveSheet())) {
                docView.setActiveSheet(range.sheet1);
            }

            // scroll to start cell of the range (do not change the active grid pane in frozen split mode)
            docView.scrollToCell(range.start, { updateFocus: false });
        }

        // initialization -----------------------------------------------------

        // initialization of the root node of this badge
        this.getNode().addClass('named-range').attr('role', 'dialog');

        // insert the additional nodes into the group
        this.addChildNodes(nameNode, actionsNode);

        // add all values to the item entry
        (function () {

            // add current label and formula expression
            var label = nameModel.getLabel();
            textNode.attr('title', label).text(_.noI18n(label));
            setFormula();
            nameNode.append(textNode, formulaNode);

            // add the edit button
            editButton = new Button({ icon: 'fa-pencil-square-o', value: label });
            editButton.on('group:change', function () {
                docView.executeControllerItem('name/edit/dialog', label);
            });
            $(editButton.getNode()).toggleClass('disabled', !docView.getApp().getController().isItemEnabled('name/edit/dialog'));
            actionsNode.append(editButton.getNode());

            // create the delete button
            deleteButton = new Button({ icon: 'fa-trash-o', value: label });
            deleteButton.on('group:change', function () {
                docView.executeControllerItem('name/delete', label);
            });
            $(deleteButton.getNode()).toggleClass('disabled', !docView.getApp().getController().isItemEnabled('name/delete'));
            actionsNode.append(deleteButton.getNode());
        }());

        // add event listeners to highlight the cell ranges
        this.getNode().on({
            mouseenter: startHoverHighlighting,
            mouseleave: endHoverHighlighting,
            click: selectNamedRange
        });

        // update the relative references in the formula expression according to the selection
        this.listenTo(docView, 'change:selection', this.createDebouncedMethod(_.noop, setFormula, { delay: 500 }));

        // destroy all class members
        this.registerDestructor(function () {
            endHoverHighlighting();
            editButton.destroy();
            deleteButton.destroy();
            docView = nameModel = nameNode = formulaNode = actionsNode = editButton = deleteButton = null;
        });

    } // class NamedRangeItem

    // exports ================================================================

    // derive this class from class UserPicture
    return Group.extend({ constructor: NamedRangeItem });

});
