/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/cellstylepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/editframework/view/control/stylesheetpicker',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Forms, StyleSheetPicker, SheetUtils, gt) {

    'use strict';

    // class CellStylePicker ==================================================

    /**
     * A drop-down menu for cell style sheets.
     *
     * @constructor
     *
     * @extends StyleSheetPicker
     *
     * @param {SpreadsheetView} docView
     *  The document view containing this instance.
     */
    function CellStylePicker(docView) {

        // the document model
        var docModel = docView.getDocModel();

        // the collection of cell styles in the document
        var cellStyles = docModel.getCellStyles();

        // base constructor ---------------------------------------------------

        StyleSheetPicker.call(this, docView, 'cell', {
            width: 150,
            icon: 'docs-table-style',
            tooltip: /*#. tool tip: predefined styles for spreadsheet cells */ gt('Cell style'),
            gridColumns: 6,
            i18nModulePath: 'io.ox/office/spreadsheet/resource/cellstylenames',
            sections: ['markup', 'headings', 'themes', 'hidden'],
            smallerVersion: {
                css: { width: 56 },
                hideLabel: true
            }
        });

        // private methods ----------------------------------------------------

        /**
         * Updates the passed button element in the drop-down menu.
         */
        function createItemHandler(event, buttonNode, styleId) {

            // the merged attributes of the passed cell style sheet
            var attributeSet = cellStyles.getStyleAttributeSet(styleId);
            // the character attributes
            var charAttrs = attributeSet.character;
            // the cell attributes
            var cellAttrs = attributeSet.cell;
            // the caption element to be formatted
            var captionNode = Forms.getCaptionNode(buttonNode);
            // text orientation settings according to localized style name
            var textOrientation = SheetUtils.getTextOrientation(captionNode.text(), cellAttrs.alignHor);
            // an additional border node as overlay (to prevent different line height in caption)
            var borderMarkup = null;

            // TODO: use canvas for complex border styles
            borderMarkup = '<div class="borders" style="';
            borderMarkup += 'background-color:' + docModel.getCssColor(cellAttrs.fillColor, 'fill') + ';';
            borderMarkup += 'border-top:' + docModel.getCssBorder(cellAttrs.borderTop, { preview: true }) + ';';
            borderMarkup += 'border-bottom:' + docModel.getCssBorder(cellAttrs.borderBottom, { preview: true }) + ';';
            borderMarkup += 'border-left:' + docModel.getCssBorder(cellAttrs.borderLeft, { preview: true }) + ';';
            borderMarkup += 'border-right:' + docModel.getCssBorder(cellAttrs.borderRight, { preview: true }) + ';';
            borderMarkup += '"></div>';
            buttonNode.prepend(borderMarkup);

            // add relevant character and cell formatting
            captionNode.css({
                fontFamily: docModel.getCssFontFamily(charAttrs.fontName),
                fontSize: Utils.minMax(Utils.round(10 + (charAttrs.fontSize - 10) / 1.5, 0.1), 6, 22) + 'pt',
                fontWeight: charAttrs.bold ? 'bold' : 'normal',
                fontStyle: charAttrs.italic ? 'italic' : 'normal',
                textDecoration: docModel.getCssTextDecoration(charAttrs),
                color: docModel.getCssTextColor(charAttrs.color, [cellAttrs.fillColor]),
                textAlign: textOrientation.cssTextAlign
            });
        }

        // initialization -----------------------------------------------------

        // add visible formatting of the cell styles to the list item buttons
        this.getMenu().on('create:item', createItemHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView = docModel = cellStyles = null;
        });

    } // class CellStylePicker

    // exports ================================================================

    // derive this class from class StyleSheetPicker
    return StyleSheetPicker.extend({ constructor: CellStylePicker });

});
