/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/control/activesheetlist', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/baseframework/app/appobjectmixin',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, RadioList, AppObjectMixin, gt) {

    'use strict';

    // class ActiveSheetList ==================================================

    /**
     * The selector for the current active sheet, as radio drop-down list.
     *
     * @constructor
     *
     * @extends RadioList
     * @extends AppObjectMixin
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Will be passed to the base class RadioList.
     */
    function ActiveSheetList(docView, initOptions) {

        // self reference
        var self = this;

        // the document model
        var docModel = docView.getDocModel();

        // whether to show the sheet name as label text
        var showNames = Utils.getBooleanOption(initOptions, 'showNames', false);

        // base constructors --------------------------------------------------

        RadioList.call(this, {
            icon: showNames ? null : 'fa-bars',
            iconClasses: showNames ? 'small-icon' : null,
            tooltip: gt('Select sheet'),
            caret: showNames,
            updateCaptionMode: showNames ? 'label icon' : 'none'
        });

        AppObjectMixin.call(this, docView.getApp());

        // private methods ----------------------------------------------------

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        var fillList = this.createDebouncedMethod($.noop, function () {
            self.clearMenu();
            docView.iterateVisibleSheets(function (sheetModel, index, sheetName) {
                self.createOptionButton(index, {
                    icon: sheetModel.isLocked() ? 'fa-lock' : null,
                    iconClasses: 'small-icon',
                    label: _.noI18n(sheetName),
                    labelStyle: 'white-space:pre;',
                    tooltip: _.noI18n(sheetName)
                });
            });
        });

        // initialization -----------------------------------------------------

        // initialization after import
        this.waitForImportSuccess(function () {
            // build the list of available sheets once after import
            fillList();
            // rebuild the list of available sheets after it has been changed
            this.listenTo(docView, 'change:sheets', fillList);
            this.listenTo(docModel, 'change:sheet:attributes', fillList); // e.g. sheet protection
        }, this);

        // register handler early for fast preview during import
        this.listenTo(docView, 'preview:activesheet', fillList);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = docView = initOptions = null;
        });

    } // class ActiveSheetList

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: ActiveSheetList });

});
