/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/rangeset', [
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/spreadsheet/utils/rangearray',
    'io.ox/office/spreadsheet/utils/intervalset'
], function (SimpleMap, RangeArray, IntervalSet) {

    'use strict';

    // class RangeSet =========================================================

    /**
     * A unique set of cell range addresses. This class provides optimized
     * methods for insertion, deletion, and access of cell range addresses,
     * while keeping the set unique (two different range addresses in this set
     * will never have the same start end end address).
     *
     * @constructor
     */
    function RangeSet() {

        // the interval set storing the column intervals of all cell ranges in this set
        this._colSet = new IntervalSet();
        // a plain map that stores all ranges by their keys, for fast direct access
        this._map = new SimpleMap();

    } // class RangeSet

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this set is empty.
     *
     * @returns {Boolean}
     *  Whether this set is empty.
     */
    RangeSet.prototype.empty = function () {
        return this._map.empty();
    };

    /**
     * Removes all elements from this set.
     *
     * @returns {RangeSet}
     *  A reference to this instance.
     */
    RangeSet.prototype.clear = function () {
        this._colSet.clear();
        this._map.clear();
        return this;
    };
    /**
     * Returns all cell range addresses contained in this set as an array.
     *
     * @returns {RangeArray}
     *  All cell range addresses contained in this set as an array, in no
     *  specific order.
     */
    RangeSet.prototype.values = function () {
        var ranges = new RangeArray();
        this.forEach(function (range) { ranges.push(range); });
        return ranges;
    };

    /**
     * Returns a shallow or deep clone of this range set.
     *
     * @param {Boolean|Function} [deep=false]
     *  If set to true or to a function, a deep clone will be created (all
     *  ranges in this set will be cloned); otherwise a shallow clone will be
     *  created (the ranges in this set will be inserted by reference into the
     *  clone). If a function has been passed, it will be invoked for each
     *  cloned cell range, receiving the new cloned cell range inserted into
     *  the cloned set as first parameter, and the old cell range of this set
     *  as second parameter.
     *
     * @param {Object} [context]
     *  The calling context used if the 'deep' parameter is a callback.
     *
     * @returns {RangeSet}
     *  The cloned range set.
     */
    RangeSet.prototype.clone = function (deep, context) {

        // start with a new empty range set
        var clone = new RangeSet();
        // user callback for deep mode
        var callback = _.isFunction(deep) ? deep : _.noop;

        // create a deep clone of the interval sets (shallow mode only for the final ranges!)
        clone._colSet = this._colSet.clone(function (newColInterval, oldColInterval) {
            newColInterval.rowSet = oldColInterval.rowSet.clone(function (newRowInterval, oldRowInterval) {
                var oldRange = oldRowInterval.range;
                var newRange = deep ? oldRange.clone() : oldRange;
                callback.call(context, newRange, oldRange);
                newRowInterval.range = newRange;
                clone._map.insert(newRange.key(), newRange);
            });
        });

        return clone;
    };

    /**
     * Returns whether this set contains a cell range address that has the same
     * position as the passed range address.
     *
     * @param {Range} range
     *  A cell range address to check.
     *
     * @returns {Boolean}
     *  Whether this set contains a cell range address that has the same
     *  position as the passed range address.
     */
    RangeSet.prototype.has = function (range) {
        return this._map.has(range.key());
    };

    /**
     * Returns the cell range address from this set that has the exact position
     * of the passed range address.
     *
     * @param {Range} range
     *  A cell range address to search for.
     *
     * @returns {Range|Null}
     *  An range object from this set with the same start and end address as
     *  the passed range address; otherwise null.
     */
    RangeSet.prototype.get = function (range) {
        return this._map.get(range.key(), null);
    };

    /**
     * Returns whether this set contains a cell range covering the passed cell
     * address.
     *
     * @param {Address} address
     *  The cell address used to look for a cell range in this set.
     *
     * @returns {Boolean}
     *  Whether this set contains a cell range covering the passed address.
     */
    RangeSet.prototype.containsAddress = function (address) {
        var colIntervals = this._colSet.findByIndex(address[0]);
        return colIntervals.some(function (colInterval) {
            return colInterval.rowSet.containsIndex(address[1]);
        });
    };

    /**
     * Returns all cell ranges in this set that cover the passed cell address.
     *
     * @param {Address} address
     *  The cell address used to look for the cell ranges in this set.
     *
     * @returns {RangeArray}
     *  An array with all cell ranges in this set that cover the passed cell
     *  address, in no specific order.
     */
    RangeSet.prototype.findByAddress = function (address) {
        var ranges = new RangeArray();
        var colIntervals = this._colSet.findByIndex(address[0]);
        colIntervals.forEach(function (colInterval) {
            var rowIntervals = colInterval.rowSet.findByIndex(address[1]);
            rowIntervals.forEach(function (rowInterval) {
                ranges.push(rowInterval.range);
            });
        });
        return ranges;
    };

    /**
     * Returns whether this set contains a range that overlaps with the passed
     * cell ranges.
     *
     * @param {RangeArray|Range} ranges
     *  An array of cell range addresses, or a single cell range address, used
     *  to look for overlapping ranges in this set.
     *
     * @returns {Boolean}
     *  Whether this set contains a cell range overlapping with the passed cell
     *  ranges.
     */
    RangeSet.prototype.overlaps = function (ranges) {

        // quick check for an exact match
        ranges = RangeArray.get(ranges);
        if (ranges.some(this.has, this)) { return true; }

        // check each range if it overlaps somehow
        return ranges.some(function (range) {
            var rowInterval = range.rowInterval();
            var colIntervals = this._colSet.findIntervals(range.colInterval());
            return colIntervals.some(function (colInterval) {
                return colInterval.rowSet.overlaps(rowInterval);
            }, this);
        }, this);
    };

    /**
     * Returns all ranges in this set that overlap with the passed cell range.
     *
     * @param {Range} range
     *  The cell range used to look for the ranges in this set.
     *
     * @returns {RangeArray}
     *  An array with all ranges in this set overlapping with the passed cell
     *  range, in no specific order.
     */
    RangeSet.prototype.findRanges = function (range) {
        var ranges = new RangeArray();
        var rowInterval = range.rowInterval();
        var colIntervals = this._colSet.findIntervals(range.colInterval());
        colIntervals.forEach(function (colInterval) {
            var rowIntervals = colInterval.rowSet.findIntervals(rowInterval);
            rowIntervals.forEach(function (interval) {
                ranges.push(interval.range);
            });
        });
        return ranges;
    };

    /**
     * Invokes the passed callback for each cell range in this set.
     *
     * @param {Function} callback
     *  The callback function to be invoked for each cell range address in this
     *  set. Receives the current range address as first parameter, and its map
     *  key as second parameter.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {RangeSet}
     *  A reference to this instance.
     */
    RangeSet.prototype.forEach = function (callback, context) {
        this._map.forEach(callback, context);
        return this;
    };

    /**
     * Creates an iterator for all cell ranges in this set.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the cell ranges have been visited completely. No
     *      more elements are available; this result object does not contain
     *      any other properties!
     *  - {Range} value
     *      The cell range address currently visited.
     */
    RangeSet.prototype.iterator = function () {
        return this._map.iterator();
    };

    /**
     * Inserts a new cell range address into this set.
     *
     * @param {Range} range
     *  The cell range address to be inserted into this set.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, an existing cell range with equal start and end address
     *  as the passed cell range will be replaced with the new cell range. By
     *  default, the existing cell range will be kept, and the passed cell
     *  range will be ignored. May be important in situations where the ranges
     *  in the set contain additional user settings in arbitrary properties.
     *
     * @returns {Range}
     *  A reference to the cell range contained in this set. This may not be
     *  the same as the passed cell range, e.g. if an equal cell range already
     *  exists in this set, and the flag 'replace' has not been set.
     */
    RangeSet.prototype.insert = function (range, replace) {
        var colInterval = this._colSet.insert(range.colInterval());
        var rowSet = colInterval.rowSet || (colInterval.rowSet = new IntervalSet());
        var rowInterval = rowSet.insert(range.rowInterval());
        if (replace || !rowInterval.range) { rowInterval.range = range; }
        return this._map.insert(range.key(), rowInterval.range);
    };

    /**
     * Removes the specified cell range address from this set.
     *
     * @param {Range} range
     *  The cell range address to be removed from this set.
     *
     * @returns {Range|Null}
     *  A reference to the removed cell range; or null, if this set does not
     *  contain a cell range that is equal to the passed cell range.
     */
    RangeSet.prototype.remove = function (range) {
        range = this._map.remove(range.key());
        if (!range) { return null; }
        var colInterval = this._colSet.get(range.colInterval());
        var rowInterval = colInterval.rowSet.remove(range.rowInterval());
        if (colInterval.rowSet.empty()) { this._colSet.remove(colInterval); }
        return rowInterval.range;
    };

    /**
     * Inserts all cell ranges of the passed set into this set.
     *
     * @param {RangeSet} rangeSet
     *  The range set to be merged into this range set.
     *
     * @param {Boolean} [replace=false]
     *  If set to true, cell ranges in the passed range set will replace equal
     *  cell ranges in this set. By default, the existing cell ranges of this
     *  set will be preferred.
     *
     * @returns {RangeSet}
     *  A reference to this instance.
     */
    RangeSet.prototype.merge = function (rangeSet, replace) {
        rangeSet.forEach(function (range) { this.insert(range, replace); }, this);
        return this;
    };

    // exports ================================================================

    return RangeSet;

});
