/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/tablecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/scalarset',
    'io.ox/office/spreadsheet/model/cellvaluecache',
    'io.ox/office/spreadsheet/model/celloperationsbuilder'
], function (Utils, SimpleMap, IteratorUtils, TimerMixin, ModelObject, AttributedModel, Operations, SheetUtils, ScalarSet, CellValueCache, CellOperationsBuilder) {

    'use strict';

    // convenience shortcuts
    var Interval = SheetUtils.Interval;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var IntervalArray = SheetUtils.IntervalArray;
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;

    // private global functions ===============================================

    /**
     * Returns the bounding range of the passed range addresses. Either range
     * address may be missing (null).
     *
     * @param {Range|Null} range1
     *  The first range address.
     *
     * @param {Range|Null} range2
     *  The second range address.
     *
     * @returns {Range|Null}
     *  The bounding range of the passed range addresses; or null, if both
     *  ranges are missing.
     */
    function combineRanges(range1, range2) {
        return (range1 && range2) ? range1.boundary(range2) : (range1 || range2 || null);
    }

    // class TableColumnModel =================================================

    /**
     * Stores filter and sorting settings for a single column in a table range.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this table column.
     *
     * @param {SheetModel} tableModel
     *  The table model instance containing this table column.
     *
     * @param {Object} [initAttrs]
     *  The initial table attribute set (style sheet reference, and explicit
     *  table attributes).
     */
    var TableColumnModel = AttributedModel.extend({ constructor: function (sheetModel, tableModel, initAttrs) {

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, sheetModel.getDocModel(), initAttrs, { families: 'filter sort', autoClear: true });

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this table model for the
         * specified sheet.
         *
         * @internal
         *  Used by the class TableCollection during clone construction. DO NOT
         *  CALL from external code!
         *
         * @param {SheetModel} targetSheetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @param {TableModel} targetTableModel
         *  The model instance of the new cloned table that will own the clone
         *  returned by this method.
         *
         * @returns {TableColumnModel}
         *  A clone of this table column model, initialized for ownership by
         *  the passed sheet and table models.
         */
        this.clone = function (targetSheetModel, targetTableModel) {
            // construct a new model instance
            return new TableColumnModel(targetSheetModel, targetTableModel, this.getExplicitAttributeSet(true));
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether this column model contains active filter rules.
         *
         * @returns {Boolean}
         *  Whether this column model contains active filter rules.
         */
        this.isFiltered = function () {
            return this.getMergedAttributeSet(true).filter.type !== 'none';
        };

        /**
         * Returns whether this column model contains active sort options.
         *
         * @returns {Boolean}
         *  Whether this column model contains active sort options.
         */
        this.isSorted = function () {
            return false; // TODO
        };

        /**
         * Returns whether this column model contains any filter rules, or sort
         * options, that may require to refresh the table range containing this
         * column.
         *
         * @returns {Boolean}
         *  Whether this column model contains any filter/sorting settings.
         */
        this.isRefreshable = function () {
            return this.isFiltered() || this.isSorted();
        };

        // initialization -----------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            sheetModel = tableModel = null;
        });

    } }); // class TableColumnModel

    // class TableModel =======================================================

    /**
     * Stores settings for a single table range in a specific sheet.
     *
     * @constructor
     *
     * @extends AttributedModel
     * @extends TimerMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this table.
     *
     * @param {String} tableName
     *  The name of this table.
     *
     * @param {Range} tableRange
     *  The address of the cell range covered by this table.
     *
     * @param {Object} [initAttrs]
     *  The initial table attribute set (style sheet reference, and explicit
     *  table attributes).
     */
    var TableModel = AttributedModel.extend({ constructor: function (sheetModel, tableName, tableRange, initAttrs) {

        // self reference
        var self = this;

        // the spreadsheet document model
        var docModel = sheetModel.getDocModel();

        // the cell collection used to resolve header labels etc.
        var cellCollection = sheetModel.getCellCollection();

        // all existing column models (with active filter or sorting settings), mapped by column index (for fast direct access)
        var columnModelMap = new SimpleMap();

        // default column model for all unmodified columns
        var defColumnModel = new TableColumnModel(sheetModel, this);

        // the cache for header cell labels (used as column names)
        var headerCache = new CellValueCache(sheetModel, tableRange);

        // base constructors --------------------------------------------------

        AttributedModel.call(this, docModel, initAttrs, { styleFamily: 'table', autoClear: true });
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Sets a new table range, and updates the column index map.
         *
         * @param {Range} newRange
         *  The new table range.
         *
         * @returns {Boolean}
         *  Whether the table range has actually been changed.
         */
        function changeRange(newRange) {

            // store the new range (unless nothing will change)
            if (tableRange.equals(newRange)) { return false; }
            tableRange = newRange.clone();

            // reset the map used to resolve column names to their indexes (will be created on demand)
            headerCache.setRanges(self.getHeaderRange());
            return true;
        }

        /**
         * Returns a map that containes the zero-based indexes of the columns
         * of this table, mapped by their keys (upper-case column names).
         *
         * @returns {SimpleMap<Number>}
         *  A map with the zero-based indexes of the columns of this table,
         *  mapped by their keys.
         */
        function getColumnIndexMap() {
            return headerCache.getCustom('colIndexMap', function (iterator) {

                // the column index map to be inserted into the value cache
                var columnIndexMap = new SimpleMap();
                // first column index of the table range (the map stores relative indexes)
                var firstCol = tableRange.start[0];

                // collect the column names from all string cells
                IteratorUtils.forEach(iterator, function (address) {
                    var value = cellCollection.getCellValue(address);
                    if (typeof value === 'string') {
                        columnIndexMap.insert(SheetUtils.getTableKey(value), address[0] - firstCol);
                    }
                });

                return columnIndexMap;
            });
        }

        /**
         * Inserts the passed table column model into this table, and registers
         * a listener for attribute change events that will be forwarded to the
         * listeners of this table.
         */
        function insertColumnModel(tableCol, columnModel) {

            // the map entry, containing the current table column index as property (may change during runtime,
            // e.g. after inserting/deleting sheet columns, needed for fast access when forwarding events below)
            var entry = { tableCol: tableCol, columnModel: columnModel };

            // store the column model as map entry (needed for fast direct access by column index)
            columnModelMap.insert(tableCol, entry);

            // listen to attribute change events, forward to own listeners
            self.listenTo(columnModel, 'change:attributes', function (event, newAttrs, oldAttrs) {

                // forward to listeners of this table model (pass current column index, not the
                // original column index passed to the method insertColumnModel(), see comment above)
                self.trigger('change:column', entry.tableCol, newAttrs, oldAttrs);

                // model can be deleted, if it does not contain any active settings
                if (!columnModel.hasExplicitAttributes()) {
                    columnModel.destroy();
                    columnModelMap.remove(entry.tableCol);
                }
            });
        }

        /**
         * Returns the specified table column model, if it exists.
         */
        function getColumnModel(tableCol) {
            var entry = columnModelMap.get(tableCol);
            return entry ? entry.columnModel : null;
        }

        /**
         * Creates a missing table column model, and returns the model
         * instance. The passed column index will not be checked for validity.
         */
        function getOrCreateColumnModel(tableCol) {

            // find an existing column model
            var columnModel = getColumnModel(tableCol);

            // create missing column model
            if (!columnModel) {
                columnModel = new TableColumnModel(sheetModel, self);
                insertColumnModel(tableCol, columnModel);
            }

            return columnModel;
        }

        /**
         * Updates internal settings after the attributes of this table have
         * been changed.
         */
        function changeAttributesHandler(event, newAttributes, oldAttributes) {

            // reset the header cache if the existence of the table header changes
            // TODO: filter needs to send column names for tables without header row
            if (newAttributes.table.headerRow !== oldAttributes.table.headerRow) {
                headerCache.setRanges(self.getHeaderRange());
            }

            // silently remove filter rules from all filtered column models, when
            // table is not active anymore (sorting settings remain valid though)
            if (!self.isFilterActive()) {
                columnModelMap.forEach(function (entry, key) {
                    // remove filter settings from the column
                    entry.columnModel.setAttributes({ filter: { type: 'none' } }, { notify: 'never' });
                    // keep the column model in the array, if any attributes are left
                    if (!entry.columnModel.hasExplicitAttributes()) {
                        entry.columnModel.destroy();
                        columnModelMap.remove(key);
                    }
                });
            }
        }

        /**
         * Creates a filter matcher for the passed entries of a discrete filter
         * rule.
         *
         * @param {Array<String|Number|Boolean>} entries
         *  The matching entries of a discrete filter rule.
         *
         * @returns {Function}
         *  A predicate function that expects the typed cell result value, and
         *  the formatted display string as parameters, and returns a Boolean
         *  whether the passed cell value matches any of the entries in the
         *  string array.
         */
        function createDiscreteFilterMatcher(entries) {

            // no matching entries: all cell values will be filtered
            if (entries.length === 0) {
                return _.constant(false);
            }

            // performance: create a flag set with all matching values as keys, instead of searching in the entry array
            var entrySet = new ScalarSet();
            entries.forEach(function (value) { entrySet.add(value); });

            // ODF: create a matcher predicate function that checks the typed result value of a cell
            if (docModel.getApp().isODF()) {
                return function discreteFilterMatcherODF(value) {
                    return entrySet.has(value);
                };
            }

            // OOXML: create a matcher predicate function that checks the lower-case display string of a cell
            return function discreteFilterMatcherOOX(result, display) {
                // bug 36264: trim space characters but no other whitespace
                return entrySet.has(display.replace(/^ +| +$/g, ''));
            };
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this table model for the
         * specified sheet.
         *
         * @internal
         *  Used by the class TableCollection during clone construction. DO NOT
         *  CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {TableModel}
         *  A clone of this table model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel, newTableName) {
            // construct a new model instance, pass all own column models as hidden parameter
            return new TableModel(targetModel, newTableName, tableRange, this.getExplicitAttributeSet(true), columnModelMap);
        };

        /**
         * Callback handler for the document operation 'changeTableColumn'.
         * Changes the attributes of a column in this table model.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeTableColumn' document operation.
         */
        this.applyChangeOperation = function (context) {

            // check that the column index exists in the table
            var tableCol = context.getInt('col');
            context.ensure((tableCol >= 0) && (tableCol < tableRange.cols()), 'invalid table column');

            // change the column attributes
            var attributes = context.getObj('attrs');
            getOrCreateColumnModel(tableCol).setAttributes(attributes);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the parent sheet model of this table range.
         *
         * @returns {SheetModel}
         *  The parent sheet model of this table range.
         */
        this.getSheetModel = function () {
            return sheetModel;
        };

        /**
         * Returns the name of this table.
         *
         * @returns {String}
         *  The name of this table.
         */
        this.getName = function () {
            return tableName;
        };

        /**
         * Returns the unique map key of this table (the uppercase name of the
         * table).
         *
         * @returns {String}
         *  The unique map key of this table.
         */
        this.getKey = function () {
            return SheetUtils.getTableKey(tableName);
        };

        /**
         * Returns whether this table id the special auto-filter range of the
         * sheet.
         *
         * @returns {Boolean}
         *  Whether this table id the special auto-filter range of the sheet.
         */
        this.isAutoFilter = function () {
            return tableName.length === 0;
        };

        /**
         * Returns the address of the cell range covered by this table.
         *
         * @returns {Range}
         *  The address of the cell range covered by this table.
         */
        this.getRange = function () {
            return tableRange.clone();
        };

        /**
         * Returns whether the passed cell address is covered by this table.
         *
         * @param {Address} address
         *  The address of the cell to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified cell is covered by this table.
         */
        this.containsCell = function (address) {
            return tableRange.containsAddress(address);
        };

        /**
         * Returns whether the passed cell range address overlaps with the cell
         * range of this table.
         *
         * @param {Range} range
         *  The address of the cell range to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed cell range address overlaps with the cell range
         *  of this table.
         */
        this.overlapsRange = function (range) {
            return tableRange.overlaps(range);
        };

        /**
         * Returns whether this table range contains a header row. Cells in the
         * header row (the top-most row) will neither be filtered nor sorted.
         *
         * @returns {Boolean}
         *  Whether this table range contains a header row.
         */
        this.hasHeaderRow = function () {
            return this.getMergedAttributeSet(true).table.headerRow;
        };

        /**
         * Returns whether this table range contains a footer row. Cells in the
         * footer row (the bottom-most row) will neither be filtered nor
         * sorted.
         *
         * @returns {Boolean}
         *  Whether this table range contains a footer row.
         */
        this.hasFooterRow = function () {
            return this.getMergedAttributeSet(true).table.footerRow;
        };

        /**
         * Returns whether filtering in this table range is activated. This
         * does not necessarily mean that a row has actually been hidden by the
         * filter rules.
         *
         * @returns {Boolean}
         *  Whether filtering in this table range is activated.
         */
        this.isFilterActive = function () {
            return this.getMergedAttributeSet(true).table.filtered;
        };

        /**
         * Returns the address of the header row of this table.
         *
         * @returns {Range|Null}
         *  The address of the header row of this table; or null if this table
         *  does not contain a header row.
         */
        this.getHeaderRange = function () {
            return this.hasHeaderRow() ? tableRange.headerRow() : null;
        };

        /**
         * Returns the address of the footer row of this table.
         *
         * @returns {Range|Null}
         *  The address of the footer row of this table; or null if this table
         *  does not contain a footer row.
         */
        this.getFooterRange = function () {
            return this.hasFooterRow() ? tableRange.footerRow() : null;
        };

        /**
         * Returns the address of the data range in this table (the cell range
         * without the header and footer rows). For the special auto-filter
         * table, the data range will be expanded to the bottom to include all
         * existing content cells following the real data range.
         *
         * @returns {Range|Null}
         *  The address of the data range of this table; or null, if no data
         *  rows are available (e.g., if the table consists of a header row
         *  only).
         */
        this.getDataRange = function () {

            // whether the table contains header and/or footer rows
            var hasHeader = this.hasHeaderRow();
            var hasFooter = this.hasFooterRow();
            // the resulting data range
            var dataRange = tableRange.clone();

            // expand data range of auto-filter to the bottom (before removing
            // the header row from the range, to not invalidate the range address)
            if (!hasFooter && this.isAutoFilter()) {
                dataRange = cellCollection.findContentRange(dataRange, { directions: 'down' });

                // restrict expansion to the next table range below the auto-filter
                if (tableRange.end[1] < dataRange.end[1]) {
                    var boundRange = Range.create(tableRange.start[0], tableRange.end[1] + 1, tableRange.end[0], docModel.getMaxRow());
                    sheetModel.getTableCollection().findTables(boundRange).forEach(function (otherTable) {
                        var otherTableRange = otherTable.getRange();
                        dataRange.end[1] = Math.min(dataRange.end[1], otherTableRange.start[1] - 1);
                    });
                }
            }

            // exclude header/footer rows from returned range
            if (hasHeader) { dataRange.start[1] += 1; }
            if (hasFooter) { dataRange.end[1] -= 1; }

            // check existence of data rows
            return (dataRange.start[1] <= dataRange.end[1]) ? dataRange : null;
        };

        /**
         * Returns the range address of the specified table region.
         *
         * @param {String} regionKey
         *  The resource key of a table region. The following table regions are
         *  supported:
         *  - 'ALL': The entire table range with header and footer row.
         *  - 'HEADERS': The entire header row, if existing.
         *  - 'DATA': The entire data range, if existing.
         *  - 'TOTALS': The entire footer row, if existing.
         *  - 'ROW': A single row of the data range, according to the passed
         *      target address.
         *  - 'HEADERS,DATA': The combined range covering the header row, and
         *      the data range.
         *  - 'DATA,TOTALS': The combined range covering the data range, and
         *      the footer row.
         *
         * @param {Address} [targetAddress]
         *  The target address needed to resolve the special table region
         *  'ROW'. Can be omitted for all other regions.

         * @returns {Range|Null}
         *  The range address of the specified table region; or null, if the
         *  passed region key is invalid, or if a region does not exist (e.g. a
         *  table range without header row).
         */
        this.getRegionRange = function (regionKey, targetAddress) {

            switch (regionKey) {
                case 'ALL':
                    return this.getRange();
                case 'HEADERS':
                    return this.getHeaderRange();
                case 'DATA':
                    return this.getDataRange();
                case 'TOTALS':
                    return this.getFooterRange();
                case 'ROW':
                    var dataRange = targetAddress ? this.getDataRange() : null;
                    return (dataRange && dataRange.containsIndex(targetAddress[1])) ? dataRange.setBoth(targetAddress[1], false) : null;
                case 'HEADERS,DATA':
                    return combineRanges(this.getHeaderRange(), this.getDataRange());
                case 'DATA,TOTALS':
                    return combineRanges(this.getDataRange(), this.getFooterRange());
            }

            return null;
        };

        /**
         * Returns the relative intervals of all visible rows in the data range
         * of this table.
         *
         * @returns {IntervalArray}
         *  The visible row intervals in the data range of this table; or an
         *  empty array, if the table does not contain any data rows. The
         *  properties 'first' and 'last' of each index interval are relative
         *  to the data range of the table (the index 0 refers to the first
         *  data row of this table!).
         */
        this.getVisibleDataRows = function () {

            // the address of the data range
            var dataRange = this.getDataRange();

            // return empty interval array, if data range is missing
            if (!dataRange) { return new IntervalArray(); }

            // the absolute row intervals
            var rowIntervals = sheetModel.getRowCollection().getVisibleIntervals(dataRange.rowInterval());

            // transform to relative row intervals
            return rowIntervals.move(-dataRange.start[1]);
        };

        /**
         * Returns whether the selected rows can be deleted from the sheet,
         * according to the position of this table. The header row of the table
         * cannot be deleted (except for auto-filters), and at least one data
         * row must remain in the table.
         *
         * @param {IntervalArray|Interval} intervals
         *  The row intervals intended to be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected rows can be deleted from the sheet.
         */
        this.canDeleteRows = function (intervals) {

            // any row in an auto-filter can be deleted
            if (this.isAutoFilter()) { return true; }

            // header rows cannot be deleted
            if (this.hasHeaderRow() && intervals.containsIndex(tableRange.start[1])) {
                return false;
            }

            // at least one data row must remain in the table
            var dataRange = this.getDataRange();
            if (dataRange && IntervalArray.get(dataRange.rowInterval()).difference(intervals).empty()) {
                return false;
            }

            return true;
        };

        /**
         * Changes the address of the cell range covered by this table.
         *
         * @param {Range} range
         *  The address of the new cell range.
         *
         * @returns {TableModel}
         *  A reference to this instance.
         */
        this.setRange = function (range) {

            // nothing to do, if the range does not change
            if (!changeRange(range)) { return this; }

            // delete column models outside the new range
            var colCount = tableRange.cols();
            columnModelMap.forEach(function (entry, key) {
                if (entry.tableCol >= colCount) {
                    entry.columnModel.destroy();
                    columnModelMap.remove(key);
                }
            });

            // notify listeners
            this.trigger('change:range', tableRange.clone());
            return this;
        };

        /**
         * Transforms the cell range address of this table, after cells have
         * been moved in the sheet, including inserted/deleted columns or rows.
         *
         * @param {MoveDescriptor} moveDesc
         *  A move descriptor that specify how to transform the target range of
         *  this table model.
         *
         * @returns {Boolean}
         *  Whether the transformed range is still valid (not completely
         *  deleted). If false is returned, this table model must be deleted
         *  from the table collection.
         */
        this.transformRange = function (moveDesc) {

            // process move descriptors that cover the table range completely in move direction
            if (!moveDesc.coversRange(tableRange)) { return true; }

            // first column index of the old table range
            var startCol = tableRange.start[0];

            // preprocessing when deleting rows in the table range
            if (!moveDesc.insert && !moveDesc.columns) {

                // bug 36226: delete tables whose header row has been deleted
                // bug 39869: deleting header rows prevented by UI except for auto-filter
                if (this.hasHeaderRow() && moveDesc.deleteIntervals.containsIndex(tableRange.start[1])) {
                    return false;
                }
            }

            // transform the table range (nothing more to do, if the range does not change at all)
            var newTableRange = moveDesc.transformRange(tableRange);
            if (!newTableRange) { return false; }
            if (!changeRange(newTableRange)) { return true; }

            // delete or remap all affected column models
            if (moveDesc.columns) {

                // create a new model map to prevent interfering with existing map entries
                var newColumnModelMap = new SimpleMap();
                columnModelMap.forEach(function (entry) {

                    // transform the absolute column index according to the operation
                    var address = new Address(startCol + entry.tableCol, tableRange.start[1]);
                    address = moveDesc.transformAddress(address);

                    // delete column model of deleted columns, remap remaining column models
                    if (address) {
                        entry.tableCol = address[0] - tableRange.start[0];
                        newColumnModelMap.insert(entry.tableCol, entry);
                    } else {
                        entry.columnModel.destroy();
                    }
                });

                columnModelMap = newColumnModelMap;
            }

            // notify listeners
            this.trigger('change:range', tableRange.clone());
            return true;
        };

        /**
         * Returns the zero-based index of the column with the passed header
         * text (contents of the header cell).
         *
         * @param {String} colName
         *  The name (contents of the header cell) of the table column.
         *
         * @returns {Number|Null}
         *  The zero-based index of the column with the specified name if
         *  existing; otherwise the value null.
         */
        this.findColumnByName = function (colName) {
            return getColumnIndexMap().get(SheetUtils.getTableKey(colName), null);
        };

        /**
         * Returns the merged column attributes of the specified table column.
         *
         * @param {Number} tableCol
         *  The zero-based index of the table column, relative to the cell
         *  range covered by this table.
         *
         * @param {Boolean} [direct=false]
         *  If set to true, the returned attribute set will be a reference to
         *  the original map stored in the table column model, which MUST NOT
         *  be modified! By default, a deep clone of the attribute set will be
         *  returned that can be freely modified.
         *
         * @returns {Object|Null}
         *  The merged attribute set of the specified table column; or null, if
         *  the passed table column index is invalid.
         */
        this.getColumnAttributeSet = function (tableCol, direct) {

            // the column model (fall-back to default model for unmodified columns)
            var columnModel = getColumnModel(tableCol) || defColumnModel;

            return columnModel.getMergedAttributeSet(direct);
        };

        /**
         * Returns the address of the data range of a single column in this
         * table, without the header and footer cell.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @returns {Range|Null}
         *  The address of the data range in the specified table column; or
         *  null, if no data rows are available (e.g., if the table consists of
         *  a header row only), or if the passed column index is invalid.
         */
        this.getColumnDataRange = function (tableCol) {

            // the data range of the entire table
            var dataRange = this.getDataRange();
            // the resulting absolute column index
            var absCol = tableRange.start[0] + tableCol;

            // check validity of passed column index, and existence of data rows
            if (!dataRange || !tableRange.containsCol(absCol)) {
                return null;
            }

            // adjust column indexes in the result range
            return dataRange.setBoth(absCol, true);
        };

        /**
         * Returns the cell contents of the data range of a single column in
         * this table.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @param {IntervalArray|Interval} [rowIntervals]
         *  If specified, an array of row intervals, or a single row interval,
         *  in the data range of this table to query the cell data for. The
         *  properties 'first' and 'last' of each interval object are relative
         *  to the data range of the table (the index 0 refers to the first
         *  data row of this table!). If omitted, returns the cell contents of
         *  all data cells in the column.
         *
         * @returns {Array<Object>}
         *  A compressed array of cell contents in the specified table column,
         *  containing the cell values, and the formatted display strings. See
         *  method CellCollection.getRangeContents() for more details.
         */
        this.getColumnContents = function (tableCol, rowIntervals) {

            // the address of the data range in the specified column
            var dataRange = this.getColumnDataRange(tableCol);

            // nothing to do, if no data range is available (e.g., header/footer rows only)
            if (!dataRange) { return []; }

            // transform the passed relative intervals to absolute row intervals
            var boundInterval = dataRange.rowInterval();
            if (rowIntervals) {
                rowIntervals = rowIntervals.clone(true).move(boundInterval.first).intersect(boundInterval);
            } else {
                rowIntervals = new IntervalArray(boundInterval);
            }

            // nothing to do, if the passed relative intervals are outside the data range
            if (rowIntervals.empty()) { return []; }

            // build the data range addresses to be queried for
            var dataRanges = RangeArray.createFromRowIntervals(rowIntervals, dataRange.start[0]);

            // fetch the cell contents from the cell collection
            return cellCollection.getRangeContents(dataRanges, { display: true, compressed: true });
        };

        /**
         * Returns whether the specified column contains active filter rules.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @returns {Boolean}
         *  Whether the specified column contains active filter rules.
         */
        this.isColumnFiltered = function (tableCol) {
            var columnModel = getColumnModel(tableCol);
            return _.isObject(columnModel) && columnModel.isFiltered();
        };

        /**
         * Returns whether the specified column contains active sort options.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @returns {Boolean}
         *  Whether the specified column contains active sort options.
         */
        this.isColumnSorted = function (tableCol) {
            var columnModel = getColumnModel(tableCol);
            return _.isObject(columnModel) && columnModel.isSorted();
        };

        /**
         * Returns whether this table contains any columns with active settings
         * (filter rules or sort options) that can be used to refresh the table
         * range.
         *
         * @returns {Boolean}
         *  Whether this table contains any columns with active settings.
         */
        this.isRefreshable = function () {
            return this.isFilterActive() && columnModelMap.some(function (entry) {
                return entry.columnModel.isRefreshable();
            });
        };

        /**
         * Creates a matcher predicate function according to the filter
         * attributes of the specified table column.
         *
         * @param {Number} tableCol
         *  The zero-based index of the column, relative to the cell range
         *  covered by this table.
         *
         * @returns {Function|Null}
         *  A predicate function that takes a cell descriptor object with the
         *  properties 'result' (typed cell result value) and 'display' (the
         *  formatted display string), and returns a Boolean whether the passed
         *  cell matches the filter rules of the specified column. This method
         *  will return null instead of a matcher function, if the specified
         *  table column does not contain any supported filter rules.
         */
        this.createFilterMatcher = function (tableCol) {

            // the current filter attributes of the passed column
            var filterAttrs = this.getColumnAttributeSet(tableCol, true).filter;

            // create a matcher function for discrete filter rule
            if (filterAttrs.type === 'discrete') {
                return createDiscreteFilterMatcher(filterAttrs.entries);
            }

            // unsupported filter type, or no existing filter rule
            return null;
        };

        /**
         * Builds and returns a list of row intervals containing all visible
         * rows not filtered by the current filter settings. For performance
         * reasons (to prevent blocked browser UI and its respective error
         * messages), the implementation of this method may run asynchronously.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Array<Number>} [options.skipColumns]
         *      The zero-based indexes of all table columns that will be
         *      excluded from filtering (its filter settings will be ignored).
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved with the
         *  visible data row intervals (instance of the class IntervalArray).
         *  The properties 'first' and 'last' of each row interval are relative
         *  to the data range of the table (the index 0 always refers to the
         *  first data row of this table!).
         */
        this.queryFilterResult = function (options) {

            // start with the entire data range
            var dataRange = this.getDataRange();

            // no data rows available
            if (!dataRange) { return $.when([]); }

            // indexes of all filtered columns
            var filterCols = [];
            // table column to be excluded from filtering
            var skipColsSet = Utils.makeSet(Utils.getArrayOption(options, 'skipColumns', []));
            // the entire row interval of the data range (absolute row indexes)
            var dataRowInterval = dataRange.rowInterval();
            // the remaining visible data intervals for the next column (relative to the data range)
            var visibleIntervals = new IntervalArray(new Interval(0, dataRowInterval.size() - 1));

            // collect the column indexes of all filtered columns
            if (this.isFilterActive()) {
                columnModelMap.forEach(function (entry) {
                    if (entry.columnModel.isFiltered() && !(entry.tableCol in skipColsSet)) {
                        filterCols.push(entry.tableCol);
                    }
                });
            }

            // process the table columns in a background loop to prevent unresponsive browser
            var promise = this.iterateArraySliced(filterCols, function (tableCol) {

                // early exit, if no visible data rows are left for filtering
                if (visibleIntervals.empty()) { return Utils.BREAK; }

                // the filter matcher for the current column
                var matcher = this.createFilterMatcher(tableCol);

                // ignore columns with unknown/unsupported filter settings
                if (!matcher) { return; }

                // query the cell contents of all remaining visible rows
                var cellArray = this.getColumnContents(tableCol, visibleIntervals);
                // the resulting row intervals which remain visible
                var newIntervals = new IntervalArray();
                // index of the current data row
                var dataRow = 0;
                // index of the current row interval, and offset inside the interval
                var intervalIndex = 0;

                // pushes a new row interval to the 'newRowIntervals' array (tries to expand last interval)
                function pushVisibleInterval(cellCount) {

                    // end position of the new interval (start position is 'dataRow')
                    var lastDataRow = dataRow + cellCount - 1;
                    // last pushed interval for expansion
                    var lastInterval = newIntervals.last();

                    // try to expand the last interval
                    if (lastInterval && (lastInterval.last + 1 === dataRow)) {
                        lastInterval.last = lastDataRow;
                    } else {
                        newIntervals.push(new Interval(dataRow, lastDataRow));
                    }
                }

                // simultaneously iterate through the row intervals and compressed result array
                cellArray.forEach(function (cellData) {

                    // the number of remaining cell to process for the current cell data object
                    var remainingCells = cellData.count;
                    // whether the current cell data results in visible rows (the filter matches)
                    var visible = _.isString(cellData.display) && matcher(cellData.value, cellData.display);
                    // the current row interval
                    var visibleInterval = null;
                    // number of cells that can be processed for the current row interval
                    var cellCount = 0;

                    // process as many remaining data rows as covered by the current cell data entry
                    while (remainingCells > 0) {

                        // the current row interval
                        visibleInterval = visibleIntervals[intervalIndex];
                        // update 'dataRow' if a new row interval has been started
                        dataRow = Math.max(dataRow, visibleInterval.first);
                        // number of cells that can be processed for the current row interval
                        cellCount = Math.min(remainingCells, visibleInterval.last - dataRow + 1);

                        // if the cell data matches the filter rule, add the row interval to the result
                        if (visible) { pushVisibleInterval(cellCount); }

                        // decrease number of remaining cells for the current cell data object
                        remainingCells -= cellCount;
                        // update data row index
                        dataRow += cellCount;
                        // if end of current row interval has been reached, go to next row interval
                        if (visibleInterval.last < dataRow) { intervalIndex += 1; }
                    }
                });

                // use the resulting filtered row intervals for the next filtered column
                visibleIntervals = newIntervals;
            });

            // return a promise that resolves with the row intervals
            return promise.then(function () { return visibleIntervals; });
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations, and the undo operations, to refresh this
         * table range according to its current filter settings, by showing or
         * hiding the affected rows in the sheet.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method TableModel.queryFilterResult().
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'rows:overflow': The current filter settings of the table would
         *      modify (show or hide) too many rows at once.
         *  - 'operation': Internal error while applying the operation.
         */
        this.generateRefreshOperations = function (generator, options) {

            // start with the entire data range; do nothing, if no data rows are available
            var dataRange = this.getDataRange();
            if (!dataRange) { return $.when(); }

            // the effective table range, e.g. after expanding auto-filter data range
            var effectiveRange = tableRange.boundary(dataRange);

            // update the effective table range (data range may have been expanded)
            if (tableRange.differs(effectiveRange)) {
                generator.generateTableOperation(Operations.CHANGE_TABLE, tableName, tableRange.toJSON(), { undo: true });
                generator.generateTableOperation(Operations.CHANGE_TABLE, tableName, effectiveRange.toJSON());
            }

            // update the visibility state of all filter rows
            return this.queryFilterResult(options).then(function (rowIntervals) {

                // the entire row interval of the data range
                var dataRowInterval = dataRange.rowInterval();

                // the new visible row intervals (queryFilterResult() returns relative row indexes)
                var visibleIntervals = rowIntervals.clone(true).move(dataRowInterval.first);
                Utils.addProperty(visibleIntervals, 'fillData', { attrs: { visible: true, filtered: false } });

                // the new hidden row intervals
                var hiddenIntervals = new IntervalArray(dataRowInterval).difference(visibleIntervals);
                Utils.addProperty(hiddenIntervals, 'fillData', { attrs: { visible: false, filtered: true } });

                // create the operations to update all affected rows
                var intervals = visibleIntervals.concat(hiddenIntervals);
                return sheetModel.getRowCollection().generateIntervalOperations(generator, intervals);
            });
        };

        /**
         * Generates the operations, and the undo operations, to modify the
         * attributes of this table range.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set with new attributes to be applied at
         *  the table range.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operation.
         */
        this.generateChangeOperations = function (generator, attributeSet) {

            // generate a 'changeTable' operation (this modifies the table attributes immediately)
            generator.generateTableOperation(Operations.CHANGE_TABLE, tableName, { attrs: this.getUndoAttributeSet(attributeSet) }, { undo: true });
            generator.generateTableOperation(Operations.CHANGE_TABLE, tableName, { attrs: attributeSet });

            // refresh the table with current filter settings applied above
            return this.generateRefreshOperations(generator);
        };

        /**
         * Generates the operations, and the undo operations, to change the
         * attributes of the specified table column, and refreshes all affected
         * settings in the sheet, such as filtered rows.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} tableCol
         *  The zero-based index of the table column to be modified, relative
         *  to the cell range covered by the table.
         *
         * @param {Object} attributeSet
         *  The attributes to be set for the specified table column.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'rows:overflow': The passed filter settings would modify (show or
         *      hide) too many rows at once.
         *  - 'operation': Internal error while applying the operation.
         */
        this.generateChangeColumnOperations = function (generator, tableCol, attributeSet) {

            // generate the undo operation, also for a missing table column (back to defaults, this will delete the column - TODO: new 'deleteTableColumn' operation!)
            var columnModel = getColumnModel(tableCol);
            var undoAttrSet = null;
            if (columnModel) {
                undoAttrSet = columnModel.getUndoAttributeSet(attributeSet);
            } else {
                undoAttrSet = _.copy(attributeSet, true);
                _.each(undoAttrSet, function (undoAttrs) {
                    _.each(undoAttrs, function (value, name) { undoAttrs[name] = null; });
                });
            }
            generator.generateTableOperation(Operations.CHANGE_TABLE_COLUMN, tableName, { col: tableCol, attrs: undoAttrSet }, { undo: true });

            // generate a 'changeTableColumn' operation (this modifies the column attributes immediately)
            generator.generateTableOperation(Operations.CHANGE_TABLE_COLUMN, tableName, { col: tableCol, attrs: attributeSet });

            // always refresh the filter (also if no attributes have been changed, the cell contents may
            // have changed in the meantime, resulting in new filtered rows)
            return this.generateRefreshOperations(generator);
        };

        /**
         * Generates the operations, and the undo operations, to delete this
         * table range from the document. Additionally, if the table contains
         * active filter rules, all operations will be generated that make the
         * affected hidden rows visible again.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while applying the operation.
         */
        this.generateDeleteOperations = function (generator) {

            // make all rows visible that are filtered by this table range
            var promise = sheetModel.getRowCollection().generateFillOperations(generator, tableRange.rowInterval(), { attrs: { visible: true, filtered: false } });

            // ignore 'rows:overflow' error (leave rows hidden, but delete the table anyways)
            promise = promise.then(null, function (result) {
                return (Utils.getStringOption(result, 'cause') === 'rows:overflow') ? $.when() : result;
            });

            // create the 'deleteTable' operation, and the undo operations
            return promise.then(function () {

                // restore the entire table with its position and attributes
                var undoProperties = tableRange.toJSON();
                undoProperties.attrs = self.getExplicitAttributeSet(true);
                generator.generateTableOperation(Operations.INSERT_TABLE, tableName, undoProperties, { undo: true });

                // restore all existing columns
                columnModelMap.forEach(function (entry) {
                    var colProperties = { col: entry.tableCol, attrs: entry.columnModel.getExplicitAttributeSet(true) };
                    generator.generateTableOperation(Operations.CHANGE_TABLE_COLUMN, tableName, colProperties, { undo: true });
                });

                // generate the operation to delete the table
                generator.generateTableOperation(Operations.DELETE_TABLE, tableName);
            });
        };

        /**
         * Generates the operations and undo operations to update and restore
         * this table model after cells have been moved in the sheet.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {MoveDescriptor} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  target range of the table model.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMoveCellsOperations = function (generator, moveDescs) {

            // find the move descriptor that manipulates this table (only one move descriptor can cover the table range)
            var moveDesc = _.find(moveDescs, function (moveDesc) { return moveDesc.coversRange(tableRange); });
            if (!moveDesc) { return $.when(); }

            // whether the move descriptor deletes rows
            var deleteRows = !moveDesc.insert && !moveDesc.columns;
            // bug 36226: delete tables whose header row has been deleted
            // bug 39869: deleting header rows prevented by UI except for auto-filter
            var deleteTable = deleteRows && this.hasHeaderRow() && moveDesc.deleteIntervals.containsIndex(tableRange.start[1]);

            // transform the table range, generate operation to delete the entire table on demand
            var transformRange = deleteTable ? null : moveDesc.transformRange(tableRange);
            if (!transformRange) { return this.generateDeleteOperations(generator); }

            // the properties for the change operation
            var properties = {};
            var undoProperties = {};

            // detect whether the table range needs to be restored on undo
            var restoredRange = moveDesc.transformRange(transformRange, { reverse: true });
            if (!restoredRange || !restoredRange.equals(tableRange)) {
                _.extend(undoProperties, tableRange.toJSON());
            }

            // update attribute 'footerRow' when deleting rows
            if (deleteRows && this.hasFooterRow() && moveDesc.deleteIntervals.containsIndex(tableRange.end[1])) {
                properties.attrs = { table: { footerRow: false } };
                undoProperties.attrs = { table: { footerRow: true } };
            }

            // generate the change operations
            if (!_.isEmpty(properties)) { generator.generateTableOperation(Operations.CHANGE_TABLE, tableName, properties); }
            if (!_.isEmpty(undoProperties)) { generator.generateTableOperation(Operations.CHANGE_TABLE, tableName, undoProperties, { undo: true }); }

            // collect the column models that will be deleted
            var deletedColumnMap = new SimpleMap();
            if (moveDesc.columns && !columnModelMap.empty()) {

                // transform the absolute column index of all columns according to the operation
                var startCol = tableRange.start[0];
                var startRow = tableRange.start[1];
                columnModelMap.forEach(function (entry) {
                    var address = new Address(startCol + entry.tableCol, startRow);
                    if (!moveDesc.transformAddress(address)) {
                        deletedColumnMap.insert(entry.tableCol, entry);
                    }
                });
            }
            // nothing more to do, if no table columns will be deleted
            if (deletedColumnMap.empty()) { return $.when(); }

            // generate the undo operations to restore the deleted table columns
            var promise = this.iterateSliced(deletedColumnMap.iterator(), function (entry) {
                var colProperties = { col: entry.tableCol, attrs: entry.columnModel.getExplicitAttributeSet(true) };
                generator.generateTableOperation(Operations.CHANGE_TABLE_COLUMN, tableName, colProperties, { undo: true });
            }, { delay: 'immediate' });

            // refresh the visible/hidden rows (deleted filter settings will cause to show hidden rows)
            return promise.then(function () {
                return self.generateRefreshOperations(generator, { skipColumns: deletedColumnMap.pluck('tableCol') });
            });
        };

        /**
         * Creates column header labels for all blank header cells, and all
         * duplicted header cells in this table range.
         *
         * @returns {AddressArray}
         *  The addresses of all header cells without a label, or with an
         *  invalid label. Each address will contain the additional property
         *  'label' with the generated label for that header cell.
         */
        this.createMissingHeaderLabels = function () {

            // the addresses of all header cells without a label
            var addresses = new AddressArray();
            // the address of the header range of the table
            var headerRange = this.getHeaderRange();

            // nothing to do for auto-filters, or tables without header row
            if (this.isAutoFilter() || !headerRange) { return addresses; }

            // collect all existing unique header labels of the table
            var labelArray = [], labelSet = {};
            var iterator = cellCollection.createAddressIterator(headerRange, { covered: true });
            IteratorUtils.forEach(iterator, function (address) {
                var label = cellCollection.getCellValue(address);
                var key = (typeof label === 'string') ? SheetUtils.getTableKey(label) : null;
                if (key && !(key in labelSet)) {
                    labelArray.push(label);
                    labelSet[key] = true;
                } else {
                    labelArray.push(null);
                }
            });

            // generate missing header labels (all elements in 'addresses')
            var prevLabel = SheetUtils.getTableColName();
            labelArray.forEach(function (label, index) {

                // a non-empty string in 'labelArray' denotes a valid header label
                if (label) {
                    prevLabel = label;
                    return;
                }

                // Try to split the header label of the preceding column into a base name and a
                // trailing integer (fall-back to translated 'Column' as base name), and generate
                // an unused column name by increasing the training index repeatedly.
                var matches = /^(.*?)(\d+)$/.exec(prevLabel);
                var baseLabel = matches ? matches[1] : SheetUtils.getTableColName();
                var labelIndex = matches ? (parseInt(matches[2], 10) + 1) : 1;
                while (SheetUtils.getTableKey(baseLabel + labelIndex) in labelSet) { labelIndex += 1; }

                // Store the new label in the result array, in the set used to test uniqueness,
                // and in 'prevLabel' for the next iteration cycle.
                var address = new Address(tableRange.start[0] + index, tableRange.start[1]);
                address.label = prevLabel = baseLabel + labelIndex;
                addresses.push(address);
                labelSet[SheetUtils.getTableKey(prevLabel)] = true;
            });

            return addresses;
        };

        // initialization -----------------------------------------------------

        // store a clone of the range to prevent external modifications
        tableRange = tableRange.clone();

        // register all table models at the document model (global map for fast look-up)
        docModel._registerTableModel(this);

        // update column models after changing table attributes
        this.on('change:attributes', changeAttributesHandler);

        // clone column models passed as hidden argument by the clone() method
        if (arguments[TableModel.length] instanceof SimpleMap) {
            arguments[TableModel.length].forEach(function (entry) {
                insertColumnModel(entry.tableCol, entry.columnModel.clone(sheetModel, this));
            }, this);
        }

        // destroy all class members
        this.registerDestructor(function () {
            docModel._unregisterTableModel(self);
            columnModelMap.forEach(function (entry) { entry.columnModel.destroy(); });
            defColumnModel.destroy();
            headerCache.destroy();
            self = docModel = sheetModel = null;
            columnModelMap = defColumnModel = headerCache = null;
        });

    } }); // class TableModel

    // class TableCollection ==================================================

    /**
     * Stores settings for table ranges in a specific sheet. Table ranges
     * contain filter settings, sorting settings, and specific cell formatting.
     *
     * Triggers the following events:
     * - 'insert:table'
     *      After a new table range has been inserted into this collection.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {TableModel} tableModel
     *          The model instance of the new table range.
     * - 'change:table'
     *      After an existing table range has been changed. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {TableModel} tableModel
     *          The model instance of the changed table range.
     *      (3) {String} type
     *          The type of the original event forwarded from the table model.
     *          Can be 'change:attributes', 'change:range', or 'change:column'.
     *      (4) {Any} [...]
     *          Additionally parameters of the original table event.
     * - 'delete:table'
     *      Before a table range will be deleted from this collection. Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {TableModel} tableModel
     *          The model instance of the table range to be deleted.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function TableCollection(sheetModel) {

        // self reference
        var self = this;

        // the document model instance
        var docModel = sheetModel.getDocModel();

        // all table models, mapped by upper-case table name
        var tableModelMap = new SimpleMap();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns a descriptor for an existing table model addressed by the
         * passed operation context.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for tables.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.insert=false]
         *      If set to true, the table model addressed by the operation must
         *      not exist in this table collection. By default, the table model
         *      must exist in the collection.
         *
         * @returns {Object}
         *  A descriptor for the table model addressed by the passed operation,
         *  with the following properties:
         *  - {String} name
         *      The original table name as contained in the passed operation.
         *  - {String} key
         *      The (upper-case) name used as map key for the table model.
         *  - {TableModel|Null} model
         *      The table model addressed by the operation; or null, if the
         *      table does not exist (see option 'insert' above).
         *
         * @throws {OperationException}
         *  If the operation does not address an existing table model, or if it
         *  addresses an existing table model that must not exist (see option
         *  'insert' above).
         */
        function getModelData(context, options) {

            // get the table name from the operation (missing/empty name addresses auto-filter)
            var tableName = context.getOptStr('table', true);
            var autoFilter = tableName.length === 0;

            // the upper-case name, used as map key
            var tableKey = SheetUtils.getTableKey(tableName);

            // check existence/absence of the table model
            var tableModel = tableModelMap.get(tableKey, null);
            if (Utils.getBooleanOption(options, 'insert', false)) {
                context.ensure(!tableModel && (autoFilter || !docModel.hasTable(tableName)), 'table exists');
            } else {
                context.ensure(tableModel, 'missing table');
            }

            return { name: tableName, key: tableKey, model: tableModel };
        }

        /**
         * Returns whether the passed table name exists already, either as
         * defined name, or as name of a table range anywhere in the document.
         *
         * @param {String} tableName
         *  The name for a table range to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed table name exists already.
         */
        function isTableNameUsed(tableName) {
            return docModel.hasTable(tableName);
        }

        /**
         * Returns whether the passed cell range address covers a table model
         * in this collection.
         *
         * @param {Range} range
         *  The cell range address to be checked.
         *
         * @param {TableModel} [excludeModel]
         *  A table model whose cell range will be ignored.
         *
         * @returns {Boolean}
         *  Whether the passed cell range address covers a table model (except
         *  the table model specified in the parameter 'excludeModel') in this
         *  collection.
         */
        function rangeCoversAnyTable(range, excludeModel) {
            return tableModelMap.some(function (tableModel) {
                return (tableModel !== excludeModel) && tableModel.overlapsRange(range);
            });
        }

        /**
         * Inserts the passed table model into this collection, triggers an
         * 'insert:table' event, and registers a listener for all events of the
         * table model that will be forwarded to the listeners of this
         * collection.
         */
        function insertTableModel(tableModel) {

            // insert the passed table model, notify listeners
            tableModelMap.insert(tableModel.getKey(), tableModel);
            self.trigger('insert:table', tableModel);

            // forward all change events of the table model
            self.listenTo(tableModel, 'triggered', function (event, type) {
                self.trigger.apply(self, ['change:table', tableModel, type].concat(_.toArray(arguments).slice(2)));
            });
        }

        /**
         * Deletes an existing table model from this collection, and triggers a
         * 'delete:table' event.
         */
        function deleteTableModel(tableModel, tableKey) {
            self.trigger('delete:table', tableModel);
            tableModelMap.remove(tableKey);
            tableModel.destroy();
        }

        /**
         * Recalculates the target ranges of all validation models, after cells
         * have been moved (including inserted/deleted columns or rows) in the
         * sheet.
         */
        function moveCellsHandler(event, moveDesc) {
            tableModelMap.forEach(function (tableModel, tableKey) {
                if (!tableModel.transformRange(moveDesc)) {
                    deleteTableModel(tableModel, tableKey);
                }
            });
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { tableModelMap: tableModelMap };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * table ranges from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {TableCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the new names for cloned table ranges (table names must be unique in the entire document)
            var tableNames = context.getOptObj('tableNames');
            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            // clone the contents of the source collection
            cloneData.tableModelMap.forEach(function (tableModel) {
                var oldTableName = tableModel.getName();
                var newTableName = oldTableName;
                if (!tableModel.isAutoFilter()) {
                    newTableName = Utils.getStringOption(tableNames, oldTableName, null);
                    context.ensure(newTableName, 'missing replacement name for table "%s"', oldTableName);
                }
                insertTableModel(tableModel.clone(sheetModel, newTableName));
            });
        };

        /**
         * Callback handler for the document operation 'insertTable'. Creates
         * and inserts a new table into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertTable' document operation.
         */
        this.applyInsertOperation = function (context) {

            // get table name and map key from operation (table must not exist, throws on error)
            var modelData = getModelData(context, { insert: true });

            // check that the cell range does not cover another table
            var tableRange = context.getRange();
            context.ensure(!rangeCoversAnyTable(tableRange), 'range covers another table');

            // create and insert the new table model, notify listeners
            modelData.model = new TableModel(sheetModel, modelData.name, tableRange, context.getOptObj('attrs'));
            insertTableModel(modelData.model);
        };

        /**
         * Callback handler for the document operation 'deleteTable'. Deletes
         * an existing table from this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteTable' document operation.
         */
        this.applyDeleteOperation = function (context) {

            // resolve the table model addressed by the operation (throws on error)
            var modelData = getModelData(context);

            // delete the table model and notify listeners
            deleteTableModel(modelData.model, modelData.key);
        };

        /**
         * Callback handler for the document operation 'changeTable'. Changes
         * the position and/or attributes of an existing table.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeTable' document operation.
         */
        this.applyChangeOperation = function (context) {

            // resolve the table model addressed by the operation (throws on error)
            var tableModel = getModelData(context).model;

            // change table range
            var tableRange = context.getOptRange();
            if (tableRange) {
                // check that the cell range does not cover another table
                context.ensure(!rangeCoversAnyTable(tableRange, tableModel), 'range covers another table');
                // set the new ranges at the table model
                tableModel.setRange(tableRange);
            }

            // change all other table attributes (change events will be forwarded)
            if (context.has('attrs')) {
                tableModel.setAttributes(context.getObj('attrs'));
            }
        };

        /**
         * Callback handler for the document operation 'changeTableColumn'.
         * Changes the attributes of a column in an existing table.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeTableColumn' document operation.
         */
        this.applyChangeColumnOperation = function (context) {
            getModelData(context).model.applyChangeOperation(context);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether the specified table exists in this collection.
         *
         * @param {String} tableName
         *  The name of the table to check for. The empty string addresses the
         *  anonymous table range used to store filter settings for the
         *  standard auto-filter of the sheet.
         *
         * @returns {Boolean}
         *  Whether the specified table exists in this collection.
         */
        this.hasTable = function (tableName) {
            return tableModelMap.has(SheetUtils.getTableKey(tableName));
        };

        /**
         * Returns whether this collection contains an auto-filter range.
         *
         * @returns {Boolean}
         *  Whether this collection contains an auto-filter range.
         */
        this.hasAutoFilter = function () {
            return tableModelMap.has('');
        };

        /**
         * Returns the model of the table with the specified name.
         *
         * @param {String} tableName
         *  The name of the table. The empty string addresses the anonymous
         *  table range used to store filter settings for the standard auto
         *  filter of the sheet.
         *
         * @returns {TableModel|Null}
         *  The model of the table with the specified name; or null, if no
         *  table exists with that name.
         */
        this.getTable = function (tableName) {
            return tableModelMap.get(SheetUtils.getTableKey(tableName), null);
        };

        /**
         * Returns all table models in this collection as plain array.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.autoFilter=false]
         *      If set to true, the table model for the auto-filter will be
         *      included into the result.
         *
         * @returns {Array<TableModel>}
         *  All table models in this collection as plain array.
         */
        this.getAllTables = function (options) {

            // return all models including the auto-filter if specified
            if (Utils.getBooleanOption(options, 'autoFilter', false)) {
                return tableModelMap.values();
            }

            // skip the model of the auto-filter
            return tableModelMap.reject(function (tableModel) { return tableModel.isAutoFilter(); });
        };

        /**
         * Returns the model of the table covering the specified cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {TableModel|Null}
         *  The model of the table at the specified position; or null, if the
         *  cell is not part of a table.
         */
        this.findTable = function (address) {
            return tableModelMap.find(function (tableModel) {
                return tableModel.containsCell(address);
            }) || null;
        };

        /**
         * Returns the models of all tables overlapping with the specified cell
         * range.
         *
         * @param {Range} range
         *  The address of a cell range.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.active=false]
         *      If set to true, the result returned by this method will be
         *      restricted to tables with activated filters/sorting (see method
         *      TableModel.isFilterActive() for details).
         *
         * @returns {Array<TableModel>}
         *  The models of all tables overlapping with the specified cell range.
         */
        this.findTables = function (range, options) {

            // whether to only find tables with active filter/sorting
            var active = Utils.getBooleanOption(options, 'active', false);

            return tableModelMap.filter(function (tableModel) {
                return tableModel.overlapsRange(range) && (!active || tableModel.isFilterActive());
            });
        };

        /**
         * Returns whether the selected rows can be deleted from the sheet,
         * according to the position of the tables in this collection. The
         * header rows of all tables but the auto-filter cannot be deleted, and
         * at least one data row must remain in each table.
         *
         * @param {IntervalArray|Interval} intervals
         *  The row intervals intended to be deleted from the sheet.
         *
         * @returns {Boolean}
         *  Whether the selected rows can be deleted from the sheet.
         */
        this.canDeleteRows = function (intervals) {
            return tableModelMap.every(function (tableModel) {
                return tableModel.canDeleteRows(intervals);
            });
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations, and the undo operations, to insert a new
         * table range into this collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} tableName
         *  The name for the new table to be inserted. The empty string refers
         *  to the anonymous table range used to store filter settings for the
         *  standard auto-filter of the sheet.
         *
         * @param {Range} range
         *  The address of the cell range covered by the new table.
         *
         * @param {Object} [attributes]
         *  The initial attribute set for the table.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'table:duplicate': The sheet already contains a table with the
         *      passed name.
         *  - 'table:overlap': The passed cell range covers an existing table.
         *  - 'operation': Internal error while applying the operation.
         */
        this.generateInsertTableOperations = function (generator, tableName, range, attributes) {

            // check existence of the table range (must be unique in entire document except for auto-filter)
            var autoFilter = tableName.length === 0;
            if (autoFilter ? this.hasAutoFilter() : isTableNameUsed(tableName)) { return SheetUtils.makeRejected('table:duplicate'); }

            // check that the range does not overlap with an existing table
            if (this.findTables(range).length > 0) { return SheetUtils.makeRejected('table:overlap'); }

            // 'real' tables must not contain merged cells (neither in header, footer, nor data rows)
            var promise = autoFilter ? this.createResolvedPromise() : sheetModel.getCellCollection().generateMergeCellsOperations(generator, range, 'unmerge');

            // create all operations and undo operations for the new table
            return promise.then(function () {

                // create the 'insertTable' operation, and the appropriate undo operation
                var properties = range.toJSON();
                if (_.isObject(attributes)) { properties.attrs = attributes; }
                generator.generateTableOperation(Operations.DELETE_TABLE, tableName, null, { undo: true });
                generator.generateTableOperation(Operations.INSERT_TABLE, tableName, properties);

                // Bug 36152: Auto-filter will hide drop-down buttons in header cells covered by merged
                // ranges, but only if the cells were already merged before creating the auto-filter.
                if (autoFilter) {
                    var mergedRanges = sheetModel.getMergeCollection().getMergedRanges(range.headerRow());
                    return self.iterateArraySliced(mergedRanges, function (mergedRange) {
                        var firstCol = Math.max(mergedRange.start[0], range.start[0]);
                        var lastCol = Math.min(mergedRange.end[0], range.end[0]);
                        // the drop-down button in the LAST column of a merged range remains visible
                        for (var col = firstCol; col < lastCol; col += 1) {
                            // no undo operations necessary (the table will be deleted completely on undo)
                            generator.generateTableOperation(Operations.CHANGE_TABLE_COLUMN, tableName, { col: col - range.start[0], attrs: { filter: { hideMergedButton: true } } });
                        }
                    }, { delay: 'immediate' });
                }
            });
        };

        /**
         * Generates the operations and undo operations to update and restore
         * the table models in this collection after cells have been moved in
         * the sheet.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {MoveDescriptor} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  target ranges of the table models in this collection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMoveCellsOperations = function (generator, moveDescs) {
            return this.iterateSliced(tableModelMap.iterator(), function (tableModel) {
                tableModel.generateMoveCellsOperations(generator, moveDescs);
            }, { delay: 'immediate' });
        };

        /**
         * Generates cell operations with calculated column header labels for
         * all blank header cells in the table ranges, after columns have been
         * inserted in the sheet.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMissingHeaderLabelOperations = function (generator) {

            // collect all missing header labels
            var addresses = new AddressArray();
            var promise = this.iterateSliced(tableModelMap.iterator(), function (tableModel) {
                addresses.append(tableModel.createMissingHeaderLabels());
            }, { delay: 'immediate' });

            // create a cell operation for all header label cells
            return promise.then(function () {
                if (addresses.empty()) { return; }
                var builder = new CellOperationsBuilder(sheetModel, generator);
                addresses.sort().forEach(function (address) {
                    builder.createCells(address, { v: address.label }, 1);
                });
                return builder.finalizeOperations({ createUndo: false });
            });
        };

        // initialization -----------------------------------------------------

        // additional processing and event handling after the document has been imported
        this.waitForImportSuccess(function () {
            // update table ranges after moving cells, or inserting/deleting columns or rows
            this.listenTo(sheetModel, 'move:cells', moveCellsHandler);
        }, this);

        // destroy all class members
        this.registerDestructor(function () {
            tableModelMap.forEach(function (tableModel) { tableModel.destroy(); });
            self = docModel = sheetModel = tableModelMap = null;
        });

    } // class TableCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: TableCollection });

});
