/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/namecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, SimpleMap, TimerMixin, ModelObject, Operations, SheetUtils, TokenArray) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;

    // class NameModel ========================================================

    /**
     * Stores settings for a single defined name in a specific sheet, or for a
     * global defined name in the document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel|SheetModel} parentModel
     *  The spreadsheet document model (for globally defined names), or the
     *  sheet model (for sheet-local names) containing this instance.
     *
     * @param {String} label
     *  The exact label of this defined name, with correct character case as
     *  used in formula expressions.
     *
     * @param {String} initFormula
     *  The formula expression bound to the defined name.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Boolean} [initOptions.hidden=false]
     *      Whether the defined name should be hidden in the user interface.
     *  @param {String} [initOptions.ref]
     *      The address of the reference cell used to interpret relative cell
     *      references in the formula expression. The reference cell must be a
     *      formula expression by itself that results in a single cell address
     *      with sheet name.
     */
    var NameModel = ModelObject.extend({ constructor: function (parentModel, label, initFormula, initOptions) {

        // the document model
        var docModel = parentModel.getDocModel();

        // the sheet model of sheet-local names
        var sheetModel = (parentModel === docModel) ? null : parentModel;

        // the token array representing the definition of this name
        var defTokenArray = new TokenArray(parentModel, 'name');

        // the token array representing the reference address for ODF
        var refTokenArray = new TokenArray(parentModel, 'name');

        // whether the defined name is hidden in the GUI
        var hidden = Utils.getBooleanOption(initOptions, 'hidden', false);

        // whether the current file format is OOXML
        var ooxml = docModel.getApp().isOOXML();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Changes the formula definition of this name model.
         *
         * @param {String} defFormula
         *  The new formula expression bound to the name model.
         *
         * @param {String} [refFormula]
         *  The address of the reference cell used to interpret relative cell
         *  references in the formula expression. The reference cell must be a
         *  formula expression by itself that results in a single cell address
         *  with sheet name.
         */
        function setFormula(defFormula, refFormula) {

            // parse the formula expression
            defTokenArray.parseFormula('op', defFormula);

            // relocate formula according to reference address (used by ODF only, OOXML always uses A1)
            if (refFormula) {

                // parse the reference address expression passed to the constructor
                refTokenArray.parseFormula('op', refFormula);

                // try to convert to a single reference address (do not pass a reference
                // sheet index, sheet name is expected to exist in the formula expression)
                var ranges = refTokenArray.resolveRangeList();

                // if the expression resolves to a single cell address, use it to relocate the formula
                if ((ranges.length === 1) && ranges.first().single()) {
                    defTokenArray.relocateRanges(ranges.first().start, Address.A1);
                }
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this name for the specified
         * sheet.
         *
         * @internal
         *  Used by the class NameCollection during clone construction. DO NOT
         *  CALL from external code!
         *
         * @param {SpreadsheetModel|SheetModel} targetModel
         *  The model instance that will own the clone returned by this method.
         *
         * @returns {NameModel}
         *  A clone of this name model, initialized for ownership by the passed
         *  sheet model.
         */
        this.clone = function (targetModel) {
            var options = { hidden: hidden };
            if (ooxml) { options.ref = refTokenArray.getFormula('op'); }
            return new NameModel(targetModel, label, defTokenArray.getFormula('op'), options);
        };

        /**
         * Callback handler for the document operation 'changeName'. Changes
         * the label and/or the formula definition of this defined name.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeName' document operation.
         *
         * @returns {Object}
         *  A flag set specifying what has changed in this defined name.
         *  - {Boolean} label
         *      Whether the label of this defined name has been changed.
         *  - {Boolean} formula
         *      Whether the formula expression has been changed.
         */
        this.applyChangeOperation = function (context) {

            // the changed flags
            var changeFlags = { label: false, formula: false };

            // set a new name for the model if specified
            if (context.has('newLabel')) {

                // get new label (will be checked to be non-empty)
                var newLabel = context.getStr('newLabel');
                if (label !== newLabel) {
                    label = newLabel;
                    changeFlags.label = true;
                }
            }

            // set a new formula expression for the model if specified
            if (context.has('formula')) {
                setFormula(context.getStr('formula', true), context.getOptStr('ref'));
                changeFlags.formula = true;
            }

            return changeFlags;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the parent sheet model of a sheet-locally defined name.
         *
         * @returns {SheetModel|Null}
         *  The parent sheet model of a sheet-locally defined name; or null for
         *  a globally defined name.
         */
        this.getSheetModel = function () {
            return sheetModel;
        };

        /**
         * Returns the exact label of this name model, with correct character
         * case as used in formula expressions.
         *
         * @returns {String}
         *  The exact label of this name model.
         */
        this.getLabel = function () {
            return label;
        };

        /**
         * Returns the unique map key of this name model (the uppercase label
         * of the defined name).
         *
         * @returns {String}
         *  The unique map key of this name model.
         */
        this.getKey = function () {
            return SheetUtils.getTableKey(label);
        };

        /**
         * Returns the token array representing the formula definition of this
         * name model.
         *
         * @returns {TokenArray}
         *  The token array containing the formula definition.
         */
        this.getTokenArray = function () {
            return defTokenArray;
        };

        /**
         * Returns the formula expression of this name model, relative to the
         * passed target address.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar to be used to generate the
         *  formula expression. See class FormulaGrammar for details.
         *
         * @param {Address} [targetAddress]
         *  The target address to relocate the formula expression to. If
         *  omitted, the current non-relocated formula expression according to
         *  the internal reference address will be returned.
         *
         * @returns {String}
         *  The formula expression of this name model.
         */
        this.getFormula = function (grammarId, targetAddress) {
            return defTokenArray.getFormula(grammarId, targetAddress ? { targetAddress: targetAddress } : null);
        };

        /**
         * Returns the formula expression of the reference address of this name
         * model. Used for the ODF format only.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar to be used to generate the
         *  formula expression. See class FormulaGrammar for details.
         *
         * @returns {String}
         *  The formula expression of the reference address of this name model.
         */
        this.getRefFormula = function (grammarId) {
            return refTokenArray.getFormula(grammarId);
        };

        /**
         * Returns whether the defined name should be hidden in the user
         * interface.
         *
         * @returns {Boolean}
         *  Whether the defined name should be hidden in the user interface.
         */
        this.isHidden = function () {
            return hidden;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expression of this defined name.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {NameModel}
         *  A reference to this instance.
         */
        this.generateFormulaOperations = function (generator, changeDesc) {

            // the properties for the change operation, and the undo operation
            var properties = {};
            var undoProperties = {};

            // resolve the formula expression for the definition of this name
            var defResult = defTokenArray.resolveOperation('op', changeDesc);
            if (defResult) {
                properties.formula = defResult.new;
                undoProperties.formula = defResult.old;
            }

            // ODF only: update the formula expression for the reference address for renamed sheets
            // TODO: move reference address to another existing sheet when deleting a sheet
            if (!ooxml && (changeDesc.type === 'renameSheet')) {
                var refResult = refTokenArray.resolveOperation('op', changeDesc);
                if (refResult) {
                    properties.ref = refResult.new;
                    undoProperties.ref = refResult.old;
                }
            }

            // generate the operation to change this name model
            if (!_.isEmpty(properties)) {
                properties.label = undoProperties.label = label;
                if (sheetModel) {
                    generator.generateSheetOperation(Operations.CHANGE_NAME, properties);
                    generator.generateSheetOperation(Operations.CHANGE_NAME, undoProperties, { undo: true });
                } else {
                    generator.generateOperation(Operations.CHANGE_NAME, properties);
                    generator.generateOperation(Operations.CHANGE_NAME, undoProperties, { undo: true });
                }
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // parse the initial formula definition
        setFormula(initFormula, Utils.getStringOption(initOptions, 'ref'));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            parentModel = docModel = sheetModel = defTokenArray = refTokenArray = null;
        });

    } }); // class NameModel

    // class NameCollection ===================================================

    /**
     * Collects all global defined names of the spreadsheet document, or the
     * defined names contained in a single sheet of the document.
     *
     * Triggers the following events:
     * - 'insert:name'
     *      After a new defined name has been inserted into the collection.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {NameModel} nameModel
     *          The model instance of the new defined name.
     * - 'change:name'
     *      After the label, or the formula definition of a defined name has
     *      been modified. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {NameModel} nameModel
     *          The model instance of the modified defined name.
     *      (3) {Object} changeFlags
     *          A flag set with information about the changed settings:
     *          - {Boolean} changeFlags.label
     *              Whether the label of the defined name has been changed.
     *          - {Boolean} changeFlags.formula
     *              Whether the formula expression of the defined name has been
     *              changed.
     * - 'delete:name'
     *      Before a defined name will be deleted from the collection. Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {NameModel} nameModel
     *          The model instance of the defined name about to be deleted.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {SpreadsheetModel|SheetModel} parentModel
     *  The spreadsheet document model (for globally defined names), or the
     *  sheet model (for sheet-local names) containing this instance.
     */
    function NameCollection(parentModel) {

        // self reference
        var self = this;

        // the document model
        var docModel = parentModel.getDocModel();

        // the sheet model, if this collection is part of a worksheet
        var sheetModel = (parentModel === docModel) ? null : parentModel;

        // the models of all defined names, mapped by upper-case name
        var nameModelMap = new SimpleMap();

        // whether the current file format is OOXML
        var ooxml = docModel.getApp().isOOXML();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, parentModel.getDocModel());
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns a descriptor for an existing name model addressed by the
         * passed operation context.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing a document operation for defined names.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.insert=false]
         *      If set to true, the name model addressed by the operation must
         *      not exist in this name collection. By default, the name model
         *      must exist in the collection.
         *
         * @returns {Object}
         *  A descriptor for the name model addressed by the passed operation,
         *  with the following properties:
         *  - {String} label
         *      The original label as contained in the passed operation.
         *  - {String} key
         *      The (upper-case) label used as map key for the name model.
         *  - {NameModel|Null} model
         *      The name model addressed by the operation; or null, if the name
         *      does not exist (see option 'insert' above).
         *
         * @throws {OperationException}
         *  If the operation does not address an existing name model, or if it
         *  addresses an existing name model that must not exist (see option
         *  'insert' above).
         */
        function getModelData(context, options) {

            // get the label from the operation
            var label = context.getStr('label');
            // the upper-case label is used as map key
            var nameKey = SheetUtils.getNameKey(label);
            // get the name model
            var nameModel = nameModelMap.get(nameKey, null);

            // check existence/absence of the name model
            if (Utils.getBooleanOption(options, 'insert', false)) {
                context.ensure(!nameModel, 'defined name exists');
            } else {
                context.ensure(nameModel, 'missing defined name');
            }

            return { label: label, key: nameKey, model: nameModel };
        }

        /**
         * Validates the passed label for a defined name. Returns a rejected
         * promise if the label is not valid.
         *
         * @param {String} label
         *  The label for a defined name to be checked.
         *
         * @returns {jQuery.Promise|Null}
         *  A rejected promise with a specific error code, if the label is not
         *  valid (see method FormulaGrammar.validateNameLabel() for a list of
         *  error codes); or the value null, if the passed label is valid.
         */
        function validateNameLabel(label) {
            var labelError = docModel.getFormulaGrammar('ui').validateNameLabel(docModel, label);
            return labelError ? SheetUtils.makeRejected(labelError) : null;
        }

        /**
         * Returns whether the passed label exists already, either as defined
         * name in this collection, or as name of a table range anywhere in the
         * document.
         *
         * @param {String} label
         *  The label for a defined name to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed label exists already.
         */
        function isNameLabelUsed(label) {
            return self.hasName(label) || docModel.hasTable(label);
        }

        /**
         * Internal helper function for the ODF file format. Returns whether
         * the passed formula array contains a complex expression, or a simple
         * cell reference.
         *
         * @param {TokenArray} tokenArray
         *  A token array with a formula.
         *
         * @returns {Boolean}
         *  Whether the passed token array represents a complex formula
         *  expression (true), or a simple cell reference (false).
         */
        function isComplexExpression(tokenArray) {
            var ranges = tokenArray.resolveRangeList();
            return !ranges || (ranges.length !== 1);
        }

        /**
         * Creates the operation properties for the passed formula expression,
         * intended to be used in the document operations 'insertName', and
         * 'changeName'. The formula expression will be checked for syntax and
         * semantic errors, and it will be relocated for usage in document
         * operations.
         *
         * @param {Object} formulaDesc
         *  The descriptor of the formula expression. See public method
         *  NameCollection.generateInsertNameOperations() for details about the
         *  properties.
         *
         * @returns {Object|Null}
         *  On success, an object with the property 'formula' containing the
         *  relocated formula expression, and an optional property 'ref' with
         *  the passed reference address (as formula expression) for ODF
         *  documents.
         */
        function getFormulaProperties(formulaDesc) {

            // the resulting operation properties
            var properties = null;

            // parse the formula; bug 40293: add sheet name to references without sheet (OOXML only) (TODO: sheet-local names)
            var tokenArray = new TokenArray(parentModel, 'name');
            tokenArray.parseFormula(formulaDesc.grammarId, formulaDesc.formula, { autoCorrect: true, refSheet: formulaDesc.refSheet, extendSheet: ooxml });

            // interpret the formula to find syntax errors or semantic errors
            var formulaResult = tokenArray.interpretFormula('any');
            if (formulaResult.type !== 'error') {

                // OOXML: relocate the formula expression from reference cell to A1
                // ODF: leave formula expression unmodified, pass reference cell with operation
                if (ooxml) {
                    properties = { formula: tokenArray.getFormula('op', { refAddress: formulaDesc.refAddress, targetAdddress: Address.A1 }) };
                } else {
                    properties = { formula: tokenArray.getFormula('op'), isExpr: isComplexExpression(tokenArray) };
                    // store reference cell as formula expression with sheet name
                    tokenArray.clearTokens();
                    tokenArray.appendRange(new Range(formulaDesc.refAddress), { sheet: formulaDesc.refSheet, abs: true });
                    properties.ref = tokenArray.getFormula('op');
                }
            }

            return properties;
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { nameModelMap: nameModelMap };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * defined names from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {NameCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            // clone the name models of the source collection
            nameModelMap = cloneData.nameModelMap.clone(function (nameModel) {
                return nameModel.clone(parentModel);
            });
        };

        /**
         * Callback handler for the document operation 'insertName'. Creates
         * and stores a new defined name, and triggers an 'insert:name' event.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertName' document operation.
         */
        this.applyInsertOperation = function (context) {

            // get name and map key from operation (name must not exist, throws on error)
            var modelData = getModelData(context, { insert: true });

            // create a new name model (formula expression may be empty)
            var nameModel = new NameModel(parentModel, modelData.label, context.getStr('formula', true), {
                hidden: context.getOptBool('hidden'),
                ref: ooxml ? null : context.getStr('ref')
            });

            // insert the model into the map, notify listeners
            nameModelMap.insert(modelData.key, nameModel);
            this.trigger('insert:name', nameModel);
        };

        /**
         * Callback handler for the document operation 'deleteName'. Deletes an
         * existing defined name from this collection, and (before) triggers a
         * 'delete:name' event.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteName' document operation.
         */
        this.applyDeleteOperation = function (context) {

            // resolve the name model addressed by the operation (throws on error)
            var modelData = getModelData(context);
            var nameModel = modelData.model;

            // delete the name model and notify listeners
            this.trigger('delete:name', nameModel);
            nameModel.destroy();
            nameModelMap.remove(modelData.key);
        };

        /**
         * Callback handler for the document operation 'changeName'. Changes
         * the label or formula definition of an existing defined name, and
         * triggers a 'change:name' event.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeName' document operation.
         */
        this.applyChangeOperation = function (context) {

            // resolve the name model addressed by the operation (throws on error)
            var modelData = getModelData(context);
            var nameModel = modelData.model;

            // apply the changes at the defined name
            var changeFlags = nameModel.applyChangeOperation(context);

            // remap the model in the map, if the label has been changed
            if (changeFlags.label) {

                // the upper-case label is used as map key
                var newKey = nameModel.getKey();
                if (modelData.key !== newKey) {
                    context.ensure(!nameModelMap.has(newKey), 'name exists');
                    nameModelMap.move(modelData.key, newKey);
                    modelData.key = newKey;
                }
            }

            // notify listeners
            if (_.some(changeFlags)) {
                this.trigger('change:name', nameModel, changeFlags);
            }
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether this collection contains a defined name with the
         * passed label.
         *
         * @param {String} label
         *  The (case-insensitive) label of a defined name.
         *
         * @returns {Boolean}
         *  Whether this collection contains a defined name with the passed
         *  label.
         */
        this.hasName = function (label) {
            return nameModelMap.has(SheetUtils.getNameKey(label));
        };

        /**
         * Returns the model of the specified defined name.
         *
         * @param {String} label
         *  The (case-insensitive) label of the defined name to be returned.
         *
         * @returns {NameModel|Null}
         *  The model of the specified defined name, if existing; otherwise
         *  null.
         */
        this.getName = function (label) {
            return nameModelMap.get(SheetUtils.getNameKey(label), null);
        };

        /**
         * Creates an iterator that visits the models of all defined names in
         * this collection.
         *
         * @returns {Object}
         *  An iterator object with the method next(). The result object of the
         *  iterator will contain the model of a defined name as value.
         */
        this.createModelIterator = function () {
            return nameModelMap.iterator();
        };

        /**
         * Returns the models of all defined names contained in this collection
         * as an array.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.skipHidden=false]
         *      If set to true, hidden defined names will not be returned.
         *
         * @returns {Array<NameModel>}
         *  The models of all defined names in this collection.
         */
        this.getAllNames = function (options) {

            // skip hidden names if specified
            if (Utils.getBooleanOption(options, 'skipHidden', false)) {
                return nameModelMap.reject(function (nameModel) { return nameModel.isHidden(); });
            }

            return nameModelMap.values();
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations, and the undo operations, to insert a new
         * defined name into this name collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} label
         *  The exact label for the new defined name.
         *
         * @param {Object} formulaDesc
         *  The descriptor of the formula expression to be bound to the new
         *  defined name. MUST contain the following properties:
         *  @param {String} formulaDesc.grammarId
         *      The identifier of the formula grammar to be used to parse the
         *      formula expression. See class FormulaGrammar for details.
         *  @param {String} formulaDesc.formula
         *      The formula expression to be bound to the new defined name.
         *  @param {Number} formulaDesc.refSheet
         *      The index of the reference sheet the formula is related to.
         *  @param {Address} formulaDesc.refAddress
         *      The address of the reference cell used to interpret relative
         *      cell references in the formula expression.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'name:used': A defined name, or a table range, with the passed
         *      label exists already.
         *  - 'name:empty': The specified label is empty.
         *  - 'name:invalid': The specified label contains invalid characters.
         *  - 'name:address': The specified label would be valid, but conflicts
         *      with the representation of a relative cell reference in A1
         *      notation, or a cell reference in R1C1 notation (either English,
         *      e.g. 'R1C1', or according to current UI language, e.g. 'Z1S1'
         *      in German).
         *  - 'formula:invalid': The passed formula expression is invalid.
         */
        this.generateInsertNameOperations = function (generator, label, formulaDesc) {

            // the label must be valid (method validateNameLabel() returns null, or a rejected promise)
            var labelError = validateNameLabel(label);
            if (labelError) { return labelError; }

            // check that the defined name does not exist yet (check table ranges in the entire document too)
            if (isNameLabelUsed(label)) { return SheetUtils.makeRejected('name:used'); }

            // parse and validate the formula expression
            var properties = getFormulaProperties(formulaDesc);
            if (!properties) { return SheetUtils.makeRejected('formula:invalid'); }
            var undoProperties = {};

            // add the label, and optionally the sheet index, to the operation properties
            properties.label = undoProperties.label = label;
            if (sheetModel) { properties.sheet = undoProperties.sheet = sheetModel.getIndex(); }

            // generate the 'insertName' operation, and the undo operation
            generator.generateOperation(Operations.DELETE_NAME, undoProperties, { undo: true });
            generator.generateOperation(Operations.INSERT_NAME, properties);
            return $.when();
        };

        /**
         * Generates the operations, and the undo operations, to delete an
         * existing defined name from this name collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} label
         *  The label of the defined name to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'operation': Internal error while creating the operations.
         */
        this.generateDeleteNameOperations = function (generator, label) {

            // check that the name exists
            var nameModel = this.getName(label);
            if (!nameModel) { return SheetUtils.makeRejected('operation'); }

            // the properties for the operation, and the undo operation
            var properties = { label: label };
            var undoProperties = {
                label: label,
                formula: nameModel.getFormula('op', Address.A1),
                hidden: nameModel.isHidden()
            };

            // add special undo properties for ODF
            if (!ooxml) {
                undoProperties.isExpr = isComplexExpression(nameModel.getTokenArray());
                undoProperties.ref = nameModel.getRefFormula('op');
            }

            // add the sheet index to the operation properties
            if (sheetModel) { properties.sheet = undoProperties.sheet = sheetModel.getIndex(); }

            // generate the 'deleteName' operation, and the undo operation
            generator.generateOperation(Operations.INSERT_NAME, undoProperties, { undo: true });
            generator.generateOperation(Operations.DELETE_NAME, properties);
            return $.when();
        };

        /**
         * Generates the operations, and the undo operations, to change the
         * label or formula definition of an existing defined name in this name
         * collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} label
         *  The label of the defined name to be changed.
         *
         * @param {String|Null} newLabel
         *  The new label for the defined name. If set to null, the label of
         *  the defined name will not be changed.
         *
         * @param {Object|Null} formulaDesc
         *  The descriptor of the formula expression to be bound to the defined
         *  name. If omitted, the formula expression of the defined name will
         *  not be changed (an invalid formula expression in the defined name
         *  will be retained). See description of the public method
         *  NameCollection.generateInsertNameOperations() for details about the
         *  expected properties.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the generated operations have
         *  been applied successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'name:used': A defined name, or a table range, with the passed
         *      label exists already.
         *  - 'name:empty': The new label is empty.
         *  - 'name:invalid': The new label contains invalid characters.
         *  - 'name:address': The new label would be valid, but conflicts with
         *      the representation of a relative cell reference in A1 notation,
         *      or a cell reference in R1C1 notation (either English, e.g.
         *      'R1C1', or according to current UI language, e.g. 'Z1S1' in
         *      German).
         *  - 'formula:invalid': The passed formula expression is invalid.
         *  - 'operation': Internal error while creating the operations.
         */
        this.generateChangeNameOperations = function (generator, label, newLabel, formulaDesc) {

            // check that the name exists
            var nameModel = this.getName(label);
            if (!nameModel) { return SheetUtils.makeRejected('operation'); }

            // the properties for the 'changeName' operation
            var properties = {};
            var undoProperties = {};

            // check the passed new label (ignore unchanged existing invalid labels, e.g. from import filter)
            newLabel = (_.isString(newLabel) && (label !== newLabel)) ? newLabel : null;
            if (newLabel) {

                // the label must be valid (method validateNameLabel() returns null, or a rejected promise)
                var labelError = validateNameLabel(newLabel);
                if (labelError) { return labelError; }

                // new label must not exist yet in the document (but allow to change case of an existing label)
                if ((SheetUtils.getNameKey(label) !== SheetUtils.getNameKey(newLabel)) && isNameLabelUsed(newLabel)) {
                    return SheetUtils.makeRejected('name:used');
                }

                // add the new label to the operation properties
                properties.newLabel = newLabel;
                undoProperties.newLabel = label;
            }

            // parse and validate the formula expression
            if (_.isObject(formulaDesc)) {
                _.extend(properties, getFormulaProperties(formulaDesc));
                if (!('formula' in properties)) { return SheetUtils.makeRejected('formula:invalid'); }

                // add the current formula expression to the undo properties (ODF: current reference address as formula too)
                undoProperties.formula = nameModel.getFormula('op');
                if (!ooxml) {
                    undoProperties.isExpr = isComplexExpression(nameModel.getTokenArray());
                    undoProperties.ref = nameModel.getRefFormula('op');
                }
            }

            // nothing to do without any changed properties
            if (_.isEmpty(properties)) { return $.when(); }

            // add the label to the operation properties (use new label to address the name in undo)
            properties.label = label;
            undoProperties.label = (newLabel === null) ? label : newLabel;

            // add the sheet index to the operation properties
            if (sheetModel) { properties.sheet = undoProperties.sheet = sheetModel.getIndex(); }

            // generate the 'changeName' operation, and the undo operation
            generator.generateOperation(Operations.CHANGE_NAME, undoProperties, { undo: true });
            generator.generateOperation(Operations.CHANGE_NAME, properties);

            // nothing more to do, if the label has not been changed
            if (!newLabel) { return $.when(); }

            // the change descriptor to be passed to the generator methods of the collections
            var changeDesc = { type: 'relabelName', sheet: sheetModel ? sheetModel.getIndex() : null, oldLabel: label, newLabel: newLabel };

            // update the formulas of all globally defined names
            var promise = docModel.getNameCollection().generateFormulaOperations(generator, changeDesc);

            // update the formula expressions for all sheet collections in the document
            promise = promise.then(function () {
                return docModel.generateOperationsForAllSheets(generator, function (sheetModel) {
                    return Utils.invokeChainedAsync(
                        function () { return sheetModel.getCellCollection().generateFormulaOperations(generator, changeDesc); },
                        function () { return sheetModel.getNameCollection().generateFormulaOperations(generator, changeDesc); },
                        function () { return sheetModel.getValidationCollection().generateFormulaOperations(generator, changeDesc); },
                        function () { return sheetModel.getCondFormatCollection().generateFormulaOperations(generator, changeDesc); },
                        function () { return sheetModel.getDrawingCollection().generateFormulaOperations(generator, changeDesc); }
                    );
                });
            });

            return promise;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the defined names in this collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateFormulaOperations = function (generator, changeDesc) {
            return this.iterateSliced(nameModelMap.iterator(), function (nameModel) {
                nameModel.generateFormulaOperations(generator, changeDesc);
            }, { delay: 'immediate' });
        };

        // initialization -----------------------------------------------------

        // additional processing and event handling after the document has been imported
        this.waitForImportSuccess(function (alreadyImported) {

            // refresh all formulas after import is finished (formulas may refer to sheets not yet imported)
            if (!alreadyImported) {
                nameModelMap.forEach(function (nameModel) {
                    nameModel.getTokenArray().refreshAfterImport();
                });
            }

            // update the sheet indexes after the sheet collection has been manipulated
            this.listenTo(docModel, 'transform:sheet', function (event, toSheet, fromSheet) {
                nameModelMap.forEach(function (nameModel) {
                    nameModel.getTokenArray().transformSheet(toSheet, fromSheet);
                });
            });

        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            nameModelMap.forEach(function (nameModel) { nameModel.destroy(); });
            self = parentModel = docModel = sheetModel = nameModelMap = null;
        });

    } // class NameCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: NameCollection });

});
