/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/mergecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, IteratorUtils, SimpleMap, TimerMixin, ModelObject, Operations, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var RangeArray = SheetUtils.RangeArray;
    var RangeSet = SheetUtils.RangeSet;

    // class MergeCollection ==================================================

    /**
     * Collects information about all merged cell ranges of a single sheet in a
     * spreadsheet document.
     *
     * Triggers the following events:
     * - 'insert:merged'
     *      After new merged ranges have been inserted into the collection.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RangeArray} ranges
     *          An array with the addresses of all new merged ranges.
     * - 'delete:merged'
     *      After merged ranges have been deleted from the collection. Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RangeArray} ranges
     *          An array with the addresses of all deleted merged ranges.
     *
     * @constructor
     *
     * @extends ModelObject
     * @extends TimerMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function MergeCollection(sheetModel) {

        // self reference
        var self = this;

        // the spreadsheet document model
        var docModel = sheetModel.getDocModel();

        // all merged ranges
        var mergedRangeSet = new RangeSet();

        // all merged ranges, mapped by key of their start addresses
        var mergedRangeMap = new SimpleMap();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed merged range into the internal containers.
         */
        function insertMergedRange(mergedRange) {
            mergedRangeSet.insert(mergedRange);
            mergedRangeMap.insert(mergedRange.start.key(), mergedRange);
        }

        /**
         * Removes the passed merged range from the internal containers.
         */
        function removeMergedRange(mergedRange) {
            mergedRangeSet.remove(mergedRange);
            mergedRangeMap.remove(mergedRange.start.key());
        }

        /**
         * Returns whether the single columns or rows of the passed range are
         * merged. The merged columns/rows must exactly fit into the range, and
         * all columns/rows in the range must be merged.
         *
         * @param {Range} range
         *  The address of a cell range.
         *
         * @returns {Boolean}
         *  Whether the columns/rows of the specified range are merged.
         */
        function isColRowMerged(range, columns) {

            // all merged ranges that are contained in the passed range
            var containedRanges = self.getMergedRanges(range, 'contain');

            // the number of merged ranges contained by the passed range must be equal to the
            // column/row count of the range, otherwise there is no need to iterate at all
            if (containedRanges.length !== range.size(columns)) {
                return false;
            }

            // check size and position of the merged ranges (if all merged ranges start and
            // end at the boundaries of the passed range, and are single-sized in the other
            // direction, the entire range is filled by single-sized merged ranges)
            return containedRanges.every(function (mergedRange) {
                return (mergedRange.size(columns) === 1) &&
                    (mergedRange.getStart(!columns) === range.getStart(!columns)) &&
                    (mergedRange.getEnd(!columns) === range.getEnd(!columns));
            });
        }

        /**
         * Generates the 'mergeCells' operation needed to merge the specified
         * cell ranges. The number of generated operations will be minimized by
         * grouping consecutive single column ranges and single row ranges into
         * merge operations with merge type 'horizontal' or 'vertical'.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray} ranges
         *  An array of cell range addresses.
         *
         * @param {Object} [options]
         *  Optional parameters passed to the operations generator which can be
         *  used to decide whether to append or prepend operations, or whether
         *  to generate undo operation.
         */
        function generateMergeOperations(generator, ranges, options) {

            // generate 'vertical' or 'horizontal' merge operations for single-column/single-row ranges
            function collectAndGenerateOperations(singleRanges, columns) {

                // the merge type for the generated operations
                var mergeType = columns ? 'vertical' : 'horizontal';

                // sort the ranges according to the direction
                var comparator = columns ? Address.compare : Address.compareVert;
                singleRanges.sort(function (range1, range2) { return comparator(range1.start, range2.start); });

                // due to the sorting, ranges that can be combined into a single operation are side-by-side in the array
                for (var arrayIndex = 0, count = 0; arrayIndex < singleRanges.length; arrayIndex += count) {

                    // the first range for the new operation
                    var range1 = singleRanges[arrayIndex];
                    var index = range1.getStart(columns);
                    var start = range1.getStart(!columns);
                    var end = range1.getEnd(!columns);

                    // continue in the array as long as following ranges are exactly adjacent to 'range1'
                    for (count = 1; arrayIndex + count < singleRanges.length; count += 1) {
                        var range2 = singleRanges[arrayIndex + count];
                        if (index + count !== (range2.getStart(columns)) || (start !== range2.getStart(!columns)) || (end !== range2.getEnd(!columns))) { break; }
                    }

                    // create one operation for all adjacent merged ranges
                    var compoundRange = columns ? range1.colRange(0, count) : range1.rowRange(0, count);
                    generator.generateRangeOperation(Operations.MERGE_CELLS, compoundRange, { type: (count > 1) ? mergeType : 'merge' }, options);
                }
            }

            // split the merged ranges into single-column, single-row, and two-dimensional ranges
            var rangeGroups = ranges.group(function (range) {
                return range.singleCol() ? 'col' : range.singleRow() ? 'row' : 'range';
            });

            // generate 'vertical' merge operations for single-column ranges
            if (rangeGroups.col) {
                collectAndGenerateOperations(rangeGroups.col, true);
            }

            // generate 'horizontal' merge operations for single-row ranges
            if (rangeGroups.row) {
                collectAndGenerateOperations(rangeGroups.row, false);
            }

            // generate undo operations for two-dimensional ranges directly
            if (rangeGroups.range) {
                rangeGroups.range.forEach(function (range) {
                    generator.generateRangeOperation(Operations.MERGE_CELLS, range, { type: 'merge' }, options);
                });
            }
        }

        /**
         * Triggers the specified event, if the passed array of range addresses
         * is not empty.
         */
        function triggerEvent(type, ranges, options) {
            if (!ranges.empty()) {
                self.trigger(type, ranges, options);
            }
        }

        /**
         * Transforms the specified merged range.
         *
         * @param {Range} mergedRange
         *  The address of a merged range to be transformed.
         *
         * @param {Array<MoveDescriptor>} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  passed merged range.
         *
         * @param {Boolean} [reverse=false]
         *  If set to true, the move descriptors will be processed in reversed
         *  order, and the opposite move operation will be used to transform
         *  the merged range.
         *
         * @returns {Range|Null}
         *  The transformed merged range if available (will contain more than
         *  one cell); otherwise null.
         */
        function transformMergedRange(mergedRange, moveDescs, reverse) {

            // transform the passed range without expanding the end of the range
            Utils.iterateArray(moveDescs, function (moveDesc) {
                mergedRange = moveDesc.transformRange(mergedRange, { reverse: reverse });
                if (!mergedRange) { return Utils.BREAK; }
            }, { reverse: reverse });

            // skip merged ranges that have been shrunken to a single cell
            return (mergedRange && !mergedRange.single()) ? mergedRange : null;
        }

        /**
         * Recalculates the position of all merged ranges, after cells have
         * been moved (including inserted/deleted columns or rows) in the
         * sheet.
         */
        function moveCellsHandler(event, moveDesc) {

            // the original merged ranges
            var oldRangeSet = mergedRangeSet;

            // fill the containers during range transformation
            mergedRangeSet = new RangeSet();
            mergedRangeMap.clear();

            // process all original merged ranges
            oldRangeSet.forEach(function (mergedRange) {
                mergedRange = transformMergedRange(mergedRange, [moveDesc]);
                if (mergedRange) { insertMergedRange(mergedRange); }
            });
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { mergedRangeSet: mergedRangeSet };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * merged ranges from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {MergeCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            // clone the contents of the source collection
            cloneData.mergedRangeSet.forEach(function (mergedRange) {
                insertMergedRange(mergedRange.clone());
            });
        };

        /**
         * Callback handler for the document operation 'mergeCells'. Merges the
         * cell range, or removes any merged ranges from the range covered by
         * the operation.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'mergeCells' document operation.
         */
        this.applyMergeCellsOperation = function (context) {

            // the range covered by the operation
            var range = context.getRange();
            // the method how to merge/unmerge the operation range
            var type = context.getEnum('type', /^(merge|horizontal|vertical|unmerge)$/);
            // all merged ranges deleted from the collection
            var removedRanges = new RangeArray();
            // all merged ranges added to the collection
            var insertedRanges = new RangeArray();

            // inserts the passed merged range into the containers, and inserts it into 'insertedRanges'
            function registerMergedRange(mergedRange) {
                insertMergedRange(mergedRange);
                insertedRanges.push(mergedRange.clone());
            }

            // implicitly remove all merged ranges that overlap with the passed range,
            // regardless of passed type (needed to merge over merged ranges)
            mergedRangeSet.findRanges(range).forEach(function (mergedRange) {
                removeMergedRange(mergedRange);
                removedRanges.push(mergedRange);
            });

            // handle the different merge types
            switch (type) {

                case 'merge':
                    if (range.cells() > 1) {
                        registerMergedRange(range);
                    }
                    break;

                case 'horizontal':
                    if (range.cols() > 1) {
                        for (var row = range.start[1]; row <= range.end[1]; row += 1) {
                            registerMergedRange(Range.create(range.start[0], row, range.end[0], row));
                        }
                    }
                    break;

                case 'vertical':
                    if (range.rows() > 1) {
                        for (var col = range.start[0]; col <= range.end[0]; col += 1) {
                            registerMergedRange(Range.create(col, range.start[1], col, range.end[1]));
                        }
                    }
                    break;

                case 'unmerge':
                    // all merged ranges have been removed already, see above
                    break;
            }

            // trigger change events with array of changed merged ranges
            triggerEvent('delete:merged', removedRanges);
            triggerEvent('insert:merged', insertedRanges);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether this collection is empty.
         *
         * @returns {Boolean}
         *  Whether this collection is empty
         */
        this.isEmpty = function () {
            return mergedRangeSet.empty();
        };

        /**
         * Returns, whether the cell at the specified address is the upper left
         * corner of a merged cell range.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell at the specified address is the upper left corner
         *  of a merged cell range.
         */
        this.isReferenceCell = function (address) {
            return this.getMergedRange(address, 'reference') !== null;
        };

        /**
         * Returns, whether the cell at the specified address is a hidden cell
         * inside a merged cell range.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell at the specified address is a hidden cell inside a
         *  merged cell range.
         */
        this.isHiddenCell = function (address) {
            return this.getMergedRange(address, 'hidden') !== null;
        };

        /**
         * Returns, whether the cell at the specified address is part of a
         * merged cell. It can be the reference cell or a hidden cell.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell is part of a merged cell range.
         */
        this.isMergedCell = function (address) {
            return this.getMergedRange(address, 'all') !== null;
        };

        /**
         * Returns whether the passed range is contained in this collection.
         *
         * @param {Range} range
         *  The cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified range is contained in this collection.
         */
        this.isMergedRange = function (range) {
            return mergedRangeSet.has(range);
        };

        /**
         * Returns whether the single columns of the passed range are merged.
         * The merged columns must exactly fit into the range, and all columns
         * in the range must be merged.
         *
         * @param {Range} range
         *  The cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified range is entirely vertically merged.
         */
        this.isVerticallyMerged = function (range) {
            return isColRowMerged(range, true);
        };

        /**
         * Returns whether the single rows of the passed range are merged. The
         * merged rows must exactly fit into the range, and all rows in the
         * range must be merged.
         *
         * @param {Range} range
         *  The cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified range is entirely horizontally merged.
         */
        this.isHorizontallyMerged = function (range) {
            return isColRowMerged(range, false);
        };

        /**
         * Returns the range address of a merged cell range for a specified
         * cell address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @param {String} [type='all']
         *  The address type of the specified address. It can be an arbitrary
         *  cell inside a merged cell range (type === 'all'), or it can be
         *  restricted to the upper left reference cell (type === 'reference')
         *  or to a hidden cell (type === 'hidden'). A hidden cell is any cell
         *  inside the merged cell range, that is not the upper left reference
         *  cell. If this address type is not specified, or if it is not one of
         *  the three types 'all', 'reference' or 'hidden', it defaults to
         *  'all'.
         *
         * @returns {Range|Null}
         *  The range address of a merged range that covers the passed cell
         *  address; or null, if no merged range exists at the address.
         */
        this.getMergedRange = function (address, type) {

            // quickly find a merged range by its start address
            var mergedRange = mergedRangeMap.get(address.key(), null);

            // do not try to search in the set for search type 'reference'
            if (type === 'reference') { return mergedRange ? mergedRange.clone() : null; }

            // return a found merged range (but return null if search type is 'hidden')
            if (mergedRange) { return (type === 'hidden') ? null : mergedRange.clone(); }

            // find a merged range covering the passed address (it will not start at that address, otherwise
            // it would have been found above, therefore return it for search types 'hidden' and 'all')
            mergedRange = mergedRangeSet.findByAddress(address).first();
            return mergedRange ? mergedRange.clone() : null;
        };

        /**
         * Returns the range addresses of all merged cell ranges that overlap
         * with one of the passed ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {String} [type='all']
         *  If set to 'all' or omitted, returns all merged ranges that overlap
         *  the passed ranges with any cell. If set to 'contain', returns only
         *  merged ranges that are completely contained in any of the passed
         *  ranges. If set to 'reference', returns only merged ranges whose
         *  reference cell is contained in any of the passed ranges.
         *
         * @returns {RangeArray}
         *  The range addresses of all merged ranges that overlap with one of
         *  the passed ranges.
         */
        this.getMergedRanges = function (ranges, type) {

            // the resulting merged ranges
            var resultRanges = new RangeArray();

            // process all passed ranges
            RangeArray.forEach(ranges, function (range) {

                // find all merged ranges overlapping with the current range
                mergedRangeSet.findRanges(range).forEach(function (mergedRange) {
                    switch (type) {
                        case 'reference':
                            if (range.containsAddress(mergedRange.start)) {
                                resultRanges.push(mergedRange);
                            }
                            break;
                        case 'contain':
                            if (range.contains(mergedRange)) {
                                resultRanges.push(mergedRange);
                            }
                            break;
                        default:
                            resultRanges.push(mergedRange);
                    }
                });
            });

            return resultRanges;
        };

        /**
         * Returns the merged range that completely contains or is equal to the
         * passed cell range. Note the difference to the method
         * MergeCollection.getMergedRanges() which returns all merged ranges
         * that are contained IN the passed ranges.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Range|Null}
         *  The address of a merged range that contains the passed range if
         *  existing, otherwise null
         */
        this.getBoundingMergedRange = function (range) {
            // if the first found overlapping merged range does not contain the range, no other
            // merged range can contain it either because merged ranges do not overlap each other
            var mergedRange = mergedRangeSet.findRanges(range).first();
            return (mergedRange && mergedRange.contains(range)) ? mergedRange.clone() : null;
        };

        /**
         * Returns whether any of the passed cell ranges overlaps with at least
         * one merged cell range.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {Boolean}
         *  Whether any of the passed cell ranges contains or overlaps with at
         *  least one merged cell range.
         */
        this.rangesOverlapAnyMergedRange = function (ranges) {
            return RangeArray.some(ranges, function (range) {
                return mergedRangeSet.overlaps(range);
            });
        };

        /**
         * Expands the passed range address, so that it will include all merged
         * ranges it is currently covering partly.
         *
         * @param {Range} range
         *  The address of a cell range that will be expanded to the merged
         *  ranges contained in this collection.
         *
         * @returns {Range}
         *  The address of the expanded cell range.
         */
        this.expandRangeToMergedRanges = function (range) {

            // shallow copy of the range set, to prevent processing them multiple times
            var pendingRangeSet = mergedRangeSet.clone();

            // repeat as long as 'pendingRangeSet' provides new merged ranges to expand with
            while (true) {

                // find all merged ranges overlapping with the current range, return the
                // current range address if no more merged ranges can be found
                var mergedRanges = pendingRangeSet.findRanges(range);
                if (mergedRanges.empty()) { return range.clone(); }

                // expand the range to all merged ranges, remove them from 'pendingRangeSet'
                range = range.boundary(mergedRanges.boundary());
                mergedRanges.forEach(pendingRangeSet.remove, pendingRangeSet);
            }
        };

        /**
         * Shrinks the passed range address, so that it will not include any
         * merged ranges it is currently covering partly.
         *
         * @param {Range} range
         *  The address of a cell range that will be shrunk until it does not
         *  partly contain any merged ranges.
         *
         * @param {Boolean} columns
         *  The direction in which the passed range will be shrunk. If set to
         *  true, the range will be shrunken at its left and right borders,
         *  otherwise at its top and bottom borders.
         *
         * @returns {Range|Null}
         *  The address of the shrunk range; or null, if no valid range was
         *  left after shrinking (the range has been shrunk to zero
         *  columns/rows).
         */
        this.shrinkRangeFromMergedRanges = (function () {

            // shrinks the passed range by all merged ranges jutting out to the left or top
            function shrinkRangeAtLeadingBorder(oldRange, columns) {

                // outer cells of the passed range
                var borderRange = columns ? oldRange.leadingCol() : oldRange.headerRow();
                // the merged ranges overlapping with the passed border range
                var mergedRanges = mergedRangeSet.findRanges(borderRange);
                // copy of the passed range that will be shrunken
                var newRange = oldRange.clone();

                // reduce the passed range by all found merged ranges that are jutting out
                mergedRanges.every(function (mergedRange) {
                    if (mergedRange.getStart(columns) < oldRange.getStart(columns)) {
                        var index = Math.max(newRange.getStart(columns), mergedRange.getEnd(columns) + 1);
                        if (index > newRange.getEnd(columns)) {
                            newRange = null;
                        } else {
                            newRange.setStart(index, columns);
                        }
                    }
                    return newRange;
                });

                return newRange;
            }

            // shrinks the passed range by all merged ranges jutting out to the right or bottom
            function shrinkRangeAtTrailingBorder(oldRange, columns) {

                // outer cells of the passed range
                var borderRange = columns ? oldRange.trailingCol() : oldRange.footerRow();
                // the merged ranges overlapping with the passed border range
                var mergedRanges = mergedRangeSet.findRanges(borderRange);
                // copy of the passed range that will be shrunken
                var newRange = oldRange.clone();

                // reduce the passed range by all found merged ranges that are jutting out
                mergedRanges.every(function (mergedRange) {
                    if (oldRange.getEnd(columns) < mergedRange.getEnd(columns)) {
                        var index = Math.min(newRange.getEnd(columns), mergedRange.getStart(columns) - 1);
                        if (index < newRange.getStart(columns)) {
                            newRange = null;
                        } else {
                            newRange.setStart(index, columns);
                        }
                    }
                    return newRange;
                });

                return newRange;
            }

            // return the implementation of the public method from the local scope
            return function (range, columns) {

                // the new shrunken range
                var newRange = null;

                // shrink the range at the left or top border as often as possible
                while (true) {
                    newRange = shrinkRangeAtLeadingBorder(range, columns);
                    if (!newRange) { return null; }
                    if (range.equals(newRange)) { break; }
                    range = newRange;
                }

                // shrink the range at the right or bottom border as often as possible
                while (true) {
                    newRange = shrinkRangeAtTrailingBorder(range, columns);
                    if (!newRange) { return null; }
                    if (range.equals(newRange)) { break; }
                    range = newRange;
                }

                return newRange;
            };
        }());

        // operation generators -----------------------------------------------

        /**
         * Generates and applies 'mergeCells' operations for all passed ranges.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address.
         *
         * @param {String} type
         *  The merge type. Must be one of:
         *  - 'merge': Merges the entire ranges.
         *  - 'horizontal': Merges the single rows in all ranges.
         *  - 'vertical': Merges the single columns in all ranges.
         *  - 'unmerge': Removes all merged ranges covered by the ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected with an object
         *  with 'cause' property set to one of the following error codes:
         *  - 'merge:overlap': The passed cell ranges overlap each other.
         *  - 'merge:overflow': Tried to merge too many ranges at once.
         */
        this.generateMergeCellsOperations = function (generator, ranges, type) {

            // creates col or row ranges in case of merging 'horizontal' or 'vertical'
            function createColRowRanges(ranges, type) {
                var horizontal = (type === 'horizontal'),
                    collection = horizontal ? sheetModel.getRowCollection() : sheetModel.getColCollection(),
                    newRanges  = new RangeArray();

                ranges.forEach(function (ran) {
                    var interval = (horizontal) ? ran.rowInterval() : ran.colInterval();

                    IteratorUtils.forEach(collection.createIterator(interval), function (desc) {
                        var start_col = horizontal ? ran.start[0] : desc.index,
                            start_row = horizontal ? desc.index : ran.start[1],
                            end_col   = horizontal ? ran.end[0] : desc.index,
                            end_row   = horizontal ? desc.index : ran.end[1];

                        newRanges.push(Range.create(start_col, start_row, end_col, end_row));
                    });
                });

                return newRanges;
            }

            // the cell collection of the sheet
            var cellCollection = sheetModel.getCellCollection();
            // whether to merge single columns in the passed ranges
            var vertical = type === 'vertical';
            // whether to merge single rows in the passed ranges
            var horizontal = type === 'horizontal';
            // whether to unmerge the ranges
            var unmerge = type === 'unmerge';

            // convert ranges to unique array
            ranges = RangeArray.get(ranges).unify();

            // merging may have no effect for specific ranges
            ranges = ranges.reject(function (range) {
                return horizontal ? range.singleCol() : vertical ? range.singleRow() : range.single();
            });

            if (horizontal || vertical) {
                ranges = createColRowRanges(ranges, type);
            }

            // nothing to do without ranges
            if (ranges.empty()) { return $.when(); }

            // do not allow to merge overlapping ranges
            if (!unmerge && ranges.overlapsSelf()) {
                return SheetUtils.makeRejected('merge:overlap');
            }

            // count number of new merged ranges, shorten the new merged ranges to the used
            // area of the cell collection (bug 30662)
            if (!unmerge) {

                // the maximum used column/row index
                var maxIndex = Math.max(0, cellCollection.getUsedCount(type !== 'horizontal') - 1);
                // the total number of created merged ranges
                var rangeCount = 0;

                // count the new merged ranges, shorten entire column/row ranges (bug 30662)
                ranges.forEach(function (range) {
                    if (vertical) {
                        // reduce range to used area for entire rows
                        if (docModel.isRowRange(range)) { range.end[0] = maxIndex; }
                        rangeCount += range.cols();
                    } else if (horizontal) {
                        // reduce range to used area for entire columns
                        if (docModel.isColRange(range)) { range.end[1] = maxIndex; }
                        rangeCount += range.rows();
                    } else {
                        rangeCount += 1;
                    }
                });

                // overflow check
                if (rangeCount > SheetUtils.MAX_MERGED_RANGES_COUNT) {
                    return SheetUtils.makeRejected('merge:overflow');
                }
            }

            // collect all current merged ranges covered by the passed ranges (will be unmerged regardless
            // of the merge type, and therefore need to be restored with undo operations)
            var restoreRanges = new RangeArray();
            var remainingRanges = mergedRangeSet.values();

            // process all ranges separately
            var promise = this.iterateArraySliced(ranges, function (range) {

                // split the remaining merged ranges (not yet covered by a range passed to this method)
                // into two groups, depending whether they are covered by the range currently processed
                var rangeGroups = remainingRanges ? remainingRanges.group(function (mergedRange) {
                    return mergedRange.overlaps(range) ? 'overlap' : 'remaining';
                }) : {};

                // generate an 'unmerge' operation (also, if merging causes to unmerge existing merged ranges)
                if (unmerge || rangeGroups.overlap) {
                    generator.generateRangeOperation(Operations.MERGE_CELLS, range, { type: 'unmerge' });
                }

                // generate a 'merge' operation for the range, and the according 'unmerge' operation for undo
                if (!unmerge) {
                    generator.generateRangeOperation(Operations.MERGE_CELLS, range, { type: 'unmerge' }, { undo: true });
                    generator.generateRangeOperation(Operations.MERGE_CELLS, range, { type: type });
                }

                // all merged ranges covered by the current range need to be restored in undo (see below)
                if (rangeGroups.overlap) {
                    restoreRanges.append(rangeGroups.overlap);
                }

                // continue with only the merged ranges not covered by a range passed to this method
                // (will become undefined, if no more merged ranges of this collection are left)
                remainingRanges = rangeGroups.remaining;

            }, { delay: 'immediate' });

            // generate the hyperlink operations
            promise = promise.then(function () {
                return sheetModel.getHyperlinkCollection().generateMergeCellsOperations(generator, ranges);
            });

            // generate the cell operations (e.g.: move values to top-left cells, expand formatting)
            promise = promise.then(function () {
                return cellCollection.generateMergeCellsOperations(generator, ranges, type);
            });

            // add all merged ranges to the undo generator that need to be restored
            return promise.then(function () {
                generateMergeOperations(generator, restoreRanges, { undo: true });
            });
        };

        /**
         * Generates the undo operations to restore the merged ranges in this
         * collection that would not be restored automatically with the reverse
         * operation of the passed move descriptor.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<MoveDescriptor>} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  merged ranges in this collection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMoveCellsOperations = function (generator, moveDescs) {

            // collect all merged ranges that cannot be restored by applying the reversed move operation
            var restoreRanges = new RangeArray();
            var promise = this.iterateSliced(mergedRangeMap.iterator(), function (mergedRange) {

                // transform the merged range back and forth to decide whether it can be restored implicitly
                var transformRange = transformMergedRange(mergedRange, moveDescs);
                var restoredRange = transformRange ? transformMergedRange(transformRange, moveDescs, true) : null;

                // collect all ranges that cannot be restored implicitly
                if (!restoredRange || restoredRange.differs(mergedRange)) {
                    restoreRanges.push(mergedRange);
                }
            }, { delay: 'immediate' });

            // add all merged ranges to the undo generator that need to be restored
            return promise.done(function () {
                generateMergeOperations(generator, restoreRanges, { undo: true });
            });
        };

        /**
         *
         */
        this.generateAutoFillOperations = SheetUtils.profileAsyncMethod('ColRowCollection.generateAutoFillOperations()', function (generator, sourceRange, targetRange, columns, reverse) {

            function prepareNewMergedRanges(baseRange) {
                var newMergedRange = baseRange.clone(),
                    multiplikator = reverse ? -1 : 1;

                newMergedRange.moveBoth(start_col, true);
                newMergedRange.moveBoth(start_row, false);
                newMergedRange.moveBoth((targetMerge * sourceSize) * multiplikator, !columns);
                op_mergedRanges.push(newMergedRange);
            }

            var sourceSize              = sourceRange.size(!columns),
                targetSize              = targetRange.size(!columns),
                targetMerge             = targetSize / sourceSize,
                start_col               = sourceRange.start[0],
                start_row               = sourceRange.start[1],
                mergedRanges            = self.getMergedRanges(sourceRange, 'contain'),
                cleanedUpMergedRanges   = [],
                op_mergedRanges         = new RangeArray();

            mergedRanges.forEach(function (mergedRange) {
                var currentRange     = mergedRange.clone();

                currentRange.setStart((currentRange.start[0] - start_col), true);
                currentRange.setEnd((currentRange.end[0] - start_col), true);

                currentRange.setStart((currentRange.start[1] - start_row), false);
                currentRange.setEnd((currentRange.end[1] - start_row), false);

                cleanedUpMergedRanges.push(currentRange);
            });

            while (targetMerge > 0) {
                cleanedUpMergedRanges.forEach(prepareNewMergedRanges);
                targetMerge--;
            }

            return self.generateMergeCellsOperations(generator, op_mergedRanges, 'merge');
        });

        // initialization -----------------------------------------------------

        // update merged ranges after moving cells, or inserting/deleting columns or rows
        this.listenTo(sheetModel, 'move:cells', moveCellsHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = sheetModel = mergedRangeSet = mergedRangeMap = null;
        });

    } // class MergeCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: MergeCollection });

});
