/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/dateutils', [
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/utils/errorcode'
], function (BaseDateUtils, FormulaUtils, ErrorCode) {

    'use strict';

    // static private functions ===============================================

    /**
     * Returns the number of leap years contained in the passed year interval.
     *
     * @param {Number} firstYear
     *  The first year in the year interval (inclusive).
     *
     * @param {Number} lastYear
     *  The last year in the year interval (inclusive). MUST be equal to or
     *  greater than the value passed in firstYear.
     *
     * @returns {Number}
     *  The number of leap years contained in the passed year interval.
     */
    function getLeapYears(firstYear, lastYear) {

        // returns the number of years divisible by the specified value
        function getDivisibleCount(divisor) {
            return Math.floor(lastYear / divisor) - Math.floor((firstYear - 1) / divisor);
        }

        // add the number of years divisible by 4, subtract the number of years divisible by 100, add the number of years divisible by 400
        return getDivisibleCount(4) - getDivisibleCount(100) + getDivisibleCount(400);
    }

    function getYearsDiff(startDate, endDate) {
        return Math.floor(getMonthsDiff(startDate, endDate) / 12);
    }

    function getMonthsDiff(startDate, endDate) {
        var diff = 12 * (endDate.getUTCFullYear() - startDate.getUTCFullYear()) + endDate.getUTCMonth() - startDate.getUTCMonth();
        if (diff > 0 && endDate.getUTCDate() < startDate.getUTCDate()) {
            diff -= 1;
        }
        return diff;
    }

    function getWeeksDiff(startDate, endDate) {
        return Math.floor(DateUtils.getDaysDiff(startDate, endDate) / 7);
    }

    function getDiff(startDate, endDate, options) {
        var whole = options.whole;
        var startFn = options.startFn;
        var diffFn = options.diffFn;

        var prefix = 1;

        // order start and end date
        if (endDate < startDate) {
            var tmpDate = startDate;
            startDate = endDate;
            endDate = tmpDate;
            prefix = -1;
        }

        if (whole) {
            return prefix * diffFn(startDate, endDate);
        } else {
            return prefix * diffFn(startFn(startDate), startFn(endDate));
        }
    }

    // static class DateUtils =================================================

    var DateUtils = _.clone(BaseDateUtils);

    // public methods ---------------------------------------------------------

    /**
     * Returns the number of years between two dates.
     * If startdate is after enddate the result will be negative.
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value.
     *
     * @param {Boolean} whole
     *  If partial is false, it identifies the month that startdate and enddate each lie in, and returns the difference between those months.
     *  if partial is true, it returns the number of whole years between startdate and enddate,
     *
     * @return {Number}
     *  months between startDate and endDate
     *  If startdate is after enddate the result will be negative.
     */
    DateUtils.getYearsDiff = function (startDate, endDate, whole) {
        return getDiff(startDate, endDate, {
            whole: whole,
            startFn: DateUtils.getYearStart,
            diffFn: getYearsDiff
        });
    };

    /**
     * Returns the number of months between two dates.
     * If startdate is after enddate the result will be negative.
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value.
     *
     * @param {Boolean} whole
     *  If partial is false, it identifies the year that startdate and enddate each lie in, and returns the difference between those years.
     *  If partial is true, it returns the number of whole months between startdate and enddate, day of the month to day of the month.
     *
     * @return {Number}
     *  months between startDate and endDate
     *  If startdate is after enddate the result will be negative.
     */
    DateUtils.getMonthsDiff = function (startDate, endDate, whole) {
        return getDiff(startDate, endDate, {
            whole: whole,
            startFn: DateUtils.getMonthStart,
            diffFn: getMonthsDiff
        });
    };

    /**
     * Returns the number of weeks between two dates.
     * If startdate is after enddate the result will be negative.
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value.
     *
     * @param {Boolean} partial
     *  If partial is false, it identifies the Monday_to_Sunday week that startdate and enddate each lie in, and returns the difference between those weeks.
     *  If partial is true, it returns the number of whole weeks between startdate and enddate - that is INT(number_of_days_difference / 7).
     *
     * @return {Number}
     *  months between startDate and endDate
     *  If startdate is after enddate the result will be negative.
     */
    DateUtils.getWeeksDiff = function (startDate, endDate, whole) {
        return getDiff(startDate, endDate, {
            whole: whole,
            startFn: DateUtils.getWeekStart,
            diffFn: getWeeksDiff
        });
    };

    /**
     * Returns the number of entire days between the passed start and end date,
     * ignoring the time contained in the passed date objects.
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value.
     *
     * @returns {Number}
     *  The number of days between start date and end date (negative, if the
     *  start date is behind the end date).
     */
    DateUtils.getDaysDiff = function (startDate, endDate) {
        // ignore time for each date separately (may otherwise distort the result by one day)
        return Math.floor(endDate.getTime() / DateUtils.MSEC_PER_DAY) - Math.floor(startDate.getTime() / DateUtils.MSEC_PER_DAY);
    };

    /**
     * Returns the number of days between the passed start and end date for a
     * specific '30/360' day count convention (based on a year with 360 days,
     * and all months with 30 days each).
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value.
     *
     * @param {String} method
     *  The day count convention. Must be one of the following values:
     *  - 'eur': The 30/360 European method.
     *  - 'us:days': The 30/360 US method as used by the function DAYS360.
     *  - 'us:frac': The 30/360 US method as used by the function YEARFRAC.
     *
     * @returns {Number}
     *  The number of days between start date and end date. The result will be
     *  negative, if the start date is behind the end date.
     */
    DateUtils.getDays360 = function (startDate, endDate, method) {

        // returns whether the passed date is the last day of February
        function isLastDayOfFeb(comps) {
            return (comps.M === 1) && (comps.D === DateUtils.getDaysInFebruary(comps.Y));
        }

        // extract original year/month/day of the passed start date
        var startComps = DateUtils.getUTCDateComponents(startDate);
        // extract original year/month/day of the passed end date
        var endComps = DateUtils.getUTCDateComponents(endDate);
        // the adjusted day of the start and end date
        var startDay = 0, endDay = 0;

        switch (method) {

            // The 30/360 European method, as used by the functions DAYS360 and YEARFRAC
            case 'eur':

                // if start date is day 31 of the month, set it to day 30
                startDay = Math.min(startComps.D, 30);
                // if end date is day 31 of the month, set it to day 30
                endDay = Math.min(endComps.D, 30);
                break;

            // The 30/360 US method, as used by the function DAYS360
            case 'us:days':

                // if start date is the last day of its month, set it to day 30 (also for February)
                startDay = isLastDayOfFeb(startComps) ? 30 : Math.min(startComps.D, 30);
                // if end date is day 31 of its month, and adjusted start day is 30,
                // set the end day to day 30 (28th/29th of February remains as is)
                endDay = (startDay === 30) ? Math.min(endComps.D, 30) : endComps.D;
                break;

            // The 30/360 US method, as used by the function YEARFRAC (slightly different than DAYS360, of course)
            case 'us:frac':

                // whether start date is the last day in February
                var startLastFeb = isLastDayOfFeb(startComps);
                // whether end date is the last day in February
                var endLastFeb = isLastDayOfFeb(endComps);

                // if start date is the last day of its month, set it to day 30 (also for February)
                startDay = startLastFeb ? 30 : Math.min(startComps.D, 30);
                // if (start date and end date are last day of February), OR if (original start day
                // is 30 or 31, and original end day is 31), set end date to day 30
                endDay = ((startLastFeb && endLastFeb) || ((startComps.D >= 30) && (endComps.D === 31))) ? 30 : endComps.D;
                break;

            default:
                FormulaUtils.throwInternal('DateTimeFuncs.getDays360(): unknown day convention code: ' + method);
        }

        // the resulting number of days, based on a year with equally-sized months of 30 days each
        return (endComps.Y - startComps.Y) * 360 + (endComps.M - startComps.M) * 30 + endDay - startDay;
    };

    /**
     * Calculates the difference between the passed dates. Implements different
     * methods as used in various sheet functions.
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value. If this date is located before the
     *  start date, the dates will be swapped internally (the result will never
     *  be negative).
     *
     * @param {Number} [dateMode=0]
     *  The date mode to be used to calculate the difference between the passed
     *  dates. The following date modes are supported:
     *  - 0 (default): The '30/360 US' method.
     *  - 1: The 'actual/actual' method (variable number of days per year).
     *  - 2: The 'actual/360' method (exact number of days, 360 days per year).
     *  - 3: The 'actual/365' method (exact number of days, 365 days per year).
     *  - 4: The '30/360 European' method.
     *
     * @returns {Object}
     *  A result descriptor with the following properties:
     *  - {Number} days
     *      The number of days between start date and end date.
     *  - {Number} daysPerYear
     *       The number of days per year, according to the passed date mode.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed date mode is invalid.
     */
    DateUtils.getYearFracData = function (startDate, endDate, dateMode) {

        // the resulting difference in days
        var days = 0;
        // the number of days per year, according to the date mode
        var daysPerYear = 0;

        // sort start and end date
        if (endDate < startDate) {
            var tmpDate = startDate;
            startDate = endDate;
            endDate = tmpDate;
        }

        // returns whether first day/month is before or equal to second day/month (assuming same year)
        function lessOrEqDate(month1, day1, month2, day2) {
            return (month1 < month2) || ((month1 === month2) && (day1 <= day2));
        }

        // return fraction according to passed mode (default to mode 0 '30/360 US')
        switch (_.isNumber(dateMode) ? dateMode : 0) {

            // method '30/360 US', special handling for last days in February
            case 0:
                days = DateUtils.getDays360(startDate, endDate, 'us:frac');
                daysPerYear = 360;
                break;

            // method 'actual/actual'
            case 1:
                days = DateUtils.getDaysDiff(startDate, endDate);
                daysPerYear = 365;

                // extract original year/month/day of the passed start date
                var startComps = DateUtils.getUTCDateComponents(startDate);
                // extract original year/month/day of the passed end date
                var endComps = DateUtils.getUTCDateComponents(endDate);

                // special behavior if end date is at most one year behind start date (regardless of actual day and month)
                if ((startComps.Y === endComps.Y) || ((startComps.Y + 1 === endComps.Y) && lessOrEqDate(endComps.M, endComps.D, startComps.M, startComps.D))) {

                    // YEARFRAC uses 366 days per year, if (start date is located in a leap year, and is
                    // before Mar-01), OR if (end date is located in a leap year, and is after Feb-28)
                    if ((DateUtils.isLeapYear(startComps.Y) && (startComps.M <= 1)) || (DateUtils.isLeapYear(endComps.Y) && lessOrEqDate(1, 29, endComps.M, endComps.D))) {
                        daysPerYear += 1;
                    }

                // otherwise: end date is more than one year after start date (including e.g. 2011-02-28 to 2012-02-29)
                } else {

                    // include the number of leap years contained in the year interval as fractional part
                    daysPerYear += getLeapYears(startComps.Y, endComps.Y) / (endComps.Y - startComps.Y + 1);
                }
                break;

            // method 'actual/360'
            case 2:
                days = DateUtils.getDaysDiff(startDate, endDate);
                daysPerYear = 360;
                break;

            // method 'actual/365'
            case 3:
                days = DateUtils.getDaysDiff(startDate, endDate);
                daysPerYear = 365;
                break;

            // method '30/360 European'
            case 4:
                days = DateUtils.getDays360(startDate, endDate, 'eur');
                daysPerYear = 360;
                break;

            // throw #NUM! error code for invalid date modes
            default:
                throw ErrorCode.NUM;
        }

        // return the result object with the number of days, and the days per year
        return { days: days, daysPerYear: daysPerYear };
    };

    /**
     * Returns the number of years between start date and end date as floating-
     * point number, according to the passed date mode.
     *
     * @param {Date} startDate
     *  The start date, as UTC date value.
     *
     * @param {Date} endDate
     *  The end date, as UTC date value. If this date is located before the
     *  start date, the dates will be swapped internally (the result will never
     *  be negative).
     *
     * @param {Number} [dateMode=0]
     *  The date mode to be used to calculate the number of years. See method
     *  DateUtils.getYearFracData() for details.
     *
     * @returns {Number}
     *  The number of years between start date and end date, as floating-point
     *  number, according to the passed date mode.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed date mode is invalid.
     */
    DateUtils.getYearFrac = function (startDate, endDate, dateMode) {
        var relation = DateUtils.getYearFracData(startDate, endDate, dateMode);
        return relation.days / relation.daysPerYear;
    };

    /**
     * returns the week day of assigned date,
     * where Monday is 0 Tuesday is 1 and so on
     *
     * @param {Date} date
     *  The original date, as UTC date value.
     *
     * @return {Number}
     *
     */
    DateUtils.getWeekDay = function (date) {
        return (date.getUTCDay() + 6) % 7;
    };

    /**
     * returns the last Monday before assigned date in UTC format
     *
     * @param {Date} date
     *  The original date, as UTC date value.
     *
     * @returns {Date}
     *  The new date, as UTC date value.
     */
    DateUtils.getWeekStart = function (date) {
        return DateUtils.addDaysToDate(date, -DateUtils.getWeekDay(date));
    };

    // exports ================================================================

    return DateUtils;

});
