/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokenarray', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/utils/sheetref',
    'io.ox/office/spreadsheet/model/formula/utils/cellref',
    'io.ox/office/spreadsheet/model/formula/parser/tokens'
], function (Utils, SheetUtils, FormulaUtils, SheetRef, CellRef, Tokens) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var RangeArray = SheetUtils.RangeArray;
    var Range3DArray = SheetUtils.Range3DArray;
    var SeparatorToken = Tokens.SeparatorToken;
    var ParenthesisToken = Tokens.ParenthesisToken;
    var SheetRefToken = Tokens.SheetRefToken;
    var ReferenceToken = Tokens.ReferenceToken;
    var NameToken = Tokens.NameToken;

    // class TokenArray =======================================================

    /**
     * Represents a formula expression that has been parsed and split into
     * single formula tokens (instances of the class BaseToken). Provides
     * support for parsing formula expressions and handling of the resulting
     * formula token arrays.
     *
     * @constructor
     *
     * @param {SpreadsheetModel|SheetModel} parentModel
     *  The spreadsheet document model, or a specific sheet model containing
     *  this instance. The parent model determines the default reference sheet
     *  index used by various methods of this class.
     *
     * @param {String} tokenType
     *  The type of the token array. May cause special behavior in specific
     *  situations. The following types are supported:
     *  - 'cell': The token array represents a cell formula. This implies
     *      special handling for circular references when interpreting the
     *      formula.
     *  - 'name': The token array represents the definition of a defined name.
     *      Reference tokens that will be located outside the sheet when
     *      relocating them will be wrapped at the sheet borders.
     *  - 'rule': The token array represents the condition value of a rule in a
     *      conditional formatting range.
     *  - 'validation': The token array represents the condition value of a
     *      data validation range.
     *  - 'chart': The token array represents the source link of a data series
     *      in a chart object.
     *  - 'custom': A custom token array without special behavior.
     */
    function TokenArray(parentModel, tokenType) {

        // the document model
        this._docModel = parentModel.getDocModel();
        // the sheet model of sheet-local names
        this._sheetModel = (parentModel === this._docModel) ? null : parentModel;
        // all callback functions to be invokes when this token array changes
        this._changeHandlers = null;

        // the array of all formula tokens
        this._allTokens = [];
        // the formula tokens with sheet references
        this._sheetTokens = [];
        // the formula reference tokens
        this._refTokens = [];
        // the formula tokens with defined names
        this._nameTokens = [];
        // flag set whether this token array contains tokens of specific types
        this._tokenTypeSet = {};

        // whether to wrap relocated range addresses at the sheet borders
        this._wrapReferences = tokenType === 'name';
        // whether to keep relative references unmodified when resolving moved cells
        this._freezeRelRefs = tokenType === 'name';
        // whether to detect circular references when interpreting the formula
        this._detectCircular = tokenType === 'cell';
        // whether to omit the own table name of structured table references inside that table
        this._unqualifiedTables = tokenType === 'cell';

        // cached compiler token tree
        this._compileDesc = null;
        // whether this token array is currently interpreted, to prevent recursion
        this._interpreting = false;

    } // class TokenArray

    // private methods --------------------------------------------------------

    /**
     * Adds a new token inserted into this token array in the various internal
     * data structurs used for caching and to optimize performance of various
     * public methods.
     *
     * @param {BaseToken} token
     *  The token to be registered in the internal data structures.
     */
    TokenArray.prototype._registerToken = function (token) {
        this._tokenTypeSet[token.getType()] = true;
        if (token instanceof SheetRefToken) { this._sheetTokens.push(token); }
        if (token instanceof ReferenceToken) { this._refTokens.push(token); }
        if (token instanceof NameToken) { this._nameTokens.push(token); }
    };

    /**
     * Invokes all registered change callback handlers.
     */
    TokenArray.prototype._invokeChangeHandlers = function (token, index) {
        if (!this._changeHandlers) { return; }
        this._changeHandlers.forEach(function (changeHandler) {
            changeHandler.call(this, token, index);
        }, this);
    };

    /**
     * Resolves a new text representation of this token array according to the
     * specified callback function.
     *
     * @param {String} grammarId
     *  The identifier of the formula grammar to be used to generate the
     *  formula expressions. See class FormulaGrammar for details.
     *
     * @param {Function} callback
     *  A callback function that receives a token from this token array, and
     *  the formula grammar for the passed grammar identifier, and must return
     *  its new string representation; or null to indicate that the token does
     *  not change.
     *
     * @returns {Object|Null}
     *  A descriptor object with the old and new formula expression, if they
     *  differ, otherwise null. The descriptor object contains the following
     *  properties:
     *  - {String} old
     *      The current text representation of this token array.
     *  - {String} new
     *      The transformed text representation of this token array.
     */
    TokenArray.prototype._resolveChangedFormula = function (grammarId, callback) {

        // the formula grammar expected by the formula tokens
        var formulaGrammar = this._docModel.getFormulaGrammar(grammarId);
        // the current string representation of the formula
        var oldFormula = '';
        // the new string representation of the formula with moved cell references
        var newFormula = '';

        // collect the old and new text representation of the formula (no early exit!)
        this._allTokens.forEach(function (token) {
            var oldText = token.getText(formulaGrammar);
            var newText = callback.call(this, token, formulaGrammar);
            oldFormula += oldText;
            newFormula += (newText === null) ? oldText : newText;
        }, this);

        return (oldFormula === newFormula) ? null : { old: oldFormula, new: newFormula };
    };

    /**
     * Returns the index of the sheet that contains this token array. If this
     * token array is owned by the document itself, the value of the option
     * 'refSheet' will be returned instead.
     *
     * @returns {Number|Null}
     *  The index of the sheet that contains this token array, if such a sheet
     *  has been passed as parent model to the constructor; or the value of the
     *  option 'refSheet', if this is a global token array with the document
     *  model as parent.
     */
    TokenArray.prototype._resolveRefSheet = function (options) {
        return this._sheetModel ? this._sheetModel.getIndex() : Utils.getIntegerOption(options, 'refSheet', null);
    };

    /**
     * Extends the passed options received by various public methods of
     * this instance, according to the type of the token array. Adds the
     * boolean options 'wrapReferences', and 'detectCircular', and inserts
     * the own reference sheet index as option 'refSheet' if available.
     */
    TokenArray.prototype._extendWithOwnOptions = function (options) {
        return _.extend({}, options, {
            refSheet: this._resolveRefSheet(options),
            wrapReferences: this._wrapReferences,
            detectCircular: this._detectCircular,
            unqualifiedTables: this._unqualifiedTables
        });
    };

    /**
     * Compiles the passed formula tokens on-the-fly, without modifying the
     * state of this instance. See class FormulaCompiler for details.
     *
     * @returns {Object}
     *  The result descriptor of the compilation process. See description of
     *  the method FormulaCompiler.compileTokens() for details.
     */
    TokenArray.prototype._compileTokens = function (tokens) {
        return this._docModel.getFormulaCompiler().compileTokens(tokens);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the parent sheet model of this token array.
     *
     * @returns {SheetModel|Null}
     *  The parent sheet model of this token array; or null for a global token
     *  array owned by the document itself.
     */
    TokenArray.prototype.getSheetModel = function () {
        return this._sheetModel;
    };

    /**
     * Returns whether this token array is empty.
     *
     * @returns {Boolean}
     *  Whether this token array is empty.
     */
    TokenArray.prototype.empty = function () {
        return this._allTokens.length === 0;
    };

    /**
     * Returns whether this token array contains at least one token of the
     * specified type.
     *
     * @param {String} type
     *  The type of the token to be checked.
     *
     * @returns {Boolean}
     *  Whether this token array contains at least one token of the specified
     *  type.
     */
    TokenArray.prototype.hasToken = function (type) {
        return type in this._tokenTypeSet;
    };

    /**
     * Returns whether this token array contains any reference tokens, or any
     * tokens referring to defined names or table ranges.
     *
     * @returns {Boolean}
     *  Whether this token array contains any reference tokens, or any tokens
     *  referring to defined names or table ranges.
     */
    TokenArray.prototype.hasRefTokens = function () {
        return this.hasToken('ref') || this.hasToken('name') || this.hasToken('table');
    };

    /**
     * Returns the text representation of the formula represented by this token
     * array.
     *
     * @param {String} grammarId
     *  The identifier of the formula grammar to be used to generate the
     *  formula expression. See class FormulaGrammar for details.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Address} [options.refAddress]
     *      The address of the original reference cell the formula is related
     *      to. If omitted, cell A1 will be used instead. If this address is
     *      different to the address resulting from the option 'targetAddress',
     *      the generated formula expression will be modified, as if the token
     *      array has been relocated according to these two addresses (without
     *      actually modifying the token array).
     *  @param {Address} [options.targetAddress]
     *      The address of the target cell for relocation. If omitted, cell A1
     *      will be used instead. If this address is different to the address
     *      resulting from the option 'refAddress', the generated formula
     *      expression will be modified, as if the token array has been
     *      relocated according to these two addresses (without actually
     *      modifying the token array).
     *
     * @returns {String}
     *  The string representation of the formula.
     */
    TokenArray.prototype.getFormula = function (grammarId, options) {

        // the formula grammar expected by the formula tokens
        var formulaGrammar = this._docModel.getFormulaGrammar(grammarId);
        // prepare the options for reference relocation
        var newOptions = this._extendWithOwnOptions(options);

        // concatenate the string representations of all formula tokens
        return this._allTokens.reduce(function (formula, token) {
            return formula + token.getText(formulaGrammar, newOptions);
        }, '');
    };

    /**
     * Resolves the token array to a list of constant strings. The formula
     * represented by this token array must consist of one or more literal
     * string tokens, separated by list operators. Whitespace tokens will be
     * ignored.
     *
     * @returns {Array<String>|Null}
     *  The string literals contained in this token array. If the formula
     *  cannot be resolved successfully to a list of strings, this method
     *  returns null.
     */
    TokenArray.prototype.resolveStringList = function () {

        // the resulting strings
        var result = [];
        // a string with a single character reflecting the token types in the token array
        var tokenTypes = '';

        // process all tokens as long as they are considered valid, collect token types
        this._allTokens.every(function (token) {
            switch (token.getType()) {
                case 'lit':
                    tokenTypes += 'L';
                    result.push(token.getValue());
                    return _.isString(_.last(result));
                case 'sep':
                    tokenTypes += ',';
                    return true;
                case 'ws':
                    return true;
                default:
                    tokenTypes += 'I'; // invalid token
                    return false;
            }
        });

        // on success, check the formula structure
        return (/^L(,L)*$/).test(tokenTypes) ? result : null;
    };

    /**
     * Resolves the token array to a list of range addresses. The formula
     * represented by this token array must be completely resolvable to a range
     * list (single reference, a list of references separated by the list
     * operator, optionally enclosed in parentheses, defined names that resolve
     * to a range list if specified with the option 'resolveNames', also
     * recursively, or a mixed list of defined names and references). To
     * extract the range addresses contained in an arbitrary formula, the
     * method TokenArray.extractRanges() can be used.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.refSheet]
     *      The index of the original reference sheet that will be used if this
     *      token array is related to the document (does not have its own
     *      reference sheet).
     *  @param {Address} [options.refAddress]
     *      The source reference address used to relocate reference tokens with
     *      relative column/row components. If omitted, cell A1 will be used as
     *      source reference cell.
     *  @param {Address} [options.targetAddress]
     *      The target reference address used to relocate reference tokens with
     *      relative column/row components. If omitted, cell A1 will be used as
     *      target reference cell.
     *  @param {Boolean} [options.resolveNames=false]
     *      If set to true, will return the range addresses referred by all
     *      defined names contained in this token array.
     *
     * @returns {Range3DArray}
     *  The addresses of the resulting cell ranges, including their sheet
     *  indexes. If the formula cannot be resolved successfully to a range
     *  list, this method returns an empty array.
     */
    TokenArray.prototype.resolveRangeList = function (options) {

        // the resulting range addresses
        var resultRanges = new Range3DArray();

        // prevent recursive processing of the token array, e.g. in defined name
        if (this._interpreting) { return resultRanges; }

        // whether to resolve defined name tokens
        var resolveNames = Utils.getBooleanOption(options, 'resolveNames', false);
        // early exit, if this token array does not contain any reference or name tokens
        var hasTokens = resolveNames ? this.hasRefTokens() : this.hasToken('ref');
        if (!hasTokens) { return resultRanges; }

        // the target address (needed to resolve addresses in table ranges)
        var targetAddress = Utils.getOption(options, 'targetAddress', Address.A1);

        // add more options (refSheet and wrapReferences)
        options = this._extendWithOwnOptions(options);

        // process all tokens as long as they are considered valid, build a signature
        // string with a single character for each token to check the formula syntax
        var tokenTypes = '';
        var valid = this._allTokens.every(function (token) {

            switch (token.getType()) {

                case 'ref':
                    // the cell range address, will be null for invalid ranges (#REF! errors)
                    var range = token.getRange3D(options);
                    // check if the token contains a valid range (formula must not contain multi-sheet references)
                    if (!range || !range.singleSheet()) { return false; }

                    // append the valid range to the result
                    resultRanges.push(range);
                    tokenTypes += 'R';
                    return true;

                case 'name':
                    // the model of the defined name referred by the passed token
                    var nameModel = resolveNames ? token.resolveNameModel(options.refSheet) : null;
                    if (!nameModel) { return false; }

                    // defined names are always defined relative to cell A1
                    var newOptions = _.clone(options);
                    newOptions.refAddress = Address.A1;

                    // prevent recursive invocation of this token array via other defined names
                    this._interpreting = true;
                    var ranges = nameModel.getTokenArray().resolveRangeList(newOptions);
                    this._interpreting = false;
                    if (ranges.empty()) { return false; }

                    // append the new ranges to the result
                    resultRanges.append(ranges);
                    tokenTypes += 'R';
                    return true;

                case 'table':
                    // try to resolve the table reference to a valid cell range address
                    var tableRange = resolveNames ? token.getRange3D(targetAddress) : null;
                    if (!tableRange) { return false; }

                    // append the valid range to the result
                    resultRanges.push(tableRange);
                    tokenTypes += 'R';
                    return true;

                case 'open':
                    tokenTypes += '<';
                    return true;

                case 'close':
                    tokenTypes += '>';
                    return true;

                case 'sep':
                    tokenTypes += ',';
                    return true;

                case 'ws':
                    return true;

                default:
                    return false;
            }
        }, this);

        // reset resulting ranges if formula structure is invalid
        return (valid && /^(R(,R)*|<R(,R)*>)$/.test(tokenTypes)) ? resultRanges : resultRanges.clear();
    };

    /**
     * Returns the addresses of all cell ranges this token array refers to.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.refSheet]
     *      The index of the original reference sheet that will be used if this
     *      token array is related to the document (does not have its own
     *      reference sheet).
     *  @param {Address} [options.refAddress]
     *      The source reference address used to relocate reference tokens with
     *      relative column/row components. If omitted, cell A1 will be used as
     *      source reference cell.
     *  @param {Address} [options.targetAddress]
     *      The target reference address used to relocate reference tokens with
     *      relative column/row components. If omitted, cell A1 will be used as
     *      target reference cell.
     *  @param {Boolean} [options.resolveNames=false]
     *      If set to true, will return the range addresses referred by all
     *      defined names and table ranges contained in this token array.
     *
     * @returns {Array<Object>}
     *  The addresses of all cell ranges contained in this token array with
     *  additional information, in an array of descriptor objects with the
     *  following properties:
     *  - {Range3D} range
     *      The cell range address represented by a formula token, with sheet
     *      indexes.
     *  - {Number} index
     *      The internal array index of the token containing the range (may be
     *      equal for different objects in the array, for example for a single
     *      name token referring to multiple cell ranges).
     *  - {String} type
     *      The type of the token containing the range. Will be either 'ref'
     *      for a regular reference token, or 'name' for a defined name that
     *      evaluates to one or more ranges in the specified sheet (only if
     *      option 'resolveNames' is set, see above).
     *  The entire array object containing the range descriptors will contain
     *  the following additional properties:
     *  - {Boolean} circular
     *      Whether trying to resolve a defined name resulted in a circular
     *      reference error (only if option 'resolveNames' is set, see above).
     */
    TokenArray.prototype.extractRanges = function (options) {

        // descriptors for the resulting ranges, with the additional 'circular' property
        var rangeInfos = [];
        rangeInfos.circular = false;

        // prevent recursive processing of the token array in defined names
        if (this._interpreting) {
            rangeInfos.circular = true;
            return rangeInfos;
        }

        // whether to resolve defined name tokens
        var resolveNames = Utils.getBooleanOption(options, 'resolveNames', false);
        // early exit, if this token array does not contain any reference or name tokens
        var hasTokens = resolveNames ? this.hasRefTokens() : this.hasToken('ref');
        if (!hasTokens) { return rangeInfos; }

        // the target address (needed to resolve addresses in table ranges)
        var targetAddress = Utils.getOption(options, 'targetAddress', Address.A1);

        // add more options (refSheet and wrapReferences)
        options = this._extendWithOwnOptions(options);

        // process all reference tokens and name tokens in this token array
        this._allTokens.forEach(function (token, index) {
            switch (token.getType()) {

                case 'ref':
                    var range = token.getRange3D(options); // returns null for invalid ranges (#REF! errors)
                    if (range) { rangeInfos.push({ range: range, index: index, type: 'ref' }); }
                    break;

                case 'name':
                    // the model of the defined name referred by the token
                    var nameModel = resolveNames ? token.resolveNameModel(options.refSheet) : null;
                    if (!nameModel) { return; }

                    // defined names are always defined relative to cell A1
                    var newOptions = _.clone(options);
                    newOptions.refAddress = Address.A1;

                    // prevent recursive invocation of this token array via other defined names
                    this._interpreting = true;
                    var nameRangeInfos = nameModel.getTokenArray().extractRanges(newOptions);
                    this._interpreting = false;

                    // insert the range descriptors with adjusted properties
                    nameRangeInfos.forEach(function (rangeInfo) {
                        rangeInfos.push({ range: rangeInfo.range, index: index, type: 'name' });
                    });

                    // update the own circular reference flag
                    rangeInfos.circular = rangeInfos.circular || nameRangeInfos.circular;
                    break;

                case 'table':
                    var tableRange = resolveNames ? token.getRange3D(targetAddress) : null;
                    if (tableRange) { rangeInfos.push({ range: tableRange, index: index, type: 'table' }); }
                    break;
            }
        }, this);

        return rangeInfos;
    };

    // operation generators ---------------------------------------------------

    /**
     * Resolves the new text representation of this token array, while changing
     * something the spreadsheet document.
     *
     * @param {String} grammarId
     *  The identifier of the formula grammar to be used to generate the
     *  formula expressions. See class FormulaGrammar for details.
     *
     * @param {Object} changeDesc
     *  The properties describing the document change. The properties that are
     *  expected in this descriptor depend on the change type in its 'type'
     *  property.
     *  - {String} changeDesc.type
     *      The type of the document operation that causes to update the
     *      formula expression. MUST be one of the following values:
     *      - 'deleteSheet': A sheet has been deleted in the document. All
     *          references containing the deleted sheet become invalid.
     *      - 'copySheet': A sheet has been cloned in the document. All
     *          references containing the source sheet will be updated to refer
     *          to the new sheet.
     *      - 'renameSheet': A sheet has been renamed in the document. All
     *          references containing the renamed sheet will be updated.
     *      - 'relabelName': The label of a defined name has been changed. The
     *          formula expressions will be updated to use the new label.
     *      - 'moveCells': Cells have been moved in a sheet. This includes
     *          inserting or deleting some columns or rows. All affected cell
     *          references will be updated.
     *  - {Number|Null} changeDesc.sheet
     *      The zero-based index of the sheet that will be changed. This
     *      property is used by all document operation types: The index of the
     *      deleted or renamed sheet; the sheet index of the relabeled defined
     *      name (null for global names), or the index of the sheet containing
     *      the moved cells.
     *  - {String} [changeDesc.sheetName]
     *      The new name of a renamed sheet. Used for change type 'renameSheet'
     *      only.
     *  - {String} [changeDesc.oldLabel]
     *      The old label of the relabeled defined name. Used for change type
     *      'relabelName' only.
     *  - {String} [changeDesc.newLabel]
     *      The new label of the relabeled defined name. Used for change type
     *      'relabelName' only.
     *  - {Array<MoveDescriptor>} [changeDesc.moveDescs]
     *      An array of move descriptors that specify how to transform the
     *      cell references after cells have been moved in a sheet. Used for
     *      change type 'moveCells' only.
     *
     * @returns {Object|Null}
     *  A descriptor object with the old and new formula expression, if they
     *  differ, otherwise null. The descriptor object contains the following
     *  properties:
     *  - {String} old
     *      The current text representation of this token array.
     *  - {String} new
     *      The transformed text representation of this token array.
     *  - {Boolean} error
     *      Whether the a cell reference has been replaced with a #REF! error.
     *      This property will be inserted for change type 'moveCells' only.
     */
    TokenArray.prototype.resolveOperation = function (grammarId, changeDesc) {

        // the effective reference sheet of this token array
        var refSheet = this._resolveRefSheet();

        switch (changeDesc.type) {

            case 'deleteSheet':
                // if the formula does not contain any sheet or table references, nothing will change
                if ((this._sheetTokens.length === 0) && !this.hasToken('table')) { return null; }

                // collect the old and new text representation of the formula
                return this._resolveChangedFormula(grammarId, function (token, config) {
                    return token.resolveDeletedSheet(config, changeDesc.sheet, refSheet);
                });

            case 'renameSheet':
                // if the formula does not contain any sheet or table references, nothing will change
                if ((this._sheetTokens.length === 0) && !this.hasToken('table')) { return null; }

                // collect the old and new text representation of the formula
                return this._resolveChangedFormula(grammarId, function (token, config) {
                    return token.resolveRenamedSheet(config, changeDesc.sheet, changeDesc.sheetName, changeDesc.tableNames);
                });

            case 'relabelName':
                // if the formula does not contain any name tokens, nothing will change
                if (this._nameTokens.length === 0) { return null; }

                // collect the old and new text representation of the formula
                return this._resolveChangedFormula(grammarId, function (token, config) {
                    return token.resolveRelabeledName(config, changeDesc.sheet, changeDesc.oldLabel, changeDesc.newLabel, refSheet);
                });

            case 'moveCells':
                // if the formula does not contain any reference tokens, nothing will change
                if (!this.hasToken('ref') && !this.hasToken('table')) { return null; }

                // create options for the method BaseToken.resolveMovedCells()
                var options = { refSheet: refSheet, freezeRelRefs: this._freezeRelRefs };
                if (changeDesc.relocate) {
                    options.refAddress = changeDesc.relocate.from;
                    options.targetAddress = changeDesc.relocate.to;
                }

                // collect the old and new text representation of the formula
                var refError = false;
                var moveResult = this._resolveChangedFormula(grammarId, function (token, config) {
                    var result = token.resolveMovedCells(config, changeDesc.sheet, changeDesc.moveDescs, options);
                    if (!result) { return null; }
                    refError = refError || result.refError;
                    return result.text;
                });
                if (moveResult) { moveResult.error = refError; }
                return moveResult;
        }

        Utils.error('TokenArray.resolveOperation(): unknown operation type "' + changeDesc.type + '"');
        return null;
    };

    // token manipulation -----------------------------------------------------

    /**
     * Registers a callback function that will be invoked after this token
     * array has changed by using any of its methods.
     *
     * @param {Function} changeHandler
     *  The callback function. Receives the following parameters:
     *  (1) {Token|Null} token
     *      A reference to the changed token, if a single token has been
     *      changed; or null, if the entire token array has changed.
     *  (2) {Number} [index]
     *      The array index of the token contained in the first parameter.
     *  The callback function will be called in the context of this token
     *  array, and its return value will be ignored.
     *
     * @returns {TokenArray}
     *  A reference to this instance.
     */
    TokenArray.prototype.registerChangeHandler = function (changeHandler) {
        (this._changeHandlers || (this._changeHandlers = [])).push(changeHandler);
        return this;
    };

    /**
     * Unregisters a callback function for change events that has been
     * registered with the method TokenArray.registerChangeHandler() before.
     *
     * @param {Function} changeHandler
     *  The callback function to be unregistered.
     *
     * @returns {TokenArray}
     *  A reference to this instance.
     */
    TokenArray.prototype.unregisterChangeHandler = function (changeHandler) {
        if (this._changeHandlers) {
            this._changeHandlers = _.without(this._changeHandlers, changeHandler);
        }
        return this;
    };

    /**
     * Inserts the passed formula tokens into this token array, and invokes the
     * registered change callback handlers.
     *
     * @param {Array<BaseToken>} tokens
     *  The new formula tokens to be inserted into this token array.
     *
     * @returns {TokenArray}
     *  A reference to this instance.
     */
    TokenArray.prototype.setTokens = function (tokens) {

        // early exit, if an empty token array will be cleared
        if (this.empty() && (tokens.length === 0)) { return this; }

        // create a shallow copy of the passed array
        this._allTokens = tokens.slice();

        // collect all existing token types in a flag set
        this._refTokens = [];
        this._sheetTokens = [];
        this._tokenTypeSet = {};
        this._allTokens.forEach(this._registerToken, this);

        // reset other properties, notify change handlers
        this._compileDesc = null;
        this._invokeChangeHandlers(null);
        return this;
    };

    /**
     * Removes all formula tokens from this token array, and invokes the
     * registered change callback handlers.
     *
     * @returns {TokenArray}
     *  A reference to this instance.
     */
    TokenArray.prototype.clearTokens = function () {
        return this.setTokens([]);
    };

    /**
     * Parses the specified formula expression, creates an array of formula
     * tokens held by this instance, and invokes the registered change callback
     * handlers.
     *
     * @param {String} grammarId
     *  The identifier of the formula grammar to be used to parse the passed
     *  formula expression. See class FormulaGrammar for details.
     *
     * @param {String} formula
     *  The formula expression to be parsed, without the leading equality sign.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options that are supported by the
     *  method FormulaParser.parseFormula().
     *
     * @returns {Array<TokenDescriptor>}
     *  An array of descriptors for all parsed formula tokens, as received from
     *  the method FormulaParser.parseFormula().
     */
    TokenArray.prototype.parseFormula = function (grammarId, formula, options) {

        // add more options specific to this token array
        options = this._extendWithOwnOptions(options);

        // parse the formula string, extract all formula tokens, invoke change handlers
        var formulaParser = this._docModel.getFormulaParser();
        var tokenDescs = formulaParser.parseFormula(grammarId, formula, options);
        var tokens = _.pluck(tokenDescs, 'token');

        // add missing tokens that can be auto-corrected
        if (Utils.getBooleanOption(options, 'autoCorrect', false)) {
            var compileDesc = this._compileTokens(tokens);
            if (!compileDesc.error) {
                for (var index = 0; index < compileDesc.missingClose; index += 1) {
                    tokens.push(new ParenthesisToken(false));
                }
            }
        }

        // set the resulting tokens to this instance
        this.setTokens(tokens);

        // return the complete token descriptors
        return tokenDescs;
    };

    /**
     * Tries to replace unresolved sheet names with existing sheet indexes in
     * the spreadsheet document, and to reparse all formulas with bad tokens.
     * Intended to be used after document import to refresh all token arrays
     * that refer to sheets that did not exist during their creation.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options that are supported by the
     *  method FormulaParser.parseFormula().
     *
     * @returns {TokenArray}
     *  A reference to this instance.
     */
    TokenArray.prototype.refreshAfterImport = function (options) {

        // reparse formula if it contains a bad token, otherwise refresh all sheet tokens
        if (this.hasToken('bad'))  {
            this.parseFormula('op', this.getFormula('op'), options);
        } else {
            this._sheetTokens.forEach(function (token) { token.refreshSheets(); });
            this._compileDesc = null;
        }

        return this;
    };

    /**
     * Invokes a callback function used to modify the specified token in this
     * token array. Afterwards, the registered change callback handlers will be
     * invoked with the changed token.
     *
     * @param {Number} index
     *  The array index of the token to be modified.
     *
     * @param {String} type
     *  A string with a single type identifier that must match the type of the
     *  token. If the token at the passed index does not match the type, the
     *  callback function will not be invoked, and the change callback handlers
     *  will not be invoked.
     *
     * @param {Function} callback
     *  The callback function invoked for the specified token. Receives the
     *  following parameters:
     *  (1) {Token} token
     *      The token instance.
     *  (2) {Number} index
     *      The token index, as passed to the method modifyToken().
     *  Must return a boolean value specifying whether the token has been
     *  modified. If the callback returns true, the registered change callback
     *  handlers will be invoked afterwards.
     *
     * @returns {TokenArray}
     *  A reference to this instance.
     */
    TokenArray.prototype.modifyToken = function (index, type, callback, context) {

        // the token to be modified
        var token = this._allTokens[index];

        // invoke the callback, notify changed token if callback returns true
        if (token && token.isType(type) && callback.call(context, token, index)) {
            this._invokeChangeHandlers(token, index);
        }

        return this;
    };

    /**
     * Appends the passed formula token to the token array, and invokes the
     * registered change callback handlers for the new token.
     *
     * @param {Token} token
     *  The new formula token. This token array will take ownership.
     *
     * @returns {TokenArray}
     *  A reference to this instance.
     */
    TokenArray.prototype.appendToken = function (token) {
        this._allTokens.push(token);
        this._registerToken(token);
        this._compileDesc = null;
        this._invokeChangeHandlers(token, this._allTokens.length - 1);
        return this;
    };

    /**
     * Appends a new reference token to this token array, that will be built
     * from the passed cell range address, and invokes the registered change
     * callback handlers for the new token.
     *
     * @param {Range} range
     *  The address of the cell range to be added as reference token to this
     *  token array.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.abs=false]
     *      If set to true, all column and row references will be marked as
     *      absolute (leading dollar signs).
     *  @param {Number} [options.sheet]
     *      The zero-based index of the sheet the reference token will point
     *      to. If omitted, no sheet reference will be inserted into the
     *      reference token (sheet-local reference).
     *
     * @returns {TokenArray}
     *  A reference to this instance.
     */
    TokenArray.prototype.appendRange = function (range, options) {

        // whether the range spans entire columns or rows
        var colRange = this._docModel.isColRange(range);
        var rowRange = this._docModel.isRowRange(range);

        // the cell references
        var abs = Utils.getBooleanOption(options, 'abs', false);
        var cell1Ref = CellRef.create(range.start, abs || rowRange, abs || (colRange && !rowRange));
        var cell2Ref = range.single() ? null : CellRef.create(range.end, abs || rowRange, abs || (colRange && !rowRange));

        // the sheet reference
        var sheet = Utils.getIntegerOption(options, 'sheet');
        var sheetRef = (typeof sheet === 'number') ? new SheetRef(sheet, true) : null;

        return this.appendToken(new ReferenceToken(this._docModel, cell1Ref, cell2Ref, sheetRef));
    };

    /**
     * Appends new reference tokens to this token array, that will be built
     * from the passed cell range addresses. If the passed range array contains
     * more than one range address, the reference tokens will be separated by
     * list operator tokens, and invokes the registered change callback
     * handlers for the new tokens.
     *
     * @param {RangeArray} ranges
     *  The addresses of the cell ranges to be added as reference tokens to
     *  this token array.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.abs=false]
     *      If set to true, all column and row references will be marked as
     *      absolute (leading dollar signs).
     *  @param {Number} [options.sheet]
     *      The zero-based index of the sheet the reference tokens will point
     *      to. If omitted, no sheet references will be inserted into the
     *      reference tokens (sheet-local references).
     *  @param {Boolean} [options.parentheses=false]
     *      If set to true, and the passed array contains more than one range,
     *      the entire range list will be enclosed into parentheses.
     *
     * @returns {TokenArray}
     *  A reference to this instance.
     */
    TokenArray.prototype.appendRangeList = function (ranges, options) {

        // whether to enclose the range list into parentheses
        var parentheses = (ranges.length > 1) && Utils.getBooleanOption(options, 'parentheses', false);

        if (parentheses) {
            this.appendToken(new ParenthesisToken(true));
        }

        ranges.forEach(function (range, index) {
            if (index > 0) { this.appendToken(new SeparatorToken()); }
            this.appendRange(range, options);
        }, this);

        if (parentheses) {
            this.appendToken(new ParenthesisToken(false));
        }

        return this;
    };

    /**
     * Relocates the cell ranges in this token array to a new reference address
     * by adjusting the relative column and row components in all reference
     * tokens, and invokes the registered change callback handlers.
     *
     * @param {Address} refAddress
     *  The address of the original reference cell the formula is related to.
     *
     * @param {Address} targetAddress
     *  The address of the new reference cell the formula will be related to.
     *
     * @returns {Boolean}
     *  Whether any token in this token array has been changed.
     */
    TokenArray.prototype.relocateRanges = function (refAddress, targetAddress) {

        // nothing to do, if the addresses are equal
        if (refAddress.equals(targetAddress)) { return false; }

        // whether any reference token has been changed
        var changed = false;

        // relocate all reference tokens, and collect changed state (no early exit!)
        this._refTokens.forEach(function (token) {
            if (token.relocateRange(refAddress, targetAddress, this._wrapReferences)) { changed = true; }
        }, this);

        // invoke all change handlers
        if (changed) { this._invokeChangeHandlers(null); }
        return changed;
    };

    /**
     * Updates all tokens with sheet references, and invokes the  registered
     * change callback handlers, after a sheet has been inserted into, deleted
     * from, or moved in the document.
     *
     * @param {Number|Null} toSheet
     *  The zero-based index of the inserted sheet, or the new position of a
     *  moved sheet, or null for a deleted sheet.
     *
     * @param {Number|Null} fromSheet
     *  The zero-based index of the deleted sheet, or the old position of a
     *  moved sheet, or null for an inserted sheet.
     *
     * @returns {Boolean}
     *  Whether any token in this token array has been changed.
     */
    TokenArray.prototype.transformSheet = function (toSheet, fromSheet) {

        // nothing to do, if the sheet indexes are equal
        if (toSheet === fromSheet) { return false; }

        // whether any sheet token has been refreshed
        var changed = false;

        // relocate all sheet tokens, and collect changed state (no early exit!)
        this._sheetTokens.forEach(function (token) {
            if (token.transformSheet(toSheet, fromSheet)) { changed = true; }
        }, this);

        // invoke all change handlers
        if (changed) { this._invokeChangeHandlers(null); }
        return changed;
    };

    // formula calculation ----------------------------------------------------

    /**
     * Compiles the formula tokens contained in this token array to a token
     * tree structure. See class FormulaCompiler for details.
     *
     * @returns {Object}
     *  The result descriptor of the compilation process. See description of
     *  the method FormulaCompiler.compileTokens() for details.
     */
    TokenArray.prototype.compileFormula = function () {
        return this._compileDesc || (this._compileDesc = this._compileTokens(this._allTokens));
    };

    /**
     * Returns the dependencies of the formula represented by this token array.
     *
     * @param {Address} refAddress
     *  The source reference address used to relocate reference tokens with
     *  relative column/row components.
     *
     * @param {Address|Range|RangeArray} depTarget
     *  The position of the target cells this formula is associated with. Can
     *  be a single cell address (e.g. for cell formulas), or a single cell
     *  range address, or an array of cell range addresses (e.g. the target
     *  ranges of a formatting rule).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.refSheet]
     *      The index of the original reference sheet that will be used if this
     *      token array is related to the document (does not have its own
     *      reference sheet).
     *
     * @returns {Object}
     *  A descriptor object with the dependency settings, with the following
     *  properties:
     *  - {Range3DArray} depRanges
     *      The addresses of all cell ranges the formula depends on, with sheet
     *      indexes, according to the reference address and target address
     *      passed to this method.
     *  - {Range3DArray} rootRanges
     *      The addresses of all cell ranges the root node of the compiler tree
     *      depends on. If this token array is associated to a defined name,
     *      these ranges will be forwarded to the name token of the parent
     *      formula expression, in order to decide how to proceed with these
     *      ranges.
     *  - {Object} names
     *      A map with the models of all defined names, mapped by their UIDs.
     *  - {Object} missingNames
     *      A flag set with the map keys of all missing defined names (as
     *      object keys) occurring in the formula expression, or in any
     *      existing defined name referred by the formula.
     *  - {String|Null} recalc
     *      The recalculation mode: Either 'always' or 'once', if at least one
     *      function in the formula contains the respective recalculation mode
     *      (either directly, or indirectly from a defined name; 'always' will
     *      be preferred over 'once'); otherwise null.
     *  - {Boolean} circular
     *      Whether trying to resolve the defined names referred by this token
     *      array resulted in a circular reference error.
     */
    TokenArray.prototype.getDependencies = function (refAddress, depTarget, options) {

        // the result descriptor object
        var deps = { depRanges: new Range3DArray(), rootRanges: new Range3DArray(), names: {}, missingNames: {}, recalc: null, circular: false };

        // compile the infix tokens to a token tree (early exit, if this token array is invalid)
        var compileDesc = this.compileFormula();
        if (compileDesc.error) { return deps; }

        // prevent recursive processing of the token array, e.g. in defined name
        if (this._interpreting) { deps.circular = true; return deps; }

        // insert the recalculation mode of the root node into the result
        deps.recalc = compileDesc.root.recalc;

        // add more options (refSheet and wrapReferences)
        options = this._extendWithOwnOptions(options);

        // a single target address for cell formulas (and others, e.g. formatting rules in range mode)
        var targetAddress = (depTarget instanceof Address) ? depTarget : null;
        var addressOptions = targetAddress ? _.extend({}, options, { refAddress: refAddress, targetAddress: targetAddress }) : null;

        // array of target ranges for other formulas, e.g. formatting rules
        var targetRanges = targetAddress ? null : RangeArray.get(depTarget);

        // traverse the compiler tree, collect references from the formula, and from referred defined names
        var rootRanges = (function collectReferences(node) {

            // the formula token represented by the compiler node
            var token = node.token;

            // reference token: return the resolved range as token references
            if (token.isType('ref')) {
                if (addressOptions) {
                    var range3d = token.getRange3D(addressOptions); // returns null for invalid references (#REF! errors)
                    return range3d ? new Range3DArray(range3d) : null;
                }
                return token.getExpandedRanges(refAddress, targetRanges, options);
            }

            // name token: resolve the source references of the defined name
            if (token.isType('name')) {

                // the model of the defined name referred by the token
                var nameModel = token.resolveNameModel(options.refSheet);

                // if the token cannot be resolved to a name, register the label used in the formula
                if (!nameModel) {
                    deps.missingNames[SheetUtils.getNameKey(token.getValue())] = true;
                    return null;
                }

                // prevent recursive invocation of this token array via other defined names
                this._interpreting = true;
                // defined names will always be evaluated relative to cell A1
                var nameDeps = nameModel.getTokenArray().getDependencies(Address.A1, depTarget, options);
                this._interpreting = false;

                // add all dependencies of the defined name to the own result
                deps.depRanges.append(nameDeps.depRanges);
                deps.names[nameModel.getUid()] = nameModel;
                _.extend(deps.names, nameDeps.names);
                _.extend(deps.missingNames, nameDeps.missingNames);
                deps.recalc = FormulaUtils.getRecalcMode(deps.recalc, nameDeps.recalc);
                deps.circular = deps.circular || nameDeps.circular;

                // return the root ranges of the defined name to caller (parent operator)
                return nameDeps.rootRanges;
            }

            // table token: resolve the target range inside the table
            if (token.isType('table')) {
                var tableRange = token.getRange3D(targetAddress);
                return tableRange ? new Range3DArray(tableRange) : null;
            }

            // operators or supported functions: resolve referencees of the operands
            if (node.operands && node.signature) {
                var rootRanges = new Range3DArray();
                node.signature.forEach(function (paramSig, index) {
                    var operandRanges = collectReferences.call(this, node.operands[index]);
                    switch (paramSig.depSpec) {
                        case 'pass':
                            rootRanges.append(operandRanges);
                            break;
                        case 'skip':
                            break;
                        default:
                            deps.depRanges.append(operandRanges);
                            break;
                    }
                }, this);
                return rootRanges;
            }

            return null;
        }.call(this, compileDesc.root));

        // add the root ranges of the root node to the result
        deps.rootRanges.append(rootRanges);

        return deps;
    };

    /**
     * Calculates the result of the formula represented by this token array.
     *
     * @param {String} contextType
     *  The formula context type influencing the type of the formula result.
     *  See method FormulaInterpreter.interpretTokens() for details.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.refSheet]
     *      The index of the original reference sheet that will be used if this
     *      token array is related to the document (does not have its own
     *      reference sheet). Used to resolve reference tokens and defined
     *      names without explicit sheet reference.
     *  @param {Address} [options.refAddress]
     *      The source reference address used to relocate reference tokens with
     *      relative column/row components. If omitted, cell A1 will be used as
     *      source reference cell.
     *  @param {Address} [options.targetAddress]
     *      The target reference address used to relocate reference tokens with
     *      relative column/row components. If omitted, cell A1 will be used as
     *      target reference cell.
     *  @param {Address} [options.recalcDirty=false]
     *      If set to true, all dirty cell formulas referred by the reference
     *      tokens of this token array will be recalculated recursively using
     *      the dependency manager of the document.
     *
     * @returns {Object}
     *  The result descriptor for the formula. See description of the method
     *  FormulaInterpreter.interpretTokens() for details.
     */
    TokenArray.prototype.interpretFormula = function (contextType, options) {

        // compile the infix tokens to a token tree
        var compileDesc = this.compileFormula();
        // the formula interpreter
        var formulaInterpreter = this._docModel.getFormulaInterpreter();

        // check compiler result
        if (compileDesc.error) {
            return formulaInterpreter.createErrorResult(compileDesc.error);
        }

        // prevent recursive interpretation
        if (this._interpreting) {
            return formulaInterpreter.createResult(FormulaUtils.CIRCULAR_ERROR);
        }

        // interpret the formula with extended options
        try {
            this._interpreting = true;
            return formulaInterpreter.interpretTokens(contextType, compileDesc.root, this._extendWithOwnOptions(options));
        } finally {
            this._interpreting = false;
        }
    };

    // exports ================================================================

    return TokenArray;

});
