/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/spreadsheet/model/formula/parser/formularesource', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/parser/desccollection',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, BaseObject, LocaleData, ErrorCode, FormulaUtils, DescriptorCollection, gt) {

    'use strict';

    // internal implementations of all operators
    var OPERATOR_IMPL_MAP = null;

    // internal implementations of all functions
    var FUNCTION_IMPL_MAP = null;

    // raw formula resource data loaded from JSON for current UI language
    var RAW_UI_RESOURCE_DATA = null;

    // raw formula resource data loaded from JSON for English (independent from UI language)
    var RAW_EN_RESOURCE_DATA = null;

    // the native prefix characters for R1C1 reference notation
    var NATIVE_R1C1_PREFIX_CHARS = 'RC';

    // maps the keys of error codes to their native names used in file format
    var NATIVE_ERROR_NAMES = {
        NULL: '#NULL!',
        DIV0: '#DIV/0!',
        VALUE: '#VALUE!',
        REF: '#REF!',
        NAME: '#NAME?',
        NUM: '#NUM!',
        NA: '#N/A',
        DATA: '#GETTING_DATA'
    };

    // maps the keys of table regions to their native names used in file format
    var NATIVE_REGION_NAMES = {
        ALL: '#All',
        HEADERS: '#Headers',
        DATA: '#Data',
        TOTALS: '#Totals',
        ROW: '#This Row'
    };

    // maps the keys of named parameters of CELL to their English names (supported in all locales)
    var NATIVE_CELL_PARAM_NAMES = {
        ADDRESS: 'address',
        COL: 'col',
        COLOR: 'color',
        CONTENTS: 'contents',
        FILENAME: 'filename',
        FORMAT: 'format',
        PARENTHESES: 'parentheses',
        PREFIX: 'prefix',
        PROTECT: 'protect',
        ROW: 'row',
        TYPE: 'type',
        WIDTH: 'width'
    };

    // maps resource keys of operators to their local names
    var LOCAL_OPERATOR_NAMES = {
        add: '+',
        sub: '-',
        mul: '*',
        div: '/',
        pow: '^',
        con: '&',
        pct: '%',
        lt: '<',
        le: '<=',
        gt: '>',
        ge: '>=',
        eq: '=',
        ne: '<>',
        // 'list' depends on locale
        isect: ' ',
        range: ':'
    };

    // maps function keys without help resources to function keys with help resources
    var MISSING_HELP_FUNC_MAP = {
        'BETA.INV': 'BETAINV',
        'BINOM.DIST': 'BINOMDIST',
        'BINOM.INV': 'CRITBINOM',
        CEILING: 'CEILING.ODF',
        'CEILING.MATH': 'CEILING.ODF',
        'CHISQ.DIST': 'CHISQDIST',
        'CHISQ.DIST.RT': 'CHIDIST',
        'CHISQ.INV': 'CHISQINV',
        'CHISQ.INV.RT': 'CHIINV',
        'CHISQ.TEST': 'CHITEST',
        'CONFIDENCE.NORM': 'CONFIDENCE',
        'COVARIANCE.P': 'COVAR',
        'ERF.PRECISE': 'ERF',
        'ERFC.PRECISE': 'ERFC',
        'EXPON.DIST': 'EXPONDIST',
        'F.DIST.RT': 'FDIST',
        'F.INV.RT': 'FINV',
        'F.TEST': 'FTEST',
        FLOOR: 'FLOOR.ODF',
        'FLOOR.MATH': 'FLOOR.ODF',
        'FORECAST.LINEAR': 'FORECAST',
        'GAMMA.DIST': 'GAMMADIST',
        'GAMMA.INV': 'GAMMAINV',
        'GAMMALN.PRECISE': 'GAMMALN',
        'HYPGEOM.DIST': 'HYPGEOMDIST',
        'LOGNORM.DIST': 'LOGNORMDIST',
        'LOGNORM.INV': 'LOGINV',
        'MODE.SNGL': 'MODE',
        'NEGBINOM.DIST': 'NEGBINOMDIST',
        'NORM.DIST': 'NORMDIST',
        'NORM.INV': 'NORMINV',
        'NORM.S.DIST': 'NORMSDIST',
        'NORM.S.INV': 'NORMSINV',
        'PERCENTILE.INC': 'PERCENTILE',
        'PERCENTRANK.INC': 'PERCENTRANK',
        'POISSON.DIST': 'POISSON',
        'QUARTILE.INC': 'QUARTILE',
        'RANK.EQ': 'RANK',
        'STDEV.P': 'STDEVP',
        'STDEV.S': 'STDEV',
        'T.INV.2T': 'TINV',
        'T.TEST': 'TTEST',
        'VAR.P': 'VARP',
        'VAR.S': 'VAR',
        'WEIBULL.DIST': 'WEIBULL',
        'Z.TEST': 'ZTEST'
    };

    // a map that contains missing help texts for functions
    var FUNC_HELP_RESOURCE = {

        // HYPGEOM.DIST contains a new parameter (existing help is mapped by HYPGEOMDIST, and will be used by HYPGEOM.DIST too)
        HYPGEOMDIST: {
            params: {
                4: {
                    //#. The name of the fifth argument of the spreadsheet function HYPGEOM.DIST (see https://help.libreoffice.org/Calc/Statistical_Functions_Part_Two#HYPGEOM.DIST)
                    n: gt.pgettext('funchelp-HYPGEOM.DIST', 'cumulative'),
                    //#. Description for the fifth argument of the spreadsheet function HYPGEOM.DIST (see https://help.libreoffice.org/Calc/Statistical_Functions_Part_Two#HYPGEOM.DIST)
                    d: gt.pgettext('funchelp-HYPGEOM.DIST', 'If set to TRUE, calculates the cumulative distribution function; if set to FALSE, calculates the probability density function.')
                }
            }
        },

        // NEGBINOM.DIST contains a new parameter (existing help is mapped by NEGBINOMDIST, and will be used by NEGBINOM.DIST too)
        NEGBINOMDIST: {
            params: {
                3: {
                    //#. The name of the fourth argument of the spreadsheet function NEGBINOM.DIST (see https://help.libreoffice.org/Calc/Statistical_Functions_Part_Four#NEGBINOM.DIST)
                    n: gt.pgettext('funchelp-NEGBINOM.DIST', 'cumulative'),
                    //#. Description for the fourth argument of the spreadsheet function NEGBINOM.DIST (see https://help.libreoffice.org/Calc/Statistical_Functions_Part_Four#NEGBINOM.DIST)
                    d: gt.pgettext('funchelp-NEGBINOM.DIST', 'If set to TRUE, calculates the cumulative distribution function; if set to FALSE, calculates the probability density function.')
                }
            }
        },

        // NORMS.DIST contains a new parameter (existing help is mapped by NORMSDIST, and will be used by NORM.S.DIST too)
        NORMSDIST: {
            params: {
                1: {
                    //#. The name of the second argument of the spreadsheet function NORM.S.DIST (see https://help.libreoffice.org/Calc/Statistical_Functions_Part_Five#NORM.S.DIST)
                    n: gt.pgettext('funchelp-NORM.S.DIST', 'cumulative'),
                    //#. Description for the second argument of the spreadsheet function NORM.S.DIST (see https://help.libreoffice.org/Calc/Statistical_Functions_Part_Five#NORM.S.DIST)
                    d: gt.pgettext('funchelp-NORM.S.DIST', 'If set to TRUE, calculates the cumulative distribution function; if set to FALSE, calculates the probability density function.')
                }
            }
        },

        // third parameter of NUMBERVALUE is not supported/documented in AOO
        NUMBERVALUE: {
            params: {
                2: {
                    //#. The name of the third argument of the spreadsheet function NUMBERVALUE (see https://help.libreoffice.org/Calc/NUMBERVALUE)
                    n: gt.pgettext('funchelp-NUMBERVALUE', 'group_separator'),
                    //#. Description for the third argument of the spreadsheet function NUMBERVALUE (see https://help.libreoffice.org/Calc/NUMBERVALUE)
                    d: gt.pgettext('funchelp-NUMBERVALUE', 'Defines the character used to separate groups of numbers (e.g. thousands, millions).')
                }
            }
        },

        // PERCENTRANK.INC contains a new parameter (existing help is mapped by PERCENTRANK, and will be used by PERCENTRANK.INC too)
        PERCENTRANK: {
            params: {
                2: {
                    //#. The name of the third argument of the spreadsheet function PERCENTRANK.INC (see https://help.libreoffice.org/Calc/Statistical_Functions_Part_Four#PERCENTRANK.INC)
                    n: gt.pgettext('funchelp-PERCENTRANK.INC', 'significance'),
                    //#. Description for the third argument of the spreadsheet function PERCENTRANK.INC (see https://help.libreoffice.org/Calc/Statistical_Functions_Part_Four#PERCENTRANK.INC)
                    d: gt.pgettext('funchelp-PERCENTRANK.INC', 'The number of significant digits to round the returned value to.')
                }
            }
        }
    };

    // imports ================================================================

    var RESOURCE_PATH = 'io.ox/office/spreadsheet/resource/formula';

    // load localized resource data
    var uiResourcePromise = LocaleData.loadResource(RESOURCE_PATH, { merge: true });
    uiResourcePromise.done(function (data) {
        RAW_UI_RESOURCE_DATA = data;

        // add missing help texts that have been defined locally in this file
        _.each(FUNC_HELP_RESOURCE, function (funcHelpMap, funcKey) {

            var funcResourceData = RAW_UI_RESOURCE_DATA.help[funcKey] || (RAW_UI_RESOURCE_DATA.help[funcKey] = {});

            // add missing function description
            if (funcHelpMap.description && !funcResourceData.d) {
                funcResourceData.d = funcHelpMap.description;
            }

            // add parameter help
            if (funcHelpMap.params) {
                var funcParams = funcResourceData.p || (funcResourceData.p = []);
                _.each(funcHelpMap.params, function (paramData, paramIndex) {
                    if (!(paramIndex in funcParams)) { funcParams[paramIndex] = paramData; }
                });
            }
        });
    });

    // load English resource data
    var enResourcePromise = (LocaleData.LOCALE === 'en_US') ? uiResourcePromise : LocaleData.loadResource(RESOURCE_PATH, { merge: true, locale: 'en_US' });
    enResourcePromise.done(function (data) { RAW_EN_RESOURCE_DATA = data; });

    // import all operator implementations
    var operatorPromise = require([
        'io.ox/office/spreadsheet/model/formula/funcs/operators'
    ], function (OPERATORS) {
        OPERATOR_IMPL_MAP = OPERATORS;
    });

    // import all function implementations
    var functionPromise = require([
        'io.ox/office/spreadsheet/model/formula/funcs/complexfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/conversionfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/databasefuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/datetimefuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/engineeringfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/financialfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/informationfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/logicalfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/mathfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/matrixfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/referencefuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/statisticalfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/textfuncs',
        'io.ox/office/spreadsheet/model/formula/funcs/webfuncs'
    ], function () {
        // merge all imported function descriptors into a single map
        FUNCTION_IMPL_MAP = _.extend.apply(_, [{}].concat(_.toArray(arguments)));
    });

    // wait for all promises
    var modulePromise = $.when(uiResourcePromise, enResourcePromise, operatorPromise, functionPromise);

    // private global functions ===============================================

    /**
     * Returns a generic name for a function parameter, used as fall-back for
     * parameters without an existing name in the help resouces.
     *
     * @param {Number} index
     *  The zero-based index of the function parameter.
     *
     * @returns {String}
     *  A generic name for a function parameter, to be used in the GUI.
     */
    function getGenericParamName(index) {
        //#. Short generic name for an unknown argument in a function of a spreadsheet formula,
        //#. used to create a dummy function signature in tooltips.
        //#. %1$d is the numeric index added to the argument.
        //#. Example for a resulting tooltip using this text: NEWFUNCTION(argument1, argument2, [argument3, ...]).
        //#, c-format
        return gt.pgettext('funchelp', 'argument%1$d', index + 1);
    }

    /**
     * Creates a descriptor collection containing the operator or function
     * implementations supported by the specified file format, resolved from
     * the passed internal implementation map.
     *
     * @param {Object} implementationMap
     *  The internal implementations of the operators or functions to be
     *  converted.
     *
     * @param {String} fileFormat
     *  The identifier of the file format.
     *
     * @param {Object} localNames
     *  The localized names of the operators or functions, mapped by unique
     *  resource key.
     *
     * @returns {DescriptorCollection}
     *  The resulting descriptor collection.
     */
    function createCollection(implementationMap, fileFormat, localNames) {

        // create a clone for deleting entries (needed to detect missing descriptors)
        localNames = _.clone(localNames);

        // process all raw descriptor objects
        var collection = DescriptorCollection.create(implementationMap, function (implementation, key) {

            // resolve file format dependent properties of the implementation
            var properties = Utils.mapProperties(implementation, function (value) {
                if (!_.isObject(value) || _.isArray(value) || _.isFunction(value)) { return value; }
                // if an object, pick a property by file format identifier
                if (_.isObject(value) && (fileFormat in value)) { return value[fileFormat]; }
            });

            // get the native name(s) of the function or operator
            var nativeNames = properties.name;
            if (_.isUndefined(nativeNames)) {
                nativeNames = [key]; // no name property: fall-back to resource key
            } else if (_.isString(nativeNames)) {
                nativeNames = [nativeNames];
            }

            // skip descriptors not supported in the specified file format
            if (!_.isArray(nativeNames) || (nativeNames.length === 0)) { return null; }

            // find the translated name
            var localName = localNames[key];
            delete localNames[key];
            // value may be an object with different translations per file format
            if (_.isObject(localName)) { localName = localName[fileFormat]; }

            // skip descriptors without translation
            if (!localName) {
                Utils.error('\xa0 missing translation: key=' + key);
                return null;
            }

            // create the resource descriptor with preferred native name (first array element)
            var descriptor = DescriptorCollection.newDescriptor(key, nativeNames[0], localName);

            // store the alternative native names contained in the array
            descriptor.altNativeNames = nativeNames.slice(1);

            // convert category list to a set
            descriptor.category = Utils.makeSet(Utils.getTokenListOption(properties, 'category', []));

            // add the hidden flag, and recalculation mode
            descriptor.hidden = Utils.getBooleanOption(properties, 'hidden', false);
            descriptor.recalc = Utils.getStringOption(properties, 'recalc', null);

            // copy parameter counts
            descriptor.minParams = properties.minParams;
            if (_.isNumber(properties.maxParams)) { descriptor.maxParams = properties.maxParams; }
            if (_.isNumber(properties.repeatParams)) { descriptor.repeatParams = properties.repeatParams; }

            // copy result type and format category
            descriptor.type = properties.type;
            descriptor.format = properties.format;

            // convert signature string to an array of parameter descriptors
            descriptor.signature = Utils.getTokenListOption(properties, 'signature', []).map(function (paramSig) {

                // split the leading type specifier from following options
                var tokens = paramSig.split('|');
                // first token is the type specifier
                var typeSpec = tokens[0];
                // base type is the leading part of the type specifier (e.g. the 'val' in 'val:bool')
                var baseType = typeSpec.split(':')[0];
                // source dependency behavior
                var depSpec = null;

                // process the additional options
                for (var index = 1; index < tokens.length; index += 1) {
                    var option = tokens[index].split(':');
                    if (option[0] === 'deps') { depSpec = option[1]; }
                }

                return { typeSpec: typeSpec, baseType: baseType, depSpec: depSpec };
            });

            // copy the resolver (callback function, or resource key, the latter will be resolved later)
            descriptor.resolve = properties.resolve;

            return descriptor;
        });

        // replace 'resolve' string properties with the implementation of the referred descriptor
        collection.forEach(function (descriptor) {
            var srcDescriptor = null;
            while (_.isString(descriptor.resolve) && (srcDescriptor = collection.get(descriptor.resolve))) {
                descriptor.resolve = srcDescriptor.resolve;
            }
        });

        return collection;
    }

    // class ParameterHelpDescriptor ==========================================

    /**
     * A descriptor with translated help texts for a single parameter of a
     * spreadsheet function.
     *
     * @constructor
     *
     * @property {String} name
     *  The translated name of the parameter.
     *
     * @property {String} description
     *  The description text of the parameter.
     *
     * @property {Boolean} optional
     *  Whether the parameter is required or optional.
     */
    function ParameterHelpDescriptor(name, description, optional) {

        this.name = _.isString(name) ? Utils.trimString(name) : '';
        this.description = _.isString(description) ? Utils.trimString(description) : '';
        this.optional = !!optional;

    } // class ParameterHelpDescriptor

    // class FunctionHelpDescriptor ===========================================

    /**
     * A descriptor with translated help texts for a spreadsheet function and
     * its parameters.
     *
     * @constructor
     *
     * @property {String} name
     *  The translated upper-case name of the function.
     *
     * @property {String} description
     *  A short description for the function itself.
     *
     * @property {Array<ParameterHelpDescriptor>} params
     *  An array with the names and descriptions of all parameters supported by
     *  the function.
     *
     * @property {Number} repeatStart
     *  The zero-based index of the first trailing parameter that can be used
     *  repeatedly in the function; or -1, if the function does not support
     *  parameter repetition.
     *
     * @property {Number} cycleLength
     *  The number of trailing repeated parameters; or 0, if the function does
     *  not support parameter repetition.
     *
     * @property {Number} cycleCount
     *  The maximum number of repetition cycles supported by the function,
     *  according to the index of first repeated parameter, and the size of the
     *  repetition cycles. Will be 0, if the function does not support
     *  parameter repetition.
     */
    function FunctionHelpDescriptor(name, description) {

        this.name = name;
        this.description = _.isString(description) ? Utils.trimString(description) : '';
        this.params = null;
        this.repeatStart = -1;
        this.cycleLength = 0;
        this.cycleCount = 0;

    } // class FunctionHelpDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Returns a shallow clone of this function help descriptor with another
     * function name.
     *
     * @param {String} name
     *  The new function name.
     *
     * @returns {FunctionHelpDescriptor}
     *  A shallow clone of this function help descriptor.
     */
    FunctionHelpDescriptor.prototype.clone = function (name) {
        var helpDesc = new FunctionHelpDescriptor(name, this.description);
        helpDesc.params = this.params;
        helpDesc.repeatStart = this.repeatStart;
        helpDesc.cycleLength = this.cycleLength;
        helpDesc.cycleCount = this.cycleCount;
        return helpDesc;
    };

    // class FormulaResource ==================================================

    /**
     * Contains and provides access to all native and localized resources used
     * in spreadsheet formulas, for example operator and function descriptors
     * (syntax descriptions, and actual implementations), function names, names
     * of error codes and boolean literals, and the help texts for functions
     * and function parameters. An instance of this class is not bound to a
     * single application instance but will be created once for all documents
     * loaded from files with a specific file format.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {String} fileFormat
     *  The identifier of the file format related to the formula resource data
     *  represented by this instance.
     *
     * @param {String} locale
     *  The locale identifier of the formula resources to be loaded. Currently,
     *  only two values are supported: The current UI language as contained in
     *  the constant LocaleData.LOCALE to load the translated formula
     *  resources, or the string 'en_US' to load the English formula resources.
     */
    var FormulaResource = BaseObject.extend({ constructor: function (fileFormat, locale) {

        // the raw JSON resource data depending on the passed locale
        var rawResourceData = null;

        // native and localized names of boolean values
        var booleanCollection = null;

        // descriptors of all supported error codes
        var errorCollection = null;

        // descriptors of all supported regions in table ranges
        var regionCollection = null;

        // collection of all supported operators
        var operatorCollection = null;

        // descriptors of all supported functions
        var functionCollection = null;

        // descriptors of all supported named parameters of the function CELL
        var cellParamCollection = null;

        // the locale data configuration for the specified locale
        var localeData = LocaleData.get(locale);

        // the native function parameter separator
        var nativeSep = null;

        // the localized function parameter separator
        var localSep = null;

        // localized R1C1 prefix characters
        var localPrefixChars = null;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Creates a help descriptor for the specified function descriptor. The
         * help descriptor (instance of FunctionHelpDescriptor) will be
         * inserted as property 'help' into the function descriptor.
         *
         *
         * @param {String} funcKey
         *  The resource key of a function implementation.
         *
         * @param {Object} rawHelpEntry
         *  The raw help resource data for the function.
         */
        function createHelpDesc(funcKey, rawHelpEntry, logProblems) {

            // writes a warning or error message for the current function
            var log = logProblems ? function (type, msg) {
                Utils[type]('\xa0 ' + msg + ' function ' + funcKey);
            } : _.noop;

            // the function descriptor resolved for the current file format (silently skip
            // function help for functions not supported by the current file format)
            var functionDesc = functionCollection.get(funcKey);
            if (!functionDesc) { return; }

            if (functionDesc.help) {
                log('warn', 'help exists already for');
                return;
            }

            // function data entry must be an object
            if (!_.isObject(rawHelpEntry)) {
                log('error', 'invalid map entry for');
                return;
            }

            // the resulting function help descriptor
            var funcHelp = functionDesc.help = new FunctionHelpDescriptor(functionDesc.localName, rawHelpEntry.d);
            if (!rawHelpEntry.d) {
                log('error', 'missing description for');
            }

            // check parameter help availability of required parameters
            var params = _.clone(rawHelpEntry.p);
            var minParams = functionDesc.minParams;
            if (params.length < minParams) {
                log('error', 'missing help for required parameters for');
                while (params.length < minParams) { params.push({ n: '', d: '' }); }
            }

            // check parameter help availability of fixed number of optional parameters
            var maxParams = functionDesc.maxParams;
            if (_.isNumber(maxParams) && (params.length < maxParams)) {
                log('error', 'missing help for optional parameters for');
                while (params.length < maxParams) { params.push({ n: '', d: '' }); }
            }

            // validate number of array elements of function parameter help
            if (_.isNumber(functionDesc.maxParams)) {
                while (params.length < functionDesc.maxParams) { params.push({ n: '', d: '' }); }
                params.splice(functionDesc.maxParams);
            }

            // check parameter help availability of variable number of optional parameters
            var repeatParams = _.isNumber(functionDesc.maxParams) ? null : (functionDesc.repeatParams || 1);
            if (_.isNumber(repeatParams)) {
                var optParams = params.length - functionDesc.minParams;
                // remove help texts of superfluous optional parameters
                if ((optParams !== 0) && (optParams !== repeatParams)) {
                    log('warn', 'superfluous variable parameters for');
                    params.splice(functionDesc.minParams + ((optParams > repeatParams) ? repeatParams : 0));
                }
            }

            // create a single parameter array with name, description, and additional information
            funcHelp.params = params.map(function (param, index) {
                var paramName = param.n || getGenericParamName(index);
                return new ParameterHelpDescriptor(paramName, param.d, functionDesc.minParams <= index);
            });

            // add properties for repeating parameters
            if (_.isNumber(repeatParams)) {
                // absolute index of first repeated parameter
                funcHelp.repeatStart = funcHelp.params.length - repeatParams;
                funcHelp.cycleLength = repeatParams;
                funcHelp.cycleCount = Math.floor((FormulaUtils.MAX_PARAM_COUNT - funcHelp.repeatStart) / repeatParams);
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the identifier of the file format represented by this
         * instance.
         *
         * @returns {String}
         *   The identifier of the file format represented by this instance.
         */
        this.getFileFormat = function () {
            return fileFormat;
        };

        /**
         * Returns the identifier of the locale represented by this instance.
         *
         * @returns {String}
         *   The identifier of the locale represented by this instance.
         */
        this.getLocale = function () {
            return locale;
        };

        /**
         * Returns the decimal separator for the current UI language.
         *
         * @param {Boolean} localized
         *  Whether to return the localized decimal separator according to the
         *  UI language (true), or the native (English) decimal separator.
         *
         * @returns {String}
         *  The decimal separator character.
         */
        this.getDec = function (localized) {
            return localized ? localeData.dec : '.';
        };

        /**
         * Returns the group separator for the current UI language.
         *
         * @param {Boolean} localized
         *  Whether to return the localized group separator according to the UI
         *  language (true), or the native (English) group separator.
         *
         * @returns {String}
         *  The decimal separator character.
         */
        this.getGroup = function (localized) {
            return localized ? localeData.group : ',';
        };

        /**
         * Returns the list separator character. The list separator is used to
         * separate parameters of functions, and in OOXML for the range list
         * operator.
         *
         * @param {Boolean} localized
         *  Whether to return the localized list separator character according
         *  to the UI language (true), or the native (English) list separator
         *  character.
         *
         * @returns {String}
         *  The list separator character.
         */
        this.getSeparator = function (localized) {
            return localized ? localSep : nativeSep;
        };

        /**
         * Returns both prefix characters used in R1C1 references (for column
         * and row).
         *
         * @param {Boolean} localized
         *  Whether to return the localized prefix characters according to the
         *  UI language (true), or the native (English) prefix characters.
         *
         * @returns {String}
         *  The prefix characters used in R1C1 references (a string with two
         *  characters, first the row prefix, then the column prefix).
         */
        this.getR1C1PrefixChars = function (localized) {
            return localized ? localPrefixChars : NATIVE_R1C1_PREFIX_CHARS;
        };

        /**
         * Returns the collection with the descriptors of all boolean names.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all boolean names. The
         *  descriptor for the boolean value false is mapped by the resource
         *  key 'f', and the descriptor for the boolean value true is mapped by
         *  the resource key 't'.
         */
        this.getBooleanCollection = function () {
            return booleanCollection;
        };

        /**
         * Returns the collection with the descriptors of all error codes
         * supported by the current file format.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all error codes supported by
         *  the current file format.
         */
        this.getErrorCollection = function () {
            return errorCollection;
        };

        /**
         * Returns the collection with the descriptors of all regions in
         * structured table references supported by the current file format.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all regions in structured
         *  table references supported by the current file format.
         */
        this.getRegionCollection = function () {
            return regionCollection;
        };

        /**
         * Returns the collection with the descriptors of all supported keys
         * for the first parameter of the function CELL.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all supported keys for the
         *  first parameter of the function CELL.
         */
        this.getCellParamCollection = function () {
            return cellParamCollection;
        };

        /**
         * Returns the collection with the descriptors of all unary and binary
         * operators supported by the current file format.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all unary and binary
         *  operators supported by the current file format.
         */
        this.getOperatorCollection = function () {
            return operatorCollection;
        };

        /**
         * Returns the collection with the descriptors of all functions
         * supported by the current file format.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all functions supported by
         *  the current file format.
         */
        this.getFunctionCollection = function () {
            return functionCollection;
        };

        /**
         * Returns a help text descriptor for the specified function.
         *
         * @param {String} key
         *  The unique resource key of the function.
         *
         * @returns {FunctionHelpDescriptor|Null}
         *  A help descriptor for the specified function; or null for invalid
         *  keys, or functions without help texts.
         */
        this.getFunctionHelp = function (key) {
            return functionCollection.getProperty(key, 'help', null);
        };

        // initialization -----------------------------------------------------

        switch (locale) {
            case LocaleData.LOCALE:
                rawResourceData = RAW_UI_RESOURCE_DATA;
                break;
            case 'en_US':
                rawResourceData = RAW_EN_RESOURCE_DATA;
                break;
            default:
                throw new Error('FormulaResource: invalid locale identifier "' + locale + '"');
        }

        // create descriptors for the names of boolean values
        booleanCollection = DescriptorCollection.create({ f: 'FALSE', t: 'TRUE' }, function (nativeName, key) {
            var localName = rawResourceData[nativeName];
            if (!localName) {
                Utils.error('\xa0 missing boolean translation: key=' + key);
                localName = nativeName;
            }
            return DescriptorCollection.newDescriptor(key, nativeName, localName);
        });

        // create error code descriptors
        errorCollection = DescriptorCollection.create(ErrorCode.NATIVE_CODES, function (error, key) {
            var nativeName = NATIVE_ERROR_NAMES[key];
            var localName = rawResourceData.errors[key];
            if (!localName) {
                Utils.error('\xa0 missing error code translation: key=' + key);
                localName = nativeName;
            }
            return DescriptorCollection.newDescriptor(key, nativeName, localName);
        });

        // resolve descriptors for regions in structured table references
        regionCollection = DescriptorCollection.create(NATIVE_REGION_NAMES, function (nativeName, key) {
            var localName = rawResourceData.regions[key];
            if (!localName) {
                Utils.error('\xa0 missing table region translation: key=' + key);
                localName = nativeName;
            }
            return DescriptorCollection.newDescriptor(key, nativeName, localName);
        });

        // resolve descriptors for named parameters of the function CELL
        cellParamCollection = DescriptorCollection.create(NATIVE_CELL_PARAM_NAMES, function (nativeName, key) {
            var localName = rawResourceData.cellParams[key];
            if (!localName) {
                Utils.error('\xa0 missing CELL parameter translation: key=' + key);
                localName = nativeName;
            }
            return DescriptorCollection.newDescriptor(key, nativeName, localName);
        });

        // function parameter separator
        nativeSep = (fileFormat === 'odf') ? ';' : ',';
        localSep = (localeData.dec === ',') ? ';' : ',';

        // resolve operator descriptors, and the maps with native and translated operator names
        operatorCollection = createCollection(OPERATOR_IMPL_MAP, fileFormat, _.extend({ list: localSep }, LOCAL_OPERATOR_NAMES));

        // resolve function descriptors, and the maps with native and translated function names
        functionCollection = createCollection(FUNCTION_IMPL_MAP, fileFormat, rawResourceData.functions);

        // the R1C1 prefix characters (ODF does not translate these characters in the UI)
        localPrefixChars = (fileFormat === 'odf') ? NATIVE_R1C1_PREFIX_CHARS : Utils.getStringOption(rawResourceData, 'RC', 'RC').toUpperCase();

        // process each entry in the function help map (build system ensures valid data)
        _.each(rawResourceData.help, function (rawHelpEntry, funcKey) {
            createHelpDesc(funcKey, rawHelpEntry, true);
        });

        // distribute existing help resources to functions without own help
        _.each(MISSING_HELP_FUNC_MAP, function (existingHelpFuncKey, missingHelpFuncKey) {
            var rawHelpEntry = Utils.getObjectOption(rawResourceData.help, existingHelpFuncKey, null);
            if (rawHelpEntry) { createHelpDesc(missingHelpFuncKey, rawHelpEntry, true); }
        });

        // add dummy help descriptors for all remaining functions without help resouces
        var missingHelpKeys = [];
        functionCollection.forEach(function (funcDesc, funcKey) {
            if (!funcDesc.help && !funcDesc.hidden) {
                createHelpDesc(funcKey, { d: '', p: [] }, false);
                missingHelpKeys.push(funcKey);
            }
        });

        // print all function names without help texts
        Utils.withLogging(function () {
            if (missingHelpKeys.length > 0) {
                Utils.error('\xa0 missing help for functions: ' + missingHelpKeys.sort().join(', '));
            }
        });

    } }); // class FormulaResource

    // static methods ---------------------------------------------------------

    /**
     * Returns an existing formula resource singleton from the internal cache,
     * if available; otherwise creates a new instance.
     *
     * @param {String} fileFormat
     *  The identifier of the file format related to the formula resource data
     *  represented by the instance returned from this method.
     *
     * @param {String} locale
     *  The locale identifier of the formula resources to be loaded. Currently,
     *  only two values are supported: The current UI language as contained in
     *  the constant LocaleData.LOCALE to load the translated formula
     *  resources, or the string 'en_US' to load the English formula resources.
     *
     * @returns {FormulaResource}
     *  A formula resource singleton for the passed file format.
     */
    FormulaResource.create = _.memoize(function (fileFormat, locale) {
        return new FormulaResource(fileFormat, locale);
    }, function (fileFormat, locale) {
        return fileFormat + ':' + locale;
    });

    // exports ================================================================

    return modulePromise.then(_.constant(FormulaResource));

});
