/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/referencefuncs', [
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/model/formula/utils/cellref',
    'io.ox/office/spreadsheet/model/formula/utils/sheetref',
    'io.ox/office/spreadsheet/model/formula/utils/matrix'
], function (IteratorUtils, SheetUtils, FormulaUtils, TokenArray, CellRef, SheetRef, Matrix) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with cell range
     * references used as values (cell address arithmetics), and functions that
     * look up specific values in a range of cells.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, cell range references can be given literally,
     * or can be the result of functions or operators. The client-side formula
     * engine provides the function signature type 'ref', and the class
     * Reference whose instances represent a single cell range address, or a
     * list of several cell range addresses (e.g. as result of the reference
     * list operator).
     *
     *************************************************************************/

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Range3D = SheetUtils.Range3D;
    var Range3DArray = SheetUtils.Range3DArray;

    // private global variables ===============================================

    var SEARCHOPERANDOPTIONS = {
        emptyParam: true,
        emptyCells: false,
        acceptErrors: true
    };

    // private global functions ===============================================

    function splitOperand(context, operand, subsets) {
        var value = operand.getRawValue();
        if (operand.isReference()) {
            value = context.convertToValueRange(value);
        }
        var res = [];
        _.each(subsets, function (subset) {
            var nv = null;
            if (operand.isMatrix()) {
                nv = Matrix.generate(subset.rows, subset.cols, function (row, col) {
                    return value.get(subset.row + row, subset.col + col);
                });
            } else if (operand.isReference()) {
                nv = Range3D.create(value.start[0] + subset.col, value.start[1] + subset.row, value.start[0] + subset.col + subset.cols - 1, value.start[1] + subset.row + subset.rows - 1, value.sheet1, value.sheet2);
            } else {
                nv = value;
            }
            res.push(context.createOperand(nv));
        });

        return res;
    }

    /**
     * @returns {Number} order and {Operand} op
     *  1  is ascending
     *  -1 is descending
     *  0  is no order
     */
    function getOrderType(context, operand) {
        var order = null;
        var last = null;

        var resultArray = [];
        var iterator = context.createScalarIterator(operand);

        IteratorUtils.some(iterator, function (value) {
            if (last !== null) {
                var subOrder = context.compareScalars(last, value);
                if (order === null) {
                    order = subOrder;
                } else if (subOrder !== order) {
                    order = 0;
                    return true;
                }
            }
            resultArray.push(value);
            last = value;
        });

        if (order === null) {
            return { op: operand, type: 1 };
        }

        if (order === 0) {
            return { op: operand, type: 0 };
        }

        if (order === -1) { resultArray.reverse(); }
        var mat = new Matrix([resultArray]);
        var op = context.createOperand(mat);
        return { op: op, type: -order };
    }

    function searchInOperand(context, lookup, inexact, searchOperand, resultOperand) {

        if (resultOperand.isValue()) { throw ErrorCode.NA; }

        var index = searchIndexInOperand(context, lookup, inexact, searchOperand);
        var res = null;
        var iterator = context.createScalarIterator(resultOperand, SEARCHOPERANDOPTIONS);

        IteratorUtils.some(iterator, function (value, result) {
            if (result.offset === index) {
                res = value;
                return true;
            }
        });
        return res;

    }

    function searchIndexInOperand(context, lookup, inexact, operand, matchSearch) {

        function getTypeId(value) {
            return _.isNumber(value) ? 1 : _.isString(value) ? 2 : _.isBoolean(value) ? 3 : (value instanceof ErrorCode) ? 4 : 0;
        }

        if (_.isUndefined(inexact)) { inexact = true; }
        if (_.isUndefined(lookup) || _.isNull(lookup)) { throw ErrorCode.VALUE; }

        var regex = null;
        if (!inexact && _.isString(lookup)) { regex = context.convertPatternToRegExp(lookup, true); }

        var lookupType = getTypeId(lookup);
        var last = null;
        var lastIndex = null;
        var resultIndex = Number.NaN;
        var iterator = context.createScalarIterator(operand, SEARCHOPERANDOPTIONS);

        IteratorUtils.some(iterator, function (value, result) {
            if (value === null) { return; }

            var index = result.offset;

            last = value;
            lastIndex = index;

            var currentType = getTypeId(value);
            if (currentType !== lookupType) { return; }

            var compare = context.compareScalars(value, lookup);
            if (compare === 0) {
                resultIndex = index;
                return true;
            }
            if (inexact && compare > 0) {
                if (index < 1) { throw ErrorCode.NA; }
                resultIndex = index - 1;
                return true;
            }
            if (regex && regex.test(value)) {
                resultIndex = index;
                return true;
            }
        });

        if (!_.isNaN(resultIndex)) { return resultIndex; }

        if (!_.isNaN(lastIndex)) {
            if (matchSearch) {
                return lastIndex;
            } else if (inexact && getTypeId(last) === lookupType && context.compareScalars(last, lookup) < 0) {
                return lastIndex;
            }
        }

        throw ErrorCode.NA;
    }

    // exports ================================================================

    return {

        ADDRESS: {
            category: 'reference',
            minParams: 2,
            maxParams: 5,
            type: 'val',
            signature: 'val:int val:int val:int val:bool val:str',
            resolve: function (row, col, absMode, a1Style, sheetName) {

                // the resulting address string
                var result = '';

                // default for missing or empty absolute mode (always defaults to 1)
                if (this.isMissingOrEmptyOperand(2)) {
                    absMode = 1;
                } else if ((absMode < 1) || (absMode > 4)) {
                    throw ErrorCode.VALUE;
                }

                // default for missing or empty A1 style (always defaults to TRUE)
                if (this.isMissingOrEmptyOperand(3)) {
                    a1Style = true;
                }

                // initialize and validate the cell reference structure
                var cellRef = new CellRef(0, 0, (absMode % 2) > 0, absMode <= 2);
                cellRef.col = (a1Style || cellRef.absCol) ? (col - 1) : col;
                cellRef.row = (a1Style || cellRef.absRow) ? (row - 1) : row;
                this.checkCellRef(cellRef, !a1Style);

                // start with encoded sheet name (enclosed in apostrophes if needed)
                // (not for missing or empty parameter, but for explicit empty string)
                if (_.isString(sheetName)) {

                    // enclose complex names in apostrophes
                    if ((sheetName.length > 0) && !this.formulaGrammar.isSimpleSheetName(this.docModel, sheetName, { range: true, external: true })) {
                        sheetName = SheetRef.encodeComplexSheetName(sheetName);
                    }

                    // use sheet name with an exclamation mark as separator
                    result = sheetName + '!';
                }

                // append the range address in A1 or R1C1 style
                return result + this.formulaGrammar.formatReference(this.docModel, null, null, cellRef, null, !a1Style);
            }
        },

        AREAS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'ref|deps:skip',
            resolve: function (ranges) { return ranges.length; }
        },

        CHOOSE: {
            category: 'reference',
            minParams: 2,
            type: 'any',
            signature: 'val:int any:lazy|deps:pass',
            resolve: function (index) {
                // CHOOSE uses one-based index
                if ((index < 1) || (index >= this.getOperandCount())) { throw ErrorCode.VALUE; }
                return this.getOperand(index);
            }
        },

        COLUMN: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref:single|deps:skip',
            resolve: function (range) {
                // use reference address, if parameter is missing
                var address = range ? range.start : this.getRefAddress();
                // function returns one-based column index
                return address[0] + 1;
            }
        },

        COLUMNS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any|deps:skip',
            resolve: function (operand) {
                switch (operand.getType()) {
                    case 'val':
                        var value = operand.getValue();
                        return (value instanceof ErrorCode) ? value : 1;
                    case 'mat':
                        return operand.getMatrix().cols();
                    case 'ref':
                        return this.convertToRange(operand.getRanges()).cols();
                    default:
                        FormulaUtils.throwInternal('ReferenceFuncs.COLUMNS.resolve(): unknown operand type');
                }
            }
        },

        FORMULATEXT: {
            category: 'reference',
            name: { ooxml: '_xlfn.FORMULATEXT', odf: 'FORMULA' },
            minParams: 1,
            maxParams: 1,
            type: 'val',
            recalc: 'always', // changing a formula may not change the result, thus no recalculation would be triggered
            signature: 'ref:single|deps:skip',
            resolve: function (range) {

                // always use the first cell in the range (TODO: matrix context)
                var address = range.start;

                // TODO: resolve self-reference while entering a cell or shared formula

                // getCellFormula() returns null for blank cells, and simple value cells
                var formula = this.getCellFormula(range.sheet1, address);
                return (formula === null) ? ErrorCode.NA : ('=' + formula);
            }
        },

        GETPIVOTDATA: {
            category: 'reference',
            hidden: true,
            minParams: 2,
            repeatParams: 2,
            type: 'val'
        },

        HLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any',
            signature: 'val any val:int val:bool',
            resolve: function (lookup, searchOperand, rowIndexParam, inexact) {
                var rowIndex = rowIndexParam - 1;
                var dim = searchOperand.getDimensions();

                if (rowIndex < 0) { throw ErrorCode.VALUE; }
                if (rowIndex >= dim.rows) { throw ErrorCode.REF; }

                var ops = splitOperand(this, searchOperand, [{ col: 0, cols: dim.cols, row: 0, rows: 1 }, { col: 0, cols: dim.cols, row: rowIndex, rows: 1 }]);

                return searchInOperand(this, lookup, inexact, ops[0], ops[1]);
            }
        },

        INDEX: {
            category: 'reference',
            minParams: 2,
            maxParams: 4,
            type: 'any',
            recalc: 'always',
            signature: 'any|deps:skip val:int val:int val:int',
            resolve: function (source, row, col, area) {
                if (source.isMatrix() && !_.isUndefined(area)) { throw ErrorCode.REF; }

                if (_.isUndefined(area)) { area = 1; }
                if (_.isUndefined(col)) { col = 1; }

                row--; col--; area--;

                if (source.isValue()) { throw ErrorCode.VALUE; }

                var value = source.getRawValue();
                if (value instanceof Range3DArray) {
                    if (area < 0 || area >= value.length)  { throw ErrorCode.REF; }
                    value = value[area];
                    source = this.createOperand(value);
                }
                var dim = source.getDimensions();

                if (col < 0 || col >= dim.cols)  { throw ErrorCode.REF; }

                var result = null;
                if (row === -1) {
                    result = splitOperand(this, source, [{ col: col, cols: 1, row: 0, rows: value.rows() }]);
                } else {
                    if (row < 0 || row >= dim.rows)  { throw ErrorCode.REF; }

                    result = splitOperand(this, source, [{ col: col, cols: 1, row: row, rows: 1 }]);
                }
                result = result[0].getRawValue();
                if (result instanceof Range3DArray && result.length === 1) {
                    result = result[0];
                }
                return result;
            }
        },

        INDIRECT: {
            category: 'reference',
            minParams: 1,
            maxParams: 2,
            type: 'ref',
            recalc: 'always',
            signature: 'val:str val:bool',
            resolve: function (refText, a1Style) {
                // default for missing or empty A1 style is true!
                a1Style = a1Style || this.isMissingOrEmptyOperand(1);

                var sheetModel = this.docModel.getSheetModel(this.getRefSheet());
                var tokenArray = new TokenArray(sheetModel, 'custom');
                var refAddress = this.getRefAddress();

                tokenArray.parseFormula('ui', refText, { refAddress: refAddress });

                var ranges = tokenArray.resolveRangeList({ resolveNames: true, refAddress: refAddress, targetAddress: refAddress });
                if (ranges.length !== 1) { throw ErrorCode.REF; }

                return ranges.first();
            }
        },

        LOOKUP: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'any',
            signature: 'val any any',
            resolve: function (lookup, searchOperand, resultOperand) {

                if (this.isMissingOrEmptyOperand(2)) {
                    var dim = searchOperand.getDimensions();
                    var ops = [];
                    if (dim.rows < dim.cols) {
                        ops = splitOperand(this, searchOperand, [{ col: 0, cols: dim.cols, row: 0, rows: 1 }, { col: 0, cols: dim.cols, row: dim.rows - 1, rows: 1 }]);
                    } else {
                        ops = splitOperand(this, searchOperand, [{ col: 0, cols: 1, row: 0, rows: dim.rows }, { col: dim.cols - 1, cols: 1, row: 0, rows: dim.rows }]);
                    }

                    return searchInOperand(this, lookup, undefined, ops[0], ops[1]);
                } else {
                    if (resultOperand.isReference()) {
                        var sDim = searchOperand.getDimensions();
                        var rDim = resultOperand.getDimensions();
                        var resultRange = null;

                        if (rDim.cols < sDim.cols) {
                            resultRange = this.convertToValueRange(resultOperand.getRawValue());
                            resultRange.end[0] = resultRange.start[0] + sDim.cols;
                            resultOperand = this.createOperand(resultRange);
                        } else if (rDim.rows < sDim.rows) {
                            resultRange = this.convertToValueRange(resultOperand.getRawValue());
                            resultRange.end[1] = resultRange.start[1] + sDim.rows;
                            resultOperand = this.createOperand(resultRange);
                        }
                    }
                    return searchInOperand(this, lookup, undefined, searchOperand, resultOperand);
                }
            }
        },

        MATCH: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val any val:int',
            resolve: function (lookup, searchOperand, order) {

                var dim = searchOperand.getDimensions();

                if (Math.min(dim.rows, dim.cols) !== 1) { throw ErrorCode.NA; }

                if (_.isUndefined(order)) { order = 1; }

                if (order !== 0) {
                    var orderType = getOrderType(this, searchOperand);

                    if (order !== 1 && order !== orderType.type) { throw ErrorCode.NA; }

                    searchOperand = orderType.op;
                }
                try {
                    return searchIndexInOperand(this, lookup, false, searchOperand, order === 1) + 1;
                } catch (e) {
                    if (order === 1 && e === ErrorCode.NA) {
                        return searchIndexInOperand(this, lookup, true, searchOperand, order === 1) + 1;
                    } else {
                        throw e;
                    }
                }
            }
        },

        OFFSET: {
            category: 'reference',
            minParams: 3,
            maxParams: 5,
            type: 'ref',
            recalc: 'always',
            signature: 'ref|deps:skip val:int val:int val:int val:int',
            resolve: function (ranges, rows, cols, height, width) {

                // reference must contain a single range address
                var range = this.convertToRange(ranges, { valueError: true });

                // apply offset
                range.start[0] += cols;
                range.start[1] += rows;
                range.end[0] += cols;
                range.end[1] += rows;

                // modify size (but not if 'height' or 'width' parameters exist but are empty)
                if (!this.isMissingOrEmptyOperand(4)) { range.end[0] = range.start[0] + width - 1; }
                if (!this.isMissingOrEmptyOperand(3)) { range.end[1] = range.start[1] + height - 1; }

                // check that the range does not left the sheet boundaries
                if (this.docModel.isValidRange(range)) { return range; }
                throw ErrorCode.REF;
            }
        },

        ROW: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val',
            signature: 'ref:single|deps:skip',
            resolve: function (range) {
                // use reference address, if parameter is missing
                var address = range ? range.start : this.getRefAddress();
                // function returns one-based row index
                return address[1] + 1;
            }
        },

        ROWS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'any|deps:skip',
            resolve: function (operand) {
                switch (operand.getType()) {
                    case 'val':
                        var value = operand.getValue();
                        return (value instanceof ErrorCode) ? value : 1;
                    case 'mat':
                        return operand.getMatrix().rows();
                    case 'ref':
                        return this.convertToRange(operand.getRanges()).rows();
                    default:
                        FormulaUtils.throwInternal('ReferenceFuncs.ROWS.resolve(): unknown operand type');
                }
            }
        },

        SHEET: {
            category: 'reference',
            name: { ooxml: '_xlfn.SHEET' },
            minParams: 0,
            maxParams: 1,
            type: 'val',
            recalc: 'always',
            signature: 'any|deps:skip', // string (sheet name) or reference
            resolve: function (operand) {

                // use reference sheet, if parameter is missing
                if (!operand) { return this.getRefSheet() + 1; }

                // use sheet index of a reference
                if (operand.isReference()) {
                    return this.convertToRange(operand.getRanges()).sheet1 + 1;
                }

                // convert values to strings (sheet name may look like a number or Boolean)
                if (operand.isValue()) {
                    var sheet = this.docModel.getSheetIndex(this.convertToString(operand.getValue()));
                    return (sheet < 0) ? ErrorCode.NA : (sheet + 1);
                }

                // always fail for matrix operands, regardless of context
                return ErrorCode.NA;
            }
        },

        SHEETS: {
            category: 'reference',
            name: { ooxml: '_xlfn.SHEETS' },
            minParams: 0,
            maxParams: 1,
            type: 'val',
            recalc: 'always',
            signature: 'ref:multi|deps:skip',
            resolve: function (range) {
                // return total number of sheets in document, if parameter is missing
                return range ? range.sheets() : this.docModel.getSheetCount();
            }
        },

        SWITCH: {
            category: 'reference',
            name: { ooxml: '_xlfn.SWITCH', odf: null },
            minParams: 3,
            // no repetition pattern, SWITCH supports a 'default value' (single parameter after a pair of parameters)!
            type: 'any',
            signature: 'val any:lazy|deps:pass any:lazy|deps:pass',
            resolve: function (selector) {

                // total number of operands
                var length = this.getOperandCount();
                // number of parameter pairs following the leading selector parameter
                var pairs = Math.floor((length - 1) / 2);

                // try to find a parameter pair matching the selector
                for (var index = 0; index < pairs; index += 1) {
                    // immediately throw on error code
                    var value = this.getOperand(index * 2 + 1, 'val');
                    // do not convert data types while comparing (number 1 as selector DOES NOT select string '1')
                    if (this.compareScalars(selector, value) === 0) {
                        return this.getOperand(index * 2 + 2);
                    }
                }

                // return default value if provided, otherwise #N/A error code (thrown automatically)
                return this.getOperand(pairs * 2 + 1);
            }
        },

        VLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any',
            signature: 'val any val:int val:bool',
            resolve: function (lookup, searchOperand, colIndexParam, inexact) {
                var colIndex = colIndexParam - 1;
                var dim = searchOperand.getDimensions();

                if (colIndex < 0) { throw ErrorCode.VALUE; }
                if (colIndex >= dim.cols) { throw ErrorCode.REF; }

                var ops = splitOperand(this, searchOperand, [{ col: 0, cols: 1, row: 0, rows: dim.rows }, { col: colIndex, cols: 1, row: 0, rows: dim.rows }]);

                return searchInOperand(this, lookup, inexact, ops[0], ops[1]);
            }
        }
    };

});
