/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/sheetdescriptor', [
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils'
], function (SimpleMap, SheetUtils, DependencyUtils) {

    'use strict';

    // convenience shortcuts
    var AddressArray = SheetUtils.AddressArray;
    var AddressSet = SheetUtils.AddressSet;

    // class SheetDescriptor ==================================================

    /**
     * This descriptor object collects all dependency settings for a sheet in
     * a spreadsheet document.
     *
     * @constructor
     *
     * @property {SheetModel} sheetModel
     *  The model of the sheet represented by this descriptor object.
     *
     * @property {SimpleMap<FormulaDescriptor>} formulaMap
     *  The formula descriptors (instances of class FormulaDescriptor) for all
     *  known token arrays of the sheet.
     *
     * @property {AddressSet} cellFormulaSet
     *  The formula descriptors (instances of class FormulaDescriptor) for all
     *  known cell formulas of the sheet, as property 'formula' in the cell
     *  address objects of an address set.
     *
     * @property {SimpleMap<SimpleMap<FormulaDescriptor>>} modelFormulaMaps
     *  The descriptors of all other formulas (but cell formulas) attached to a
     *  specific parent model object, e.g. the conditions of formatting rules.
     *  The elements of this map are maps of formula descriptors. The inner
     *  maps are keyed by the UIDs of the parent model. The formula descriptors
     *  are keyed by the complete formula keys of the token arrays (the UID of
     *  the parent model, combined with the internal keys of the token arrays
     *  used inside the parent model).
     *
     * @param {SimpleMap<FormulaDescriptor>} missingNamesMap
     *  The formula descriptors for all known token arrays that contain
     *  references to non-existing defined names. These formulas are considered
     *  to be dirty, if a defined name has been inserted, or an existing defind
     *  name has changed its label, which may result in validating a #NAME?
     *  error. The property is a map with the labels of missing names as key,
     *  and stores a formula map (keyed by formula keys) per missing name.
     *
     * @param {SimpleMap<FormulaDescriptor>} recalcAlwaysMap
     *  The formula descriptors for all known token arrays with recalculation
     *  mode 'always'. These formulas will always be recalculated whenever
     *  something in the document has changed, regardless of their
     *  dependencies.
     *
     * @param {SimpleMap<FormulaDescriptor>} recalcOnceMap
     *  The formula descriptors for all known token arrays with recalculation
     *  mode 'once'. These formulas will be recalculated once after the
     *  document has been imported, regardless of their dependencies.
     *
     * @property {AddressArray} dirtyValueCells
     *  The addresses of all dirty value cells (cells with changed values) in
     *  the sheet. The cell addresses will be collected, and will cause to
     *  update  all formulas known by the dependency manager that depend
     *  directly or indirectly on these cells.
     *
     * @property {AddressArray} dirtyFormulaCells
     *  The addresses of all dirty formula cells (cells with changed formula
     *  expression) in the sheet. The cell addresses will be collected, and
     *  will cause to recalculate all formulas, in order to get the correct
     *  results e.g. after generating new cell formulas without the results.
     */
    function SheetDescriptor(sheetModel) {

        this.sheetModel = sheetModel;

        this.formulaMap = new SimpleMap();
        this.cellFormulaSet = new AddressSet();
        this.modelFormulaMaps = new SimpleMap();
        this.missingNamesMap = new SimpleMap();
        this.recalcAlwaysMap = new SimpleMap();
        this.recalcOnceMap = new SimpleMap();

        this.dirtyValueCells = new AddressArray();
        this.dirtyFormulaCells = new AddressArray();

    } // class SheetDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Inserts the passed formula descriptor into the formula caches of this
     * sheet descriptor.
     *
     * @param {String} formulaKey
     *  The formula key of the formula descriptor, used as map key.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The formula descriptor to be inserted into the formula caches.
     */
    SheetDescriptor.prototype.insertFormula = function (formulaKey, formulaDesc) {

        // insert the formula descriptor into the own maps
        this.formulaMap.insert(formulaKey, formulaDesc);
        switch (formulaDesc.recalc) {
            case 'always': this.recalcAlwaysMap.insert(formulaKey, formulaDesc); break;
            case 'once': this.recalcOnceMap.insert(formulaKey, formulaDesc); break;
        }

        // insert the formula into the type-specific containers
        if (formulaDesc.ruleModel) {
            var parentUid = formulaDesc.ruleModel.getUid();
            this.modelFormulaMaps.getOrConstruct(parentUid, SimpleMap).insert(formulaKey, formulaDesc);
        } else {
            var address = formulaDesc.getCellAddress().clone();
            this.cellFormulaSet.insert(address).formula = formulaDesc;
        }

        // map the formula descriptor by missing name labels
        _.each(formulaDesc.missingNames, function (flag, key) {
            var formulaMap = this.missingNamesMap.getOrConstruct(key, SimpleMap);
            formulaMap.insert(formulaKey, formulaDesc);
        }, this);

        DependencyUtils.withLogging(function () {
            DependencyUtils.log('new dependencies for ' + formulaKey + ': ' + formulaDesc.references);
        });
    };

    /**
     * Deletes the specified formula descriptor from the formula caches of this
     * sheet descriptor.
     *
     * @param {String} formulaKey
     *  The formula key of the formula descriptor to be deleted.
     *
     * @returns {FormulaDescriptor|Null}
     *  The formula descriptor deleted from this sheet descriptor, or null.
     */
    SheetDescriptor.prototype.removeFormula = function (formulaKey) {

        // find and remove an existing descriptor registered for this sheet
        var formulaDesc = this.formulaMap.remove(formulaKey);
        if (!formulaDesc) { return null; }

        // remove the formula descriptor from the own maps
        this.recalcAlwaysMap.remove(formulaKey);
        this.recalcOnceMap.remove(formulaKey);
        _.each(formulaDesc.missingNames, function (flag, key) {
            this.missingNamesMap.with(key, function (formulaMap) {
                formulaMap.remove(formulaKey);
                if (formulaMap.empty()) { this.missingNamesMap.remove(key); }
            }, this);
        }, this);

        // remove the formula from the type-specific containers
        if (formulaDesc.ruleModel) {
            var parentUid = formulaDesc.ruleModel.getUid();
            this.modelFormulaMaps.with(parentUid, function (formulaMap) {
                formulaMap.remove(formulaKey);
                if (formulaMap.empty()) {
                    this.modelFormulaMaps.remove(parentUid);
                }
            }, this);
        } else {
            this.cellFormulaSet.remove(formulaDesc.getCellAddress());
        }

        DependencyUtils.log('removed dependencies for ' + formulaKey);
        return formulaDesc;
    };

    /**
     * Clears all cached addresses of dirty cells in the sheet.
     *
     * @returns {SheetDescriptor}
     *  A reference to this instance.
     */
    SheetDescriptor.prototype.clearDirtyCells = function () {
        this.dirtyValueCells.clear();
        this.dirtyFormulaCells.clear();
        return this;
    };

    // exports ================================================================

    return SheetDescriptor;

});
