/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/formuladictionary', [
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils'
], function (SimpleMap, IteratorUtils, SheetUtils, DependencyUtils) {

    'use strict';

    // convenience shortcuts
    var AddressSet = SheetUtils.AddressSet;
    var RangeSet = SheetUtils.RangeSet;

    // class FormulaDictionary ================================================

    /**
     * A container for all registered formula descriptors in a spreadsheet
     * document. The descriptors will be mapped by their source references, in
     * order to provide efficient methods for looking up formulas depending on
     * changed cells, or changed ancestor formulas in a dependency chain.
     *
     * @constructor
     */
    function FormulaDictionary() {

        // several formula descriptor maps attached to addresses in address sets, mapped by sheet UID
        this._cellSets = new SimpleMap();
        // several formula descriptor maps attached to ranges in range sets, mapped by sheet UID
        this._rangeSets = new SimpleMap();
        // formula descriptors depending on a defined name, mapped by name key
        this._nameMaps = new SimpleMap();

    } // class FormulaDictionary

    // private methods --------------------------------------------------------

    /**
     * Inserts a formula descriptor into the passed address set or range set.
     *
     * @param {AddressSet|RangeSet} destSet
     *  The target address set or range set to insert the passed formula
     *  descriptor into.
     *
     * @param {Address|Range} destPos
     *  An address or range used to look up the parent object in the target set
     *  that stores formula descriptors in a custom property 'formulaMap'. If
     *  the set does not contain such an object, this object will be inserted,
     *  and a new formula map will be created. The passed formula descriptor
     *  will be inserted into the formula map of the parent object then.
     *
     * @param {String} formulaKey
     *  The formula key of the passed formula descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The formula descriptor to be inserted into the parent object.
     */
    FormulaDictionary.prototype._insertFormula = function (destSet, destPos, formulaKey, formulaDesc) {
        var parentObj = destSet.insert(destPos);
        var formulaMap = parentObj.formulaMap || (parentObj.formulaMap = new SimpleMap());
        formulaMap.insert(formulaKey, formulaDesc);
    };

    /**
     * Removes a formula descriptor from the passed address set or range set.
     *
     * @param {AddressSet|RangeSet} destSet
     *  The target address set or range set to remove the passed formula
     *  descriptor from.
     *
     * @param {Address|Range} destPos
     *  An address or range to look up the parent object in the target map that
     *  stores formula descriptors in a custom property 'formulaMap'. The
     *  passed formula descriptor will be removed from this formula map. The
     *  parent object will be removed from the passed target set, if its
     *  formula map becomes empty.
     *
     * @param {String} formulaKey
     *  The formula key of the formula descriptor to be removed.
     *
     * @returns {FormulaDescriptor|Null}
     *  The formula descriptor that has been removed from the parent object.
     */
    FormulaDictionary.prototype._removeFormula = function (destSet, destPos, formulaKey) {
        var parentObj = destSet.get(destPos);
        if (!parentObj) { return null; }
        var formulaMap = parentObj.formulaMap;
        var formulaDesc = formulaMap.remove(formulaKey);
        if (formulaMap.empty()) { destSet.remove(parentObj); }
        return formulaDesc;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this dictionary is completely empty.
     *
     * @returns {Boolean}
     *  Whether this dictionary is completely empty.
     */
    FormulaDictionary.prototype.empty = function () {
        return this._cellSets.empty() && this._rangeSets.empty() && this._nameMaps.empty();
    };

    /**
     * Creates a clone of this formula dictionary, referring to the same
     * formula descriptors as this dictionary.
     *
     * @returns {FormulaDictionary}
     *  A clone of this formula dictionary.
     */
    FormulaDictionary.prototype.clone = function () {

        // start with an empty dictionary
        var clone = new FormulaDictionary();

        // clone the formula maps of the cell address sets
        clone._cellSets = this._cellSets.clone(function (cellSet) {
            return cellSet.clone(function (newAddress, oldAddress) {
                newAddress.formulaMap = oldAddress.formulaMap.clone();
            });
        });

        // clone the formula maps of the cell range sets
        clone._rangeSets = this._rangeSets.clone(function (rangeSet) {
            return rangeSet.clone(function (newRange, oldRange) {
                newRange.formulaMap = oldRange.formulaMap.clone();
            });
        });

        // clone the formula descriptors mapped by name keys
        clone._nameMaps = this._nameMaps.clone(function (formulaMap) {
            return formulaMap.clone();
        });

        return clone;
    };

    /**
     * Inserts a new formula descriptor into this dictionary. The formula
     * descriptor will be mapped by all of its source references.
     *
     * @param {String} formulaKey
     *  The key of the passed formula descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The descriptor of a formula to be inserted into this dictionary.
     */
    FormulaDictionary.prototype.insertFormula = function (formulaKey, formulaDesc) {

        // insert the formula descriptor into the formula maps for all source ranges
        formulaDesc.references.forEachRange(function (range, sheetUid) {
            if (range.cells() <= DependencyUtils.SINGLE_CELL_THRESHOLD) {
                var cellSet = this._cellSets.getOrConstruct(sheetUid, AddressSet);
                IteratorUtils.forEach(range.iterator(), function (address) {
                    this._insertFormula(cellSet, address, formulaKey, formulaDesc);
                }, this);
            } else {
                var rangeSet = this._rangeSets.getOrConstruct(sheetUid, RangeSet);
                this._insertFormula(rangeSet, range, formulaKey, formulaDesc);
            }
        }, this);

        // insert the formula descriptor into the formula maps for all defined names
        formulaDesc.references.forEachNameKey(function (nameKey) {
            this._nameMaps.getOrConstruct(nameKey, SimpleMap).insert(formulaKey, formulaDesc);
        }, this);
    };

    /**
     * Removes a formula descriptor from this dictionary.
     *
     * @param {String} formulaKey
     *  The key of the passed formula descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The descriptor of a formula to be removed form this dictionary.
     */
    FormulaDictionary.prototype.removeFormula = function (formulaKey, formulaDesc) {

        // insert the formula descriptor into the formula maps for all source ranges
        formulaDesc.references.forEachRange(function (range, sheetUid) {
            if (range.cells() <= DependencyUtils.SINGLE_CELL_THRESHOLD) {
                this._cellSets.with(sheetUid, function (cellSet) {
                    IteratorUtils.forEach(range.iterator(), function (address) {
                        this._removeFormula(cellSet, address, formulaKey);
                    }, this);
                }, this);
            } else {
                this._rangeSets.with(sheetUid, function (rangeSet) {
                    this._removeFormula(rangeSet, range, formulaKey);
                }, this);
            }
        }, this);

        // remove the formula descriptor from the formula maps for all defined names
        formulaDesc.references.forEachNameKey(function (nameKey) {
            this._nameMaps.with(nameKey, function (formulaMap) {
                formulaMap.remove(formulaKey);
                if (formulaMap.empty()) { this._nameMaps.remove(nameKey); }
            }, this);
        }, this);
    };

    /**
     * Returns the descriptors of all formulas that depend directly on the
     * specified cell address.
     *
     * @param {String} sheetUid
     *  The UID of the sheet containing the cell with the passed address.
     *
     * @param {Address} address
     *  The address of the cell to look up formula descriptors for.
     *
     * @returns {SimpleMap}
     *  All formula descriptors depending on the specified cell address, mapped
     *  by formula key.
     */
    FormulaDictionary.prototype.findFormulasForAddress = function (sheetUid, address) {

        // the collected formula descriptors
        var formulaMap = new SimpleMap();

        // extends the formula map with the formulas depending on the passed parent object
        function extendFormulaMap(parentObj) {
            formulaMap.merge(parentObj.formulaMap);
        }

        // collect formula descriptors depending directly on the passed key
        this._cellSets.with(sheetUid, function (cellSet) {
            var address2 = cellSet.get(address);
            if (address2) { extendFormulaMap(address2); }
        }, this);

        // collect formula descriptors depending on large cell ranges
        this._rangeSets.with(sheetUid, function (rangeSet) {
            rangeSet.findByAddress(address).forEach(extendFormulaMap);
        }, this);

        return formulaMap;
    };

    /**
     * Returns the descriptors of all formulas that depend directly on any cell
     * in the specified cell range address.
     *
     * @param {String} sheetUid
     *  The UID of the sheet containing the cell range with the passed address.
     *
     * @param {Range} range
     *  The address of the cell range to look up formula descriptors for.
     *
     * @returns {SimpleMap}
     *  All formula descriptors depending on the specified cell range address,
     *  mapped by formula key.
     */
    FormulaDictionary.prototype.findFormulasForRange = function (sheetUid, range) {

        // the collected formula descriptors
        var formulaMap = new SimpleMap();

        // extends the formula map with the formulas depending on the passed parent object
        function extendFormulaMap(parentObj) {
            formulaMap.merge(parentObj.formulaMap);
        }

        // collect formula descriptors depending directly on the passed key
        this._cellSets.with(sheetUid, function (cellSet) {
            cellSet.findAddresses(range).forEach(extendFormulaMap);
        }, this);

        // collect formula descriptors depending on large cell ranges
        this._rangeSets.with(sheetUid, function (rangeSet) {
            rangeSet.findRanges(range).forEach(extendFormulaMap);
        }, this);

        return formulaMap;
    };

    /**
     * Returns the descriptors of all formulas that depend directly on the
     * defined name with the specified key.
     *
     * @param {String} nameKey
     *  The key of the defined name to look up formula descriptors for.
     *
     * @returns {SimpleMap}
     *  All formula descriptors depending on the specified defined name, mapped
     *  by formula key.
     */
    FormulaDictionary.prototype.findFormulasForName = function (nameKey) {
        return this._nameMaps.get(nameKey) || new SimpleMap();
    };

    // exports ================================================================

    return FormulaDictionary;

});
