/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/formuladescriptor', [
    'io.ox/office/spreadsheet/model/formula/deps/referencecollection'
], function (ReferenceCollection) {

    'use strict';

    // class FormulaDescriptor ================================================

    /**
     * A descriptor object for a token array in a spreadsheet document, that
     * contains all its dependency settings.
     *
     * @constructor
     *
     * @property {String} sheetUid
     *  The UID of the sheet model containing the formula expression.
     *
     * @property {TokenArray} tokenArray
     *  The token array represented by this descriptor object.
     *
     * @property {RuleModel|Null} ruleModel
     *  For cell formulas, this property will be null (will be used to detect
     *  cell formulas); for conditions of formtting rules, this will be the
     *  parent rule model containing the formula.
     *
     * @property {ReferenceCollection} references
     *  A collection with source references the token array depends on
     *  directly. If this collection is empty, the token array does not depend
     *  on any other external object in the spreadsheet document.
     *
     * @property {Object} missingNames
     *  A flag set with the lower-case labels of all missing defined names (as
     *  object keys) occurring in the formula expression, or in any existing
     *  defined name referred by the formula.
     *
     * @property {String|Null} recalc
     *  The recalculation mode of the formula expression, as deduced from the
     *  recalculation modes of the functions: Either 'always', 'once', or null.
     *
     * @property {Boolean} circular
     *  If set to true, a circular dependency has been detected for the formula
     *  represented by this descriptor. In this case, resolving the dependency
     *  chain for this formula will be stopped to prevent endless loops.
     */
    function FormulaDescriptor(docModel, sheetUid, tokenArray, refAddress, ruleModel) {

        // public properties
        this.sheetUid = sheetUid;
        this.tokenArray = tokenArray;
        this.ruleModel = ruleModel || null;
        this.references = null;
        this.missingNames = null;
        this.recalc = null;
        this.circular = false;

        // private properties
        this._docModel = docModel;
        this._refAddress = refAddress;

        // initialization -----------------------------------------------------

        // calculate the initial dependencies, and create the reference collection
        this.refreshReferences();

    } // class FormulaDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the formula represented by this descriptor does not have
     * any external dependencies, and therefore does not need to be handled by
     * the dependency manager.
     *
     * @returns {Boolean}
     *  Whether the formula represented by this descriptor does not have any
     *  external dependencies.
     */
    FormulaDescriptor.prototype.isFixed = function () {
        return !this.recalc && _.isEmpty(this.missingNames) && this.references.empty();
    };

    /**
     * Recalculates the dependency setings for the formula represented by this
     * descriptor.
     */
    FormulaDescriptor.prototype.refreshReferences = function () {

        // get the target position for resolving the dependencies
        var depTarget = this.ruleModel ? this.ruleModel.getDependencyTarget() : this.getCellAddress();

        // resolve the dependencies of the token array (cell ranges, and defined names)
        var dependencies = this.tokenArray.getDependencies(this._refAddress, depTarget);
        this.references = new ReferenceCollection(this._docModel, dependencies);

        // copy the other settings
        this.circular = dependencies.circular;
        this.recalc = dependencies.recalc;
        this.missingNames = dependencies.missingNames;
    };

    /**
     * Returns the reference address of the formula, i.e. the address of the
     * cell the relative references are associated to. For cell formulas, this
     * is the address of the cell containing the formula. For shared formulas,
     * for conditional formatting rules, and for data validation rules, this is
     * the top-left cell of the bounding range of the parent object.
     *
     * @returns {Address}
     *  The reference address of the formula.
     */
    FormulaDescriptor.prototype.getRefAddress = function () {
        return this._refAddress;
    };

    /**
     * Returns the target address of a cell formula, i.e. the address of the
     * cell the formula is associated with.
     *
     * @returns {Address|Null}
     *  The target address of a cell formula; or null for other types of
     *  formulas, e.g. conditions of formatting rules.
     */
    FormulaDescriptor.prototype.getCellAddress = function () {
        return this.ruleModel ? null : this._refAddress;
    };

    // exports ================================================================

    return FormulaDescriptor;

});
