/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/dependencyutils', [
    'io.ox/office/tk/utils/logger'
], function (Logger) {

    'use strict';

    // static class DependencyUtils ===========================================

    var DependencyUtils = {};

    // constants --------------------------------------------------------------

    /**
     * Duration for a single time slice of the background update task of the
     * dependency manager (in milliseconds).
     *
     * @constant
     */
    DependencyUtils.TIMER_SLICE = 150;

    /**
     * Duration between two time slices of the background update task of the
     * dependency manager (in milliseconds).
     *
     * @constant
     */
    DependencyUtils.TIMER_SLICE_DELAY = 100;

    /**
     * Duration for a single time slice of the background update task of the
     * dependency manager for high-priority mode (in milliseconds).
     *
     * @constant
     */
    DependencyUtils.PRIORITY_TIMER_SLICE = 200;

    /**
     * Duration between two time slices of the background update task of the
     * dependency manager for high-priority mode (in milliseconds).
     *
     * @constant
     */
    DependencyUtils.PRIORITY_TIMER_SLICE_DELAY = 50;

    /**
     * Delay for starting the background update task of the dependency manager
     * after receiving a document event (in milliseconds).
     *
     * @constant
     */
    DependencyUtils.UPDATE_DEBOUNCE_DELAY = 100;

    /**
     * Additional delay for the first background update task of the dependency
     * manager after import (in milliseconds).
     *
     * @constant
     */
    DependencyUtils.IMPORT_UPDATE_DELAY = 900;

    /**
     * The maximum number of cells in a cell range to manage the dependencies
     * of these cells separately instead in a range set, for performance.
     *
     * @constant
     */
    DependencyUtils.SINGLE_CELL_THRESHOLD = 16;

    // logger interface -------------------------------------------------------

    // debug logger for the dependency manager
    Logger.extend(DependencyUtils, { enable: 'spreadsheet:log-deps', prefix: 'DEP' });

    // static methods ---------------------------------------------------------

    /**
     * Returns the cell key for the specified sheet UID and cell address.
     *
     * @param {String} sheetUid
     *  The UID of a sheet in a spreadsheet document.
     *
     * @param {Address} address
     *  The cell address to create a cell key for.
     *
     * @returns {String}
     *  A unique map key for the specified cell.
     */
    DependencyUtils.getCellKey = function (sheetUid, address) {
        return sheetUid + '!' + address.key();
    };

    /**
     * Returns a readable label for the passed defined name for debug logging.
     *
     * @param {NameModel} nameModel
     *  The model instance of a defined name.
     *
     * @returns {String}
     *  A readable label for the passed defined name for debug logging.
     */
    DependencyUtils.getNameLabel = function (nameModel) {
        var sheetModel = nameModel.getSheetModel();
        return 'name:' + (sheetModel ? (sheetModel.getName() + '!') : '') + nameModel.getLabel();
    };

    /**
     * Returns a readable label for the passed formatting rule for debug
     * logging.
     *
     * @param {RuleModel} ruleModel
     *  The model instance of a formatting rule.
     *
     * @returns {String}
     *  A readable label for the passed formatting rule for debug logging.
     */
    DependencyUtils.getRuleLabel = function (ruleModel) {
        return 'rule:' + ruleModel.getSheetModel().getName() + '!' + ruleModel.getUid();
    };

    // exports ================================================================

    return DependencyUtils;

});
