/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/dependencymanager', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/simplemap',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils',
    'io.ox/office/spreadsheet/model/formula/deps/formuladescriptor',
    'io.ox/office/spreadsheet/model/formula/deps/sheetdescriptor',
    'io.ox/office/spreadsheet/model/formula/deps/formuladictionary',
    'io.ox/office/spreadsheet/utils/config'
], function (Utils, SimpleMap, IteratorUtils, BaseObject, TriggerObject, TimerMixin, SheetUtils, FormulaUtils, DependencyUtils, FormulaDescriptor, SheetDescriptor, FormulaDictionary, Config) {

    'use strict';

    // convenience shortcuts
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;

    // general options for background timers, for first-level loops (mapped by priority mode)
    var OUTER_TIMER_OPTIONS_MAP = {
        low:  { slice: DependencyUtils.TIMER_SLICE, interval: DependencyUtils.TIMER_SLICE_DELAY },
        high: { slice: DependencyUtils.PRIORITY_TIMER_SLICE, interval: DependencyUtils.PRIORITY_TIMER_SLICE_DELAY }
    };

    // general options for background timers, for emebdded loops (mapped by priority mode)
    var NESTED_TIMER_OPTIONS_MAP = {
        low:  _.extend({ delay: 'immediate' }, OUTER_TIMER_OPTIONS_MAP.low),
        high: _.extend({ delay: 'immediate' }, OUTER_TIMER_OPTIONS_MAP.high)
    };

    // class TimerHelper ======================================================

    /**
     * A small helper class to execute background loops with different timing
     * settings.
     *
     * @constructor
     *
     * @extends BaseObject
     * @extends TimerMixin
     */
    var TimerHelper = BaseObject.extend({ constructor: function () {

        // the number of nested background loops currently running
        var loopDepth = 0;

        // whether to run the background loops with higher priority
        var priority = 'low';

        // base constructors --------------------------------------------------

        BaseObject.call(this);
        TimerMixin.call(this);

        // public methods -----------------------------------------------------

        /**
         * Changes the execution priority of the background loops.
         *
         * @param {String} newPriority
         *  The new priority mode. Must be one of 'low' or 'high'.
         *
         * @returns {TimerHelper}
         *  A reference to this instance.
         */
        this.setPriority = function (newPriority) {
            priority = newPriority;
            return this;
        };

        /**
         * Processes the passed iterator in a background loop.
         *
         * @param {Object} iterator
         *  The iterator to be processed.
         *
         * @param {Function} callback
         *  The callback function invoked for every iterator value.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise representing the background loop.
         */
        this.iterate = function (iterator, callback, context) {
            var options = ((loopDepth === 0) ? OUTER_TIMER_OPTIONS_MAP : NESTED_TIMER_OPTIONS_MAP)[priority];
            loopDepth += 1;
            return this.iterateSliced(iterator, callback.bind(context), options).always(function () { loopDepth -= 1; });
        };

        /**
         * Invokes the passed callback function forever in a background loop.
         *
         * @param {Function} callback
         *  The callback function invoked repeatedly. Receives the zero-based
         *  index as first parameter. The return value Utils.BREAK will exit
         *  the loop.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise representing the background loop.
         */
        this.loop = function (callback, context) {
            var iterator = IteratorUtils.createIndexIterator(Number.POSITIVE_INFINITY);
            return this.iterate(iterator, callback, context);
        };

    } }); // class TimerHelper

    // class PendingCollection ================================================

    /**
     * Encapsulates all data that will be collected by a dependency manager
     * when processing document change events. Will be used to find and update
     * all dirty formulas depending on the data collected in an instance of
     * this class.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @property {Boolean} recalcAll
     *  Specifies whether the next update cycle has to recalculate all existing
     *  formulas, instead of just the dirty formulas according to the collected
     *  data in this instance.
     *
     * @property {Boolean} recalcVolatile
     *  Specifies whether the next update cycle has to recalculate all volatile
     *  formulas, also if they do not depend on any changed cell.
     *
     * @property {Array<Object>} documentChanges
     *  A queue of change events received from the document model, that need to
     *  be processed by the dependency manager. This array contains objects
     *  with a sheet model, and specific data according to the change event
     *  (e.g. changed cells, changed formula expressions, or moved cells).
     *
     * @property {SimpleMap<NameModel>} changedNamesMap
     *  A map with the model instances of all dirty defined names. After
     *  inserting a defined name, or changing its formula expression, all
     *  depending formulas need to be recalculated.
     *
     * @property {SimpleMap<NameModel>} changedLabelsMap
     *  A map with the model instances of all defined names with changed label.
     *  After inserting a defined name, or changing its label, all formulas
     *  that contain unresolved names need to be recalculated in order to ged
     *  rid of #NAME? errors.
     */
    var PendingCollection = TriggerObject.extend({ constructor: function () {

        TriggerObject.call(this);

        this.recalcAll = false;
        this.recalcVolatile = true;
        this.documentChanges = [];
        this.changedNamesMap = new SimpleMap();
        this.changedLabelsMap = new SimpleMap();

    } }); // class PendingCollection

    // public methods ---------------------------------------------------------

    /**
     * Resets all collected data of this instance.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.clear = function () {
        this.recalcAll = false;
        this.recalcVolatile = true;
        this.documentChanges = [];
        this.changedNamesMap.clear();
        this.changedLabelsMap.clear();
        return this;
    };

    /**
     * Registers a new document change event, and notifies the own listeners
     * with a 'change' event.
     *
     * @param {String} type
     *  An internal type identifier of the change event.
     *
     * @param {SheetModel|Null} sheetModel
     *  The model of a sheet associated to the change event, or null for global
     *  document events.
     *
     * @param {Object} [properties]
     *  Additional properties depending on the event type.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.registerDocumentChange = function (type, sheetModel, properties, options) {
        var index = Utils.getIntegerOption(options, 'index', this.documentChanges.length);
        this.documentChanges.splice(index, 0, _.extend({ type: type, sheetModel: sheetModel }, properties));
        return this.trigger('change');
    };

    /**
     * Deletes all pending document change events that pass a truth test, and
     * notifies the own listeners with a 'change' event, if at least one
     * pending document change event has been deleted.
     *
     * @param {Function} predicate
     *  A predicate function that receives a document change descriptor, and
     *  must return whether to delete this event from the queue.
     *
     * @param {Object} [context]
     *  The calling context for the predicate function.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.deleteDocumentChanges = function (predicate, context) {
        var oldLength = this.documentChanges.length;
        this.documentChanges = this.documentChanges.filter(function (docChange) {
            return !predicate.call(context, docChange);
        });
        if (oldLength !== this.documentChanges.length) { this.trigger('change'); }
        return this;
    };

    /**
     * Registers a defined name in this collection that has been inserted into
     * the spreadsheet document, or that has been changed (either the formula
     * expression, or the label).
     *
     * @param {NameModel} nameModel
     *  The model instance of the new or changed defined name.
     *
     * @param {Boolean} formulaChanged
     *  Whether the formula expression of the defined name has been changed.
     *  MUST be set to true for new defined names inserted into the document.
     *
     * @param {Boolean} labelChanged
     *  Whether the label of the defined name has been changed. MUST be set to
     *  true for new defined names inserted into the document.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.registerNameModel = function (nameModel, formulaChanged, labelChanged) {
        if (formulaChanged) { this.changedNamesMap.insert(nameModel.getUid(), nameModel); }
        if (labelChanged) { this.changedLabelsMap.insert(nameModel.getUid(), nameModel); }
        return this;
    };

    /**
     * Unregisters a defined name from this collection that will be deleted
     * from the spreadsheet document.
     *
     * @param {NameModel} nameModel
     *  The model instance of the defined name that is about to be deleted.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.unregisterNameModel = function (nameModel) {
        this.changedNamesMap.insert(nameModel.getUid(), nameModel);
        this.changedLabelsMap.remove(nameModel.getUid());
        return this;
    };

    /**
     * Unregisters all sheet-locally defined names of the specified sheet model
     * from this collection.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the sheet whose defined names will be removed
     *  from this collection.
     *
     * @returns {PendingCollection}
     *  A reference to this instance.
     */
    PendingCollection.prototype.unregisterAllNameModels = function (sheetModel) {
        var iterator = sheetModel.getNameCollection().createModelIterator();
        IteratorUtils.forEach(iterator, this.unregisterNameModel, this);
        return this;
    };

    // class ResultCollection =================================================

    /**
     * Encapsulates all data that will be collected during a single update
     * cycle of a dependency manager.
     *
     * @constructor
     *
     * @property {SimpleMap<FormulaDescriptor>} formulaMap
     *  All dirty formula descriptors collected by an update cycle of a
     *  dependency manager. This map contains all formula descriptors contained
     *  in the property map 'cellFormulaMap', and all formula descriptors that
     *  have caused to add the target ranges map of a formatting rule to the
     *  property 'ruleRangesMap'.
     *
     * @property {SimpleMap<FormulaDescriptor>} cellFormulaMap
     *  The descriptors of all dirty cell formulas collected by an update cycle
     *  of a dependency manager. All formulas in this map will be calculated
     *  while the calculation of another formula looks up the cell contents of
     *  that dirty formulas. After calculation of a formula has finished, the
     *  formula result will be stored in the property 'resultMaps', and will be
     *  returned when looking up the cell value the next time.
     *
     * @property {SimpleMap<RangeArray>} ruleRangesMap
     *  The target ranges of all dirty formatting rules collected by an update
     *  cycle of a dependency manager, mapped by sheet UID. These target ranges
     *  will be repainted after the update cycle has been completed.
     *
     * @property {SimpleMap<SimpleMap<Object>>} resultMaps
     *  The results of all dirty formulas that have been calculated, mapped as
     *  submaps by sheet UID. Each submap contains the interpreter result
     *  objects and cell addresses, mapped by cell key. Looking up the results
     *  of dirty formula cells will first check if this map contains a
     *  calculated result. If not, the formula will be calculated, and the
     *  result will be stored in this map.
     *
     * @property {Array<Array<FormulaDescriptor>>} referenceCycles
     *  All formula cycles found during calculation of formulas. Each element
     *  in the array is another array with formula descriptors representing the
     *  reference cycle.
     *
     * @property {Boolean} hasErrors
     *  Whether any of the formulas could not be calculated correctly, e.g. due
     *  to unimplemented functions.
     */
    function ResultCollection() {

        this.formulaMap = new SimpleMap();
        this.cellFormulaMap = new SimpleMap();
        this.ruleRangesMap = new SimpleMap();
        this.resultMaps = new SimpleMap();
        this.referenceCycles = [];
        this.hasErrors = false;

    } // class ResultCollection

    // public methods ---------------------------------------------------------

    /**
     * Inserts a new formula descriptor into the internal collections of this
     * instance.
     *
     * @param {String} formulaKey
     *  The unique key of the formula descriptor.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The descriptor for the new formula to be registered.
     *
     * @returns {ResultCollection}
     *  A reference to this instance.
     */
    ResultCollection.prototype.insertFormula = function (formulaKey, formulaDesc) {
        this.formulaMap.insert(formulaKey, formulaDesc);
        if (formulaDesc.ruleModel) {
            this.ruleRangesMap.getOrConstruct(formulaDesc.sheetUid, RangeArray).append(formulaDesc.ruleModel.getTargetRanges());
        } else {
            this.cellFormulaMap.insert(formulaKey, formulaDesc);
        }
        return this;
    };

    // class DependencyManager ================================================

    /**
     * Collects all change events in the document, and recalculates all formula
     * expressions depending on these changes.
     *
     * Triggers the following events:
     * - 'recalc:start'
     *      When the background task for calculating the dependent formulas has
     *      started.
     * - 'recalc:end'
     *      When the background task for calculating the dependent formulas has
     *      been finished successfully. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event}
     *          The jQuery event object.
     *      (2) {Array<AddressArray>} referenceCycles
     *          An array of arrays of cell addresses with sheet UIDs. Each of
     *          the inner arrays represents a reference cycle found during
     *          calculation of dirty formulas.
     * - 'recalc:cancel'
     *      When the background task for calculating the dependent formulas has
     *      been aborted due to new document changes.
     * - 'recalc:overflow'
     *      Will be triggered once, if the total number of formulas in the
     *      document exceeds a specific limit in the server configuration.
     *      After that, the dependency manager will not process any formulas in
     *      the document anymore.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    var DependencyManager = TriggerObject.extend({ constructor: function (docModel) {

        // private properties -------------------------------------------------

        // self reference
        var self = this;

        // the application containing this dependency manager
        var app = docModel.getApp();

        // A helper instance to execute background timer loops with different
        // priorities (interval durations).
        //
        var timerHelper = new TimerHelper();

        // Stores the descriptors of all registered formulas in the document.
        //
        // The formula descriptors (instances of the class FormulaDescriptor)
        // will be mapped by their formula keys.
        //
        var formulaDescMap = new SimpleMap();

        // The total number of all registered formulas currently contained in
        // the spreadsheet document.
        //
        var formulaCount = 0;

        // Stores a sheet descriptor object per existing sheet in the document.
        //
        // The sheet descriptors (instances of the class SheetDescriptor) will
        // be mapped by the UID of the sheet model.
        //
        var sheetDescMap = new SimpleMap();

        // A dictionary for all formula descriptors of this dependency manager.
        //
        // This is the central storage of this dependency manager that will be
        // used to find formulas depending on specific dirty cells, or other
        // changed document contents, in a very efficient way.
        //
        var formulaDictionary = new FormulaDictionary();

        // Collected data for all pending document change events, and other
        // information needed to decide which formulas need to be recalculated
        // in the next update cycle.
        //
        var pendingCollection = new PendingCollection();

        // All dirty formula descriptors collected by an update cycle of this
        // dependency manager, and the formula results already calculated.
        //
        var resultCollection = null;

        // Whether this dependency manager did not finish its initial update
        // cycle after importing the document yet.
        //
        // Will be set to false, after the first update cycle has finished
        // successfully.
        //
        var initialUpdateCycle = true;

        // Specifies whether the initial update cycle needs to recalculate all
        // formulas in the document.
        //
        // This flag will be initialized after importing the document, and will
        // be reset after the first successful update cycle.
        //
        var calcOnLoad = false;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private methods ----------------------------------------------------

        /**
         * Creates a new operations generator, and invokes the passed callback
         * function. If the spreadsheet document is in edit mode, the generated
         * operations will be sent to the server.
         *
         * @param {Function} callback
         *  The callback function. Receives the operations generator as first
         *  parameter.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         */
        function generateAndSendOperations(callback, context) {

            // create a new generator that applies the generated operations immediately
            var generator = docModel.createOperationsGenerator({ applyImmediately: true });

            // invoke the callback function with the new generator
            callback.call(context, generator);

            // send the operations to the server, but only if the document is in edit mode
            if (docModel.getEditMode()) {
                docModel.sendOperations(generator, { defer: !app.isLocallyModified() });
            }
        }

        /**
         * Sends the 'calcOnLoad' document flag to the server, if it has not
         * been set yet.
         */
        function setCalcOnLoadFlag() {
            if (!calcOnLoad) {
                generateAndSendOperations(function (generator) {
                    generator.generateDocAttrsOperation({ calcOnLoad: true });
                });
                calcOnLoad = true;
            }
        }

        /**
         * Inserts a new formula descriptor into all collections of this
         * dependency manager.
         *
         * @param {String} formulaKey
         *  The unique key of the formula descriptor.
         *
         * @param {FormulaDescriptor} formulaDesc
         *  The descriptor for the new formula to be registered.
         *
         * @returns {Boolean}
         *  Whether the formula descriptor has been inserted into the internal
         *  formula collections. If the return value is false, this method has
         *  triggered a 'recalc:overflow' event, and has stopped the entire
         *  recalculation cycle.
         */
        function insertFormula(formulaKey, formulaDesc) {

            // update the total number of formulas
            formulaCount += 1;

            // If the maximum number of formulas has been exceeded, detach this dependency manager
            // from all document model events (this prevents processing any further edit actions),
            // trigger the 'recalc:overflow' event (this will abort the update cycle currently
            // running), and set the 'calcOnLoad' document flag.
            if (formulaCount > Config.MAX_FORMULA_COUNT) {
                self.stopListeningTo(docModel);
                pendingCollection.clear();
                self.trigger('recalc:overflow');
                setCalcOnLoadFlag();
                return false;
            }

            // register the formula descriptor in the global map
            formulaDescMap.insert(formulaKey, formulaDesc);

            // insert the formula descriptor into the sheet descriptor maps
            sheetDescMap.get(formulaDesc.sheetUid).insertFormula(formulaKey, formulaDesc);

            // insert the formula descriptor into the dictionary
            formulaDictionary.insertFormula(formulaKey, formulaDesc);

            return true;
        }

        /**
         * Removes a formula descriptor from all collections of this dependency
         * manager.
         *
         * @param {String} formulaKey
         *  The unique key of the formula descriptor.
         *
         * @returns {FormulaDescriptor|Null}
         *  The formula descriptor that has been removed from the collections;
         *  or null, if no formula descriptor was registered for the passed
         *  formula key.
         */
        function removeFormula(formulaKey) {

            // remove the formula descriptor from the global map
            var formulaDesc = formulaDescMap.remove(formulaKey);
            if (!formulaDesc) { return null; }

            // update the total number of formulas in this dependency manager, if the formula exists
            formulaCount -= 1;

            // remove the formula descriptor from the sheet descriptor maps (check existence,
            // sheet descriptor has been removed already while deleting a sheet in the document)
            sheetDescMap.with(formulaDesc.sheetUid, function (sheetDesc) {
                sheetDesc.removeFormula(formulaKey);
            });

            // remove the formula descriptor from the dictionary
            formulaDictionary.removeFormula(formulaKey, formulaDesc);
            return formulaDesc;
        }

        /**
         * Refreshes the dependency settings of a formula descriptor that is
         * registered for this dependency manager.
         *
         * @param {String} formulaKey
         *  The unique key of the formula descriptor.
         *
         * @returns {FormulaDescriptor|Null}
         *  The formula descriptor that has been refreshed; or null, if no
         *  formula descriptor was registered for the passed formula key.
         */
        function refreshFormula(formulaKey) {

            // find and remove an existing formula descriptor
            var formulaDesc = removeFormula(formulaKey);
            if (!formulaDesc) { return null; }

            // refresh the source dependencies, and reinsert the descriptor
            formulaDesc.refreshReferences();
            insertFormula(formulaKey, formulaDesc);
            return formulaDesc;
        }

        /**
         * Registers the addresses of cells that contain a changed value.
         *
         * @param {SheetDescriptor} sheetDesc
         *  The descriptor of the sheet containing the changed cells.
         *
         * @param {AddressArray} addresses
         *  The addresses of all changed cells.
         */
        var registerValueCells = DependencyUtils.profileMethod('registerValueCells()', function (sheetDesc, addresses) {
            sheetDesc.dirtyValueCells.append(addresses);
        });

        /**
         * Registers the addresses of cells that contain a changed formula
         * expression.
         *
         * @param {SheetDescriptor} sheetDesc
         *  The descriptor of the sheet containing the changed cells.
         *
         * @param {AddressArray} addresses
         *  The addresses of all changed cells.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.update=false]
         *      If set to true, this method will update an existing formula
         *      descriptor, otherwise it will insert a new formula descriptor.
         *  @param {Boolean} [options.recalc=false]
         *      If set to true, the registered formulas will be recalculated
         *      (will not be done e.g. after importing the document).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the formula cells have been
         *  processed completely.
         */
        var registerFormulaCells = DependencyUtils.profileAsyncMethod('registerFormulaCells()', function (sheetDesc, iterator, options) {

            // the unique identifier of the sheet, used as map key
            var sheetUid = sheetDesc.sheetModel.getUid();
            // the cell collection of the sheet
            var cellCollection = sheetDesc.sheetModel.getCellCollection();
            // whether to update an existing formula descriptor
            var update = Utils.getBooleanOption(options, 'update', false);
            // whether to recalculate the formula
            var recalc = Utils.getBooleanOption(options, 'recalc', false);

            return timerHelper.iterate(iterator, function (address) {

                // the cell key will be used as formula descriptor key
                var formulaKey = DependencyUtils.getCellKey(sheetUid, address);

                // first, remove an existing formula descriptor from the dictionary
                if (update) { removeFormula(formulaKey); }

                // nothing more to do, if there is no token array available for the key (e.g. deleted cell formulas)
                var tokenArray = cellCollection.getCellTokenArray(address);
                if (!tokenArray) { return; }

                // formula cells without a valid result value must be recalculated
                recalc = recalc || (cellCollection.getCellValue(address) === null);

                // create a new descriptor for the token array (but ignore it if it does not contain any dependencies)
                var formulaDesc = new FormulaDescriptor(docModel, sheetUid, tokenArray, address);
                if (recalc || !formulaDesc.isFixed()) {
                    // immediately escape the loop, if the number of formulas exceeds the configuration limit
                    if (!insertFormula(formulaKey, formulaDesc)) { return Utils.BREAK; }
                }

                // register the new formula for recalculation in the next update cycle
                if (recalc) { sheetDesc.dirtyFormulaCells.push(address); }
            });
        });

        /**
         * Updates the document event queue after cells have been moved in the
         * document.
         *
         * @param {SheetDescriptor} sheetDesc
         *  The descriptor of the sheet containing the moved cells.
         *
         * @param {MoveDescriptor} moveDesc
         *  A descriptor that specifies how the cells have been moved.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the move descriptor has been
         *  processed completely.
         */
        var registerMovedCells = DependencyUtils.profileAsyncMethod('registerMovedCells()', function (sheetDesc, moveDesc) {

            // the unique identifier of the sheet, used as map key
            var sheetUid = sheetDesc.sheetModel.getUid();
            // the cell collection of the sheet
            var cellCollection = sheetDesc.sheetModel.getCellCollection();

            // move all cell formula descriptors to their new positions (collect all addresses before
            // updating the formulas, an iterator is not stable against modification of the set)
            var oldAddresses = sheetDesc.cellFormulaSet.findAddresses(moveDesc.dirtyRange);
            // transform the addresses
            var newAddresses = moveDesc.transformAddresses(oldAddresses);

            // first, remove all formula descriptors (mixed remove/insert may overwrite old descriptors!)
            var promise = timerHelper.iterate(oldAddresses.iterator(), function (address) {
                removeFormula(DependencyUtils.getCellKey(sheetUid, address));
            });

            // insert all remaining formula descriptors
            promise = promise.then(function () {
                return timerHelper.iterate(newAddresses.iterator(), function (address) {

                    // get the token array from the new formula position
                    var tokenArray = cellCollection.getCellTokenArray(address);
                    if (!tokenArray) { return; }

                    // create a new descriptor for the token array
                    var formulaDesc = new FormulaDescriptor(docModel, sheetUid, tokenArray, address);

                    // immediately escape the loop, if the number of formulas exceeds the configuration limit
                    if (!insertFormula(DependencyUtils.getCellKey(sheetUid, address), formulaDesc)) {
                        return Utils.BREAK;
                    }
                });
            });

            return promise;
        });

        /**
         * Registers a new or changed formatting rule.
         *
         * @param {RuleModel}
         *  The model instance of the new or changed formatting rule.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.update=false]
         *      If set to true, this method will update an existing formula
         *      descriptor, otherwise it will insert a new formula descriptor.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the formatting rule has been
         *  processed completely.
         */
        var registerRuleModel = DependencyUtils.profileAsyncMethod('registerRuleModel()', function (ruleModel, options) {

            // remove old settings when updating an existing formula descriptor
            if (Utils.getBooleanOption(options, 'update', false)) {
                unregisterRuleModel(ruleModel);
            }

            var sheetUid = ruleModel.getSheetModel().getUid();
            var ruleUid = ruleModel.getUid();
            var refAddress = ruleModel.getRefAddress();

            return timerHelper.iterate(ruleModel.createTokenArrayIterator(), function (tokenArray, result) {
                var formulaDesc = new FormulaDescriptor(docModel, sheetUid, tokenArray, refAddress, ruleModel);
                if (!formulaDesc.isFixed()) { insertFormula(ruleUid + '!' + result.key, formulaDesc); }
            });
        });

        /**
         * Unregisters a deleted formatting rule.
         *
         * @param {RuleModel}
         *  The model instance of the formatting rule about to be deleted.
         */
        var unregisterRuleModel = DependencyUtils.profileMethod('unregisterRuleModel()', function (ruleModel) {

            var sheetDesc = sheetDescMap.get(ruleModel.getSheetModel().getUid());
            sheetDesc.modelFormulaMaps.with(ruleModel.getUid(), function (formulaMap) {
                formulaMap.forEach(function (formulaDesc, formulaKey) {
                    removeFormula(formulaKey);
                });
            });
        });

        /**
         * Updates the settings of this dependency manager for a new sheet in
         * the spreadsheet document.
         *
         * @param {SheetModel} sheetModel
         *  The model instance of the new sheet.
         *
         * @param {Boolean} [recalc=false]
         *  Whether to recalculate all formulas in the new sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the sheet model has been
         *  processed completely.
         */
        var registerSheetModel = DependencyUtils.profileAsyncMethod('registerSheetModel()', function (sheetModel, recalcAll) {

            // create the sheet descriptor object for the new sheet
            var sheetDesc = new SheetDescriptor(sheetModel);
            sheetDescMap.insert(sheetModel.getUid(), sheetDesc);

            // collect the formula cells in the new sheet (copied sheets may contain formula cells)
            var cellCollection = sheetModel.getCellCollection();
            var iterator = cellCollection.createAddressIterator(docModel.getSheetRange(), { type: 'anyformula', covered: true });
            var promise = registerFormulaCells(sheetDesc, iterator, recalcAll ? { recalc: true } : null);

            // collect the formatting rules in the new sheet (copied sheets may contain conditional formattings)
            promise = promise.then(function () {
                var condFormatCollection = sheetModel.getCondFormatCollection();
                return timerHelper.iterate(condFormatCollection.createModelIterator(), registerRuleModel);
            });

            return promise;
        });

        /**
         * Updates the settings of this dependency manager before a sheet in
         * the spreadsheet document will be deleted.
         *
         * @param {SheetModel} sheetModel
         *  The model instance of the deleted sheet.
         */
        var unregisterSheetModel = DependencyUtils.profileMethod('unregisterSheetModel()', function (sheetModel) {

            // unregister the sheet-locally defined names
            pendingCollection.unregisterAllNameModels(sheetModel);

            // remove the sheet descriptor object for the deleted sheet (sheet descriptor may not
            // exist yet, e.g. when inserting/deleting a sheet very quickly in unit tests)
            var sheetDesc = sheetDescMap.remove(sheetModel.getUid());
            if (!sheetDesc) { return; }

            // remove all formulas of the deleted sheet from the internal structures
            sheetDesc.formulaMap.forEach(function (formulaDesc, formulaKey) {
                removeFormula(formulaKey, formulaDesc);
            });
        });

        var logDirtyFormulas = DependencyUtils.isLoggingActive() ? function (formulaMap) {
            formulaMap.forEach(function (formulaDesc, formulaKey) {
                DependencyUtils.log('key=' + formulaKey + ' references=' + formulaDesc.references + ' circular=' + formulaDesc.circular);
            });
        } : _.noop;

        /**
         * Collects the descriptors of all dirty formulas depending directly on
         * changed cells, or indirectly on other dirty formulas, in the class
         * member variable 'resultCollection', recalculates all collected dirty
         * formulas, and writes the calculated results into the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all dirty formulas have been
         *  collected and recalculated, and the cells in the document have been
         *  updated.
         */
        var recalcDirtyFormulas = DependencyUtils.profileAsyncMethod('recalcDirtyFormulas()', function () {

            // realculate-all mode: calculate and collect the results of all cell formulas in the document
            if (pendingCollection.recalcAll) {
                return timerHelper.iterate(docModel.createSheetIterator(), function (sheetModel) {
                    var iterator = sheetModel.getCellCollection().createAddressIterator(docModel.getSheetRange(), { type: 'anyformula', covered: true });
                    return timerHelper.iterate(iterator, function (address) {
                        self.getCellValue(sheetModel, address);
                    });
                });
            }

            // copy of the formula dictionary, that will be reduced during lookup to improve performance
            var lookupDictionary = formulaDictionary.clone();
            // staring point of the abortable promise chain
            var promise = self.createResolvedPromise();

            // inserts a new formula descriptor into the dirty formula map, and removes it from the lookup dictionary
            function registerDirtyFormula(formulaKey, formulaDesc) {
                lookupDictionary.removeFormula(formulaKey, formulaDesc);
                resultCollection.insertFormula(formulaKey, formulaDesc);
            }

            // inserts a map of formula descriptors into the dirty formula map, and removes them from the lookup dictionary
            function registerDirtyFormulas(formulaMap) {
                formulaMap.forEach(function (formulaDesc, formulaKey) {
                    registerDirtyFormula(formulaKey, formulaDesc);
                });
            }

            // Collect all formulas depending on defined names with changed formula expression, and refresh the dependencies
            // of all found formulas. The dependencies will include the new source references of the changed defined names.
            if (!pendingCollection.changedNamesMap.empty()) {
                promise = promise.then(DependencyUtils.profileAsyncMethod('collecting formulas for dirty names...', function () {
                    return timerHelper.iterate(pendingCollection.changedNamesMap.iterator(), function (nameModel, result) {
                        registerDirtyFormulas(lookupDictionary.findFormulasForName(result.key));
                    });
                }));
                promise = promise.then(DependencyUtils.profileAsyncMethod('refreshing formulas for dirty names...', function () {
                    return timerHelper.iterate(resultCollection.formulaMap.iterator(), function (formulaDesc, result) {
                        refreshFormula(result.key);
                    });
                }));
            }

            // Collect all formulas with a #NAME? error, after a defined name changed its label, and refresh
            // their dependencies too, in order to resolve #NAME? errors to valid name references.
            if (!pendingCollection.changedLabelsMap.empty()) {
                promise = promise.then(DependencyUtils.profileAsyncMethod('refreshing formulas for dirty name labels...', function () {
                    // visit all sheets (each sheet descriptor contains its own map of formulas with missing names)...
                    return timerHelper.iterate(sheetDescMap.iterator(), function (sheetDesc) {
                        // process all defined names with changed label for the current sheet...
                        return timerHelper.iterate(pendingCollection.changedLabelsMap.iterator(), function (nameModel) {
                            // if the sheet contains formulas with #NAME? errors using the current label...
                            return sheetDesc.missingNamesMap.with(nameModel.getKey(), function (destFormulaMap) {
                                // visit all these formulas with #NAME? errors...
                                return timerHelper.iterate(destFormulaMap.iterator(), function (formulaDesc, result) {
                                    // the formula may have been refreshed already (multiple unresolved names)
                                    var formulaKey = result.key;
                                    if (!resultCollection.formulaMap.has(formulaKey)) {
                                        refreshFormula(formulaKey);
                                        if (formulaDesc.references.containsName(nameModel)) {
                                            registerDirtyFormula(formulaKey, formulaDesc);
                                        }
                                    }
                                });
                            });
                        });
                    });
                }));
            }

            // Collect all initially dirty formulas from all sheets in the document.
            promise = promise.then(DependencyUtils.profileAsyncMethod('collecting initially dirty formulas...', function () {
                return timerHelper.iterate(sheetDescMap.iterator(), function (sheetDesc, result) {

                    // Add all formulas that are permanently dirty (recalculation mode 'always'). These formulas will
                    // always be recalculated, regardless whether they depend on any dirty cells.
                    if (initialUpdateCycle || pendingCollection.recalcVolatile) {
                        registerDirtyFormulas(sheetDesc.recalcAlwaysMap);
                    }

                    // Add all formulas that are initially dirty (recalculation mode 'once'). These formulas will be
                    // recalculated once after importing the document.
                    if (initialUpdateCycle) {
                        registerDirtyFormulas(sheetDesc.recalcOnceMap);
                    }

                    // the sheet UID, used as map key
                    var sheetUid = result.key;

                    // Get the address arrays of dirty value cells, and dirty formula cells. Dirty value cells that are
                    // formulas by themselves need to be recalculated (the correct formula result may have been destroyed).
                    var dirtyFormulaCells = sheetDesc.dirtyFormulaCells.clone();
                    var dirtyValueCells = sheetDesc.dirtyValueCells.reject(function (address) {
                        if (formulaDescMap.has(DependencyUtils.getCellKey(sheetUid, address))) {
                            dirtyFormulaCells.push(address);
                            return true;
                        }
                    });
                    if (dirtyFormulaCells.empty() && dirtyValueCells.empty()) { return; }

                    // Join the value cells to ranges for faster lookup of dependent formulas.
                    dirtyFormulaCells = dirtyFormulaCells.unify();
                    var dirtyValueRanges = RangeArray.mergeAddresses(dirtyValueCells);

                    DependencyUtils.withLogging(function () {
                        var sheetName = sheetDesc.sheetModel.getName();
                        if (!dirtyFormulaCells.empty()) {
                            DependencyUtils.log('dirty formulas in sheet ' + sheetName + ': ' + RangeArray.mergeAddresses(dirtyFormulaCells));
                        }
                        if (!dirtyValueRanges.empty()) {
                            DependencyUtils.log('dirty cells in sheet ' + sheetName + ': ' + dirtyValueRanges);
                        }
                    });

                    // staring point of the abortable promise chain
                    var promise2 = self.createResolvedPromise();

                    // Collect all dirty cell formulas. Cells with changed formula expressions must be recalculated in
                    // order to get a valid initial result.
                    if (!dirtyFormulaCells.empty()) {
                        promise2 = promise2.then(function () {
                            return timerHelper.iterate(dirtyFormulaCells.iterator(), function (address) {
                                var formulaKey = DependencyUtils.getCellKey(sheetUid, address);
                                sheetDesc.formulaMap.with(formulaKey, function (formulaDesc) {
                                    registerDirtyFormula(formulaKey, formulaDesc);
                                });
                            });
                        });
                    }

                    // Collect all formulas depending directly on dirty value cells.
                    if (!dirtyValueRanges.empty()) {
                        promise2 = promise2.then(function () {
                            return timerHelper.iterate(dirtyValueRanges.iterator(), function (range) {
                                registerDirtyFormulas(lookupDictionary.findFormulasForRange(sheetUid, range));
                            });
                        });
                    }

                    return promise2;
                }).done(function () {
                    logDirtyFormulas(resultCollection.formulaMap);
                });
            }));

            // Extend the map of dirty formulas with other formulas depending on them.
            promise = promise.then(function () {

                // descriptors of all cell formulas to collect dependent formulas for
                var pendingFormulaMap = resultCollection.cellFormulaMap.clone();

                // repeat as long as new formulas can be found depending on the formulas in 'pendingFormulaMap'
                return timerHelper.loop(function (index) {

                    // exit the (endless) loop, if the last iteration step did not find a new formula
                    if (pendingFormulaMap.empty()) { return Utils.BREAK; }

                    return DependencyUtils.takeAsyncTime('collecting dependent formulas for chain depth: ' + (index + 1) + '...', function () {

                        // all new formula descriptors not yet contained in the dirty formula map
                        var newFormulaMap = new SimpleMap();

                        // process all formulas whose dependencies have not been calculated yet
                        var promise2 = timerHelper.iterate(pendingFormulaMap.iterator(), function (sourceFormulaDesc) {

                            // process cell formulas only
                            if (sourceFormulaDesc.ruleModel) { return; }

                            // find all formulas that depend directly on the passed cell formula
                            var destFormulaMap = lookupDictionary.findFormulasForAddress(sourceFormulaDesc.sheetUid, sourceFormulaDesc.getCellAddress());
                            return timerHelper.iterate(destFormulaMap.iterator(), function (destFormulaDesc, destResult) {

                                // nothing more to do, if the formula has already been collected
                                var destFormulaKey = destResult.key;
                                if (resultCollection.formulaMap.has(destFormulaKey)) { return; }

                                // insert the formula descriptor into the maps
                                registerDirtyFormula(destFormulaKey, destFormulaDesc);
                                newFormulaMap.insert(destFormulaKey, destFormulaDesc);
                            });
                        });

                        // initialize the next iteration cycle
                        return promise2.done(function () {
                            logDirtyFormulas(newFormulaMap);
                            pendingFormulaMap = newFormulaMap;
                        });
                    });
                });
            });

            // calculate and collect all formula results, as long as dirty formulas are contained in the map
            promise = promise.then(DependencyUtils.profileAsyncMethod('recalculating all dirty formulas...', function () {
                return timerHelper.loop(function () {

                    // get a formula descriptor from the dirty formula map
                    var formulaDesc = resultCollection.cellFormulaMap.getAny(null);
                    if (!formulaDesc) { return Utils.BREAK; }

                    // calculate and store the result of the formula (this may recursively calculate other dirty
                    // formulas as well, which will be removed from the dirty formula map)
                    var sheetModel = sheetDescMap.get(formulaDesc.sheetUid).sheetModel;
                    self.getCellValue(sheetModel, formulaDesc.getCellAddress());
                });
            }));

            return promise;
        });

        /**
         * Writes the collected formula results into the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the cells in the document have
         *  been updated.
         */
        var refreshCellValues = DependencyUtils.profileAsyncMethod('refreshDocument()', function () {

            // wait for another operations generator process currently running (needed to prevent internal
            // application error due to nested operation generators), this may even abort the entire update
            // cycle, e.g. after deleting a sheet which makes the calculated formula results useless
            var promise = docModel.getActionsPromise();

            // write the collected results (synchronously) into the document
            return promise.done(DependencyUtils.profileMethod('updating document contents...', function () {

                // generate the cell operations, and send them to the server
                generateAndSendOperations(function (generator) {

                    // update the 'calcOnLoad' flag in the original spreadsheet document
                    var newCalcOnLoad = resultCollection.hasErrors ? true : initialUpdateCycle ? false : calcOnLoad;
                    if (newCalcOnLoad !== calcOnLoad) {
                        generator.generateDocAttrsOperation({ calcOnLoad: newCalcOnLoad });
                        calcOnLoad = newCalcOnLoad;
                    }

                    // create a 'changeCell' operation per sheet
                    resultCollection.resultMaps.forEach(function (resultMap, sheetUid) {

                        // remove all formula results from the map that do not change the old cell value
                        resultMap.forEach(function (resultDesc, cellKey) {
                            if (!resultDesc.changed) { resultMap.remove(cellKey); }
                        });
                        if (resultMap.empty()) { return; }

                        // generate cell operations for the remaining results
                        var sheetModel = sheetDescMap.get(sheetUid).sheetModel;
                        generator.setSheetIndex(sheetModel.getIndex());
                        sheetModel.getCellCollection().generateFormulaResultOperations(generator, resultMap);

                        // log addresses of all formula cells with changed results in a 'Recalc All' cycle
                        Utils.withLogging(function () {
                            if (!pendingCollection.recalcAll) { return; }
                            var addresses = resultMap.pluck('address');
                            Utils.warn('DependencyManager.runUpdateCycle(): changed formula results in sheet "' + sheetModel.getName() + '":');
                            Utils.warn('\xa0 ' + addresses.join(' ') + ' (' + addresses.length + ' cells)');
                        });
                    });
                });

                // refresh all formatting rules containing dirty formulas locally (without operations)
                if (pendingCollection.recalcAll) {
                    IteratorUtils.forEach(docModel.createSheetIterator(), function (sheetModel) {
                        sheetModel.getCondFormatCollection().refreshRanges();
                    });
                } else {
                    resultCollection.ruleRangesMap.forEach(function (ruleRanges, sheetUid) {
                        sheetDescMap.get(sheetUid).sheetModel.refreshRanges(ruleRanges);
                    });
                }
            }));
        });

        /**
         * Starts the background task that processes all collected and pending
         * document events, refreshes the dependency chains, recalculates all
         * dirty formulas, and updates the model contents (cells, conditional
         * formattings, etc.).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.continue=false]
         *      If set to true, an update cycle currently running will be
         *      continued. By default, a running update cycle will be aborted,
         *      and a fresh update cycle will be started.
         *  @param {Boolean} [options.priority=false]
         *      If set to true, the update cycle will be started with high
         *      timer priority. If used together with the option 'continue',
         *      the continued update cycle will be swiched from low to high
         *      priority.
         *  @param {Boolean} [options.volatile=true]
         *      If set to false (default is true!), volatile formulas will NOT
         *      be recalculated (unless they depend on dirty cells). This
         *      option will affect new update cycles only, but not a running
         *      continued update cycle.
         */
        var startUpdateCycle = (function () {

            // the timer promise representing the update cycle currently running
            var updateTimer = null;
            // first update after import will be delayed further
            var firstUpdate = true;
            // deferred object for a full cycle (remains pending until any cycle has been completed)
            var fullCycleDef = null;
            // the abortable promise of the deferred to be returned to external callers
            var fullCyclePromise = null;

            // aborts the update cycle currently running
            function abortUpdateCycle() {
                if (updateTimer) {
                    DependencyUtils.log('cancelling update cycle');
                    updateTimer.abort(); // additionally, this sets 'updateTimer' to null
                }
            }

            // immediately abort an update cycle, if a formula count overflow has been detected
            self.on('recalc:overflow', abortUpdateCycle);

            // starts a new background task (debounced)
            var runUpdateCycle = self.createDebouncedMethod(_.noop, Utils.profileAsyncMethod('dependency update cycle', function () {

                // trigger the start event (additional delay for initial update after import)
                var delay = firstUpdate ? DependencyUtils.IMPORT_UPDATE_DELAY : 0;
                updateTimer = self.executeDelayed(function () { self.trigger('recalc:start'); }, delay);
                firstUpdate = false;

                // collect all dirty formulas and the results, used later e.g. in the public method getCellValue()
                resultCollection = new ResultCollection();

                // Process all pending document changes (changed value cells, changed formula
                // expressions, moved cells, etc.), before starting to update the token arrays.
                updateTimer = updateTimer.then(processPendingDocumentChanges);

                // Collect and recalculate all dirty token arrays (all token arrays referring to
                // dirty cells, unless all formulas have to be recalculated).
                updateTimer = updateTimer.then(recalcDirtyFormulas);

                // Update the values of the formula cells in the document model.
                updateTimer = updateTimer.then(refreshCellValues);

                // Notify listeners, and clean up the caches after the update cycle has finished successfully.
                updateTimer.done(function () {

                    // notify the listeners
                    self.trigger('recalc:end', resultCollection.referenceCycles);

                    // clean up all internal data
                    sheetDescMap.forEach(function (sheetDesc) { sheetDesc.clearDirtyCells(); });
                    pendingCollection.clear();
                    resultCollection = null;
                    initialUpdateCycle = false;

                    // resolve the deferred object that represents a full calculation cycle
                    fullCycleDef.resolve();
                });

                // notify listeners if the update cycle has been aborted; 'fullCycleDef' remains pending
                updateTimer.fail(function () { self.trigger('recalc:cancel'); });

                // returning the promise is required by Logger.profileAsyncMethod() method
                return updateTimer.always(function () { updateTimer = null; });

            }, { delay: DependencyUtils.UPDATE_DEBOUNCE_DELAY }));

            // the implementation of the method startUpdateCycle() returned from local scope
            return function (options) {

                // abort a running background task (unless the task shall be continued)
                if (!Utils.getBooleanOption(options, 'continue', false)) {
                    abortUpdateCycle();
                }

                // start (or continue) timers with high priority if specified
                if (Utils.getBooleanOption(options, 'priority', false)) {
                    timerHelper.setPriority('high');
                }

                // create the promise that will be returned to external callers from public methods
                // (remains in pending state until an update cycle can be finished successfully)
                if (!fullCycleDef) {
                    fullCycleDef = new $.Deferred();
                    fullCyclePromise = self.createAbortablePromise(fullCycleDef);
                    fullCyclePromise.done(function () { timerHelper.setPriority('low'); });
                    fullCyclePromise.always(function () { fullCycleDef = null; });
                }

                // whether to recalculate all volatile formulas that do NOT depend on dirty cells
                pendingCollection.recalcVolatile = Utils.getBooleanOption(options, 'volatile', true);

                // start the new background task (unless an update cycle exists already and will be reused)
                if (!updateTimer) { runUpdateCycle();  }

                return fullCyclePromise;
            };
        }());

        // document event queue -----------------------------------------------

        /**
         * Processes all document change events that have been collected in the
         * 'pendingDocumentChanges' queue.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise that will be resolved after all collected
         *  document changes have been processed, and removed from the queue.
         */
        var processPendingDocumentChanges = DependencyUtils.profileAsyncMethod('processPendingDocumentChanges()', function () {

            // shift and process all elements in 'pendingDocumentChanges' in a background loop
            return timerHelper.loop(function () {

                // get the next pending change descriptor from the array
                var docChange = pendingCollection.documentChanges[0];
                if (!docChange) { return Utils.BREAK; }

                // sheet model and sheet descriptor (null for global document events, e.g. globally defined names)
                var sheetModel = docChange.sheetModel;
                var sheetDesc = sheetModel ? sheetDescMap.get(sheetModel.getUid()) : null;

                var promise = $.when();
                switch (docChange.type) {
                    case 'insertSheet':
                        promise = registerSheetModel(sheetModel, docChange.recalcAll);
                        break;
                    case 'changeValues':
                        registerValueCells(sheetDesc, docChange.addresses);
                        break;
                    case 'changeFormulas':
                        promise = registerFormulaCells(sheetDesc, docChange.addresses.iterator(), { update: true, recalc: true });
                        break;
                    case 'moveCells':
                        promise = registerMovedCells(sheetDesc, docChange.moveDesc);
                        break;
                    case 'insertRule':
                        promise = registerRuleModel(docChange.ruleModel);
                        break;
                    case 'changeRule':
                        promise = registerRuleModel(docChange.ruleModel, { update: true });
                        break;
                    case 'changeName':
                        pendingCollection.registerNameModel(docChange.nameModel, docChange.formulaChanged, docChange.labelChanged);
                        break;
                }

                // remove the change descriptor after it has been processed completetly (not before!)
                return promise.done(function () {
                    pendingCollection.documentChanges.shift();
                });
            });
        });

        // event handlers -----------------------------------------------------

        /**
         * Handler for 'insert:sheet:after' events sent by the document model.
         * Registers a new sheet and all its formula cells in this dependency
         * manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the new sheet model.
         *
         * @param {SheetModel} sheetModel
         *  The new sheet model.
         */
        function insertSheetHandler(event, sheet, sheetModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: inserted sheet "' + sheetModel.getName() + '"'); });
            pendingCollection.registerDocumentChange('insertSheet', sheetModel);
        }

        /**
         * Handler for 'delete:sheet:before' events sent by the document model.
         * Removes all settings of a sheet that will be deleted from the
         * document.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model to be deleted.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model the will be deleted.
         */
        function deleteSheetHandler(event, sheet, sheetModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: deleted sheet "' + sheetModel.getName() + '"'); });

            // delete all pending document change events associated to the deleted sheet
            pendingCollection.deleteDocumentChanges(function (docChange) { return docChange.sheetModel === sheetModel; });
            // immediately remove all other cached settings of the sheet
            unregisterSheetModel(sheetModel);
        }

        /**
         * Handler for 'insert:name' events sent by the document model. Inserts
         * all settings for a new defined name into this dependency manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number|Null} sheet
         *  The index of the sheet model containing the defined name, or null
         *  for globally defined names.
         *
         * @param {NameModel} nameModel
         *  The model of the new defined name.
         */
        function insertNameHandler(event, sheet, nameModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: inserted name "' + DependencyUtils.getNameLabel(nameModel) + '"'); });
            pendingCollection.registerDocumentChange('changeName', nameModel.getSheetModel(), { nameModel: nameModel, formulaChanged: true, labelChanged: true });
        }

        /**
         * Handler for 'delete:name' events sent by the document model. Removes
         * all settings of a deleted defined name from this dependency manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number|Null} sheet
         *  The index of the sheet model containing the defined name, or null
         *  for globally defined names.
         *
         * @param {NameModel} nameModel
         *  The model of the defined name that will be deleted.
         */
        function deleteNameHandler(event, sheet, nameModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: deleted name "' + DependencyUtils.getNameLabel(nameModel) + '"'); });

            // delete all cached pending change events for the deleted name
            pendingCollection.deleteDocumentChanges(function (docChange) { return docChange.nameModel === nameModel; });
            // immediately remove all other cached settings of the deleted name
            pendingCollection.unregisterNameModel(nameModel);
            startUpdateCycle();
        }

        /**
         * Handler for 'change:name' events sent by the document model. Updates
         * the settings of an existing defined name, whose formula expression
         * has been changed.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number|Null} sheet
         *  The index of the sheet model containing the defined name, or null
         *  for globally defined names.
         *
         * @param {NameModel} nameModel
         *  The model of the defined name with a changed formula expression.
         */
        function changeNameHandler(event, sheet, nameModel, changeFlags) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: changed name "' + DependencyUtils.getNameLabel(nameModel) + '"'); });
            pendingCollection.registerDocumentChange('changeName', nameModel.getSheetModel(), { nameModel: nameModel, formulaChanged: changeFlags.formula, labelChanged: changeFlags.label });
        }

        /**
         * Handler for 'change:cells' events sent by the document model.
         * Registers changed values and formulas in the document.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the changed cells.
         *
         * @param {ChangeDescriptor} changeDesc
         *  The descriptor with the addresses of all changed cells.
         */
        function changeCellsHandler(event, sheet, changeDesc, external, results) {

            // ignore the change events that have been generated due to the own last update cycle
            if (results) { return; }

            var sheetModel = docModel.getSheetModel(sheet);

            // update the token arrays depending on cells that have changed their values
            if (!changeDesc.valueCells.empty()) {
                pendingCollection.registerDocumentChange('changeValues', sheetModel, { addresses: changeDesc.valueCells });
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: changed value cells in sheet "' + sheetModel.getName() + '": ' + RangeArray.mergeAddresses(changeDesc.valueCells));
                });
            }

            // update the dependency chains if the formula expressions of cells have changed
            if (!changeDesc.formulaCells.empty()) {
                pendingCollection.registerDocumentChange('changeFormulas', sheetModel, { addresses: changeDesc.formulaCells });
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: changed formula cells in sheet "' + sheetModel.getName() + '": ' + RangeArray.mergeAddresses(changeDesc.formulaCells));
                });
            }
        }

        /**
         * Handler for 'move:cells' events sent by the document model. Updates
         * this dependency manager after cells have been moved in the document,
         * including inserted or deleted columns/rows.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the moved cells.
         *
         * @param {MoveDescriptor} moveDesc
         *  The descriptor with the positions of all moved cells.
         */
        function moveCellsHandler(event, sheet, moveDesc) {

            var sheetModel = docModel.getSheetModel(sheet);
            DependencyUtils.withLogging(function () {
                DependencyUtils.log('Event: moved cells in sheet "' + sheetModel.getName() + '": ' + moveDesc.dirtyRange);
            });

            // Immediately transform the addresses of dirty value cells and formula cells in the pending document events.
            // This is needed to prevent updating formula descriptors that are located at a different position in the
            // cell collection, before the move operation will be processed.
            pendingCollection.documentChanges.forEach(function (docChange) {
                if ((docChange.sheetModel === sheetModel) && docChange.addresses) {
                    docChange.addresses = moveDesc.transformAddresses(docChange.addresses);
                }
            });

            // Insert the move operation at the beginning of the pending document events queue, in order to ensure that
            // the formula descriptors will be relocated before there dependencies will be refreshed.
            var index = Utils.findFirstIndex(pendingCollection.documentChanges, function (docChange) { return !('moveDesc' in docChange); }, { sorted: true });
            pendingCollection.registerDocumentChange('moveCells', sheetModel, { moveDesc: moveDesc }, { index: index });
        }

        /**
         * Handler for 'change:rows' events sent by the document model. Updates
         * this dependency manager after rows have been shown or hidden in the
         * document. All volatile formulas will be recalculated.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the changed rows.
         */
        function changeRowsHandler(event, sheet, interval, attributes, styleId, changeFlags) {
            if (changeFlags.visibility) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: changed row visibility in sheet "' + docModel.getSheetName(sheet) + '"');
                });
                startUpdateCycle();
            }
        }

        /**
         * Handler for 'insert:rule' events sent by the document model. Inserts
         * all settings for a new formatting rule into this dependency manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the rule.
         *
         * @param {RuleModel} ruleModel
         *  The model of the new formatting rule.
         */
        function insertRuleHandler(event, sheet, ruleModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: inserted rule "' + DependencyUtils.getRuleLabel(ruleModel) + '"'); });
            pendingCollection.registerDocumentChange('insertRule', ruleModel.getSheetModel(), { ruleModel: ruleModel });
        }

        /**
         * Handler for 'delete:rule' events sent by the document model. Removes
         * all settings of a deleted formatting rule from this dependency
         * manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the rule.
         *
         * @param {RuleModel} ruleModel
         *  The model of the deleted formatting rule.
         */
        function deleteRuleHandler(event, sheet, ruleModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: deleted rule "' + DependencyUtils.getRuleLabel(ruleModel) + '"'); });

            // delete all cached pending change events for the deleted rule
            pendingCollection.deleteDocumentChanges(function (docChange) { return docChange.ruleModel === ruleModel; });
            // immediately remove all other cached settings of the deleted rule
            unregisterRuleModel(ruleModel);
        }

        /**
         * Handler for 'change:rule' events sent by the document model. Updates
         * all settings of a changed formatting rule in this dependency
         * manager.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object sent to all event handlers.
         *
         * @param {Number} sheet
         *  The index of the sheet model containing the rule.
         *
         * @param {RuleModel} ruleModel
         *  The model of the changed formatting rule.
         */
        function changeRuleHandler(event, sheet, ruleModel) {
            DependencyUtils.withLogging(function () { DependencyUtils.log('Event: changed rule "' + DependencyUtils.getRuleLabel(ruleModel) + '"'); });
            pendingCollection.registerDocumentChange('changeRule', ruleModel.getSheetModel(), { ruleModel: ruleModel });
        }

        // public methods -----------------------------------------------------

        /**
         * Recalculates all dirty formulas in the document, and refreshes all
         * cells with outdated formula results, and all other document contents
         * that depend on formula results (e.g. charts, or conditional
         * formatting). The background task that calculates the formula results
         * will run with high priority (in difference to background tasks
         * started automatically due to change events of the document processed
         * by this dependency manager).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.all=false]
         *      If set to true, all availavle formulas in the document will be
         *      recalculated, regardless of their diryt state.
         *  @param {Boolean} [options.volatile=true]
         *      If set to false (default is true!), volatile formulas will NOT
         *      be recalculated (unless they depend on dirty cells). This
         *      option will be ignored, if the option 'all' has been set.
         *  @param {Number} [options.timeout]
         *      The maximum time allowed for the recalculation cycle, in
         *      milliseconds. If the time elapses before all formulas have been
         *      calculated, the recalculation cycle will be aborted, and the
         *      'calcOnLoad' document flag will be set.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all dirty formulas have been
         *  recalculated successfully.
         */
        this.recalcFormulas = function (options) {

            // initialize the 'recalcAll' flag in the internal data
            if (Utils.getBooleanOption(options, 'all', false)) {
                pendingCollection.recalcAll = true;
            }

            // start the update cycle, or continue a running update cycle
            var promise = startUpdateCycle({
                continue: !pendingCollection.recalcAll,
                priority: true,
                volatile: Utils.getBooleanOption(options, 'volatile', true)
            });

            // initialize a timer that aborts the returned promise after the specified delay
            var timeout = Utils.getIntegerOption(options, 'timeout', 0);
            if (timeout > 0) {

                // create a new deferred object that represents this method invocation
                // (this allows to call the method multiple times with individual timeouts)
                var deferred = new $.Deferred();
                promise.done(deferred.resolve.bind(deferred));
                promise.fail(deferred.reject.bind(deferred));
                promise.progress(deferred.notify.bind(deferred));

                // create a promise with timeout from the new deferred object,
                // set the 'calcOnLoad' document flag if the timeout is reached
                promise = this.createAbortablePromise(deferred, setCalcOnLoadFlag, timeout);
            }

            return promise;
        };

        /**
         * Returns an up-to-date cell value for the specified cell in the
         * document. If the cell is a formula cell that this dependency manager
         * has marked as dirty, the formula will be recalculated, and all cells
         * referred by the formula that are dirty too will be recalculated
         * recursively. Otherwise, the current value of the cell as stored in
         * the cell model will be returned.
         *
         * @param {SheetModel} sheetModel
         *  The model of the sheet containing the cell to be resolved.
         *
         * @param {Address} address
         *  The address of the cell whose value will be resolved.
         *
         * @returns {Object}
         *  A descriptor object with information about the resolved cell value,
         *  with the following properties:
         *  - {Any} value
         *      The resolved scalar value of the specified cell.
         *  - {String} type
         *      How the cell value has been resolved. If set to 'calculated',
         *      the value has just been calculated by interpreting the dirty
         *      cell formula. If set to 'cached', the result has already been
         *      calculated by a preceding call of this metjod during the update
         *      cycle  currently running. If set to 'fixed', the value has been
         *      returned directly from the cell model (either because the cell
         *      is not a formula cell, or the formula was not dirty).
         *  - {Number} time
         *      The time in milliseconds it took to calculate the result of a
         *      dirty formula cell (type 'calculated'). Will be zero for all
         *      other value types ('cached' and 'fixed').
         */
        this.getCellValue = (function () {

            // stack with the addresses of all formulas currently calculated recursively
            var formulaAddressStack = new AddressArray();

            // calculates the result of a cell formula
            function calculateCellFormula(sheetUid, tokenArray, refAddress, targetAddress, oldValue) {

                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('calculating formula in cell ' + sheetUid + '!' + targetAddress);
                });

                // take the time it needs to calculate the formula (and all formulas it depends on)
                var startTime = _.now();

                var formulaAddress = targetAddress.clone();
                formulaAddress.sheetUid = sheetUid;
                formulaAddressStack.push(formulaAddress);

                // first, calculate the formula result
                var result = tokenArray.interpretFormula('val', {
                    refAddress: refAddress,
                    targetAddress: targetAddress,
                    recalcDirty: true
                });

                // detect circular reference errors, and other warnings and errors
                var hasError = false;
                if (result.code === 'circular') {
                    DependencyUtils.withLogging(function () {
                        var addressNames = formulaAddressStack.map(function (address) { return address.sheetUid + '!' + address; });
                        DependencyUtils.warn('reference cycle found: ' + addressNames.join());
                    });
                    resultCollection.referenceCycles.push(formulaAddressStack.clone());
                } else if (result.type !== 'valid') {
                    hasError = resultCollection.hasErrors = true;
                }
                formulaAddressStack.pop();

                // insert the new result into the result map of the correct sheet (but keep old result
                // in the map, if the formula has been recalculated already due to a circular reference)
                var addressKey = targetAddress.key();
                var resultMap = resultCollection.resultMaps.getOrConstruct(sheetUid, SimpleMap);
                if (resultMap.has(addressKey)) { return null; }
                return resultMap.insert(addressKey, {
                    address: targetAddress,
                    result: result,
                    changed: hasError || !FormulaUtils.equalScalars(result.value, oldValue, { withCase: true, nullMode: 'less' }),
                    time: _.now() - startTime
                });
            }

            // the implementation of the method getCellValue()
            function getCellValue(sheetModel, address) {

                // the result descriptor
                var valueDesc = { value: null, type: null, time: 0 };

                if (!app || app.isInQuit()) {
                    return valueDesc;
                }

                // first, try to find a result already calculated by a preceding call of this method
                var sheetUid = sheetModel.getUid();
                var addressKey = address.key();
                var resultDesc = resultCollection.resultMaps.with(sheetUid, function (resultMap) {
                    return resultMap.get(addressKey, null);
                });
                if (resultDesc) {
                    valueDesc.value = resultDesc.result.value;
                    valueDesc.type = 'cached';
                    return valueDesc;
                }

                // the current cell value or formula result
                var value = sheetModel.getCellCollection().getCellValue(address);

                // in recalculate-all mode, resolve the token array directly from the cell collection
                if (pendingCollection.recalcAll) {

                    var tokenArray = sheetModel.getCellCollection().getCellTokenArray(address);
                    resultDesc = tokenArray ? calculateCellFormula(sheetUid, tokenArray, address, address, value) : null;

                } else {

                    // try to find a dirty formula, and calculate its result
                    var formulaKey = DependencyUtils.getCellKey(sheetUid, address);
                    resultCollection.cellFormulaMap.with(formulaKey, function (formulaDesc) {
                        // calculate the result of the cell formula
                        resultDesc = calculateCellFormula(sheetUid, formulaDesc.tokenArray, formulaDesc.getRefAddress(), address, value);
                        // remove the formula descriptor (MUST be done after calculation, needed to detect circular references!)
                        // if the formula has been removed already due to a circular reference, keep that result in the map
                        resultCollection.cellFormulaMap.remove(formulaKey);
                    });
                }

                if (resultDesc) {
                    valueDesc.value = resultDesc.result.value;
                    valueDesc.type = 'calculated';
                    valueDesc.time = resultDesc.time;
                    return valueDesc;
                }

                // return the current cell value for all other formulas, and simple cells
                valueDesc.value = value;
                valueDesc.type = 'fixed';
                return valueDesc;
            }

            return getCellValue;
        }());

        // initialization -----------------------------------------------------

        // initialization after document import
        docModel.waitForImportSuccess(function () {

            // initialize dependency manager for all existing sheets after import
            calcOnLoad = docModel.getDocumentAttribute('calcOnLoad');
            IteratorUtils.forEach(docModel.createSheetIterator(), function (sheetModel) {
                pendingCollection.registerDocumentChange('insertSheet', sheetModel, { recalcAll: calcOnLoad });
            });

            // keep track of all changes in the document
            this.listenTo(docModel, 'insert:sheet:after', insertSheetHandler);
            this.listenTo(docModel, 'delete:sheet:before', deleteSheetHandler);
            this.listenTo(docModel, 'insert:name', insertNameHandler);
            this.listenTo(docModel, 'delete:name', deleteNameHandler);
            this.listenTo(docModel, 'change:name', changeNameHandler);
            this.listenTo(docModel, 'change:cells', changeCellsHandler);
            this.listenTo(docModel, 'move:cells', moveCellsHandler);
            this.listenTo(docModel, 'change:rows', changeRowsHandler);
            this.listenTo(docModel, 'insert:rule', insertRuleHandler);
            this.listenTo(docModel, 'delete:rule', deleteRuleHandler);
            this.listenTo(docModel, 'change:rule', changeRuleHandler);

            // start a new update cycle, after document change events have been processed
            this.listenTo(pendingCollection, 'change', function () { startUpdateCycle(); });

            // initial update cycle for the imported sheets
            startUpdateCycle();
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            pendingCollection.destroy();
            timerHelper.destroy();
            self = app = docModel = timerHelper = null;
            formulaDescMap = sheetDescMap = formulaDictionary = null;
            pendingCollection = resultCollection = null;
        });

    } }); // class DependencyManager

    // exports ================================================================

    return DependencyManager;

});
