/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/drawingmodelmixin', [
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Operations, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var Range = SheetUtils.Range;

    // horizontal drawing anchor attribute names
    var COL_ATTRIBUTE_NAMES = {
        offset: 'left',
        size: 'width',
        startIndex: 'startCol',
        startOffset: 'startColOffset',
        endIndex: 'endCol',
        endOffset: 'endColOffset'
    };

    // vertical drawing anchor attribute names
    var ROW_ATTRIBUTE_NAMES = {
        offset: 'top',
        size: 'height',
        startIndex: 'startRow',
        startOffset: 'startRowOffset',
        endIndex: 'endRow',
        endOffset: 'endRowOffset'
    };

    // the names of all anchor attributes in the 'drawing' attribute family
    var ANCHOR_ATTR_NAMES = ['anchorType'].concat(_.values(COL_ATTRIBUTE_NAMES)).concat(_.values(ROW_ATTRIBUTE_NAMES));

    // class DrawingModelMixin ================================================

    /**
     * A mix-in class that extends the public API of any drawing model object
     * that has been inserted into a sheet of a spreadsheet document.
     *
     * @constructor
     *
     * @internal
     *  This is a mix-in class supposed to extend an existing instance of the
     *  class DrawingModel or any of its sub classes. Expects the symbol 'this'
     *  to be bound to an instance of DrawingModel.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing the drawing collection with this drawing
     *  model object.
     *
     * @param {Function} cloneConstructor
     *  A callback function invoked from the public method clone() which has to
     *  create and return a new instance cloned from this drawing model.
     *
     * @param {Function} [generateRestoreOperations]
     *  A callback function that will be invoked from the own public method
     *  DrawingModelMixin.generateDeleteOperations() while generating document
     *  operations that will delete this drawing object. An implementation must
     *  generate more undo operations to restore the complete drawing object.
     *
     * @param {Function} [generateFormulaOperations]
     *  A callback function that will be invoked from the own public method
     *  DrawingModelMixin.generateFormulaOperations() while generating document
     *  operations that involve to update the formula expressions in this
     *  drawing object. Receives all parameters passed to the public method
     *  mentioned before. May return an attribute set with additional
     *  formatting attributes to be applied at this drawing object, or null.
     */
    function DrawingModelMixin(sheetModel, cloneConstructor, generateRestoreOperations, generateFormulaOperations) {

        // self reference
        var self = this;

        // the document model instance
        var docModel = sheetModel.getDocModel();

        // the drawing collection containing this drawing model
        var drawingCollection = sheetModel.getDrawingCollection();

        // private methods ----------------------------------------------------

        /**
         * Returns the anchor attributes of this drawing object.
         *
         * @returns {Object}
         *  A copy of the anchor attributes of this drawing object.
         */
        function getAnchorAttributes() {
            var drawingAttrs = self.getMergedAttributeSet(true).drawing;
            return ANCHOR_ATTR_NAMES.reduce(function (attrs, name) {
                attrs[name] = drawingAttrs[name];
                return attrs;
            }, {});
        }

        /**
         * Generates the anchor attributes needed to update the position of
         * this drawing object, after cells have been moved in the document.
         *
         * @param {MoveDescriptor} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  position of this drawing object.
         *
         * @returns {Object|Null}
         *  The value null to indicate that this drawing object will be deleted
         *  from the sheet by the passed move descriptors; otherwise the anchor
         *  attributes that need to ne applied at this drawing object to update
         *  its position. An empty object indicates that the drawing object
         *  will not be modified at all.
         */
        function generateMoveCellsAttributes(moveDescs) {

            // copy of the current anchor attributes, will be recalculated below
            var anchorAttrs = getAnchorAttributes();
            // the cell range covered by the drawing object
            var drawingRange = self.getRange();

            // process all move descriptors, returning true indicates to delete the drawing
            var deleteDrawing = moveDescs.some(function (moveDesc) {

                // nothing to do, if the move descriptor will not touch the drawing object
                if (!moveDesc.coversRange(drawingRange)) { return false; }

                // whether to modify the column or row indexes of the range
                var columns = moveDesc.columns;
                // the attribute names according to the passed direction
                var ATTR_NAMES = columns ? COL_ATTRIBUTE_NAMES : ROW_ATTRIBUTE_NAMES;
                // the maximum column/row
                var maxIndex = docModel.getMaxIndex(columns);

                // calculate new column/row indexes; if the drawing object has one-cell or absolute anchor mode,
                // these attributes will be overwritten when recalculating the inactive anchor attributes
                if (moveDesc.insert) {

                    // insertion mode: process the target intervals from first to last
                    return moveDesc.targetIntervals.some(function (interval) {

                        // the number of columns/rows to be inserted
                        var move = interval.size();
                        // the column/row interval covered by the drawing model
                        var startIndex = drawingRange.getStart(columns);
                        var endIndex = drawingRange.getEnd(columns);
                        // the maximum number of columns/rows the drawing can be moved
                        var maxMove = Math.min(move, maxIndex - endIndex);

                        // do nothing, if drawing cannot be move anymore
                        if (maxMove === 0) { return false; }

                        // move end position, if the columns/rows are inserted inside the drawing object
                        if (interval.first <= endIndex) {
                            anchorAttrs[ATTR_NAMES.endIndex] += maxMove;
                            drawingRange.end.move(maxMove, columns);
                            // move completely, if the columns/rows are inserted before the drawing object
                            if (interval.first <= startIndex) {
                                anchorAttrs[ATTR_NAMES.startIndex] += maxMove;
                                drawingRange.start.move(maxMove, columns);
                            }
                        }
                    });
                }

                // deletion mode: process the target intervals in reversed order
                return moveDesc.targetIntervals.someReverse(function (interval) {

                    // the number of columns/rows to be inserted
                    var move = interval.size();
                    // the column/row interval covered by the drawing model
                    var startIndex = drawingRange.getStart(columns);
                    var endIndex = drawingRange.getEnd(columns);

                    // Delete drawing object with two-cell anchor mode, if it is contained in the deleted column/row
                    // interval, or if it ends at offset zero in the first column/row after the interval.
                    if ((anchorAttrs.anchorType === 'twoCell') && (interval.first <= startIndex) && (endIndex <= interval.last)) {
                        return true;
                    }

                    // adjust start/end position of drawings overlapping or following the deleted interval
                    if (interval.last < endIndex) {
                        // move end position, if the columns/rows are deleted before the end
                        anchorAttrs[ATTR_NAMES.endIndex] -= move;
                        drawingRange.end.move(-move, columns);
                    } else {
                        // cut end of drawing object at the deleted interval
                        if (interval.first === 0) { return true; }
                        anchorAttrs[ATTR_NAMES.endIndex] = interval.first;
                        anchorAttrs[ATTR_NAMES.endOffset] = 0;
                        drawingRange.setEnd(interval.first - 1, columns);
                    }
                    if (interval.last < startIndex) {
                        // move start position, if the columns/rows are deleted before the start
                        anchorAttrs[ATTR_NAMES.startIndex] -= move;
                        drawingRange.start.move(-move, columns);
                    } else if (interval.first <= startIndex) {
                        // cut start of drawing object at the deleted interval
                        anchorAttrs[ATTR_NAMES.startIndex] = interval.first;
                        anchorAttrs[ATTR_NAMES.startOffset] = 0;
                        drawingRange.setStart(interval.first, columns);
                    }
                });
            });

            // update the inactive anchor attributes according to anchor type (this may return to the old cell anchor)
            return deleteDrawing ? null : _.extend(anchorAttrs, drawingCollection.getInactiveAnchorAttributes(anchorAttrs, { displayAnchor: true }));
        }

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this drawing object model
         * for the specified sheet.
         *
         * @internal
         *  Used by the class SheetDrawingCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {DrawingModelMixin}
         *  A clone of this drawing model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            return cloneConstructor.call(this, targetModel);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the sheet model instance that contains this drawing model.
         *
         * @returns {SheetModel}
         *  The sheet model instance that contains this drawing model.
         */
        this.getSheetModel = function () {
            return sheetModel;
        };

        /**
         * Returns the position of this drawing model in the drawing collection
         * of the sheet.
         *
         * @returns {Array<Number>|Null}
         *  The sheet position of the drawing object, if it is still contained
         *  in the drawing collection; otherwise null.
         */
        this.getSheetPosition = function () {
            return drawingCollection.getModelPosition(this, { deep: true });
        };

        /**
         * Returns the complete document position of this drawing model, as
         * used in document operations, including the leading sheet index.
         *
         * @returns {Array<Number>|Null}
         *  The complete position of the drawing object, if it is still
         *  contained in the drawing collection, including the zero-based sheet
         *  index as first array element; otherwise null.
         */
        this.getDocumentPosition = function () {
            var position = this.getSheetPosition();
            return position ? [sheetModel.getIndex()].concat(position) : null;
        };

        /**
         * Returns the address of the cell range covered by this drawing model
         * object.
         *
         * @returns {Range}
         *  The address of the cell range covered by this drawing model.
         */
        this.getRange = function () {
            var anchorAttrs = this.getMergedAttributeSet(true).drawing;
            var range = Range.create(anchorAttrs.startCol, anchorAttrs.startRow, anchorAttrs.endCol, anchorAttrs.endRow);
            // shorten by one column/row, if the end offset is zero
            if ((anchorAttrs.endColOffset === 0) && !range.singleCol()) { range.end[0] -= 1; }
            if ((anchorAttrs.endRowOffset === 0) && !range.singleRow()) { range.end[1] -= 1; }
            return range;
        };

        /**
         * Returns the sheet rectangle in pixels covered by this drawing
         * object.
         *
         * @returns {Object|Null}
         *  The position of this drawing object in the sheet in pixels,
         *  according to the current sheet zoom factor, in the properties
         *  'left', 'top', 'width', and 'height'; or null if the drawing object
         *  is hidden by itself, or is located entirely in hidden columns or
         *  rows and thus invisible.
         */
        this.getRectangle = function () {
            // always return null for invisible drawing objects
            return this.isVisible() ? drawingCollection.getRectangleForAnchor(this.getMergedAttributeSet(true).drawing) : null;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations, and the undo operations, to change the
         * attributes of this drawing object.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of this drawing object in the sheet, as expected by
         *  the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateChangeOperations = function (generator, position, attributeSet) {
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: this.getUndoAttributeSet(attributeSet) }, { undo: true });
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position, { attrs: attributeSet });
            return this;
        };

        /**
         * Generates the operations, and the undo operations, to delete the
         * specified drawing objects from this collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of this drawing object in the sheet, as expected by
         *  the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateDeleteOperations = function (generator, position) {

            // undo: create an 'insertDrawing' operation to restore the drawing model with its attributes
            var undoProperties = { type: this.getType(), attrs: this.getExplicitAttributeSet(true) };
            generator.generateDrawingOperation(Operations.INSERT_DRAWING, position, undoProperties, { undo: true });

            // additional undo operations for complex drawing contents
            if (_.isFunction(generateRestoreOperations)) {
                generateRestoreOperations.call(this, generator, position);
            }

            // generate the delete operation for this drawing object
            generator.generateDrawingOperation(Operations.DELETE_DRAWING, position);
            return this;
        };

        /**
         * Generates the operations and undo operations to update and restore
         * the inactive anchor attributes of this drawing object, after the
         * size or visibility of the columns or rows in the sheet has been
         * changed.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of this drawing object in the sheet, as expected by
         *  the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateUpdateAnchorOperations = function (generator, position) {

            // update the inactive anchor attributes according to the anchor type of the drawing object
            var anchorAttrs = getAnchorAttributes();
            _.extend(anchorAttrs, drawingCollection.getInactiveAnchorAttributes(anchorAttrs, { displayAnchor: true }));

            // create a change operation, if some attributes will be changed in this drawing
            var attributeSet = this.getReducedAttributeSet({ drawing: anchorAttrs });
            if (!_.isEmpty(attributeSet)) {
                this.generateChangeOperations(generator, position, attributeSet);
            }

            return this;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of the source links in this drawing object
         * that refer to some cells in this document.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of this drawing object in the sheet, as expected by
         *  the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {DrawingModelMixin}
         *  A reference to this instance.
         */
        this.generateFormulaOperations = function (generator, position, changeDesc) {

            // update the position of top-level drawing objects, if cells in the own sheet will be moved
            var anchorAttrs = null;
            if ((changeDesc.type === 'moveCells') && (position.length === 1) && (changeDesc.sheet === sheetModel.getIndex())) {

                // calculate the new anchor attributes
                anchorAttrs = generateMoveCellsAttributes(changeDesc.moveDescs);

                // delete the drawing object if the parent cells have been deleted
                if (!anchorAttrs) {
                    return this.generateDeleteOperations(generator, position);
                }
            }

            // the formatting attributes to be applied for this drawing object
            var attributeSet = null;

            // derived classes may generate new operations for child objects, and additional formatting attributes
            if (_.isFunction(generateFormulaOperations)) {
                attributeSet = generateFormulaOperations.call(this, generator, position, changeDesc);
            }

            // add the new anchor attributes to the attribute set
            if (!_.isEmpty(anchorAttrs)) {
                if (attributeSet) {
                    attributeSet.drawing = _.isEmpty(attributeSet.drawing) ? anchorAttrs : _.extend(attributeSet.drawing, anchorAttrs);
                } else {
                    attributeSet = { drawing: anchorAttrs };
                }
            }

            // reduce the attribute to the attributes that will actually change
            if (attributeSet) {
                attributeSet = this.getReducedAttributeSet(attributeSet);
            }

            // create a change operation, if some attributes will be changed in this drawing
            if (!_.isEmpty(attributeSet)) {
                this.generateChangeOperations(generator, position, attributeSet);
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = sheetModel = drawingCollection = cloneConstructor = null;
        });

    } // class DrawingModelMixin

    // exports ================================================================

    return DrawingModelMixin;

});
