/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/drawingcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/model/drawingmodel',
    'io.ox/office/drawinglayer/model/shapemodel',
    'io.ox/office/drawinglayer/model/imagemodel',
    'io.ox/office/drawinglayer/model/drawingcollection',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/drawing/drawingmodelmixin',
    'io.ox/office/spreadsheet/model/drawing/chartmodel',
    'io.ox/office/tk/utils/iteratorutils'
], function (Utils, DrawingUtils, DrawingModel, ShapeModel, ImageModel, DrawingCollection, Operations, SheetUtils, DrawingModelMixin, SheetChartModel, IteratorUtils) {

    'use strict';

    // convenience shortcuts
    var Interval = SheetUtils.Interval;

    // maps 'anchorType' attribute values to the internal anchor mode used for anchor attributes
    var INTERNAL_ANCHOR_MODES = {
        twoCell: 'twoCell',
        oneCell: 'oneCell',
        absolute: 'absolute',
        twoCellAsOneCell: 'twoCell',
        twoCellAsAbsolute: 'twoCell'
    };

    // maps 'anchorType' attribute values to display anchor mode defining runtime behavior
    var DISPLAY_ANCHOR_MODES = {
        twoCell: 'twoCell',
        oneCell: 'oneCell',
        absolute: 'absolute',
        twoCellAsOneCell: 'oneCell',
        twoCellAsAbsolute: 'absolute'
    };

    // class SheetDrawingModel ================================================

    /**
     * The model of a generic drawing object contained in a sheet.
     *
     * @constructor
     *
     * @extends DrawingModel
     * @extends DrawingModelMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this drawing object.
     *
     * @param {String} type
     *  The type of this drawing object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the object.
     */
    var SheetDrawingModel = DrawingModel.extend({ constructor: function (sheetModel, type, initAttributes) {

        // base constructors --------------------------------------------------

        DrawingModel.call(this, sheetModel.getDocModel(), type, initAttributes);
        DrawingModelMixin.call(this, sheetModel, function cloneConstructor(targetModel) {
            return new SheetDrawingModel(targetModel, this.getType(), this.getExplicitAttributeSet(true));
        });

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            sheetModel = initAttributes = null;
        });

    } }); // class SheetImageModel

    // class SheetShapeModel ==================================================

    /**
     * The model of a shape object contained in a sheet.
     *
     * @constructor
     *
     * @extends ShapeModel
     * @extends DrawingModelMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this shape object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the object.
     */
    var SheetShapeModel = ImageModel.extend({ constructor: function (sheetModel, initAttributes) {

        // base constructors --------------------------------------------------

        ShapeModel.call(this, sheetModel.getDocModel(), initAttributes);
        DrawingModelMixin.call(this, sheetModel, function cloneConstructor(targetModel) {
            return new SheetShapeModel(targetModel, this.getExplicitAttributeSet(true));
        });

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            sheetModel = initAttributes = null;
        });

    } }); // class SheetShapeModel

    // class SheetImageModel ==================================================

    /**
     * The model of a image/picture object contained in a sheet.
     *
     * @constructor
     *
     * @extends ImageModel
     * @extends DrawingModelMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this image object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the object.
     */
    var SheetImageModel = ImageModel.extend({ constructor: function (sheetModel, initAttributes) {

        // base constructors --------------------------------------------------

        ImageModel.call(this, sheetModel.getDocModel(), initAttributes);
        DrawingModelMixin.call(this, sheetModel, function cloneConstructor(targetModel) {
            return new SheetImageModel(targetModel, this.getExplicitAttributeSet(true));
        });

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            sheetModel = initAttributes = null;
        });

    } }); // class SheetImageModel

    // class SheetDrawingCollection ===========================================

    /**
     * Represents the drawing collection of a single sheet in a spreadsheet
     * document.
     *
     * @constructor
     *
     * @extends DrawingCollection
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function SheetDrawingCollection(sheetModel) {

        // self reference
        var self = this;

        // the document model
        var docModel = sheetModel.getDocModel();

        // the column and row collections (size and formatting of all columns/rows)
        var colCollection = sheetModel.getColCollection();
        var rowCollection = sheetModel.getRowCollection();

        // maps drawing model indexes (used as positions) to drawing models
        var drawingModels = [];

        // base constructor ---------------------------------------------------

        DrawingCollection.call(this, docModel, {
            clientCreateModelHandler: createDrawingModel,
            clientResolveModelHandler: resolveDrawingModel,
            clientResolvePositionHandler: resolveDrawingPosition,
            clientRegisterModelHandler: registerDrawingModel,
            clientUnregisterModelHandler: unregisterDrawingModel
        });

        // private methods ----------------------------------------------------

        /**
         * Constructs a drawing model instance for the specified drawing type.
         *
         * @param {String} type
         *  The type of the drawing model to be created.
         *
         * @param {Object} [attributes]
         *  Initial formatting attributes for the drawing model.
         *
         * @returns {DrawingModel|Null}
         *  The new drawing model instance, if the passed type is supported.
         */
        function createDrawingModel(type, attributes) {

            switch (type) {
                case 'shape':
                    return new SheetShapeModel(sheetModel, attributes);

                case 'image':
                    return new SheetImageModel(sheetModel, attributes);

                case 'chart':
                    return new SheetChartModel(sheetModel, attributes);

                default:
                    if (/^(group|diagram|ole|undefined)$/.test(type)) {
                        return new SheetDrawingModel(sheetModel, type, attributes);
                    }
            }

            return null;
        }

        /**
         * Tries to return a drawing model contained in this sheet at the
         * specified position.
         *
         * @param {Array<Number>} position
         *  An arbitrary document position.
         *
         * @returns {DrawingModel|Null}
         *  The drawing model at the specified position, if available;
         *  otherwise null.
         */
        function resolveDrawingModel(position) {
            return ((position.length === 1) && drawingModels[position[0]]) || null;
        }

        /**
         * Tries to return the position of the specified drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  An arbitrary drawing model.
         *
         * @param {Array<Number>|Null}
         *  The position of the passed drawing model, if the model is part of
         *  this drawing collection, otherwise null.
         */
        function resolveDrawingPosition(drawingModel) {
            var index = drawingModels.indexOf(drawingModel);
            return (index >= 0) ? [index] : null;
        }

        /**
         * Registers a new drawing model that has just been inserted into the
         * drawing collection of this sheet.
         *
         * @param {DrawingModel} drawingModel
         *  The new drawing model.
         *
         * @param {Array<Number>} position
         *  The position of the new drawing object.
         *
         * @returns {Boolean}
         *  Whether the passed position is valid, and the drawing model has
         *  been registered successfully.
         */
        function registerDrawingModel(drawingModel, position) {
            var index = (position.length === 1) ? position[0] : -1;
            if ((index >= 0) && (index <= drawingModels.length)) {
                drawingModels.splice(index, 0, drawingModel);
                // add missing inactive anchor attributes that can be calculated from the existing anchor attributes
                var anchorAttrs = self.getInactiveAnchorAttributes(drawingModel.getMergedAttributeSet(true).drawing);
                drawingModel.setAttributes({ drawing: anchorAttrs }, { notify: 'never' });
                return true;
            }
            return false;
        }

        /**
         * Unregisters a drawing model that has been deleted from the drawing
         * collection of this sheet.
         *
         * @param {DrawingModel} drawingModel
         *  The deleted drawing model.
         *
         * @param {Array<Number>} position
         *  The position of the deleted drawing object.
         *
         * @returns {Boolean}
         *  Whether the passed position is valid, and the drawing model has
         *  been unregistered successfully.
         */
        function unregisterDrawingModel(drawingModel, position) {
            var index = (position.length === 1) ? position[0] : -1;
            if (drawingModel === drawingModels[index]) {
                drawingModels.splice(index, 1);
                return true;
            }
            return false;
        }

        /**
         * Returns all available information about the position of a drawing
         * object, according to the passed drawing attributes.
         *
         * @param {Object} drawingAttrs
         *  The values of the drawing anchor attributes, and the anchor type.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.displayAnchor=false]
         *      If set to true, uses the runtime behavior defined by the
         *      'anchorType' formatting attribute as anchor mode, instead of
         *      the internal anchor mode normally used.
         *
         * @returns {Object}
         *  The result object with position information for the passed drawing
         *  attribute set, in the following properties:
         *  - {String} anchorMode
         *      The effective anchor mode (one of 'absolute', 'oneCell', or
         *      'twoCell'), according to the anchor type attribute, and the
         *      passed options.
         *  - {Number} leftHmm
         *      The absolute horizontal offset of the drawing object in 1/100
         *      mm, independent from the current sheet zoom factor.
         *  - {Number} topHmm
         *      The absolute vertical offset of the drawing object in 1/100 mm,
         *      independent from the current sheet zoom factor.
         *  - {Number} widthHmm
         *      The width of the drawing object in 1/100 mm, independent from
         *      the current sheet zoom factor.
         *  - {Number} heightHmm
         *      The height of the drawing object in 1/100 mm, independent from
         *      the current sheet zoom factor.
         *  - {Number} left
         *      The absolute horizontal offset of the drawing object in pixels,
         *      according to the current sheet zoom factor.
         *  - {Number} top
         *      The absolute vertical offset of the drawing object in pixels,
         *      according to the current sheet zoom factor.
         *  - {Number} width
         *      The width of the drawing object in pixels, according to the
         *      current sheet zoom factor.
         *  - {Number} height
         *      The height of the drawing object in pixels, according to the
         *      current sheet zoom factor.
         *  - {Number} startCol
         *      The zero-based index of the column containing the leading
         *      border of the drawing object.
         *  - {Number} startColOffset
         *      The relative offset inside the column containing the leading
         *      border of the drawing object, in 1/100 mm.
         *  - {Number} startRow
         *      The zero-based index of the row containing the top border of
         *      the drawing object.
         *  - {Number} startRowOffset
         *      The relative offset inside the row containing the top border of
         *      the drawing object, in 1/100 mm.
         *  - {Number} endCol
         *      The zero-based index of the column containing the trailing
         *      border of the drawing object.
         *  - {Number} endColOffset
         *      The relative offset inside the column containing the trailing
         *      border of the drawing object, in 1/100 mm.
         *  - {Number} endRow
         *      The zero-based index of the row containing the bottom border of
         *      the drawing object.
         *  - {Number} endRowOffset
         *      The relative offset inside the row containing the bottom border
         *      of the drawing object, in 1/100 mm.
         *  - {Boolean} hiddenCols
         *      Whether the drawing object is located entirely in a range of
         *      hidden columns.
         *  - {Boolean} hiddenRows
         *      Whether the drawing object is located entirely in a range of
         *      hidden rows.
         */
        function getAnchorInformation(drawingAttrs, options) {

            // whether to use the display anchor instead of the anchor type
            var displayAnchor = Utils.getBooleanOption(options, 'displayAnchor', false);
            // the actual anchor mode used to find inactive anchor attributes
            var anchorMode = (displayAnchor ? DISPLAY_ANCHOR_MODES : INTERNAL_ANCHOR_MODES)[drawingAttrs.anchorType] || 'twoCell';
            // the resulting position
            var position = { anchorMode: anchorMode };
            // collection entry of the column/row collection
            var entryData = null;

            // get start position of the drawing object
            switch (anchorMode) {
                case 'twoCell':
                case 'oneCell':
                    // left position in 1/100 mm and pixels, start column index and offset
                    position.leftHmm = colCollection.getEntryOffsetHmm(drawingAttrs.startCol, drawingAttrs.startColOffset);
                    position.left = colCollection.getEntryOffset(drawingAttrs.startCol, drawingAttrs.startColOffset);
                    position.startCol = drawingAttrs.startCol;
                    position.startColOffset = drawingAttrs.startColOffset;

                    // top position in 1/100 mm and pixels, start row index and offset
                    position.topHmm = rowCollection.getEntryOffsetHmm(drawingAttrs.startRow, drawingAttrs.startRowOffset);
                    position.top = rowCollection.getEntryOffset(drawingAttrs.startRow, drawingAttrs.startRowOffset);
                    position.startRow = drawingAttrs.startRow;
                    position.startRowOffset = drawingAttrs.startRowOffset;
                    break;

                case 'absolute':
                    // left position in 1/100 mm and pixels, start column index and offset
                    entryData = colCollection.getEntryByOffset(drawingAttrs.left);
                    position.leftHmm = entryData.offsetHmm + entryData.relOffsetHmm;
                    position.left = entryData.offset + entryData.relOffset;
                    position.startCol = entryData.index;
                    position.startColOffset = entryData.relOffsetHmm;

                    // top position in 1/100 mm and pixels, start row index and offset
                    entryData = rowCollection.getEntryByOffset(drawingAttrs.top);
                    position.topHmm = entryData.offsetHmm + entryData.relOffsetHmm;
                    position.top = entryData.offset + entryData.relOffset;
                    position.startRow = entryData.index;
                    position.startRowOffset = entryData.relOffsetHmm;
                    break;
            }

            // get effective size of the drawing object
            switch (anchorMode) {
                case 'twoCell':
                    // width in 1/100 mm and pixels; end column index and offset
                    position.widthHmm = colCollection.getEntryOffsetHmm(drawingAttrs.endCol, drawingAttrs.endColOffset) - position.leftHmm;
                    position.width = colCollection.getEntryOffset(drawingAttrs.endCol, drawingAttrs.endColOffset) - position.left;
                    position.endCol = drawingAttrs.endCol;
                    position.endColOffset = drawingAttrs.endColOffset;

                    // height in 1/100 mm and pixels, end row index and offset
                    position.heightHmm = rowCollection.getEntryOffsetHmm(drawingAttrs.endRow, drawingAttrs.endRowOffset) - position.topHmm;
                    position.height = rowCollection.getEntryOffset(drawingAttrs.endRow, drawingAttrs.endRowOffset) - position.top;
                    position.endRow = drawingAttrs.endRow;
                    position.endRowOffset = drawingAttrs.endRowOffset;
                    break;

                case 'oneCell':
                case 'absolute':
                    // width in 1/100 mm and pixels; end column index and offset
                    entryData = colCollection.getEntryByOffset(position.leftHmm + drawingAttrs.width);
                    position.widthHmm = entryData.offsetHmm + entryData.relOffsetHmm - position.leftHmm;
                    position.width = entryData.offset + entryData.relOffset - position.left;
                    position.endCol = entryData.index;
                    position.endColOffset = entryData.relOffsetHmm;

                    // height in 1/100 mm and pixels, end row index and offset
                    entryData = rowCollection.getEntryByOffset(position.topHmm + drawingAttrs.height);
                    position.heightHmm = entryData.offsetHmm + entryData.relOffsetHmm - position.topHmm;
                    position.height = entryData.offset + entryData.relOffset - position.top;
                    position.endRow = entryData.index;
                    position.endRowOffset = entryData.relOffsetHmm;
                    break;
            }

            // whether the drawing is located entirely in hidden columns or rows
            position.hiddenCols = colCollection.isIntervalHidden(new Interval(position.startCol, position.endCol));
            position.hiddenRows = rowCollection.isIntervalHidden(new Interval(position.startRow, position.endRow));

            return position;
        }

        // protected methods --------------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * drawing objects from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {SheetDrawingCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // clone the contents of the source collection
            var iterator = collection.createModelIterator();
            IteratorUtils.forEach(iterator, function (drawingModel, result) {
                this.insertModelInstance(result.position, drawingModel.clone(sheetModel));
            }, this);
        };

        /**
         * Callback handler for the document operation 'insertDrawing'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertDrawing' document operation.
         *
         * @param {Array<Number>} position
         *  The insertion position of the new drawing object in this collection
         *  (without leading sheet index).
         */
        this.applyInsertDrawingOperation = function (context, position) {
            var result = this.insertModel(position, context.getStr('type'), context.getOptObj('attrs'));
            context.ensure(result, 'cannot insert drawing');
        };

        /**
         * Callback handler for the document operation 'deleteDrawing'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteDrawing' document operation.
         *
         * @param {Array<Number>} position
         *  The position of the drawing object to be deleted in this collection
         *  (without leading sheet index).
         */
        this.applyDeleteDrawingOperation = function (context, position) {
            var result = this.deleteModel(position);
            context.ensure(result, 'cannot delete drawing');
        };

        // public methods -----------------------------------------------------

        /**
         * Creates an iterator that visits all drawing models in this sheet.
         * The drawing models will be visited in order of their document
         * positions (Z order).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.deep=false]
         *      If set to true, drawing models embedded in other drawing models
         *      will be visited too. Otherwise, only the top-level drawing
         *      models in the sheet will be visited.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the iterator will visit the drawing models in
         *      reversed order.
         *  @param {String} [options.type]
         *      If specified, only the drawing models with that type will be
         *      visited by the iterator.
         *
         * @returns {Object}
         *  An iterator object that implements the standard EcmaScript iterator
         *  protocol, i.e. it provides the method next() that returns a result
         *  object with the following properties:
         *  - {Boolean} done
         *      If set to true, the collection has been visited completely. No
         *      more drawing models are available; this result object does not
         *      contain any other properties!
         *  - {DrawingModel} value
         *      The drawing model currently visited. This property will be
         *      omitted, if the iterator is done (see property 'done').
         *  - {Array<Number>} position
         *      The position of the drawing model in the sheet, to be used as
         *      position in drawing document operations. This  property will be
         *      omitted, if the iterator is done (see property  'done').
         *  - {Number} index
         *      The index of the drawing model in its parent (Z order). This
         *      property will be omitted, if the iterator is done (see property
         *      'done').
         */
        this.createModelIterator = function (options) {

            // whether to search deeply in embedded collections
// TODO            var deep = Utils.getBooleanOption(options, 'deep', false);
            // whether to restrict the iterator to a specific type
            var type = Utils.getStringOption(options, 'type', null);
            // whether to iterate in reversed order
            var reverse = Utils.getBooleanOption(options, 'reverse', false);

            // create an array iterator for the drawing models
            var iterator = IteratorUtils.createArrayIterator(drawingModels, { reverse: reverse });

            // convert array index to drawing position, filter for a specific drawing type
            return IteratorUtils.createTransformIterator(iterator, function (drawingModel, result) {
                if (type && (type !== drawingModel.getType())) { return null; }
                result.position = [result.index];
                return result;
            });
        };

        /**
         * Returns the inactive anchor attributes of the passed attribute set,
         * according to the anchor mode and anchor attributes.
         *
         * @param {Object} drawingAttrs
         *  The values of the drawing anchor attributes, and the anchor type.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.displayAnchor=false]
         *      If set to true, uses the runtime behavior defined by the
         *      'anchorType' formatting attribute as anchor mode, instead of
         *      the internal anchor mode normally used.
         *
         * @returns {Object}
         *  The current values of the inactive anchor attributes, according to
         *  the anchor mode.
         */
        this.getInactiveAnchorAttributes = function (drawingAttrs, options) {

            // current position of the drawing object
            var positionInfo = getAnchorInformation(drawingAttrs, options);

            switch (positionInfo.anchorMode) {
                case 'twoCell':
                    return {
                        left: positionInfo.leftHmm,
                        top: positionInfo.topHmm,
                        width: positionInfo.widthHmm,
                        height: positionInfo.heightHmm
                    };

                case 'oneCell':
                    return {
                        left: positionInfo.leftHmm,
                        top: positionInfo.topHmm,
                        endCol: positionInfo.endCol,
                        endColOffset: positionInfo.endColOffset,
                        endRow: positionInfo.endRow,
                        endRowOffset: positionInfo.endRowOffset
                    };

                case 'absolute':
                    return {
                        startCol: positionInfo.startCol,
                        startColOffset: positionInfo.startColOffset,
                        startRow: positionInfo.startRow,
                        startRowOffset: positionInfo.startRowOffset,
                        endCol: positionInfo.endCol,
                        endColOffset: positionInfo.endColOffset,
                        endRow: positionInfo.endRow,
                        endRowOffset: positionInfo.endRowOffset
                    };
            }

            return null;
        };

        /**
         * Returns the sheet rectangle in pixels covered by a drawing object
         * with the passed anchor attributes.
         *
         * @param {Object} drawingAttrs
         *  The values of the drawing anchor attributes, and the anchor type.
         *
         * @returns {Object|Null}
         *  The position of this drawing object in the sheet in pixels,
         *  according to the current sheet zoom factor, in the properties
         *  'left', 'top', 'width', and 'height'; or null if the drawing object
         *  is hidden by itself, or is located entirely in hidden columns or
         *  rows and thus invisible.
         */
        this.getRectangleForAnchor = function (drawingAttrs) {

            // the complete position information object
            var positionInfo = getAnchorInformation(drawingAttrs);

            // return null for hidden drawing objects
            if (positionInfo.hiddenCols || positionInfo.hiddenRows) {
                return null;
            }

            // keep minimum size of 5 pixels
            if (positionInfo.width < 5) {
                positionInfo.left = Math.max(0, positionInfo.left - Math.round((5 - positionInfo.width) / 2));
                positionInfo.width = 5;
            }
            if (positionInfo.height < 5) {
                positionInfo.top = Math.max(0, positionInfo.top - Math.round((5 - positionInfo.height) / 2));
                positionInfo.height = 5;
            }

            // return the pixel position and size as object
            return { left: positionInfo.left, top: positionInfo.top, width: positionInfo.width, height: positionInfo.height };
        };

        /**
         * Returns a drawing attribute set containing the anchor attributes
         * needed to set the position of a drawing object to the passed sheet
         * rectangle.
         *
         * @param {Object} rectangle
         *  The position of a drawing object in the sheet in pixels, according
         *  to the current sheet zoom factor, in the properties 'left', 'top',
         *  'width', and 'height'.
         *
         * @returns {Object}
         *  A drawing attribute set with the anchor attributes (contained in
         *  the sub-object 'drawing' of the resulting attribute set) for the
         *  new position of the drawing object.
         */
        this.getAttributeSetForRectangle = function (rectangle) {

            // collection entry of the column/row collection
            var entryData = null;
            // the new anchor attributes
            var attributes = {};

            // horizontal start position
            entryData = colCollection.getEntryByOffset(rectangle.left, { pixel: true });
            attributes.left = entryData.offsetHmm + entryData.relOffsetHmm;
            attributes.startCol = entryData.index;
            attributes.startColOffset = entryData.relOffsetHmm;

            // object width and horizontal end position
            entryData = colCollection.getEntryByOffset(rectangle.left + rectangle.width, { pixel: true });
            attributes.width = entryData.offsetHmm + entryData.relOffsetHmm - attributes.left;
            attributes.endCol = entryData.index;
            attributes.endColOffset = entryData.relOffsetHmm;

            // vertical start position
            entryData = rowCollection.getEntryByOffset(rectangle.top, { pixel: true });
            attributes.top = entryData.offsetHmm + entryData.relOffsetHmm;
            attributes.startRow = entryData.index;
            attributes.startRowOffset = entryData.relOffsetHmm;

            // object height and vertical end position
            entryData = rowCollection.getEntryByOffset(rectangle.top + rectangle.height, { pixel: true });
            attributes.height = entryData.offsetHmm + entryData.relOffsetHmm - attributes.top;
            attributes.endRow = entryData.index;
            attributes.endRowOffset = entryData.relOffsetHmm;

            return { drawing: attributes };
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations, and the undo operations, to insert new
         * drawing objects into this collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Number>} position
         *  The position of the first new drawing object. All drawing objects
         *  will be inserted subsequently.
         *
         * @param {Array<Object>} drawingData
         *  The type and formatting attributes of the drawing objects to be
         *  inserted. Each array element MUST be an object with the following
         *  properties:
         *  - {String} element.type
         *      The type identifier of the drawing object.
         *  - {Object} element.attrs
         *      Initial attributes for the drawing object, especially the
         *      anchor attributes specifying the physical position of the
         *      drawing object in the sheet.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked every time after a new
         *  'insertDrawing' operation has been generated for a drawing object.
         *  Allows to create additional operations for the drawing objects.
         *  Receives the following parameters:
         *  (1) {SheetOperationsGenerator} generator
         *      The operations generator (already containing the initial
         *      'insertDrawing' operation of the current drawing).
         *  (2) {Number} sheet
         *      The zero-based index of this sheet.
         *  (3) {Array<Number>} position
         *      The effective document position of the current drawing object
         *      in the sheet, as inserted into the 'insertDrawing' operation.
         *  (4) {Object} data
         *      The data element from the array parameter 'drawingData' that is
         *      currently processed.
         *  The callback function may return a promise to defer processing of
         *  subsequent drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected on any error.
         */
        this.generateInsertDrawingOperations = function (generator, position, drawingData, callback) {

            // the index of the sheet containing this collection
            var sheet = sheetModel.getIndex();

            // generate all operations for the new drawing objects, and the undo operations
            return this.iterateArraySliced(drawingData, function (data) {

                // generate the undo operations that delete the new drawings (in reversed order)
                generator.generateDrawingOperation(Operations.DELETE_DRAWING, position, null, { undo: true, prepend: true });

                // the merged attribute set (with default values for all missing attributes)
                var mergedAttributeSet = docModel.getStyleCollection('drawing').getMergedAttributes(data.attrs);
                // the inactive anchor attributes
                var anchorAttrs = self.getInactiveAnchorAttributes(mergedAttributeSet.drawing);
                // create a deep copy of the original attributes
                var attributeSet = _.copy(data.attrs, true);
                // the drawing attributes in the attribute set (create on demand)
                var drawingAttrrs = attributeSet.drawing || (attributeSet.drawing = {});

                // add the generic anchor attributes to the attribute set
                ['left', 'top', 'width', 'height'].forEach(function (attrName) {
                    if (attrName in anchorAttrs) {
                        drawingAttrrs[attrName] = anchorAttrs[attrName];
                    }
                });

                // create the 'insertDrawing' operation at the current position
                generator.generateDrawingOperation(Operations.INSERT_DRAWING, position, { type: data.type, attrs: attributeSet });
                // invoke callback function for more operations (may return a promise)
                var result = _.isFunction(callback) ? callback.call(self, generator, sheet, position.slice(), data) : null;

                // adjust document position for the next drawing object
                position = position.slice();
                position[position.length - 1] += 1;

                // callback may want to defer processing the following drawings
                return result;
            }, { delay: 'immediate' });
        };

        /**
         * Generates the operations, and the undo operations, to delete the
         * specified drawing objects from this collection.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Array<Number>>} positions
         *  An array with positions of the drawing objects to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected on any error.
         */
        this.generateDeleteDrawingOperations = function (generator, positions) {

            // sort by position, remove positions of embedded drawings whose parents will be deleted too
            DrawingUtils.optimizeDrawingPositions(positions);

            // create the delete operations (in reversed order), and the according undo operations
            return this.iterateArraySliced(positions, function (position) {

                // the drawing model to be deleted
                var drawingModel = resolveDrawingModel(position);
                // a local generator to be able to prepend all undo operations at once
                var generator2 = sheetModel.createOperationsGenerator();

                // generate the operations, and insert them into the passed generator
                drawingModel.generateDeleteOperations(generator2, position);
                generator.appendOperations(generator2);
                generator.prependOperations(generator2, { undo: true });

            }, { delay: 'immediate', reverse: true });
        };

        /**
         * Generates the operations, and the undo operations, to change the
         * formatting attributes of the specified drawing objects.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Array<Number>>} positions
         *  An array with positions of the drawing objects to be modified.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set to be applied at the drawing objects.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully.
         */
        this.generateChangeDrawingOperations = function (generator, positions, attributes) {

            // create the change operations, and the according undo operations
            return this.iterateArraySliced(positions, function (position) {
                var drawingModel = resolveDrawingModel(position);
                drawingModel.generateChangeOperations(generator, position, attributes);
            }, { delay: 'immediate' });
        };

        /**
         * Generates the operations and undo operations to update and restore
         * the inactive anchor attributes of the drawing objects, after the
         * size or visibility of the columns or rows in the sheet has been
         * changed.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateUpdateAnchorOperations = function (generator) {
            return this.iterateSliced(this.createModelIterator(), function (drawingModel, result) {
                drawingModel.generateUpdateAnchorOperations(generator, result.position);
            }, { delay: 'immediate' });
        };

        /**
         * Generates the operations and undo operations to update and restore
         * the formula expressions of the source links in all drawing objects
         * that refer to some cells in this document.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateFormulaOperations = function (generator, changeDesc) {

            // special processing needed when moving cells in a sheet (including inserting/deleting columns or rows) which may delete drawings
            if (changeDesc.type === 'moveCells') {
                return this.iterateSliced(this.createModelIterator({ deep: true, reverse: true }), function (drawingModel, result) {

                    // a local generator to be able to prepend all undo operations at once (for deleted drawings)
                    var generator2 = sheetModel.createOperationsGenerator();

                    // generate the operations, and insert them into the passed generator
                    drawingModel.generateFormulaOperations(generator2, result.position, changeDesc);
                    generator.appendOperations(generator2);
                    generator.prependOperations(generator2, { undo: true });

                }, { delay: 'immediate' });
            }

            return this.iterateSliced(this.createModelIterator({ deep: true }), function (drawingModel, result) {
                drawingModel.generateFormulaOperations(generator, result.position, changeDesc);
            }, { delay: 'immediate' });
        };

        // initialization -----------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            // drawing models are owned by the base class DrawingCollection, do not destroy them here!
            self = docModel = sheetModel = colCollection = rowCollection = drawingModels = null;
        });

    } // class SheetDrawingCollection

    // exports ================================================================

    // derive this class from class DrawingCollection
    return DrawingCollection.extend({ constructor: SheetDrawingCollection });

});
