/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/axismodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/chartstyleutil',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/drawing/titlemodel',
    'io.ox/office/spreadsheet/model/drawing/gridlinemodel'
], function (Utils, ChartStyleUtil, AttributedModel, Operations, TitleModel, GridlineModel) {

    'use strict';

    // class AxisModel ========================================================

    /**
     * Representation of a simgle axis in a chart drawing object.
     *
     * @constructor
     *
     * @extends AttributedModel
     */
    var AxisModel = AttributedModel.extend({ constructor: function (chartModel, axisId, otherType) {

        var self = this;
        var titleModel = null;
        var gridModel = null;
        var zeroLine = null;
        var format = null;

        var dataAxis = chartModel.getModelData()['axis' + axisId.toUpperCase()];
        if (dataAxis) {
            titleModel = new TitleModel(chartModel, axisId);
            gridModel = new GridlineModel(chartModel, axisId, {}, dataAxis);

            var otherAxis = chartModel.getModelData()['axis' + otherType.toUpperCase()];
            if (otherAxis.stripLines) {
                zeroLine = otherAxis.stripLines[0];
                if (zeroLine && !zeroLine.zero) {
                    zeroLine = null;
                }
            }
        }

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, chartModel.getDocModel(), null, { families: 'axis line character' });

        // private methods ----------------------------------------------------

        function isHorizontal() {
            return chartModel.getAxisIdForDrawing(axisId) === 'x';
        }

        function refreshMinMax() {
            var data = chartModel.getModelData();
            if (!data.series.length) {
                return;
            }

            if (isHorizontal()) {
                if (format && format.format) {
                    dataAxis.labelAutoFit = false;
                } else {
                    dataAxis.labelAutoFit = true;
                }
            } else {
                dataAxis.labelAutoFit = true;
            }

            if (axisId === 'x') {
                refreshAxisX(data);
            } else {
                refreshAxisY(data);
            }
        }

        function makeMin(min, max) {
            if ((!format || !format.date) && (max === min || (min / max < 5 / 6))) {
                return 0;
            }
            return min - ((max - min) / 2);
        }

        function makeMax(min, max) {
            var res = max + 0.2 * (max - min);
            if (res < max) {
                res = max;
            }
            return res;
        }

        function distance(a, b) {
            return Math.abs(a - b);
        }

        function updateInterval(number) {
            var normalized = Utils.normalizeNumber(number);

            var distances = [];
            distances.push({ value: 1, dist: distance(normalized.mant, 1) });
            distances.push({ value: 2, dist: distance(normalized.mant, 2) });
            distances.push({ value: 5, dist: distance(normalized.mant, 5) });
            distances.push({ value: 10, dist: distance(normalized.mant, 10) });

            var newMant = null;
            var minDist = 999999;

            distances.forEach(function (d) {
                if (d.dist < minDist) {
                    minDist = d.dist;
                    newMant = d.value;
                }
            });

            return newMant * Math.pow(10, normalized.exp);
        }

        function updateMinMax(minValue, maxValue, precision, intervalCount, xAxis, allLabels) {
            if (minValue < maxValue) {
                var interval = (maxValue - minValue) / intervalCount;
                interval = round(interval, precision - 1);
                interval = updateInterval(interval);

                if (interval === 0) {
                    interval = null;
                    dataAxis.minimum = minValue;
                    dataAxis.maximum = maxValue;
                    dataAxis.interval = null;
                } else {
                    var intCount = 1 + ((maxValue - minValue) / interval);

                    if (Math.abs(maxValue) < Math.abs(minValue)) {
                        maxValue = Utils.roundUp(maxValue, interval);
                        minValue = maxValue - (interval * intCount);
                    } else {
                        minValue = Utils.roundDown(minValue, interval);
                        maxValue = minValue + (interval * intCount);
                    }
                }
                setMinMax(minValue, maxValue, interval, xAxis, allLabels);
            }
        }

        function setMinMax(min, max, interval, xAxis, allLabels) {

            if (min < 0 && format && format.format && format.format.category === 'date') { min = 0; }

            dataAxis.minimum = min;
            dataAxis.maximum = max;
            dataAxis.interval = interval;

            var stripLines = dataAxis.stripLines;
            var i = 0;

            if (!interval) {
                if (stripLines.length > 1) {
                    for (i = stripLines.length - 1; i >= 0; i -= 1) {
                        var stripLine = stripLines[i];
                        if (!stripLine.zero) {
                            stripLines.splice(i, 1);
                        }
                    }
                }
                return;
            }

            var index = 0;
            var formatter = chartModel.getDocModel().getNumberFormatter();
            var steps = (dataAxis.maximum - dataAxis.minimum) / dataAxis.interval;
            var lastLine = null;
            var line = null;

            for (i = 0; i <= steps; i += 1) {
                var value = dataAxis.minimum + i * dataAxis.interval;

                line = stripLines[index];
                if (line && line.zero) {
                    index++;
                    line = stripLines[index];
                }
                if (!line) {
                    line = {
                        value: 0,
                        color: 'transparent',
                        label: '',
                        labelBackgroundColor: 'transparent',
                        labelPlacement: 'outside'
                    };
                    stripLines[index] = line;
                }
                line.labelFontColor = 'transparent';
                line.label = '';
                line.value = value;
                line.labelPlacement = 'outside';

                if (allLabels || dataAxis.labelAutoFit || (index % 2)) {
                    var realValue = value * Math.pow(10, format.exp);
                    if (format && format.format) {
                        line.label = formatter.formatValue(format.format, realValue);
                    } else {
                        line.label = formatter.formatStandardNumber(realValue, 8);
                    }
                    if (!line.label || (lastLine && lastLine.label === line.label)) {
                        line.label = '';
                        lastLine = null;
                    } else {
                        lastLine = line;
                    }

                    if (xAxis && lastLine && lastLine.label.length > 1) {
                        if (value === max && i === steps) {
                            line.label = '';
                        } else if (value === min && !i && dataAxis.labelAngle) {
                            line.label = '';
                        }
                    }
                }
                index++;
            }
            for (; index < stripLines.length; index++) {
                line = stripLines[index];
                line.labelFontColor = 'transparent';
                line.label = '';
            }
        }

        function refreshAxisX(data) {
            var area = chartModel.getMergedAttributeSet(true).chart.type.indexOf('area') === 0;
            if (!data.linearX && (chartModel.isXYType() || area)) {
                var minValue = Number.POSITIVE_INFINITY;
                var maxValue = Number.NEGATIVE_INFINITY;

                _.each(data.series, function (dd) {
                    _.each(dd.dataPoints, function (dataPoint) {
                        var value = dataPoint[axisId];
                        minValue = Math.min(minValue, value);
                        maxValue = Math.max(maxValue, value);
                    });
                });

                // TODO: disable this block to reproduce Bug 46447
                if (chartModel.isXYType()) {
                    var intervalCount = Utils.minMax(data.series.length, 2, 8);
                    var diff = maxValue - minValue;
                    maxValue +=  diff / intervalCount;
                    minValue -=  diff / intervalCount;
                }

                if (!isFinite(minValue) || !isFinite(maxValue)) {
                    setMinMax();
                    return;
                }
                if (minValue === maxValue) {
                    maxValue = minValue + 0.5;
                    minValue -= 0.5;
                }

                updateMinMax(minValue, maxValue, 2, 4, true, area);
            } else {
                setMinMax(null, null, null, true);
            }
        }

        function round(value, precision) {
            if ((precision > 0) && (precision < 20)) {
                return parseFloat(value.toPrecision(precision));
            }
            return value;
        }

        function refreshAxisY(data) {
            var chartAttrs = chartModel.getMergedAttributeSet(true).chart;
            var stacking = chartAttrs.stacking;

            var givenMin = null;
            var givenMax = null;

            var axisInfo = self.getMergedAttributeSet(true);
            if (axisInfo) {
                if (axisInfo.min !== 'auto') {
                    givenMin = axisInfo.min;
                }
                if (axisInfo.max !== 'auto') {
                    givenMax = axisInfo.max;
                }
            }

            var minValue = Number.POSITIVE_INFINITY;
            var maxValue = Number.NEGATIVE_INFINITY;

            if (stacking === 'percentStacked' || (stacking === 'stacked' && data.series.length > 1)) {
                var first = data.series[0];
                _.times(first.dataPoints.length, function (j) {
                    var value = 0;
                    _.each(data.series, function (dd) {
                        var jValue = dd.dataPoints[j];
                        if (!jValue) { return; }
                        var useValue = jValue[axisId];
                        value += useValue;
                    });
                    minValue = Math.min(minValue, value);
                    maxValue = Math.max(maxValue, value);
                });
            } else {
                _.each(data.series, function (dd) {
                    _.each(dd.dataPoints, function (dataPoint) {
                        var value = dataPoint[axisId];
                        minValue = Math.min(minValue, value);
                        maxValue = Math.max(maxValue, value);
                    });
                });
            }

            var ignoreMax = false;

            if (chartModel.isXYType()) {
                var diff = maxValue - minValue;
                maxValue += diff / 8;
                minValue -= diff / 8;
            } else if (stacking === 'percentStacked') {
                ignoreMax = true;
                format = {
                    exp: -2,
                    format: chartModel.getDocModel().getNumberFormatter().getParsedFormat('0%', { grammarId: 'ui' })
                };
                //Trial & error...
                if (maxValue >= 0 && minValue >= 0) {
                    // scenario 1
                    minValue = givenMin || 0;
                    maxValue = givenMax || 110;
                } else if (maxValue <= 0 && minValue <= 0) {
                    // scenario 2
                    maxValue = givenMax || 0;
                    minValue = givenMin || -110;
                } else {
                    // scenario 3
                    maxValue =  givenMax || 110;
                    minValue =  givenMin || -110;
                }
            } else {
                if (maxValue >= 0 && minValue >= 0) {
                    // scenario 1
                    minValue = givenMin || makeMin(minValue, maxValue);
                    maxValue = givenMax || makeMax(minValue, maxValue);
                } else if (maxValue <= 0 && minValue <= 0) {
                    // scenario 2
                    maxValue = givenMax || -makeMin(-maxValue, -minValue);
                    minValue = givenMin || -makeMax(-maxValue, -minValue);
                } else {
                    // scenario 3
                    var ma = maxValue;
                    var mi = minValue;
                    maxValue =  givenMax || makeMax(mi, ma);
                    minValue =  givenMin || -makeMax(-ma, -mi);
                }
            }

            if (!isFinite(minValue) || !isFinite(maxValue)) {
                setMinMax();
                return;
            }

            if (minValue === maxValue) {
                maxValue = minValue + 0.5;
                minValue -= 0.5;
            }

            var precision = 2;
            if (format.date) {
                precision = 0;
            }

            if (Math.abs(maxValue - minValue) > 0.00000000001) {
                minValue = round(minValue, precision);
                maxValue = round(maxValue, precision);
            }

            updateMinMax(minValue, maxValue, precision, 8, ignoreMax);
        }

        // public methods -----------------------------------------------------

        this.refreshInfo = function () {
            refreshMinMax();

            if (!dataAxis) { return; }

            var attrs = this.getMergedAttributeSet(true);

            ChartStyleUtil.handleLineProps(chartModel, attrs.line, dataAxis, 'line');
            ChartStyleUtil.handleLineProps(chartModel, attrs.line, dataAxis, 'tick');

            if (attrs.axis.label) {
                ChartStyleUtil.handleCharacterProps(chartModel, attrs.character, dataAxis, 'label');
            } else {
                dataAxis.labelFontColor = 'transparent';
                dataAxis.labelFontSize = 1;
            }

            if (zeroLine) {
                zeroLine.color = dataAxis.lineColor;
                zeroLine.thickness = dataAxis.lineThickness;
            }

            var overwriteLabels = false;
            var stripLines = dataAxis.stripLines;
            _.each(stripLines, function (stripLine) {
                if (!stripLine.zero) {
                    if (attrs.axis.label) {
                        ChartStyleUtil.handleCharacterProps(chartModel, attrs.character, stripLine, 'label');
                    } else {
                        stripLine.labelFontColor = 'transparent';
                        stripLine.labelFontSize = 1;
                    }
                    overwriteLabels = true;
                }
            });
            if (overwriteLabels) {
                dataAxis.labelFontColor = 'transparent';
            }

            gridModel.refreshInfo();

            var titleAtt = titleModel.getMergedAttributeSet(true);
            dataAxis.title = titleAtt.text.link[0];
            if (dataAxis.title) {
                ChartStyleUtil.handleCharacterProps(chartModel, titleAtt.character, dataAxis, 'title');
            }
        };

        this.setFormat = function (newFormat) {
            format = newFormat;
        };

        this.getGrid = function () {
            return gridModel;
        };

        this.getTitle = function () {
            return titleModel;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the undo operations needed to restore this chart axis.
         *
         * @param {SheetOperationsGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {Array<Number>} position
         *  The position of the parent chart object in the sheet, as expected
         *  by the method SheetOperationsGenerator.generateDrawingOperation().
         *
         * @returns {AxisModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator, position) {

            // restore this axis
            if (this.hasExplicitAttributes()) {
                var properties = { axis: axisId, attrs: this.getExplicitAttributeSet(true) };
                generator.generateDrawingOperation(Operations.SET_CHART_AXIS_ATTRIBUTES, position, properties, { undo: true });
            }

            // restore the axis title, and the grid line settings
            if (titleModel) { titleModel.generateRestoreOperations(generator, position); }
            if (gridModel) { gridModel.generateRestoreOperations(generator, position); }

            return this;
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            if (gridModel) { gridModel.destroy(); }
            if (titleModel) { titleModel.destroy(); }
            chartModel = gridModel = titleModel = null;
        });

    } }); // class AxisModel

    // exports ================================================================

    return AxisModel;

});
