/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/celloperationsbuilder', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Utils, Operations, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;

    // class JSONBuffer =======================================================

    /**
     * A two-dimensional buffer of JSON cell content objects, as used in cell
     * operations.
     *
     * @constructor
     */
    function JSONBuffer(builder) {

        this._builder = builder;
        this._matrix = [];

        // create an initial empty row
        this.addRow();

    } // class JSONBuffer

    // public methods ---------------------------------------------------------

    /**
     * Appends a new empty row to this cell data buffer.
     *
     * @param {Number} [count=1]
     *  The initial repetition count for the new row.
     */
    JSONBuffer.prototype.addRow = function (count) {
        this._matrix.push({ c: [], r: count || 1 });
    };

    /**
     * Returns whether the passed JSON data objects for a cell operation have
     * equal properties (ignoring the 'r' property for repetition).
     *
     * @param {Object} jsonData1
     *  The first JSON cell data object, as used in cell operations.
     *
     * @param {Object} jsonData2
     *  The second JSON cell data object, as used in cell operations.
     *
     * @returns {Boolean}
     *  Whether the passed JSON data objects for a cell operation have equal
     *  properties (ignoring the 'r' property for repetition).
     */
    JSONBuffer.prototype.isEqualJSON = function (jsonData1, jsonData2) {
        return (jsonData1.u === jsonData2.u) &&
            (jsonData1.v === jsonData2.v) &&
            (jsonData1.e === jsonData2.e) &&
            (jsonData1.f === jsonData2.f) &&
            (jsonData1.si === jsonData2.si) &&
            _.isEqual(jsonData1.sr, jsonData2.sr) &&
            this._builder.areEqualStyleIds(jsonData1.s, jsonData2.s);
    };

    /**
     * Appends the passed JSON cell data object to the end of this cell data
     * buffer. Tries to compress the buffer by updating the repetition count of
     * the last existing cell entry if possible.
     *
     * @param {Object} jsonData
     *  The JSON cell data object to be added to this buffer.
     *
     * @param {Number} [count=1]
     *  The repetition count for the passed JSON cell data.
     */
    JSONBuffer.prototype.addCell = function (jsonData, count) {
        var lastRow = _.last(this._matrix);
        var lastData = _.last(lastRow.c);
        count = count || 1;
        if (lastData && this.isEqualJSON(lastData, jsonData)) {
            lastData.r += count;
        } else {
            jsonData.r = count;
            lastRow.c.push(jsonData);
        }
    };

    /**
     * Generates the 'changeCells' operation for this cell data buffer.
     *
     * @param {SheetOperationsGenerator} generator
     *  The operations generator to be filled with the operation.
     *
     * @param {Address} address
     *  The address of the top-left cell of this data buffer.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.undo=false]
     *      If set to true, an undo operation will be created.
     *  @param {Boolean} [options.filter=false]
     *      If set to true, an operation for export filters will be created
     *      (remote editor clients will ignore the operation).
     */
    JSONBuffer.prototype.generateOperation = function (generator, address, options) {

        // clone the start address to be able to modify it below
        address = address.clone();

        // number of leading skipped cells in all rows
        var leadingSkip = Number.MAX_VALUE;
        // the JSON keys of all cell data arrays in the matrix
        var cellKeys = [];

        // remove trailing garbage from the rows, remove useless repetition properties from cell entries
        this._matrix.forEach(function (currRow) {

            // the cell data array
            var cells = currRow.c;

            // remove trailing skip cell entries
            var lastData = _.last(cells);
            if (lastData && CellOperationsBuilder.isEmptyJSON(lastData)) {
                cells.pop();
            }

            // collect number of leading skipped cells (ignore empty rows completely)
            var firstData = cells[0];
            if (firstData) {
                leadingSkip = CellOperationsBuilder.isEmptyJSON(firstData) ? Math.min(leadingSkip, firstData.r) : 0;
            }

            // remove empty cell arrays completely, collect unique keys for cell arrays
            if (cells.length === 0) {
                delete currRow.c;
                cellKeys.push(null);
            } else {
                cellKeys.push(Utils.stringifyJSON(cells));
            }

        }, this);

        // remove leading skipped rows from the matrix
        for (var firstRow = this._matrix[0]; firstRow && !firstRow.c; firstRow = this._matrix[0]) {
            address[1] += firstRow.r;
            this._matrix.shift();
            cellKeys.shift();
        }

        // maps unique keys of cell arrays to array indexes of row data objects
        var cellKeyMap = {};

        // process all rows of the matrix
        this._matrix.forEach(function (currRow, index) {

            // collapse rows with the same cell entries (repeat as long as the next row can be merged
            // with the current row; the next row entry will always be removed immediately from the array)
            var cellKey = cellKeys[index];
            while (cellKey === cellKeys[index + 1]) {
                currRow.r += this._matrix[index + 1].r;
                this._matrix.splice(index + 1, 1);
                cellKeys.splice(index + 1, 1);
            }

            // remove r:1 property from the row entry
            if (currRow.r === 1) { delete currRow.r; }

            // add array index of an existing cell array in a preceding row data entry
            if (cellKey) {
                var oldIndex = cellKeyMap[cellKey];
                if (_.isNumber(oldIndex)) {
                    currRow.c = oldIndex;
                    return;
                }
                cellKeyMap[cellKey] = index;
            }

            // further processing of the cell array
            var cells = currRow.c;
            if (!cells) { return; }

            // remove leading skipped cells from the row
            if (leadingSkip > 0) {
                var firstData = cells[0];
                if (firstData.r === leadingSkip) {
                    cells.shift();
                } else {
                    firstData.r -= leadingSkip;
                }
            }

            // remove r:1 properties from all cell entries
            cells.forEach(function (cellData) {
                if (cellData.r === 1) { delete cellData.r; }
            });

        }, this);

        // update start address according to leading skipped cells
        address[0] += leadingSkip;

        // remove trailing skipped rows
        var lastRow = _.last(this._matrix);
        if (lastRow && !('c' in lastRow)) {
            this._matrix.pop();
        }

        // nothing to do without any changed cells
        if (this._matrix.length > 0) {
            var properties = { contents: this._matrix };
            if (Utils.getBooleanOption(options, 'filter', false)) { properties.scope = 'filter'; }
            generator.generateCellOperation(Operations.CHANGE_CELLS, address, properties, options);
        }
    };

    /**
     * Releases all properties of this instance.
     */
    JSONBuffer.prototype.destroy = function () {
        this._builder = this._matrix = null;
    };

    // class CellOperationsBuilder ============================================

    /**
     * A builder to generate cell operations and undo operations for multiple
     * cells in a cell collection.
     *
     * @constructor
     *
     * @param {SheetModel} sheetModel
     *  The model of the sheet this instance is associated to.
     *
     * @param {SheetOperationsGenerator} generator
     *  The operations generator that will be filled with the new document
     *  operations.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Address} [initOptions.startAddress=Address.A1]
     *      The address of the first cell to generate operations for, if known.
     *      By default, address A1 will be used to be sure to include all cells
     *      in the generated operations.
     */
    function CellOperationsBuilder(sheetModel, generator, initOptions) {

        // the collection of cell auto-styles of the document
        this._autoStyles = sheetModel.getDocModel().getCellAutoStyles();

        // the configuration for the formula grammar used in operations
        this._formulaGrammar = sheetModel.getDocModel().getFormulaGrammar('op');

        // the cell collection this builder works on
        this._cellCollection = sheetModel.getCellCollection();

        // the operations generator
        this._generator = generator;

        // the JSON data for the cell operations
        this._cellBuffer = new JSONBuffer(this);

        // the JSON data for the undo operations
        this._undoBuffer = new JSONBuffer(this);

        // the cell address of the start cell for this builder
        this._startAddress = Utils.getOption(initOptions, 'startAddress', Address.A1).clone();

        // the cell address of the first cell not yet manipulated by this builder
        this._nextAddress = this._startAddress.clone();

        // the addresses of all cells changed by this builder
        this._changedAddresses = new AddressArray();

        // changed shared formula IDs
        this._sharedFormulas = new AddressArray();

    } // class CellOperationsBuilder

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed JSON data object for a cell operation will
     * not change a cell at all.
     *
     * @param {Object} jsonData
     *  The JSON cell data object, as used in cell operations.
     *
     * @returns {Boolean}
     *  Whether the JSON data object for a cell operation will not change a
     *  cell at all.
     */
    CellOperationsBuilder.isEmptyJSON = function (jsonData) {
        return !('u' in jsonData) && !('v' in jsonData) && !('e' in jsonData) &&
            !('f' in jsonData) && !('si' in jsonData) && !('sr' in jsonData) && !('s' in jsonData);
    };

    // private methods --------------------------------------------------------

    /**
     * Assigns an error code as string to the 'e' operation property, otherwise
     * the passed value to the 'v' property of the passed JSON cell object, to
     * be able to distinguish plain strings looking like error codes.
     *
     * @param {Object} jsonData
     *  The JSON cell data object to be extended with a value property.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The cell value to be inserted into the passed JSON object. An error
     *  code will be inserted as 'e' string property, all other values will be
     *  inserted as 'v' property.
     */
    CellOperationsBuilder.prototype._putValue = function (jsonData, value) {
        if (value instanceof ErrorCode) {
            jsonData.e = this._formulaGrammar.getErrorName(value);
        } else {
            jsonData.v = value;
        }
    };

    /**
     * Appends the JSON data for a single cell to the data buffers.
     */
    CellOperationsBuilder.prototype._createCell = function (contents, options) {

        // the model of the cell to be manipulated
        var cellModel = this._cellCollection.getCellModel(this._nextAddress);
        // whether to delete the cell model explicitly
        var deleteCell = contents.u === true;
        // the resulting JSON contents object
        var cellJSON = {};
        // the resulting JSON contents object for the undo operation
        var undoJSON = {};

        // true if shared formula cell address should be collected if cell changes (value, formula or delete cell) occurred.
        var collectSharedFormulas = Utils.getBooleanOption(options, 'collectsharedformulas', true);
        // True if the value formula is changed or the cell will be deleted
        var cellValueChanges = false;

        // create the properties in the JSON data that will really change the cell model
        if (!deleteCell) {

            // copy the cell value to the JSON data (unless it does not change)
            var oldValue = cellModel ? cellModel.v : null;
            if (('v' in contents) && (contents.v !== oldValue)) {
                this._putValue(cellJSON, contents.v);
                this._putValue(undoJSON, oldValue);
                cellValueChanges = true;
            }

            // copy the cell formula to the JSON data (unless it does not change)
            var oldFormula = cellModel ? cellModel.f : null;
            if (('f' in contents) && (contents.f !== oldFormula)) {
                cellJSON.f = contents.f;
                undoJSON.f = oldFormula;
                cellValueChanges = true;
            }

            // copy the cell shared formula id to the JSON data (unless it does not change)
            var oldSharedFormulaID = cellModel ? cellModel.si : null;
            if (('si' in contents) && (contents.si !== oldSharedFormulaID)) {
                cellJSON.si = contents.si;
                undoJSON.si = oldSharedFormulaID;
            }

            // copy the cell shared formula range to the JSON data (unless it does not change)
            var oldSharedFormulaRange = cellModel ? cellModel.sr : null;
            if (('sr' in contents) && ((contents.sr === null && oldSharedFormulaRange !== null) || (contents.sr !== null && oldSharedFormulaRange === null) || (contents.sr && oldSharedFormulaRange && contents.sr.differs(oldSharedFormulaRange)))) {
                cellJSON.sr = contents.sr ? contents.sr.toJSON() : null;
                undoJSON.sr = oldSharedFormulaRange ? oldSharedFormulaRange.toJSON() : null;
            }
            if ('sa' in contents) {
                undoJSON.sa = contents.sa;
            }

            // the default column/row auto-style identifier for undefined cells, with current column/row defaults
            var defStyleId = this._cellCollection.getDefaultStyleId(this._nextAddress);
            // range mode 'row:merge': ignore the row style that has been set during this operation cycle when creating the cell auto-style
            var ignoreRowStyle = contents.rangeMode === 'row:merge';
            // the identifier of the base auto-style used to create the new cell auto-style
            var baseStyleId = cellModel ? cellModel.s : ignoreRowStyle ? this._cellCollection.getDefaultStyleId(this._nextAddress, true) : defStyleId;
            // the resulting auto-style identifier (try to get a cached identifier from a previous call)
            var newStyleId = this._generator.generateAutoStyle(baseStyleId, contents);

            // When changing an existing cell model, add the auto-style if it changes, also if set from or to the default style.
            if (cellModel && !this.areEqualStyleIds(cellModel.s, newStyleId)) {
                cellJSON.s = newStyleId;
                undoJSON.s = cellModel.s;
            }

            // When the cell does not exist, the auto-style must be set if it differs from the default column/row
            // style (also set the default style to override an existing row style); or when creating the cell due
            // to a new value or formula, and the auto-style does not remain the default auto-style.
            if (!cellModel && (!this.areEqualStyleIds(defStyleId, newStyleId) || (!_.isEmpty(cellJSON) && !this.isDefaultStyleId(newStyleId)))) {
                cellJSON.s = newStyleId;
            }

            // Check whether to undefine (delete) an existing cell model implicitly: It must be (or become) blank, it must not
            // contain a formula, and its auto-style must be (or become) equal to the default auto-style of the row or column.
            deleteCell = cellModel &&
                (('v' in contents) ? (contents.v === null) : (oldValue === null)) &&
                (('f' in contents) ? (contents.f === null) : (oldFormula === null)) &&
                (('si' in contents) ? (contents.si === null) : (oldSharedFormulaID === null)) &&
                (('sr' in contents) ? (contents.sr === null) : (oldSharedFormulaRange === null)) &&
                this.areEqualStyleIds(defStyleId, newStyleId);

            // the cell simply needs to be deleted on undo, if it does not exist yet and will be created
            if (!cellModel && !_.isEmpty(cellJSON)) {
                undoJSON = { u: true };
            }
        }

        // create simple contents objects when an existing cell model will be deleted
        // (the flag 'deleteCell' may have been set in the previous if-block)
        if (deleteCell && cellModel) {

            // the property 'u' marks a cell to be deleted (undefined)
            cellJSON = { u: true };

            // restore the cell value, formula, and auto-style
            undoJSON = {};
            if (cellModel.v !== null) { this._putValue(undoJSON, cellModel.v); }
            if (cellModel.f !== null) { undoJSON.f = cellModel.f; }
            if (cellModel.si !== null) { undoJSON.si = cellModel.si; }
            if (cellModel.sr !== null) { undoJSON.sr = cellModel.sr.toJSON(); }
            if (!this.isDefaultStyleId(cellModel.s)) { undoJSON.s = cellModel.s; }

            cellValueChanges = true;
        }

        // Collect the Shared Formula IDs
        if (!deleteCell && cellValueChanges && collectSharedFormulas && cellModel && cellModel.si !== null) {
            var sharedFormulaAddresses = this._sharedFormulas[cellModel.si];
            if (!sharedFormulaAddresses) {
                sharedFormulaAddresses = new AddressArray();
                this._sharedFormulas[cellModel.si] = sharedFormulaAddresses;
            }
            sharedFormulaAddresses.push(cellModel.a.clone());
        }

        // whether the cell will be changed at all (method JSONBuffer.addCell() changes the JSON object in-place)
        var changed = !_.isEmpty(cellJSON);

        // insert the JSON objects into the data buffers, update addresses
        this._cellBuffer.addCell(cellJSON);
        this._undoBuffer.addCell(undoJSON);

        return changed;
    };

    /**
     * Destroy the JSONBuffers and delete all properties.
     */
    CellOperationsBuilder.prototype._reset = function () {
        // base class will delete all instance properties
        this._cellBuffer.destroy();
        this._undoBuffer.destroy();

        // reset all properties to prevent dangling references
        _.each(this, function (value, name, self) { delete self[name]; });
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the passed auto-style identifier is considered to be the
     * default auto-style. Convenience shortcut for the same method of the
     * auto-style collection.
     */
    CellOperationsBuilder.prototype.isDefaultStyleId = function (styleId) {
        return this._autoStyles.isDefaultStyleId(styleId);
    };

    /**
     * Returns whether the passed auto-style identifiers are considered to be
     * equal. Convenience shortcut for the same method of the auto-style
     * collection.
     */
    CellOperationsBuilder.prototype.areEqualStyleIds = function (styleId1, styleId2) {
        return this._autoStyles.areEqualStyleIds(styleId1, styleId2);
    };

    /**
     * Processes the passed cell properties, and stores all data needed to be
     * inserted into the resulting cell operations.
     *
     * @param {Address} address
     *  The address of the cell to be processed. The addresses passed to
     *  repeated invocations of this method MUST be increasing!
     *
     * @param {Object} contents
     *  The cell properties to be applied to the specified cell.
     *
     * @param {Object} count
     *  The repetition count for the cell data (consecutive cells in the
     *  current row).
     */
    CellOperationsBuilder.prototype.createCells = function (address, contents, count, options) {

        // jump to a new row, if the row index changes
        var incRows = address[1] - this._nextAddress[1];
        if (incRows > 0) {
            this._nextAddress[0] = this._startAddress[0]; // back to start column
            this._nextAddress[1] = address[1];

            // create separate entries for the skipped empty rows
            if (incRows > 1) {
                this._cellBuffer.addRow(incRows - 1);
                this._undoBuffer.addRow(incRows - 1);
            }
            this._cellBuffer.addRow();
            this._undoBuffer.addRow();
        }

        // skip cells in the current row
        var incCols = address[0] - this._nextAddress[0];
        if (incCols > 0) {
            this._cellBuffer.addCell({}, incCols);
            this._undoBuffer.addCell({}, incCols);
            this._nextAddress[0] = address[0];
        }

        // generate the JSON data for the passed cell properties repeatedly
        for (var index = 0; index < count; index += 1) {
            if (this._createCell(contents, options)) {
                this._changedAddresses.push(this._nextAddress.clone());
            }
            this._nextAddress[0] += 1;
        }
    };

    /**
     * Generates the remaining document operation, and the undo operation, for
     * all cells collected by this builder instance, and invalidates the entire
     * builder instance.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.createCells=true]
     *      Whether to create the document operations for the changed cells.
     *  @param {Boolean} [options.createUndo=true]
     *      Whether to create the undo operations for the changed cells.
     *  @param {Boolean} [options.filter=false]
     *      If set to true, operations for export filters will be created
     *      (remote editor clients will ignore the operations).
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved when all operations have been
     *  finalized. The promise returns the addresses (RangeArray) of all cell ranges containing changed cells.
     */
    CellOperationsBuilder.prototype.finalizeOperations = SheetUtils.profileMethod('CellOperationsBuilder.finalizeOperations()', function (options) {

        var self = this;
        // whether to generate the operation for the export filters only
        var filter = Utils.getBooleanOption(options, 'filter', false);

        // generate the undo operations
        if (!filter && Utils.getBooleanOption(options, 'createUndo', true)) {
            this._undoBuffer.generateOperation(this._generator, this._startAddress, { undo: true });
        }

        // generate operations for the collected cell data
        if (Utils.getBooleanOption(options, 'createCells', true)) {
            this._cellBuffer.generateOperation(this._generator, this._startAddress, { filter: filter });
        }

        // return the changed cell ranges (merged from the changed cell addresses)
        var changedRanges = RangeArray.mergeAddresses(this._changedAddresses);

        var promise;
        if (_.isEmpty(this._sharedFormulas)) {
            promise = this._cellCollection.createResolvedPromise();
        } else {
            promise = this._cellCollection.generateSharedFormulaUpdateOperations(this._generator, this._sharedFormulas);
        }

        return promise.then(function () {
            // base class will delete all instance properties
            self._reset();
            return changedRanges;
        });
    });

    // exports ================================================================

    return CellOperationsBuilder;

});
