/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellmodel', [
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (ErrorCode, TokenArray) {

    'use strict';

    // class CellModel ========================================================

    /**
     * Light-weight representation of a defined cell in a cell collection.
     *
     * @constructor
     *
     * @param {Address} address
     *  The address of the cell model.
     *
     * @property {Address} a
     *  The address of the cell represented by this instance.
     *
     * @property {Number|String|Boolean|ErrorCode|Null} v
     *  The typed value, or formula result, of the cell. The value null is used
     *  to represent a blank cell.
     *
     * @property {String|Null} f
     *  The native formula expression of the cell (without the leading equality
     *  sign); or null for simple value cells without a formula.
     *
     * @property {Number|Null} si
     *  The identifier of a shared formula this cell refers to; or null if this
     *  cell is not part of a shared formula.
     *
     * @property {Range|Null} sr
     *  The bounding range of the shared formula this cell refers to; or null
     *  if this cell is not the anchor cell of a shared formula.
     *
     * @property {String} s
     *  The identifier of the auto-style for the cell.
     *
     * @property {String|Null} d
     *  The formatted  display string of the cell, according to its current
     *  value and number format. May be null, if the value cannot be formatted
     *  with the current number format. In this case, the cell renderer will
     *  show a 'railroad track' error (repeated hash marks) in the cell.
     *
     * @property {TokenArray|Null} t
     *  The token array containing the parsed formula expression stored in the
     *  property 'f' of this cell model.
     *
     * @property {String|Null} url
     *  The dynamic URL returned by a cell formula (HYPERLINK function).
     *
     * @property {CellModelVector|Null} cv
     *  Refernce to the cell model vector in the column matrix that contains
     *  this cell model; or null, if this cell model is currently not part of
     *  the column matrix.
     *
     * @property {CellModelVector|Null} rv
     *  Refernce to the cell model vector in the row matrix that contains this
     *  cell model; or null, if this cell model is currently not part of the
     *  row matrix.
     */
    function CellModel(address) {

        // 'real' model properties
        this.a = address.clone();
        this.v = null;
        this.f = null;
        this.si = null;
        this.sr = null;
        this.s = '';

        // calculated properties, cached for performance
        this.d = '';
        this.t = null;
        this.url = null;

        // references to cell vectors in the column/row matrixes
        this.cv = null;
        this.rv = null;

    } // class CellModel

    // static methods ---------------------------------------------------------

    /**
     * Compares the addresses of the passed cell models for a strict ordering,
     * e.g. for sorting cell models.
     *
     * @param {CellModel} cellModel1
     *  The first cell model whose address will be compared with the address of
     *  the second cell model.
     *
     * @param {CellModel} cellModel2
     *  The second cell model whose address will be compared with the address
     *  of the first cell model.
     *
     * @returns {Number}
     *  An integer specifying how the addresses of the passed cell models are
     *  related to each other. See method Address.compare() for details.
     */
    CellModel.compare = function (cellModel1, cellModel2) {
        return cellModel1.a.compareTo(cellModel2.a);
    };

    /**
     * Returns whether the passed cell model contains a value (is not blank).
     * Can be used as callback predicate for filters.
     *
     * @param {CellModel} [cellModel]
     *  A cell model, or null, or undefined. The latter will be treated as
     *  undefined blank cells, and this method will return false.
     *
     * @returns {Boolean}
     *  Whether the passed cell model contains a value (is not blank).
     */
    CellModel.hasValue = function (cellModel) {
        return cellModel && !cellModel.isBlank();
    };

    /**
     * Returns whether the passed cell model contains a formula. Can be used as
     * callback predicate for filters.
     *
     * @param {CellModel} [cellModel]
     *  A cell model, or null, or undefined. The latter will be treated as
     *  undefined blank cells, and this method will return false.
     *
     * @returns {Boolean}
     *  Whether the passed cell model contains a formula.
     */
    CellModel.hasFormula = function (cellModel) {
        return cellModel && cellModel.isFormula();
    };

    /**
     * Returns whether the passed cell model contains a formula or a shared formula. Can be used as
     * callback predicate for filters.
     *
     * @param {CellModel} [cellModel]
     *  A cell model, or null, or undefined. The latter will be treated as
     *  undefined blank cells, and this method will return false.
     *
     * @returns {Boolean}
     *  Whether the passed cell model contains a formula or a shared formula.
     */
    CellModel.hasAnyFormula = function (cellModel) {
        return cellModel && (cellModel.isFormula() || cellModel.isSharedFormula());
    };

    /**
     * Returns whether the passed cell model contains a shared formula. Can be used as
     * callback predicate for filters.
     *
     * @param {CellModel} [cellModel]
     *  A cell model, or null, or undefined. The latter will be treated as
     *  undefined blank cells, and this method will return false.
     *
     * @returns {Boolean}
     *  Whether the passed cell model contains a shared formula.
     */
    CellModel.hasSharedFormula = function (cellModel) {
        return cellModel && cellModel.isSharedFormula();
    };

    /**
     * Returns the column index of the passed cell model. Used in sorted model
     * arrays as sorter callback.
     *
     * @param {CellModel} cellModel
     *  The cell model to return the column index for.
     *
     * @returns {Number}
     *  The column index of the passed cell model.
     */
    CellModel.col = function (cellModel) {
        return cellModel.a[0];
    };

    /**
     * Returns the row index of the passed cell model. Used in sorted model
     * arrays as sorter callback.
     *
     * @param {CellModel} cellModel
     *  The cell model to return the row index for.
     *
     * @returns {Number}
     *  The row index of the passed cell model.
     */
    CellModel.row = function (cellModel) {
        return cellModel.a[1];
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns an exact clone of this cell model.
     *
     * @param {SheetModel} sheetModel
     *  The target sheet model that will contain this cell model.
     *
     * @returns {CellModel}
     *  An exact clone of this cell model for the specified sheet model.
     */
    CellModel.prototype.clone = function (sheetModel) {
        var cellModel = new CellModel(this.a);
        cellModel.v = this.v;
        cellModel.f = this.f;
        cellModel.si = this.si;
        cellModel.sr = this.sr ? this.sr.clone() : null;
        cellModel.s = this.s;
        cellModel.d = this.d;
        cellModel.url = this.url;
        cellModel.getTokenArray(sheetModel);
        return cellModel;
    };

    /**
     * Returns a unique key for the cell address of this cell model, intended
     * to be used as map key.
     *
     * @returns {String}
     *  A unique key for the cell address of this cell model.
     */
    CellModel.prototype.key = function () {
        return this.a.key();
    };

    /**
     * Returns whether this cell model is blank (no value, and no formula).
     *
     * @returns {Boolean}
     *  Whether this cell model is blank (no value, and no formula).
     */
    CellModel.prototype.isBlank = function () {
        return (this.v === null) && (this.f === null);
    };

    /**
     * Returns whether this cell model contains a number value.
     *
     * @returns {Boolean}
     *  Whether this cell model contains a number value.
     */
    CellModel.prototype.isNumber = function () {
        return typeof this.v === 'number';
    };

    /**
     * Returns whether this cell model contains a string value.
     *
     * @returns {Boolean}
     *  Whether this cell model contains a string value.
     */
    CellModel.prototype.isText = function () {
        return typeof this.v === 'string';
    };

    /**
     * Returns whether this cell model contains a boolean value.
     *
     * @returns {Boolean}
     *  Whether this cell model contains a boolean value.
     */
    CellModel.prototype.isBoolean = function () {
        return typeof this.v === 'boolean';
    };

    /**
     * Returns whether the cell contains an error code.
     *
     * @returns {Boolean}
     *  Whether the cell contains an error code.
     */
    CellModel.prototype.isError = function () {
        return this.v instanceof ErrorCode;
    };

    /**
     * Returns whether this cell model contains a formula expression or is a shared formula.
     *
     * @returns {Boolean}
     *  Whether this cell model contains a formula expression or a shared formula.
     */
    CellModel.prototype.isFormula = function () {
        return !!this.f;
    };

    /**
     * Returns if the cell model contains a shared formula.
     *
     * @returns {Boolean} true if the cell model contains a shared formula, otherwise false.
     */
    CellModel.prototype.isSharedFormula = function () {
        return this.si !== null;
    };

    /**
     * Returns if the cell model contains a shared formula but not a formula.
     *
     * @returns {Boolean} true if the cell model contains a shared formula without formula, otherwise false.
     */
    CellModel.prototype.isSharedFormulaWithoutFormula = function () {
        return this.si !== null && !this.f;
    };

    /**
     * Returns the token array containing the parsed formula expression of this
     * cell model.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing this cell model, needed to create a new
     *  token array instance on demand.
     *
     * @returns {TokenArray|Null}
     *  The token array of this cell model; or null, if the cell model does not
     *  contain a formula expression.
     */
    CellModel.prototype.getTokenArray = function (sheetModel) {
        if (!this.f || this.t) { return this.t; }
        this.t = new TokenArray(sheetModel, 'cell');
        this.t.parseFormula('op', this.f, { refAddress: this.a });
        return this.t;
    };

    /**
     * Returns the formula expression of this cell model.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing this cell model, needed to create a new
     *  token array instance on demand.
     *
     * @param {String} grammarId
     *  The identifier of the formula grammar for the formula expression.
     *
     * @returns {String|Null}
     *  The formula expression of this cell model; or null, if the cell model
     *  does not contain a formula expression.
     */
    CellModel.prototype.getFormula = function (sheetModel, grammarId) {
        // shortcut for operation grammar: return the 'f' property directly
        return !this.isFormula() ? null : (grammarId === 'op') ? this.f : this.getTokenArray(sheetModel).getFormula(grammarId, { refAddress: this.a, targetAddress: this.a });
    };

    /**
     * Changes the value of this cell model, and returns whether the cell value
     * has changed.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The new value for the cell.
     *
     * @returns {Boolean}
     *  Whether the cell value has changed.
     */
    CellModel.prototype.setValue = function (value) {
        if (this.v === value) { return false; }
        this.v = value;
        return true;
    };

    /**
     * Changes the formula expression of this cell model, and returns whether
     * the formula expression has changed.
     *
     * @param {String|Null} formula
     *  The new formula expression for the cell. The value null will delete the
     *  formula expression, and the cell becomes a simple value cell.
     *
     * @returns {Boolean}
     *  Whether the formula expression has changed.
     */
    CellModel.prototype.setFormula = function (formula) {
        formula = formula || null;
        if (this.f === formula) { return false; }
        this.f = formula;
        this.t = null;
        return true;
    };

    /**
     * Change the value of the shared formula id.
     *
     * @param {Number|Null} id
     *  The identifier of the shared formula, or null to remove the identifier
     *  of a shared formula from this cell model.
     *
     * @returns {Boolean}
     *  Whether the identifier of the shared formula has changed.
     */
    CellModel.prototype.setSharedFormulaID = function (id) {
        if (this.si === id) { return false; }
        this.si = id;
        return true;
    };

    /**
     * Changes the bounding range of the shared formula.
     *
     * @param {Range|Null} range
     *  The new bounding range of the shared formula, or null to remove the
     *  bounding range from this cell model.
     *
     * @returns {Boolean}
     *  Whether the bounding range of the shared formula has changed.
     */
    CellModel.prototype.setSharedFormulaRange = function (range) {
        if (((this.sr === null) && (range === null)) ||
            (this.sr && range && this.sr.equals(range))) {
            return false;
        }
        this.sr = range ? range.clone() : null;
        return true;
    };

    /**
     * Changes the dynamic URL of this cell model, and returns whether the cell
     * URL has changed.
     *
     * @param {String|Null} url
     *  The new dynamic URL for the cell, or null to remove the current URL.
     *
     * @returns {Boolean}
     *  Whether the dynamic cell URL has changed.
     */
    CellModel.prototype.setURL = function (url) {
        if (this.url === url) { return false; }
        this.url = url;
        return true;
    };

    // exports ================================================================

    return CellModel;

});
