/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/drawingcontroller', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/view/chartcreator',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Color, DrawingUtils, DrawingLabels, Operations, TokenArray, ChartCreator, gt) {

    'use strict';

    // class DrawingControllerMixin ===========================================

    /**
     * Drawing controller (will) encapsulate(s) all drawing features of the controller
     * For now Chart are part of Drawings!
     *
     * @constructor
     */
    function DrawingControllerMixin(app, docModel, docView) {

        var self = this,

            sourceSelector = false,

            chartSourceToken = null,

            // the token arrays containing the highlighted ranges of the selected drawing object
            tokenArrays = [],

            // the unique identifier of the range highlighting mode
            highlightUid = null;

        // private methods ----------------------------------------------------

        /**
         * Shows a warning alert box for the 'exchange direction' function for
         * chart series.
         */
        function yellChartDirection(type) {

            switch (type) {
                case 'sheets':
                    docView.yell({
                        type: 'warning',
                        message:
                            //#. Warning text: A chart object in a spreadsheet contains complex data source (cells from multiple sheets).
                            gt('Data references are located on different sheets.')
                    });
                    break;

                case 'directions':
                    docView.yell({
                        type: 'warning',
                        message:
                            //#. Warning text: A chart object in a spreadsheet contains complex data source (not a simple cell range).
                            gt('Data references are too complex for this operation.')
                    });
                    break;

                default:
                    Utils.warn('DrawingController.yellChartDirection(): unknown warning type: "' + type + '"');
            }

            return $.Deferred().reject();
        }

        /**
         * changeSelection
         * if only one drawing is selected and this is a chart, the function highlights its source cells via token array
         */
        function changeSelectionHandler(evt, selection) {

            if (highlightUid) {
                docView.endRangeHighlighting(highlightUid);
                highlightUid = null;
            }

            tokenArrays = [];

            if (selection.drawings.length === 1) {

                var chartModel = docView.getDrawingCollection().findModel(selection.drawings[0], { type: 'chart' });

                if (chartModel) {
                    chartModel.getTokenArrays(tokenArrays);
                }
            }

            // highlight the source ranges of the chart in the active sheet
            if (tokenArrays.length > 0) {
                highlightUid = docView.startRangeHighlighting(tokenArrays, { priority: true, resolveNames: true });
            }
        }

        /**
         * Returns the position of the selected drawing object, if exactly one
         * drawing object is currently selected.
         *
         * @returns {Array<Number>|Null}
         *  The position of the selected drawing object in the active sheet
         *  (without leading sheet index), if available; otherwise null.
         */
        function getDrawingPosition() {
            var drawingPositions = docView.getSelectedDrawings();
            return (drawingPositions.length === 1) ? drawingPositions[0] : null;
        }

        /**
         * @returns {DrawingModel|Null}
         *  The drawing model currently selected.
         */
        function getDrawingModel(type) {
            var drawingIndices = getDrawingPosition();
            if (drawingIndices) {
                return docView.getDrawingCollection().findModel(drawingIndices, { type: type });
            }
        }

        /**
         * Generates and applies a document operation for a chart object with correct undo operation.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the drawing operations have
         *  been applied successfully, or that will be rejected on any error.
         */
        function createAndApplyChartOperation(chartModel, opName, properties, callback) {
            return docView.getSheetModel().createAndApplyOperations(function (generator) {

                var position = getDrawingPosition();
                var promise = callback ? $.when(callback(generator, position)) : $.when();

                return promise.done(function () {
                    generateChartOperation(generator, chartModel, opName, position, properties);
                });
            }, { storeSelection: true });
        }

        /**
         * Generates a document operation for a chart object with correct undo operation.
         */
        function generateChartOperation(generator, chartModel, opName, position, properties) {
            var undoProperties = null;
            switch (opName) {
                case Operations.SET_CHART_LEGEND_ATTRIBUTES:
                    var legendModel = chartModel.getLegendModel(properties.axis);
                    undoProperties = { attrs: legendModel.getUndoAttributeSet(properties.attrs) };
                    break;

                case Operations.SET_CHART_TITLE_ATTRIBUTES:
                    var titleModel = chartModel.getTitleModel(properties.axis);
                    undoProperties = { axis: properties.axis, attrs: titleModel.getUndoAttributeSet(properties.attrs) };
                    break;

                case Operations.SET_CHART_AXIS_ATTRIBUTES:
                    var axisModel = chartModel.getAxisModel(properties.axis);
                    undoProperties = { axis: properties.axis, attrs: axisModel.getUndoAttributeSet(properties.attrs) };
                    break;

                case Operations.SET_CHART_GRIDLINE_ATTRIBUTES:
                    var gridModel = chartModel.getAxisModel(properties.axis).getGrid();
                    undoProperties = { axis: properties.axis, attrs: gridModel.getUndoAttributeSet(properties.attrs) };
                    break;

                case Operations.SET_DRAWING_ATTRIBUTES:
                    undoProperties = { attrs: chartModel.getUndoAttributeSet(properties.attrs) };
                    break;

                default:
                    Utils.error('generateChartOperation(): unknown operation "' + opName + '"');
                    return;
            }

            generator.generateDrawingOperation(opName, position, undoProperties, { undo: true });
            generator.generateDrawingOperation(opName, position, properties);
        }

        /**
         * Returns the title text from the passed attribute set.
         *
         * @param {Object} [attributes]
         *  The merged attribute set for a chart title. If omitted, returns an
         *  empty string.
         *
         * @returns {String}
         *  The title text from the passed attribute set.
         */
        function getChartTitle(attributes) {
            return _.isObject(attributes) ? (attributes.text.link[0] || '') : '';
        }

        /**
         * Changes the attributes of the specified title in the selected chart
         * object.
         *
         * @param {ChartModel} chartModel
         *  The chart model to be manipulated.
         *
         * @param {String} titleId
         *  The identifier of the title object. Must be one of 'main', 'x',
         *  'y', or 'z'.
         *
         * @param {Object} attrs
         *  The attribute set to be set for the title.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operation has been
         *  applied successfully, or that will be rejected on any error.
         */
        function setChartTitleAttributes(chartModel, titleId, attrs) {
            return createAndApplyChartOperation(chartModel, Operations.SET_CHART_TITLE_ATTRIBUTES, { axis: titleId, attrs: attrs });
        }

        /**
         * Changes the text of the specified title in the selected chart
         * object.
         *
         * @param {ChartModel} chartModel
         *  The chart model to be manipulated.
         *
         * @param {String} titleId
         *  The identifier of the title object. Must be one of 'main', 'x',
         *  'y', or 'z'.
         *
         * @param {String} title
         *  The new text contents of the title. An empty string will remove the
         *  title from the chart.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operation has been
         *  applied successfully, or that will be rejected on any error.
         */
        function setChartTitle(chartModel, titleId, title) {
            return setChartTitleAttributes(chartModel, titleId, { text: { link: [title] }, character: ChartCreator.getHeadChar() });
        }

        function selectNewChartSource() {

            var // enter custom selection mode
                promise = docView.enterCustomSelectionMode('chartsource', {
                    statusLabel: /*#. change source data for a chart object in a spreadsheet */ gt('Select source data')
                });

            // set selected range as source data
            promise.done(function (selection) {

                var range = selection.ranges[0];
                var sheet = docView.getActiveSheet();

                if (!chartSourceToken) {
                    chartSourceToken = new TokenArray(docView.getSheetModel(), 'chart');
                }

                chartSourceToken.clearTokens().appendRange(range, { sheet: sheet, abs: true });

                var chart = getDrawingModel('chart');

                var possSources = chart.getDataSourceInfo();
                if (possSources.warn) {
                    return ChartCreator.updateSeries(app, range, sheet, chart);
                }

                return ChartCreator.updateSeries(app, range, sheet, chart, possSources.axis);
            });
        }

        /**
         * changeChartType maps the assigned id to chart-data
         * all ids are mapped in DrawingLabels.CHART_TYPE_STYLES
         *
         * There is a special behavior for bubble-chart, change to bubble or from bubble.
         * all series-data will be removed an initialized complete new by the ChartCreator
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operation has been
         *  applied successfully, or that will be rejected on any error.
         */
        function changeChartType(id) {

            var chartModel = getDrawingModel('chart');
            var data = DrawingLabels.CHART_TYPE_STYLES[id];
            var oldData = chartModel.getMergedAttributeSet(true).chart;

            var oldNoAx = oldData.type.indexOf('pie') === 0 || oldData.type.indexOf('donut') === 0;
            var noAx = data.type.indexOf('pie') === 0 || data.type.indexOf('donut') === 0;

            var oldBubble = oldData.type.indexOf('bubble') === 0;
            var bubble = data.type.indexOf('bubble') === 0;

            if (oldNoAx !== noAx || oldBubble !== bubble) {
                // special behavior for bubble & pie charts!!!
                // fix for Bug 46917
                var possSources = chartModel.getDataSourceInfo();
                if (possSources.warn) {
                    return yellChartDirection(possSources.warn);
                }
                return ChartCreator.updateSeries(app, possSources.range, possSources.sheet, chartModel, possSources.axis, data);
            }

            return createAndApplyChartOperation(chartModel, Operations.SET_DRAWING_ATTRIBUTES, { attrs: { chart: data } }, function (generator, position) {
                if ((data.type === 'pie2d' || data.type === 'donut2d')) {
                    var noShape = { line: ChartCreator.getNoneShape(), axis: { label: false } };
                    generateChartOperation(generator, chartModel, Operations.SET_CHART_AXIS_ATTRIBUTES,     position, { attrs: noShape, axis: 'x' });
                    generateChartOperation(generator, chartModel, Operations.SET_CHART_AXIS_ATTRIBUTES,     position, { attrs: noShape, axis: 'y' });
                    generateChartOperation(generator, chartModel, Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { attrs: noShape, axis: 'x' });
                    generateChartOperation(generator, chartModel, Operations.SET_CHART_GRIDLINE_ATTRIBUTES, position, { attrs: noShape, axis: 'y' });
                    generateChartOperation(generator, chartModel, Operations.SET_CHART_TITLE_ATTRIBUTES,    position, { attrs: { text: { link: [] }, character: null }, axis: 'x' });
                    generateChartOperation(generator, chartModel, Operations.SET_CHART_TITLE_ATTRIBUTES,    position, { attrs: { text: { link: [] }, character: null }, axis: 'y' });
                }
            });
        }

        function isFirstRowColEnabled(chartModel, rowCol) {
            if (!chartModel) { return null; }

            var possSources = chartModel.getDataSourceInfo();
            if (possSources.warn) { return null; }
            if (possSources.axis) { rowCol = Math.abs(rowCol - 1); }
            if (rowCol === 1) {
                return chartModel.isTitleLabel() || chartModel.getFirstPointsCount() > 1;
            } else {
                return chartModel.isNamesLabel() || chartModel.getSeriesCount() > 1;
            }
        }

        function isFirstRowCol(chartModel, rowCol) {
            if (!chartModel) { return null; }

            var possSources = chartModel.getDataSourceInfo();
            if (possSources.warn) { return null; }
            if (possSources.axis) { rowCol = Math.abs(rowCol - 1); }
            if (rowCol === 1) {
                return chartModel.isTitleLabel();
            } else {
                return chartModel.isNamesLabel();
            }
        }

        function setFirstRowCol(chartModel, rowCol, value) {
            if (!chartModel) { return null; }
            var possSources = chartModel.getDataSourceInfo();
            if (possSources.warn) { return null; }

            if (possSources.axis) { rowCol = Math.abs(rowCol - 1); }

            if (rowCol === 1) {
                return ChartCreator.updateSeries(app, possSources.range, possSources.sheet, chartModel, possSources.axis, undefined, value, chartModel.isNamesLabel());
            } else {
                return ChartCreator.updateSeries(app, possSources.range, possSources.sheet, chartModel, possSources.axis, undefined, chartModel.isTitleLabel(), value);
            }

        }

        function exchange(chartModel) {
            var possSources = chartModel.getDataSourceInfo();
            if (possSources.warn) {
                return yellChartDirection(possSources.warn);
            }
            return ChartCreator.updateSeries(app, possSources.range, possSources.sheet, chartModel, 1 - possSources.axis, undefined, chartModel.isNamesLabel(), chartModel.isTitleLabel());
        }

        function addAxisDef(axisId, defs) {

            var // the base path of axis items
                keyPath = 'drawing/chart/axis/' + axisId + '/';

            function getCorrectAxisId(chartModel) {
                return chartModel ? chartModel.getAxisIdForDrawing(axisId) : axisId;
            }

            function getStandardLineAttributes(visible) {
                return { line: visible ? ChartCreator.getStandardShape() : ChartCreator.getNoneShape() };
            }

            function setAxisAttributes(chartModel, attrs) {
                return createAndApplyChartOperation(chartModel, Operations.SET_CHART_AXIS_ATTRIBUTES, { axis: getCorrectAxisId(chartModel), attrs: attrs });
            }

            function setGridLineAttributes(chartModel, attrs) {
                return createAndApplyChartOperation(chartModel, Operations.SET_CHART_GRIDLINE_ATTRIBUTES, { axis: getCorrectAxisId(chartModel), attrs: attrs });
            }

            // *** axis items ***

            // parent item providing access to a chart axis model
            defs[keyPath + 'model'] = {
                parent: ['drawing/chart', 'drawing/chart/axes/enabled'],
                get: function (chartModel) { return chartModel ? chartModel.getAxisModel(getCorrectAxisId(chartModel)) : null; }
            };

            // parent item providing access to the attributes of a chart axis model
            defs[keyPath + 'attributes'] = {
                parent: [keyPath + 'model', 'drawing/chart'],
                get: function (axisModel) { return axisModel ? axisModel.getMergedAttributeSet(true) : null; },
                set: function (attributes) { return setAxisAttributes(this.getParentValue(1), attributes); }
            };

            // return or modify the visibility of the axis caption labels
            defs[keyPath + 'labels/visible'] = {
                parent: [keyPath + 'attributes', 'drawing/chart'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.axis.label === true) && this.areParentsEnabled(); },
                set: function (visible) { return setAxisAttributes(this.getParentValue(1), { axis: { label: visible } }); }
            };

            // return or modify the visibility of the axis line
            defs[keyPath + 'line/visible'] = {
                parent: [keyPath + 'attributes', 'drawing/chart'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.line.type !== 'none'); },
                set: function (visible) { return setAxisAttributes(this.getParentValue(1), getStandardLineAttributes(visible)); }
            };

            // *** grid line items ***

            // parent item providing access to a chart axis grid model
            defs[keyPath + 'grid/model'] = {
                parent: keyPath + 'model',
                get: function (axisModel) { return axisModel ? axisModel.getGrid() : null; }
            };

            // parent item providing access to the attributes of a chart axis grid model
            defs[keyPath + 'grid/attributes'] = {
                parent: [keyPath + 'grid/model', 'drawing/chart'],
                get: function (gridModel) { return gridModel ? gridModel.getMergedAttributeSet(true) : null; },
                set: function (attributes) { return setGridLineAttributes(this.getParentValue(1), attributes); }
            };

            // return or modify the visibility of the axis grid lines
            defs[keyPath + 'grid/visible'] = {
                parent: [keyPath + 'grid/attributes', 'drawing/chart'],
                get: function (attributes) { return _.isObject(attributes) && (attributes.line.type !== 'none'); },
                set: function (visible) { return setGridLineAttributes(this.getParentValue(1), getStandardLineAttributes(visible)); }
            };

            // *** title items ***

            // parent item providing access to a chart axis title model
            defs[keyPath + 'title/model'] = {
                parent: keyPath + 'model',
                get: function (axisModel) { return axisModel ? axisModel.getTitle() : null; }
            };

            // parent item providing access to the attributes of a chart axis title model
            defs[keyPath + 'title/attributes'] = {
                parent: [keyPath + 'title/model', 'drawing/chart'],
                get: function (titleModel) { return titleModel ? titleModel.getMergedAttributeSet(true) : null; },
                set: function (attributes) {
                    var chartModel = this.getParentValue(1);
                    return setChartTitleAttributes(chartModel, getCorrectAxisId(chartModel), attributes);
                }
            };

            // return or modify the text contents of a chart axis title
            defs[keyPath + 'title/text'] = {
                parent: [keyPath + 'title/attributes', 'drawing/chart'],
                get: function (attributes) { return getChartTitle(attributes); },
                set: function (title) {
                    var chartModel = this.getParentValue(1);
                    return setChartTitle(chartModel, getCorrectAxisId(chartModel), title);
                }
            };

            // remove the main title from a chart (disabled if title does not exist)
            defs[keyPath + 'title/delete'] = {
                parent: [keyPath + 'title/text', 'drawing/chart'],
                enable: function () { return this.getValue().length > 0; },
                set: function () {
                    var chartModel = this.getParentValue(1);
                    return setChartTitle(chartModel, getCorrectAxisId(chartModel), '');
                }
            };
        }

        /**
         * private registerDefinitions handles all controller definitions for Drawings and Charts.
         * Including the Insert-Drawing button which is physical part of the original Sidepane but is better placed here
         */
        function registerDefinitions() {
            var defs = {};

            defs['image/insert'] = {
                parent: 'sheet/operation/unlocked',
                set: function (imageDesc) { return docView.insertImages(imageDesc); }
            };

            // show image drive, file or url dialog
            defs['image/insert/dialog'] = {
                parent: 'sheet/operation/unlocked',
                set: function (dialogType) { return docView.showInsertImageDialog(dialogType); }
            };

            defs['chart/insert'] = {
                parent: 'sheet/operation/unlocked',
                enable: function () {
                    var ranges = docView.getSelectedRanges();
                    return (ranges.length === 1) && !docView.getSheetModel().isSingleCellInRange(ranges[0]);
                },
                set: function (id) { return ChartCreator.createChart(app, DrawingLabels.CHART_TYPE_STYLES[id]); }
            };

            defs['drawing/operation'] = {
                parent: 'sheet/operation/unlocked',
                enable: function () { return docView.hasDrawingSelection(); }
            };

            defs['drawing/model'] = {
                parent: 'drawing/operation',
                get: getDrawingModel
            };

            defs['drawing/type/label'] = {
                parent: 'drawing/model',
                get: function (drawingModel) { return DrawingLabels.getDrawingTypeLabel(drawingModel ? drawingModel.getType() : ''); }
            };

            defs['drawing/delete'] = {
                parent: 'drawing/operation',
                set: function () { return docView.deleteDrawings(); }
            };

            defs['drawing/image'] = {
                parent: 'drawing/operation',
                enable: function () { return _.isObject(this.getValue()); },
                get: function () { return getDrawingModel('image'); }
            };

            defs['drawing/chart'] = {
                parent: 'drawing/operation',
                enable: function () { return _.isObject(this.getValue()); },
                get: function () { return getDrawingModel('chart'); }
            };

            defs['drawing/chartlabels'] = {
                parent: 'drawing/chart',
                get: function () { return docView.getChartLabelsMenu().isVisible(); },
                set: function (state) { docView.getChartLabelsMenu().toggle(state); }
            };

            defs['drawing/chartexchange'] = {
                parent: 'drawing/chart',
                enable: function (chartModel) { return chartModel.getSeriesCount() > 0; },
                set: function () {
                    var chart = getDrawingModel('chart');
                    return exchange(chart);
                }
            };

            defs['drawing/chartfirstcol'] = {
                parent: 'drawing/chart',
                enable: function (chartModel) { return isFirstRowColEnabled(chartModel, 0); },
                get: function (chartModel) { return isFirstRowCol(chartModel, 0); },
                set: function (state) { return setFirstRowCol(getDrawingModel('chart'), 0, state); }
            };

            defs['drawing/chartfirstrow'] = {
                parent: 'drawing/chart',
                enable: function (chartModel) { return isFirstRowColEnabled(chartModel, 1); },
                get: function (chartModel) { return isFirstRowCol(chartModel, 1); },
                set: function (state) { return setFirstRowCol(getDrawingModel('chart'), 1, state); }
            };

            defs['drawing/charttype'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getChartTypeForGui() : null; },
                set: changeChartType
            };

            // parent item providing access to the attributes of a chart model
            defs['drawing/chart/attributes'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getMergedAttributeSet(true) : null; },
                set: function (attributes) { this.getParentValue().setAttributes(attributes); }
            };

            defs['drawing/chartvarycolor'] = {
                parent: ['drawing/chart', 'drawing/chart/attributes'],
                enable: function (chartModel) { return chartModel.isVaryColorEnabled(); },
                get: function (chartModel, attributes) { return attributes ? attributes.chart.varyColors : null; },
                set: function (state) { this.getParentValue().changeVaryColors(state); }
            };

            defs['drawing/chartdatalabel'] = {
                parent: ['drawing/chart', 'drawing/chart/attributes'],
                get: function (chartModel, attributes) { return attributes ? attributes.chart.dataLabel : false; },
                set: function (state) { return createAndApplyChartOperation(this.getParentValue(), Operations.SET_DRAWING_ATTRIBUTES, { attrs: { chart: { dataLabel: state } } }); }
            };

            defs['drawing/chartmarkeronly'] = {
                parent: ['drawing/chart'],
                enable: function (chartModel) { var data = chartModel ? chartModel.getMergedAttributeSet(true).chart : {}; return data.type === 'line2d' || data.type === 'scatter2d'; },
                get: function (chartModel) { return chartModel ? chartModel.isMarkerOnly() : false; },
                set: function (state) { this.getParentValue().setMarkerOnly(state); }
            };

            defs['drawing/chartcolorset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getColorSet() : null; },
                set: function (colorset) { this.getParentValue().changeColorSet(colorset); }
            };

            defs['drawing/chartstyleset'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getStyleSet() : null; },
                set: function (colorset) { this.getParentValue().changeStyleSet(colorset); }
            };

            defs['drawing/chartsource'] = {
                parent: 'drawing/chart',
                get: function () { return sourceSelector; },
                set: selectNewChartSource
            };

            defs['drawing/chartlegend/pos'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getLegendModel().getMergedAttributeSet(true).legend.pos : null; },
                set: function (pos) { return createAndApplyChartOperation(this.getParentValue(), Operations.SET_CHART_LEGEND_ATTRIBUTES, { attrs: { legend: { pos: pos } } }); }
            };

            defs['drawing/chart/axes/enabled'] = {
                parent: 'drawing/chart',
                enable: function (chartModel) { return chartModel.isAxesEnabled(); }
            };

            addAxisDef('x', defs);
            addAxisDef('y', defs);

            // parent item providing access to the main title model of a chart
            defs['drawing/chart/title/model'] = {
                parent: 'drawing/chart',
                get: function (chartModel) { return chartModel ? chartModel.getTitleModel('main') : null; }
            };

            // parent item providing access to the attributes of the main title model of a chart
            defs['drawing/chart/title/attributes'] = {
                parent: ['drawing/chart/title/model', 'drawing/chart'],
                get: function (titleModel) { return titleModel ? titleModel.getMergedAttributeSet(true) : null; },
                set: function (attributes) { return setChartTitleAttributes(this.getParentValue(1), 'main', attributes); }
            };

            // return or modify the text contents of the main title of a chart
            defs['drawing/chart/title/text'] = {
                parent: ['drawing/chart/title/attributes', 'drawing/chart'],
                get: function (attributes) { return getChartTitle(attributes); },
                set: function (title) { return setChartTitle(this.getParentValue(1), 'main', title); }
            };

            // remove the main title from a chart (disabled if title does not exist)
            defs['drawing/chart/title/delete'] = {
                parent: ['drawing/chart/title/text', 'drawing/chart'],
                enable: function () { return this.getValue().length > 0; },
                set: function () { return setChartTitle(this.getParentValue(1), 'main', ''); }
            };

            // IMAGE

            defs['drawing/border/style'] = {
                parent: 'drawing/image',
                get: function (imageModel) { return DrawingUtils.getPresetBorder(imageModel ? imageModel.getMergedAttributeSet(true).line : null); },
                set: function (preset) {
                    var attributes = this.getParentValue().getMergedAttributeSet(true),
                        options = { line: DrawingUtils.resolvePresetBorder(preset) };

                    if (attributes.line.color.type === 'auto') {
                        options.line.color = { type: 'rgb', value: '000000' };
                    }

                    return docView.changeDrawings(options);
                }
            };

            defs['drawing/border/color'] = {
                parent: 'drawing/image',
                get: function (imageModel) { return imageModel ? imageModel.getMergedAttributeSet(true).line.color : Color.AUTO; },
                set: function (color) {
                    var attributes = this.getParentValue().getMergedAttributeSet(true),
                        options = { line: { color: color } };

                    if (attributes.line.type !== 'solid') {
                        options.line.type = 'solid';
                    }

                    return docView.changeDrawings(options);
                }
            };

            self.registerDefinitions(defs);
        }

        // initialization -----------------------------------------------------

        registerDefinitions();

        this.listenTo(docView, 'change:selection', changeSelectionHandler);

        this.registerDestructor(function () {
            self = app = docModel = docView = chartSourceToken = tokenArrays = null;
        });

    } // class DrawingControllerMixin

    // exports ================================================================

    return DrawingControllerMixin;

});
