/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH.
 *
 * @author Peter Seliger <peter.seliger@open-xchange.com>
 */

define('io.ox/office/presentation/view/slidepreviewmixin', [

    'io.ox/office/tk/utils',
    'io.ox/office/textframework/utils/dom',

    'less!io.ox/office/presentation/view/style'

], function (Utils, DOM) {

    'use strict';

    // mixin class SlidePreviewMixin ==========================================

    /**
     * Additional preview specific behavior for the presentation view.
     *
     * Does provide a set of specific functionality that is just for rendering
     * the preview of any presentation slide that most recently has been changed.
     *
     * @mixin (applicator/functor)
     *
     * @used-by PresentationView class
     *
     * @param {TextApplication} app
     *  The application containing this view instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options that are supported by the
     *  constructor of the base class BaseView.
     */
    function SlidePreviewMixin(app, initOptions) {

        var
            self = this,                    // self reference :: PresentationView

            docModel,                       // {TextModel} docModel - The document model created by the passed application.

            slidepane,                      // the `slide pane` reference.

            $pageNode,

            $thumbRenderBox,

            imgNodeBlueprint,

            $masterPreviewBlueprint,
            $standardPreviewBlueprint,

            isMasterView   = false,
            isStandardView = false,

            thumb_scale_value,
            css_value__thumb_scale,

            THUMBNAIL_HEIGHT  = 50,

            SELECTOR__MASTER_SLIDE          = '.masterslidelayer',
            SELECTOR__LAYOUT_SLIDE          = '.layoutslidelayer',
            SELECTOR__PAGE_CONTENT          = '.pagecontent',
            SELECTOR__SLIDE                 = '.slide',

            SELECTOR__SLIDE_CONTAINER       = '.slidecontainer',
            SELECTOR__SLIDEPANE_CONTAINER   = '.slide-pane-container',

            SELECTOR__OVERLAY               = '.collaborative-overlay',

            CLASS_NAME__INVISIBLE_DRAWINGS  = 'invisibledrawings',
            CLASS_NAME__INVISIBLE_SLIDE     = 'invisibleslide',
            CLASS_NAME__NOT_SELECTABLE      = 'notselectable',
            CLASS_NAME__SELECTED            = 'selected',
            CLASS_NAME__APP_TOOLTIP         = 'app-tooltip',

            CLASS_NAME__EDIT_MODE           = 'edit-mode',

          //CLASS_NAME__THUMB_MODE          = 'thumb-mode',
            CLASS_NAME__PREVIEW_MODE        = 'preview-mode',

            ATTRIBUTE__CONTENT_EDITABLE     = 'contenteditable',

            default_target_of_mpo_s = { $currentMaster: null, currentMasterId: '' }, // Default Target Of Master Preview Operations.

            previewStore        = {}, // stores all available previews each as always up to date DOM fragment.
            attributionMap      = {}, // keeps track of any preview's attribution state.

            thumbRegistry       = {}, // keeps track of render state of all available thumbs and, if necessary, stores them too.

            fieldUpdateCache    = [], // does cache all data objects that have been received by listening to
                                      // '...Fields:update' events before all internal states were initialized.

            formattedSlideCache = [], // does cache all slide id's that have been notified to be finally formatted
                                      // via the model before all internal states were initialized.

            REGX_ATTRIBUTE_NAME_WHITELIST = (/^(?:link|href|src|alt)$/);

        // modify initial `initHandler` with mixin specific `init` functionality via basic function composition.
        //
        initOptions.initHandler = (function (proceed) {
            return function () {
              //proceed.call(self, arguments); // not necessary.

                proceed();     // the `PresentationView`s `initHandler` accessible via `initOptions`
                initHandler(); // the `SlidePreviewMixin`s specific `initHandler` functionality.
            };
        }(initOptions.initHandler));
        //
        // if there was the opportunity introducing `Function.prototype[.before[.after[.around]]]`, the code
        // above could be rewritten to `initOptions.initHandler = initOptions.initHandler.after(initHandler);`

        // local / private methods --------------------------------------------
        //

        /*
        function deletePreview(slideId) {

            deleteAttributionItem(slideId);
            deleteThumbItem(slideId);

            return (delete previewStore[slideId]);
        }*/
        function putPreview(slideId, $preview) {
            previewStore[slideId] = $preview;
        }
        function getPreview(slideId) {
          // deprecated:
          //return previewStore[slideId] ? previewStore[slideId].clone() : $('<div>');
            var
                $preview        = previewStore[slideId],
                attributionItem = attributionMap[slideId];

            return (attributionItem && !attributionItem.isMustFormat && $preview) ? $preview.clone() : $('<div>');

        //  if (attributionItem) {
        //      if (attributionItem.isMustFormat) {
        //
        //      // NOTE :
        //          - not that way ...
        //          - ... 'request:formatting' by now has to be triggered explicitly via `triggerPreviewFormatting`.
        //
        //        //docModel.trigger('request:formatting', slideIdList);
        //      } else {
        //          $preview = previewStore[slideId].clone();
        //      }
        //  }
        //  return $preview;
        }/*
        function getPreviewList() {
            return Object.keys(previewStore).reduce(function (collector, slideId) {

                collector.push(getPreview(slideId));
                return collector;

            }, []);
        }*/

        function unflagAttributionItemAsMustFormat(slideId) {
            attributionMap[slideId].isMustFormat = false;
        }
        function flagAttributionItemAsMustFormat(slideId) {
            attributionMap[slideId].isMustFormat = true;
        }/*

        function deleteAttributionItem(slideId) {
            return (delete attributionMap[slideId]);
        }*/
        function createAttributionItem(slideId) {
            attributionMap[slideId] = {};
            flagAttributionItemAsMustFormat(slideId);
        }

        function unflagThumbItemAsMustRender(slideId) {
            thumbRegistry[slideId].isMustRender = false;
        }
        function flagThumbItemAsMustRender(slideId) {
            thumbRegistry[slideId].isMustRender = true;
        }/*

        function deleteThumbItem(slideId) {
            return (delete thumbRegistry[slideId]);
        }*/
        function createThumbItem(slideId/*, idx, list*/) {
            thumbRegistry[slideId] = {
                thumb: null
            };
            flagThumbItemAsMustRender(slideId);
        }

        var getThumbDeferred = (function () {
            var
                firstLevelRequestQueue  = [],   // queue of pending thumb request/render promises of high priority (thumbs already have been rendered).
                secondLevelRequestQueue = [],   // queue of pending thumb request/render promises of low priority (thumbs still need to be rendered).
                requestTimer            = null; // background loop processing all pending thumb request/render promises.

            // direct callback: called every time when `getThumbDeferred` has been called.
            function registerRequestPromise(deferredRequest, slideId) {
                var
                    renderStateItem   = thumbRegistry[slideId],
                    requestDescriptor = {
                        deferred: deferredRequest,
                        slideId:  slideId
                    };
                if (renderStateItem.isMustRender) {
                    secondLevelRequestQueue.push(requestDescriptor);
                } else {
                    firstLevelRequestQueue.push(requestDescriptor);
                }
            }

            // deferred callback: called once after the specified timeout.
            function getThumbDeferredFromPrioritizedQueues() {

                // check if the background loop is already running
                if (requestTimer) { return; }

                // create a new background loop that processes all pending tables
                requestTimer = self.repeatSliced(function () {

                    var requestDescriptor = firstLevelRequestQueue.shift() || secondLevelRequestQueue.shift();
                    if (!requestDescriptor) {

                        return Utils.BREAK;
                    }
                    var
                        deferredRequest = requestDescriptor.deferred,
                        slideId         = requestDescriptor.slideId,

                        renderStateItem = thumbRegistry[slideId];

                    if (renderStateItem.isMustRender) {
                        renderStateItem.thumb = createLayoutThumb(slideId);
                    }
                    deferredRequest.resolve({ thumb: renderStateItem.thumb });

                }, { infoString: 'Text: SlidePreviewMixin: getThumbDeferred' });

                // forget timer reference as soon as every request/render promise is out of 'pending' state.
                requestTimer.always(function () { requestTimer = null; });
            }

            // create and return the debounced and queued `getThumbDeferred` method with a delay of 10ms per thumb request/render promise.
            return self.createDebouncedMethod(registerRequestPromise, getThumbDeferredFromPrioritizedQueues, { delay: 20/* delay: 100, maxDelay: 400*/ });
        }());

        // function getThumbDeferred(deferredRequest, slideId) {
        //     var renderStateItem = thumbRegistry[slideId];
        //
        //     if (renderStateItem.isMustRender) {
        //         renderStateItem.thumb = createLayoutThumb(slideId);
        //     }
        //     deferredRequest.resolve({ thumb: renderStateItem.thumb });
        // }

        function requestThumb(slideId) {
            var thumbRequest = $.Deferred();

            if (slideId in thumbRegistry) {

              //window.setTimeout((function (thumbRequest, slideId) {
              //    return function () {
              //
              //        getThumbDeferred(thumbRequest, slideId);
              //    };
              //}(thumbRequest, slideId)), 0);

                getThumbDeferred(thumbRequest, slideId);
            } else {
                thumbRequest.reject(new ReferenceError('Slide ID does not match.'));
            }
            // return the `deferred` 's `promise` object
            return thumbRequest.promise();
        }

        function createLayoutThumb(slideId) {
          //Utils.info('+++ createLayoutThumb :: slideId', slideId);
            var
                $layoutThumb,
                $previewThumb = getPreview(slideId)/*.addClass(CLASS_NAME__THUMB_MODE)*/;

            applyThumbScaling($previewThumb);

            $previewThumb.prependTo($thumbRenderBox);

            $layoutThumb = preserveComputedElementStyles($previewThumb, $previewThumb.clone());
          //$layoutThumb.css('position', 'static');
          //$layoutThumb.css('z-index', 'auto');
            $layoutThumb.addClass('thumb');

            // check back with local function `renderLayoutThumb` as of 'io.ox/office/presentation/view/control/layoutslidepicker'
            $layoutThumb.data({ width: Math.round($layoutThumb.width() * thumb_scale_value)/*, height: Math.round($layoutThumb.height() * thumb_scale_value)*/ });

            $previewThumb.remove();
            unflagThumbItemAsMustRender(slideId);

            return $layoutThumb;
        }

        function computeInitialThumbScaling() {
            thumb_scale_value       = (Math.round((THUMBNAIL_HEIGHT / Utils.convertHmmToLength(docModel.getSlideDocumentSize().height, 'px', 1)) * 100000) / 100000);
            css_value__thumb_scale  = 'scale(' + thumb_scale_value + ')';
        }

        function applyThumbScaling($fragment) {
            $fragment.css('transform-origin', '0 0'); // class name bound LESS/CSS rule.
            $fragment.css('transform', css_value__thumb_scale);
        }

        /**
         * Helper that creates a base64-encoded image from a provided canvas element whilst preserving size measurements.
         */
        function createImageFromCanvas(canvasNode) {
            var
                imgNode = imgNodeBlueprint.cloneNode();

            imgNode.src           = canvasNode.toDataURL();

            imgNode.width         = canvasNode.width;         // canvasNode.offsetWidth
            imgNode.height        = canvasNode.height;        // canvasNode.offsetHeight

            imgNode.className     = canvasNode.className;
            imgNode.style.cssText = canvasNode.style.cssText;

            return imgNode;
        }

    //  /**
    //   *  helper
    //   */
    //  function createImageFromElementDeferred(node) {
    //      //var
    //      //    canvas = createCanvasFromElement(node),
    //      //    img    = createImageFromCanvas(canvas);
    //      //
    //      //return img;
    //
    //      return createCanvasFromElementDeferred(node).then(function (canvas) {
    //
    //          return createImageFromCanvas(canvas);
    //      });
    //  }

        /**
         * Sanitizing helper that will preserve any canvas content (in this case,
         * any shapes appearance) by stepwise taking a base64-encoded image-snapshot
         * of each canvas source element and replacing the corresponding canvas target
         * element with this very snapshot.
         */
        function preserveShapes($sourceNode, $targetNode) {
            var
                sourceList = $sourceNode.find('canvas').toArray(),
                targetList = $targetNode.find('canvas').toArray();

            sourceList.forEach(function (sourceNode, idx/*, list*/) {
                var
                    targetNode = targetList[idx],
                    imgNode    = createImageFromCanvas(sourceNode);

                $(targetNode).replaceWith(imgNode);
            });
        }

        /**
         *  helper
         */
        function preserveComputedElementStyles($sourceNode, $targetNode, styleNameFilter) {
            var
                sourceNode,
                targetNode     = $targetNode[0],

                sourceNodeList = $sourceNode.children().toArray(),
                targetNodeList = $targetNode.children().toArray();

            // @TODO - countercheck for x-browser compatibility
            _.toArray(targetNode.attributes).forEach(function (attrNode) {

                if (!REGX_ATTRIBUTE_NAME_WHITELIST.test(attrNode.nodeName)) {

                    targetNode.removeAttributeNode(attrNode);

                  //targetNode.removeAttribute(attrNode.nodeName);
                  //$targetNode.removeAttr(attrNode.nodeName);
                }
            });
            if (targetNode.nodeName.toLowerCase() === 'br') {
                targetNode = $('<span/>')[0];
            }
            targetNode.style.cssText = DOM.getComputedElementCssText($sourceNode[0], styleNameFilter);

            while ((sourceNode = sourceNodeList.pop()) && (targetNode = targetNodeList.pop())) {
                preserveComputedElementStyles($(sourceNode), $(targetNode), styleNameFilter);
            }
            $targetNode.find('br').remove();

            return $targetNode;
        }

        /**
         * Sanitizing helper that assures any preview fragment
         * of being in a visible state.
         */
        function preventInvisibleslide(node/*, idx, list*/) {
            var $node = $(node);

            $node.removeClass(CLASS_NAME__INVISIBLE_SLIDE);
            $node.children().toArray().forEach(preventInvisibleslide);
        }

        /**
         * Sanitizing helper that prevents any preview fragment
         * from being in a selectable state.
         */
        function preventNotselectable(node/*, idx, list*/) {
            var $node = $(node);

            $node.removeClass(CLASS_NAME__NOT_SELECTABLE);
            $node.children().toArray().forEach(preventNotselectable);
        }

        /**
         * Sanitizing helper that prevents any `slide` classified preview
         * fragment from being in a selectable state.
         */
        function assureNotselectableSlide(node/*, idx, list*/) {
            var $node = $(node);

            if ($node.hasClass('slide')) {
                $node.addClass(CLASS_NAME__NOT_SELECTABLE);
            }
            $node.children().toArray().forEach(assureNotselectableSlide);
        }

        /**
         * Sanitizing helper that prevents any preview fragment
         * from being in a selected state.
         */
        function preventSelectedElement(node/*, idx, list*/) {
            var $node = $(node);

            $node.removeClass(CLASS_NAME__SELECTED);
            $node.children().toArray().forEach(preventSelectedElement);
        }

        /**
         * Sanitizing helper that prevents any preview fragment
         * from being in an editable state.
         */
        function preventContendEditable(node/*, idx, list*/) {
            var $node = $(node);

            if (_.isString($node.attr(ATTRIBUTE__CONTENT_EDITABLE))) {
                $node.removeAttr(ATTRIBUTE__CONTENT_EDITABLE);
            }
            $node.children().toArray().forEach(preventContendEditable);
        }

        /**
         * Sanitizing helper that prevents any preview fragment
         * from triggering the tooltip display on hover.
         */
        function preventApplicationTooltip(node/*, idx, list*/) {
            var $node = $(node);

            $node.removeClass(CLASS_NAME__APP_TOOLTIP);
            $node.children().toArray().forEach(preventApplicationTooltip);
        }

        function getSlideIdFromFieldData(data) {
            return data.slideID;
        }
        function updateField(data/*, idx, list*/) {
            var
                $preview = previewStore[data.slideID],
                $field   = $preview.find('[data-fid="' + data.fieldID + '"] > span');

            if ($field && $field[0]) {

              //$field.eq(0).text(data.value);
                $field[0].innerHTML = data.value;
            }
        }
        function handleFieldUpdate(evt, fieldDataList) {

            fieldDataList.forEach(updateField);

            var
                filterByCurrentView = isStandardView ? docModel.isStandardSlideId : docModel.isLayoutOrMasterId,
                slideIdList         = fieldDataList.map(getSlideIdFromFieldData).filter(filterByCurrentView);

            self.trigger('previewupdate:after', slideIdList);
        }
        var handleFieldUpdateBeforeCompiledView = function (evt, fieldDataList) {
            fieldUpdateCache = fieldUpdateCache.concat(fieldDataList);
        };

        var handleSlideStateFormattedBeforeCompiledView = function (evt, slideIdList) {
            formattedSlideCache = fieldUpdateCache.concat(slideIdList);
        };

        /**
         * Generic sanitizer for both blueprint compiler methods
         * `compileMasterPreviewBlueprint` and `compileMasterPreviewBlueprint`.
         * This method cleans up any given jQuery-fied DOM fragment
         * in order to make it the render base of any preview.
         *
         * @param {jQuery.ElementNode} $node
         *  Any `pageNode` classified jQuery-fied element node.
         *
         * @returns {jQuery.ElementNode}
         *  Either a jQuery-fied blueprint for standard or for master/layout previews.
         */
        function getSanitizedPreviewBlueprint($node) {

            $node.removeClass(CLASS_NAME__EDIT_MODE);

            $node.find(SELECTOR__MASTER_SLIDE).empty();
            $node.find(SELECTOR__LAYOUT_SLIDE).empty();
            $node.find(SELECTOR__PAGE_CONTENT).empty();

            $node.addClass(CLASS_NAME__PREVIEW_MODE);

            preventContendEditable($node);

            $node.css('margin', 0); // TODO to be refactored / to be applied within every sanitize process.

            return $node;
        }

        /**
         * Function that initially clones all master/layout layers
         * as render source for a continuous updating process of
         * each of a slide's related/connected master/layout preview.
         *
         * This blueprint does become the render base for any process
         * that does update, scale, render, clone and replace the
         * master/layout preview of the most recently changes/updated
         * master/layout slide(s).
         */
        function compileMasterPreviewBlueprint() {
            var time = (new Date());

            $masterPreviewBlueprint = getSanitizedPreviewBlueprint($pageNode.clone());

          //$masterPreviewBlueprint.find(SELECTOR__OVERLAY).remove();
            $masterPreviewBlueprint.find(SELECTOR__OVERLAY).empty();

          //Utils.info('+++ compileMasterPreviewBlueprint +++');
          //Utils.info('+++ compileMasterPreviewBlueprint :: $masterPreviewBlueprint ', $masterPreviewBlueprint);

            Utils.info('+++ compileMasterPreviewBlueprint :: time : ', (new Date() - time));
        }

        /**
         * Function that initially clones all presentation layers
         * as render source for a continuous updating process of
         * each of a slide's related/connected standard preview.
         *
         * This blueprint does become the render base for any process
         * that does update, scale, render, clone and replace the
         * standard preview of the most recently changes/updated
         * standard slide(s).
         */
        function compileStandardPreviewBlueprint() {
            var time = (new Date());

            $standardPreviewBlueprint = getSanitizedPreviewBlueprint($pageNode.clone());

            $standardPreviewBlueprint.find(SELECTOR__OVERLAY).remove();
          //$standardPreviewBlueprint.find(SELECTOR__OVERLAY).empty();

          //Utils.info('+++ compileStandardPreviewBlueprint +++');
          //Utils.info('+++ compileStandardPreviewBlueprint :: $standardPreviewBlueprint ', $standardPreviewBlueprint);

            Utils.info('+++ compileStandardPreviewBlueprint :: time : ', (new Date() - time));
        }

        /**
         * This method does set or update the visibility states of depended preview layers.
         *
         * @param {jQuery.ElementNode} $preview
         *  The jQuery element node all DOM operations do act upon.
         *
         * Note:
         *      A similar method can be found in 'io.ox/office/presentation/view/view'.
         *      There it is called `handleLayoutMasterVisibility`. The following description
         *      has been copied partly from there ...
         *
         * Helper function to handle the visibility of the layout and the master slide. This is
         * required, if the slide attribute 'followMasterShapes' is modified. It is important, that
         * not the complete slide gets 'display: none' but only the drawing children. Otherwise
         * the background would also vanish. But that would not be the correct behavior.
         *
         * @param {String} [layoutId]
         *  The id of the layout slide.
         *
         * @param {String} [masterId]
         *  The id of the master slide.
         *
         * @param {Boolean} showLayoutSlide
         *  Whether the specified layout slide shall be made visible or not.
         *
         * @param {Boolean} showMasterSlide
         *  Whether the specified master slide shall be made visible or not.
         */
        function setLayoutMasterVisibility($preview, layoutId, masterId, showLayoutSlide, showMasterSlide) {
            var
                $layout = $preview
                    .find(SELECTOR__LAYOUT_SLIDE).eq(0)
                    .children('[data-container-id="' + layoutId + '"]').eq(0)
                    .find(SELECTOR__SLIDE).eq(0),

                $master = $preview
                    .find(SELECTOR__MASTER_SLIDE).eq(0)
                    .children('[data-container-id="' + masterId + '"]').eq(0)
                    .find(SELECTOR__SLIDE).eq(0);

            if (layoutId) { $layout.toggleClass(CLASS_NAME__INVISIBLE_DRAWINGS, !showLayoutSlide); }
            if (masterId) { $master.toggleClass(CLASS_NAME__INVISIBLE_DRAWINGS, !showMasterSlide); }
        }

        /**
         * Master/Layout DOM Fragment Factory
         *
         * This method is a helper factory that creates jQuery-fied DOM
         * fragments according to the provided master/layout slide model ID.
         *
         * @param {String} slideId
         *  Any master/layout slide model ID.
         *
         * @returns {jQuery.ElementNode}
         *  Either a jQuery-fied `$master` or a `$layout` layer DOM fragment.
         */
        function createLayoutFragment(slideId) {
            var
                $fragmentSource = (docModel.getSlideById(slideId) || $()).parent(SELECTOR__SLIDE_CONTAINER),
              //$fragmentSource = docModel.getSlideById(slideId).parent(SELECTOR__SLIDE_CONTAINER),
                $fragment       = $fragmentSource.clone();

            preserveShapes($fragmentSource, $fragment);

            preventContendEditable($fragment);
            preventSelectedElement($fragment);

            preventApplicationTooltip($fragment);

            preventNotselectable($fragment);
            preventInvisibleslide($fragment);

            return $fragment;
        }

        /**
         * Initial Renderer of any Standard Preview
         *
         * This method reduces a list of master/layout slide model ID's.
         * Therefore it makes use of a `collector` parameter.
         * For every master/layout slide model ID it accordingly renders
         * a master/layout preview into the DOM.
         *
         * @param {Object} collector
         *  The `collector` object with 2 additional key/value pairs attached to it.
         *  @param {jQuery.ElementNode} [collector.$currentMaster]
         *      In order to build a master/layout structure efficiently, an already
         *      created master object, that is related to many of its yet to be created
         *      layout objects, gets passed through until the next new master object
         *      is going to be built.
         *  @param {String} [collector.currentMasterId]
         *      In order to build a master/layout structure efficiently, for every
         *      already created master object, that is related to many of its yet
         *      to be created layout objects, it's (master) ID gets passed through
         *      until the next new master object is going to be built. master/layout
         *      visibility gets computed via this ID.
         *
         * @param {String} slideId
         *  Any master/layout slide model ID.
         */
        function assignMasterPreviewInitially(collector, slideId) {

            createAttributionItem(slideId);
            createThumbItem(slideId);

            var
                $preview = $masterPreviewBlueprint.clone(),

                $master,
                $layout,

                masterId, showMasterSlide,
                layoutId, showLayoutSlide;

            //Utils.info('+++ assignMasterPreviewInitially +++');
            //Utils.info('+++ assignMasterPreviewInitially - slideId : ', slideId);

            //Utils.info('+++ assignMasterPreviewInitially - ' + SELECTOR__MASTER_SLIDE + ' : ', $preview.find(SELECTOR__MASTER_SLIDE));
            //Utils.info('+++ assignMasterPreviewInitially - ' + SELECTOR__LAYOUT_SLIDE + ' : ', $preview.find(SELECTOR__LAYOUT_SLIDE));

            if (docModel.isMasterSlideId(slideId)) {

                $master = createLayoutFragment(slideId);

                collector.$currentMaster = $master.clone();

                showMasterSlide = true;
                masterId        = collector.currentMasterId = slideId;
            } else {
                layoutId        = slideId;
                masterId        = collector.currentMasterId || docModel.getMasterSlideId(layoutId);       // fail silently.

                showLayoutSlide = true;
                showMasterSlide = docModel.isFollowMasterShapeSlide(layoutId);

                $master = (collector.$currentMaster && collector.$currentMaster.clone()) || $('<div/>');  // fail silently.
                $layout = createLayoutFragment(slideId);

                assureNotselectableSlide($master);

                $layout.prependTo($preview.find(SELECTOR__LAYOUT_SLIDE));
            }
            $master.prependTo($preview.find(SELECTOR__MASTER_SLIDE));

            setLayoutMasterVisibility($preview, layoutId, masterId, showLayoutSlide, showMasterSlide);

            putPreview(slideId, $preview);

            return collector;
        }

        /**
         * Rerender Process of an already existing Master/Layout Preview
         *
         * This method iterates a list of master/layout slide model ID's.
         *
         * The whole iteration process gets provided an additional
         * `followMasterSlideIdIndex` context. Thus this method easily
         * can access additional information of how to recompute the
         * visibility states of all preview layers.
         *
         * This method takes any master/layout slide model ID. It then
         * accordingly creates either a new `$master` layer only or both
         * a pair of `$master` / `$layout` layers in dependency of either
         * the provided ID is a master or a layout ID only. The newly
         * created layer(s) does/do replace the former one(s). Right after,
         * all visibility states get recomputed.
         *
         * @param {String} slideId
         *  Any master/layout slide model ID.
         *
         * @this {Object} followMasterSlideIdIndex
         *  An object used as registry/map that as key contains every `slideId`
         *  that has master/layout dependencies and therefore will trigger the
         *  recalculation of all visibility states of this targeted layer set.
         */
        function reassignMasterPreview(slideId) {
            var
                followMasterSlideIdIndex = this,

                $preview      = getPreview(slideId),

                $master,
                $layout,

                masterId, showMasterSlide,
                layoutId, showLayoutSlide;

            if (!$preview) {                        // @TODO - refactor - get rid of this guard that is just a temporary fix that keeps code running whilst working on task DOCS-381.
                assignMasterPreviewInitially(default_target_of_mpo_s, slideId);
                return;
            }

            if (docModel.isMasterSlideId(slideId)) {

                $master         = createLayoutFragment(slideId);

                showMasterSlide = true;
                masterId        = slideId;
            } else {
                layoutId        = slideId;
                masterId        = docModel.getMasterSlideId(layoutId);

                showLayoutSlide = true;
                showMasterSlide = docModel.isFollowMasterShapeSlide(layoutId);

                $master         = createLayoutFragment(masterId);
                $layout         = createLayoutFragment(layoutId);

                assureNotselectableSlide($master);

                $layout.prependTo($preview.find(SELECTOR__LAYOUT_SLIDE).empty());
            }
            $master.prependTo($preview.find(SELECTOR__MASTER_SLIDE).empty());

            setLayoutMasterVisibility($preview, layoutId, masterId, showLayoutSlide, showMasterSlide);

            if (slideId in followMasterSlideIdIndex) {

                layoutId = docModel.getLayoutSlideId(slideId);
                masterId = docModel.getMasterSlideId(layoutId);

                showLayoutSlide = docModel.isFollowMasterShapeSlide(slideId);
                showMasterSlide = showLayoutSlide ? docModel.isFollowMasterShapeSlide(layoutId) : false;

                setLayoutMasterVisibility($preview, layoutId, masterId, showLayoutSlide, showMasterSlide);
            }
            putPreview(slideId, $preview);

            flagThumbItemAsMustRender(slideId);
        }

        /**
         * Standard Preview DOM Fragment Factory
         *
         * This method takes any standard slide model ID. It then accordingly creates
         * a complete set of jQuery-fied layers (`$master`, `$layout`, `$content`), in
         * order to return a jQuery-fied standard preview representative that already
         * features all current visibility states.
         *
         * @param {String} slideId
         *  Any standard slide model ID.
         *
         * @returns {jQuery.ElementNode}
         *  A complete set of jQuery-fied layers (`$master`, `$layout`, `$content`)
         *  that represent a standard preview DOM fragment, including all of its
         *  current visibility states.
         */
        function createStandardPreview(slideId) {
            var
                slideIdSet      = docModel.getSlideIdSet(slideId),

                masterId        = slideIdSet.masterId,
                layoutId        = slideIdSet.layoutId,

                $masterSource   = docModel.getSlideById(masterId).parent(SELECTOR__SLIDE_CONTAINER),
                $layoutSource   = docModel.getSlideById(layoutId).parent(SELECTOR__SLIDE_CONTAINER),
                $contentSource  = docModel.getSlideById(slideId),

                $master         = $masterSource.clone(),
                $layout         = $layoutSource.clone(),
                $content        = $contentSource.clone(),

                $preview        = $standardPreviewBlueprint.clone(),

                showLayoutSlide = docModel.isFollowMasterShapeSlide(slideId),
                showMasterSlide = showLayoutSlide ? docModel.isFollowMasterShapeSlide(layoutId) : false;

            preserveShapes($masterSource, $master);
            preserveShapes($layoutSource, $layout);
            preserveShapes($contentSource, $content);

            preventContendEditable($master);
            preventContendEditable($layout);
            preventContendEditable($content);

            preventSelectedElement($master);
            preventSelectedElement($layout);
            preventSelectedElement($content);

            preventApplicationTooltip($master);
            preventApplicationTooltip($layout);
            preventApplicationTooltip($content);

            assureNotselectableSlide($master);
            assureNotselectableSlide($layout);

            preventInvisibleslide($master);
            preventInvisibleslide($layout);
            $content.removeClass(CLASS_NAME__INVISIBLE_SLIDE);  // this line acknowledges remote clients.

            $master.prependTo($preview.find(SELECTOR__MASTER_SLIDE));
            $layout.prependTo($preview.find(SELECTOR__LAYOUT_SLIDE));
            $content.prependTo($preview.find(SELECTOR__PAGE_CONTENT));

            setLayoutMasterVisibility($preview, layoutId, masterId, showLayoutSlide, showMasterSlide);

            return $preview;
        }

        /**
         * Rerender Process of an already existing Standard Preview
         *
         * This method iterates a list of standard slide model ID's.
         *
         * The whole iteration process gets provided an additional
         * `followMasterSlideIdIndex` context. Thus this method easily
         * can access additional information of how to recompute the
         * visibility states of all preview layers.
         *
         * This method takes any standard slide model ID. It then accordingly
         * creates a minimum preview subset only. A new jQuery-fied `$content`
         * layer gets created. It does replace the former one. Right after,
         * all visibility states get recomputed.
         *
         * @param {String} slideId
         *  Any standard slide model ID.
         *
         * @this {Object} followMasterSlideIdIndex
         *  An object used as registry/map that as key contains every `slideId`
         *  that has master/layout dependencies and therefore will trigger the
         *  recalculation of all visibility states of this targeted layer set.
         */
        function reassignStandardPreview(slideId) {
            var
                followMasterSlideIdIndex = this,

                $preview        = getPreview(slideId),
                $container      = $preview ? $preview.find(SELECTOR__PAGE_CONTENT) : $(), // TODO

                $contentSource  = docModel.getSlideById(slideId),
                $content        = $contentSource ? $contentSource.clone() : $(),  // TODO

                layoutId, masterId, showLayoutSlide, showMasterSlide;

            preserveShapes($contentSource, $content);

            preventContendEditable($content);
            preventSelectedElement($content);

            preventApplicationTooltip($content);

            $content.removeClass(CLASS_NAME__INVISIBLE_SLIDE);  // this line acknowledges remote clients.

            $content.prependTo($container.empty());

            if (slideId in followMasterSlideIdIndex) {

                layoutId = docModel.getLayoutSlideId(slideId);
                masterId = docModel.getMasterSlideId(layoutId);

                showLayoutSlide = docModel.isFollowMasterShapeSlide(slideId);
                showMasterSlide = showLayoutSlide ? docModel.isFollowMasterShapeSlide(layoutId) : false;

                setLayoutMasterVisibility($preview, layoutId, masterId, showLayoutSlide, showMasterSlide);
            }
            putPreview(slideId, $preview);

            flagThumbItemAsMustRender(slideId); // right now there is no need for standard-slide thumb-previews.
        }

        /**
         * Initial Renderer of any Standard Preview
         *
         * This method iterates a list of standard slide model ID's.
         * For every standard slide model ID it accordingly renders
         * a standard preview into the DOM.
         *
         * @param {String} slideId
         *  Any standard slide model ID.
         */
        function assignStandardPreviewInitially(slideId/*, idx, list*/) {

            createAttributionItem(slideId);
            createThumbItem(slideId);       // right now there is no need for standard-slide thumb-previews.

            var
                $preview = createStandardPreview(slideId);

            putPreview(slideId, $preview);
        }

        function assignEveryMasterPreviewItemInitially() {
            var time = (new Date());

            docModel.getMasterSlideOrder().reduce(assignMasterPreviewInitially, default_target_of_mpo_s);

            Utils.info('+++ assignEveryMasterPreviewItemInitially :: time : ', (new Date() - time));
        }
        function assignEveryStandardPreviewItemInitially() {
            var time = (new Date());

            docModel.getStandardSlideOrder().forEach(assignStandardPreviewInitially);

            Utils.info('+++ assignEveryStandardPreviewItemInitially :: time : ', (new Date() - time));
        }/*

        function registerEveryMasterAttributionItem() {
            var time = (new Date());

            docModel.getMasterSlideOrder().forEach(createAttributionItem);

            Utils.info('+++ registerEveryMasterAttributionItem :: time : ', (new Date() - time));
        }
        function registerEveryStandardAttributionItem() {
            var time = (new Date());

            docModel.getStandardSlideOrder().forEach(createAttributionItem);

            Utils.info('+++ registerEveryStandardAttributionItem :: time : ', (new Date() - time));
        }*/

        function triggerPreviewFormatting(slideIdList) {
            Utils.info('+++ triggerPreviewFormatting :: slideIdList - before : ', slideIdList);

            slideIdList = slideIdList.filter(function (slideId/*, idx, list*/) {
                return attributionMap[slideId].isMustFormat;
            });
            Utils.info('+++ triggerPreviewFormatting :: slideIdList - after : ', slideIdList);

            docModel.trigger('request:formatting', slideIdList);
        }

        function compilePreviewStatesInitially() {
            var time = (new Date());

            attributionMap  = {};

            previewStore    = {};
            thumbRegistry   = {};

            assignEveryStandardPreviewItemInitially();
            assignEveryMasterPreviewItemInitially();

          //registerEveryStandardAttributionItem();
          //registerEveryMasterAttributionItem();

            Utils.info('+++ compilePreviewStatesInitially :: time : ', (new Date() - time));
            Utils.info('+++ compilePreviewStatesInitially :: attributionMap : ', attributionMap);
            Utils.info('+++ compilePreviewStatesInitially :: previewStore : ', previewStore);
            Utils.info('+++ compilePreviewStatesInitially :: thumbRegistry : ', thumbRegistry);
        }

        function compileAllDisplayStates() {

            isMasterView   = docModel.isMasterView();
            isStandardView = !isMasterView;

            Utils.info('+++ compileAllDisplayStates :: isMasterView, isStandardView : ', isMasterView, isStandardView);
        }

        function handlePreviewCreation(evt, slideIdList) {

            Utils.info('+++ ENTER handlePreviewCreation +++');
          //compileAllDisplayStates();

            var
                standardSlideIdList   = [],
                layoutOrMasterIdList  = [];

            slideIdList.forEach(function (slideId) {
                if (docModel.isStandardSlideId(slideId)) {

                    assignStandardPreviewInitially(slideId);

                    standardSlideIdList.push(slideId);
                } else {
                    assignMasterPreviewInitially(default_target_of_mpo_s, slideId);

                    layoutOrMasterIdList.push(slideId);
                }
                unflagAttributionItemAsMustFormat(slideId);
            });

            if (isStandardView) {
                self.trigger('previewupdate:after', standardSlideIdList);
            } else {
                self.trigger('previewupdate:after', layoutOrMasterIdList);
            }
        }

        /**
         * Handler for any of a document's 'slidestate:update' events.
         *
         * This handler is aware of a document's view modeindirectSlideIdList
         * and accordingly triggers the preview update process.
         */
        function handlePreviewUpdate(evt, slideIdList, followMasterSlideIdIndex, indirectSlideIdList) {

            Utils.info('+++ ENTER handlePreviewUpdate +++');
            //compileAllDisplayStates();

            // setting default values
            slideIdList = slideIdList || [];
            followMasterSlideIdIndex = followMasterSlideIdIndex || {};
            indirectSlideIdList = indirectSlideIdList || [];

            //Utils.info('+++ handlePreviewUpdate +++');
            //Utils.info('handlePreviewUpdate :: evt : ', evt);
            //Utils.info('handlePreviewUpdate :: slideIdList : ', slideIdList);
            //Utils.info('handlePreviewUpdate :: followMasterSlideIdIndex : ', followMasterSlideIdIndex);
            //Utils.info('handlePreviewUpdate :: indirectSlideIdList : ', indirectSlideIdList);

            if (isStandardView) {
                if (docModel.isStandardSlideId(slideIdList[0])) {
                  //Utils.info('handlePreviewUpdate :: isStandardView AND isStandardSlideId');

                    slideIdList.forEach(reassignStandardPreview, followMasterSlideIdIndex);
                } else {
                  //Utils.info('handlePreviewUpdate :: isStandardView AND isLayoutSlideId');

                    slideIdList.forEach(reassignMasterPreview, followMasterSlideIdIndex);

                    indirectSlideIdList.forEach(function (slideId) { assignStandardPreviewInitially(slideId); unflagAttributionItemAsMustFormat(slideId); });
                }
                slideIdList = slideIdList.concat(indirectSlideIdList);

                slideIdList = slideIdList.filter(docModel.isStandardSlideId);
            } else {
                if (docModel.isStandardSlideId(slideIdList[0])) {
                  //Utils.info('handlePreviewUpdate :: isMasterView AND isStandardSlideId');

                    slideIdList.forEach(reassignStandardPreview, followMasterSlideIdIndex);
                } else {
                  //Utils.info('handlePreviewUpdate :: isMasterView AND isLayoutSlideId');

                    slideIdList.forEach(reassignMasterPreview, followMasterSlideIdIndex);

                    indirectSlideIdList.forEach(function (slideId) { assignStandardPreviewInitially(slideId); unflagAttributionItemAsMustFormat(slideId); });
                }
                slideIdList = slideIdList.concat(indirectSlideIdList);

                slideIdList = slideIdList.filter(docModel.isLayoutOrMasterId);
            }
            self.trigger('previewupdate:after', slideIdList);
        }

        function handleInsertSlide(evt, data) {
          //data :: { id: id, isMasterView: isMasterView, documentReloaded: documentReloaded }
            var slideId = data.id;

            // do nothing, if the document was reloaded with a snapshot
            if (Utils.getBooleanOption(data, 'documentReloaded', false)) { return; }

            if (docModel.isStandardSlideId(slideId)) {

                assignStandardPreviewInitially(slideId);

                if (isStandardView) {
                    self.trigger('previewupdate:after', [slideId]);
                }
            } else {
                assignMasterPreviewInitially(default_target_of_mpo_s, slideId);

                if (isMasterView) {
                    self.trigger('previewupdate:after', [slideId]);
                }
            }
            unflagAttributionItemAsMustFormat(slideId);
        }

        function handleSlideSizeChange() {

            compileStandardPreviewBlueprint();
            compileMasterPreviewBlueprint();

            compilePreviewStatesInitially();
            docModel.getMasterSlideOrder().concat(docModel.getStandardSlideOrder()).forEach(unflagAttributionItemAsMustFormat);

            computeInitialThumbScaling();
        }

        function compileThumbRenderBase() {
            computeInitialThumbScaling();

            $thumbRenderBox = $pageNode.parent(); // currently <div class="app-content" ... />
        }

        function compileView() {
            Utils.info('+++ ENTER compileView +++');

            var time = (new Date());

            if (!imgNodeBlueprint) { imgNodeBlueprint = $('<img/>')[0]; }

            compileStandardPreviewBlueprint();
            compileMasterPreviewBlueprint();

            compileAllDisplayStates();
            compilePreviewStatesInitially();

            compileThumbRenderBase();

            self.getPreview       = getPreview;
          //self.getPreviewList   = getPreviewList;

            self.requestThumb     = requestThumb;
          //self.requestThumbList = requestThumbList;

            self.stopListeningTo(docModel, 'slidestate:formatted', handleSlideStateFormattedBeforeCompiledView);

            handlePreviewCreation({}, formattedSlideCache);
            handleSlideStateFormattedBeforeCompiledView = formattedSlideCache = null;

            self.listenTo(docModel, 'slidestate:formatted', handlePreviewCreation);
            self.listenTo(docModel, 'slidestate:update', handlePreviewUpdate);

            self.stopListeningTo(docModel, 'slideNumFields:update', handleFieldUpdateBeforeCompiledView);
            self.stopListeningTo(docModel, 'slideDateFields:update', handleFieldUpdateBeforeCompiledView);

            handleFieldUpdate({}, fieldUpdateCache);
            handleFieldUpdateBeforeCompiledView = fieldUpdateCache = null;

            self.listenTo(docModel, 'slideNumFields:update', handleFieldUpdate);
            self.listenTo(docModel, 'slideDateFields:update', handleFieldUpdate);

            self.listenTo(docModel, 'inserted:slide', handleInsertSlide);

            Utils.info('+++ compileView :: time : ', (new Date() - time));

            self.trigger('slidepreview:init:after', {});

            //TODO clean up - countercheck for references
            //self.trigger('renderpreviewinitially:after');
        }

        //
        // --------------------------------------------------------------------

        // public methods -----------------------------------------------------
        //

        this.getPreview       = $.noop;
      //this.getPreviewList   = $.noop;

        this.requestThumb     = $.noop;
      //this.requestThumbList = $.noop;

        //
        // --------------------------------------------------------------------

        // initialization -----------------------------------------------------
        //

        /**
         * lazy initialization after composition.
         */
        function initHandler() {

            docModel  = app.getModel();
            $pageNode = docModel.getNode();

            slidepane = self.getSlidePane();
            if (slidepane) {
                var
                    $paneRoot = slidepane.getNode(),
                    $paneNode = $paneRoot.find(SELECTOR__SLIDEPANE_CONTAINER).eq(0);

                preventContendEditable($paneNode);

                self.listenTo(slidepane, 'visiblesliderange:changed', function (evt, slideRange, visibleSlideIdList) {
                    triggerPreviewFormatting(visibleSlideIdList);
                });
            }
            Utils.info('+++ initHandler :: $pageNode : ', $pageNode);

            self.listenTo(docModel, 'change:activeView:after', compileAllDisplayStates);

            self.listenTo(docModel, 'slideModel:init', compileView);
          //self.listenTo(docModel, 'formatmanager:init', compileView);                                   // - too late.
                                                                                                          //
          //self.listenTo(docModel, 'formatmanager:init', registerSlidestateHandlingBeforeCompiledView);  // - too complicated.
            self.listenTo(docModel, 'slidestate:formatted', handleSlideStateFormattedBeforeCompiledView);

            self.listenTo(docModel, 'slideNumFields:update', handleFieldUpdateBeforeCompiledView);
            self.listenTo(docModel, 'slideDateFields:update', handleFieldUpdateBeforeCompiledView);

            // slide size has changed - e.g. orientation change.
            self.listenTo(docModel, 'slidesize:change', handleSlideSizeChange);
        }

        //
        // --------------------------------------------------------------------

    } // mixin SlidePreviewMixin

    // exports ================================================================

    return SlidePreviewMixin;
});
